package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: TrackerServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/TrackerServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.PagerBean;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.TrackedEvent;
import fr.inra.agrosyst.api.entities.security.TrackedEventTopiaDao;
import fr.inra.agrosyst.api.entities.security.TrackedEventType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.services.security.TrackedEventFilter;
import fr.inra.agrosyst.api.services.security.TrackerService;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class TrackerServiceImpl extends AbstractAgrosystService implements TrackerService {

    private static final Log log = LogFactory.getLog(TrackerServiceImpl.class);

    protected BusinessAuthorizationService businessAuthorizationService;

    protected AgrosystUserTopiaDao usersDao;
    protected TrackedEventTopiaDao eventsDao;

    public void setBusinessAuthorizationService(BusinessAuthorizationService businessAuthorizationService) {
        this.businessAuthorizationService = businessAuthorizationService;
    }

    public void setUsersDao(AgrosystUserTopiaDao usersDao) {
        this.usersDao = usersDao;
    }

    public void setEventsDao(TrackedEventTopiaDao eventsDao) {
        this.eventsDao = eventsDao;
    }

    @Override
    public ResultList<TrackedEvent> list(TrackedEventFilter filter) {
        businessAuthorizationService.checkIsAdmin();

        List<TrackedEvent> elements = eventsDao.findAll(); // TODO AThimel 14/02/14 Apply filter
        PagerBean pagerBean = new PagerBean();
        pagerBean.setPageIndex(0);
        pagerBean.setPageSize(elements.size());
        pagerBean.setPagesNumber(1);
        ResultList<TrackedEvent> result = ResultList.of(elements, pagerBean);
        return result;
    }

    protected AgrosystUser getCurrentUser() {
        AgrosystUser result = null;
        try {
            String userId = getSecurityContext().getUserId();
            result = usersDao.forTopiaIdEquals(userId).findUnique();
        } catch (Exception eee) {
            // Aïe
        }
        return result;
    }

    protected void create(TrackedEventType type, List<String> args) {
        eventsDao.create(
                TrackedEvent.PROPERTY_DATE, context.getCurrentDate(),
                TrackedEvent.PROPERTY_AUTHOR, getCurrentUser(),
                TrackedEvent.PROPERTY_TYPE, type,
                TrackedEvent.PROPERTY_ARGS, context.getGson().toJson(args)
        );
    }

    public void userCreated(AgrosystUser user) {
        List<String> args = Lists.newArrayList(user.getEmail());
        create(TrackedEventType.USER_CREATION, args);
    }

    public void userModified(AgrosystUser user, boolean passwordChange) {
        List<String> args = Lists.newArrayList(user.getEmail(), String.valueOf(passwordChange));
        AgrosystUser currentUser = getCurrentUser();
        TrackedEventType type = user.equals(currentUser) ? TrackedEventType.USER_MOD_SELF : TrackedEventType.USER_MOD;
        create(type, args);
    }

    public void userActivation(AgrosystUser user, boolean active) {
        List<String> args = Lists.newArrayList(user.getEmail(), String.valueOf(active));
        create(TrackedEventType.USER_ACTIVATION, args);
    }

    protected List<String> roleToArgs(UserRole role) {
        List<String> result = Lists.newArrayList(role.getType().name());
        switch (role.getType()) {
            case DOMAIN_RESPONSIBLE:
                Domain domain = context.getDaoSupplier().getDomainDao().forCodeEquals(role.getDomainCode()).findAny();
                result.add(domain.getName());
                result.add("Toutes campagnes");
                break;
            case GROWING_PLAN_RESPONSIBLE:
                GrowingPlan growingPlan = context.getDaoSupplier().getGrowingPlanDao().forCodeEquals(role.getGrowingPlanCode()).findAny();
                result.add(growingPlan.getName());
                result.add("Toutes campagnes");
                break;
            case GS_DATA_PROCESSOR:
                if (Strings.isNullOrEmpty(role.getGrowingSystemCode())) {
                    GrowingSystem growingSystem = context.getDaoSupplier().getGrowingSystemDao().forTopiaIdEquals(role.getGrowingSystemId()).findUnique();
                    result.add(growingSystem.getName());
                    result.add(String.valueOf(growingSystem.getGrowingPlan().getDomain().getCampaign()));
                } else {
                    GrowingSystem growingSystem = context.getDaoSupplier().getGrowingSystemDao().forCodeEquals(role.getGrowingSystemCode()).findAny();
                    result.add(growingSystem.getName());
                    result.add("Toutes campagnes");
                }
                break;
            case NETWORK_RESPONSIBLE:
                Network network = context.getDaoSupplier().getNetworkDao().forTopiaIdEquals(role.getNetworkId()).findUnique();
                result.add(network.getName());
                break;
        }
        result.add(role.getAgrosystUser().getEmail());
        return result;
    }

    public void roleAdded(UserRole role) {
        List<String> args = roleToArgs(role);
        create(TrackedEventType.USER_ROLE_ADD, args);
    }

    public void roleRemoved(UserRole role) {
        List<String> args = roleToArgs(role);
        create(TrackedEventType.USER_ROLE_REMOVE, args);
    }

}
