package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: SecurityHelper.java 3733 2014-03-12 15:33:21Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/SecurityHelper.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Map;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.security.ComputedUserPermission;
import fr.inra.agrosyst.api.entities.security.PermissionObjectType;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 0.8
 */
public class SecurityHelper {

    protected static final String ZONE_GROWING_SYSTEM = Joiner.on(".").join(Zone.PROPERTY_PLOT, Plot.PROPERTY_GROWING_SYSTEM);

    public static final int PERMISSION_READ_VALIDATED = 1;
    public static final int PERMISSION_READ_RAW = 3;
    public static final int PERMISSION_WRITE = 7;
    public static final int PERMISSION_ADMIN = 15;

    protected static final String IN_SELECT_CUP_ACTION_HIGHER_THAN =
            " %s.%s IN ( " +
                    "   SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
                    "   FROM " + ComputedUserPermission.class.getName() + " cup" +
                    "   WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
                    "   AND   cup." + ComputedUserPermission.PROPERTY_TYPE + " = :%s" +
                    "   AND   cup." + ComputedUserPermission.PROPERTY_ACTION + " >= :cup_action" +
                    " ) ";
    protected static final String IN_SELECT_CUP_ACTION_EQUALS =
            " %s.%s IN ( " +
                    "   SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
                    "   FROM " + ComputedUserPermission.class.getName() + " cup" +
                    "   WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
                    "   AND   cup." + ComputedUserPermission.PROPERTY_TYPE + " = :%s" +
                    "   AND   cup." + ComputedUserPermission.PROPERTY_ACTION + " = :cup_action_read_validated" +
                    " ) ";

    protected static void addMinimumActionSecurityFilter(StringBuilder query,
                                                         Map<String, Object> args,
                                                         SecurityContext securityContext,
                                                         String alias,
                                                         PermissionObjectType cupTypeCode,
                                                         PermissionObjectType cupTypeId,
                                                         String propCode,
                                                         String propId,
                                                         int action) {

        if (!securityContext.isAdmin()) {

            String userId = securityContext.getUserId();

            String codeInReadRaw = String.format(IN_SELECT_CUP_ACTION_HIGHER_THAN, alias, propCode, "cup_type_code");
            String topiaIdInReadRaw = String.format(IN_SELECT_CUP_ACTION_HIGHER_THAN, alias, propId, "cup_type_id");

            String securityQuery = String.format(
                    " AND ( %s OR %s ) ",
                    codeInReadRaw,
                    topiaIdInReadRaw);

            query.append(securityQuery);
            args.put("cup_userId", userId);
            args.put("cup_type_code", cupTypeCode);
            args.put("cup_type_id", cupTypeId);
            args.put("cup_action", action);
        }
    }

    protected static void addReadSecurityFilter(StringBuilder query,
                                                Map<String, Object> args,
                                                SecurityContext securityContext,
                                                String alias,
                                                PermissionObjectType cupTypeCode,
                                                PermissionObjectType cupTypeId,
                                                String propCode,
                                                String propId) {

        addMinimumActionSecurityFilter(query, args, securityContext, alias,
                cupTypeCode, cupTypeId, propCode, propId,
                SecurityHelper.PERMISSION_READ_RAW);
    }

    protected static void addReadValidatedSecurityFilter(StringBuilder query,
                                                         Map<String, Object> args,
                                                         SecurityContext securityContext,
                                                         String alias,
                                                         PermissionObjectType cupTypeCode,
                                                         PermissionObjectType cupTypeId,
                                                         String propCode,
                                                         String propId) {

        addMinimumActionSecurityFilter(query, args, securityContext, alias,
                cupTypeCode, cupTypeId, propCode, propId,
                SecurityHelper.PERMISSION_READ_VALIDATED);
    }

    protected static void addWriteSecurityFilter(StringBuilder query,
                                                 Map<String, Object> args,
                                                 SecurityContext securityContext,
                                                 String alias,
                                                 PermissionObjectType cupTypeCode,
                                                 PermissionObjectType cupTypeId,
                                                 String propCode,
                                                 String propId) {
        addMinimumActionSecurityFilter(query, args, securityContext, alias,
                cupTypeCode, cupTypeId, propCode, propId,
                SecurityHelper.PERMISSION_WRITE);
    }

    protected static void addReadSecurityFilterOnValidableEntity(StringBuilder query,
                                                                 Map<String, Object> args,
                                                                 SecurityContext securityContext,
                                                                 String alias,
                                                                 PermissionObjectType cupTypeCode,
                                                                 PermissionObjectType cupTypeId,
                                                                 String propCode,
                                                                 String propId,
                                                                 String propValidated) {

        if (!securityContext.isAdmin()) {

            String userId = securityContext.getUserId();

            String codeInReadRaw = String.format(IN_SELECT_CUP_ACTION_HIGHER_THAN, alias, propCode, "cup_type_code");
            String topiaIdInReadRaw = String.format(IN_SELECT_CUP_ACTION_HIGHER_THAN, alias, propId, "cup_type_id");
            String validated = alias + "." + propValidated;
            String codeInReadValidated = String.format(IN_SELECT_CUP_ACTION_EQUALS, alias, propCode, "cup_type_code");
            String topiaIdInReadValidated = String.format(IN_SELECT_CUP_ACTION_EQUALS, alias, propId, "cup_type_id");

            String securityQuery = String.format(
                    " AND ( ( %s OR %s ) OR ( %s = true AND ( %s OR %s ) ) ) ",
                    codeInReadRaw,
                    topiaIdInReadRaw,
                    validated,
                    codeInReadValidated,
                    topiaIdInReadValidated);

            query.append(securityQuery);
            args.put("cup_userId", userId);
            args.put("cup_type_code", cupTypeCode);
            args.put("cup_type_id", cupTypeId);
            args.put("cup_action", SecurityHelper.PERMISSION_READ_RAW);
            args.put("cup_action_read_validated", SecurityHelper.PERMISSION_READ_VALIDATED);
        }

    }

    public static void addDomainFilter(StringBuilder query,
                                       Map<String, Object> args,
                                       SecurityContext securityContext,
                                       String alias) {
        addReadSecurityFilterOnValidableEntity(query, args, securityContext, alias,
                PermissionObjectType.DOMAIN_CODE, PermissionObjectType.DOMAIN_ID,
                Domain.PROPERTY_CODE, Domain.PROPERTY_TOPIA_ID, Domain.PROPERTY_VALIDATED);
    }

    public static void addWritableDomainFilter(StringBuilder query,
                                               Map<String, Object> args,
                                               SecurityContext securityContext,
                                               String alias) {
        addWriteSecurityFilter(query, args, securityContext, alias,
                PermissionObjectType.DOMAIN_CODE, PermissionObjectType.DOMAIN_ID,
                Domain.PROPERTY_CODE, Domain.PROPERTY_TOPIA_ID);
    }

    public static void addWritableDomainFilterForDecisionRuleCreation(StringBuilder query,
                                                                      Map<String, Object> args,
                                                                      SecurityContext securityContext,
                                                                      String alias) {

        StringBuilder subQuery1 = new StringBuilder();
        addWriteSecurityFilter(subQuery1, args, securityContext, alias,
                PermissionObjectType.DOMAIN_CODE, PermissionObjectType.DOMAIN_ID,
                Domain.PROPERTY_CODE, Domain.PROPERTY_TOPIA_ID);

        StringBuilder subQuery2 = new StringBuilder();
        if (!securityContext.isAdmin()) {

            subQuery2.append(
                    String.format(
                            IN_SELECT_CUP_ACTION_HIGHER_THAN,
                            alias,
                            Domain.PROPERTY_TOPIA_ID + " IN ( SELECT gp." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID + " FROM " + GrowingPlan.class.getName() + " gp WHERE gp." + GrowingPlan.PROPERTY_CODE,
                            "cup_gp_code"));
            subQuery2.append(" ) "); // Because of opened bracket

            args.put("cup_gp_code", PermissionObjectType.GROWING_PLAN_CODE);

            // Sql is : select * from domain where topiaId in (select domain from growingplan where code in ( select object from computeduserpermission where type='GROWING_PLAN_CODE' and userid='fr.inra.agrosyst.api.entities.security.AgrosystUser_9c7ce4d0-8ef4-4dd4-bf8e-ca9870784adf' and action >= 7));

            // Here's the sample generated HQL :

//            FROM fr.inra.agrosyst.api.entities.Domain D
//            WHERE 1 = 1
//            AND D.active = :active0
//            AND (
//                D.code IN (
//                    SELECT DISTINCT cup.object
//                    FROM fr.inra.agrosyst.api.entities.security.ComputedUserPermission cup
//                    WHERE cup.userId = :cup_userId
//                    AND   cup.type = :cup_type_code
//                    AND   cup.action >= :cup_action
//                ) OR
//                D.topiaId IN (
//                    SELECT DISTINCT cup.object
//                    FROM fr.inra.agrosyst.api.entities.security.ComputedUserPermission cup
//                    WHERE cup.userId = :cup_userId
//                    AND   cup.type = :cup_type_id
//                    AND   cup.action >= :cup_action
//                ) OR
//                D.topiaId IN (
//                    SELECT gp.domain.topiaId
//                    FROM fr.inra.agrosyst.api.entities.GrowingPlan
//                    WHERE gp.code IN (
//                        SELECT DISTINCT cup.object
//                        FROM fr.inra.agrosyst.api.entities.security.ComputedUserPermission cup
//                        WHERE cup.userId = :cup_userId
//                        AND   cup.type = :cup_gp_code
//                        AND   cup.action >= :cup_action
//                    )
//                )
//            )
//            ORDER BY lower (D.name)


        }

        if (subQuery1.length() > 0 && subQuery2.length() > 0) {
            String hql = subQuery1.toString();
            Preconditions.checkState(hql.endsWith(" ) "));
            hql = hql.substring(0, hql.length() - 3);
            hql += " OR ";
            hql += subQuery2.toString();
            hql += " ) ";
            query.append(hql);
        }
    }

    public static void addGrowingPlanFilter(StringBuilder query,
                                            Map<String, Object> args,
                                            SecurityContext securityContext,
                                            String alias) {
        addReadSecurityFilterOnValidableEntity(query, args, securityContext, alias,
                PermissionObjectType.GROWING_PLAN_CODE, PermissionObjectType.GROWING_PLAN_ID,
                GrowingPlan.PROPERTY_CODE, GrowingPlan.PROPERTY_TOPIA_ID, GrowingPlan.PROPERTY_VALIDATED);
    }

    public static void addGrowingSystemFilter(StringBuilder query,
                                              Map<String, Object> args,
                                              SecurityContext securityContext,
                                              String alias) {
        addReadSecurityFilterOnValidableEntity(query, args, securityContext, alias,
                PermissionObjectType.GROWING_SYSTEM_CODE, PermissionObjectType.GROWING_SYSTEM_ID,
                GrowingSystem.PROPERTY_CODE, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_VALIDATED);
    }

    public static void addZoneFilter(StringBuilder query,
                                     Map<String, Object> args,
                                     SecurityContext securityContext,
                                     String alias) {

        if (!securityContext.isAdmin()) {
            // Pas de growingsystem, il faut le droit sur le domaine

            StringBuilder subQuery1 = new StringBuilder(" SELECT p1.topiaId FROM " + Plot.class.getName() + " p1 ");
            subQuery1.append(" WHERE p1." + Plot.PROPERTY_GROWING_SYSTEM + " IS NULL ");
            addReadSecurityFilterOnValidableEntity(subQuery1, args, securityContext, "p1",
                    PermissionObjectType.DOMAIN_CODE, PermissionObjectType.DOMAIN_ID,
                    Joiner.on(".").join(Plot.PROPERTY_DOMAIN, Domain.PROPERTY_CODE),
                    Joiner.on(".").join(Plot.PROPERTY_DOMAIN, Domain.PROPERTY_TOPIA_ID),
                    Joiner.on(".").join(Plot.PROPERTY_DOMAIN, Domain.PROPERTY_VALIDATED)
            );

            // Dans l'une des 2 requêtes, il faut renommer les propriétés cup_type_code et cup_type_id
            String subQuery1String = subQuery1.toString();
            for (String toReplace : Sets.newHashSet("cup_type_code", "cup_type_id")) {
                final String dotPlusToReplace = ":" + toReplace;
                while (subQuery1String.contains(dotPlusToReplace)) {
                    int index = subQuery1String.indexOf(dotPlusToReplace);
                    String part0 = subQuery1String.substring(0, index);
                    String part1 = ":domain_" + toReplace;
                    String part2 = subQuery1String.substring(index + dotPlusToReplace.length());
                    subQuery1String = part0 + part1 + part2;
                }
            }
            args.remove("cup_type_code");
            args.remove("cup_type_id");
            args.put("domain_cup_type_code", PermissionObjectType.DOMAIN_CODE);
            args.put("domain_cup_type_id", PermissionObjectType.DOMAIN_ID);

            // growing system, il faut le droit sur le sdc
            StringBuilder subQuery2 = new StringBuilder(" SELECT p2.topiaId FROM " + Plot.class.getName() + " p2 ");
            subQuery2.append(" WHERE p2." + Plot.PROPERTY_GROWING_SYSTEM + " IS NOT NULL ");
            addReadSecurityFilterOnValidableEntity(subQuery2, args, securityContext, "p2",
                    PermissionObjectType.GROWING_SYSTEM_CODE, PermissionObjectType.GROWING_SYSTEM_ID,
                    Joiner.on(".").join(Plot.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_CODE),
                    Joiner.on(".").join(Plot.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_TOPIA_ID),
                    Joiner.on(".").join(Plot.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_VALIDATED)
            );

            query.append(String.format(
                    " AND ( " +
                            alias + "." + Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_TOPIA_ID + " IN ( %s ) " +
                            " OR " +
                            alias + "." + Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_TOPIA_ID + " IN ( %s ) " +
                            " ) ", subQuery1String, subQuery2));

        }

    }

    public static void addWritableGrowingSystemFilter(StringBuilder query,
                                                      Map<String, Object> args,
                                                      SecurityContext securityContext,
                                                      String alias) {
        addWriteSecurityFilter(query, args, securityContext, alias,
                PermissionObjectType.GROWING_SYSTEM_CODE, PermissionObjectType.GROWING_SYSTEM_ID,
                GrowingSystem.PROPERTY_CODE, GrowingSystem.PROPERTY_TOPIA_ID);
    }

    public static void addDecisionRuleFilter(StringBuilder query,
                                             Map<String, Object> args,
                                             SecurityContext securityContext,
                                             String alias) {


        // Chargement des decision rule sur domaines :
// SELECT dr.topiaId, dr.name, dr.domainCode FROM decisionrule dr
// WHERE dr.active IS true
// AND (
//    dr.domainCode IN (
//            SELECT DISTINCT cup.object
//            FROM ComputedUserPermission cup
//            WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//            AND   cup.type = 'DOMAIN_CODE'
//            AND   cup.action >= 3
//    )
//    OR dr.domainCode IN (
//            SELECT d.code FROM domain d WHERE d.topiaId IN (
//                SELECT DISTINCT cup.object
//                FROM ComputedUserPermission cup
//                WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//                AND   cup.type = 'DOMAIN_ID'
//                AND   cup.action >= 3
//        )
//    )
//
//    OR dr.domainCode IN (
//            SELECT d.code
//            FROM domain d
//            WHERE d.validated = true
//            AND d.code IN (
//                SELECT DISTINCT cup.object
//                FROM ComputedUserPermission cup
//                WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//                AND   cup.type = 'DOMAIN_CODE'
//                AND   cup.action = 1
//        )
//    )
//    OR dr.domainCode IN (
//            SELECT d.code
//            FROM domain d
//            WHERE d.validated = true
//            AND d.topiaId IN (
//                SELECT DISTINCT cup.object
//                FROM ComputedUserPermission cup
//                WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//                AND   cup.type = 'DOMAIN_ID'
//                AND   cup.action = 1
//        )
//    )
// )

        // Vu que les domaines sont automatiquement validés, on peut simplifier :

        // Chargement des decision rule sur domaines :
// SELECT dr.topiaId, dr.name, dr.domainCode FROM decisionrule dr
// WHERE dr.active IS true
// AND (
//    dr.domainCode IN (
//            SELECT DISTINCT cup.object
//            FROM ComputedUserPermission cup
//            WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//            AND   cup.type = 'DOMAIN_CODE'
//            AND   cup.action >= 1
//    )
//    OR dr.domainCode IN (
//            SELECT d.code FROM domain d WHERE d.topiaId IN (
//                SELECT DISTINCT cup.object
//                FROM ComputedUserPermission cup
//                WHERE cup.userId = 'fr.inra.agrosyst.api.entities.security.AgrosystUser_99b18b32-f31c-4848-9dc5-d805cfb61803'
//                AND   cup.type = 'DOMAIN_ID'
//                AND   cup.action >= 1
//        )
//    )
//

        StringBuilder subQuery2 = new StringBuilder();

        // Les domaines sont tous validés
        addReadValidatedSecurityFilter(subQuery2, args, securityContext, alias,
                PermissionObjectType.DOMAIN_CODE,
                PermissionObjectType.DOMAIN_ID,
                DecisionRule.PROPERTY_DOMAIN_CODE,
                "domainCode IN ( SELECT d.code FROM " + Domain.class.getName() + " d WHERE d.topiaId ");

        if (subQuery2.length() > 0) {
            subQuery2.append(" ) "); // Because of opened bracket
        }
        query.append(subQuery2);

    }

    public static void addPracticedSystemFilter(StringBuilder query,
                                                Map<String, Object> args,
                                                SecurityContext securityContext,
                                                String alias) {

        String propertyValidated = PracticedSystem.PROPERTY_VALIDATED + " = true " +
                "AND " + alias + "." + Joiner.on('.').join(PracticedSystem.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_VALIDATED);

        addReadSecurityFilterOnValidableEntity(query, args, securityContext, alias,
                PermissionObjectType.GROWING_SYSTEM_CODE, PermissionObjectType.GROWING_SYSTEM_ID,
                Joiner.on('.').join(PracticedSystem.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_CODE),
                Joiner.on('.').join(PracticedSystem.PROPERTY_GROWING_SYSTEM, GrowingSystem.PROPERTY_TOPIA_ID),
                propertyValidated);
//        if (!securityContext.isAdmin()) {
//            query += " AND ( ps." + PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_CODE + " IN (" +
//                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
//                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
//                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
//                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_code" +
//                    ")" +
//                    " OR ps." + PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID + " IN (" +
//                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
//                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
//                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
//                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_id" +
//                    ")" +
//                    " )";
//            args.put("cup_userId", securityContext.getUserId());
//            args.put("cup_type_code", PermissionObjectType.GROWING_SYSTEM_CODE);
//            args.put("cup_type_id", PermissionObjectType.GROWING_SYSTEM_ID);
//        }
    }

    public static void addPracticedPlotFilter(StringBuilder query,
                                              Map<String, Object> args,
                                              SecurityContext securityContext,
                                              String alias) {

        StringBuilder subQuery = new StringBuilder();
        addPracticedSystemFilter(subQuery, args, securityContext, "ps");

        if (subQuery.length() > 0) {
            String subFilter = String.format(
                    " AND %s.%s IN ( SELECT ps.%s FROM %s ps WHERE 1=1 %s ) ",
                    alias,
                    Joiner.on('.').join(PracticedPlot.PROPERTY_PRACTICED_SYSTEM, PracticedSystem.PROPERTY_TOPIA_ID),
                    PracticedSystem.PROPERTY_TOPIA_ID,
                    PracticedSystem.class.getName(),
                    subQuery.toString()
            );
            query.append(subFilter);
        }

//        if (!securityContext.isAdmin()) {
//            query += " AND ( pp." + PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_CODE + " IN (" +
//                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
//                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
//                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
//                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_code" +
//                    ")" +
//                    " OR pp." + PROPERTY_GROWING_SYSTEM_ID + " IN (" +
//                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
//                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
//                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
//                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_id" +
//                    ")" +
//                    " )";
//            args.put("cup_userId", securityContext.getUserId());
//            args.put("cup_type_code", PermissionObjectType.GROWING_SYSTEM_CODE);
//            args.put("cup_type_id", PermissionObjectType.GROWING_SYSTEM_ID);
//        }

    }
}
