package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: BusinessAuthorizationServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/BusinessAuthorizationServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.topia.persistence.TopiaIdFactory;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import fr.inra.agrosyst.api.entities.AttachmentMetadata;
import fr.inra.agrosyst.api.entities.AttachmentMetadataTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkManager;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRuleTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeTopiaDao;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.performance.PerformanceTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlotTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.PermissionObjectType;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.services.security.AgrosystAccessDeniedException;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.common.CacheService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class BusinessAuthorizationServiceImpl extends AuthorizationServiceImpl implements BusinessAuthorizationService {

    protected static final Predicate<NetworkManager> IS_ACTIVE_NETWORK_MANAGER = new Predicate<NetworkManager>() {
        @Override
        public boolean apply(NetworkManager input) {
            // TODO AThimel 22/10/13 Manage case where toDate is in the future
            return input.isActive() && input.getToDate() == null;
        }
    };

    protected PracticedSystemTopiaDao practicedSystemDao;
    protected ManagementModeTopiaDao managementModeDao;
    protected DecisionRuleTopiaDao decisionRuleDao;
    protected ZoneTopiaDao zoneDao;
    protected AttachmentMetadataTopiaDao attachmentMetadataDao;
    protected PerformanceTopiaDao performanceDao;
    protected PlotTopiaDao plotDao;
    protected PracticedPlotTopiaDao practicedPlotDao;

    public void setPracticedSystemDao(PracticedSystemTopiaDao practicedSystemDao) {
        this.practicedSystemDao = practicedSystemDao;
    }

    public void setManagementModeDao(ManagementModeTopiaDao managementModeDao) {
        this.managementModeDao = managementModeDao;
    }

    public void setDecisionRuleDao(DecisionRuleTopiaDao decisionRuleDao) {
        this.decisionRuleDao = decisionRuleDao;
    }

    public void setZoneDao(ZoneTopiaDao zoneDao) {
        this.zoneDao = zoneDao;
    }

    public void setAttachmentMetadataDao(AttachmentMetadataTopiaDao attachmentMetadataDao) {
        this.attachmentMetadataDao = attachmentMetadataDao;
    }

    public void setPerformanceDao(PerformanceTopiaDao performanceDao) {
        this.performanceDao = performanceDao;
    }

    public void setPlotDao(PlotTopiaDao plotDao) {
        this.plotDao = plotDao;
    }

    public void setPracticedPlotDao(PracticedPlotTopiaDao practicedPlotDao) {
        this.practicedPlotDao = practicedPlotDao;
    }

    protected boolean hasObjectPermissionAdmin(String userId, PermissionObjectType type, String object) {
        boolean result = hasPermissionActionLevel(userId, type, object, SecurityHelper.PERMISSION_ADMIN);
        return result;
    }

    protected boolean hasObjectPermissionWritable(String userId, PermissionObjectType type, String object) {
        boolean result = hasPermissionActionLevel(userId, type, object, SecurityHelper.PERMISSION_WRITE);
        return result;
    }

    protected boolean hasObjectPermissionWritable(String userId, String object) {
        boolean result = hasPermissionActionLevel(userId, object, SecurityHelper.PERMISSION_WRITE);
        return result;
    }

    protected boolean hasObjectPermissionReadable(String userId, PermissionObjectType type, String object) {
        boolean result = hasPermissionActionLevel(userId, type, object, SecurityHelper.PERMISSION_READ_VALIDATED);
        return result;
    }

    protected boolean hasObjectPermissionReadable(String userId, String object) {
        boolean result = hasPermissionActionLevel(userId, object, SecurityHelper.PERMISSION_READ_VALIDATED);
        return result;
    }

    protected boolean hasPermissionActionLevel(String userId, String object, int expectedActionLevel) {
        int maxActionLevel = getHighestPermissionLevel(userId, null, object);
        boolean result = maxActionLevel >= expectedActionLevel;
        return result;
    }

    protected boolean hasPermissionActionLevel(String userId, PermissionObjectType type, String object, int expectedActionLevel) {
        int maxActionLevel = getHighestPermissionLevel(userId, type, object);
        boolean result = maxActionLevel >= expectedActionLevel;
        return result;
    }

    protected int getHighestPermissionLevel(String userId, PermissionObjectType type, String object) {
        int result = computedUserPermissionDao.getMaxAction(userId, object, type);
        return result;
    }

    public boolean isDomainWritable(String domainId) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_ID, domainId);

        if (!result) {
            Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
            result = hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
        }

        return result;
    }

    @Override
    public boolean areDomainPlotsEditable(String domainId) {
        boolean result;
        if (isDomainWritable(domainId)) {
            result = true;
        } else {
            Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
            // XXX AThimel 12/03/14 En fait il faudrait regarder sur tous les dispositifs à partir du code de dommaine
            Iterable<GrowingPlan> growingPlans = growingPlanDao.forDomainEquals(domain).findAllLazy();
            result = Iterables.any(growingPlans, new Predicate<GrowingPlan>() {
                @Override
                public boolean apply(GrowingPlan input) {
                    return isGrowingPlanAdministrable(input.getCode());
                }
            });
        }
        return result;
    }

    @Override
    public boolean isDomainAdministrable(String domainCode) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionAdmin(userId, PermissionObjectType.DOMAIN_CODE, domainCode);

        return result;
    }

    protected boolean isDomainCodeWritable(String domainCode) {

        String userId = getUserId();

        // XXX AThimel 18/02/14 Maybe iterate over domains for a given code and test each ID ?

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_CODE, domainCode);

        return result;
    }

    public boolean isGrowingPlanWritable(String growingPlanId) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, PermissionObjectType.GROWING_PLAN_ID, growingPlanId);

        if (!result) {
            GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
            result = hasObjectPermissionWritable(userId, PermissionObjectType.GROWING_PLAN_CODE, growingPlan.getCode());
        }

        return result;
    }

    @Override
    public boolean isGrowingPlanAdministrable(String growingPlanCode) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionAdmin(userId, PermissionObjectType.GROWING_PLAN_CODE, growingPlanCode);

        return result;
    }

    public boolean isGrowingSystemWritable(String growingSystemId) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, PermissionObjectType.GROWING_SYSTEM_ID, growingSystemId);

        if (!result) {
            GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
            result = hasObjectPermissionWritable(userId, PermissionObjectType.GROWING_SYSTEM_CODE, growingSystem.getCode());
        }

        return result;
    }

    @Override
    public boolean isGrowingSystemAdministrable(String growingSystemCode) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionAdmin(userId, PermissionObjectType.GROWING_SYSTEM_CODE, growingSystemCode);

        return result;
    }

    public boolean isPracticedSystemWritable(String practicedSystemId) {
        PracticedSystem practicedSystem = practicedSystemDao.forTopiaIdEquals(practicedSystemId).findUnique();
        String growingSystemId = practicedSystem.getGrowingSystem().getTopiaId(); // TODO AThimel 21/10/13 Optimize it
        boolean result = isGrowingSystemWritable(growingSystemId);
        return result;
    }

    @Override
    public boolean isNetworkWritable(String networkId) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, PermissionObjectType.NETWORK_ID, networkId);

        return result;
    }

    @Override
    public boolean isManagementModeWritable(String managementModeId) {
        ManagementMode managementMode = managementModeDao.forTopiaIdEquals(managementModeId).findUnique();
        String growingSystemId = managementMode.getGrowingSystem().getTopiaId(); // TODO AThimel 21/10/13 Optimize it
        boolean result = isGrowingSystemWritable(growingSystemId);
        return result;
    }

    @Override
    public boolean isManagementModeReadable(String managementModeId) {
        ManagementMode managementMode = managementModeDao.forTopiaIdEquals(managementModeId).findUnique();
        String growingSystemId = managementMode.getGrowingSystem().getTopiaId(); // TODO AThimel 21/10/13 Optimize it
        boolean result = isGrowingSystemReadable(growingSystemId);
        return result;
    }

    @Override
    // https://forge.codelutin.com/issues/4439
    public boolean isDecisionRuleWritable(String decisionRuleId) {
        DecisionRule decisionRule = decisionRuleDao.forTopiaIdEquals(decisionRuleId).findUnique();
        boolean result = isDomainCodeWritable(decisionRule.getDomainCode());
        if (!result) {
            List<Domain> domains = domainDao.forCodeEquals(decisionRule.getDomainCode()).findAll();
            List<GrowingPlan> growingPlans = growingPlanDao.forDomainIn(domains).findAll();
            for (GrowingPlan growingPlan : growingPlans) {
                String growingPlanId = growingPlan.getTopiaId();
                if (isGrowingPlanWritable(growingPlanId)) {
                    result = true;
                    break;
                }
            }
        }
        return result;
    }

    @Override
    // https://forge.codelutin.com/issues/4439
    public boolean isDecisionRuleReadable(String decisionRuleId) {
        DecisionRule decisionRule = decisionRuleDao.forTopiaIdEquals(decisionRuleId).findUnique();
        boolean result = isDomainCodeReadable(decisionRule.getDomainCode());
        if (!result) {
            List<Domain> domains = domainDao.forCodeEquals(decisionRule.getDomainCode()).findAll();
            List<GrowingPlan> growingPlans = growingPlanDao.forDomainIn(domains).findAll();
            for (GrowingPlan growingPlan : growingPlans) {
                String growingPlanId = growingPlan.getTopiaId();
                if (isGrowingPlanReadable(growingPlanId)) {
                    result = true;
                    break;
                }
            }
        }
        return result;
    }

    @Override
    public boolean isZoneWritable(String zoneId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();

        boolean result;
        GrowingSystem growingSystem = zone.getPlot().getGrowingSystem();
        if (growingSystem == null) {
            String domainId = zone.getPlot().getDomain().getTopiaId();
            result = isDomainWritable(domainId);
        } else {
            result = isGrowingSystemWritable(growingSystem.getTopiaId());
        }
        return result;
    }

    @Override
    public void domainCreated(Domain domain) {

        String userId = getUserId();

        AgrosystUser user = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

        userRoleDao.create(
                UserRole.PROPERTY_AGROSYST_USER, user,
                UserRole.PROPERTY_TYPE, RoleType.DOMAIN_RESPONSIBLE,
                UserRole.PROPERTY_DOMAIN_CODE, domain.getCode()
        );

        dropComputedPermissions0(userId);

        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }

    @Override
    public void growingPlanCreated(GrowingPlan growingPlan) {
        String userId = getUserId();

        AgrosystUser user = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

        userRoleDao.create(
                UserRole.PROPERTY_AGROSYST_USER, user,
                UserRole.PROPERTY_TYPE, RoleType.GROWING_PLAN_RESPONSIBLE,
                UserRole.PROPERTY_GROWING_PLAN_CODE, growingPlan.getCode()
        );

        dropComputedPermissions0(userId);
        Domain domain = growingPlan.getDomain();
        domainIsDirty(domain);

        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }

    @Override
    public void growingSystemCreated(GrowingSystem growingSystem) {
        GrowingPlan growingPlan = growingSystem.getGrowingPlan();
        growingPlanIsDirty(growingPlan);

        cacheService.clear(); // for network projections
    }

    @Override
    public void networkCreated(Network network) {
        String userId = getUserId();

        resetNetworkManagers(network);

        dropComputedPermissions0(userId);

        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }

    @Override
    public void networkUpdated(Network network) {
        resetNetworkManagers(network);
        // TODO AThimel 08/10/13 This is a bit aggressive, find a better way
        objectsAreDirty(PermissionObjectType.NETWORK_ID);
    }

    protected void resetNetworkManagers(Network network) {
        List<UserRole> existingRoles = Lists.newArrayList(userRoleDao.forProperties(
                UserRole.PROPERTY_TYPE, RoleType.NETWORK_RESPONSIBLE,
                UserRole.PROPERTY_NETWORK_ID, network.getTopiaId()).findAll());
        userRoleDao.deleteAll(existingRoles);

        Iterable<NetworkManager> activeNetworkManagers = Iterables.filter(network.getManagers(), IS_ACTIVE_NETWORK_MANAGER);
        List<UserRole> newRoles = Lists.newArrayList();
        for (NetworkManager manager : activeNetworkManagers) {

            UserRole newRole = userRoleDao.create(
                    UserRole.PROPERTY_AGROSYST_USER, manager.getAgrosystUser(),
                    UserRole.PROPERTY_TYPE, RoleType.NETWORK_RESPONSIBLE,
                    UserRole.PROPERTY_NETWORK_ID, network.getTopiaId()
            );

            Iterator<UserRole> existingRolesIt = existingRoles.iterator();
            boolean found = false;
            while (existingRolesIt.hasNext() && !found) {
                UserRole existingRole = existingRolesIt.next();
                if (existingRole.getAgrosystUser().equals(newRole.getAgrosystUser())) {
                    existingRolesIt.remove();
                    found = true;
                }
            }
            if (!found) {
                newRoles.add(newRole);
            }
        }

        for (UserRole existingRole : existingRoles) {
            trackerService.roleRemoved(existingRole);
        }
        for (UserRole newRole : newRoles) {
            trackerService.roleAdded(newRole);
        }
    }

    protected void growingPlanIsDirty(GrowingPlan growingPlan) {
        objectIsDirty(PermissionObjectType.GROWING_PLAN_ID, growingPlan.getTopiaId());
        objectIsDirty(PermissionObjectType.GROWING_PLAN_CODE, growingPlan.getCode());

        domainIsDirty(growingPlan.getDomain());
    }

    protected void domainIsDirty(Domain domain) {
        objectIsDirty(PermissionObjectType.DOMAIN_ID, domain.getTopiaId());
        objectIsDirty(PermissionObjectType.DOMAIN_CODE, domain.getCode());
    }

    @Override
    public void checkIsAdmin() throws AgrosystAccessDeniedException {
        if (!isAdmin()) {
            // TODO AThimel 07/10/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkDomainReadable(String domainId) throws AgrosystAccessDeniedException {
        boolean result = isDomainReadable(domainId);

        if (!result) {
            // TODO AThimel 07/10/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean isDomainReadable(String domainId) {
        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionReadable(userId, PermissionObjectType.DOMAIN_ID, domainId);

        if (!result) {
            Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
            result = hasObjectPermissionReadable(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
        }
        return result;
    }

    /**
     * @deprecated just for decision rule changes
     */
    @Deprecated
    protected void checkDomainCodeReadable(String domainCode) throws AgrosystAccessDeniedException {
        boolean result = isDomainCodeReadable(domainCode);

        if (!result) {
            // TODO AThimel 07/10/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean isDomainCodeReadable(String domainCode) {
        String userId = getUserId();

        return isAdmin() ||
                hasObjectPermissionReadable(userId, PermissionObjectType.DOMAIN_CODE, domainCode);
    }

    @Override
    public boolean shouldAnonymizeDomain(String domainId) {
        boolean result = shouldAnonymizeDomain(domainId, false).getLeft();
        return result;
    }

    public Pair<Boolean, Boolean> shouldAnonymizeDomain(String domainId, boolean allowUnreadable) {
        // TODO AThimel 26/02/14 Introduce cache
        boolean canRead = true;
        boolean result = false;

        if (!isAdmin()) {
            String userId = getUserId();

            int highestPermissionLevel = getHighestPermissionLevel(userId, PermissionObjectType.DOMAIN_ID, domainId);
            if (highestPermissionLevel == 0) {
                Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
                highestPermissionLevel = getHighestPermissionLevel(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
            }

            Preconditions.checkState(allowUnreadable || highestPermissionLevel > 0);
            result = highestPermissionLevel < SecurityHelper.PERMISSION_READ_RAW;
            canRead = highestPermissionLevel >= SecurityHelper.PERMISSION_READ_VALIDATED;
        }
        return Pair.of(result, canRead);
    }

//    @Override
//    public boolean shouldAnonymizeDomainByCode(String domainCode) {
//        List<Domain> domains = domainDao.forCodeEquals(domainCode).findAll();
//        // Si on en trouve pour lequel c'est pas nécessaire d'anonymiser, alors on anonymise pas
//        boolean result = Iterables.any(domains, new Predicate<Domain>() {
//            @Override
//            public boolean apply(Domain input) {
//                return !shouldAnonymizeDomain(input.getTopiaId());
//            }
//        });
//        return result;
//    }

    @Override
    public boolean shouldAnonymizeGrowingPlan(String growingPlanId) {
        boolean result = shouldAnonymizeGrowingPlan(growingPlanId, false).getLeft();
        return result;
    }

    public  Pair<Boolean, Boolean> shouldAnonymizeGrowingPlan(String growingPlanId, boolean allowUnreadable) {

        // TODO AThimel 26/02/14 Introduce cache
        boolean canRead = true;
        boolean result = false;

        if (!isAdmin()) {
            String userId = getUserId();

            int highestPermissionLevel = getHighestPermissionLevel(userId, PermissionObjectType.GROWING_PLAN_ID, growingPlanId);
            if (highestPermissionLevel == 0) {
                GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
                highestPermissionLevel = getHighestPermissionLevel(userId, PermissionObjectType.GROWING_PLAN_CODE, growingPlan.getCode());
            }

            Preconditions.checkState(allowUnreadable || highestPermissionLevel > 0);
            result = highestPermissionLevel < SecurityHelper.PERMISSION_READ_RAW;
            canRead = highestPermissionLevel >= SecurityHelper.PERMISSION_READ_VALIDATED;
        }
        return Pair.of(result, canRead);
    }

//    @Override
//    public boolean shouldAnonymizeGrowingPlanByCode(String growingPlanCode) {
//        List<GrowingPlan> growingPlans = growingPlanDao.forCodeEquals(growingPlanCode).findAll();
//        // Si on en trouve pour lequel c'est pas nécessaire d'anonymiser, alors on anonymise pas
//        boolean result = Iterables.any(growingPlans, new Predicate<GrowingPlan>() {
//            @Override
//            public boolean apply(GrowingPlan input) {
//                return !shouldAnonymizeGrowingPlan(input.getTopiaId());
//            }
//        });
//        return result;
//    }

    @Override
    public void checkGrowingPlanReadable(String growingPlanId) throws AgrosystAccessDeniedException {

        boolean result = isGrowingPlanReadable(growingPlanId);

        if (!result) {
            // TODO AThimel 07/10/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean isGrowingPlanReadable(String growingPlanId) {
        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionReadable(userId, PermissionObjectType.GROWING_PLAN_ID, growingPlanId);

        if (!result) {
            GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
            result = hasObjectPermissionReadable(userId, PermissionObjectType.GROWING_PLAN_CODE, growingPlan.getCode());
        }
        return result;
    }

    @Override
    public boolean isGrowingSystemReadable(String growingSystemId) {
        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionReadable(userId, PermissionObjectType.GROWING_SYSTEM_ID, growingSystemId);

        if (!result) {
            GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
            result = hasObjectPermissionReadable(userId, PermissionObjectType.GROWING_SYSTEM_CODE, growingSystem.getCode());
        }

        return result;
    }

    @Override
    public void checkGrowingSystemReadable(String growingSystemId) throws AgrosystAccessDeniedException {
        if (!isGrowingSystemReadable(growingSystemId)) {
            // TODO AThimel 07/10/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkPracticedSystemReadable(String practicedSystemId) throws AgrosystAccessDeniedException {
        PracticedSystem practicedSystem = practicedSystemDao.forTopiaIdEquals(practicedSystemId).findUnique();
        String growingSystemId = practicedSystem.getGrowingSystem().getTopiaId(); // TODO AThimel 21/10/13 Optimize it
        checkGrowingSystemReadable(growingSystemId);
    }

    @Override
    public void checkDecisionRuleReadable(String decisionRuleId) throws AgrosystAccessDeniedException {
        DecisionRule decisionRule = decisionRuleDao.forTopiaIdEquals(decisionRuleId).findUnique();
        checkDomainCodeReadable(decisionRule.getDomainCode());
    }

    @Override
    public void checkManagementModeReadable(String managementModeId) throws AgrosystAccessDeniedException {
        ManagementMode managementMode = managementModeDao.forTopiaIdEquals(managementModeId).findUnique();
        String growingSystemId = managementMode.getGrowingSystem().getTopiaId(); // TODO AThimel 21/10/13 Optimize it
        checkGrowingSystemReadable(growingSystemId);
    }

    @Override
    public void checkEffectiveCropCyclesReadable(String zoneId) throws AgrosystAccessDeniedException {
        boolean result = isZoneReadable(zoneId);
        if (!result) {
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean isZoneReadable(String zoneId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();
        boolean result;
        if (zone.getPlot().getGrowingSystem() == null) {
            String domainId = zone.getPlot().getDomain().getTopiaId();
            result = isDomainReadable(domainId);
        } else {
            String growingSystemId = zone.getPlot().getGrowingSystem().getTopiaId();
            result = isGrowingSystemReadable(growingSystemId);
        }
        return result;
    }

    @Override
    public void checkCreateOrUpdateDomain(String domainId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(domainId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else if (!isDomainWritable(domainId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateGrowingPlan(String growingPlanId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(growingPlanId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else if (!isGrowingPlanWritable(growingPlanId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateGrowingSystem(String growingSystemId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(growingSystemId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else if (!isGrowingSystemWritable(growingSystemId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateNetwork(String networkId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(networkId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else if (!isNetworkWritable(networkId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdatePracticedSystem(String practicedSystemTopiaId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(practicedSystemTopiaId)) {
            // TODO AThimel 21/10/13 create authorisation check
        } else if (!isPracticedSystemWritable(practicedSystemTopiaId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateDecisionRule(String decisionRuleId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(decisionRuleId)) {
            // TODO AThimel 21/10/13 create authorisation check
        } else if (!isDecisionRuleWritable(decisionRuleId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateManagementMode(String managementModeId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(managementModeId)) {
            // TODO AThimel 21/10/13 create authorisation check
        } else if (!isManagementModeWritable(managementModeId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkCreateOrUpdateEffectiveCropCycles(String zoneId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(zoneId)) {
            // TODO AThimel 21/10/13 create authorisation check
        } else if (!isZoneWritable(zoneId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isDomainValidable(String domainId) {

        // TODO AThimel 29/11/13 Check validable not only writable
        return isDomainWritable(domainId);
//        String userId = getUserId();
//
//        boolean result = isAdmin() ||
//                hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_ID, domainId);
//
//        if (!result) {
//            Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
//            result = hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
//        }
//
//        return result;
    }

    @Override
    public void checkValidateDomain(String domainId) throws AgrosystAccessDeniedException {
        if (!isDomainValidable(domainId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isGrowingPlanValidable(String growingPlanId) {

        // TODO AThimel 29/11/13 Check validable not only writable
        return isGrowingPlanWritable(growingPlanId);
//        String userId = getUserId();
//
//        boolean result = isAdmin() ||
//                hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_ID, domainId);
//
//        if (!result) {
//            Domain domain = domainDao.forTopiaIdEquals(domainId).finfUnique;
//            result = hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
//        }
//
//        return result;
    }

    @Override
    public void checkValidateGrowingPlan(String growingPlanId) throws AgrosystAccessDeniedException {
        if (!isGrowingPlanWritable(growingPlanId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isGrowingSystemValidable(String growingSystemId) {

        // TODO AThimel 29/11/13 Check validable not only writable
        return isGrowingSystemWritable(growingSystemId);
//        String userId = getUserId();
//
//        boolean result = isAdmin() ||
//                hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_ID, domainId);
//
//        if (!result) {
//            Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique;
//            result = hasObjectPermissionWritable(userId, PermissionObjectType.DOMAIN_CODE, domain.getCode());
//        }
//
//        return result;
    }

    @Override
    public void checkValidateGrowingSystem(String growingSystemId) throws AgrosystAccessDeniedException {
        if (!isGrowingSystemWritable(growingSystemId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isPracticedSystemValidable(String practicedSystemId) {
        PracticedSystem practicedSystem = practicedSystemDao.forTopiaIdEquals(practicedSystemId).findUnique();
        String growingSystemId = practicedSystem.getGrowingSystem().getTopiaId();
        boolean result = isGrowingSystemWritable(growingSystemId); // TODO AThimel 25/02/14 Correct ?
        return result;
    }

    @Override
    public void checkValidatePracticedSystem(String practicedSystemId) throws AgrosystAccessDeniedException {
        if (!isPracticedSystemValidable(practicedSystemId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean areAttachmentsAddableOrDeletable(String objectReferenceId) {
        boolean result = canCreateOrDeleteAttachment(objectReferenceId);
        return result;
    }

    @Override
    public void checkAddAttachment(String objectReferenceId) {
        if (!canCreateOrDeleteAttachment(objectReferenceId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public void checkDeleteAttachment(String attachmentMetadataId) {
        AttachmentMetadata attachmentMetadata = attachmentMetadataDao.forTopiaIdEquals(attachmentMetadataId).findUnique();
        String objectReferenceId = attachmentMetadata.getObjectReferenceId();
        if (!canCreateOrDeleteAttachment(objectReferenceId)) {
            // TODO AThimel 11/12/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean canCreateOrDeleteAttachment(String referenceId) {

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionWritable(userId, referenceId);

        TopiaIdFactory topiaIdFactory = context.getPersistenceContext().getTopiaIdFactory();
        if (!result && topiaIdFactory.isTopiaId(referenceId) && ManagementMode.class.equals(topiaIdFactory.getClassName(referenceId))) {
            result = isManagementModeWritable(referenceId);
        }

        if (!result && topiaIdFactory.isTopiaId(referenceId) && Performance.class.equals(topiaIdFactory.getClassName(referenceId))) {
            result = isPerformanceWritable(referenceId);
        }

        if (!result) {
            if (referenceId.endsWith("-measurement")) {
                referenceId = referenceId.substring(0, referenceId.lastIndexOf("-measurement"));
            }
            if (topiaIdFactory.isTopiaId(referenceId) && Zone.class.equals(topiaIdFactory.getClassName(referenceId))) {
                result = isZoneWritable(referenceId);
            }
        }

        if (!result) {
            // User might not have permission on all campaigns (i.e code) but only a specific one (i.e topiaId)
            Iterable<DecisionRule> decisionRules = decisionRuleDao.forCodeEquals(referenceId).findAllLazy();
            Iterator<DecisionRule> iterator = decisionRules.iterator();
            while (!result && iterator.hasNext()) {
                DecisionRule decisionRule = iterator.next();
                result = isDecisionRuleWritable(decisionRule.getTopiaId());
            }
        }

        return result;
    }

    @Override
    public void checkReadAttachment(String attachmentMetadataId) {

        AttachmentMetadata attachmentMetadata = attachmentMetadataDao.forTopiaIdEquals(attachmentMetadataId).findUnique();
        String referenceId = attachmentMetadata.getObjectReferenceId();

        String userId = getUserId();

        boolean result = isAdmin() ||
                hasObjectPermissionReadable(userId, referenceId);

        TopiaIdFactory topiaIdFactory = context.getPersistenceContext().getTopiaIdFactory();
        if (!result && topiaIdFactory.isTopiaId(referenceId) && ManagementMode.class.equals(topiaIdFactory.getClassName(referenceId))) {
            result = isManagementModeReadable(referenceId);
        }

        if (!result && topiaIdFactory.isTopiaId(referenceId) && Performance.class.equals(topiaIdFactory.getClassName(referenceId))) {
            result = isPerformanceReadable(referenceId);
        }

        if (!result) {
            if (referenceId.endsWith("-measurement")) {
                referenceId = referenceId.substring(0, referenceId.lastIndexOf("-measurement"));
            }
            if (topiaIdFactory.isTopiaId(referenceId) && Zone.class.equals(topiaIdFactory.getClassName(referenceId))) {
                result = isZoneReadable(referenceId);
            }
        }

        if (!result) {
            // User might not have permission on all campaigns (i.e code) but only a specific one (i.e topiaId)
            Iterable<GrowingSystem> growingSystems = growingSystemDao.forCodeEquals(referenceId).findAllLazy();
            Iterator<GrowingSystem> iterator = growingSystems.iterator();
            while (!result && iterator.hasNext()) {
                GrowingSystem growingSystem = iterator.next();
                result = hasObjectPermissionReadable(userId, growingSystem.getTopiaId());
            }
        }

        if (!result) {
            // User might not have permission on all campaigns (i.e code) but only a specific one (i.e topiaId)
            Iterable<DecisionRule> decisionRules = decisionRuleDao.forCodeEquals(referenceId).findAllLazy();
            Iterator<DecisionRule> iterator = decisionRules.iterator();
            while (!result && iterator.hasNext()) {
                DecisionRule decisionRule = iterator.next();
                result = isDecisionRuleReadable(decisionRule.getTopiaId());
            }
        }

        // TODO AThimel 13/01/14 Maybe other entities needs this kind of consideration ?

        if (!result) {
            // TODO AThimel 11/12/13 Give some information to the created exception
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isPerformanceWritable(String performanceId) {

        boolean result = isAdmin();

        if (!result) {

            String userId = getUserId();
            AgrosystUser user = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

            long count = performanceDao.forProperties(
                    Performance.PROPERTY_TOPIA_ID, performanceId,
                    Performance.PROPERTY_AUTHOR, user).count();

            result = count == 1L;
        }

        return result;
    }



    @Override
    public void checkPerformanceReadable(String performanceId) throws AgrosystAccessDeniedException {
        if (!isPerformanceReadable(performanceId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    protected boolean isPerformanceReadable(String performanceId) {
        // Pour les performances, il n'y a pas de différence entre les droits en lecture et écriture : seul le créateur peut les lire/écrire
        return isPerformanceWritable(performanceId);
    }

    @Override
    public void checkCreateOrUpdatePerformance(String performanceId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(performanceId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else if (!isPerformanceWritable(performanceId)) {
            throw new AgrosystAccessDeniedException();
        }
    }

    @Override
    public boolean isPlotWritable(String plotId) {
        Plot plot = plotDao.forTopiaIdEquals(plotId).findUnique();
        String domainId = plot.getDomain().getTopiaId();
        boolean result = isDomainWritable(domainId);
        return result;
    }

    @Override
    public void checkPlotReadable(String plotId) throws AgrosystAccessDeniedException {
        Plot plot = plotDao.forTopiaIdEquals(plotId).findUnique();
        String domainId = plot.getDomain().getTopiaId();
        checkDomainReadable(domainId);
    }

    @Override
    public void checkCreateOrUpdatePlot(String plotId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(plotId)) {
            // TODO AThimel 18/10/13 create authorisation check
        } else {
            if (!isPlotWritable(plotId)) {
                throw new AgrosystAccessDeniedException();
            }
        }

    }


    @Override
    public void checkPracticedPlotReadable(String practicedPlotId) throws AgrosystAccessDeniedException {
        PracticedPlot practicedPlot = practicedPlotDao.forTopiaIdEquals(practicedPlotId).findUnique();
        String practicedSystemId = practicedPlot.getPracticedSystem().getTopiaId();
        checkPracticedSystemReadable(practicedSystemId);
    }

    @Override
    public boolean isPracticedPlotWritable(String practicedPlotId) {
        PracticedPlot practicedPlot = practicedPlotDao.forTopiaIdEquals(practicedPlotId).findUnique();
        String practicedSystemId = practicedPlot.getPracticedSystem().getTopiaId();
        boolean result = isPracticedSystemWritable(practicedSystemId);
        return result;

    }
    @Override
    public void checkCreateOrUpdatePracticedPlot(String practicedPlotId, String practicedSystemId) throws AgrosystAccessDeniedException {
        if (Strings.isNullOrEmpty(practicedPlotId)) {
            checkCreateOrUpdatePracticedSystem(practicedSystemId);
        } else {
            if (!isPracticedPlotWritable(practicedPlotId)) {
                throw new AgrosystAccessDeniedException();
            }
        }
    }

}
