package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AuthorizationServiceImpl.java 5028 2015-07-10 06:33:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/AuthorizationServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.ComputedUserPermission;
import fr.inra.agrosyst.api.entities.security.ComputedUserPermissionTopiaDao;
import fr.inra.agrosyst.api.entities.security.PermissionObjectType;
import fr.inra.agrosyst.api.entities.security.RoleType;
import fr.inra.agrosyst.api.entities.security.UserRole;
import fr.inra.agrosyst.api.entities.security.UserRoleTopiaDao;
import fr.inra.agrosyst.api.services.AgrosystFilter;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.network.NetworkFilter;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.security.UsersRolesAndDependencies;
import fr.inra.agrosyst.api.services.referential.ImportResult;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.security.AuthorizationService;
import fr.inra.agrosyst.api.services.security.InvalidTokenException;
import fr.inra.agrosyst.api.services.security.UserRoleDto;
import fr.inra.agrosyst.api.services.security.UserRoleEntityDto;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.Users;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.CacheService;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AuthorizationServiceImpl extends AbstractAgrosystService implements AuthorizationService {

    private static final Log log = LogFactory.getLog(AuthorizationServiceImpl.class);

    public static final Function<Domain, String> GET_DOMAIN_ENTITY_LABEL = new Function<Domain, String>() {
        @Override
        public String apply(Domain domain) {
            if (domain == null) {
                return "n/c";
            }
            return String.format("%s (%s)", domain.getName(), domain.getLocation().getCodePostal());
        }
    };

    public static final Function<GrowingPlan, String> GET_GROWING_PLAN_ENTITY_LABEL = new Function<GrowingPlan, String>() {
        @Override
        public String apply(GrowingPlan growingPlan) {
            if (growingPlan == null) {
                return "n/c";
            }
            return String.format("%s (%s)", growingPlan.getName(), growingPlan.getDomain().getName());
        }
    };

    public static final Function<GrowingSystem, String> GET_GROWING_SYSTEM_ENTITY_LABEL = new Function<GrowingSystem, String>() {
        @Override
        public String apply(GrowingSystem growingSystem) {
            if (growingSystem == null) {
                return "n/c";
            }
            return String.format("%s (%s)", growingSystem.getName(), growingSystem.getGrowingPlan().getDomain().getName());
        }
    };

    protected AuthenticationService authenticationService;
    protected GrowingSystemService growingSystemService;
    protected DomainService domainService;
    protected NetworkService networkService;
    protected GrowingPlanService growingPlanService;
    protected TrackerServiceImpl trackerService;
    protected CacheService cacheService;

    protected UserRoleTopiaDao userRoleDao;
    protected AgrosystUserTopiaDao agrosystUserDao;
    protected ComputedUserPermissionTopiaDao computedUserPermissionDao;
    protected NetworkTopiaDao networkDao;
    protected DomainTopiaDao domainDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected GrowingSystemTopiaDao growingSystemDao;


    public void setAuthenticationService(AuthenticationService authenticationService) {
        this.authenticationService = authenticationService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setNetworkService(NetworkService networkService) {
        this.networkService = networkService;
    }

    public void setGrowingPlanService(GrowingPlanService growingPlanService) {
        this.growingPlanService = growingPlanService;
    }

    public void setTrackerService(TrackerServiceImpl trackerService) {
        this.trackerService = trackerService;
    }

    public void setUserRoleDao(UserRoleTopiaDao userRoleDao) {
        this.userRoleDao = userRoleDao;
    }

    public void setAgrosystUserDao(AgrosystUserTopiaDao agrosystUserDao) {
        this.agrosystUserDao = agrosystUserDao;
    }

    public void setComputedUserPermissionDao(ComputedUserPermissionTopiaDao computedUserPermissionDao) {
        this.computedUserPermissionDao = computedUserPermissionDao;
    }

    public void setNetworkDao(NetworkTopiaDao networkDao) {
        this.networkDao = networkDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    public void setGrowingPlanDao(GrowingPlanTopiaDao growingPlanDao) {
        this.growingPlanDao = growingPlanDao;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    protected String getUserId() {
        String token = getSecurityContext().getAuthenticationToken();
        String userId = checkComputedPermissions(token);
        return userId;
    }

    protected String getUserIdOrFail(String authenticationToken) {

        String result = authenticationService.getAuthenticatedUserId(authenticationToken);

        if (Strings.isNullOrEmpty(result)) {
            throw new InvalidTokenException("The given authentication token is invalid, no user associated with it");
        }

        return result;
    }

    protected boolean hasRole(final String authenticationToken, final RoleType roleType) {

        Callable<Boolean> loader = new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                String userId = getUserIdOrFail(authenticationToken);

                UserRole role = userRoleDao.findRole(userId, roleType);
                boolean result = role != null;
                return result;
            }
        };

        boolean result = cacheService.get(CacheService.CACHE_HAS_ROLE, authenticationToken + "_" + roleType, loader);
        return result;
    }

    public boolean isAdmin() {
        String token = getSecurityContext().getAuthenticationToken();
        boolean result = isAdmin(token);
        return result;
    }

    @Override
    public boolean isAdmin(String authenticationToken) {
        boolean result = hasRole(authenticationToken, RoleType.ADMIN);
        return result;
    }

    public boolean isIsDataProcessor() {
        String authenticationToken = getSecurityContext().getAuthenticationToken();
        boolean result = isIsDataProcessor(authenticationToken);
        return result;
    }

    @Override
    public boolean isIsDataProcessor(String authenticationToken) {
        boolean result = hasRole(authenticationToken, RoleType.IS_DATA_PROCESSOR);
        return result;
    }

    @Override
    public boolean canCreateUser(String authenticationToken) {
        String userId = getUserIdOrFail(authenticationToken);

        boolean result = hasRole(authenticationToken, RoleType.ADMIN);
        if (!result) {
            UserRole networkManagerRole = userRoleDao.findNetworkResponsibleRole(userId);
            result = networkManagerRole != null;
        }
        return result;
    }

    public void dropComputedPermissions(String authenticationToken) {
        String userId = getUserIdOrFail(authenticationToken);

        dropComputedPermissions0(userId);
        getTransaction().commit();
    }

    protected void dropComputedPermissions0(String userId) {
        List<ComputedUserPermission> permissions = computedUserPermissionDao.forUserIdEquals(userId).findAll();
        if (permissions != null) {
            if (log.isDebugEnabled()) {
                log.debug(String.format("%d permission(s) are about to be deleted for user '%s'", permissions.size(), userId));
            }
            computedUserPermissionDao.deleteAll(permissions);
        }
    }

    public String checkComputedPermissions(String authenticationToken, String doAsUserId) {
        if (!Strings.isNullOrEmpty(doAsUserId) && isAdmin(authenticationToken)) {
            return checkComputedPermissionFromUserId(doAsUserId);
        } else {
            return checkComputedPermissions(authenticationToken);
        }
    }

    public String checkComputedPermissions(String authenticationToken) {
        String userId = getUserIdOrFail(authenticationToken);

        return checkComputedPermissionFromUserId(userId);
    }

    protected String checkComputedPermissionFromUserId(String userId) {
        long start = System.currentTimeMillis();

        // permissions has to be reloaded if there is no permission or if one of them is dirty
        long usersPermissionCount = computedUserPermissionDao.getUsersPermissionCount(userId);
        boolean hasDeletion = false;
        int newPermissionsCount = 0;
        if (usersPermissionCount == 0L || computedUserPermissionDao.getUsersDirtyCount(userId) >= 1L) {

            if (usersPermissionCount >= 1L) {
                dropComputedPermissions0(userId);
                hasDeletion = true;
            }

            Collection<ComputedUserPermission> permissions = computeUserPermissions(userId);
            newPermissionsCount = permissions.size();
            for (ComputedUserPermission permission : permissions) {
                computedUserPermissionDao.create(permission);
            }
        }

        // commit only if some permissions has been deleted or created
        if (hasDeletion || newPermissionsCount > 0) {
            getTransaction().commit();
        }

        if (log.isTraceEnabled()) {
            long duration = System.currentTimeMillis() - start;
            String text = "User's permissions computation took %dms. %d new permission(s) created.";
            String message = String.format(text, duration, newPermissionsCount);
            log.trace(message);
        }

        return userId;
    }

    protected Function<UserRole, UserRoleDto> getUserRoleToDtoFunction(final boolean includeEntity, final boolean includeUser) {
        Function<UserRole, UserRoleDto> function = new Function<UserRole, UserRoleDto>() {

            @Override
            public UserRoleDto apply(UserRole input) {
                UserRoleDto result = new UserRoleDto();
                result.setTopiaId(input.getTopiaId());
                result.setType(input.getType());

                if (includeEntity) {
                    if (!RoleType.IS_DATA_PROCESSOR.equals(input.getType()) && !RoleType.ADMIN.equals(input.getType())) {
                        UserRoleEntityDto entityDto = new UserRoleEntityDto();
                        result.setEntity(entityDto);

                        String identifier;
                        String label;
                        Integer campaign = null;

                        switch (input.getType()) {
                            case NETWORK_RESPONSIBLE:
                                identifier = input.getNetworkId();
                                label = networkDao.forTopiaIdEquals(identifier).findUnique().getName();
                                break;
                            case DOMAIN_RESPONSIBLE:
                                identifier = input.getDomainCode();
                                Domain domain = domainDao.forCodeEquals(identifier).findAny();
                                label = GET_DOMAIN_ENTITY_LABEL.apply(domain);
                                break;
                            case GROWING_PLAN_RESPONSIBLE:
                                identifier = input.getGrowingPlanCode();
                                GrowingPlan growingPlan = growingPlanDao.forCodeEquals(identifier).findAny();
                                label = GET_GROWING_PLAN_ENTITY_LABEL.apply(growingPlan);
                                break;
                            case GS_DATA_PROCESSOR:
                                GrowingSystem growingSystem;
                                if (!Strings.isNullOrEmpty(input.getGrowingSystemId())) {
                                    growingSystem = growingSystemDao.forTopiaIdEquals(input.getGrowingSystemId()).findUnique();
                                    campaign = growingSystem.getGrowingPlan().getDomain().getCampaign();
                                } else {
                                    growingSystem = growingSystemDao.forCodeEquals(input.getGrowingSystemCode()).findAny();
                                }
                                identifier = growingSystem.getCode();
                                label = GET_GROWING_SYSTEM_ENTITY_LABEL.apply(growingSystem);
                                break;
                            default:
                                throw new UnsupportedOperationException("Not supposed to happen");
                        }

                        entityDto.setIdentifier(identifier);
                        entityDto.setLabel(label);
                        entityDto.setCampaign(campaign);
                    }
                } else if (RoleType.GS_DATA_PROCESSOR.equals(input.getType())) {
                    // C'est un cas particulier : On ne demande pas l'entité mais il faut malgré tout indiquer la campagne
                    if (!Strings.isNullOrEmpty(input.getGrowingSystemId())) {
                        UserRoleEntityDto entityDto = new UserRoleEntityDto();
                        GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(input.getGrowingSystemId()).findUnique();
                        Integer campaign = growingSystem.getGrowingPlan().getDomain().getCampaign();
                        entityDto.setCampaign(campaign);
                        result.setEntity(entityDto);
                    }
                }

                if (includeUser) {
                    AgrosystUser user = input.getAgrosystUser();
                    UserDto userDto = Users.TO_USER_DTO.apply(user);
                    result.setUser(userDto);
                }

                return result;

            }

        };

        return function;
    }

    @Override
    public List<UserRoleDto> getUserRoles(String userId) {
        List<UserRoleDto> result = Lists.newLinkedList();

        List<UserRole> roles = userRoleDao.findAllForUserId(userId);
        Iterables.addAll(result, Iterables.transform(roles, getUserRoleToDtoFunction(true, false)));

        return result;
    }

    @Override
    public List<UserRoleDto> getEntityRoles(RoleType roleType, String entityCode) {
        List<UserRoleDto> result = Lists.newLinkedList();

        List<UserRole> roles = getUserRoles0(roleType, entityCode);

        Iterables.addAll(result, Iterables.transform(roles, getUserRoleToDtoFunction(false, true)));
        return result;
    }

    protected List<UserRole> getUserRoles0(RoleType roleType, String entityCode) {
        Map<String, Object> properties = Maps.newHashMap();
        properties.put(UserRole.PROPERTY_TYPE, roleType);
        switch (roleType) {
            case DOMAIN_RESPONSIBLE:
                properties.put(UserRole.PROPERTY_DOMAIN_CODE, entityCode);
                break;
            case GROWING_PLAN_RESPONSIBLE:
                properties.put(UserRole.PROPERTY_GROWING_PLAN_CODE, entityCode);
                break;
            case GS_DATA_PROCESSOR:
                properties.put(UserRole.PROPERTY_GROWING_SYSTEM_CODE, entityCode);
                break;
            default:
                throw new UnsupportedOperationException("Unexcpected role type: " + roleType);
        }

        List<UserRole> roles = userRoleDao.forProperties(properties).findAll();

        // Cas particulier des exploitants SdC où il se peut que seule la propriété growingSystemId soir valuée
        if (RoleType.GS_DATA_PROCESSOR.equals(roleType)) {
            // On recherche tous les SdC avec le code et pour chacun d'entre eux, on prend les rôles associés
            List<String> relatedGrowingSystemIds = growingSystemDao.forCodeEquals(entityCode).findAllIds();
            List<UserRole> gsCampaignRoles = userRoleDao.newQueryBuilder().
                    addEquals(UserRole.PROPERTY_TYPE, RoleType.GS_DATA_PROCESSOR).
                    addIn(UserRole.PROPERTY_GROWING_SYSTEM_ID, relatedGrowingSystemIds).findAll();
            Iterables.addAll(roles, gsCampaignRoles);
        }
        return roles;
    }

    @Override
    public List<UserRoleEntityDto> searchPossibleEntities(RoleType roleType, String termRaw) {
        String term = Strings.emptyToNull(termRaw);
        // TODO AThimel 03/10/13 Add security filtering according to user permissions
        List<UserRoleEntityDto> dtos = Lists.newLinkedList();
        switch (roleType) {
            case ADMIN:
            case IS_DATA_PROCESSOR:
                break;
            case DOMAIN_RESPONSIBLE:
                DomainFilter domainFilter = new DomainFilter();
                domainFilter.setDomainName(term);
                domainFilter.setPageSize(AgrosystFilter.ALL_PAGE_SIZE);
                ResultList<Domain> filteredDomains = domainService.getFilteredDomains(domainFilter);
                Iterables.addAll(dtos, Iterables.transform(filteredDomains, new Function<Domain, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(Domain input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getCode());
                        result.setLabel(GET_DOMAIN_ENTITY_LABEL.apply(input));
                        return result;
                    }
                }));
                break;
            case NETWORK_RESPONSIBLE:
                NetworkFilter networkFilter = new NetworkFilter();
                networkFilter.setNetworkName(term);
                networkFilter.setPageSize(AgrosystFilter.ALL_PAGE_SIZE);
                ResultList<Network> filteredNetworks = networkService.getFilteredNetworks(networkFilter);
                Iterables.addAll(dtos, Iterables.transform(filteredNetworks, new Function<Network, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(Network input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getTopiaId());
                        result.setLabel(input.getName());
                        return result;
                    }
                }));
                break;
            case GROWING_PLAN_RESPONSIBLE:
                GrowingPlanFilter growingPlanFilter = new GrowingPlanFilter();
                growingPlanFilter.setGrowingPlanName(term);
                growingPlanFilter.setPageSize(AgrosystFilter.ALL_PAGE_SIZE);
                ResultList<GrowingPlan> filteredGrowingPlans = growingPlanService.getFilteredGrowingPlans(growingPlanFilter);
                Iterables.addAll(dtos, Iterables.transform(filteredGrowingPlans, new Function<GrowingPlan, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(GrowingPlan input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getCode());
                        result.setLabel(GET_GROWING_PLAN_ENTITY_LABEL.apply(input));
                        return result;
                    }
                }));
                break;
            case GS_DATA_PROCESSOR:
                GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
                growingSystemFilter.setGrowingSystemName(term);
                growingSystemFilter.setPageSize(AgrosystFilter.ALL_PAGE_SIZE);
                ResultList<GrowingSystem> filteredGrowingSystems = growingSystemService.getFilteredGrowingSystems(growingSystemFilter);
                Iterables.addAll(dtos, Iterables.transform(filteredGrowingSystems, new Function<GrowingSystem, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(GrowingSystem input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getCode());
                        result.setLabel(GET_GROWING_SYSTEM_ENTITY_LABEL.apply(input));
                        return result;
                    }
                }));
                break;
            default:
                throw new UnsupportedOperationException("Unexpected type: " + roleType);
        }

        Multimap<String, UserRoleEntityDto> index = Multimaps.index(dtos, new Function<UserRoleEntityDto, String>() {
            @Override
            public String apply(UserRoleEntityDto input) {
                return input.getIdentifier();
            }
        });

        Set<String> keySet = index.keySet();
        List<UserRoleEntityDto> result = Lists.newArrayListWithCapacity(keySet.size());

        for (String identifier : keySet) {
            Collection<UserRoleEntityDto> identifierEntities = index.get(identifier);
            UserRoleEntityDto entity = Iterables.getFirst(identifierEntities, null);
            result.add(entity);
        }
        return result;
    }

    @Override
    public List<UserRoleEntityDto> searchEntities(RoleType roleType, String termRaw, Integer campaign) {
        // TODO AThimel 03/10/13 Add security filtering according to user permissions
        List<UserRoleEntityDto> dtos = Lists.newLinkedList();
        switch (roleType) {
            case ADMIN:
            case IS_DATA_PROCESSOR:
                break;
            case DOMAIN_RESPONSIBLE:
                List<Domain> filteredDomains = domainService.getDomainWithName(termRaw);
                Iterables.addAll(dtos, Iterables.transform(filteredDomains, new Function<Domain, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(Domain input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getCode());
                        result.setLabel(GET_DOMAIN_ENTITY_LABEL.apply(input));
                        return result;
                    }
                }));
                break;
            case NETWORK_RESPONSIBLE:
                List<Network> filteredNetworks = networkService.getNteworkWithName(termRaw);
                Iterables.addAll(dtos, Iterables.transform(filteredNetworks, new Function<Network, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(Network input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getTopiaId());
                        result.setLabel(input.getName());
                        return result;
                    }
                }));
                break;
            case GROWING_PLAN_RESPONSIBLE:
                List<GrowingPlan> filteredGrowingPlans = growingPlanService.getGrowingPlanWithName(termRaw);
                Iterables.addAll(dtos, Iterables.transform(filteredGrowingPlans, new Function<GrowingPlan, UserRoleEntityDto>() {
                    @Override
                    public UserRoleEntityDto apply(GrowingPlan input) {
                        UserRoleEntityDto result = new UserRoleEntityDto();
                        result.setIdentifier(input.getCode());
                        result.setLabel(GET_GROWING_PLAN_ENTITY_LABEL.apply(input));
                        return result;
                    }
                }));
                break;
            case GS_DATA_PROCESSOR:
                List<GrowingSystem> filteredGrowingSystems = growingSystemService.getGrowingSystemWithNameAndCampaign(termRaw, campaign);
                // all growing systems must have same code.
                Map<String, GrowingSystem> orderedGrowingSystems = Maps.newHashMap();
                for(GrowingSystem growingSystem : filteredGrowingSystems) {
                    GrowingSystem gs = orderedGrowingSystems.get(growingSystem.getCode());
                    if (gs == null) {
                        orderedGrowingSystems.put(growingSystem.getCode(), growingSystem);
                    }
                }
                if (orderedGrowingSystems.size() == 1) {
                    Iterables.addAll(dtos, Iterables.transform(orderedGrowingSystems.values(), new Function<GrowingSystem, UserRoleEntityDto>() {
                        @Override
                        public UserRoleEntityDto apply(GrowingSystem input) {
                            UserRoleEntityDto result = new UserRoleEntityDto();
                            result.setIdentifier(input.getCode());
                            result.setLabel(GET_GROWING_SYSTEM_ENTITY_LABEL.apply(input));
                            return result;
                        }
                    }));
                }
                break;
            default:
                throw new UnsupportedOperationException("Unexpected type: " + roleType);
        }

        Multimap<String, UserRoleEntityDto> index = Multimaps.index(dtos, new Function<UserRoleEntityDto, String>() {
            @Override
            public String apply(UserRoleEntityDto input) {
                return input.getIdentifier();
            }
        });

        Set<String> keySet = index.keySet();
        List<UserRoleEntityDto> result = Lists.newArrayListWithCapacity(keySet.size());

        for (String identifier : keySet) {
            Collection<UserRoleEntityDto> identifierEntities = index.get(identifier);
            UserRoleEntityDto entity = Iterables.getFirst(identifierEntities, null);
            result.add(entity);
        }
        return result;
    }

    @Override
    public void saveUserRoles(String userId, List<UserRoleDto> userRoles) {
        createOrUpdateUserRoles(userId, userRoles, true);
    }

    @Override
    public void addUserRoles(String userId, List<UserRoleDto> userRoles) {
        createOrUpdateUserRoles(userId, userRoles, false);
    }

    protected void createOrUpdateUserRoles(String userId, List<UserRoleDto> userRoles, Boolean removePreviousRoles) {

        List<UserRole> roles = userRoleDao.findAllForUserId(userId);
        AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

        ImmutableMap<String, UserRole> rolesIndex = Maps.uniqueIndex(roles, Entities.GET_TOPIA_ID);
        Set<String> rolesIds = Sets.newHashSet(rolesIndex.keySet());

        doCreateOrUpdateUserRoles(userRoles, agrosystUser, rolesIndex, rolesIds);

        removePreviousRoles(removePreviousRoles, rolesIndex, rolesIds);

        dropComputedPermissions0(userId);

        getTransaction().commit();

        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }

    protected void removePreviousRoles(Boolean removePreviousRoles, ImmutableMap<String, UserRole> rolesIndex, Set<String> rolesIds) {
        if (removePreviousRoles) {
            for (String rolesId : rolesIds) {
                UserRole userRole = rolesIndex.get(rolesId);
                userRoleDao.delete(userRole);
                trackerService.roleRemoved(userRole);
            }
        }
    }

    protected void doCreateOrUpdateUserRoles(List<UserRoleDto> userRoles, AgrosystUser agrosystUser, ImmutableMap<String, UserRole> rolesIndex, Set<String> rolesIds) {
        if (userRoles != null) {
            for (UserRoleDto userRole : userRoles) {
                String roleId = userRole.getTopiaId();
                UserRole roleEntity;
                if (Strings.isNullOrEmpty(roleId)) {
                    roleEntity = userRoleDao.newInstance();
                    roleEntity.setAgrosystUser(agrosystUser);
                } else {
                    roleEntity = rolesIndex.get(roleId);
                    Preconditions.checkState(roleEntity.getAgrosystUser().equals(agrosystUser));
                    rolesIds.remove(roleId);
                }

                String identifier = null;
                if (userRole.getEntity() != null) {
                    identifier = userRole.getEntity().getIdentifier();
                }
                RoleType type = userRole.getType();
                roleEntity.setType(type);
                switch (type) {
                    case ADMIN:
                    case IS_DATA_PROCESSOR:
                        break;
                    case DOMAIN_RESPONSIBLE:
                        Preconditions.checkState(identifier != null);
                        roleEntity.setDomainCode(identifier);
                        break;
                    case NETWORK_RESPONSIBLE:
                        Preconditions.checkState(identifier != null);
                        roleEntity.setNetworkId(identifier);
                        break;
                    case GROWING_PLAN_RESPONSIBLE:
                        Preconditions.checkState(identifier != null);
                        roleEntity.setGrowingPlanCode(identifier);
                        break;
                    case GS_DATA_PROCESSOR:
                        Preconditions.checkState(identifier != null);
                        Integer campaign = userRole.getEntity().getCampaign();
                        if (campaign == null) {
                            roleEntity.setGrowingSystemCode(identifier);
                            roleEntity.setGrowingSystemId(null);
                        } else {
                            HashSet<Integer> set = Sets.newHashSet(campaign);
                            List<GrowingSystem> list = growingSystemDao.findAllByCodeAndCampaign(identifier, set);
                            GrowingSystem growingSystem = list.iterator().next();
                            roleEntity.setGrowingSystemCode(null);
                            roleEntity.setGrowingSystemId(growingSystem.getTopiaId());
                        }
                        break;
                    default:
                        throw new UnsupportedOperationException("Unexpected type: " + type);
                }

                if (Strings.isNullOrEmpty(roleId)) {
                    userRoleDao.create(roleEntity);
                    trackerService.roleAdded(roleEntity);
                } else {
                    userRoleDao.update(roleEntity);
                }
            }
        }
    }

    @Override
    public void saveEntityUserRoles(RoleType roleType, String entityCode, List<UserRoleDto> roleDtos) {

        List<UserRole> roles = getUserRoles0(roleType, entityCode);

        ImmutableMap<String, UserRole> rolesIndex = Maps.uniqueIndex(roles, Entities.GET_TOPIA_ID);
        Set<String> rolesIds = Sets.newHashSet(rolesIndex.keySet());

        for (UserRoleDto userRole : roleDtos) {
            String userId = userRole.getUser().getTopiaId();
            AgrosystUser user = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

            String roleId = userRole.getTopiaId();
            UserRole roleEntity;
            if (Strings.isNullOrEmpty(roleId)) {
                roleEntity = userRoleDao.newInstance();
            } else {
                roleEntity = rolesIndex.get(roleId);
                rolesIds.remove(roleId);
            }
            roleEntity.setAgrosystUser(user);

            RoleType type = userRole.getType();
            roleEntity.setType(type);
            switch (type) {
                case DOMAIN_RESPONSIBLE:
                    roleEntity.setDomainCode(entityCode);
                    break;
                case GROWING_PLAN_RESPONSIBLE:
                    roleEntity.setGrowingPlanCode(entityCode);
                    break;
                case GS_DATA_PROCESSOR:
                    Integer campaign = null;
                    if (userRole.getEntity() != null) { // Entity may be null if no campaign is specified
                        campaign = userRole.getEntity().getCampaign();
                    }
                    if (campaign == null) {
                        roleEntity.setGrowingSystemCode(entityCode);
                        roleEntity.setGrowingSystemId(null);
                    } else {
                        HashSet<Integer> set = Sets.newHashSet(campaign);
                        List<GrowingSystem> list = growingSystemDao.findAllByCodeAndCampaign(entityCode, set);
                        GrowingSystem growingSystem = list.iterator().next();
                        roleEntity.setGrowingSystemCode(null);
                        roleEntity.setGrowingSystemId(growingSystem.getTopiaId());
                    }
                    break;
                default:
                    throw new UnsupportedOperationException("Unexpected type: " + type);
            }
            if (Strings.isNullOrEmpty(roleId)) {
                userRoleDao.create(roleEntity);
                trackerService.roleAdded(roleEntity);
            } else {
                userRoleDao.update(roleEntity);
            }

            dropComputedPermissions0(userId);
        }

        for (String rolesId : rolesIds) {
            UserRole userRole = rolesIndex.get(rolesId);
            userRoleDao.delete(userRole);
            trackerService.roleRemoved(userRole);
        }

        getTransaction().commit();

        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }

    @Override
    public List<UserDto> getDomainResponsibles(String domainCode) {
        List<AgrosystUser> users = userRoleDao.findAllRoleUsers(
                RoleType.DOMAIN_RESPONSIBLE, UserRole.PROPERTY_DOMAIN_CODE, domainCode);
        List<UserDto> result = Lists.transform(users, Users.TO_USER_DTO);
        return result;
    }

    @Override
    public List<UserDto> getGrowingPlanResponsibles(String growingPlanCode) {
        List<AgrosystUser> users = userRoleDao.findAllRoleUsers(
                RoleType.GROWING_PLAN_RESPONSIBLE, UserRole.PROPERTY_GROWING_PLAN_CODE, growingPlanCode);
        List<UserDto> result = Lists.transform(users, Users.TO_USER_DTO);
        return result;
    }

    protected Set<ComputedUserPermission> computeUserPermissions(String userId) {
        Set<ComputedUserPermission> result = Sets.newLinkedHashSet();

        List<UserRole> roles = userRoleDao.findAllForUserId(userId);
        if (roles != null) {
            for (UserRole role : roles) {

                switch (role.getType()) {
                    case ADMIN:
                        // AThimel 26/09/13 No permission to generate
                        break;
                    case IS_DATA_PROCESSOR: {
                        List<ComputedUserPermission> permissions = computeIsDataProcessorPermissions(userId);
                        result.addAll(permissions);
                        break;
                    }
                    case DOMAIN_RESPONSIBLE: {
                        List<ComputedUserPermission> permissions = computeDomainResponsiblePermissions(userId, role);
                        result.addAll(permissions);
                        break;
                    }
                    case GROWING_PLAN_RESPONSIBLE: {
                        List<ComputedUserPermission> permissions = computeGrowingPlanResponsiblePermissions(userId, role);
                        result.addAll(permissions);
                        break;
                    }
                    case GS_DATA_PROCESSOR: {
                        List<ComputedUserPermission> permissions = computeGrowingSystemDataProcessorPermissions(userId, role);
                        result.addAll(permissions);
                        break;
                    }
                    case NETWORK_RESPONSIBLE: {
                        List<ComputedUserPermission> permissions = computeNetworkResponsiblePermissions(userId, role);
                        result.addAll(permissions);
                        break;
                    }
                    default:
                        throw new UnsupportedOperationException("Unexpected type: " + role);
                }

            }
        }
        return result;
    }

    protected List<ComputedUserPermission> computeIsDataProcessorPermissions(String userId) {

        List<ComputedUserPermission> result = Lists.newLinkedList();

        // TODO AThimel 05/12/13 Maybe it would be better to process a different way, with no permission generated and role managed on all queries ?

        Set<String> domains = domainDao.getAllDomainCodes();
        for (String domain : domains) {
            ComputedUserPermission permission = computedUserPermissionDao.newDomainReadValidatedPermission(userId, domain);
            result.add(permission);
        }
        Set<String> growingPlans = growingPlanDao.getAllGrowingPlanCodes();
        for (String growingPlan : growingPlans) {
            ComputedUserPermission permission = computedUserPermissionDao.newGrowingPlanReadValidatedPermission(userId, growingPlan);
            result.add(permission);
        }
        Set<String> growingSystems = growingSystemDao.getAllGrowingSystemCodes();
        for (String growingSystem : growingSystems) {
            ComputedUserPermission permission = computedUserPermissionDao.newGrowingSystemReadValidatedPermission(userId, growingSystem);
            result.add(permission);
        }

        return result;
    }


    // XXX AThimel 01/10/13 Use a separate class in order to do the following treatments ?

    protected List<ComputedUserPermission> computeGrowingSystemDataProcessorPermissions(String userId, UserRole role) {

        List<ComputedUserPermission> result = Lists.newLinkedList();

        String growingSystemCode = role.getGrowingSystemCode();
        String growingSystemId = role.getGrowingSystemId();
        Preconditions.checkState(!Strings.isNullOrEmpty(growingSystemCode) || !Strings.isNullOrEmpty(growingSystemId));

        {
            ComputedUserPermission permission;
            if (!Strings.isNullOrEmpty(growingSystemCode)) {
                permission = computedUserPermissionDao.newGrowingSystemReadValidatedPermission(userId, growingSystemCode);
            } else {
                permission = computedUserPermissionDao.newSpecificGrowingSystemReadValidatedPermission(userId, growingSystemId);
            }
            result.add(permission);
        }

        {
            ComputedUserPermission domainPermission;
            ComputedUserPermission growingPlanPermission;
            if (!Strings.isNullOrEmpty(growingSystemCode)) {
                GrowingSystem growingSystem = growingSystemDao.forCodeEquals(growingSystemCode).findAny();
                GrowingPlan growingPlan = growingSystem.getGrowingPlan();
                String growingPlanCode = growingPlan.getCode();
                growingPlanPermission = computedUserPermissionDao.newGrowingPlanReadValidatedPermission(userId, growingPlanCode);
                String domainCode = growingPlan.getDomain().getCode();
                domainPermission = computedUserPermissionDao.newDomainReadValidatedPermission(userId, domainCode);
            } else {
                GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
                GrowingPlan growingPlan = growingSystem.getGrowingPlan();
                String growingPlanId = growingPlan.getTopiaId();
                growingPlanPermission = computedUserPermissionDao.newSpecificGrowingPlanReadValidatedPermission(userId, growingPlanId);
                String domainId = growingPlan.getDomain().getTopiaId();
                domainPermission = computedUserPermissionDao.newSpecificDomainReadValidatedPermission(userId, domainId);
            }
            result.add(growingPlanPermission);
            result.add(domainPermission);
        }

        return result;
    }

    protected List<ComputedUserPermission> computeGrowingPlanResponsiblePermissions(String userId, UserRole role) {

        List<ComputedUserPermission> result = Lists.newLinkedList();

        String growingPlanCode = role.getGrowingPlanCode();
        Preconditions.checkState(!Strings.isNullOrEmpty(growingPlanCode));
        final Set<String> growingPlanCodes = ImmutableSet.of(growingPlanCode);

        {
            ComputedUserPermission permission = computedUserPermissionDao.newGrowingPlanAdminPermission(userId, growingPlanCode);
            result.add(permission);
        }

        {
            GrowingPlan growingPlan = growingPlanDao.forCodeEquals(growingPlanCode).findAny();
            String domainCode = growingPlan.getDomain().getCode();
            ComputedUserPermission permission = computedUserPermissionDao.newDomainReadPermission(userId, domainCode);
            result.add(permission);
        }

        Callable<LinkedHashSet<String>> gp2gsLoader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                LinkedHashSet<String> result = computedUserPermissionDao.getProjectionHelper().growingPlansToGrowingSystemsCode(growingPlanCodes);
                return result;
            }
        };
        LinkedHashSet<String> growingSystemCodes = cacheService.get("growingPlansToGrowingSystemsCode", growingPlanCodes, gp2gsLoader);
        if (growingSystemCodes != null) {
            for (String growingSystemCode : growingSystemCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newGrowingSystemAdminPermission(userId, growingSystemCode);
                result.add(permission);
            }
        }

        return result;
    }

    protected List<ComputedUserPermission> computeDomainResponsiblePermissions(String userId, UserRole role) {

        List<ComputedUserPermission> result = Lists.newLinkedList();

        String domainCode = role.getDomainCode();
        Preconditions.checkState(!Strings.isNullOrEmpty(domainCode));
        final Set<String> domainCodes = ImmutableSet.of(domainCode);

        {
            ComputedUserPermission permission = computedUserPermissionDao.newDomainAdminPermission(userId, domainCode);
            result.add(permission);
        }

        Callable<LinkedHashSet<String>> d2gpLoader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                LinkedHashSet<String> result = growingPlanDao.domainsToGrowingPlansCode(domainCodes);
                return result;
            }
        };
        Set<String> growingPlanCodes = cacheService.get("domainToGrowingPlanCodes", domainCodes, d2gpLoader);
        if (growingPlanCodes != null) {
            for (String growingPlanCode : growingPlanCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newGrowingPlanReadPermission(userId, growingPlanCode);
                result.add(permission);
            }
        }

        Callable<LinkedHashSet<String>> d2gsLoader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                LinkedHashSet<String> result = growingPlanDao.getProjectionHelper().domainsToGrowingSystemsCode(domainCodes);
                return result;
            }
        };
        LinkedHashSet<String> growingSystemCodes = cacheService.get("domainToGrowingSystemCodes", domainCodes, d2gsLoader);
        if (growingSystemCodes != null) {
            for (String growingSystemCode : growingSystemCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newGrowingSystemReadPermission(userId, growingSystemCode);
                result.add(permission);
            }
        }

        return result;
    }

    protected List<ComputedUserPermission> computeNetworkResponsiblePermissions(String userId, UserRole role) {

        List<ComputedUserPermission> result = Lists.newLinkedList();

        // Permission on networks
        String networkId = role.getNetworkId();
        Preconditions.checkState(!Strings.isNullOrEmpty(networkId));
        Set<String> networkIdSet = ImmutableSet.of(networkId);

        {
            ComputedUserPermission permission = computedUserPermissionDao.newNetworkAdminPermission(userId, networkId);
            result.add(permission);
        }

        // Permissions on domains
        Set<String> domainCodes = computedUserPermissionDao.getProjectionHelper().networksToDomainsCode(networkIdSet);
        if (domainCodes != null) {
            for (String domainCode : domainCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newDomainReadPermission(userId, domainCode);
                result.add(permission);
            }
        }

        // Permissions on growingPlans
        Set<String> growingPlanCodes = computedUserPermissionDao.getProjectionHelper().networksToGrowingPlansCode(networkIdSet);
        if (growingPlanCodes != null) {
            for (String growingPlanCode : growingPlanCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newGrowingPlanReadPermission(userId, growingPlanCode);
                result.add(permission);
            }
        }

        // Permissions on growingSystems
        Set<String> growingSystemCodes = computedUserPermissionDao.getProjectionHelper().networksToGrowingSystemsCode(networkIdSet);
        if (growingSystemCodes != null) {
            for (String growingSystemCode : growingSystemCodes) {
                ComputedUserPermission permission = computedUserPermissionDao.newGrowingSystemReadPermission(userId, growingSystemCode);
                result.add(permission);
            }
        }

        return result;
    }

    protected void markPermissionsAsDirty(ComputedUserPermissionTopiaDao cupDAO, Iterable<ComputedUserPermission> toDirty) {
        for (ComputedUserPermission permission : toDirty) {
            permission.setDirty(true);
            cupDAO.update(permission);
        }
    }

    protected void objectsAreDirty(PermissionObjectType type) {
        List<ComputedUserPermission> toDirty = computedUserPermissionDao.forProperties(
                ComputedUserPermission.PROPERTY_TYPE, type,
                ComputedUserPermission.PROPERTY_DIRTY, false).findAll();

        markPermissionsAsDirty(computedUserPermissionDao, toDirty);
    }

    protected void objectIsDirty(PermissionObjectType type, String object) {

        List<ComputedUserPermission> toDirty = computedUserPermissionDao.forProperties(
                ComputedUserPermission.PROPERTY_TYPE, type,
                ComputedUserPermission.PROPERTY_OBJECT, object,
                ComputedUserPermission.PROPERTY_DIRTY, false).findAll();

        markPermissionsAsDirty(computedUserPermissionDao, toDirty);

    }

    @Override
    public ImportResult importRoles(InputStream rolesFileStream) {

        Import<ImportUserRoleDto> importer = null;
        ImportResult result = new ImportResult();
        try {
            UserRolesImportModel model = new UserRolesImportModel();
            importer = Import.newImport(model, rolesFileStream);
            Map<String, List<UserRoleDto>> userRolesDtoMap = Maps.newHashMap();

            for (ImportUserRoleDto entity : importer) {

                String email = entity.getUserEmail();
                Preconditions.checkArgument(!Strings.isNullOrEmpty(email));
                // make sure email is in lower case
                email = entity.getUserEmail().toLowerCase();

                AgrosystUser agrosystUser = agrosystUserDao.forEmailEquals(email).findAnyOrNull();
                if (agrosystUser == null) {
                    // as requested, do nothing for existing accounts
                    result.incIgnored();
                } else if (entity.getType() == RoleType.ADMIN || entity.getType() == RoleType.IS_DATA_PROCESSOR){

                    List<UserRoleDto> userRoleDtos = userRolesDtoMap.get(agrosystUser.getTopiaId());
                    if (userRoleDtos == null) {
                        userRoleDtos = Lists.newArrayList();
                        userRolesDtoMap.put(agrosystUser.getTopiaId(), userRoleDtos);
                    }
                    UserRoleDto userRoleDto = new UserRoleDto();
                    userRoleDto.setType(entity.getType());
                    userRoleDtos.add(userRoleDto);
                } else {

                    // targeted entity
                    String term = entity.getTargetedEntity();

                    List<UserRoleEntityDto> userRoleEntityDtos = Lists.newArrayList();
                    userRoleEntityDtos.addAll(searchEntities(entity.getType(), term, entity.getCampaign()));

                    if (userRoleEntityDtos.size() == 1) {
                        // valid there are no ambiguity for the targeted entity
                        List<UserRoleDto> userRoleDtos = userRolesDtoMap.get(agrosystUser.getTopiaId());
                        if (userRoleDtos == null) {
                            userRoleDtos = Lists.newArrayList();
                            userRolesDtoMap.put(agrosystUser.getTopiaId(), userRoleDtos);
                        }
                        UserRoleDto userRoleDto = new UserRoleDto();
                        userRoleDto.setType(entity.getType());
                        UserRoleEntityDto userRoleEntityDto = userRoleEntityDtos.get(0);
                        userRoleEntityDto.setCampaign(entity.getCampaign());
                        userRoleDto.setEntity(userRoleEntityDto);
                        userRoleDtos.add(userRoleDto);

                    } else if (userRoleEntityDtos.isEmpty()){
                        // no entity found
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find any entity for term " + term);
                        }
                        result.incIgnored();
                    } else {
                        // too many entities return to be able to have unique target
                        if (log.isWarnEnabled()) {
                            log.warn("Too many entities founds for term " + term);
                        }
                        result.incIgnored();
                    }
                }
            }
            for (Map.Entry<String, List<UserRoleDto>> entry : userRolesDtoMap.entrySet()) {
                String key = entry.getKey();
                List<UserRoleDto> userRoleDtos = entry.getValue();
                saveUserRoles(key, userRoleDtos);
                result.incCreated();
            }

            getTransaction().commit();
        } finally {
            IOUtils.closeQuietly(importer);
        }

        return result;
    }

    protected void importPz0UserRoles(String userId, List<UserRoleDto> userRoles) {

        List<UserRole> roles = userRoleDao.findAllForUserId(userId);
        AgrosystUser agrosystUser = agrosystUserDao.forTopiaIdEquals(userId).findUnique();

        ImmutableMap<String, UserRole> rolesIndex = Maps.uniqueIndex(roles, Entities.GET_TOPIA_ID);
        Set<String> rolesIds = Sets.newHashSet(rolesIndex.keySet());

        doCreateOrUpdateUserRoles(userRoles, agrosystUser, rolesIndex, rolesIds);

        dropComputedPermissions0(userId);
    }

    @Override
    public void importPz0UsersRoles(Map<Class, ImportResults> allResults) {

        ImportResults importResults = allResults.remove(UserRoleDto.class);
        if (importResults != null) {
            UsersRolesAndDependencies usersRolesAndDependencies = (UsersRolesAndDependencies) importResults.getEntityAndDepsByCsvIds().get("usersRolesImport");
            Map<String, List<UserRoleDto>> allUsersRoles = usersRolesAndDependencies.getUsersRoles();
            int count = 1;
            int total = allUsersRoles.entrySet().size();
            for (Map.Entry<String, List<UserRoleDto>> stringListEntry : allUsersRoles.entrySet()) {
                String userId = stringListEntry.getKey();
                long start = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info(String.format("Début sauvegarde du roles utilisateur %d/%d.", count++, total));
                }

                List<UserRoleDto> userRole = stringListEntry.getValue();
                importPz0UserRoles(userId, userRole);

                long p1 = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info(String.format("Fin de sauvegarde du roles utilisateur en %d:", (p1- start)));
                }
            }
        }
        cacheService.clear(CacheService.CACHE_HAS_ROLE);
    }
}
