package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AuthenticationServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/AuthenticationServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Map;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.entities.security.StoredToken;
import fr.inra.agrosyst.api.entities.security.StoredTokenTopiaDao;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.security.AuthorizationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.api.services.users.Users;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.users.UserServiceImpl;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class AuthenticationServiceImpl extends AbstractAgrosystService implements AuthenticationService {

    private static final Log log = LogFactory.getLog(AuthenticationServiceImpl.class);

    protected static final Map<String, String> USER_TOKENS = Maps.newConcurrentMap(); // TODO AThimel 24/09/13 replication

    protected AuthorizationService authorizationService;

    protected StoredTokenTopiaDao storedTokenDao;
    protected AgrosystUserTopiaDao agrosystUserDao;

    public void setAuthorizationService(AuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setStoredTokenDao(StoredTokenTopiaDao storedTokenDao) {
        this.storedTokenDao = storedTokenDao;
    }

    public void setAgrosystUserDao(AgrosystUserTopiaDao agrosystUserDao) {
        this.agrosystUserDao = agrosystUserDao;
    }

    private void saveToken(String token, String userId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(token));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(userId));
        if (getConfig().isTokenStorageEnabled()) {
            if (log.isDebugEnabled()) {
                log.debug("Token will be stored directly in database");
            }
            storedTokenDao.create(
                    StoredToken.PROPERTY_TOKEN, token,
                    StoredToken.PROPERTY_USER_ID, userId);
            getTransaction().commit();
        } else {
            USER_TOKENS.put(token, userId);
        }
    }

    private String getTokenUserId(String token) {
        String result = null;
        if (getConfig().isTokenStorageEnabled()) {
            StoredToken storedToken = storedTokenDao.forTokenEquals(token).findAnyOrNull();
            if (storedToken != null) {
                result = storedToken.getUserId();
            }
        } else {
            result = USER_TOKENS.get(token);
        }
        return result;
    }

    private void dropToken(String token) {
        if (getConfig().isTokenStorageEnabled()) {
            StoredToken storedToken = storedTokenDao.forTokenEquals(token).findAnyOrNull();
            if (storedToken != null) {
                storedTokenDao.delete(storedToken);
                getTransaction().commit();
            }
        } else {
            USER_TOKENS.remove(token);
        }
    }

    @Override
    public UserDto login(String email, String password) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(email), "email cannot be empty");

        if (log.isInfoEnabled()) {
            String message = String.format("Check user authentication with email: '%s'", email);
            log.info(message);
        }

        String encodedPassword = UserServiceImpl.hashPassword(password);
        AgrosystUser user = agrosystUserDao.forProperties(
                AgrosystUser.PROPERTY_EMAIL, email.toLowerCase(),
                AgrosystUser.PROPERTY_PASSWORD, encodedPassword,
                AgrosystUser.PROPERTY_ACTIVE, true).findAnyOrNull();

        UserDto dto = null;
        if (user == null) {
            if (log.isWarnEnabled()) {
                String message = String.format("Could not authenticate user with email: '%s' and the given password", email);
                log.warn(message);
            }
        } else {
            dto = Users.TO_USER_DTO.apply(user);
        }

        if (dto != null) {
            String authenticationToken = UUID.randomUUID().toString();
            dto.setAuthenticationToken(authenticationToken);
            saveToken(authenticationToken, dto.getTopiaId());
        }

        return dto;
    }

    @Override
    public void logout(String token) {
        if (!Strings.isNullOrEmpty(token)) {
            String userId = getTokenUserId(token);
            if (!Strings.isNullOrEmpty(userId)) {
                authorizationService.dropComputedPermissions(token);
            }
            dropToken(token);
        }
    }

    @Override
    public String getAuthenticatedUserId(String token) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(token), "token cannot be empty");
        String result = getTokenUserId(token);
        return result;
    }

    @Override
    public UserDto getAuthenticatedUser(String token) {
        String userId = getAuthenticatedUserId(token);
        UserDto result = null;
        if (userId != null) {
            AgrosystUser user = agrosystUserDao.forTopiaIdEquals(userId).findUnique();
            result = Users.TO_USER_DTO.apply(user);
        }
        return result;
    }

    @Override
    public long getConnectedUsersCount() {
        if (getConfig().isTokenStorageEnabled()) {
            return storedTokenDao.getConnectedUserCount(context.getCurrentDate());
        } else {
            return Integer.valueOf(USER_TOKENS.size()).longValue();
        }
    }
}
