package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AnonymizeServiceImpl.java 3847 2014-03-22 01:39:13Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/AnonymizeServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.hash.Hashing;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainImpl;
import fr.inra.agrosyst.api.entities.GeoPoint;
import fr.inra.agrosyst.api.entities.GeoPointImpl;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanImpl;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemImpl;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotImpl;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneImpl;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeImpl;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.performance.PerformanceImpl;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemImpl;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationImpl;
import fr.inra.agrosyst.api.entities.security.HashedValueTopiaDao;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.domain.PlotDto;
import fr.inra.agrosyst.api.services.domain.ZoneDto;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemDto;
import fr.inra.agrosyst.api.services.performance.PerformanceDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemDto;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 0.8
 */
public class AnonymizeServiceImpl extends AuthorizationServiceImpl implements AnonymizeService {

    protected static final String XXXXX = "xxxxx";
    protected static final int XXXXX_INT = -88888;
    protected static final double XXXXX_DOUBLE = -8888.8d;

    protected static final Function<RefLocation, RefLocation> ANONYMIZE_LOCATION = new Function<RefLocation, RefLocation>() {
        @Override
        public RefLocation apply(RefLocation input) {
            Binder<RefLocation, RefLocation> gpsDataBinder = BinderFactory.newBinder(RefLocation.class);
            RefLocation result = new RefLocationImpl();
            gpsDataBinder.copyExcluding(input, result,
                    RefLocation.PROPERTY_LATITUDE,
                    RefLocation.PROPERTY_LONGITUDE,
                    RefLocation.PROPERTY_CODE_INSEE,
                    RefLocation.PROPERTY_CODE_POSTAL,
                    RefLocation.PROPERTY_COMMUNE
            );

            result.setLongitude(XXXXX_DOUBLE);
            result.setLatitude(XXXXX_DOUBLE);
            result.setCodeInsee(XXXXX);
            result.setCodePostal(XXXXX);
            result.setCommune(XXXXX);
            return result;
        }
    };

    protected BusinessAuthorizationService authorizationService;

    protected HashedValueTopiaDao hashedValuesDao;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setHashedValuesDao(HashedValueTopiaDao hashedValuesDao) {
        this.hashedValuesDao = hashedValuesDao;
    }

    public AnonymizeContext newContext() {
        AnonymizeContext result = new AnonymizeContext(authorizationService, getAnonymizeFunction());
        return result;
    }

    public Function<String, String> getAnonymizeFunction() {
        return new Function<String, String>() {
            @Override
            public String apply(String input) {
                return anonymize(input);
            }
        };
    }

    @Override
    public String anonymize(String clear) {
        String userId = getUserId();
        Preconditions.checkState(!Strings.isNullOrEmpty(userId));

        String hashed = Hashing.crc32().hashString(clear, Charsets.UTF_8).toString();

        if (hashedValuesDao.checkValue(clear, hashed)) {
            getTransaction().commit();
        }

        return hashed;
    }

    protected Function<Domain, Domain> getAnonymizeDomainFunction() {
        Function<Domain, Domain> result = getAnonymizeDomainFunction(false);
        return result;
    }

    protected Function<Domain, Domain> getAnonymizeDomainFunction(final boolean allowUnreadable) {
        Function<Domain, Domain> result = new Function<Domain, Domain>() {
            @Override
            public Domain apply(Domain input) {

                if (input == null) {
                    return null;
                }
                Domain result;
                String domainId = input.getTopiaId();
                Pair<Boolean, Boolean> anonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId, allowUnreadable);
                Boolean shouldAnonymizeDomain = anonymizeDomain.getLeft();
                if (shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<Domain, Domain> binder = BinderFactory.newBinder(Domain.class);
                    result = new DomainImpl();
                    binder.copyExcluding(input, result,
                            Domain.PROPERTY_TOPIA_ID,
                            Domain.PROPERTY_NAME,
                            Domain.PROPERTY_MAIN_CONTACT,
                            Domain.PROPERTY_SIRET,
                            Domain.PROPERTY_LOCATION
                    );

                    result.setName(anonymize(input.getName()));
                    result.setMainContact(XXXXX);
                    result.setSiret(XXXXX);
                    result.setLocation(ANONYMIZE_LOCATION.apply(input.getLocation()));

                    Boolean canReadDomain = anonymizeDomain.getRight();
                    if (canReadDomain) {
                        result.setTopiaId(input.getTopiaId());
                    }
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public Function<Domain, DomainDto> getDomainToDtoFunction(final boolean includeResponsibles) {
        Function<Domain, DomainDto> result = newContext().
                includeDomainResponsibles(includeResponsibles).
                getDomainToDtoFunction();
        return result;
    }

    @Override
    public Domain checkForDomainAnonymization(Domain domain) {
        Domain result = getAnonymizeDomainFunction().apply(domain);
        return result;
    }

    @Override
    public Collection<Domain> checkForDomainsAnonymization(Collection<Domain> domains) {
        Collection<Domain> result = Collections2.transform(domains, getAnonymizeDomainFunction());
        return result;
    }

    @Override
    public ResultList<Domain> checkForDomainsAnonymization(ResultList<Domain> domains) {
        ResultList<Domain> result = ResultList.transformLazy(domains, getAnonymizeDomainFunction());
        return result;
    }

    @Override
    public Map<String, String> getDomainsAsMap(Iterable<Domain> domains) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (Domain domain : domains) {
            String domainId = domain.getTopiaId();
            if (authorizationService.shouldAnonymizeDomain(domainId)) {
                result.put(domainId, anonymize(domain.getName()));
            } else {
                result.put(domainId, domain.getName());
            }
        }
        return result;
    }

    @Override
    public Plot checkForPlotAnonymization(Plot plot) {
        Plot result = getAnonymizePlotFunction().apply(plot);
        return result;
    }

    @Override
    public List<Plot> checkForPlotsAnonymization(List<Plot> plots) {
        List<Plot> result = Lists.transform(plots, getAnonymizePlotFunction(true));
        return result;
    }

    protected Function<GeoPoint, GeoPoint> getAnonymizeGeoPointFunction() {
        Function<GeoPoint, GeoPoint> result = new Function<GeoPoint, GeoPoint>() {
            @Override
            public GeoPoint apply(GeoPoint input) {

                GeoPoint result = null;
                if (input != null) {
                    Domain domain = input.getDomain();
                    String domainId = domain.getTopiaId();
                    if (authorizationService.shouldAnonymizeDomain(domainId)) {
                        // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                        Binder<GeoPoint, GeoPoint> binder = BinderFactory.newBinder(GeoPoint.class);
                        result = new GeoPointImpl();
                        binder.copyExcluding(input, result,
                                GeoPoint.PROPERTY_LATITUDE,
                                GeoPoint.PROPERTY_LONGITUDE,
                                GeoPoint.PROPERTY_DOMAIN
                        );

                        result.setLatitude(XXXXX_DOUBLE);
                        result.setLongitude(XXXXX_DOUBLE);
                        result.setDomain(null); // Do not push domain value
                    } else {
                        // No need to modify instance
                        result = input;
                    }
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public List<GeoPoint> checkForGeoPointAnonymization(List<GeoPoint> geoPoints) {
        List<GeoPoint> result = Lists.transform(geoPoints, getAnonymizeGeoPointFunction());
        return result;
    }

    protected Function<Plot, Plot> getAnonymizePlotFunction() {
        Function<Plot, Plot> result = getAnonymizePlotFunction(false);
        return result;
    }

    protected Function<Plot, Plot> getAnonymizePlotFunction(final boolean allowUnreadable) {
        final Function<Domain, Domain> anonymizeDomainFunction = getAnonymizeDomainFunction(allowUnreadable);
        final Function<GrowingSystem, GrowingSystem> anonymizeGrowingSystemFunction = getAnonymizeGrowingSystemFunction(allowUnreadable);
        Function<Plot, Plot> result = new Function<Plot, Plot>() {
            @Override
            public Plot apply(Plot input) {

                Plot result = null;
                if (input != null) {
                    Domain domain = input.getDomain();
                    String domainId = domain.getTopiaId();
                    Pair<Boolean, Boolean> domainReadableStatus = authorizationService.shouldAnonymizeDomain(domainId, allowUnreadable);
                    boolean shouldAnonymizeDomain = domainReadableStatus.getLeft();

                    boolean shouldAnonymizeGrowingPlan = false;
                    boolean plotReadable = true;
                    if (input.getGrowingSystem() != null) {
                        String growingPlanId = input.getGrowingSystem().getGrowingPlan().getTopiaId();
                        Pair<Boolean, Boolean> gpReadableStatus = authorizationService.shouldAnonymizeGrowingPlan(growingPlanId, allowUnreadable);
                        shouldAnonymizeGrowingPlan = gpReadableStatus.getLeft();
                        plotReadable = authorizationService.isGrowingSystemReadable(input.getGrowingSystem().getTopiaId());
                    }
                    if (shouldAnonymizeDomain || shouldAnonymizeGrowingPlan) {
                        // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                        Binder<Plot, Plot> binder = BinderFactory.newBinder(Plot.class);
                        result = new PlotImpl();
                        binder.copyExcluding(input, result,
                                Plot.PROPERTY_TOPIA_ID,
                                Plot.PROPERTY_NAME,
                                Plot.PROPERTY_PAC_ILOT_NUMBER,
                                Plot.PROPERTY_LOCATION,
                                Plot.PROPERTY_LATITUDE,
                                Plot.PROPERTY_LONGITUDE,
                                Plot.PROPERTY_DOMAIN,
                                Plot.PROPERTY_GROWING_SYSTEM
                        );

                        if (shouldAnonymizeDomain) {
                            result.setName(anonymize(input.getName()));
                            result.setPacIlotNumber(XXXXX_INT);
                            result.setLocation(ANONYMIZE_LOCATION.apply(input.getLocation()));
                            result.setLatitude(XXXXX_DOUBLE);
                            result.setLongitude(XXXXX_DOUBLE);
                            result.setDomain(anonymizeDomainFunction.apply(domain));
                        } else {
                            result.setName(input.getName());
                            result.setPacIlotNumber(input.getPacIlotNumber());
                            result.setLocation(input.getLocation());
                            result.setLatitude(input.getLatitude());
                            result.setLongitude(input.getLongitude());
                            result.setDomain(domain);
                        }

                        if (shouldAnonymizeGrowingPlan) {
                            result.setGrowingSystem(anonymizeGrowingSystemFunction.apply(input.getGrowingSystem()));
                        } else {
                            result.setGrowingSystem(input.getGrowingSystem());
                        }

                        if (plotReadable) {
                            result.setTopiaId(input.getTopiaId());
                        }
                    } else {
                        // No need to modify instance
                        result = input;
                    }
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public Function<Plot, PlotDto> getPlotToDtoFunction() {
        Function<Plot, PlotDto> result = newContext().
                getPlotToDtoFunction();
        return result;
    }

    protected Function<Zone, Zone> getAnonymizeZoneFunction() {
        final Function<Plot, Plot> anonymizePlotFunction = getAnonymizePlotFunction();

        Function<Zone, Zone> result = new Function<Zone, Zone>() {
            @Override
            public Zone apply(Zone input) {

                if (input == null) {
                    return null;
                }

                Zone result;
                String domainId = input.getPlot().getDomain().getTopiaId();
                boolean shouldAnonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId);
                if (shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<Zone, Zone> binder = BinderFactory.newBinder(Zone.class);
                    result = new ZoneImpl();
                    binder.copyExcluding(input, result,
                            Zone.PROPERTY_PLOT
                    );

                    result.setPlot(anonymizePlotFunction.apply(input.getPlot()));
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public Zone checkForZoneAnonymization(Zone zone) {
        Zone result = getAnonymizeZoneFunction().apply(zone);
        return result;
    }

    @Override
    public Function<Zone, ZoneDto> getZoneToDtoFunction() {
        return getZoneToDtoFunction(true);
    }

    @Override
    public Function<Zone, ZoneDto> getZoneToDtoFunction(final boolean includePlot) {
        Function<Zone, ZoneDto> result = newContext().
                includeZonePlot(includePlot).
                getZoneToDtoFunction();
        return result;
    }

    protected Function<GrowingPlan, GrowingPlan> getAnonymizeGrowingPlanFunction() {
        Function<GrowingPlan, GrowingPlan> result = getAnonymizeGrowingPlanFunction(false);
        return result;
    }

    protected Function<GrowingPlan, GrowingPlan> getAnonymizeGrowingPlanFunction(final boolean allowUnreadable) {
        final Function<Domain, Domain> anonymizeDomainFunction = getAnonymizeDomainFunction(allowUnreadable);
        Function<GrowingPlan, GrowingPlan> result = new Function<GrowingPlan, GrowingPlan>() {
            @Override
            public GrowingPlan apply(GrowingPlan input) {

                if (input == null) {
                    return null;
                }
                GrowingPlan result;
                String growingPlanId = input.getTopiaId();
                String domainId = input.getDomain().getTopiaId();
                Pair<Boolean, Boolean> anonymizeGrowingPlan = authorizationService.shouldAnonymizeGrowingPlan(growingPlanId, allowUnreadable);
                boolean shouldAnonymizeGrowingPlan = anonymizeGrowingPlan.getLeft();
                boolean canReadGrowingPlan = anonymizeGrowingPlan.getRight();
                boolean shouldAnonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId, allowUnreadable).getLeft();
                if (shouldAnonymizeGrowingPlan || shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<GrowingPlan, GrowingPlan> binder = BinderFactory.newBinder(GrowingPlan.class);
                    result = new GrowingPlanImpl();
                    binder.copyExcluding(input, result,
                            GrowingPlan.PROPERTY_TOPIA_ID,
                            GrowingPlan.PROPERTY_NAME,
                            GrowingPlan.PROPERTY_DOMAIN
                    );

                    if (shouldAnonymizeGrowingPlan) {
                        result.setName(anonymize(input.getName()));
                    } else {
                        result.setName(input.getName());
                    }

                    if (canReadGrowingPlan) {
                        result.setTopiaId(input.getTopiaId());
                    }
                    result.setDomain(anonymizeDomainFunction.apply(input.getDomain()));
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public GrowingPlan checkForGrowingPlanAnonymization(GrowingPlan growingPlan) {
        GrowingPlan result = getAnonymizeGrowingPlanFunction().apply(growingPlan);
        return result;
    }

    @Override
    public Function<GrowingPlan, GrowingPlanDto> getGrowingPlanToDtoFunction(final boolean includeResponsibles) {
        Function<GrowingPlan, GrowingPlanDto> result = newContext().
                includeGrowingPlanResponsibles(includeResponsibles).
                getGrowingPlanToDtoFunction();
        return result;
    }

    @Override
    public Map<String, String> getGrowingPlansAsMap(Iterable<GrowingPlan> growingPlans) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (GrowingPlan growingPlan : growingPlans) {
            String growingPlanId = growingPlan.getTopiaId();
            if (authorizationService.shouldAnonymizeGrowingPlan(growingPlanId)) {
                result.put(growingPlanId, anonymize(growingPlan.getName()));
            } else {
                result.put(growingPlanId, growingPlan.getName());
            }
        }
        return result;
    }

    protected Function<GrowingSystem, GrowingSystem> getAnonymizeGrowingSystemFunction() {
        Function<GrowingSystem, GrowingSystem> result = getAnonymizeGrowingSystemFunction(false);
        return result;
    }

    protected Function<GrowingSystem, GrowingSystem> getAnonymizeGrowingSystemFunction(final boolean allowUnreadable) {
        final Function<GrowingPlan, GrowingPlan> anonymizeGrowingPlanFunction = getAnonymizeGrowingPlanFunction(allowUnreadable);
        Function<GrowingSystem, GrowingSystem> result = new Function<GrowingSystem, GrowingSystem>() {
            @Override
            public GrowingSystem apply(GrowingSystem input) {

                if (input == null) {
                    return null;
                }
                GrowingSystem result;
                String growingPlanId = input.getGrowingPlan().getTopiaId();
                String domainId = input.getGrowingPlan().getDomain().getTopiaId();
                boolean shouldAnonymizeGrowingPlan = authorizationService.shouldAnonymizeGrowingPlan(growingPlanId, allowUnreadable).getLeft();
                boolean shouldAnonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId, allowUnreadable).getLeft();
                if (shouldAnonymizeGrowingPlan || shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<GrowingSystem, GrowingSystem> binder = BinderFactory.newBinder(GrowingSystem.class);
                    result = new GrowingSystemImpl();
                    binder.copyExcluding(input, result,
                            GrowingSystem.PROPERTY_TOPIA_ID,
                            GrowingSystem.PROPERTY_GROWING_PLAN
                    );

                    if (authorizationService.isGrowingSystemReadable(input.getTopiaId())) {
                        result.setTopiaId(input.getTopiaId());
                    }

                    result.setGrowingPlan(anonymizeGrowingPlanFunction.apply(input.getGrowingPlan()));
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public GrowingSystem checkForGrowingSystemAnonymization(GrowingSystem growingSystem) {
        GrowingSystem result = getAnonymizeGrowingSystemFunction().apply(growingSystem);
        return result;
    }

    @Override
    public Function<GrowingSystem, GrowingSystemDto> getGrowingSystemToDtoFunction() {
        Function<GrowingSystem, GrowingSystemDto> result = newContext().
                getGrowingSystemToDtoFunction();
        return result;
    }

    protected Function<PracticedSystem, PracticedSystem> getAnonymizePracticedSystemFunction() {
        final Function<GrowingSystem, GrowingSystem> anonymizeGrowingSystemFunction = getAnonymizeGrowingSystemFunction();
        Function<PracticedSystem, PracticedSystem> result = new Function<PracticedSystem, PracticedSystem>() {
            @Override
            public PracticedSystem apply(PracticedSystem input) {

                if (input == null) {
                    return null;
                }
                PracticedSystem result;
                String growingPlanId = input.getGrowingSystem().getGrowingPlan().getTopiaId();
                String domainId = input.getGrowingSystem().getGrowingPlan().getDomain().getTopiaId();
                boolean shouldAnonymizeGrowingPlan = authorizationService.shouldAnonymizeGrowingPlan(growingPlanId);
                boolean shouldAnonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId);
                if (shouldAnonymizeGrowingPlan || shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<PracticedSystem, PracticedSystem> binder = BinderFactory.newBinder(PracticedSystem.class);
                    result = new PracticedSystemImpl();
                    binder.copyExcluding(input, result,
                            PracticedSystem.PROPERTY_GROWING_SYSTEM
                    );

                    result.setGrowingSystem(anonymizeGrowingSystemFunction.apply(input.getGrowingSystem()));
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public PracticedSystem checkForPracticedSystemAnonymization(PracticedSystem practicedSystem) {
        PracticedSystem result = getAnonymizePracticedSystemFunction().apply(practicedSystem);
        return result;
    }

    @Override
    public Function<PracticedSystem, PracticedSystemDto> getPracticedSystemToDtoFunction() {
        Function<PracticedSystem, PracticedSystemDto> result = newContext().
                getPracticedSystemToDtoFunction();
        return result;
    }

    protected Function<ManagementMode, ManagementMode> getAnonymizeManagementModeFunction() {
        final Function<GrowingSystem, GrowingSystem> anonymizeGrowingSystemFunction = getAnonymizeGrowingSystemFunction();
        Function<ManagementMode, ManagementMode> result = new Function<ManagementMode, ManagementMode>() {
            @Override
            public ManagementMode apply(ManagementMode input) {

                if (input == null) {
                    return null;
                }
                ManagementMode result;
                String growingPlanId = input.getGrowingSystem().getGrowingPlan().getTopiaId();
                String domainId = input.getGrowingSystem().getGrowingPlan().getDomain().getTopiaId();
                boolean shouldAnonymizeGrowingPlan = authorizationService.shouldAnonymizeGrowingPlan(growingPlanId);
                boolean shouldAnonymizeDomain = authorizationService.shouldAnonymizeDomain(domainId);
                if (shouldAnonymizeGrowingPlan || shouldAnonymizeDomain) {
                    // AThimel 12/12/13 It is very important to copy entity before changing the value to avoid changes value to be written to DB
                    Binder<ManagementMode, ManagementMode> binder = BinderFactory.newBinder(ManagementMode.class);
                    result = new ManagementModeImpl();
                    binder.copyExcluding(input, result,
                            ManagementMode.PROPERTY_GROWING_SYSTEM
                    );

                    result.setGrowingSystem(anonymizeGrowingSystemFunction.apply(input.getGrowingSystem()));
                } else {
                    // No need to modify instance
                    result = input;
                }

                return result;
            }
        };
        return result;
    }

    @Override
    public ManagementMode checkForManagementModeAnonymization(ManagementMode managementMode) {
        ManagementMode result = getAnonymizeManagementModeFunction().apply(managementMode);
        return result;
    }

    @Override
    public Function<PracticedPlot, PracticedPlotDto> getPracticedPlotToDtoFunction() {
        Function<PracticedPlot, PracticedPlotDto> result = newContext().
                getPracticedPlotToDtoFunction();
        return result;
    }


    @Override
    public ResultList<Performance> checkForPerformancesAnonymization(ResultList<Performance> performances) {
        ResultList<Performance> result = ResultList.transformLazy(performances, getAnonymizePerformanceFunction());
        return result;
    }

    protected Function<Performance, Performance> getAnonymizePerformanceFunction() {
        final Function<Domain, Domain> anonymizeDomainFunction = getAnonymizeDomainFunction();
        final Function<GrowingSystem, GrowingSystem> anonymizeGrowingSystemFunction = getAnonymizeGrowingSystemFunction();
        final Function<Plot, Plot> anonymizePlotFunction = getAnonymizePlotFunction();
        final Function<Zone, Zone> anonymizeZoneFunction = getAnonymizeZoneFunction();

        Function<Performance, Performance> result = new Function<Performance, Performance>() {
            @Override
            public Performance apply(Performance input) {

                Performance result = null;
                if (input != null) {

                    Binder<Performance, Performance> binder = BinderFactory.newBinder(Performance.class);
                    result = new PerformanceImpl();
                    binder.copyExcluding(input, result,
                            Performance.PROPERTY_DOMAINS,
                            Performance.PROPERTY_GROWING_SYSTEMS,
                            Performance.PROPERTY_PLOTS,
                            Performance.PROPERTY_ZONES
                    );

                    if (input.getDomains() != null) {
                        result.setDomains(Collections2.transform(input.getDomains(), anonymizeDomainFunction));
                    }
                    if (input.getGrowingSystems() != null) {
                        result.setGrowingSystems(Collections2.transform(input.getGrowingSystems(), anonymizeGrowingSystemFunction));
                    }
                    if (input.getPlots() != null) {
                        result.setPlots(Collections2.transform(input.getPlots(), anonymizePlotFunction));
                    }
                    if (input.getZones() != null) {
                        result.setZones(Collections2.transform(input.getZones(), anonymizeZoneFunction));
                    }
                }
                return result;
            }
        };
        return result;
    }

    @Override
    public Function<Performance, PerformanceDto> getPerformanceToDtoFunction() {
        Function<Performance, PerformanceDto> result = newContext().
                includeZonePlot(false).
                limitPerformanceElements(Integer.MAX_VALUE).
                getPerformanceToDtoFunction();
        return result;
    }

}
