package fr.inra.agrosyst.services.security;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AnonymizeContext.java 4121 2014-07-03 12:26:54Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/security/AnonymizeContext.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;

import org.nuiton.topia.persistence.TopiaEntity;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.domain.PlotDto;
import fr.inra.agrosyst.api.services.domain.ZoneDto;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemDto;
import fr.inra.agrosyst.api.services.performance.PerformanceDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemDto;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.services.users.UserDto;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class AnonymizeContext {

    protected static final String XXXXX = "xxxxx";

    protected BusinessAuthorizationService authorizationService;
    protected Function<String, String> anonymizeFunction;

    protected Function<Domain, DomainDto> domainToDtoFunction;
    protected Function<GrowingPlan, GrowingPlanDto> growingPlanToDtoFunction;
    protected Function<GrowingSystem, GrowingSystemDto> growingSystemToDtoFunction;
    protected Function<Plot, PlotDto> plotToDtoFunction;
    protected Function<Zone, ZoneDto> zoneToDtoFunction;
    protected Function<PracticedSystem, PracticedSystemDto> practicedSystemToDtoFunction;
    protected Function<PracticedPlot, PracticedPlotDto> practicedPlotToDtoFunction;
    protected Function<Performance, PerformanceDto> performanceToDtoFunction;

    protected Function<Domain, Boolean> shouldAnonymizeDomain;
    protected Function<GrowingPlan, Boolean> shouldAnonymizeGrowingPlan;

    protected boolean includeDomainResponsibles = false;
    protected boolean includeGrowingPlanResponsibles = false;
    protected boolean includeZonePlot = false;
    protected int limitPerformanceElements = Integer.MAX_VALUE;


    /**
     * Method that caches a transform result for a given Function. Using the return Function will avoid transforming
     * twice the same entity
     *
     * @param function the base function doing the transformation
     * @return the same function with cache facility
     */
    public <K, V> Function<K, V> cache(final Function<K, V> function) {
        // The cache instance storing the transformed entities
        final Map<String, V> cache = Maps.newHashMap();

        Function<K, V> result = new Function<K, V>() {
            @Override
            public V apply(K input) {
                if (input == null) {
                    return null;
                }
                // Look for the entity key in the cache
                String key;
                if (input instanceof String) {
                    key = (String) input;
                } else if (input instanceof TopiaEntity) {
                    key = ((TopiaEntity) input).getTopiaId();
                } else {
                    throw new UnsupportedOperationException("Unsupported key type : " + input.getClass());
                }
                V result = cache.get(key);

                // If no result found, use the function
                if (result == null) {
                    result = function.apply(input);
                    cache.put(key, result);
                }
                return result;
            }
        };
        return result;
    }


    public AnonymizeContext(BusinessAuthorizationService authorizationService, Function<String, String> anonymizeFunction) {
        this.authorizationService = authorizationService;
        this.anonymizeFunction = cache(anonymizeFunction);
    }

    public AnonymizeContext includeDomainResponsibles(boolean b) {
        this.includeDomainResponsibles = b;
        return this;
    }

    public AnonymizeContext includeGrowingPlanResponsibles(boolean b) {
        includeGrowingPlanResponsibles = b;
        return this;
    }

    public AnonymizeContext includeZonePlot(boolean b) {
        includeZonePlot = b;
        return this;
    }

    public AnonymizeContext limitPerformanceElements(int i) {
        this.limitPerformanceElements = i;
        return this;
    }


    public Function<Domain, DomainDto> getDomainToDtoFunction() {
        if (domainToDtoFunction == null) {
            domainToDtoFunction = cache(newDomainToDtoFunction());
        }
        return domainToDtoFunction;
    }

    public Function<GrowingPlan, GrowingPlanDto> getGrowingPlanToDtoFunction() {
        if (growingPlanToDtoFunction == null) {
            growingPlanToDtoFunction = cache(newGrowingPlanToDtoFunction());
        }
        return growingPlanToDtoFunction;
    }

    public Function<GrowingSystem, GrowingSystemDto> getGrowingSystemToDtoFunction() {
        if (growingSystemToDtoFunction == null) {
            growingSystemToDtoFunction = cache(newGrowingSystemToDtoFunction());
        }
        return growingSystemToDtoFunction;
    }

    public Function<Plot, PlotDto> getPlotToDtoFunction() {
        if (plotToDtoFunction == null) {
            plotToDtoFunction = cache(newPlotToDtoFunction());
        }
        return plotToDtoFunction;
    }

    public Function<Zone, ZoneDto> getZoneToDtoFunction() {
        if (zoneToDtoFunction == null) {
            zoneToDtoFunction = cache(newZoneToDtoFunction());
        }
        return zoneToDtoFunction;
    }

    public Function<PracticedSystem, PracticedSystemDto> getPracticedSystemToDtoFunction() {
        if (practicedSystemToDtoFunction == null) {
            practicedSystemToDtoFunction = cache(newPracticedSystemToDtoFunction());
        }
        return practicedSystemToDtoFunction;
    }

    public Function<PracticedPlot, PracticedPlotDto> getPracticedPlotToDtoFunction() {
        if (practicedPlotToDtoFunction == null) {
            practicedPlotToDtoFunction = cache(newPracticedPlotToDtoFunction());
        }
        return practicedPlotToDtoFunction;
    }

    public Function<Performance, PerformanceDto> getPerformanceToDtoFunction() {
        if (performanceToDtoFunction == null) {
            performanceToDtoFunction = cache(newPerformanceToDtoFunction());
        }
        return performanceToDtoFunction;
    }

    public Function<Domain, Boolean> getShouldAnonymizeDomain() {
        if (shouldAnonymizeDomain == null) {
            shouldAnonymizeDomain = cache(new Function<Domain, Boolean>() {
                @Override
                public Boolean apply(Domain input) {
                    return authorizationService.shouldAnonymizeDomain(input.getTopiaId());
                }
            });
        }
        return shouldAnonymizeDomain;
    }

    public Function<GrowingPlan, Boolean> getShouldAnonymizeGrowingPlan() {
        if (shouldAnonymizeGrowingPlan == null) {
            shouldAnonymizeGrowingPlan = cache(new Function<GrowingPlan, Boolean>() {
                @Override
                public Boolean apply(GrowingPlan input) {
                    return authorizationService.shouldAnonymizeGrowingPlan(input.getTopiaId());
                }
            });
        }
        return shouldAnonymizeGrowingPlan;
    }

    protected Function<Domain, DomainDto> newDomainToDtoFunction() {
        return new Function<Domain, DomainDto>() {
            @Override
            public DomainDto apply(Domain input) {
                if (input == null) {
                    return null;
                }
                DomainDto result = new DomainDto();
                result.setTopiaId(input.getTopiaId());
                result.setCode(input.getCode());
                result.setCampaign(input.getCampaign());
                result.setType(input.getType());
                result.setLocation(input.getLocation());
                result.setActive(input.isActive());

                boolean shouldAnonymizeDomain = getShouldAnonymizeDomain().apply(input);
                if (shouldAnonymizeDomain) {
                    result.setName(anonymizeFunction.apply(input.getName()));
                    result.setMainContact(XXXXX);
                } else {
                    result.setName(input.getName());
                    result.setMainContact(input.getMainContact());
                }

                List<UserDto> responsibles;
//                if (includeResponsibles && !shouldAnonymizeDomain) {
                if (includeDomainResponsibles) {
                    responsibles = authorizationService.getDomainResponsibles(input.getCode());
                } else {
                    responsibles = Lists.newArrayList();
                }
                result.setResponsibles(responsibles);
                return result;
            }
        };
    }


    protected Function<GrowingPlan, GrowingPlanDto> newGrowingPlanToDtoFunction() {
        return new Function<GrowingPlan, GrowingPlanDto>() {
            @Override
            public GrowingPlanDto apply(GrowingPlan input) {
                if (input == null) {
                    return null;
                }
                GrowingPlanDto result = new GrowingPlanDto();
                result.setTopiaId(input.getTopiaId());
                result.setCode(input.getCode());
                result.setDomain(getDomainToDtoFunction().apply(input.getDomain()));
                result.setType(input.getType());
                result.setActive(input.isActive());

                boolean shouldAnonymizeGrowingPlan = getShouldAnonymizeGrowingPlan().apply(input);
                if (shouldAnonymizeGrowingPlan) {
                    result.setName(anonymizeFunction.apply(input.getName()));
                } else {
                    result.setName(input.getName());
                }

                List<UserDto> responsibles;
//                if (includeResponsibles && !shouldAnonymizeGrowingPlan) {
                if (includeGrowingPlanResponsibles) {
                    responsibles = authorizationService.getGrowingPlanResponsibles(input.getCode());
                } else {
                    responsibles = Lists.newArrayList();
                }
                result.setResponsibles(responsibles);
                return result;
            }
        };
    }


    protected Function<GrowingSystem, GrowingSystemDto> newGrowingSystemToDtoFunction() {
        return new Function<GrowingSystem, GrowingSystemDto>() {
            @Override
            public GrowingSystemDto apply(GrowingSystem input) {
                if (input == null) {
                    return null;
                }
                GrowingSystemDto result = new GrowingSystemDto();
                result.setTopiaId(input.getTopiaId());
                result.setCode(input.getCode());
                result.setName(input.getName());
                result.setValidated(input.isValidated());
                result.setActive(input.isActive());
                result.setDephyNumber(input.getDephyNumber());
                result.setSector(input.getSector());
                result.setGrowingPlan(getGrowingPlanToDtoFunction().apply(input.getGrowingPlan()));
                if (input.getNetworks() != null) {
                    Map<String, String> networks = Maps.newLinkedHashMap();
                    result.setNetworks(networks);
                    for (Network network : input.getNetworks()) {
                        networks.put(network.getTopiaId(), network.getName());
                    }
                }
                return result;
            }
        };
    }

    protected Function<Plot, PlotDto> newPlotToDtoFunction() {
        return new Function<Plot, PlotDto>() {
            @Override
            public PlotDto apply(Plot input) {
                if (input == null) {
                    return null;
                }
                PlotDto result = new PlotDto();
                result.setTopiaId(input.getTopiaId());
                result.setCode(input.getCode());
                result.setArea(input.getArea());
                result.setGrowingSystem(getGrowingSystemToDtoFunction().apply(input.getGrowingSystem()));
                result.setDomain(getDomainToDtoFunction().apply(input.getDomain()));
                if (getShouldAnonymizeDomain().apply(input.getDomain())) {
                    result.setName(anonymizeFunction.apply(input.getName()));
                } else {
                    result.setName(input.getName());
                }
                return result;
            }
        };

    }


    protected Function<Zone, ZoneDto> newZoneToDtoFunction() {
        return new Function<Zone, ZoneDto>() {
            @Override
            public ZoneDto apply(Zone input) {
                if (input == null) {
                    return null;
                }
                ZoneDto result = new ZoneDto();
                result.setTopiaId(input.getTopiaId());
                result.setCode(input.getCode());
                result.setName(input.getName());
                Plot plot = input.getPlot();
                if (includeZonePlot) {
                    result.setPlot(getPlotToDtoFunction().apply(plot));
                } else {
                    if (plot != null) {
                        PlotDto plotDto = new PlotDto();
                        plotDto.setTopiaId(plot.getTopiaId());
                        result.setPlot(plotDto);
                    }
                }
                return result;
            }
        };

    }

    protected Function<PracticedSystem, PracticedSystemDto> newPracticedSystemToDtoFunction() {
        return new Function<PracticedSystem, PracticedSystemDto>() {
            @Override
            public PracticedSystemDto apply(PracticedSystem input) {
                if (input == null) {
                    return null;
                }
                PracticedSystemDto result = new PracticedSystemDto();
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setGrowingSystem(getGrowingSystemToDtoFunction().apply(input.getGrowingSystem()));
                result.setCampaigns(input.getCampaigns());
                result.setActive(input.isActive());
                result.setValidated(input.isValidated());
                return result;
            }
        };
    }

    protected Function<PracticedPlot, PracticedPlotDto> newPracticedPlotToDtoFunction() {
        return new Function<PracticedPlot, PracticedPlotDto>() {
            @Override
            public PracticedPlotDto apply(PracticedPlot input) {
                if (input == null) {
                    return null;
                }
                PracticedPlotDto result = new PracticedPlotDto();
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setPracticedSystem(getPracticedSystemToDtoFunction().apply(input.getPracticedSystem()));
                return result;
            }
        };
    }

    protected Function<Performance, PerformanceDto> newPerformanceToDtoFunction() {

        return new Function<Performance, PerformanceDto>() {
            @Override
            public PerformanceDto apply(Performance input) {
                if (input == null) {
                    return null;
                }
                PerformanceDto result = new PerformanceDto();
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setComputeStatus(input.getComputeStatus());
                result.setUpdateDate(input.getUpdateDate());

                if (input.getDomains() != null) {
                    result.setDomains(Lists.newArrayList(Iterables.transform(Iterables.limit(input.getDomains(), limitPerformanceElements), getDomainToDtoFunction())));
                }
                if (input.getGrowingSystems() != null) {
                    result.setGrowingSystems(Lists.newArrayList(Iterables.transform(Iterables.limit(input.getGrowingSystems(), limitPerformanceElements), getGrowingSystemToDtoFunction())));
                }
                if (input.getPlots() != null) {
                    result.setPlots(Lists.newArrayList(Iterables.transform(Iterables.limit(input.getPlots(), limitPerformanceElements), getPlotToDtoFunction())));
                }
                if (input.getZones() != null) {
                    result.setZones(Lists.newArrayList(Iterables.transform(Iterables.limit(input.getZones(), limitPerformanceElements), getZoneToDtoFunction())));
                }
                return result;
            }
        };
    }

}
