package fr.inra.agrosyst.services.referential.csv;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AbstractAgrosystModel.java 5089 2015-09-09 07:44:14Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/referential/csv/AbstractAgrosystModel.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.MaterielTransportUnit;
import fr.inra.agrosyst.api.entities.MaterielWorkRateUnit;
import fr.inra.agrosyst.api.entities.TypeDEPHY;
import fr.inra.agrosyst.api.entities.action.FertiOrgaUnit;
import fr.inra.agrosyst.api.entities.referential.VitesseCouv;
import fr.inra.agrosyst.api.entities.security.RoleType;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.csv.ImportableColumn;
import org.nuiton.csv.ValueFormatter;
import org.nuiton.csv.ValueParser;
import org.nuiton.csv.ext.AbstractImportModel;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

public abstract class AbstractAgrosystModel<E> extends AbstractImportModel<E>{

    /**
     * String to integer converter.
     */
    protected static final ValueParser<Integer> INT_PARSER = new ValueParser<Integer>() {
        @Override
        public Integer parse(String value) throws ParseException {
            int result = 0;
            if (!Strings.isNullOrEmpty(value)) {
                result = Integer.valueOf(value);
            }
            return result;
        }
    };

    /**
     * String to integer converter (null allowed).
     */
    protected static final ValueParser<Integer> INTEGER_WITH_NULL_PARSER = new ValueParser<Integer>() {
        @Override
        public Integer parse(String value) throws ParseException {
            Integer result = null;
            if (!Strings.isNullOrEmpty(value)) {
                result = Integer.valueOf(value);
            }
            return result;
        }
    };

    /**
     * String to double converter.
     * Can handle , or . as decimal separator and ' ' as thousand separator
     */
    protected static final ValueParser<Double> DOUBLE_PARSER =  new ValueParser<Double>() {
        @Override
        public Double parse(String value) throws ParseException {
            double result = 0;
            if (!value.isEmpty()) {
                // " " est un espace insécable, pas un " "
                result = Double.valueOf(value.replace(',','.').replace(" ", ""));
            }
            return result;
        }
    };

    /**
     * String to double converter (null allowed).
     * Can handle , or . as decimal separator and ' ' as thousand separator
     */
    protected static final ValueParser<Double> DOUBLE_WITH_NULL_PARSER =  new ValueParser<Double>() {
        @Override
        public Double parse(String value) throws ParseException {
            Double result = null;
            if (!value.isEmpty()) {
                // " " est un espace insécable, pas un " "
                result = Double.valueOf(value.replace(',','.').replace(" ", ""));
            }
            return result;
        }
    };

    /**
     * Zero to empty string converter.
     */
    protected static final ValueParser<String> ZERO_TO_EMPTY_PARSER = new ValueParser<String>() {
        @Override
        public String parse(String value) throws ParseException {
            String result = value.trim();
            result = result.replaceAll("\\s{2,}", " "); // replace multiple spaces
            if (result.equals("0")) {
                result = "";
            }
            return result;
        }
    };

    /**
     * Date converter converter.
     */
    protected static final ValueParser<Date> DATE_PARSER = new ValueParser<Date>() {
        @Override
        public Date parse(String value) throws ParseException {
            Date result = null;
            if (!Strings.isNullOrEmpty(value)) {
                SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
                result = format.parse(value);
            }
            return result;
        }
    };

    /**
     * O/N boolean parser.
     */
    protected static final ValueParser<Boolean> O_N_PARSER = new ValueParser<Boolean>() {
        @Override
        public Boolean parse(String value) throws ParseException {
            Boolean result;
            if ("O".equalsIgnoreCase(value)) {
                result = Boolean.TRUE;
            } else {
                result = Boolean.FALSE;
            }
            return result;
        }
    };

    protected static final ValueFormatter<Boolean> O_N_FORMATTER = new ValueFormatter<Boolean>() {
        @Override
        public String format(Boolean value) {
            String result = value ? "O" : "N";
            return result;
        }
    };

    protected static final ValueParser<Boolean> T_F_PARSER = new ValueParser<Boolean>() {
        @Override
        public Boolean parse(String value) throws ParseException {
            Boolean result;
            if ("t".equalsIgnoreCase(value)) {
                result = Boolean.TRUE;
            } else {
                result = Boolean.FALSE;
            }
            return result;
        }
    };

    protected static final ValueFormatter<Boolean> T_F_FORMATTER = new ValueFormatter<Boolean>() {
        @Override
        public String format(Boolean value) {
            String result = value ? "t" : "f";
            return result;
        }
    };

    protected static final ValueFormatter<Integer> INTEGER_FORMATTER = new ValueFormatter<Integer>() {
        @Override
        public String format(Integer value) {
            String result;
            if (value != null) {
                result = String.valueOf(value);
            } else {
                result = "";
            }
            return result;
        }
    };

    protected static final ValueFormatter<Double> DOUBLE_FORMATTER = new ValueFormatter<Double>() {
        @Override
        public String format(Double value) {
            String result;
            if (value != null) {
                result = String.valueOf(value);
            } else {
                result = "";
            }
            return result;
        }
    };
    
    protected static final ValueFormatter<Date> DATE_FORMATTER = new ValueFormatter<Date>() {
        @Override
        public String format(Date value) {
            String result;
            if (value != null) {
                SimpleDateFormat format = new SimpleDateFormat("dd/MM/yyyy");
                result = format.format(value);
            } else {
                result = "";
            }
            return result;
        }
    };

    protected static final ValueFormatter<AgrosystInterventionType> AGROSYST_INTERVENTION_TYPE_FORMATTER = new ValueFormatter<AgrosystInterventionType>() {
        @Override
        public String format(AgrosystInterventionType value) {
            String result;
            if (value != null) {
                result = value.name().toLowerCase();
            } else {
                result = "";
            }
            return result;
        }
    };

    protected static final ValueFormatter<BioAgressorType> AGROSYST_BIO_AGRESSOR_TYPE_FORMATTER = new ValueFormatter<BioAgressorType>() {
        @Override
        public String format(BioAgressorType value) {
            String result;
            if (value != null) {
                if (value == BioAgressorType.ACTION_SUR_LE_METABOLISME){
                    result = "Action sur le métabolisme";
                } else if (value == BioAgressorType.AUXILIAIRE_BIOLOGIQUE){
                    result = "Auxiliaire biologique";
                } else if (value == BioAgressorType.GENERIQUE){
                    result = "Générique";
                } else if (value == BioAgressorType.MALADIE){
                    result = "Maladie";
                } else if (value == BioAgressorType.MALADIE_PHYSIOLOGIQUE){
                    result = "Maladie physiologique";
                } else if (value == BioAgressorType.PLANTE_PARASITE){
                    result = "Plante parasite";
                } else if (value == BioAgressorType.PLANTES_ENVAHISSANTES_ORIGINE_EXOTIQUE){
                    result = "Plantes envahissantes d'origine exotique";
                } else if (value == BioAgressorType.RAVAGEUR){
                    result = "Ravageur";
                } else if (value == BioAgressorType.VIRUS){
                    result = "Virus";
                } else {
                    throw new UnsupportedOperationException("type non supporté : " + value);
                }

            } else {
                result = "";
            }

            return result;
        }
    };

    protected static final ValueFormatter<FertiOrgaUnit> AGROSYST_FERTI_ORGA_UNIT_FORMATTER = new ValueFormatter<FertiOrgaUnit>() {

        @Override
        public String format(FertiOrgaUnit fertiOrgaUnit) {
            String result;
            if (fertiOrgaUnit != null) {
                if(fertiOrgaUnit == FertiOrgaUnit.KG_M_CUB) {
                    result = "kg/m3";
                } else if (fertiOrgaUnit == FertiOrgaUnit.KG_T) {
                    result = "kg/T";
                } else {
                    throw new UnsupportedOperationException("unité non supportée : " + fertiOrgaUnit);
                }
            } else {
                result = "";
            }
            return result;
        }
    };

    protected static final ValueParser<AgrosystInterventionType> AGROSYST_INTERVENTION_TYPE_PARSER = new ValueParser<AgrosystInterventionType>() {
        @Override
        public AgrosystInterventionType parse(String value) throws ParseException {
            AgrosystInterventionType result = null;
            if (!Strings.isNullOrEmpty(value)) {
                result = AgrosystInterventionType.valueOf(value.toUpperCase());
            }
            return result;
        }
    };

    protected static final ValueParser<BioAgressorType> AGROSYST_BIO_AGRESSOR_TYPE_PARSER = new ValueParser<BioAgressorType>() {
        @Override
        public BioAgressorType parse(String value) throws ParseException {
            BioAgressorType result = null;
            if (!Strings.isNullOrEmpty(value)) {
                //result = BioAgressorType.valueOf(value.toUpperCase());
                if (value.equalsIgnoreCase("Action sur le métabolisme")){
                    result = BioAgressorType.ACTION_SUR_LE_METABOLISME;
                } else if (value.equalsIgnoreCase("Auxiliaire biologique")){
                    result = BioAgressorType.AUXILIAIRE_BIOLOGIQUE;
                } else if (value.equalsIgnoreCase("Générique")){
                    result = BioAgressorType.GENERIQUE;
                } else if (value.equalsIgnoreCase("Maladie")){
                    result = BioAgressorType.MALADIE;
                } else if (value.equalsIgnoreCase("Maladie physiologique")){
                    result = BioAgressorType.MALADIE_PHYSIOLOGIQUE;
                } else if (value.equalsIgnoreCase("Plante parasite")){
                    result = BioAgressorType.PLANTE_PARASITE;
                } else if (value.equalsIgnoreCase("Plantes envahissantes d'origine exotique")){
                    result = BioAgressorType.PLANTES_ENVAHISSANTES_ORIGINE_EXOTIQUE;
                } else if (value.equalsIgnoreCase("Ravageur")){
                    result = BioAgressorType.RAVAGEUR;
                } else if (value.equalsIgnoreCase("Virus")){
                    result = BioAgressorType.VIRUS;
                } else {
                    throw new UnsupportedOperationException("type non supporté : " + value);
                }
            }
            return result;
        }
    };

    protected static final ValueParser<FertiOrgaUnit> AGROSYST_FERTI_ORGA_UNIT_PARSER = new ValueParser<FertiOrgaUnit>() {
        @Override
        public FertiOrgaUnit parse(String value) throws ParseException {
            FertiOrgaUnit result = null;
            if (!Strings.isNullOrEmpty(value)) {
                if (value.equalsIgnoreCase("kg/T")) {
                    result = FertiOrgaUnit.KG_T;
                } else if (value.equalsIgnoreCase("kg/m3")) {
                    result = FertiOrgaUnit.KG_M_CUB;
                } else {
                    throw new UnsupportedOperationException("Unité non supporté : " + value);
                }
            }
            return result;
        }
    };

    protected static final ValueParser<VitesseCouv> VITESSE_COUV_PARSER = new ValueParser<VitesseCouv>() {
        @Override
        public VitesseCouv parse(String value) throws ParseException {
            VitesseCouv result = VitesseCouv.valueOf(value);
            return result;
        }
    };

    protected static final ValueFormatter<VitesseCouv> VITESSE_COUV_FORMATTER = new ValueFormatter<VitesseCouv>() {
        @Override
        public String format(VitesseCouv value) {
            return value.name();
        }
    };

    /**
     * Convertit une chaine contenant un pourcentage obligatoire en son equivalent en double.
     */
    protected static final ValueParser<Double> PERCENT_DOUBLE_PARSER = new ValueParser<Double>() {
        @Override
        public Double parse(String value) throws ParseException {
            Double result = null;
            if (value != null) {
                if (value.matches("\\d{1,2}([\\.,]\\d+)?\\s*%")) {
                    String str = StringUtils.removeEnd(value, "%").trim();
                    str = str.replace(',', '.');
                    result = Double.parseDouble(str) / 100;
                }
            }
            return result;
        }
    };

    /**
     * Convertit un double en une chaine formattant le double en pourcentage.
     */
    protected static final ValueFormatter<Double> PERCENT_DOUBLE_FORMATTER = new ValueFormatter<Double>() {
        @Override
        public String format(Double value) {
            return String.format("%f%%", value);
        }
    };

    protected AbstractAgrosystModel(char separator) {
        super(separator);
    }

    protected Map<String, String> headerMap;

    protected String getHeaderId(String header) {
        return header.toLowerCase();
    }

    /**
     * Override method to build map to manage case insesitive headers.
     */
    @Override
    public void pushCsvHeaderNames(List<String> headerNames) {
        headerMap = Maps.uniqueIndex(headerNames, new Function<String, String>() {
            @Override
            public String apply(String input) {
                return getHeaderId(input);
            }
        });
    }

    /**
     * Retourne le nom réel de l'header 
     * @param header header string
     * @return header in file
     */
    protected String getRealHeader(String header) {
        String result = null;
        if (headerMap != null) { // non dynamic mode
            result = headerMap.get(getHeaderId(header));
        }
        // si il existe pas
        if (result == null) {
            result = header;
        }
        return result;
    }

    protected boolean hasHeader(String header) {
        return headerMap.containsKey(getHeaderId(header));
    }

    @Override
    public <T> ImportableColumn<E, T> newIgnoredColumn(String headerName) {
        return super.newIgnoredColumn(getRealHeader(headerName));
    }

    @Override
    public ImportableColumn<E, String> newMandatoryColumn(String headerName) {
        return super.newMandatoryColumn(getRealHeader(headerName));
    }

    @Override
    public ImportableColumn<E, String> newMandatoryColumn(String headerName, String propertyName) {
        return super.newMandatoryColumn(getRealHeader(headerName), propertyName);
    }

    @Override
    public <T> ImportableColumn<E, T> newMandatoryColumn(String headerName, String propertyName, ValueParser<T> valueParser) {
        return super.newMandatoryColumn(getRealHeader(headerName), propertyName, valueParser);
    }

    protected static final ValueParser<RoleType> ROLE_TYPE_PARSER = new ValueParser<RoleType>() {
        @Override
        public RoleType parse(String value) throws ParseException {
            RoleType result = null;
            if (!Strings.isNullOrEmpty(value)) {
                result = RoleType.valueOf(value.toUpperCase());
            }
            return result;
        }
    };

    protected static final ValueParser<MaterielWorkRateUnit> AGROSYST_MATERIEL_WORK_RATE_UNIT_PARSER = new ValueParser<MaterielWorkRateUnit>() {
        @Override
        public MaterielWorkRateUnit parse(String value) throws ParseException {
            MaterielWorkRateUnit result = null;
            if (!Strings.isNullOrEmpty(value)){
                String valueTrim = StringUtils.remove(value, ' ');
                if (valueTrim.equalsIgnoreCase("bal/h")) {
                    result = MaterielWorkRateUnit.BAL_H;
                } else if (valueTrim.equalsIgnoreCase("ha/h")) {
                    result = MaterielWorkRateUnit.HA_H;
                } else if (valueTrim.equalsIgnoreCase("m3/h")) {
                    result = MaterielWorkRateUnit.M3_H;
                } else if (valueTrim.equalsIgnoreCase("qx/h")) {
                    result = MaterielWorkRateUnit.QX_H;
                } else if (valueTrim.equalsIgnoreCase("sacs/h")) {
                    result = MaterielWorkRateUnit.SACS_H;
                } else if (valueTrim.equalsIgnoreCase("t/benne")) {
                    result = MaterielWorkRateUnit.T_BENNE;
                } else if (valueTrim.equalsIgnoreCase("trous/h")) {
                    result = MaterielWorkRateUnit.TROUS_H;
                } else if (valueTrim.equalsIgnoreCase("voy/h")) {
                    result = MaterielWorkRateUnit.VOY_H;
                } else if (valueTrim.equalsIgnoreCase("t/ha")) {
                    result = MaterielWorkRateUnit.T_HA;
                } else if (valueTrim.equalsIgnoreCase("h/ha")) {
                    result = MaterielWorkRateUnit.H_HA;
                } else if (valueTrim.equalsIgnoreCase("t/h")) {
                    result = MaterielWorkRateUnit.T_H;
                } else if (valueTrim.equalsIgnoreCase("€/h")) {
                    result = MaterielWorkRateUnit.EURO_H;
                } else if (valueTrim.equalsIgnoreCase("0")) {
                    result = null;
                } else {
                    throw new UnsupportedOperationException("Unité non supporté : " + value);
                }
            }
            return result;
        }
    };

    protected static final ValueFormatter<MaterielWorkRateUnit> AGROSYST_MATERIEL_WORK_RATE_UNIT_FORMATTER = new ValueFormatter<MaterielWorkRateUnit>() {

        @Override
        public String format(MaterielWorkRateUnit materielWorkRateUnit) {
            String result;
            if (materielWorkRateUnit != null) {
                if(materielWorkRateUnit == MaterielWorkRateUnit.BAL_H) {
                    result = "bal/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.HA_H) {
                    result = "ha/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.M3_H) {
                    result = "m3/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.QX_H) {
                    result = "qx/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.SACS_H) {
                    result = "sacs/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.T_BENNE) {
                    result = "t/benne";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.TROUS_H) {
                    result = "trous/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.VOY_H) {
                    result = "voy/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.T_HA) {
                    result = "t/ha";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.H_HA) {
                    result = "h/ha";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.T_H) {
                    result = "t/h";
                } else if (materielWorkRateUnit == MaterielWorkRateUnit.EURO_H) {
                    result = "€/h";
                } else {
                    throw new UnsupportedOperationException("unité non supportée : " + materielWorkRateUnit);
                }
            } else {
                result = "";
            }
            return result;
        }
    };

    protected static final ValueParser<MaterielTransportUnit> AGROSYST_MATERIEL_TRANSPORT_UNIT_PARSER = new ValueParser<MaterielTransportUnit>() {
        @Override
        public MaterielTransportUnit parse(String value) throws ParseException {
            MaterielTransportUnit result = null;
            if (!Strings.isNullOrEmpty(value)) {
                value = StringUtils.remove(value, ' ');
                if (value.equalsIgnoreCase("l") || value.equalsIgnoreCase("litres")) {
                    result = MaterielTransportUnit.L;
                } else if (value.equalsIgnoreCase("m3")) {
                    result = MaterielTransportUnit.M3;
                } else if (value.equalsIgnoreCase("t")) {
                    result = MaterielTransportUnit.T;
                } else {
                    throw new UnsupportedOperationException("Unité non supporté : " + value);
                }
            }
            return result;
        }
    };

    protected static final ValueFormatter<MaterielTransportUnit> AGROSYST_MATERIEL_TRANSPORT_UNIT_FORMATTER = new ValueFormatter<MaterielTransportUnit>() {

        @Override
        public String format(MaterielTransportUnit materielTransportUnit) {
            String result;
            if (materielTransportUnit != null) {
                if(materielTransportUnit == MaterielTransportUnit.L) {
                    result = "l";
                } else if (materielTransportUnit == MaterielTransportUnit.M3) {
                    result = "m3";
                } else if (materielTransportUnit == MaterielTransportUnit.T) {
                    result = "T";
                } else {
                    throw new UnsupportedOperationException("unité non supportée : " + materielTransportUnit);
                }
            } else {
                result = "";
            }
            return result;
        }
    };

    public static final ValueParser<TypeDEPHY> TYPE_DEPHY_PARSER = new ValueParser<TypeDEPHY>() {
        @Override
        public TypeDEPHY parse(String value) throws ParseException {
            TypeDEPHY result = null;

            if (!Strings.isNullOrEmpty(value)) {
                if (value.equalsIgnoreCase("DEPHY-FERME")) {
                    result = TypeDEPHY.DEPHY_FERME;
                } else if (value.equalsIgnoreCase("DEPHY-EXPE")) {
                    result = TypeDEPHY.DEPHY_EXPE;
                } else if(value.equalsIgnoreCase("Hors DEPHY")) {
                    result = TypeDEPHY.NOT_DEPHY;
                } else {
                    result = null;
                }
            }
            return result;
        }
    };

}
