package fr.inra.agrosyst.services.referential;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ReferentialServiceImpl.java 5075 2015-09-01 12:55:16Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/referential/ReferentialServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.MaterielType;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.referential.*;
import fr.inra.agrosyst.api.services.itk.Itk;
import fr.inra.agrosyst.api.services.practiced.RefStadeEdiDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.referential.MineralProductType;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.api.services.referential.TypeMaterielFilter;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.CacheService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

public class ReferentialServiceImpl extends AbstractAgrosystService implements ReferentialService {

    private static final Log log = LogFactory.getLog(ReferentialServiceImpl.class);

    protected RefFertiMinUNIFATopiaDao refFertiMinUNIFATopiaDao;
    protected final String DEFAULT_SOURCE = "AGROSYST_USER";

    public void setRefFertiMinUNIFATopiaDao(RefFertiMinUNIFATopiaDao refFertiMinUNIFATopiaDao) {
        this.refFertiMinUNIFATopiaDao = refFertiMinUNIFATopiaDao;
    }


    protected static final Predicate<RefEspeceToVariete> IS_GEVES = new Predicate<RefEspeceToVariete>() {
        @Override
        public boolean apply(RefEspeceToVariete input) {
            return "geves".equalsIgnoreCase(input.getReferentiel_source());
        }
    };

    protected static final Function<RefEspeceToVariete, Integer> TO_VARIETE_GEVES_SPECIES_ID = new Function<RefEspeceToVariete, Integer>() {
        @Override
        public Integer apply(RefEspeceToVariete input) {
            Integer result = Integer.valueOf(input.getCode_espece_autre_referentiel());
            return result;
        }
    };

    protected static final Function<RefEspeceToVariete, String> TO_VARIETE_PLANT_GRAPE_SPECIES_ID = new Function<RefEspeceToVariete, String>() {
        @Override
        public String apply(RefEspeceToVariete input) {
            String result = input.getCode_espece_autre_referentiel();
            return result;
        }
    };

    protected static final Function<RefActaGroupeCultures, Integer> GET_GROUPE_ID_CULTURE = new Function<RefActaGroupeCultures, Integer>() {
        @Override
        public Integer apply(RefActaGroupeCultures input) {
            return input.getId_groupe_culture();
        }
    };

    protected static final Comparator<RefStadeEdiDto> STADE_EDI_DTO_COMPARATOR = new Comparator<RefStadeEdiDto>() {
        @Override
        public int compare(RefStadeEdiDto o1, RefStadeEdiDto o2) {
            return o1.getLabel().compareTo(o2.getLabel());
        }
    };


    protected static final Function<CroppingPlanSpecies, String> GET_REF_ESPECE_VARIETY_KEY = new Function<CroppingPlanSpecies, String>() {
        @Override
        public String apply(CroppingPlanSpecies input) {
            String id = input.getSpecies().getTopiaId();
            if (input.getVariety() != null) {
                id = id + "_" + input.getVariety().getTopiaId();
            }
            return id;
        }
    };

    protected CacheService cacheService;

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    @Override
    public Map<MaterielType, List<String>> getTypeMateriel1List() {
        Map<MaterielType, List<String>> result = Maps.newLinkedHashMap();

        result.put(MaterielType.TRACTEUR, getPersistenceContext().getRefMaterielTractionDao().findTypeMateriel1Values());
        result.put(MaterielType.AUTOMOTEUR, getPersistenceContext().getRefMaterielAutomoteurDao().findTypeMateriel1Values());
        result.put(MaterielType.OUTIL, getPersistenceContext().getRefMaterielOutilDao().findTypeMateriel1Values());
        result.put(MaterielType.IRRIGATION, getPersistenceContext().getRefMaterielIrrigationDao().findTypeMateriel1Values());
        // fast removed since 6723
        //result.put(MaterielType.AUTRE, Collections.singletonList(MaterielType.AUTRE.name()));
        return result;
    }

    protected AbstractRefMaterielTopiaDao getRefMaterielDao(MaterielType type) {
        AbstractRefMaterielTopiaDao dao;
        switch (type) {
            case TRACTEUR:
                dao = getPersistenceContext().getRefMaterielTractionDao();
                break;
            case AUTOMOTEUR:
                dao = getPersistenceContext().getRefMaterielAutomoteurDao();
                break;
            case OUTIL:
                dao = getPersistenceContext().getRefMaterielOutilDao();
                break;
            case IRRIGATION:
                dao = getPersistenceContext().getRefMaterielIrrigationDao();
                break;
            default:
                dao = getPersistenceContext().getRefMaterielDao();
        }
        return dao;
    }

    @Override
    public List<String> getTypeMateriel2List(TypeMaterielFilter filter) {
        MaterielType type = filter.getType();
        Preconditions.checkArgument(type != null, "Type de matériel inconnu");
        AbstractRefMaterielTopiaDao dao = getRefMaterielDao(type);

        // declare dao
        List result = dao.findPropertyValues(RefMateriel.PROPERTY_TYPE_MATERIEL2,
                filter.getTypeMateriel1(), null, null);
        return result;
    }

    @Override
    public List<String> getTypeMateriel3List(TypeMaterielFilter filter) {
        MaterielType type = filter.getType();
        Preconditions.checkArgument(type != null, "Type de matériel inconnu");
        AbstractRefMaterielTopiaDao dao = getRefMaterielDao(type);

        List result = dao.findPropertyValues(RefMateriel.PROPERTY_TYPE_MATERIEL3,
                filter.getTypeMateriel1(), filter.getTypeMateriel2(), null);
        return result;
    }

    @Override
    public List<String> getTypeMateriel4List(TypeMaterielFilter filter) {
        MaterielType type = filter.getType();
        Preconditions.checkArgument(type != null, "Type de matériel inconnu");
        AbstractRefMaterielTopiaDao dao = getRefMaterielDao(type);
        List result = dao.findPropertyValues(RefMateriel.PROPERTY_TYPE_MATERIEL4,
                filter.getTypeMateriel1(), filter.getTypeMateriel2(), filter.getTypeMateriel3());
        return result;
    }

    @Override
    public Map<String, String[]> getMaterielUniteMap(TypeMaterielFilter filter) {
        MaterielType type = filter.getType();
        Preconditions.checkArgument(type != null, "Type de matériel inconnu");
        AbstractRefMaterielTopiaDao dao = getRefMaterielDao(type);
        Map result = dao.findPropertyValuesAsMap(RefMateriel.PROPERTY_UNITE_PAR_AN,
                RefMateriel.PROPERTY_UNITE, filter.getTypeMateriel1(), filter.getTypeMateriel2(),
                filter.getTypeMateriel3(), filter.getTypeMateriel4());
        return result;
    }

    @Override
    public RefMateriel findMateriel(String materielTopiaId) {
        // declare dao
        RefMateriel result = getPersistenceContext().getRefMaterielDao().forTopiaIdEquals(materielTopiaId).findUnique();
        return result;
    }

    @Override
    public Map<Integer, String> getSolArvalisRegions() {
        Map<Integer, String> result = getPersistenceContext().getRefSolArvalisDao().getAllSolArvalisRegions();
        return result;
    }

    @Override
    public List<RefSolArvalis> getSolArvalis(Integer regionCode) {
        List<RefSolArvalis> result;
        if (regionCode == null) {
            result = Lists.newArrayList();
        } else {
            result = getPersistenceContext().getRefSolArvalisDao().findAllForRegion(regionCode);
        }
        return result;
    }

    @Override
    public RefSolArvalis findSolArvalis(String solArvalisTopiaId) {
        RefSolArvalis result = getPersistenceContext().getRefSolArvalisDao().forTopiaIdEquals(solArvalisTopiaId).findUnique();
        return result;
    }

    @Override
    public List<RefEspece> findSpecies(String filter) {
        List<RefEspece> result = getPersistenceContext().getRefEspeceDao().findActiveEspeces(filter, 20);
        return result;
    }

    @Override
    public List<RefVariete> findVarietes(String speciesId, String filter) {
        RefEspece species = getPersistenceContext().getRefEspeceDao().forTopiaIdEquals(speciesId).findUnique();

        List<RefEspeceToVariete> varietes = getPersistenceContext().getRefEspeceToVarieteDao().forCode_espece_ediEquals(species.getCode_espece_botanique()).findAll();

        List<RefVariete> result = Lists.newArrayList();
        int maxResults = 20;

        // Variétés Geves
        Iterable<RefEspeceToVariete> gevesVarietes = Iterables.filter(varietes, IS_GEVES);
        Set<Integer> varietesGevesIds = Sets.newHashSet(Iterables.transform(gevesVarietes, TO_VARIETE_GEVES_SPECIES_ID));
        List<RefVarieteGeves> varietesGeves = getPersistenceContext().getRefVarieteGevesDao().findAllActiveVarietes(varietesGevesIds, filter, maxResults);
        result.addAll(varietesGeves);

        // Variétés PlantGrape
        Iterable<RefEspeceToVariete> plantGrapeVarietes = Iterables.filter(varietes, Predicates.not(IS_GEVES));
        Set<String> varietesPlantGrapeIds = Sets.newHashSet(Iterables.transform(plantGrapeVarietes, TO_VARIETE_PLANT_GRAPE_SPECIES_ID));
        List<RefVarietePlantGrape> varietesPlantGrappe = getPersistenceContext().getRefVarietePlantGrapeDao().findAllVarietes(varietesPlantGrapeIds, filter, maxResults - result.size() - 1);
        result.addAll(varietesPlantGrappe);

        return result;
    }

    @Override
    public boolean validVarietesFromSpeciesId(RefVariete variete, String speciesId) {

        RefEspece species = getPersistenceContext().getRefEspeceDao().forTopiaIdEquals(speciesId).findUnique();

        boolean result = validVarietesFromCodeEspeceEdi(variete, species.getCode_espece_botanique());

        return result;
    }

    @Override
    public boolean validVarietesFromCodeEspeceEdi(RefVariete variete, String code_espece_botanique) {
        boolean result;

        Preconditions.checkArgument(variete != null, StringUtils.isNoneBlank(code_espece_botanique));

        List<RefEspeceToVariete> varietes = getPersistenceContext().getRefEspeceToVarieteDao().forCode_espece_ediEquals(code_espece_botanique).findAll();

        List<RefVariete> speciesVariety = Lists.newArrayList();

        // Variétés Geves
        Iterable<RefEspeceToVariete> gevesVarietes = Iterables.filter(varietes, IS_GEVES);
        Set<Integer> varietesGevesIds = Sets.newHashSet(Iterables.transform(gevesVarietes, TO_VARIETE_GEVES_SPECIES_ID));
        List<RefVarieteGeves> varietesGeves = getPersistenceContext().getRefVarieteGevesDao().findAllActiveVarietes(varietesGevesIds, null, Integer.MAX_VALUE);
        speciesVariety.addAll(varietesGeves);

        // Variétés PlantGrape
        Iterable<RefEspeceToVariete> plantGrapeVarietes = Iterables.filter(varietes, Predicates.not(IS_GEVES));
        Set<String> varietesPlantGrapeIds = Sets.newHashSet(Iterables.transform(plantGrapeVarietes, TO_VARIETE_PLANT_GRAPE_SPECIES_ID));
        List<RefVarietePlantGrape> varietesPlantGrappe = getPersistenceContext().getRefVarietePlantGrapeDao().findAllVarietes(varietesPlantGrapeIds, null, Integer.MAX_VALUE);
        speciesVariety.addAll(varietesPlantGrappe);

        result = speciesVariety.contains(variete);

        return result;
    }

    @Override
    public RefEspece getSpecies(String speciesId) {
        RefEspece result = getPersistenceContext().getRefEspeceDao().forTopiaIdEquals(speciesId).findUnique();
        return result;
    }

    @Override
    public RefVariete getVariete(String varieteId) {
        RefVariete result = getPersistenceContext().getRefVarieteDao().forTopiaIdEquals(varieteId).findUnique();
        return result;
    }

    @Override
    public RefClonePlantGrape getClonePlantGrape(String clonePlantGrapeId) {
        RefClonePlantGrape result = getPersistenceContext().getRefClonePlantGrapeDao().forTopiaIdEquals(clonePlantGrapeId).findUnique();
        return result;
    }

    @Override
    public List<RefVariete> findGraftSupports(String filter) {

        List<RefVariete> result = Lists.newArrayList();
        int maxResults = 20;

        // Variétés Geves
        int graftSupportCodeSection = getConfig().getSpeciesGraftSupportCodeSection();
        List<RefVarieteGeves> gevesVarietes = getPersistenceContext().getRefVarieteGevesDao().findActiveGraftSupport(
                filter, graftSupportCodeSection, maxResults);
        result.addAll(gevesVarietes);

        // Variétés PlantGrape
        if (result.size() < maxResults) {
            String speciesGraftSupportUtilisation = getConfig().getSpeciesGraftSupportUtilisation();
            List<RefVarietePlantGrape> plantGrapeVarietes = getPersistenceContext().getRefVarietePlantGrapeDao().findGraftSupport(
                    filter, speciesGraftSupportUtilisation, maxResults - result.size());
            result.addAll(plantGrapeVarietes);
        }

        return result;
    }

    @Override
    public List<RefClonePlantGrape> findGraftClones(String speciesId, String varietyId, String filter) {
        Preconditions.checkArgument(speciesId != null);
        Preconditions.checkArgument(varietyId != null);

        // it can be something else than an RefVarietePlantGrape that's why it can be null
        RefVarietePlantGrape variety = getPersistenceContext().getRefVarietePlantGrapeDao().forTopiaIdEquals(varietyId).findUniqueOrNull();

        //Preconditions.checkState(species != null);
        //Preconditions.checkState(variety != null);

        List<RefClonePlantGrape> result = Lists.newArrayList();
        if (variety != null) {
            List<RefClonePlantGrape> gevesVarietes = getPersistenceContext().getRefClonePlantGrapeDao().findGraftClones(
                    filter, variety.getCodeVar(), 20);
            result.addAll(gevesVarietes);
        }
        return result;
    }

    @Override
    public RefLocation getRefLocation(String refLocationArvalisId) {
        RefLocation result = getPersistenceContext().getRefLocationDao().forTopiaIdEquals(refLocationArvalisId).findUnique();
        return result;
    }

    @Override
    public List<RefLocation> findActiveCommunes(String filter) {

        List<RefLocation> result = getPersistenceContext().getRefLocationDao().findActiveLocations(filter, 20);
        return result;
    }

    @Override
    public Map<Integer, String> findAllActiveOtex18Code() {
        Map<Integer, String> result = getPersistenceContext().getRefOTEXDao().findAllActiveOtex18Code();
        return result;
    }

    @Override
    public Map<Integer, String> findAllActiveCodeOtex70ByOtex18code(Integer otex18code) {
        Map<Integer, String> result = getPersistenceContext().getRefOTEXDao().findAllActiveCodeOtex70ByOtex18code(otex18code);
        return result;
    }

    @Override
    public List<RefOrientationEDI> findAllReferentielEDI() {
        List<RefOrientationEDI> result = getPersistenceContext().getRefOrientationEDIDao().newQueryBuilder().setOrderByArguments(RefOrientationEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public RefOrientationEDI findOrientation(String orientationTopiaId) {
        RefOrientationEDI result = getPersistenceContext().getRefOrientationEDIDao().forTopiaIdEquals(orientationTopiaId).findUnique();
        return result;
    }

    @Override
    public List<RefInterventionAgrosystTravailEDI> findAllActiveAgrosystActions() {
        List<RefInterventionAgrosystTravailEDI> result = findAllActiveAgrosystActions(null);
        return result;
    }

    @Override
    public List<RefInterventionAgrosystTravailEDI> findAllActiveAgrosystActions(final AgrosystInterventionType interventionType) {

        Callable<LinkedList<RefInterventionAgrosystTravailEDI>> loader = new Callable<LinkedList<RefInterventionAgrosystTravailEDI>>() {
            @Override
            public LinkedList<RefInterventionAgrosystTravailEDI> call() throws Exception {
                RefInterventionAgrosystTravailEDITopiaDao refInterventionAgrosystTravailEDIDao = getPersistenceContext().getRefInterventionAgrosystTravailEDIDao();
                LinkedList<RefInterventionAgrosystTravailEDI> result = refInterventionAgrosystTravailEDIDao.findAllActive(interventionType);
                return result;
            }
        };

        List<RefInterventionAgrosystTravailEDI> result = cacheService.get("activeAgrosystActions", interventionType, loader);
        return result;
    }

    @Override
    public RefInterventionAgrosystTravailEDI getRefInterventionAgrosystTravailEDI(String topiaId) {
        RefInterventionAgrosystTravailEDI result = getPersistenceContext().getRefInterventionAgrosystTravailEDIDao().forTopiaIdEquals(topiaId).findUnique();
        return result;
    }

    @Override
    public List<RefParcelleZonageEDI> getAllActiveParcelleZonage() {
        List<RefParcelleZonageEDI> result = getPersistenceContext().getRefParcelleZonageEDIDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<RefSolTextureGeppa> getAllActiveSolTextures() {
        List<RefSolTextureGeppa> result = getPersistenceContext().getRefSolTextureGeppaDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<RefSolProfondeurIndigo> getAllActiveSolProfondeurs() {
        List<RefSolProfondeurIndigo> result = getPersistenceContext().getRefSolProfondeurIndigoDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<RefSolCaracteristiqueIndigo> getAllActiveSolCaracteristiques() {
        List<RefSolCaracteristiqueIndigo> result = getPersistenceContext().getRefSolCaracteristiqueIndigoDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<MineralProductType> findAllActiveMineralProductTypes() {
        List<MineralProductType> result = getPersistenceContext().getRefFertiMinUNIFADao().findAllActiveFertiMinProductType();
        return result;
    }

    @Override
    public List<String> findAllActiveFertiMinShape(Integer categ) {
        List<String> result = getPersistenceContext().getRefFertiMinUNIFADao().findAllActiveFertiMinShape(categ);
        return result;
    }

    @Override
    public List<RefFertiMinUNIFA> findAllActiveRefFertiMinUnifaByCategAndShape(Integer categ, String fertilizerShape, String productId) {
        List<RefFertiMinUNIFA> result = getPersistenceContext().getRefFertiMinUNIFADao().findAllActiveRefFertiMinUnifaByCategAndShape(categ, fertilizerShape, productId);
        return result;
    }

    @Override
    public List<RefUniteEDI> findAllActiveRefUnitesEDI() {
        List<RefUniteEDI> result = getPersistenceContext().getRefUniteEDIDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<RefFertiOrga> findAllActiveOrganicProductTypes() {
        List<RefFertiOrga> result = getPersistenceContext().getRefFertiOrgaDao().findAllActive();
        return result;
    }

    @Override
    public List<RefFertiEngraisorg> findAllActiveRefFertiorgs() {
        List<RefFertiEngraisorg> result = getPersistenceContext().getRefFertiEngraisorgDao().forActiveEquals(true).findAll();
        return result;
    }

    @Override
    public List<RefStadeEdiDto> getRefStadesEdi(Integer vegetativeProfileRaw) {
        int vegetativeProfile = MoreObjects.firstNonNull(vegetativeProfileRaw, 9999);
        List<RefStadeEDI> entities = getPersistenceContext().getRefStadeEDIDao().forProperties(
                RefStadeEDI.PROPERTY_PROFIL_VEGETATIF, vegetativeProfile,
                RefStadeEDI.PROPERTY_ACTIVE, true).findAll();
        List<RefStadeEdiDto> result = Lists.newArrayList(Iterables.transform(entities, Itk.STADE_EDI_TO_DTO));
        Collections.sort(result, STADE_EDI_DTO_COMPARATOR);
        return result;
    }

    @Override
    public Map<String, String> getActaTreatementCodesAndNames() {
        Map<String, String> result = getPersistenceContext().getRefActaTraitementsProduitDao().findAllTreatementCodesAndNames();
        return result;
    }

    @Override
    public Map<AgrosystInterventionType, List<String>> getAllActiveActaTreatmentProductTypes() {

        Callable<LinkedHashMap<AgrosystInterventionType, List<String>>> loader = new Callable<LinkedHashMap<AgrosystInterventionType, List<String>>>() {
            @Override
            public LinkedHashMap<AgrosystInterventionType, List<String>> call() throws Exception {
                RefActaTraitementsProduitsCategTopiaDao dao = getPersistenceContext().getRefActaTraitementsProduitsCategDao();
                LinkedHashMap<AgrosystInterventionType, List<String>> result = dao.findAllActiveActaTreatmentProductType();
                return result;
            }
        };

        Map<AgrosystInterventionType, List<String>> result = cacheService.get("activeActaTreatmentProductTypes", loader);
        return result;
    }

    @Override
    public List<RefElementVoisinage> getAllActiveElementVoisinages() {
        List<RefElementVoisinage> result = getPersistenceContext().getRefElementVoisinageDao().forActiveEquals(true)
                .setOrderByArguments(RefElementVoisinage.PROPERTY_IAE_NOM).findAll();
        return result;
    }

    @Override
    public List<RefBioAgressor> getTreatmentTargets(BioAgressorType category) {
        List<RefBioAgressor> result = Lists.newArrayList();
        if (BioAgressorType.ADVENTICE.equals(category)) {
            Iterables.addAll(result, getPersistenceContext().getRefAdventiceDao().forProperties(RefAdventice.PROPERTY_ACTIVE, true).findAll());
        } else {
            Iterables.addAll(result, getPersistenceContext().getRefNuisibleEDIDao().forProperties(RefNuisibleEDI.PROPERTY_REFERENCE_PARAM, category, RefNuisibleEDI.PROPERTY_ACTIVE, true).findAll());
        }
        return result;
    }

    protected Integer getActaSpeciesId(String speciesId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(speciesId));

        RefEspece espece = getPersistenceContext().getRefEspeceDao().forTopiaIdEquals(speciesId).findAny();

        RefLienCulturesEdiActa acta = getPersistenceContext().getRefLienCulturesEdiActaDao().forProperties(
                RefLienCulturesEdiActa.PROPERTY_CODE_ESPECE_BOTANIQUE, espece.getCode_espece_botanique(),
                RefLienCulturesEdiActa.PROPERTY_CODE_QUALIFIANT_AEE, espece.getCode_qualifiant_AEE(),
                RefLienCulturesEdiActa.PROPERTY_CODE_TYPE_SAISONNIER_AEE, espece.getCode_type_saisonnier_AEE(),
                RefLienCulturesEdiActa.PROPERTY_CODE_DESTINATION_AEE, espece.getCode_destination_AEE(),
                RefLienCulturesEdiActa.PROPERTY_ACTIVE, true
        ).findAnyOrNull();
        Integer speciesActaId = acta == null ? null : acta.getId_culture();
        return speciesActaId;
    }


    public RefActaDosageSPC computeActaReferenceDose(String phytoProductId, String speciesId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(phytoProductId));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(speciesId));

        RefActaDosageSPC result = null;
        try {
            // Get the ACTA id_culture
            Integer actaSpeciesId = getActaSpeciesId(speciesId);
            Set<Integer> actaSpeciesIds = actaSpeciesId == null ? Sets.<Integer>newHashSet() : Sets.newHashSet(actaSpeciesId);

            // Get all related cultures/groups and add to the set
            List<RefActaGroupeCultures> culturesList = getPersistenceContext().getRefActaGroupeCulturesDao().forProperties(RefActaGroupeCultures.PROPERTY_ACTIVE, true, RefActaGroupeCultures.PROPERTY_ID_CULTURE, actaSpeciesId).findAll();

            Iterable<Integer> ids = Iterables.transform(culturesList, GET_GROUPE_ID_CULTURE);
            Iterables.addAll(actaSpeciesIds, ids);

            // do not load RefActaDosageSPC for couple phyto product x Zones cultivées #5566
            actaSpeciesIds.remove(getConfig().getActaDosageSpcCroppingZonesGroupId());

            // Ask Dao for the minimal value over all the expected acta species
            result = getPersistenceContext().getRefActaDosageSPCDao().findMinimalValue(phytoProductId, actaSpeciesIds);

            // if result is null we look for value for couple phyto product x Zones cultivées #5566
            if (result == null) {
                result = getPersistenceContext().getRefActaDosageSPCDao().forProperties(
                        RefActaDosageSPC.PROPERTY_ACTIVE, true,
                        RefActaDosageSPC.PROPERTY_ID_PRODUIT, phytoProductId,
                        RefActaDosageSPC.PROPERTY_ID_CULTURE, getConfig().getActaDosageSpcCroppingZonesGroupId()
                ).addNotNull(RefActaDosageSPC.PROPERTY_DOSAGE_SPC_VALEUR).findUniqueOrNull();
            }
        } catch (Exception eee) { // May happen if getActaSpeciesId does not match any actaSpecies
            if (log.isWarnEnabled()) {
                log.warn("Exception during reference dose computation", eee);
            }
        }

        return result;
    }

    @Override
    public RefActaDosageSPC computeActaReferenceDose(String phytoId_Produit, Set<String> speciesIds) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(phytoId_Produit));
        Preconditions.checkArgument(!speciesIds.isEmpty() && !Strings.isNullOrEmpty(speciesIds.iterator().next()));

        RefActaDosageSPC result = null;
        for (String speciesId : speciesIds) {
            RefActaDosageSPC dose = computeActaReferenceDose(phytoId_Produit, speciesId);
            // Keep the minimum value
            if (result == null || (dose != null && dose.getDosage_spc_valeur() < result.getDosage_spc_valeur())) {
                result = dose;
            }
        }
        return result;
    }

    protected RefActaDosageSPC computeActaReferenceDoseAndValidUnit(String phytoId_Produit, Set<String> speciesIds) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(phytoId_Produit));
        Preconditions.checkArgument(!speciesIds.isEmpty() && !Strings.isNullOrEmpty(speciesIds.iterator().next()));

        RefActaDosageSPC result = null;

        PhytoProductUnit commonUnit = null;
        for (String speciesId : speciesIds) {
            RefActaDosageSPC dose = computeActaReferenceDose(phytoId_Produit, speciesId);

            if (dose != null) {
                PhytoProductUnit unit = dose.getDosage_spc_unite();
                commonUnit = commonUnit == null ? unit : commonUnit;

                if (unit == commonUnit) {
                    // Keep the minimum value
                    if (result == null || (dose.getDosage_spc_valeur() < result.getDosage_spc_valeur())) {
                        result = dose;
                    }
                } else {
                    result = null;
                    break;
                }
            }


        }
        return result;
    }

    @Override
    public List<RefTraitSdC> getAllActiveGrowingSystemCharacteristics() {
        List<RefTraitSdC> result = getPersistenceContext().getRefTraitSdCDao().forActiveEquals(true)
                .setOrderByArguments(RefTraitSdC.PROPERTY_TYPE_TRAIT_SDC, RefTraitSdC.PROPERTY_NOM_TRAIT)
                .findAll();
        return result;
    }


    @Override
    public List<RefActaTraitementsProduit> getActaTraitementsProduits(AgrosystInterventionType interventionType, String productType) {
        List<RefActaTraitementsProduit> result = getPersistenceContext().getRefActaTraitementsProduitDao().findAllActiveTreatmentTypesForProductType(interventionType, productType);
        return result;
    }

    @Override
    public RefBioAgressor getBioAgressor(String bioAgressorId) {
        RefBioAgressor result = getPersistenceContext().getRefBioAgressorDao().forTopiaIdEquals(bioAgressorId).findUnique();
        return result;
    }

    @Override
    public RefDepartmentShape getDepartmentShape(String departmentCode) {
        RefDepartmentShape result = getPersistenceContext().getRefDepartmentShapeDao().forDepartmentEquals(departmentCode).findAnyOrNull();
        return result;
    }

    @Override
    public Map<String, String> getAllRefStationMeteoMap() {
        List<RefStationMeteo> list = getPersistenceContext().getRefStationMeteoDao().newQueryBuilder()
                .addEquals(RefStationMeteo.PROPERTY_ACTIVE, true)
                .setOrderByArguments(RefStationMeteo.PROPERTY_COMMUNE_SITE).findAll();
        Map<String, String> result = Maps.newLinkedHashMap();
        for (RefStationMeteo station : list) {
            result.put(station.getTopiaId(), station.getCommuneSite());
        }
        return result;
    }

    @Override
    public RefFertiMinUNIFA getNewRefFertiMinUNIFA() {
        RefFertiMinUNIFA fertiMinUNIFA = refFertiMinUNIFATopiaDao.newInstance();
        return fertiMinUNIFA;
    }

    @Override
    public boolean isValidRefFertiMinProduct(RefFertiMinUNIFA product) {
        Preconditions.checkState(product != null);
        double totalWeight = 0;
        totalWeight += product.getN() != null ? product.getN() : 0;
        totalWeight += product.getP2O5() != null ? product.getP2O5() : 0;
        totalWeight += product.getBore() != null ? product.getBore() : 0;
        totalWeight += product.getCalcium() != null ? product.getCalcium() : 0;
        totalWeight += product.getCuivre() != null ? product.getCuivre() : 0;
        totalWeight += product.getFer() != null ? product.getFer() : 0;
        totalWeight += product.getK2O() != null ? product.getK2O() : 0;
        totalWeight += product.getManganese() != null ? product.getManganese() : 0;
        totalWeight += product.getMgO() != null ? product.getMgO() : 0;
        totalWeight += product.getMolybdene() != null ? product.getMolybdene() : 0;
        totalWeight += product.getOxyde_de_sodium() != null ? product.getOxyde_de_sodium() : 0;
        totalWeight += product.getsO3() != null ? product.getsO3() : 0;
        totalWeight += product.getZinc() != null ? product.getZinc() : 0;
        boolean result = totalWeight <= 100;
        return result;
    }

    @Override
    public RefFertiMinUNIFA createOrUpdateRefMineralProductToInput(RefFertiMinUNIFA product) {
        Preconditions.checkState(product != null , isValidRefFertiMinProduct(product));
        RefFertiMinUNIFA result;
        // on retrouve le produit par rapport à sa forme, sa catégorie et sa composition et non pas par rapport à son topiaId
        RefFertiMinUNIFA existingProduct = refFertiMinUNIFATopiaDao.forNaturalId(
                product.getCateg(),
                product.getForme(),
                product.getN(),
                product.getP2O5(),
                product.getK2O(),
                product.getBore(),
                product.getCalcium(),
                product.getFer(),
                product.getManganese(),
                product.getMolybdene(),
                product.getMgO(),
                product.getOxyde_de_sodium(),
                product.getsO3(),
                product.getCuivre(),
                product.getZinc()).findUniqueOrNull();
        // product not found so it's a new product added by user.
        if (existingProduct == null) {
            product.setTopiaId(null);
            String source = StringUtils.isNotBlank(product.getSource()) ? product.getSource() : DEFAULT_SOURCE;
            product.setSource(source);
            product.setActive(true);
            result =  refFertiMinUNIFATopiaDao.create(product);
        } else if (!existingProduct.isActive()){
            existingProduct.setActive(true);
            result = refFertiMinUNIFATopiaDao.update(existingProduct);
        } else {
            result = existingProduct;
        }
        return result;
    }

    @Override
    public void importRefFertiMinUnifaPz0(Map<Class, ImportResults> allResults) {
        ImportResults results = allResults.remove(RefFertiMinUNIFA.class);
        if (results != null && results.getIgnoredRecords() == 0) {
            Collection<EntityAndDependencies> products = results.getEntityAndDepsByCsvIds().values();
            int count = 1;
            int total = products.size();
            for (EntityAndDependencies product : products) {

                long start = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info(String.format("Début sauvegarde d'un produit fertilisant minéral %d/%d.", count++, total));
                }

                RefFertiMinUNIFA entity = (RefFertiMinUNIFA) product.getEntity();
                createOrUpdateRefMineralProductToInput(entity);

                long p1 = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info("Fin de sauvegarde d'un produit fertilisant minéral, traitement réalisé en:" + (p1- start));
                }
            }
        }
    }

    @Override
    public Map<String, String> getCroppingPlanSpeciesCodeByRefEspeceAndVarietyKey(Collection<CroppingPlanSpecies> allCroppingPlanSpecies) {
        Map<String,String> speciesKeyToCode = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(allCroppingPlanSpecies)) {
            for (CroppingPlanSpecies croppingPlanSpecies : allCroppingPlanSpecies) {
                speciesKeyToCode.put(GET_REF_ESPECE_VARIETY_KEY.apply(croppingPlanSpecies), croppingPlanSpecies.getCode());
            }
        }
        return speciesKeyToCode;
    }

    @Override
    public Map<String, String> getRefEspeceAndVarietyKeyByCroppingPlanSpeciesCode(Collection<CroppingPlanSpecies> allCroppingPlanSpecies) {
        Map<String,String> speciesKeyToCode = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(allCroppingPlanSpecies)) {
            for (CroppingPlanSpecies croppingPlanSpecies : allCroppingPlanSpecies) {
                speciesKeyToCode.put(croppingPlanSpecies.getCode(), GET_REF_ESPECE_VARIETY_KEY.apply(croppingPlanSpecies));
            }
        }
        return speciesKeyToCode;
    }


}
