package fr.inra.agrosyst.services.referential;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ImportServiceImpl.java 4972 2015-06-09 13:14:05Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/referential/ImportServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.PriceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefActaDosageSPC;
import fr.inra.agrosyst.api.entities.referential.RefActaGroupeCultures;
import fr.inra.agrosyst.api.entities.referential.RefActaSubstanceActive;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitsCateg;
import fr.inra.agrosyst.api.entities.referential.RefAdventice;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolAnnuelle;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolPerenne;
import fr.inra.agrosyst.api.entities.referential.RefCultureEdiGroupeCouvSol;
import fr.inra.agrosyst.api.entities.referential.RefDepartmentShape;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceToVariete;
import fr.inra.agrosyst.api.entities.referential.RefFertiEngraisorg;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFATopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefFertiOrga;
import fr.inra.agrosyst.api.entities.referential.RefGesCarburant;
import fr.inra.agrosyst.api.entities.referential.RefGesEngrais;
import fr.inra.agrosyst.api.entities.referential.RefGesPhyto;
import fr.inra.agrosyst.api.entities.referential.RefGesSemence;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLienCulturesEdiActa;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referential.RefMaterielIrrigation;
import fr.inra.agrosyst.api.entities.referential.RefMaterielOutil;
import fr.inra.agrosyst.api.entities.referential.RefMaterielTraction;
import fr.inra.agrosyst.api.entities.referential.RefMesure;
import fr.inra.agrosyst.api.entities.referential.RefNrjCarburant;
import fr.inra.agrosyst.api.entities.referential.RefNrjEngrais;
import fr.inra.agrosyst.api.entities.referential.RefNrjGesOutil;
import fr.inra.agrosyst.api.entities.referential.RefNrjPhyto;
import fr.inra.agrosyst.api.entities.referential.RefNrjSemence;
import fr.inra.agrosyst.api.entities.referential.RefNuisibleEDI;
import fr.inra.agrosyst.api.entities.referential.RefOTEX;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefPhytoSubstanceActiveIphy;
import fr.inra.agrosyst.api.entities.referential.RefProtocoleVgObs;
import fr.inra.agrosyst.api.entities.referential.RefSaActaIphy;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalis;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalisTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolCaracteristiqueIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefStadeEDI;
import fr.inra.agrosyst.api.entities.referential.RefStadeNuisibleEDI;
import fr.inra.agrosyst.api.entities.referential.RefStationMeteo;
import fr.inra.agrosyst.api.entities.referential.RefSupportOrganeEDI;
import fr.inra.agrosyst.api.entities.referential.RefTraitSdC;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgriculture;
import fr.inra.agrosyst.api.entities.referential.RefTypeNotationEDI;
import fr.inra.agrosyst.api.entities.referential.RefTypeTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefUniteEDI;
import fr.inra.agrosyst.api.entities.referential.RefUnitesQualifiantEDI;
import fr.inra.agrosyst.api.entities.referential.RefValeurQualitativeEDI;
import fr.inra.agrosyst.api.entities.referential.RefVarieteGeves;
import fr.inra.agrosyst.api.entities.referential.RefVarietePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefZoneClimatiqueIphy;
import fr.inra.agrosyst.api.entities.referential.ReferentialEntity;
import fr.inra.agrosyst.api.entities.referential.Referentials;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoCaseGroundWater;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoFuzzySetGroundWater;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoRulesGroundWater;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesuRunoffPotRulesParc;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.referential.ImportResult;
import fr.inra.agrosyst.api.services.referential.ImportService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.CacheService;
import fr.inra.agrosyst.services.referential.csv.CommuneInseeModel;
import fr.inra.agrosyst.services.referential.csv.CommunePostCodeModel;
import fr.inra.agrosyst.services.referential.csv.CommuneRegionAgricoleModel;
import fr.inra.agrosyst.services.referential.csv.CommunesPostCodeOsmModel;
import fr.inra.agrosyst.services.referential.csv.RefActaDosageSPCModel;
import fr.inra.agrosyst.services.referential.csv.RefActaGroupeCulturesModel;
import fr.inra.agrosyst.services.referential.csv.RefActaSubstanceActiveModel;
import fr.inra.agrosyst.services.referential.csv.RefActaTraitementsProduitModel;
import fr.inra.agrosyst.services.referential.csv.RefActaTraitementsProduitsCategModel;
import fr.inra.agrosyst.services.referential.csv.RefAdventiceModel;
import fr.inra.agrosyst.services.referential.csv.RefClonePlantGrapeModel;
import fr.inra.agrosyst.services.referential.csv.RefCouvSolAnnuelleModel;
import fr.inra.agrosyst.services.referential.csv.RefCouvSolPerenneModel;
import fr.inra.agrosyst.services.referential.csv.RefCultureEdiGroupeCouvSolModel;
import fr.inra.agrosyst.services.referential.csv.RefDepartmentShapeModel;
import fr.inra.agrosyst.services.referential.csv.RefElementVoisinageModel;
import fr.inra.agrosyst.services.referential.csv.RefEspeceModel;
import fr.inra.agrosyst.services.referential.csv.RefEspeceToVarieteModel;
import fr.inra.agrosyst.services.referential.csv.RefFertiEngraisOrgModel;
import fr.inra.agrosyst.services.referential.csv.RefFertiMinUNIFAModel;
import fr.inra.agrosyst.services.referential.csv.RefFertiOrgaModel;
import fr.inra.agrosyst.services.referential.csv.RefGesCarburantModel;
import fr.inra.agrosyst.services.referential.csv.RefGesEngraisModel;
import fr.inra.agrosyst.services.referential.csv.RefGesPhytoModel;
import fr.inra.agrosyst.services.referential.csv.RefGesSemenceModel;
import fr.inra.agrosyst.services.referential.csv.RefInterventionAgrosystTravailEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefLegalStatusModel;
import fr.inra.agrosyst.services.referential.csv.RefLienCulturesEdiActaModel;
import fr.inra.agrosyst.services.referential.csv.RefLocationDto;
import fr.inra.agrosyst.services.referential.csv.RefMaterielAutomoteurModel;
import fr.inra.agrosyst.services.referential.csv.RefMaterielIrrigationModel;
import fr.inra.agrosyst.services.referential.csv.RefMaterielOutilModel;
import fr.inra.agrosyst.services.referential.csv.RefMaterielTracteurModel;
import fr.inra.agrosyst.services.referential.csv.RefMesureModel;
import fr.inra.agrosyst.services.referential.csv.RefNrjCarburantModel;
import fr.inra.agrosyst.services.referential.csv.RefNrjEngraisModel;
import fr.inra.agrosyst.services.referential.csv.RefNrjGesOutilModel;
import fr.inra.agrosyst.services.referential.csv.RefNrjPhytoModel;
import fr.inra.agrosyst.services.referential.csv.RefNrjSemenceModel;
import fr.inra.agrosyst.services.referential.csv.RefNuisibleEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefOTEXModel;
import fr.inra.agrosyst.services.referential.csv.RefOrientationEdiModel;
import fr.inra.agrosyst.services.referential.csv.RefParcelleZonageEdiModel;
import fr.inra.agrosyst.services.referential.csv.RefPhytoSubstanceActiveIphyModel;
import fr.inra.agrosyst.services.referential.csv.RefProtocoleVgObsModel;
import fr.inra.agrosyst.services.referential.csv.RefSaActaIphyModel;
import fr.inra.agrosyst.services.referential.csv.RefSolArvalisModel;
import fr.inra.agrosyst.services.referential.csv.RefSolCaracteristiqueIndigoModel;
import fr.inra.agrosyst.services.referential.csv.RefSolProfondeurIndigoModel;
import fr.inra.agrosyst.services.referential.csv.RefSolTextureGeppaModel;
import fr.inra.agrosyst.services.referential.csv.RefStadeEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefStadeNuisibleEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefStationMeteoModel;
import fr.inra.agrosyst.services.referential.csv.RefSupportOrganeEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefTraitSdCModel;
import fr.inra.agrosyst.services.referential.csv.RefTypeAgricultureModel;
import fr.inra.agrosyst.services.referential.csv.RefTypeNotationEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefTypeTravailEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefUniteEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefUnitesQualifiantEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefValeurQualitativeEDIModel;
import fr.inra.agrosyst.services.referential.csv.RefVarieteGevesModel;
import fr.inra.agrosyst.services.referential.csv.RefVarietePlantGrapeModel;
import fr.inra.agrosyst.services.referential.csv.RefZoneClimatiqueIphyModel;
import fr.inra.agrosyst.services.referential.csv.RegionDto;
import fr.inra.agrosyst.services.referential.csv.SolsArvalisRegionsModel;
import fr.inra.agrosyst.services.referential.csv.iphy.RefRcesoCaseGroundWaterModel;
import fr.inra.agrosyst.services.referential.csv.iphy.RefRcesoFuzzySetGroundWaterModel;
import fr.inra.agrosyst.services.referential.csv.iphy.RefRcesoRulesGroundWaterModel;
import fr.inra.agrosyst.services.referential.csv.iphy.RefRcesuRunoffPotRulesParcModel;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;
import org.nuiton.csv.ImportModel;
import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class ImportServiceImpl extends AbstractAgrosystService implements ImportService {

    private static final Log log = LogFactory.getLog(ImportServiceImpl.class);

    public static final Function<RefLocationDto, String> GET_REF_LOCATION_DTO_CODE_INSEE = new Function<RefLocationDto, String>() {
        @Override
        public String apply(RefLocationDto input) {
            String rawCodeInsee = input.getCodeInsee();
            if (rawCodeInsee == null) {
                String codeDepartement = input.getDepartement();
                String codeCommune = input.getCodeCommune();
                if (codeDepartement.length() == 3) { // Cas particulier des DOM/TOM
                    codeCommune = Strings.padStart(codeCommune, 2, '0');
                } else {
                    codeCommune = Strings.padStart(codeCommune, 3, '0');
                }
                rawCodeInsee = codeDepartement + codeCommune;
            }
            String result = Strings.padStart(rawCodeInsee, 5, '0');
            return result;
        }
    };

    public static final Function<RefLocationDto, String> GET_REF_LOCATION_DTO_PRETTY_COMMUNE = new Function<RefLocationDto, String>() {
        @Override
        public String apply(RefLocationDto input) {
            String result = input.getNomCommune();
            String article = input.getArticleCommune();
            if (StringUtils.isNotBlank(article)) {
                article = article.trim();
                if (article.startsWith("(")) {
                    article = article.substring(1);
                }
                if (article.endsWith(")")) {
                    article = article.substring(0, article.length() - 1);
                }
                if (!article.endsWith("'")) {
                    article += " ";
                }
                result = article + result;
            }
            return result;
        }
    };

    protected static final Function<RegionDto, String> GET_REGION_NAME = new Function<RegionDto, String>() {
        @Override
        public String apply(RegionDto input) {
            return input.getName();
        }
    };

    protected RefLocationTopiaDao refLocationDao;
    protected RefSolArvalisTopiaDao refSolArvalisDao;
    protected RefFertiMinUNIFATopiaDao refFertiMinUNIFATopiaDao;
    protected PriceTopiaDao priceTopiaDao;
    protected CacheService cacheService;


    public void setRefLocationDao(RefLocationTopiaDao refLocationDao) {
        this.refLocationDao = refLocationDao;
    }

    public void setRefSolArvalisDao(RefSolArvalisTopiaDao refSolArvalisDao) {
        this.refSolArvalisDao = refSolArvalisDao;
    }

    public void setRefFertiMinUNIFATopiaDao(RefFertiMinUNIFATopiaDao refFertiMinUNIFATopiaDao) {
        this.refFertiMinUNIFATopiaDao = refFertiMinUNIFATopiaDao;
    }

    public void setPriceTopiaDao(PriceTopiaDao priceTopiaDao) {
        this.priceTopiaDao = priceTopiaDao;
    }

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    protected <T extends ReferentialEntity> ImportResult runSimpleImport(InputStream input,
                                                                         Class<T> entityClass,
                                                                         ImportModel<T> model,
                                                                         Function<T, String> getNaturalIdFunction) {

        Import<T> importer = null;

        ImportResult result = new ImportResult();
        long start = System.currentTimeMillis();

        String entityClassName = entityClass.getSimpleName();
        try {
            // declare dao
            TopiaDao<T> dao = context.getDaoSupplier().getDao(entityClass);

            List<T> all = dao.findAll();
            Map<String, T> map = Maps.uniqueIndex(all, getNaturalIdFunction);
            Set<String> alreadyDone = Sets.newHashSet();

            importer = Import.newImport(model, input);
            Binder<T, T> binder = BinderFactory.newBinder(entityClass);
            int line = 0;
            for (T entity : importer) {
                entity.setActive(true);
                String id = getNaturalIdFunction.apply(entity);
                if (alreadyDone.contains(id)) {
                    if (log.isWarnEnabled()) {
                        String message = String.format("Duplicate entry found (type=%s, L%d): %s", entityClassName, line, id);
                        log.warn(message);
                    }
                    result.incIgnored();
                } else {
                    alreadyDone.add(id);
                    if (map.containsKey(id)) {
                        T existing = map.get(id);
                        binder.copyExcluding(entity, existing, TopiaEntity.PROPERTY_TOPIA_ID,
                                TopiaEntity.PROPERTY_TOPIA_CREATE_DATE, TopiaEntity.PROPERTY_TOPIA_VERSION);
                        dao.update(existing);
                        result.incUpdated();
                    } else {
                        dao.create(entity);
                        result.incCreated();
                    }
                }
                line++;
            }

            getTransaction().commit();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Error during import: " + eee.getMessage(), eee);
            }
            result.addError(eee.getMessage());
        } finally {
            IOUtils.closeQuietly(importer);
            IOUtils.closeQuietly(input);
        }
        long end = System.currentTimeMillis();
        result.setDuration(end - start);
        if (log.isInfoEnabled()) {
            String message = String.format("Import finished (type=%s). Result: %s", entityClassName, result);
            log.info(message);
        }

        // At the end of each import, make sure to invalidate caches
        cacheService.clear();

        return result;
    }

    @Override
    public ImportResult importMaterielTracteursCSV(InputStream contentStream) {
        // declare nuiton csv import model
        ImportModel<RefMaterielTraction> csvModel = new RefMaterielTracteurModel();

        ImportResult result = runSimpleImport(contentStream, RefMaterielTraction.class, csvModel, Referentials.GET_MATERIEL_TRACTION_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importMaterielAutomoteursCSV(InputStream contentStream) {
        // declare nuiton csv import model
        ImportModel<RefMaterielAutomoteur> csvModel = new RefMaterielAutomoteurModel();

        ImportResult result = runSimpleImport(contentStream, RefMaterielAutomoteur.class, csvModel, Referentials.GET_MATERIEL_AUTOMOTEUR_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importMaterielOutilsCSV(InputStream contentStream) {
        // declare nuiton csv import model
        ImportModel<RefMaterielOutil> csvModel = new RefMaterielOutilModel();

        ImportResult result = runSimpleImport(contentStream, RefMaterielOutil.class, csvModel, Referentials.GET_MATERIEL_OUTIL_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importMaterielIrrigationCSV(InputStream contentStream) {

        ImportModel<RefMaterielIrrigation> csvModel = new RefMaterielIrrigationModel();

        ImportResult result = runSimpleImport(contentStream, RefMaterielIrrigation.class, csvModel, Referentials.GET_MATERIEL_IRRIGATION_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importCommuneInseeCSV(InputStream communesStream, InputStream postCodesStream, InputStream raStream) {

        // déclaration du modèle d'import nuiton-csv
        ImportModel<RefLocationDto> csvModel = new CommuneInseeModel();
        Import<RefLocationDto> importer = null;

        ImportResult result = new ImportResult();
        long start = System.currentTimeMillis();

        try {

            // Lecture des 2 autres flux
            Map<String, RefLocationDto> postCodes = importCommunePostCodesCSV(postCodesStream);
            Multimap<String, RefLocationDto> regionsAgricoles = importCommuneRegionAgricoleCSV(raStream);

            // declare dao
            List<RefLocation> all = refLocationDao.findAll();
            Map<String, RefLocation> map = Maps.uniqueIndex(all, Referentials.GET_LOCATION_CODE_INSEE);

            // import data
            importer = Import.newImport(csvModel, new InputStreamReader(communesStream, getConfig().getFileEncoding()));
            for (RefLocationDto dto : importer) {

                // compute the codeInsee
                String codeInsee = GET_REF_LOCATION_DTO_CODE_INSEE.apply(dto);
                dto.setCodeInsee(codeInsee);

                RefLocation location;
                if (map.containsKey(codeInsee)) {
                    location = map.get(codeInsee);
                } else {
                    location = refLocationDao.newInstance();
                }
                location.setCodeInsee(codeInsee);
                location.setActive(true);

                // commune
                String commune = GET_REF_LOCATION_DTO_PRETTY_COMMUNE.apply(dto);
                location.setCommune(commune);

                // region & departement
                int region = dto.getRegion();
                String departement = dto.getDepartement();
                location.setRegion(region);
                location.setDepartement(departement);

                String codeInseeForPostCodes = codeInsee.replace("2A", "20").replace("2B", "20");
                if (postCodes.containsKey(codeInseeForPostCodes)) {
                    RefLocationDto codePostalDto = postCodes.get(codeInseeForPostCodes);
                    location.setCodePostal(Strings.padStart(codePostalDto.getCodePostal(), 5, '0'));
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Pas de code postal pour la commune: " + dto);
                    }
                }

                if (regionsAgricoles.containsKey(codeInsee)) {
                    Collection<RefLocationDto> raDtos = regionsAgricoles.get(codeInsee);
                    if (raDtos.size() > 1) {
                        if (log.isWarnEnabled()) {
                            log.warn("Plusieurs régions agricoles pour ce code INSEE. On prend la première." + raDtos);
                        }
                    }
                    RefLocationDto raDto = raDtos.iterator().next();
                    location.setPetiteRegionAgricoleCode(raDto.getCodePetiteRegionAgricole());
                    location.setPetiteRegionAgricoleNom(raDto.getNomPetiteRegionAgricole());
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Pas de région agricole pour la commune: " + dto);
                    }
                }

                if (location.isPersisted()) {
                    refLocationDao.update(location);
                    result.incUpdated();
                } else {
                    refLocationDao.create(location);
                    result.incCreated();
                }
            }

            getTransaction().commit();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Error during import: " + eee.getMessage(), eee);
            }
            result.addError(eee.getMessage());
        } finally {
            IOUtils.closeQuietly(importer);
        }
        long end = System.currentTimeMillis();
        result.setDuration(end - start);
        if (log.isInfoEnabled()) {
            log.info("Import communes INSEE terminé: " + result);
        }
        return result;
    }

    @Override
    public ImportResult importCommuneOsmCSV(InputStream communesAndPostCodesStream, InputStream raStream) {

        // déclaration du modèle d'import nuiton-csv
        ImportModel<RefLocationDto> csvModel = new CommunesPostCodeOsmModel();
        Import<RefLocationDto> importer = null;

        ImportResult result = new ImportResult();
        long start = System.currentTimeMillis();

        try {

            // Lecture des 2 autres flux
            Multimap<String, RefLocationDto> regionsAgricoles = importCommuneRegionAgricoleCSV(raStream);

            // declare dao
            List<RefLocation> all = refLocationDao.findAll();
            Map<String, RefLocation> map = Maps.uniqueIndex(all, Referentials.GET_LOCATION_CODE_INSEE);

            // import data
            importer = Import.newImport(csvModel, new InputStreamReader(communesAndPostCodesStream, getConfig().getFileEncoding()));
            for (RefLocationDto dto : importer) {

                // compute the codeInsee
                String codeInsee = GET_REF_LOCATION_DTO_CODE_INSEE.apply(dto);
                dto.setCodeInsee(codeInsee);

                RefLocation location;
                if (map.containsKey(codeInsee)) {
                    location = map.get(codeInsee);
                } else {
                    location = refLocationDao.newInstance();
                }
                location.setCodeInsee(codeInsee);
                location.setActive(true);

                // geoloc
                location.setLatitude(dto.getLatitude());
                location.setLongitude(dto.getLongitude());

                // commune
                String commune = GET_REF_LOCATION_DTO_PRETTY_COMMUNE.apply(dto);
                location.setCommune(commune);

                // region & departement
                int region = dto.getRegion();
                String departement = dto.getDepartement();
                String codePostal = dto.getCodePostal();
                location.setRegion(region);
                location.setDepartement(departement);
                codePostal = codePostal.split(";")[0];
                location.setCodePostal(Strings.padStart(codePostal, 5, '0'));

                if (regionsAgricoles.containsKey(codeInsee)) {
                    Collection<RefLocationDto> raDtos = regionsAgricoles.get(codeInsee);
                    if (raDtos.size() > 1) {
                        if (log.isWarnEnabled()) {
                            log.warn("Plusieurs régions agricoles pour ce code INSEE. On prend la première." + raDtos);
                        }
                    }
                    RefLocationDto raDto = raDtos.iterator().next();
                    location.setPetiteRegionAgricoleCode(raDto.getCodePetiteRegionAgricole());
                    location.setPetiteRegionAgricoleNom(raDto.getNomPetiteRegionAgricole());
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Pas de région agricole pour la commune: " + dto);
                    }
                }

                if (location.isPersisted()) {
                    refLocationDao.update(location);
                    result.incUpdated();
                } else {
                    refLocationDao.create(location);
                    result.incCreated();
                }
            }

            getTransaction().commit();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Error during import: " + eee.getMessage(), eee);
            }
            result.addError(eee.getMessage());
        } finally {
            IOUtils.closeQuietly(importer);
        }
        long end = System.currentTimeMillis();
        result.setDuration(end - start);
        if (log.isInfoEnabled()) {
            log.info("Import communes OSM terminé: " + result);
        }
        return result;
    }

    protected ImmutableMap<String, RefLocationDto> importCommunePostCodesCSV(InputStream postCodesStream) throws UnsupportedEncodingException {
        // déclaration du modèle d'import nuiton-csv
        ImportModel<RefLocationDto> csvModel = new CommunePostCodeModel();
        Import<RefLocationDto> importer = null;

        try {
            // lecture du CSV et création de l'index
            importer = Import.newImport(csvModel, new InputStreamReader(postCodesStream, getConfig().getFileEncoding()));
            ImmutableMap<String, RefLocationDto> result = Maps.uniqueIndex(importer, GET_REF_LOCATION_DTO_CODE_INSEE);
            return result;
        } finally {
            IOUtils.closeQuietly(importer);
        }
    }

    protected ImmutableMultimap<String, RefLocationDto> importCommuneRegionAgricoleCSV(InputStream raStream) throws UnsupportedEncodingException {
        // déclaration du modèle d'import nuiton-csv
        ImportModel<RefLocationDto> csvModel = new CommuneRegionAgricoleModel();
        Import<RefLocationDto> importer = null;

        try {
            // lecture du CSV et création de l'index
            importer = Import.newImport(csvModel, new InputStreamReader(raStream, getConfig().getFileEncoding()));
            ImmutableMultimap<String, RefLocationDto> result = Multimaps.index(importer, GET_REF_LOCATION_DTO_CODE_INSEE);

            return result;
        } finally {
            IOUtils.closeQuietly(importer);
        }
    }

    @Override
    public ImportResult importSolArvalisCSV(InputStream solsStream, InputStream regionsStreamRaw) {

        ImportResult result = new ImportResult();
        Import<RegionDto> importRegion = null;
        Import<RefSolArvalis> importerSol = null;
        InputStream regionsStream = null;
        
        long start = System.currentTimeMillis();
        try {
            regionsStream = regionsStreamRaw;
            if (regionsStream == null) {
                regionsStream = ImportServiceImpl.class.getResourceAsStream("/referentiels/sols_regions.csv");
            }
            
            // first import all region into memory
            ImportModel<RegionDto> csvModelRegion = new SolsArvalisRegionsModel();
            importRegion = Import.newImport(csvModelRegion, regionsStream);
            Map<String, RegionDto> regionMaps = Maps.uniqueIndex(importRegion, GET_REGION_NAME);

            List<RefSolArvalis> all = refSolArvalisDao.findAll();
            Map<String, RefSolArvalis> map = Maps.uniqueIndex(all, Referentials.GET_SOL_ARVALIS_NATURAL_ID);

            // import data
            ImportModel<RefSolArvalis> csvModelSol = new RefSolArvalisModel();
            importerSol = Import.newImport(csvModelSol, new InputStreamReader(solsStream, getConfig().getFileEncoding()));
            Binder<RefSolArvalis, RefSolArvalis> binder = BinderFactory.newBinder(RefSolArvalis.class);
            for (RefSolArvalis refSolsArvalis : importerSol) {
                refSolsArvalis.setActive(true);
                RegionDto region = regionMaps.get(refSolsArvalis.getSol_region());
                if (region == null) {
                    throw new AgrosystTechnicalException("No such region " + refSolsArvalis.getSol_region() + " in sols region file");
                }
                refSolsArvalis.setSol_region_code(region.getCode());
                String id = Referentials.GET_SOL_ARVALIS_NATURAL_ID.apply(refSolsArvalis);
                if (map.containsKey(id)) {
                    RefSolArvalis existingSol = map.get(id);
                    binder.copyExcluding(refSolsArvalis, existingSol, TopiaEntity.PROPERTY_TOPIA_ID,
                            TopiaEntity.PROPERTY_TOPIA_CREATE_DATE, TopiaEntity.PROPERTY_TOPIA_VERSION);
                    refSolArvalisDao.update(existingSol);
                    result.incUpdated();
                } else {
                    refSolArvalisDao.create(refSolsArvalis);
                    result.incCreated();
                }
            }

            getTransaction().commit();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Error during import: " + eee.getMessage(), eee);
            }
            result.addError(eee.getMessage());
        } finally {
            IOUtils.closeQuietly(regionsStream);
            IOUtils.closeQuietly(importerSol);
            IOUtils.closeQuietly(importRegion);
            IOUtils.closeQuietly(solsStream);
            IOUtils.closeQuietly(regionsStreamRaw);
        }
        long end = System.currentTimeMillis();
        result.setDuration(end - start);
        if (log.isInfoEnabled()) {
            log.info("Import Sols Arvalis: " + result);
        }
        return result;
    }

    @Override
    public ImportResult importLegalStatusCSV(InputStream statusStream) {
        ImportResult result = runSimpleImport(
                statusStream,
                RefLegalStatus.class,
                new RefLegalStatusModel(),
                Referentials.GET_LEGAL_STATUS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importEspeces(InputStream stream) {
        ImportResult result = runSimpleImport(
                stream,
                RefEspece.class,
                new RefEspeceModel(),
                Referentials.GET_ESPECE_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importVarietesGeves(InputStream stream) {
        ImportResult result = runSimpleImport(
                stream,
                RefVarieteGeves.class,
                new RefVarieteGevesModel(),
                Referentials.GET_VARIETE_GEVES_NATURAL_ID);
        
        return result;
    }

    @Override
    public ImportResult importVarietesPlantGrape(InputStream stream) {
        ImportResult result = runSimpleImport(
                stream,
                RefVarietePlantGrape.class,
                new RefVarietePlantGrapeModel(),
                Referentials.GET_VARIETE_PLANT_GRAPE_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importClonesPlantGrape(InputStream stream) {
        ImportResult result = runSimpleImport(
                stream,
                RefClonePlantGrape.class,
                new RefClonePlantGrapeModel(),
                Referentials.GET_CLONE_PLANT_GRAPE_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importEspecesToVarietes(InputStream stream) {
        ImportResult result = runSimpleImport(
                stream,
                RefEspeceToVariete.class,
                new RefEspeceToVarieteModel(),
                Referentials.GET_ESPECE_TO_VARIETE_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importOtexCSV(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefOTEX.class,
                new RefOTEXModel(),
                Referentials.GET_OTEX_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importOrientationEdiCSV(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefOrientationEDI.class,
                new RefOrientationEdiModel(),
                Referentials.GET_ORIENTATION_EDI_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importInterventionAgrosystTravailEdiCSV(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefInterventionAgrosystTravailEDI.class,
                new RefInterventionAgrosystTravailEDIModel(),
                Referentials.GET_INTERVENTION_AGROSYST_TRAVAIL_EDI_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importTypeTravailEdiCSV(InputStream contentStream){
        ImportResult result = runSimpleImport(
                contentStream,
                RefTypeTravailEDI.class,
                new RefTypeTravailEDIModel(),
                Referentials.GET_TYPE_TRAVAIL_EDI_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importStadesEdiCSV(InputStream contentStream){
        ImportResult result = runSimpleImport(
                contentStream,
                RefStadeEDI.class,
                new RefStadeEDIModel(),
                Referentials.GET_STADE_EDI_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importSolTextureGeppa(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefSolTextureGeppa.class,
                new RefSolTextureGeppaModel(),
                Referentials.GET_SOL_TEXTURE_GEPPA_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importZonageParcelleEdi(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefParcelleZonageEDI.class,
                new RefParcelleZonageEdiModel(),
                Referentials.GET_PARCELLE_ZONAGE_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importSolProfondeurIndigo(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefSolProfondeurIndigo.class,
                new RefSolProfondeurIndigoModel(),
                Referentials.GET_SOL_PROFONDEUR_INDIGO_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importSolCarateristiquesIndigo(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefSolCaracteristiqueIndigo.class,
                new RefSolCaracteristiqueIndigoModel(),
                Referentials.GET_SOL_CARACTERISTIQUES_INDIGO_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importUniteEDI(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefUniteEDI.class,
                new RefUniteEDIModel(),
                Referentials.GET_UNITES_EDI_NATURAL_ID);
        return result;
    }

    protected Map<String, List<Price>> getRefFertiMinIndexedPrices() {
        Map<String, List<Price>> result = new HashMap<String, List<Price>>();
        List<Price> prices = priceTopiaDao.getPricesForObjectIdLike(RefFertiMinUNIFA.PROPERTY_CATEG + "%");
        for (Price price : prices) {
            List<Price> objectPrices = result.get(price.getObjectId());
            if (objectPrices == null) {
                objectPrices = Lists.newArrayList();
                result.put(price.getObjectId(), objectPrices);
            }
            objectPrices.add(price);
        }
        return result;
    }

    @Override
    public ImportResult importFertiMinUNIFA(InputStream contentStream) {
        Import<RefFertiMinUNIFA> importerRefFertiMinUNIFA;
        ImportResult result = new ImportResult();
        long start = System.currentTimeMillis();

        // import data
        try {
            ImportModel<RefFertiMinUNIFA> csvModelRefFertiMinUNIFAModel = new RefFertiMinUNIFAModel();
            importerRefFertiMinUNIFA = Import.newImport(csvModelRefFertiMinUNIFAModel, new InputStreamReader(contentStream, getConfig().getFileEncoding()));

            List<RefFertiMinUNIFA> persistedFertiMinUnifas = refFertiMinUNIFATopiaDao.findAll();
            Map<String, RefFertiMinUNIFA> fertiMinUNIFAMapWithNull = Maps.newHashMap(Maps.uniqueIndex(persistedFertiMinUnifas, Referentials.GET_FERTI_MIN_UNIFA_NATURAL_ID_WITH_NULL));
            Binder<RefFertiMinUNIFA, RefFertiMinUNIFA> binder = BinderFactory.newBinder(RefFertiMinUNIFA.class);

            List<RefFertiMinUNIFA> productToCreates = Lists.newArrayList();
            List<RefFertiMinUNIFA> productToUpdates = Lists.newArrayList();
            List<Price> pricesToUpdates = Lists.newArrayList();

            Map<String, List<Price>> objectPrices = getRefFertiMinIndexedPrices();

            for (RefFertiMinUNIFA importedData : importerRefFertiMinUNIFA) {
                    RefFertiMinUNIFA validProduct = fertiMinUNIFAMapWithNull.get(Referentials.GET_FERTI_MIN_UNIFA_NATURAL_ID_WITH_NULL.apply(importedData));

                    if (validProduct != null) {
                        binder.copyExcluding(importedData, validProduct, RefFertiMinUNIFA.PROPERTY_TOPIA_ID, RefFertiMinUNIFA.PROPERTY_TOPIA_CREATE_DATE, RefFertiMinUNIFA.PROPERTY_TOPIA_VERSION);
                        validProduct.setActive(true);
                        productToUpdates.add(validProduct);
                        result.incUpdated();

                        List<Price> prices = objectPrices.get(validProduct.getTopiaId());
                        if (prices != null) {
                            for (Price price : prices) {
                                price.setObjectId(StringUtils.stripAccents(Referentials.GET_FERTI_MIN_UNIFA_NATURAL_ID_WITH_NULL.apply(importedData)));
                                pricesToUpdates.add(price);
                            }
                        }

                    } else {
                        productToCreates.add(importedData);
                        result.incCreated();
                        fertiMinUNIFAMapWithNull.put(Referentials.GET_FERTI_MIN_UNIFA_NATURAL_ID_WITH_NULL.apply(importedData), importedData);
                    }
            }

            refFertiMinUNIFATopiaDao.updateAll(productToUpdates);
            refFertiMinUNIFATopiaDao.createAll(productToCreates);
            priceTopiaDao.updateAll(pricesToUpdates);

            getTransaction().commit();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Error during import: " + e.getMessage(), e);
            }
            result.addError(e.getMessage());
        } finally {
            IOUtils.closeQuietly(contentStream);
        }
        long end = System.currentTimeMillis();
        result.setDuration(end - start);
        if (log.isInfoEnabled()) {
            log.info("Import FertiMinUnifa: " + result);
        }

        return result;
    }

    @Override
    public ImportResult importAdventices(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefAdventice.class,
                new RefAdventiceModel(),
                Referentials.GET_ADVENTICES_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNuisiblesEDI(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNuisibleEDI.class,
                new RefNuisibleEDIModel(),
                Referentials.GET_NUISIBLES_EDI_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importFertiOrga(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefFertiOrga.class,
                new RefFertiOrgaModel(),
                Referentials.GET_FERTI_ORGA_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importFertiEngraisOrg(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefFertiEngraisorg.class,
                new RefFertiEngraisOrgModel(),
                Referentials.GET_FERTI_ENGRAISORG_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importStationMeteo(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefStationMeteo.class,
                new RefStationMeteoModel(),
                Referentials.GET_STATION_METEO_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importTypeAgriculture(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefTypeAgriculture.class,
                new RefTypeAgricultureModel(),
                Referentials.GET_TYPE_AGRICULTURE_NATURAL_ID);
        return result;
    }
 
    @Override
    public ImportResult importGesCarburants(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefGesCarburant.class,
                new RefGesCarburantModel(),
                Referentials.GET_GES_CARBURANTS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importGesEngrais(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefGesEngrais.class,
                new RefGesEngraisModel(),
                Referentials.GET_GES_ENGRAIS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importGesPhyto(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefGesPhyto.class,
                new RefGesPhytoModel(),
                Referentials.GET_GES_PHYTO_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importGesSemences(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefGesSemence.class,
                new RefGesSemenceModel(),
                Referentials.GET_GES_SEMENCES_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNrjCarburants(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNrjCarburant.class,
                new RefNrjCarburantModel(),
                Referentials.GET_NRJ_CARBURANTS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNrjEngrais(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNrjEngrais.class,
                new RefNrjEngraisModel(),
                Referentials.GET_NRJ_ENGRAIS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNrjPhyto(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNrjPhyto.class,
                new RefNrjPhytoModel(),
                Referentials.GET_NRJ_PHYTO_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNrjSemences(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNrjSemence.class,
                new RefNrjSemenceModel(),
                Referentials.GET_NRJ_SEMENCES_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importNrjGesOutils(InputStream contentStream) {
        ImportResult result = runSimpleImport(
                contentStream,
                RefNrjGesOutil.class,
                new RefNrjGesOutilModel(),
                Referentials.GET_NRJ_GES_OUTILS_NATURAL_ID);
        return result;
    }

    @Override
    public ImportResult importMesure(InputStream stream) {
        return runSimpleImport(
                stream,
                RefMesure.class,
                new RefMesureModel(),
                Referentials.GET_MESURE_NATURAL_ID);
    }

    @Override
    public ImportResult importSupportOrganeEDI(InputStream stream) {
        return runSimpleImport(
                stream,
                RefSupportOrganeEDI.class,
                new RefSupportOrganeEDIModel(),
                Referentials.GET_SUPPORT_ORGANE_EDI_NATURAL_ID);
    }

    @Override
    public ImportResult importStadeNuisibleEDI(InputStream stream) {
        return runSimpleImport(
                stream,
                RefStadeNuisibleEDI.class,
                new RefStadeNuisibleEDIModel(),
                Referentials.GET_STADE_NUISIBLE_EDI_NATURAL_ID);
    }

    @Override
    public ImportResult importTypeNotationEDI(InputStream stream) {
        return runSimpleImport(
                stream,
                RefTypeNotationEDI.class,
                new RefTypeNotationEDIModel(),
                Referentials.GET_TYPE_NOTATION_EDI_NATURAL_ID);
    }

    @Override
    public ImportResult importValeurQualitativeEDI(InputStream stream) {
        return runSimpleImport(
                stream,
                RefValeurQualitativeEDI.class,
                new RefValeurQualitativeEDIModel(),
                Referentials.GET_VALEUR_QUALITATIVE_EDI_NATURAL_ID);
    }

    @Override
    public ImportResult importUnitesQualifiantEDI(InputStream stream) {
        return runSimpleImport(
                stream,
                RefUnitesQualifiantEDI.class,
                new RefUnitesQualifiantEDIModel(),
                Referentials.GET_UNITES_QUALIFIANT_EDI_NATURAL_ID);
    }

    @Override
    public ImportResult importActaTraitementsProduits(InputStream stream) {
        return runSimpleImport(
                stream,
                RefActaTraitementsProduit.class,
                new RefActaTraitementsProduitModel(),
                Referentials.GET_ACTA_TRAITEMENTS_PRODUITS_NATURAL_ID
        );
    }

    @Override
    public ImportResult importActaTraitementsProduitsCateg(InputStream stream) {
        return runSimpleImport(
                stream,
                RefActaTraitementsProduitsCateg.class,
                new RefActaTraitementsProduitsCategModel(),
                Referentials.GET_ACTA_TRAITEMENTS_PRODUITS_CATEG_NATURAL_ID
        );
    }

    @Override
    public ImportResult importActaSubstanceActive(InputStream stream) {
        return runSimpleImport(
                stream,
                RefActaSubstanceActive.class,
                new RefActaSubstanceActiveModel(),
                Referentials.GET_ACTA_SUBSTANCE_ACTIVE_NATURAL_ID
        );
    }
    
    @Override
    public ImportResult importProtocoleVgObs(InputStream stream) {
        return runSimpleImport(
                stream,
                RefProtocoleVgObs.class,
                new RefProtocoleVgObsModel(),
                Referentials.GET_PROTOCOLE_VG_OBS_NATURAL_ID
        );
    }
    
    @Override
    public ImportResult importElementVoisinage(InputStream stream) {
        return runSimpleImport(
                stream,
                RefElementVoisinage.class,
                new RefElementVoisinageModel(),
                Referentials.GET_ELEMENT_VOISINAGE_NATURAL_ID
        );
    }

    @Override
    public ImportResult importRcesoRulesGroundWater(InputStream stream) {
        return runSimpleImport(
                stream,
                RefRcesoRulesGroundWater.class,
                new RefRcesoRulesGroundWaterModel(),
                Referentials.GET_RCESO_RULES_GROUND_WATER_NATURAL_ID
        );
    }

    @Override
    public ImportResult importRcesoFuzzySetGroundWater(InputStream stream) {
        return runSimpleImport(
                stream,
                RefRcesoFuzzySetGroundWater.class,
                new RefRcesoFuzzySetGroundWaterModel(),
                Referentials.GET_RCESO_FUZZYSET_GROUND_WATER_NATURAL_ID
        );
    }

    @Override
    public ImportResult importRcesoCaseGroundWater(InputStream stream) {
        return runSimpleImport(
                stream,
                RefRcesoCaseGroundWater.class,
                new RefRcesoCaseGroundWaterModel(),
                Referentials.GET_RCESO_CASE_GROUND_WATER_NATURAL_ID
        );
    }

    @Override
    public ImportResult importRcesuRunoffPotRulesParc(InputStream stream) {
        return runSimpleImport(
                stream,
                RefRcesuRunoffPotRulesParc.class,
                new RefRcesuRunoffPotRulesParcModel(),
                Referentials.GET_RCESU_RUNOFF_POT_RULES_PARC_NATURAL_ID
        );
    }

    @Override
    public ImportResult importPhytoSubstanceActiveIphy(InputStream stream) {
        return runSimpleImport(
                stream,
                RefPhytoSubstanceActiveIphy.class,
                new RefPhytoSubstanceActiveIphyModel(),
                Referentials.GET_PHYTO_SUBSTANCE_ACTIVE_IPHY_NATURAL_ID
        );
    }

    @Override
    public ImportResult importActaDosageSpc(InputStream stream) {
        return runSimpleImport(
                stream,
                RefActaDosageSPC.class,
                new RefActaDosageSPCModel(),
                Referentials.GET_ACTA_DOSAGE_SPC_NATURAL_ID
        );
    }

    @Override
    public ImportResult importActaGroupeCultures(InputStream stream) {
        return runSimpleImport(
                stream,
                RefActaGroupeCultures.class,
                new RefActaGroupeCulturesModel(),
                Referentials.GET_ACTA_GROUPE_CULTURES_NATURAL_ID
        );
    }

    @Override
    public ImportResult importLienCulturesEdiActa(InputStream stream) {
        return runSimpleImport(
                stream,
                RefLienCulturesEdiActa.class,
                new RefLienCulturesEdiActaModel(),
                Referentials.GET_LIEN_CULTURES_EDI_ACTA_NATURAL_ID
        );
    }

    @Override
    public ImportResult importSaActaIphy(InputStream stream) {
        return runSimpleImport(
                stream,
                RefSaActaIphy.class,
                new RefSaActaIphyModel(),
                Referentials.GET_SA_ACTA_IPHY_NATURAL_ID
        );
    }

    @Override
    public ImportResult importTraitSdC(InputStream stream) {
        return runSimpleImport(
                stream,
                RefTraitSdC.class,
                new RefTraitSdCModel(),
                Referentials.GET_TRAIT_SDC_NATURAL_ID
        );
    }

    @Override
    public ImportResult importCouvSolAnnuelle(InputStream stream) {
        return runSimpleImport(
                stream,
                RefCouvSolAnnuelle.class,
                new RefCouvSolAnnuelleModel(),
                Referentials.GET_COUV_SOL_ANNUELLE_NATURAL_ID
        );
    }

    @Override
    public ImportResult importCultureEdiGroupeCouvSol(InputStream stream) {
        return runSimpleImport(
                stream,
                RefCultureEdiGroupeCouvSol.class,
                new RefCultureEdiGroupeCouvSolModel(),
                Referentials.GET_CULTURE_EDI_GROUP_COUV_SOL_NATURAL_ID
        );
    }

    @Override
    public ImportResult importCouvSolPerenne(InputStream stream) {
        return runSimpleImport(
                stream,
                RefCouvSolPerenne.class,
                new RefCouvSolPerenneModel(),
                Referentials.GET_COUV_SOL_PERENNE_NATURAL_ID
        );
    }

    @Override
    public ImportResult importDepartmentShapes(InputStream stream) {
        return runSimpleImport(
                stream,
                RefDepartmentShape.class,
                new RefDepartmentShapeModel(),
                Referentials.GET_DEPARTMENT_SHAPE_NATURAL_ID
        );
    }

    @Override
    public ImportResult importZoneClimatiqueIphy(InputStream stream) {
        return runSimpleImport(
                stream,
                RefZoneClimatiqueIphy.class,
                new RefZoneClimatiqueIphyModel(),
                Referentials.GET_ZONE_CLIMATIQUE_NATURAL_ID
        );
    }

}
