package fr.inra.agrosyst.services.practiced;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PracticedPlotServiceImpl.java 5028 2015-07-10 06:33:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/practiced/PracticedPlotServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.SolHorizon;
import fr.inra.agrosyst.api.entities.SolHorizonTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlotTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinageTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppaTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.plot.SolHorizonDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotDto;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotFilter;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.practicedPlot.PracticedPlotAndDependencies;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class PracticedPlotServiceImpl extends AbstractAgrosystService implements PracticedPlotService {

    protected AnonymizeService anonymizeService;
    protected BusinessAuthorizationService authorizationService;

    protected PracticedPlotTopiaDao practicedPlotTopiaDao;
    protected PracticedSystemTopiaDao practicedSystemDao;
    protected RefLocationTopiaDao locationDao;
    protected RefParcelleZonageEDITopiaDao parcelleZonageEDIDao;
    protected RefSolTextureGeppaTopiaDao refSolTextureGeppaDao;
    protected RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao;
    protected SolHorizonTopiaDao solHorizonDao;
    protected RefElementVoisinageTopiaDao refElementVoisinageTopiaDao;

    private static final Log log = LogFactory.getLog(PracticedPlotServiceImpl.class);

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setPracticedPlotTopiaDao(PracticedPlotTopiaDao practicedPlotTopiaDao) {
        this.practicedPlotTopiaDao = practicedPlotTopiaDao;
    }

    public void setPracticedSystemDao(PracticedSystemTopiaDao practicedSystemDao) {
        this.practicedSystemDao = practicedSystemDao;
    }

    public void setLocationDao(RefLocationTopiaDao locationDao) {
        this.locationDao = locationDao;
    }

    public void setParcelleZonageEDIDao(RefParcelleZonageEDITopiaDao parcelleZonageEDIDao) {
        this.parcelleZonageEDIDao = parcelleZonageEDIDao;
    }

    public void setRefSolTextureGeppaDao(RefSolTextureGeppaTopiaDao refSolTextureGeppaDao) {
        this.refSolTextureGeppaDao = refSolTextureGeppaDao;
    }

    public void setRefSolProfondeurIndigoDao(RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao) {
        this.refSolProfondeurIndigoDao = refSolProfondeurIndigoDao;
    }

    public void setSolHorizonDao(SolHorizonTopiaDao solHorizonDao) {
        this.solHorizonDao = solHorizonDao;
    }

    public void setRefElementVoisinageTopiaDao(RefElementVoisinageTopiaDao refElementVoisinageTopiaDao) {
        this.refElementVoisinageTopiaDao = refElementVoisinageTopiaDao;
    }

    @Override
    public ResultList<PracticedPlot> getFilteredPracticedPlots(PracticedPlotFilter filter) {
        ResultList<PracticedPlot> result = practicedPlotTopiaDao.getFilteredPracticedPlots(filter, getSecurityContext());
        return result;
    }

    @Override
    public ResultList<PracticedPlotDto> getFilteredPracticedPlotsDto(PracticedPlotFilter filter) {
        ResultList<PracticedPlot> practicedPlots = getFilteredPracticedPlots(filter);
        return ResultList.transform(practicedPlots, anonymizeService.getPracticedPlotToDtoFunction());
    }

    @Override
    public PracticedPlot getPracticedPlot(String practicedPlotTopiaId) {
        PracticedPlot result;
        if (StringUtils.isBlank(practicedPlotTopiaId)) {
            result = practicedPlotTopiaDao.newInstance();
        } else {
            result = practicedPlotTopiaDao.forTopiaIdEquals(practicedPlotTopiaId).findUnique();
        }
        return result;
    }

    @Override
    public PracticedPlot createOrUpdatePracticedPlot(PracticedPlot practicedPlot,
                                                     String practicedSystemId,
                                                     String locationId,
                                                     String selectedSurfaceTextureId,
                                                     String selectedSubSoilTextureId,
                                                     String selectedSolDepthId,
                                                     List<String> selectedPlotZoningIds,
                                                     List<SolHorizonDto> solHorizons,
                                                     List<String> adjacentElementIds) {

        PracticedSystem practicedSystem = getPracticedSystemForPracticedPlot(practicedPlot, practicedSystemId);

        authorizationService.checkCreateOrUpdatePracticedPlot(practicedPlot.getTopiaId(), practicedSystem.getTopiaId());

        RefLocation location = getRefLocation(locationId);

        RefSolTextureGeppa surfaceTexture = getSurfaceTexture(practicedPlot, selectedSurfaceTextureId);

        RefSolTextureGeppa subSoilTexture = getSubSoilTexture(practicedPlot, selectedSubSoilTextureId);

        RefSolProfondeurIndigo solDepth = getSolDepth(practicedPlot, selectedSolDepthId);

        PracticedPlot result = createOrUpdatePracticedPlotWithoutCommit(practicedPlot, practicedSystem, location, surfaceTexture, subSoilTexture, solDepth, selectedPlotZoningIds, solHorizons, adjacentElementIds);

        getTransaction().commit();
        return result;
    }

    protected PracticedPlot createOrUpdatePracticedPlotWithoutCommit(PracticedPlot practicedPlot,
                                                                     PracticedSystem practicedSystem,
                                                                     RefLocation location,
                                                                     RefSolTextureGeppa surfaceTexture,
                                                                     RefSolTextureGeppa subSoilTexture,
                                                                     RefSolProfondeurIndigo solDepth,
                                                                     List<String> selectedPlotZoningIds,
                                                                     List<SolHorizonDto> solHorizons,
                                                                     List<String> adjacentElementIds) {
        validatePracticedPlot(practicedPlot, practicedSystem);

        practicedPlot.setPracticedSystem(practicedSystem);

        practicedPlot.setLocation(location);

        // plot's surface texture
        practicedPlot.setSurfaceTexture(surfaceTexture);

        // plot's sub soil texture
        practicedPlot.setSubSoilTexture(subSoilTexture);

        // plot's sol profondeur
        practicedPlot.setSolDepth(solDepth);

        // persist sol horizon
        managePracticePlotSolHorizon(practicedPlot, solHorizons);

        // add zonings entities
        Collection<RefParcelleZonageEDI> plotZonings = practicedPlot.getPlotZonings() == null ? new ArrayList<RefParcelleZonageEDI>() : practicedPlot.getPlotZonings();
        practicedPlot.setPlotZonings(plotZonings);


        Collection<RefParcelleZonageEDI> nonDeletedPlotZonings = getPlotZonings(practicedPlot, selectedPlotZoningIds, plotZonings);
        plotZonings.retainAll(nonDeletedPlotZonings);

        // persist plot
        PracticedPlot result;
        if (practicedPlot.isPersisted()) {
            result = practicedPlotTopiaDao.update(practicedPlot);
        } else {
            result = practicedPlotTopiaDao.create(practicedPlot);
        }

        // adjacent elements
        manageAdjacentElements(adjacentElementIds, result);
        return result;
    }

    protected void manageAdjacentElements(List<String> adjacentElementIds, PracticedPlot result) {
        Collection<RefElementVoisinage> currentAdjacentElements = result.getAdjacentElements();
        if (currentAdjacentElements == null) {
            currentAdjacentElements = Lists.newArrayList();
            result.setAdjacentElements(currentAdjacentElements);
        }
        Collection<RefElementVoisinage> nonDeletedAdjacentElements = Lists.newArrayList();
        if (adjacentElementIds != null) {
            for (String adjacentElementId : adjacentElementIds) {
                RefElementVoisinage refElementVoisinage = refElementVoisinageTopiaDao.forTopiaIdEquals(adjacentElementId).findUnique();
                if (!currentAdjacentElements.contains(refElementVoisinage)) {
                    currentAdjacentElements.add(refElementVoisinage);
                }
                nonDeletedAdjacentElements.add(refElementVoisinage);
            }
        }
        currentAdjacentElements.retainAll(nonDeletedAdjacentElements);
    }

    protected void managePracticePlotSolHorizon(PracticedPlot practicedPlot, List<SolHorizonDto> solHorizons) {
        if (solHorizons != null) {
            Collection<SolHorizon> currentHorizons = practicedPlot.getSolHorizon();
            if (currentHorizons == null) {
                currentHorizons = Lists.newArrayList();
                practicedPlot.setSolHorizon(currentHorizons);
            }
            Map<String, SolHorizon> currentHorizonMap = Maps.uniqueIndex(currentHorizons, Entities.GET_TOPIA_ID);
            Set<SolHorizon> nonDeletedHorizon = Sets.newHashSet();
            for (SolHorizonDto solHorizonDto : solHorizons) {
                String topiaId = solHorizonDto.getTopiaId();
                SolHorizon solHorizon;
                if (StringUtils.isNotBlank(topiaId)) {
                    solHorizon = currentHorizonMap.get(topiaId);
                } else {
                    solHorizon = solHorizonDao.newInstance();
                }

                solHorizon.setComment(solHorizonDto.getComment());
                solHorizon.setLowRating(solHorizonDto.getLowRating());
                solHorizon.setStoniness(solHorizonDto.getStoniness());
                String solTextureId = solHorizonDto.getSolTextureId();
                RefSolTextureGeppa horizonSolTexture = null;
                if (StringUtils.isNotBlank(solTextureId)) {
                    horizonSolTexture = refSolTextureGeppaDao.forTopiaIdEquals(solTextureId).findUnique();
                }
                solHorizon.setSolTexture(horizonSolTexture);

                nonDeletedHorizon.add(solHorizon);
                currentHorizons.add(solHorizon);
            }
            currentHorizons.retainAll(nonDeletedHorizon);
        }
    }

    protected RefSolProfondeurIndigo getSolDepth(PracticedPlot practicedPlot, String selectedSolProfondeurId) {
        RefSolProfondeurIndigo solProfondeur = practicedPlot.getSolDepth();
        if (!Strings.isNullOrEmpty(selectedSolProfondeurId) && (solProfondeur == null || !selectedSolProfondeurId.equals(solProfondeur.getTopiaId()))) {
            solProfondeur = refSolProfondeurIndigoDao.forTopiaIdEquals(selectedSolProfondeurId).findUnique();
        }
        return solProfondeur;
    }

    protected RefSolTextureGeppa getSubSoilTexture(PracticedPlot practicedPlot, String selectedSubSoilTextureId) {
        RefSolTextureGeppa subSoilTexture = practicedPlot.getSubSoilTexture();
        if (!Strings.isNullOrEmpty(selectedSubSoilTextureId) && (subSoilTexture == null || !selectedSubSoilTextureId.equals(subSoilTexture.getTopiaId()))) {
            subSoilTexture = refSolTextureGeppaDao.forTopiaIdEquals(selectedSubSoilTextureId).findUnique();
        }
        return subSoilTexture;
    }

    protected RefSolTextureGeppa getSurfaceTexture(PracticedPlot practicedPlot, String selectedSurfaceTextureId) {
        RefSolTextureGeppa surfaceTexture = practicedPlot.getSurfaceTexture();
        if (!Strings.isNullOrEmpty(selectedSurfaceTextureId) && (surfaceTexture == null || !selectedSurfaceTextureId.equals(surfaceTexture.getTopiaId()))) {
            surfaceTexture = refSolTextureGeppaDao.forTopiaIdEquals(selectedSurfaceTextureId).findUnique();
        }
        return surfaceTexture;
    }

    protected Collection<RefParcelleZonageEDI> getPlotZonings(PracticedPlot practicedPlot, List<String> selectedPlotZoningIds, Collection<RefParcelleZonageEDI> plotZonings) {
        Collection<RefParcelleZonageEDI> nonDeletedPlotZonings = Lists.newArrayList();
        if (selectedPlotZoningIds != null) {
            // si la parcelle est hors zonage, on vide la liste
            List<RefParcelleZonageEDI> parcelleZonageEDIs;
            if (practicedPlot.isOutOfZoning()) {
                selectedPlotZoningIds.clear();
                parcelleZonageEDIs = Lists.newArrayList();
            } else {
                parcelleZonageEDIs = parcelleZonageEDIDao.forTopiaIdIn(selectedPlotZoningIds).findAll();
                Preconditions.checkArgument(parcelleZonageEDIs.size() == selectedPlotZoningIds.size(), "Parcelles de la zone non retouvées !");
            }

            for (RefParcelleZonageEDI parcelleZonageEDI : parcelleZonageEDIs) {
                if (!plotZonings.contains(parcelleZonageEDI)) {
                    plotZonings.add(parcelleZonageEDI);
                }
                nonDeletedPlotZonings.add(parcelleZonageEDI);
            }
        }
        return nonDeletedPlotZonings;
    }

    protected RefLocation getRefLocation(String locationTopiaId) {
        RefLocation location = null;
        if (StringUtils.isNotBlank(locationTopiaId)) {
            location = locationDao.forTopiaIdEquals(locationTopiaId).findUnique();
        }
        return location;
    }

    protected PracticedSystem getPracticedSystemForPracticedPlot(PracticedPlot practicedPlot, String practicedSystemTopiaId) {
        PracticedSystem practicedSystem;
        if (!practicedPlot.isPersisted()) {
            Preconditions.checkArgument(StringUtils.isNotBlank(practicedSystemTopiaId), "L'identifiant su sytstème synthétisé est requis.");
            practicedSystem = practicedSystemDao.forTopiaIdEquals(practicedSystemTopiaId).findUnique();
            practicedPlot.setPracticedSystem(practicedSystem);

        } else {
            practicedSystem = practicedPlot.getPracticedSystem();
        }
        return practicedSystem;
    }

    protected void validatePracticedPlot(PracticedPlot practicedPlot, PracticedSystem practicedSystem) {
        // valid unique
        if (!practicedPlot.isPersisted()) {
            Preconditions.checkArgument(!practicedPlotTopiaDao.forPracticedSystemEquals(practicedSystem).exists());
        }
        Preconditions.checkArgument(practicedPlot.getWaterFlowDistance() != null);
        Preconditions.checkArgument(practicedPlot.getBufferStrip() != null);
        Preconditions.checkArgument(practicedPlot.getMaxSlope() != null);
    }

    @Override
    public List<PracticedSystem> getPracticedSystemsWithoutPracticedPlot(NavigationContext navigationContext) {
        List<PracticedSystem> result = practicedSystemDao.getPracticedSystemsWithoutPlot(navigationContext, getSecurityContext());
        return result;
    }

    @Override
    public void importPZ0PracticedPlots(Map<Class, ImportResults> allResults) {
        ImportResults importResults = allResults.remove(PracticedPlot.class);
        if (importResults != null && importResults.getIgnoredRecords() == 0) {
            try {
                Map<String, EntityAndDependencies> entitiesAndDependencies = importResults.getEntityAndDepsByCsvIds();
                int count = 1;
                int total = entitiesAndDependencies.values().size();
                for (EntityAndDependencies entityAndDependencies : entitiesAndDependencies.values()) {
                    PracticedPlotAndDependencies practicedPlotAndDependencies = (PracticedPlotAndDependencies) entityAndDependencies;
                    PracticedPlot practicedPlot = practicedPlotAndDependencies.getEntity();
                    PracticedSystem practicedSystem = practicedPlot.getPracticedSystem();
                    RefLocation location = practicedPlot.getLocation();
                    List<String> selectedPlotZoningIds = practicedPlotAndDependencies.getSelectedPlotZoningIds();
                    RefSolTextureGeppa surfaceTexture = practicedPlot.getSurfaceTexture();
                    RefSolTextureGeppa subSoilTexture = practicedPlot.getSubSoilTexture();
                    RefSolProfondeurIndigo solDepth = practicedPlot.getSolDepth();
                    List<SolHorizonDto> solHorizons = practicedPlotAndDependencies.getSolHorizonDtos();
                    List<String> adjacentElementIds = practicedPlotAndDependencies.getAdjacentElementIds();

                    long start = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info(String.format("Début sauvegarde de la parcelle du synthétisé %s, campagne %s - %d/%s.", practicedSystem.getName(), practicedSystem.getCampaigns(), count++, total));
                    }
                    PracticedPlot persistedPlot = createOrUpdatePracticedPlotWithoutCommit(practicedPlot, practicedSystem, location, surfaceTexture, subSoilTexture, solDepth, selectedPlotZoningIds, solHorizons, adjacentElementIds);
                    practicedPlotAndDependencies.setEntity(persistedPlot);

                    long p2 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Fin de sauvegarde de la parcelle du synthétisé, traitement réalisé en:" + (p2- start));
                    }
                }
            } catch (Exception e) {
                throw (new AgrosystImportException("Echec de persistance des parcelle pour systèmes synthétisé", e));
            }
        }
    }
}
