package fr.inra.agrosyst.services.plot.export;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PlotExportMetadata.java 4005 2014-04-11 14:52:55Z bleny $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/plot/export/PlotExportMetadata.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.FrostProtectionType;
import fr.inra.agrosyst.api.entities.HosesPositionning;
import fr.inra.agrosyst.api.entities.IrrigationSystemType;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PompEngineType;
import fr.inra.agrosyst.api.entities.SolWaterPh;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneType;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinageTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppaTopiaDao;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.common.export.UniqueDualLinkedHashMap;
import org.apache.commons.collections4.BidiMap;

import java.util.Map;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class PlotExportMetadata {

    /**
     * Current bean info.
     */
    public static abstract class PlotCommonBeanInfo implements EntityExportTabInfo {

        @Override
        public BidiMap<String, String> getCommonColumns() {
            BidiMap<String, String> result = new UniqueDualLinkedHashMap<String, String>();
            result.put("plotName", "Parcelle");
            result.put("domainName", "Domaine");
            result.put("campaign", "Campagne");
            return result;
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> result = new UniqueDualLinkedHashMap<String, String>();
            return result;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = Maps.newLinkedHashMap();
            return result;
        }

        @Override
        public Map<String, Function<String, Object>> getCustomParsers() {
            Map<String, Function<String, Object>> result = Maps.newHashMap();
            return result;
        }

        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = Maps.newHashMap();
            return result;
        }
    }

    /**
     * Current bean info.
     */
    public static class PlotMainBeanInfo extends PlotCommonBeanInfo {

        @Override
        public String getBeanTitle() {
            return "Généralités";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(RefLocation.PROPERTY_CODE_POSTAL, "Commune (Code Postal)");
            columns.put(RefLocation.PROPERTY_COMMUNE, "Commune (Nom)");
            columns.put("growingSystemName", "Système De Culture");
            columns.put(Plot.PROPERTY_AREA, "Surface totale");
            columns.put(Plot.PROPERTY_PAC_ILOT_NUMBER, "Numéro d'ilôt PAC");
            columns.put(Plot.PROPERTY_MAX_SLOPE, "Pente maxi");
            columns.put(Plot.PROPERTY_WATER_FLOW_DISTANCE, "Distance à un cours d'eau");
            columns.put(Plot.PROPERTY_BUFFER_STRIP, "Bande enherbée");
            columns.put(Plot.PROPERTY_LATITUDE, "Latitude GPS du centre de la parcelle");
            columns.put(Plot.PROPERTY_LONGITUDE, "Longitude GPS du centre de la parcelle");
            columns.put(Plot.PROPERTY_COMMENT, "Commentaire sur la parcelle");
            columns.put(Plot.PROPERTY_ACTIVITY_END_COMMENT, "Motif de fin d'utilisation");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();

            result.put("growingSystemName", new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getGrowingSystem() != null) {
                        result = plot.getGrowingSystem().getName();
                    }
                    return result;
                }
            });
            result.put(RefLocation.PROPERTY_CODE_POSTAL, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getLocation() != null) {
                        result = plot.getLocation().getCodePostal();
                    }
                    return result;
                }
            });
            result.put(RefLocation.PROPERTY_COMMUNE, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getLocation() != null) {
                        result = plot.getLocation().getCommune();
                    }
                    return result;
                }
            });

            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            return result;
        }
    }
    
    /**
     * Current bean info.
     */
    public static class PlotZoningBeanInfo extends PlotCommonBeanInfo {
        protected RefParcelleZonageEDITopiaDao refParcelleZonageEDITopiaDao;
        public void setRefParcelleZonageEDITopiaDao(RefParcelleZonageEDITopiaDao refParcelleZonageEDITopiaDao) {
            this.refParcelleZonageEDITopiaDao = refParcelleZonageEDITopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Zonage";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Plot.PROPERTY_OUT_OF_ZONING, "Déclarer la parcelle hors de tout zonage");
            columns.put("zonage", "Zonage");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();
            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Plot.PROPERTY_PLOT_ZONINGS, refParcelleZonageEDITopiaDao.findPropertyDistinctValues(RefParcelleZonageEDI.PROPERTY_LIBELLE_ENGAGEMENT_PARCELLE));
            return result;
        }
    }
    
    /**
     * Current bean info.
     */
    public static class PlotEquipmentBeanInfo extends PlotCommonBeanInfo {

        @Override
        public String getBeanTitle() {
            return "Equipements";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Plot.PROPERTY_IRRIGATION_SYSTEM, "Système d'irrigation");
            columns.put(Plot.PROPERTY_IRRIGATION_SYSTEM_TYPE, "Type de système d'irrigation");
            columns.put(Plot.PROPERTY_POMP_ENGINE_TYPE, "Type de moteur de pompe");
            columns.put(Plot.PROPERTY_HOSES_POSITIONNING, "Positionnement des tuyaux d'arrosage");
            columns.put(Plot.PROPERTY_FERTIGATION_SYSTEM, "Système de fertirrigation");
            columns.put(Plot.PROPERTY_WATER_ORIGIN, "Origine de l'eau");
            columns.put(Plot.PROPERTY_DRAINAGE, "Drainage");
            columns.put(Plot.PROPERTY_DRAINAGE_YEAR, "Année de réalisation du drainage");
            columns.put(Plot.PROPERTY_FROST_PROTECTION, "Protection anti-gel");
            columns.put(Plot.PROPERTY_FROST_PROTECTION_TYPE, "Type de protection anti-gel");
            columns.put(Plot.PROPERTY_HAIL_PROTECTION, "Protection anti-grêle");
            columns.put(Plot.PROPERTY_RAINPROOF_PROTECTION, "Protection anti-pluie");
            columns.put(Plot.PROPERTY_PEST_PROTECTION, "Protection anti-insectes");
            columns.put(Plot.PROPERTY_OTHER_EQUIPMENT, "Autre équipement");
            columns.put(Plot.PROPERTY_EQUIPMENT_COMMENT, "Commentaire sur l’équipement de la parcelle");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();
            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Plot.PROPERTY_IRRIGATION_SYSTEM_TYPE, ExportUtils.allStringOf(IrrigationSystemType.class));
            result.put(Plot.PROPERTY_POMP_ENGINE_TYPE, ExportUtils.allStringOf(PompEngineType.class));
            result.put(Plot.PROPERTY_HOSES_POSITIONNING, ExportUtils.allStringOf(HosesPositionning.class));
            result.put(Plot.PROPERTY_FROST_PROTECTION_TYPE, ExportUtils.allStringOf(FrostProtectionType.class));
            return result;
        }
    }
    
    /**
     * Current bean info.
     */
    public static class PlotGroundBeanInfo extends PlotCommonBeanInfo {
        protected RefSolTextureGeppaTopiaDao refSolTextureGeppaTopiaDao;
        protected RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoTopiaDao;
        public void setRefSolProfondeurIndigoTopiaDao(RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoTopiaDao) {
            this.refSolProfondeurIndigoTopiaDao = refSolProfondeurIndigoTopiaDao;
        }
        public void setRefSolTextureGeppaTopiaDao(RefSolTextureGeppaTopiaDao refSolTextureGeppaTopiaDao) {
            this.refSolTextureGeppaTopiaDao = refSolTextureGeppaTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Sols";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Plot.PROPERTY_GROUND, "Sol du domaine");
            columns.put(Plot.PROPERTY_SURFACE_TEXTURE, "Texture moyenne de la surface");
            columns.put(Plot.PROPERTY_SUB_SOIL_TEXTURE, "Texture moyenne du sous-sol");
            columns.put(Plot.PROPERTY_SOL_STONINESS, "Pierrosité moyenne");
            columns.put(Plot.PROPERTY_SOL_DEPTH, "Classe profondeur maxi enracinement");
            columns.put(Plot.PROPERTY_SOL_MAX_DEPTH, "Profondeur maxi enracinement");
            // c'est chaud : columns.put("reserveUtile", "Réserve utile");
            columns.put(Plot.PROPERTY_SOL_ORGANIC_MATERIAL_PERCENT, "Pourcentage Matière Organique");
            columns.put(Plot.PROPERTY_SOL_BATTANCE, "Battance");
            columns.put(Plot.PROPERTY_SOL_WATER_PH, "PH eau");
            columns.put(Plot.PROPERTY_SOL_HYDROMORPHISMS, "Hydromorphie");
            columns.put(Plot.PROPERTY_SOL_LIMESTONE, "Calcaire");
            columns.put(Plot.PROPERTY_SOL_ACTIVE_LIMESTONE, "Proportion calcaire actif");
            columns.put(Plot.PROPERTY_SOL_TOTAL_LIMESTONE, "Proportion calcaire total");
            columns.put(Plot.PROPERTY_SOL_COMMENT, "Commentaires");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();
            result.put(Plot.PROPERTY_GROUND, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof Plot);
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getGround() != null) {
                        result = plot.getGround().getName();
                    }
                    return result;
                }
            });
            result.put(Plot.PROPERTY_SURFACE_TEXTURE, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof Plot);
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getSurfaceTexture() != null) {
                        result = plot.getSurfaceTexture().getClasses_texturales_GEPAA();
                    }
                    return result;
                }
            });
            result.put(Plot.PROPERTY_SUB_SOIL_TEXTURE, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof Plot);
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getSubSoilTexture() != null) {
                        result = plot.getSubSoilTexture().getClasses_texturales_GEPAA();
                    }
                    return result;
                }
            });
            result.put(Plot.PROPERTY_SOL_DEPTH, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof Plot);
                    Plot plot = (Plot) input;
                    Object result = null;
                    if (plot.getSolDepth() != null) {
                        result = plot.getSolDepth().getLibelle_classe();
                    }
                    return result;
                }
            });
            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Plot.PROPERTY_SOL_DEPTH, refSolProfondeurIndigoTopiaDao.findPropertyDistinctValues(RefSolProfondeurIndigo.PROPERTY_LIBELLE_CLASSE));
            result.put(Plot.PROPERTY_SURFACE_TEXTURE, refSolTextureGeppaTopiaDao.findPropertyDistinctValues(RefSolTextureGeppa.PROPERTY_CLASSES_TEXTURALES__GEPAA));
            result.put(Plot.PROPERTY_SUB_SOIL_TEXTURE, refSolTextureGeppaTopiaDao.findPropertyDistinctValues(RefSolTextureGeppa.PROPERTY_CLASSES_TEXTURALES__GEPAA));
            result.put(Plot.PROPERTY_SOL_WATER_PH, ExportUtils.allStringOf(SolWaterPh.class));
            return result;
        }
    }
    
    /**
     * Current bean info.
     */
    public static class PlotZoneBeanInfo extends PlotCommonBeanInfo {

        @Override
        public String getBeanTitle() {
            return "Zones";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Zone.PROPERTY_NAME, "Nom de la zone");
            columns.put(Zone.PROPERTY_TYPE, "Type");
            columns.put(Zone.PROPERTY_AREA, "Surface (ha)");
            columns.put(Zone.PROPERTY_LATITUDE, "Latitude du centre de la zone");
            columns.put(Zone.PROPERTY_LONGITUDE, "Longitude du centre de la zone");
            columns.put(Zone.PROPERTY_COMMENT, "Commentaire");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();
            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Zone.PROPERTY_TYPE, ExportUtils.allStringOf(ZoneType.class));
            return result;
        }
    }
    
    /**
     * Current bean info.
     */
    public static class PlotAdjacentBeanInfo extends PlotCommonBeanInfo {
        protected RefElementVoisinageTopiaDao refElementVoisinageTopiaDao;
        public void setRefElementVoisinageTopiaDao(RefElementVoisinageTopiaDao refElementVoisinageTopiaDao) {
            this.refElementVoisinageTopiaDao = refElementVoisinageTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Éléments de voisinage";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Plot.PROPERTY_ADJACENT_ELEMENTS, "Éléments de voisinage");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();
            result.put(Plot.PROPERTY_ADJACENT_ELEMENTS, new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof RefElementVoisinage);
                    RefElementVoisinage refElementVoisinage = (RefElementVoisinage) input;
                    Object result = refElementVoisinage.getIae_nom();
                    return result;
                }
            });
            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Plot.PROPERTY_SOL_DEPTH, refElementVoisinageTopiaDao.findPropertyDistinctValues(RefElementVoisinage.PROPERTY_IAE_NOM));
            return result;
        }
    }
}
