package fr.inra.agrosyst.services.plot;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PlotServiceImpl.java 5028 2015-07-10 06:33:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/plot/PlotServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.GroundTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.SolHorizon;
import fr.inra.agrosyst.api.entities.SolHorizonTopiaDao;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneImpl;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.ZoneType;
import fr.inra.agrosyst.api.entities.measure.MeasurementSession;
import fr.inra.agrosyst.api.entities.measure.MeasurementSessionTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinage;
import fr.inra.agrosyst.api.entities.referential.RefElementVoisinageTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDI;
import fr.inra.agrosyst.api.entities.referential.RefParcelleZonageEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigo;
import fr.inra.agrosyst.api.entities.referential.RefSolProfondeurIndigoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppa;
import fr.inra.agrosyst.api.entities.referential.RefSolTextureGeppaTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.domain.PlotDto;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.plot.SolHorizonDto;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.plot.PlotAndDependencies;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.EntityUsageService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.EntityImporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.plot.export.PlotExportEntity;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotAdjacentBeanInfo;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotEquipmentBeanInfo;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotGroundBeanInfo;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotMainBeanInfo;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotZoneBeanInfo;
import fr.inra.agrosyst.services.plot.export.PlotExportMetadata.PlotZoningBeanInfo;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * @author cosse
 *
 */
public class PlotServiceImpl extends AbstractAgrosystService implements PlotService {

    private static final Log log = LogFactory.getLog(PlotServiceImpl.class);


    /** Nom de la zone principale créée automatiquement. */
    public static final String PLOT_DEFAULT_ZONE_NAME = "Zone principale";

    protected static final Function<Zone, String> GET_ZONE_CODE = new Function<Zone, String>() {
        @Override
        public String apply(Zone zone) {
            String result = zone.getCode();
            return result;
        }
    };

    protected AnonymizeService anonymizeService;
    protected BusinessAuthorizationService authorizationService;
    protected EntityUsageService entityUsageService;

    protected PlotTopiaDao plotDao;
    protected RefLocationTopiaDao locationDao;
    protected DomainTopiaDao domainDao;
    protected RefParcelleZonageEDITopiaDao parcelleZonageEDIDao;
    protected GrowingSystemTopiaDao growingSystemDao;
    protected RefSolTextureGeppaTopiaDao refSolTextureGeppaDao;
    protected GroundTopiaDao solDao;
    protected RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao;
    protected SolHorizonTopiaDao solHorizonDao;
    protected ZoneTopiaDao zoneDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected RefElementVoisinageTopiaDao refElementVoisinageDao;
    protected MeasurementSessionTopiaDao measurementSessionDao;

    public void setEntityUsageService(EntityUsageService entityUsageService) {
        this.entityUsageService = entityUsageService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setPlotDao(PlotTopiaDao plotDao) {
        this.plotDao = plotDao;
    }

    public void setLocationDao(RefLocationTopiaDao locationDao) {
        this.locationDao = locationDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    public void setParcelleZonageEDIDao(RefParcelleZonageEDITopiaDao parcelleZonageEDIDao) {
        this.parcelleZonageEDIDao = parcelleZonageEDIDao;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    public void setRefSolTextureGeppaDao(RefSolTextureGeppaTopiaDao refSolTextureGeppaDao) {
        this.refSolTextureGeppaDao = refSolTextureGeppaDao;
    }

    public void setSolDao(GroundTopiaDao solDao) {
        this.solDao = solDao;
    }

    public void setRefSolProfondeurIndigoDao(RefSolProfondeurIndigoTopiaDao refSolProfondeurIndigoDao) {
        this.refSolProfondeurIndigoDao = refSolProfondeurIndigoDao;
    }

    public void setSolHorizonDao(SolHorizonTopiaDao solHorizonDao) {
        this.solHorizonDao = solHorizonDao;
    }

    public void setZoneDao(ZoneTopiaDao zoneDao) {
        this.zoneDao = zoneDao;
    }

    public void setGrowingPlanDao(GrowingPlanTopiaDao growingPlanDao) {
        this.growingPlanDao = growingPlanDao;
    }

    public void setRefElementVoisinageDao(
            RefElementVoisinageTopiaDao refElementVoisinageDao) {
        this.refElementVoisinageDao = refElementVoisinageDao;
    }

    public void setMeasurementSessionDao(MeasurementSessionTopiaDao measurementSessionDao) {
        this.measurementSessionDao = measurementSessionDao;
    }

    @Override
    public Plot getPlot(String plotTopiaId) {
        Plot result;
        if (StringUtils.isBlank(plotTopiaId)) {
            result = plotDao.newInstance();
            result.setActive(true);
            result.setCode(UUID.randomUUID().toString());
        } else {
            result = plotDao.forTopiaIdEquals(plotTopiaId).findUnique();
            result = anonymizeService.checkForPlotAnonymization(result);
        }
        return result;
    }

    @Override
    public Plot createOrUpdatePlot(Plot plot, String domainId, String locationId, String growingSystemId,
                                   Collection<String> selectedPlotZoningIds, String selectedSolId,
                                   String selectedSurfaceTextureId, String selectedSubSoilTextureId,
                                   String selectedSolDepthId, List<SolHorizonDto> solHorizons, Collection<Zone> zones,
                                   List<String> adjacentElementIds) {

        authorizationService.checkCreateOrUpdatePlot(plot.getTopiaId());

        Plot result = doCreateOrUpdatePlot(plot, domainId, locationId, growingSystemId,
                selectedPlotZoningIds, selectedSolId,
                selectedSurfaceTextureId, selectedSubSoilTextureId,
                selectedSolDepthId, solHorizons, zones,
                adjacentElementIds);
        getTransaction().commit();
        return result;
    }

    protected Plot createOrUpdatePlotWithoutCommit(Plot plot, String domainId, String locationId, String growingSystemId,
                                                   Collection<String> selectedPlotZoningIds, String selectedSolId,
                                                   String selectedSurfaceTextureId, String selectedSubSoilTextureId,
                                                   String selectedSolDepthId, List<SolHorizonDto> solHorizons, Collection<Zone> zones,
                                                   List<String> adjacentElementIds) {
        Plot result = doCreateOrUpdatePlot(plot, domainId, locationId, growingSystemId,
                selectedPlotZoningIds, selectedSolId,
                selectedSurfaceTextureId, selectedSubSoilTextureId,
                selectedSolDepthId, solHorizons, zones,
                adjacentElementIds);
        return result;
    }

    protected void validPlotPreconditions(Plot plot, String domainId, String locationId, Collection<Zone> zones) {
        Preconditions.checkArgument(StringUtils.isNotBlank(plot.getName()), "Le nom de la parcelle et requis");
        Preconditions.checkArgument(StringUtils.isNotBlank(domainId), "Aucun domaine renseigné pour la parcelle");
        Preconditions.checkArgument(StringUtils.isNotBlank(locationId), "La commune n'est pas renseignée pour la parcelle");
        Preconditions.checkArgument(plot.getWaterFlowDistance() != null, "La distance à un cours d'eau n'est renseignée pour la parcelle");
        Preconditions.checkArgument(plot.getMaxSlope() != null, "la pente maximum n'est renseignée pour la parcelle");
        Preconditions.checkArgument(plot.getBufferStrip() != null, "la bande enherbée n'est renseignée pour la parcelle");
        Preconditions.checkArgument(CollectionUtils.isNotEmpty(zones), "La liste des zones ne peut être vide, une zone principale est requise !");
    }

    protected Plot doCreateOrUpdatePlot(Plot plot, String domainId, String locationId, String growingSystemId,
                                        Collection<String> selectedPlotZoningIds, String selectedSolId,
                                        String selectedSurfaceTextureId, String selectedSubSoilTextureId,
                                        String selectedSolDepthId, List<SolHorizonDto> solHorizons, Collection<Zone> zones,
                                        List<String> adjacentElementIds) {

        // persist plot
        Plot result;

        validPlotPreconditions(plot, domainId, locationId, zones);

        addPlotDomain(plot, domainId);

        addPlotLocation(plot, locationId);

        addPlotGrowingSystem(plot, growingSystemId);

        addZoningsEntities(plot, selectedPlotZoningIds);

        addGround(plot, selectedSolId);

        RefSolTextureGeppa surfaceTexture = getRefSolTextureGeppa(selectedSurfaceTextureId);
        plot.setSurfaceTexture(surfaceTexture);

        RefSolTextureGeppa subSoilTexture = getRefSolTextureGeppa(selectedSubSoilTextureId);
        plot.setSubSoilTexture(subSoilTexture);

        addPlotSolProfondeurIndigo(plot, selectedSolDepthId);

        persistPlotSolHorizons(plot, solHorizons);

        addPlotAdjacentElements(plot, adjacentElementIds);

        if (plot.isPersisted()) {
            result = plotDao.update(plot);
        } else {
            // create a random plot code, used to link plots each other
            setPlotCode(plot);
            result = plotDao.create(plot);
        }

        // persist zones
        persistPlotZones(plot, zones, result);

        return result;
    }

    private void setPlotCode(Plot plot) {
        if (StringUtils.isBlank(plot.getCode())) {
            plot.setCode(UUID.randomUUID().toString());
        }
    }

    protected void addPlotDomain(Plot plot, String domainId) {
        Domain domain = plot.getDomain();
        if (domain != null && !domain.getTopiaId().equals(domainId)) {
            domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        } else if (domain == null) {
            domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        }
        plot.setDomain(domain);
    }

    protected void addPlotLocation(Plot plot, String locationId) {
        RefLocation location = locationDao.forTopiaIdEquals(locationId).findUnique();
        plot.setLocation(location);
    }

    protected void addPlotGrowingSystem(Plot plot, String growingSystemId) {
        GrowingSystem previousGrowingSystem = plot.getGrowingSystem();

        String previousGrowingSystemId = null;

        if (previousGrowingSystem != null) {
            previousGrowingSystemId = previousGrowingSystem.getTopiaId();
        }
        // A growing system is removed
        if (StringUtils.isBlank(growingSystemId) && previousGrowingSystemId != null) {
            plot.setGrowingSystem(null);
        }
        // A new growing system is affected
        if (!StringUtils.isBlank(growingSystemId) && (previousGrowingSystemId == null || !previousGrowingSystemId.contentEquals(growingSystemId))){
            GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
            plot.setGrowingSystem(growingSystem);
        }
    }

    protected void addZoningsEntities(Plot plot, Collection<String> selectedPlotZoningIds) {
        Collection<RefParcelleZonageEDI> plotZonings = plot.getPlotZonings();
        if (plotZonings == null) {
            plotZonings = Lists.newArrayList();
            plot.setPlotZonings(plotZonings);
        }
        Collection<RefParcelleZonageEDI> nonDeletedPlotZonings = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(selectedPlotZoningIds)) {
            // si la parcelle est hors zonage, on vide la liste
            if (plot.isOutOfZoning()) {
                selectedPlotZoningIds.clear();
            }
            for (String selectedPlotZoningId : selectedPlotZoningIds) {
                RefParcelleZonageEDI parcelleZonageEDI = parcelleZonageEDIDao.forTopiaIdEquals(selectedPlotZoningId).findUnique();
                if (!plotZonings.contains(parcelleZonageEDI)) {
                    plotZonings.add(parcelleZonageEDI);
                }
                nonDeletedPlotZonings.add(parcelleZonageEDI);
            }
        }
        plotZonings.retainAll(nonDeletedPlotZonings);
    }

    protected void addGround(Plot plot, String selectedSolId) {
        Ground sol;
        if (StringUtils.isNotBlank(selectedSolId)) {
            sol = solDao.forTopiaIdEquals(selectedSolId).findUnique();
        } else {
            sol = null;
        }
        plot.setGround(sol);
    }

    protected RefSolTextureGeppa getRefSolTextureGeppa(String selectedSubSoilTextureId) {
        RefSolTextureGeppa subSoilTexture;
        if (StringUtils.isNotBlank(selectedSubSoilTextureId)) {
            subSoilTexture = refSolTextureGeppaDao.forTopiaIdEquals(selectedSubSoilTextureId).findUnique();
        } else {
            subSoilTexture = null;
        }
        return subSoilTexture;
    }

    protected void addPlotSolProfondeurIndigo(Plot plot, String selectedSolDepthId) {
        RefSolProfondeurIndigo solProfondeur;
        if (StringUtils.isNotBlank(selectedSolDepthId)) {
            solProfondeur = refSolProfondeurIndigoDao.forTopiaIdEquals(selectedSolDepthId).findUnique();
        } else {
            solProfondeur = null;
        }

        plot.setSolDepth(solProfondeur);
    }

    protected void persistPlotSolHorizons(Plot plot, List<SolHorizonDto> solHorizons) {
        if (solHorizons != null) {
            Collection<SolHorizon> currentHorizons = plot.getSolHorizon();
            if (currentHorizons == null) {
                currentHorizons = Lists.newArrayList();
                plot.setSolHorizon(currentHorizons);
            }
            Map<String, SolHorizon> currentHorizonMap = Maps.uniqueIndex(currentHorizons, Entities.GET_TOPIA_ID);
            Set<SolHorizon> nonDeletedHorizon = Sets.newHashSet();
            for (SolHorizonDto solHorizonDto : solHorizons) {
                String topiaId = solHorizonDto.getTopiaId();
                SolHorizon solHorizon;
                if (StringUtils.isNotBlank(topiaId)) {
                    solHorizon = currentHorizonMap.get(topiaId);
                } else {
                    solHorizon = solHorizonDao.newInstance();
                }

                solHorizon.setComment(solHorizonDto.getComment());
                solHorizon.setLowRating(solHorizonDto.getLowRating());
                solHorizon.setStoniness(solHorizonDto.getStoniness());
                String solTextureId = solHorizonDto.getSolTextureId();
                RefSolTextureGeppa horizonSolTexture = null;
                if (StringUtils.isNotBlank(solTextureId)) {
                    horizonSolTexture = refSolTextureGeppaDao.forTopiaIdEquals(solTextureId).findUnique();
                }
                solHorizon.setSolTexture(horizonSolTexture);

                nonDeletedHorizon.add(solHorizon);
                currentHorizons.add(solHorizon);
            }
            currentHorizons.retainAll(nonDeletedHorizon);
        }
    }

    protected void addPlotAdjacentElements(Plot plot, List<String> adjacentElementIds) {
        Collection<RefElementVoisinage> currentAdjacentElements = plot.getAdjacentElements();
        if (currentAdjacentElements == null) {
            currentAdjacentElements = Lists.newArrayList();
            plot.setAdjacentElements(currentAdjacentElements);
        }
        Collection<RefElementVoisinage> nonDeletedAdjacentElements = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(adjacentElementIds)) {
            for (String adjacentElementId : adjacentElementIds) {
                RefElementVoisinage refElementVoisinage = refElementVoisinageDao.forTopiaIdEquals(adjacentElementId).findUnique();
                if (!currentAdjacentElements.contains(refElementVoisinage)) {
                    currentAdjacentElements.add(refElementVoisinage);
                }
                nonDeletedAdjacentElements.add(refElementVoisinage);
            }
        }
        currentAdjacentElements.retainAll(nonDeletedAdjacentElements);
    }

    protected void persistPlotZones(Plot plot, Collection<Zone> zones, Plot result) {
        List<Zone> currentZones = zoneDao.forPlotEquals(result).findAll();
        Map<String, Zone> currentZonesMap = Maps.newHashMap(Maps.uniqueIndex(currentZones, Entities.GET_TOPIA_ID));
        for (Zone zoneDto : zones) {
            String topiaId = zoneDto.getTopiaId();
            Zone zone;
            if (StringUtils.isBlank(topiaId)) {
                zone = zoneDao.newInstance();
                zone.setPlot(plot);
                if (StringUtils.isBlank(zoneDto.getCode())) {
                    zone.setCode(UUID.randomUUID().toString());
                } else {
                    zone.setCode(zoneDto.getCode());
                }
            } else {
                zone = currentZonesMap.remove(topiaId);
            }

            zone.setName(zoneDto.getName());
            zone.setType(zoneDto.getType());
            zone.setArea(zoneDto.getArea());
            zone.setLatitude(zoneDto.getLatitude());
            zone.setLongitude(zoneDto.getLongitude());
            zone.setComment(zoneDto.getComment());
            zone.setActive(zoneDto.isActive());

            if (StringUtils.isBlank(topiaId)) {
                zoneDao.create(zone);
            } else {
                zoneDao.update(zone);
            }
        }

        // on peut supprimer des zones qui ne sont pas utilisées ici
        // si cela plante à cause d'une dépendance transient, ca veut dire que la suppression
        // de zone n'est pas autorisée
        zoneDao.deleteAll(currentZonesMap.values());

        // on sauvegarde l'information de zones supprimées pour pouvoir le signaler
        // à l'utilisateur
        if (!currentZonesMap.isEmpty()) {
            plot.setDeletedZones(true);
        }
    }

    @Override
    public void updatePlotsGrowingSystemRelationship(GrowingSystem growingSystem, Collection<String> plotIds) {

        String growingSystemTopiaId = growingSystem.getTopiaId();

        List<Plot> plots;

        if (growingSystemTopiaId == null) {
            plots = new ArrayList<Plot>();
        } else {
            plots = plotDao.forGrowingSystemEquals(growingSystem).findAll();
        }

        Map<String, Plot> indexedPlots = Maps.uniqueIndex(plots, Entities.GET_TOPIA_ID);

        Set<String> selectedPlotIds;
        if (plotIds != null) {
            selectedPlotIds = Sets.newHashSet(plotIds);
        } else {
            selectedPlotIds = Sets.newHashSet();
        }

        Set<String> originePlotIds = indexedPlots.keySet();
        Set<String> plotRemovedRelationships = Sets.difference(originePlotIds, selectedPlotIds);

        // remove the relationship
        for (String plotRemovedRelationship : plotRemovedRelationships) {
            Plot plot = plotDao.forTopiaIdEquals(plotRemovedRelationship).findUnique();
            if (plot.getGrowingSystem()!=null && plot.getGrowingSystem().getTopiaId().contentEquals(growingSystemTopiaId)) {
                plot.setGrowingSystem(null);
            }
        }

        if (plotIds != null) {
            for (String plotId : plotIds) {
                Plot plot = indexedPlots.get(plotId);
                // Case where the plot was not affected to this given GrowingSystem.
                if (plot == null) {
                    plot = plotDao.forTopiaIdEquals(plotId).findUnique();
                    // If the plot is already affected to an other GrowingSystem there is no right to change it.
                    if (plot.getGrowingSystem() == null) {
                        plot.setGrowingSystem(growingSystem);
                        plot.setActive(true);
                    }
                }
            }
        }
    }

    @Override
    public List<Plot> findAllByGrowingSystem(GrowingSystem growingSystem) {
        List<Plot> result = plotDao.forGrowingSystemEquals(growingSystem).findAll();
        return result;
    }

    @Override
    public List<Plot> findAllFreeAndGrowingSystemPlots(GrowingSystem growingSystem, String growingPlanTopiaId) {

        // get growing plan instance
        GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanTopiaId).findUnique();

        // get free plots
        List<Plot> plots1 = plotDao.findAllFreePlotInDomain(growingPlan.getDomain().getTopiaId());
        List<Plot> result = Lists.newArrayList(plots1);

        // get all plots associated to current growing system
        if (growingSystem != null) {
            List<Plot> plots2 = plotDao.forGrowingSystemEquals(growingSystem).findAll();
            result.addAll(plots2);
        }

        return result;
    }

    @Override
    public List<Plot> getFreePlotForGrowingPlan(String growingPlanTopiaId) {

        // get growing plan instance
        GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanTopiaId).findUnique();

        // get free plots
        List<Plot> plots = plotDao.findAllFreePlotInDomain(growingPlan.getDomain().getTopiaId());
        return plots;
    }

    @Override
    public LinkedHashMap<Integer, String> getRelatedPlots(String plotCode) {
        LinkedHashMap<Integer, String> result = plotDao.findAllRelatedPlots(plotCode);
        return result;
    }

    @Override
    public LinkedHashMap<Integer, String> getRelatedZones(String zoneCode) {
        LinkedHashMap<Integer, String> result = zoneDao.findAllRelatedZones(zoneCode);
        return result;
    }

    @Override
    public boolean validMergingPlots(List<String> plotTopiaIds){
        // valid if plots have same growing system can be any.
        Long nbGS = plotDao.countAllGrowingSystemsForPlots(Sets.newConcurrentHashSet(plotTopiaIds));
        boolean result = nbGS == 0 || nbGS == 1;
        return result;
    }

    @Override
    public Plot mergePlots(List<String> plotTopiaIds) {
        Preconditions.checkArgument(plotTopiaIds.size() >= 2);
        Preconditions.checkArgument(validMergingPlots(plotTopiaIds));

        // find all instance for ids
        List<Plot> plots = plotDao.forTopiaIdIn(plotTopiaIds).findAll();

        // find plot with max surface
        Plot remainPlot = plots.get(0);
        for (int i = 1; i < plots.size(); i++) {
            Plot loopPlot = plots.get(i);
            if (loopPlot.getArea() > remainPlot.getArea()) {
                remainPlot = loopPlot;
            }
        }

        // merge plot
        for (Plot plot : plots) {
            if (!plot.equals(remainPlot)) {
                // move zones
                List<Zone> zones = zoneDao.forPlotEquals(plot).findAll();
                for (Zone zone : zones) {
                    // ajout du nom de la parcelle d'origine pour différencier les
                    // différente zone principale résultantes
                    String name = zone.getName();
                    name += " (" + plot.getName() + ")";
                    zone.setName(name);
                    // move zone
                    zone.setPlot(remainPlot);
                }
                // add area
                remainPlot.setArea(remainPlot.getArea() + plot.getArea());
                // copy previous lineage
                if (plot.getLineage() != null) {
                    remainPlot.addAllLineage(plot.getLineage());
                }
                // add lineage for delete
                remainPlot.addLineage(plot.getCode());
                // delete moved plot
                plotDao.delete(plot);
            }
        }

        getTransaction().commit();
        return remainPlot;
    }

    @Override
    public Plot duplicatePlot(String topiaId) {
        Plot plot = plotDao.forTopiaIdEquals(topiaId).findUnique();
        Plot result = extendPlot(new ExtendContext(true), plot.getDomain(), plot);
        getTransaction().commit();
        return result;
    }

    @Override
    public Plot extendPlot(ExtendContext extendContext, Domain clonedDomain, Plot plot) {

        // perform clone
        Binder<Plot, Plot> binder = BinderFactory.newBinder(Plot.class);
        Plot clonedPlot = plotDao.newInstance();
        binder.copyExcluding(plot, clonedPlot,
                Plot.PROPERTY_DOMAIN,
                Plot.PROPERTY_TOPIA_ID,
                Plot.PROPERTY_TOPIA_VERSION,
                Plot.PROPERTY_TOPIA_CREATE_DATE,
                Plot.PROPERTY_SOL_HORIZON,
                Plot.PROPERTY_PLOT_ZONINGS,
                Plot.PROPERTY_ADJACENT_ELEMENTS,
                Plot.PROPERTY_LINEAGE,
                Plot.PROPERTY_GROUND);
        clonedPlot.setDomain(clonedDomain);

        // always reset deleted zone
        clonedPlot.setDeletedZones(false);

        // break code is case of duplication
        if (extendContext.isDuplicateOnly()) {
            clonedPlot.setCode(UUID.randomUUID().toString());
            clonedPlot.addLineage(plot.getCode());
            clonedPlot.setGround(plot.getGround());
        } else {
            Preconditions.checkNotNull(extendContext.getGroundCache(), "Dans le cas d'une prolongation, cette map ne peut pas être null");
            Ground ground = extendContext.getGroundCache().get(plot.getGround());
            clonedPlot.setGround(ground);
        }

        // clone sol horizon
        if (plot.getSolHorizon() != null) {
            Binder<SolHorizon, SolHorizon> binderSH = BinderFactory.newBinder(SolHorizon.class);
            for (SolHorizon solHorizon : plot.getSolHorizon()) {
                SolHorizon solHorizonClone = solHorizonDao.newInstance();
                binderSH.copyExcluding(solHorizon, solHorizonClone,
                        SolHorizon.PROPERTY_TOPIA_ID,
                        SolHorizon.PROPERTY_TOPIA_VERSION,
                        SolHorizon.PROPERTY_TOPIA_CREATE_DATE);
                clonedPlot.addSolHorizon(solHorizonClone);
            }
        }

        // force collection copy instead of collection reference copy
        if (plot.getPlotZonings() != null) {
            clonedPlot.setPlotZonings(Lists.newArrayList(plot.getPlotZonings()));
        }
        if (plot.getAdjacentElements() != null) {
            clonedPlot.setAdjacentElements(Lists.newArrayList(plot.getAdjacentElements()));
        }

        // persist clone
        clonedPlot = plotDao.create(clonedPlot);

        // clone plot zones
        List<Zone> zones = zoneDao.forPlotEquals(plot).addEquals(Zone.PROPERTY_ACTIVE, true).findAll();
        Binder<Zone, Zone> zoneBinder = BinderFactory.newBinder(Zone.class);
        for (Zone zone : zones) {
            Zone zoneClone = zoneDao.newInstance();
            zoneBinder.copyExcluding(zone, zoneClone,
                    Zone.PROPERTY_TOPIA_ID,
                    Zone.PROPERTY_TOPIA_VERSION,
                    Zone.PROPERTY_TOPIA_CREATE_DATE,
                    Zone.PROPERTY_PLOT,
                    Zone.PROPERTY_LINEAGE);
            zoneClone.setPlot(clonedPlot);

            // break code is cas of duplication
            if (extendContext.isDuplicateOnly()) {
                zoneClone.setCode(UUID.randomUUID().toString());
                zoneClone.addLineage(zone.getCode());
            }

            zoneDao.create(zoneClone);
        }

        return clonedPlot;
    }

    @Override
    public List<Plot> findAllByDomain(Domain domain) {
        List<Plot> result = plotDao.findAllByDomainOrderByName(domain);
        result = anonymizeService.checkForPlotsAnonymization(result);
        return result;
    }

    @Override
    public List<Zone> getPlotZones(Plot plot) {
        List<Zone> result;
        if (plot.isPersisted()) {
            result = zoneDao.forPlotEquals(plot).findAll();
        } else {
            result = new ArrayList<Zone>();

            // add default zone
            Zone defaultZone = zoneDao.newInstance();
            defaultZone.setName(PLOT_DEFAULT_ZONE_NAME);
            defaultZone.setType(ZoneType.PRINCIPALE);
            defaultZone.setActive(true);
            defaultZone.setCode(UUID.randomUUID().toString());
            result.add(defaultZone);
        }
        return result;
    }

    @Override
    public List<MeasurementSession> getPlotMeasurementSessions(String plotTopiaId) {
        List<MeasurementSession> result = measurementSessionDao.forProperties(MeasurementSession.PROPERTY_ZONE + "." +
                Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_TOPIA_ID, plotTopiaId)
                .setOrderByArguments(MeasurementSession.PROPERTY_ZONE, MeasurementSession.PROPERTY_START_DATE, MeasurementSession.PROPERTY_END_DATE).findAll();
        return result;
    }

    @Override
    public UsageList<Zone> getZonesAndUsages(String plotTopiaId) {
        List<Zone> zones = zoneDao.forProperties(Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_TOPIA_ID, plotTopiaId).findAll();
        UsageList<Zone> result = entityUsageService.getZonesUsageList(zones);
        return result;
    }

    @Override
    public Plot unactivatePlot(String plotTopiaId, boolean activate) {
        Plot plot = plotDao.forTopiaIdEquals(plotTopiaId).findUnique();
        plot.setActive(activate);
        getTransaction().commit();
        return plot;
    }

    @Override
    public List<Zone> getZonesWithoutCycle(String zoneId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(zoneId));

        Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();

        List<Zone> result = zoneDao.findZonesWithoutCycle(zoneId, zone.getPlot().getDomain().getTopiaId());
        return result;
    }

    @Override
    public List<PlotDto> getPlots(Collection<String> plotIds) {
        List<Plot> plots = plotDao.forTopiaIdIn(plotIds).findAll();
        List<PlotDto> result = Lists.transform(plots, anonymizeService.getPlotToDtoFunction());
        return result;
    }

    @Override
    public InputStream exportPlotsAsXlsStream(List<String> plotIds) {
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        // get all possible bean infos
        PlotMainBeanInfo plotMainBeanInfo = newInstance(PlotMainBeanInfo.class);
        PlotZoningBeanInfo plotZoningBeanInfo = newInstance(PlotZoningBeanInfo.class);
        PlotEquipmentBeanInfo plotEquipmentBeanInfo = newInstance(PlotEquipmentBeanInfo.class);
        PlotGroundBeanInfo plotGroundBeanInfo = newInstance(PlotGroundBeanInfo.class);
        PlotAdjacentBeanInfo plotAdjacentBeanInfo = newInstance(PlotAdjacentBeanInfo.class);
        PlotZoneBeanInfo plotZoneBeanInfo = newInstance(PlotZoneBeanInfo.class);

        // add all sheet (necessary for empty export : model)
        ExportUtils.addAllBeanInfo(sheet, plotMainBeanInfo, plotZoningBeanInfo, plotEquipmentBeanInfo, plotGroundBeanInfo,
                plotAdjacentBeanInfo, plotZoneBeanInfo);

        // NuisibleMaladiesPhysiologiquesAux trop compliqué
        List<PlotExportEntity> zoningEntities = (List<PlotExportEntity>)sheet.get(plotZoningBeanInfo);

        try {
            if (CollectionUtils.isNotEmpty(plotIds)) {
                Iterable<Plot> plots = plotDao.forTopiaIdIn(plotIds).findAll();
                for (Plot plot : plots) {

                    // anonymize plot
                    plot = anonymizeService.checkForPlotAnonymization(plot);

                    // Common data for all tabs
                    PlotExportEntity model = new PlotExportEntity();
                    model.setCampaign(plot.getDomain().getCampaign());
                    model.setDomainName(plot.getDomain().getName());
                    model.setPlotName(plot.getName());

                    ExportUtils.export(sheet, model, plot, plotMainBeanInfo);

                    // zoning
                    if (plot.isOutOfZoning()) {
                        PlotExportEntity export = (PlotExportEntity) model.clone();
                        ExportUtils.setExtraField(export, Plot.PROPERTY_OUT_OF_ZONING, plot.isOutOfZoning());
                        zoningEntities.add(export);
                    } else {
                        Collection<RefParcelleZonageEDI> zonings = CollectionUtils.emptyIfNull(plot.getPlotZonings());
                        for (RefParcelleZonageEDI zoning : zonings) {
                            PlotExportEntity export = (PlotExportEntity) model.clone();
                            ExportUtils.setExtraField(export, Plot.PROPERTY_OUT_OF_ZONING, plot.isOutOfZoning());
                            ExportUtils.setExtraField(export, "zonage", zoning.getLibelle_engagement_parcelle());
                            zoningEntities.add(export);
                        }
                    }

                    // others
                    ExportUtils.export(sheet, model, plot, plotEquipmentBeanInfo);
                    ExportUtils.export(sheet, model, plot, plotGroundBeanInfo);
                    ExportUtils.export(sheet, model, CollectionUtils.emptyIfNull(plot.getAdjacentElements()), plotAdjacentBeanInfo);

                    // zones
                    List<Zone> zones = zoneDao.forPlotEquals(plot).findAll();
                    ExportUtils.export(sheet, model, zones, plotZoneBeanInfo);
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    @Override
    public void importPlotsForXlsStream(InputStream is) {

        // get all possible bean infos
        PlotMainBeanInfo plotMainBeanInfo = newInstance(PlotMainBeanInfo.class);
        PlotZoningBeanInfo plotZoningBeanInfo = newInstance(PlotZoningBeanInfo.class);
        PlotEquipmentBeanInfo plotEquipmentBeanInfo = newInstance(PlotEquipmentBeanInfo.class);
        PlotGroundBeanInfo plotGroundBeanInfo = newInstance(PlotGroundBeanInfo.class);
        PlotAdjacentBeanInfo plotAdjacentBeanInfo = newInstance(PlotAdjacentBeanInfo.class);
        PlotZoneBeanInfo plotZoneBeanInfo = newInstance(PlotZoneBeanInfo.class);

        // technical import
        EntityImporter importer = new EntityImporter();
        Map<EntityExportTabInfo, List<PlotExportEntity>> datas = importer.importFromStream(is,
                PlotExportEntity.class,
                plotMainBeanInfo,
                plotZoningBeanInfo,
                plotEquipmentBeanInfo,
                plotGroundBeanInfo,
                plotAdjacentBeanInfo,
                plotZoneBeanInfo);

        try {
            // custom import (main)
            List<PlotExportEntity> mainBeanInfo = datas.get(plotMainBeanInfo);
            MultiKeyMap<Object, Plot> plotCache = new MultiKeyMap<Object, Plot>();
            for (PlotExportEntity beanInfo : mainBeanInfo) {
                // get domain
                Domain domain = domainDao.forProperties(
                        Domain.PROPERTY_NAME, beanInfo.getDomainName(),
                        Domain.PROPERTY_CAMPAIGN, beanInfo.getCampaign()).findUnique();

                // create new instance
                Plot plot = plotDao.newInstance();
                plot.setName(beanInfo.getPlotName());
                plot.setDomain(domain);
                plot.setCode(UUID.randomUUID().toString());
                plot.setActive(true);

                // import
                ExportUtils.copyFields(beanInfo, plot,
                        Plot.PROPERTY_AREA,
                        Plot.PROPERTY_PAC_ILOT_NUMBER,
                        Plot.PROPERTY_MAX_SLOPE,
                        Plot.PROPERTY_WATER_FLOW_DISTANCE,
                        Plot.PROPERTY_BUFFER_STRIP,
                        Plot.PROPERTY_LATITUDE,
                        Plot.PROPERTY_LONGITUDE,
                        Plot.PROPERTY_COMMENT,
                        Plot.PROPERTY_ACTIVITY_END_COMMENT);

                // complex fields
                String codePostal = beanInfo.getExtraAsString(RefLocation.PROPERTY_CODE_POSTAL);
                if (StringUtils.isNotBlank(codePostal)) {
                    RefLocation location = locationDao.forCodePostalEquals(codePostal).findAny(); // FIXME any ?
                    plot.setLocation(location);
                }
                String growingSystemName = beanInfo.getExtraAsString("growingSystemName");
                if (StringUtils.isNotBlank(growingSystemName)) {
                    GrowingSystem growingSystem = growingSystemDao.forProperties(
                            GrowingSystem.PROPERTY_NAME, growingSystemName,
                            GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN, domain).findUniqueOrNull();
                    plot.setGrowingSystem(growingSystem);
                }

                // put in cache
                plot = plotDao.create(plot);
                plotCache.put(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName(), plot);
            }


            // custom import (equipement)
            List<PlotExportEntity> zoningBeanInfo = datas.get(plotZoningBeanInfo);
            for (PlotExportEntity beanInfo : zoningBeanInfo) {
                Plot plot = plotCache.get(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName());

                // import
                ExportUtils.copyFields(beanInfo, plot,
                        Plot.PROPERTY_OUT_OF_ZONING);

                // complex fields
                String zoningElement = beanInfo.getExtraAsString("zonage");
                if (StringUtils.isNotBlank(zoningElement)) {
                    RefParcelleZonageEDI refParcelleZonageEDI = parcelleZonageEDIDao.forLibelle_engagement_parcelleEquals(zoningElement).findUnique();
                    plot.addPlotZonings(refParcelleZonageEDI);
                }
            }

            // custom import (equipement)
            List<PlotExportEntity> equipementBeanInfo = datas.get(plotEquipmentBeanInfo);
            for (PlotExportEntity beanInfo : equipementBeanInfo) {
                Plot plot = plotCache.get(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName());

                // import
                ExportUtils.copyFields(beanInfo, plot,
                        Plot.PROPERTY_IRRIGATION_SYSTEM,
                        Plot.PROPERTY_IRRIGATION_SYSTEM_TYPE,
                        Plot.PROPERTY_POMP_ENGINE_TYPE,
                        Plot.PROPERTY_HOSES_POSITIONNING,
                        Plot.PROPERTY_FERTIGATION_SYSTEM,
                        Plot.PROPERTY_WATER_ORIGIN,
                        Plot.PROPERTY_DRAINAGE,
                        Plot.PROPERTY_DRAINAGE_YEAR,
                        Plot.PROPERTY_FROST_PROTECTION,
                        Plot.PROPERTY_FROST_PROTECTION_TYPE,
                        Plot.PROPERTY_HAIL_PROTECTION,
                        Plot.PROPERTY_RAINPROOF_PROTECTION,
                        Plot.PROPERTY_PEST_PROTECTION,
                        Plot.PROPERTY_OTHER_EQUIPMENT,
                        Plot.PROPERTY_EQUIPMENT_COMMENT);

            }

            // custom import (ground)
            List<PlotExportEntity> groundBeanInfo = datas.get(plotGroundBeanInfo);
            for (PlotExportEntity beanInfo : groundBeanInfo) {
                Plot plot = plotCache.get(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName());

                // import
                ExportUtils.copyFields(beanInfo, plot,
                        Plot.PROPERTY_SOL_STONINESS,
                        Plot.PROPERTY_SOL_MAX_DEPTH,
                        Plot.PROPERTY_SOL_ORGANIC_MATERIAL_PERCENT,
                        Plot.PROPERTY_SOL_BATTANCE,
                        Plot.PROPERTY_SOL_WATER_PH,
                        Plot.PROPERTY_SOL_HYDROMORPHISMS,
                        Plot.PROPERTY_SOL_LIMESTONE,
                        Plot.PROPERTY_SOL_ACTIVE_LIMESTONE,
                        Plot.PROPERTY_SOL_TOTAL_LIMESTONE,
                        Plot.PROPERTY_SOL_COMMENT);

                // complex fields
                String domainSol = beanInfo.getExtraAsString(Plot.PROPERTY_GROUND);
                if (StringUtils.isNotBlank(domainSol)) {
                    Ground ground = solDao.forProperties(
                            Ground.PROPERTY_DOMAIN, plot.getDomain(),
                            Ground.PROPERTY_NAME, domainSol).findUnique();
                    plot.setGround(ground);
                }
                String solSurface = beanInfo.getExtraAsString(Plot.PROPERTY_SURFACE_TEXTURE);
                if (StringUtils.isNotBlank(solSurface)) {
                    RefSolTextureGeppa refSolTextureGeppa = refSolTextureGeppaDao.forClasses_texturales_GEPAAEquals(solSurface).findUnique();
                    plot.setSurfaceTexture(refSolTextureGeppa);
                }
                String subsolSurface = beanInfo.getExtraAsString(Plot.PROPERTY_SURFACE_TEXTURE);
                if (StringUtils.isNotBlank(subsolSurface)) {
                    RefSolTextureGeppa refSolTextureGeppa = refSolTextureGeppaDao.forClasses_texturales_GEPAAEquals(subsolSurface).findUnique();
                    plot.setSubSoilTexture(refSolTextureGeppa);
                }
                String solDepth = beanInfo.getExtraAsString(Plot.PROPERTY_SOL_DEPTH);
                if (StringUtils.isNotBlank(solDepth)) {
                    RefSolProfondeurIndigo profondeur = refSolProfondeurIndigoDao.forLibelle_classeEquals(solDepth).findUnique();
                    plot.setSolDepth(profondeur);
                }

            }

            // custom import (equipement)
            List<PlotExportEntity> adjacentBeanInfo = datas.get(plotAdjacentBeanInfo);
            for (PlotExportEntity beanInfo : adjacentBeanInfo) {
                Plot plot = plotCache.get(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName());

                // complex fields
                String adjacentElement = beanInfo.getExtraAsString(Plot.PROPERTY_ADJACENT_ELEMENTS);
                if (StringUtils.isNotBlank(adjacentElement)) {
                    RefElementVoisinage refElementVoisinage = refElementVoisinageDao.forIae_nomEquals(adjacentElement).findUnique();
                    plot.addAdjacentElements(refElementVoisinage);
                }
            }

            // custom import (zones)
            List<PlotExportEntity> zoneBeanInfo = datas.get(plotZoneBeanInfo);
            for (PlotExportEntity beanInfo : zoneBeanInfo) {
                Plot plot = plotCache.get(beanInfo.getDomainName(), beanInfo.getCampaign(), beanInfo.getPlotName());

                Zone zone = new ZoneImpl();
                zone.setPlot(plot);

                ExportUtils.copyFields(beanInfo, zone,
                        Zone.PROPERTY_NAME,
                        Zone.PROPERTY_TYPE,
                        Zone.PROPERTY_AREA,
                        Zone.PROPERTY_LATITUDE,
                        Zone.PROPERTY_LONGITUDE,
                        Zone.PROPERTY_COMMENT);

                zone.setActive(true);
                zone.setCode(UUID.randomUUID().toString());
                zoneDao.create(zone);
            }

            // persist
            for (Plot plot : plotCache.values()) {
                if (plot.isPersisted()) {
                    plotDao.update(plot);
                }
            }
            getTransaction().commit();
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy fields", ex);
        }
    }

    @Override
    public List<Plot> getAllGrowingSystemPlot(GrowingSystem growingSystem) {
        Preconditions.checkNotNull(growingSystem);
        List<Plot> result = plotDao.forGrowingSystemEquals(growingSystem).findAll();
        return result;
    }

    @Override
    public void importPZ0Plots(Map<Class, ImportResults> allResults) {
        ImportResults plotsImportResults = allResults.remove(Plot.class);
        if (plotsImportResults != null && plotsImportResults.getIgnoredRecords() == 0) {
            try {
                Map<String, EntityAndDependencies> entitiesAndDependencies = plotsImportResults.getEntityAndDepsByCsvIds();
                int count = 1;
                int total = entitiesAndDependencies.values().size();
                for (EntityAndDependencies entityAndDependencies : entitiesAndDependencies.values()) {
                    PlotAndDependencies plotAndDependencies = (PlotAndDependencies) entityAndDependencies;
                    Plot plot = plotAndDependencies.getEntity();

                    long start = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info(String.format("Début sauvegarde de la parcelle %s - %d/%d.", plot.getName(), count++, total));
                    }

                    String domainId = plotAndDependencies.getDomainId();
                    String locationId = plotAndDependencies.getLocationId();
                    String growingSystemId = plotAndDependencies.getGrowingSystemId();
                    Collection<String> selectedPlotZoningIds = plotAndDependencies.getSelectedPlotZoningIds();
                    String selectedSolId = plotAndDependencies.getSelectedSolId();
                    String selectedSurfaceTextureId = plotAndDependencies.getSelectedSurfaceTextureId();
                    String selectedSubSoilTextureId = plotAndDependencies.getSelectedSubSoilTextureId();
                    String selectedSolDepthId = plotAndDependencies.getSelectedSolDepthId();
                    List<SolHorizonDto> solHorizons = plotAndDependencies.getSolHorizonDtos();
                    List<String> adjacentElementIds = plotAndDependencies.getAdjacentElementIds();
                    Map<String, Zone> zoneByCsvIds = plotAndDependencies.getZones();

                    Plot persistedPlot = createOrUpdatePlotWithoutCommit(plot, domainId, locationId, growingSystemId, selectedPlotZoningIds, selectedSolId, selectedSurfaceTextureId, selectedSubSoilTextureId, selectedSolDepthId, solHorizons, zoneByCsvIds.values(), adjacentElementIds);
                    entityAndDependencies.setEntity(persistedPlot);

                    List<Zone> persistedZones = getPlotZones(persistedPlot);
                    Map<String, Zone> persistedZoneByCode = Maps.uniqueIndex(persistedZones, GET_ZONE_CODE);

                    for (Map.Entry<String, Zone> zoneToSaveIter : zoneByCsvIds.entrySet()) {
                        Zone zoneToSave = zoneToSaveIter.getValue();
                        Zone persistedZone = persistedZoneByCode.get(zoneToSave.getCode());
                        zoneToSaveIter.setValue(persistedZone);
                    }

                    long p1 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Fin de sauvegarde de la parcelle, traitement réalisé en:" + (p1- start));
                    }
                }
            }  catch (Exception e) {
                throw (new AgrosystImportException("Echec de persistance des parcelles", e));
            }
        }
    }
}
