package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorYield.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorYield.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.HarvestingActionTopiaDao;
import fr.inra.agrosyst.api.entities.action.HarvestingYeald;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;

/**
 * Présentation de l’indicateur.
 * 
 * Le rendement décrit le volume ou la masse de la production principale ou des co-produits obtenus pour un ha et par culture. Pour les cultures assolées non fourragères, la production principale correspond aux grains ou à la masse de légumes récoltés. Pour les cultures fourragères, la production principale correspond à la biomasse récoltée. Pour les cultures pérennes, la production principale correspond au poids total de fruits récoltés. Le rendement de l’intervention pour la production principale peut se décliner en i) rendement qualité supérieure, ii) rendement qualité inférieure et iii) rendement non commercialisé. Les actions concernées par cet indicateur sont toutes les actions de type « récolte ».
 * 
 * Formule de calcul
 *   Rendementintervention = Production * PSCi
 *   
 * Avec,
 * Rendementintervention (t ha-1)
 * Production : quantité de production de co-produits ou de production principale (totale, qualité
 *   supérieure, qualité inférieure, non commercialisée) récoltée (t ha-1). Donnée saisie par l’utilisateur.
 * PSCi : proportion de surface concernée, sans unité. PSCi est calculé (voir le calcul ici) sur
 *   la base de données saisies par l’utilisateur.
 * @author Eric Chatellier
 */
public class IndicatorYield extends AbstractIndicator {

    protected HarvestingActionTopiaDao harvestingActionTopiaDao;

    public void setHarvestingActionTopiaDao(HarvestingActionTopiaDao harvestingActionTopiaDao) {
        this.harvestingActionTopiaDao = harvestingActionTopiaDao;
    }

    @Override
    public String getIndicatorCategory() {
        return "Indicateurs de résultats agronomiques";
    }

    @Override
    public String getIndicatorLabel(int i) {
        return "Rendement (t/ha)";
    }

    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {

        List<HarvestingAction> harvestingActions = harvestingActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();

        double meanYieldSum = 0;

        for (HarvestingAction action : harvestingActions) {
            if (action.getHarvestingYealds() != null) {
                for (HarvestingYeald yeald : action.getHarvestingYealds()) {
                    meanYieldSum += yeald.getYealdAverage();
                }
            }
        }


        double value = meanYieldSum * getToolPSCi(intervention);
        return newResult(value);
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {

        List<HarvestingAction> harvestingActions = harvestingActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();

        double meanYieldSum = 0;
        for (HarvestingAction action : harvestingActions) {
            if (action.getHarvestingYealds() != null) {
                for (HarvestingYeald yeald : action.getHarvestingYealds()) {
                    meanYieldSum += yeald.getYealdAverage();
                }
            }
        }

        double value = meanYieldSum * getToolPSCi(intervention);
        return newResult(value);
    }
}
