package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorWorkTime.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorWorkTime.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Calendar;
import java.util.Date;
import java.util.regex.Matcher;

import org.apache.commons.lang3.StringUtils;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;

/**
 * Temps de travail.
 * 
 * Présentation de l’indicateur
 * Le temps de travail est calculé sur la base de l’indicateur de Systerre1. Il correspond au temps
 * de travail lié aux interventions culturales et éventuellement au temps d’observation si celui-ci
 * a été enregistré. Il ne tient pas compte du temps consacré à la gestion du domaine. Le temps de
 * travail est exprimé en h ha-1. Dans certains cas, en particulier pour certaines interventions
 * manuelles, ce n’est pas le débit de chantier qui est saisi mais le temps passé à la parcelle.
 * Il faut dans ce cas utiliser directement cette information en la divisant par le nombre d’ha de
 * la parcelle. Toutes les interventions sont concernées par cet indicateur.
 * Formule de calcul
 * Cas où le débit de chantier est saisi en ha h-1 :
 * 
 * <pre>
 *                            1
 * Temps de travaili = ------------------ x PCSi
 *                     Débit de chantieri
 * </pre>
 * 
 * Avec,
 * Temps de travaili (h ha-1),
 * Débit de chantieri : Débit de chantier de l’attelage utilisé pour l’intervention (ha h-1). Donnée
 * issue d’un référentiel ou saisie/modifiée par l’utilisateur. Pour les automoteurs – récolteuses
 * (toutes filières), le débit de chantier = performance en ha/h dans le référentiel. Pour les attelages
 * (traction + outil), le débit de chantier = performance en ha/h de l’outil ayant le débit de
 * chantier le plus faible (performance minimum)
 * PSCi : proportion de surface concernée, sans unité. PSCi est calculé (voir le calcul ici) sur la
 * base de données saisies par l’utilisateur.
 * Cas où le débit de chantier n’est pas saisi :
 * 
 * Avec,
 * Temps de travaili (h ha-1),
 * Temps passéi : temps total passé à l’intervention sur la zone concernée (h). Donnée saisie par l’utilisateur.
 * Surface zone : surface de la zone concernée par l’intervention (ha). Donnée saisie par l’utilisateur.
 * Cas où la saisie et la donnée de référence ne sont pas exprimées en ha h-1 : 
 * Dans le cas des chantiers de pulvérisation et d’épandage (fumier, lisier, etc.) les débits de
 * chantier des outils sont exprimés en voy/h.
 * La conversion des voy/h en une performance de travail en ha/h, se fait suivant la formule : 
 * 
 * Avec : 
 * Performance (voy/h): donnée « performance » du référentiel BCMA.
 * Volume de chargement par voyage: donnée saisie par l’utilisateur.
 * Dosage : donnée saisie par l’utilisateur.
 * Dans le cas où les données de référence ne sont exprimées ni en ha/h ni en voy/h (bal/ha), ne pas
 * proposer de débit de chantier à l’utilisateur. L’utilisateur saisira directement la valeur de son
 * débit de chantier en ha/h (cf. cas où le débit de chantier est saisi en ha h-1).
 * 
 * Echelles de présentation de la variable calculée
 * Echelle de la variable d’intérêt :
 * Le temps de travail peut être calculé pour :
 * 1 intervention (échelle de saisie des informations / échelle utilisée pour la procédure de validation des informations saisies)
 * x interventions (permet les regroupements d’intervention pour décliner l’indicateur pour chaque mois de l’année : temps de travail janvier, temps de travail février, etc.).
 * 1 campagne culturale
 * 1 campagne agricole
 * toutes les campagnes agricoles
 * 
 * Echelle spatiale :
 * Le temps de travail peut être présenté aux échelles spatiales de :
 * la zone,
 * la parcelle,
 * la sole du SdC
 * le domaine
 * 
 * Echelle temporelle :
 * Le temps de travail peut être présenté aux échelles temporelles de :
 * 1 mois
 * 1 année
 * Plusieurs années
 * 
 * Les modalités de changement d’échelle sont présentées ici.
 * Données de référence
 * La donnée de référence pour le calcul du temps de travail est le débit de chantier qui dépend de l’attelage utilisé (Source BCMA). Onglets BCMA_xxx du fichier référentiel.
 * 
 * @author Eric Chatellier
 */
public class IndicatorWorkTime extends AbstractIndicator {

    @Override
    public String getIndicatorCategory() {
        return "Résultats socio-techniques";
    }

    @Override
    public String getIndicatorLabel(int i) {
        String result;
        if (0 <= i && i < 12) {
            result = String.format("Temps de travail %s (h/ha)", MONTHS[i]);
        } else {
            result = "Temps de travail (h/ha)";
        }

        return result;
    }

    /**
     * Compute and return work time on intervention.
     * 
     * @param intervention
     * @param growingSystem
     * @param campaigns
     * @param croppingPlanEntryCode
     * @return
     */
    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {

        Double[] workTimes = null;
        Double workTime = null;
        if (intervention.getWorkRate() != null) {
            // c'est un débit de chantier
            workTime = 1 / intervention.getWorkRate() * getToolPSCi(intervention);
        }
        
        if (workTime != null) {
            workTimes = newArray(13, 0.0); // 12 months + total
            // le temps de travail total de l'intervention est appliqué au premier mois de l'intervention
            int month = getStartMonth(intervention);
            workTimes[month] = workTime;
            workTimes[12] = workTime; // sum
        }

        return newResult(workTimes);
    }

    protected int getStartMonth(PracticedIntervention intervention) {
        int startMonth = -1;

        String startDate = intervention.getStartingPeriodDate();

        if (StringUtils.isEmpty(startDate)) {
            return startMonth;
        }

        Matcher mStartDate = PRACTICED_DATE_PATTERN.matcher(startDate);

        if (mStartDate.find()) {
            startMonth = Integer.parseInt(mStartDate.group(2)) - 1;
        }
        return startMonth;
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {

        Double[] workTimes = null;
        Double workTime = null;
        if (intervention.getWorkRate() != null) {
            // c'est un débit de chantier
            workTime = 1 / intervention.getWorkRate() * getToolPSCi(intervention);
        }
        
        if (workTime != null) {
            workTimes = newArray(13, 0.0); // 12 months + sum
            // le temps de travail total de l'intervention est appliqué au premier mois de l'intervention
            int month = getStartMonth(intervention);
            workTimes[month] = workTime;
            workTimes[12] = workTime; // sum
        }

        return newResult(workTimes);
    }
    
    protected int getStartMonth(EffectiveIntervention intervention) {
        int startMonth = -1;

        Date startDate = intervention.getStartInterventionDate();
        if (startDate == null) {
            return startMonth;
        }

        Calendar testCalendar = Calendar.getInstance();
        testCalendar.setTime(startDate);
        startMonth = testCalendar.get(Calendar.MONTH);

        return startMonth;
    }
}
