package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorTransitCount.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorTransitCount.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Calendar;
import java.util.Date;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;

/**
 * Nombre de passages.
 * 
 * Présentation de l’indicateur
 * Le nombre de passages correspond au nombre d’interventions culturales, manuelles ou de traction
 * animale, faites sur un hectare. Il est exprimé par type d’intervention en passage ha-1. Toutes
 * les interventions sont concernées par cet indicateur.
 * 
 * Formule de calcul
 * Le nombre de passages est une donnée saisie par l’utilisateur. Par défaut de saisie, sa valeur
 * est 1 pour chaque intervention.
 * 
 * @author Eric Chatellier
 */
public class IndicatorTransitCount extends AbstractIndicator {

    @Override
    public String getIndicatorCategory() {
        return "Résultats socio-techniques";
    }

    @Override
    public String getIndicatorLabel(int i) {
        String result;
        if (0 <= i && i < 12) {
            result = String.format("Nombre de passages %s (passage/ha)", MONTHS[i]);
        } else {
            result = "Nombre de passages (passage/ha)";
        }

        return result;
    }

    protected int getStartMonth(EffectiveIntervention intervention) {
        int startMonth = -1;

        Date startDate = intervention.getStartInterventionDate();
        if (startDate == null) {
            return startMonth;
        }

        Calendar testCalendar = Calendar.getInstance();
        testCalendar.setTime(startDate);
        startMonth = testCalendar.get(Calendar.MONTH);

        return startMonth;
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {

        Double[] transitCounts = null;
        Integer transitCount = intervention.getTransitCount();
        if (transitCount != null) {
            transitCounts = newArray(13, 0.0);
            
            int month = getStartMonth(intervention);
            transitCounts[month] = transitCount.doubleValue();
            transitCounts[12] = transitCount.doubleValue(); // sum
        }

        return transitCounts;
    }

    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem, String campaigns,
            String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {
        // Can't be computed for practiced
        return null;
    }
}
