package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorSurfaceUTH.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorSurfaceUTH.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.services.performance.IndicatorWriter;

/**
 * Surface par UTH.
 * 
 * Présentation de l’indicateur
 * La surface par UTH est calculée sur la base de l’indicateur de Systerre1. C’est une indication de
 * la charge de travail des travailleurs d’un domaine. Cet indicateur est exprimé en ha UTH-1.
 * Formule de calcul
 * <pre>
 *                   SAU totale
 * Surface par UTH = ----------
 *                   MO totale
 * </pre>
 * 
 * Avec,
 * Surface par UTH : Surface par unité de travail humain (ha UTH-1)
 * SAU totale : Surface Agricole Utile totale du domaine (ha). Donnée saisie par l’utilisateur.
 * MO totale : Main d’œuvre totale du domaine (UMO). Donnée saisie par l’utilisateur.
 * Echelles de présentation de la variable calculée
 * Le calcul est directement réalisé à l’échelle du domaine, en utilisant des données saisies au niveau du domaine.
 * Echelle temporelle :
 * La surface par UTH peut être présentée aux échelles temporelles de :
 * 1 année
 * plusieurs années
 * 
 * Les modalités de changement d’échelle sont présentées ici.
 * 
 * Données de référence
 * Aucune donnée de référence pour ce calcul
 * 
 * @author Eric Chatellier
 */
public class IndicatorSurfaceUTH extends AbstractIndicator {

    @Override
    public String getIndicatorCategory() {
        return "Résultats socio-techniques";
    }

    @Override
    public String getIndicatorLabel(int i) {
        return "Surface par UTH (ha)";
    }

    @Override
    public void computePracticed(IndicatorWriter writer, Domain domain) {
        if (domain.getUsedAgriculturalArea() != null) {
            double totalMO = getDomainTotalMO(domain);
            Double usedAgriArea = domain.getUsedAgriculturalArea();
            if (totalMO != 0 && usedAgriArea != null) {
                double areaBYMo = usedAgriArea / totalMO;
                writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(0), String.valueOf(domain.getCampaign()), domain, areaBYMo);
            }
        }
    }

    @Override
    public void computeEffective(IndicatorWriter writer, Domain domain) {
        if (domain.getUsedAgriculturalArea() != null) {
            double totalMO = getDomainTotalMO(domain);
            Double usedAgriArea = domain.getUsedAgriculturalArea();
            if (totalMO != 0 && usedAgriArea != null) {
                double areaBYMo = usedAgriArea / totalMO;
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(0), domain.getCampaign(), domain, areaBYMo);
            }
        }
    }

    public double getDomainTotalMO(Domain domain) {
        double result = 0;
        if (domain.getOtherWorkForce() != null) {
            result += domain.getOtherWorkForce();
        }
        if (domain.getPermanentEmployeesWorkForce() != null) {
            result += domain.getPermanentEmployeesWorkForce();
        }
        if (domain.getTemporaryEmployeesWorkForce() != null) {
            result += domain.getTemporaryEmployeesWorkForce();
        }
        return result;
    }

    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem, String campaigns,
            String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {
        // Can't compute at intervention level
        return null;
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone, CroppingPlanEntry croppingPlanEntry,
            CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {
        // Can't compute at intervention level
        return null;
    }
}
