package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorIPhy.java 4340 2014-09-17 12:44:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorIPhy.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;

import fr.inra.agrosyst.api.entities.action.SeedingProductInputImpl;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.keyvalue.MultiKey;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.collections4.map.MultiValueMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.BasicPlot;
import fr.inra.agrosyst.api.entities.BufferStrip;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.MaxSlope;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.ToolsCouplingTopiaDao;
import fr.inra.agrosyst.api.entities.WaterFlowDistance;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractActionTopiaDao;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.AbstractInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnection;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNodeTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectiveInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlot;
import fr.inra.agrosyst.api.entities.practiced.PracticedPlotTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemTopiaDao;
import fr.inra.agrosyst.api.entities.referential.PeriodeSemis;
import fr.inra.agrosyst.api.entities.referential.RefActaSubstanceActive;
import fr.inra.agrosyst.api.entities.referential.RefActaSubstanceActiveTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolAnnuelle;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolAnnuelleTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolPerenne;
import fr.inra.agrosyst.api.entities.referential.RefCouvSolPerenneTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefCultureEdiGroupeCouvSol;
import fr.inra.agrosyst.api.entities.referential.RefCultureEdiGroupeCouvSolTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefPhytoSubstanceActiveIphy;
import fr.inra.agrosyst.api.entities.referential.RefPhytoSubstanceActiveIphyTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSaActaIphy;
import fr.inra.agrosyst.api.entities.referential.RefSaActaIphyTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefZoneClimatiqueIphy;
import fr.inra.agrosyst.api.entities.referential.RefZoneClimatiqueIphyTopiaDao;
import fr.inra.agrosyst.api.entities.referential.TypeCulture;
import fr.inra.agrosyst.api.entities.referential.VitesseCouv;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoCaseGroundWater;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoCaseGroundWaterTopiaDao;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoFuzzySetGroundWater;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoFuzzySetGroundWaterTopiaDao;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesoRulesGroundWaterTopiaDao;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesuRunoffPotRulesParc;
import fr.inra.agrosyst.api.entities.referential.iphy.RefRcesuRunoffPotRulesParcTopiaDao;
import fr.inra.agrosyst.services.performance.IndicatorWriter;

public class IndicatorIPhy extends Indicator {

    private static final Log log = LogFactory.getLog(IndicatorIPhy.class);

    protected static final String[] RESULTS = new String[]{"rceso", "rcesu_rd", "rcesu_de", "rcair_vo", "rcair_de", "iphyeso", "iphyesu", "iphyair", "iphy"};

    @Override
    public String getIndicatorCategory() {
        return "I-Phy";
    }

    @Override
    public String getIndicatorLabel(int i) {
        return RESULTS[i];
    }

    /**
     * Résultat transitif servant à calculer les différentes étapes de Iphy.
     */
    static class IPhyStep {

        /** Couverture de sol de la culture. */
        protected double couvertureSol;

        /** les receso par substance active (pour le calcul de iphyeso). RefActaTraitementsProduits,RefPhytoSubstanceActiveIphy > double */
        protected MultiKeyMap<Object, Double> rcesos;
        /** les rcesu_rd par substance active (pour le calcul de iphyesu). RefActaTraitementsProduits,RefPhytoSubstanceActiveIphy > double */
        protected MultiKeyMap<Object, Double> rcesurds;
        /** les rcesu_de par substance active (pour le calcul de iphyesu). RefActaTraitementsProduits,RefPhytoSubstanceActiveIphy > double */
        protected MultiKeyMap<Object, Double> rcesudes;
        /** les iphyesu par substance active (pour le calcul de iphyair). RefActaTraitementsProduits,RefPhytoSubstanceActiveIphy > double */
        protected MultiKeyMap<Object, Double> rcairvos;
        /** les iphyair par substance active (pour le calcul de iphyair). RefActaTraitementsProduits,RefPhytoSubstanceActiveIphy > double */
        protected MultiKeyMap<Object, Double> rcairdes;
        /** les iphyeso par substance active. */
        protected MultiKeyMap<Object, Double> iphyesos;
        /** les iphyesu par substance active. */
        protected MultiKeyMap<Object, Double> iphyesus;
        /** les iphyair par substance active. */
        protected MultiKeyMap<Object, Double> iphyairs;
        /** les iphys par substance active. */
        protected MultiKeyMap<Object, Double> iphys;

        /** rceso (Eau souterraines). */
        protected double rceso;
        /** rcesu_rd (Eau superficielle par ruisellement/drainage). */
        protected double rcesu_rd;
        /** rcesu_de (Eau superficielle par dérive). */
        protected double rcesu_de;
        /** rcair_vo (Air par volatilisation). */
        protected double rcair_vo;
        /** rcair_de (Air par dérive). */
        protected double rcair_de;
        /** iphyeso. */
        protected double iphyeso;
        /** ihpyesu. */
        protected double iphyesu;
        /** iphyair. */
        protected double iphyair;
        /** iphy. */
        protected double iphy;
    }

    protected CroppingPlanEntryTopiaDao croppingPlanEntryDao;
    protected ToolsCouplingTopiaDao toolsCouplingDAO;
    protected AbstractActionTopiaDao abstractActionTopiaDao;
    protected AbstractInputTopiaDao abstractInputTopiaDao;

    protected PracticedSystemTopiaDao practicedSystemDao;
    protected PracticedSeasonalCropCycleTopiaDao practicedSeasonalCropCycleDao;
    protected PracticedPerennialCropCycleTopiaDao practicedPerennialCropCycleDao;
    protected PracticedCropCycleConnectionTopiaDao practicedCropCycleConnectionDao;
    protected PracticedInterventionTopiaDao practicedInterventionDAO;

    protected EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao;
    protected EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao;
    protected EffectiveInterventionTopiaDao effectiveInterventionTopiaDao;
    protected EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao;
    protected EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao;

    public void setCroppingPlanEntryDao(CroppingPlanEntryTopiaDao croppingPlanEntryDao) {
        this.croppingPlanEntryDao = croppingPlanEntryDao;
    }

    public void setToolsCouplingDAO(ToolsCouplingTopiaDao toolsCouplingDAO) {
        this.toolsCouplingDAO = toolsCouplingDAO;
    }

    public void setAbstractActionTopiaDao(AbstractActionTopiaDao abstractActionTopiaDao) {
        this.abstractActionTopiaDao = abstractActionTopiaDao;
    }

    public void setAbstractInputTopiaDao(AbstractInputTopiaDao abstractInputTopiaDao) {
        this.abstractInputTopiaDao = abstractInputTopiaDao;
    }

    public void setPracticedSystemDao(PracticedSystemTopiaDao practicedSystemDao) {
        this.practicedSystemDao = practicedSystemDao;
    }

    public void setPracticedSeasonalCropCycleDao(PracticedSeasonalCropCycleTopiaDao practicedSeasonalCropCycleDao) {
        this.practicedSeasonalCropCycleDao = practicedSeasonalCropCycleDao;
    }

    public void setPracticedPerennialCropCycleDao(PracticedPerennialCropCycleTopiaDao practicedPerennialCropCycleDao) {
        this.practicedPerennialCropCycleDao = practicedPerennialCropCycleDao;
    }

    public void setPracticedCropCycleConnectionDao(PracticedCropCycleConnectionTopiaDao practicedCropCycleConnectionDao) {
        this.practicedCropCycleConnectionDao = practicedCropCycleConnectionDao;
    }

    public void setPracticedInterventionDAO(PracticedInterventionTopiaDao practicedInterventionDAO) {
        this.practicedInterventionDAO = practicedInterventionDAO;
    }

    public void setEffectivePerennialCropCycleTopiaDao(EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao) {
        this.effectivePerennialCropCycleTopiaDao = effectivePerennialCropCycleTopiaDao;
    }

    public void setEffectiveInterventionTopiaDao(EffectiveInterventionTopiaDao effectiveInterventionTopiaDao) {
        this.effectiveInterventionTopiaDao = effectiveInterventionTopiaDao;
    }

    public void setEffectiveCropCycleConnectionTopiaDao(EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao) {
        this.effectiveCropCycleConnectionTopiaDao = effectiveCropCycleConnectionTopiaDao;
    }

    public EffectiveCropCycleConnectionTopiaDao getEffectiveCropCycleConnectionTopiaDao() {
        return effectiveCropCycleConnectionTopiaDao;
    }

    public void setEffectiveSeasonalCropCycleTopiaDao(EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao) {
        this.effectiveSeasonalCropCycleTopiaDao = effectiveSeasonalCropCycleTopiaDao;
    }

    public void setEffectiveCropCycleNodeTopiaDao(EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao) {
        this.effectiveCropCycleNodeTopiaDao = effectiveCropCycleNodeTopiaDao;
    }

    protected PracticedPlotTopiaDao practicedPlotTopiaDao;
    protected PhytoProductInputTopiaDao phytoProductInputTopiaDao;
    protected RefRcesoCaseGroundWaterTopiaDao rcesoCaseGroundWaterTopiaDao;
    protected RefRcesoFuzzySetGroundWaterTopiaDao rcesoFuzzySetGroundWaterTopiaDao;
    protected RefRcesoRulesGroundWaterTopiaDao rcesoRulesGroundWaterTopiaDao;
    protected RefActaSubstanceActiveTopiaDao actaSubstanceActiveTopiaDao;
    protected RefPhytoSubstanceActiveIphyTopiaDao phytoSubstanceActiveIphyTopiaDao;
    protected RefRcesuRunoffPotRulesParcTopiaDao rcesuRunoffPotRulesParcTopiaDao;
    protected RefSaActaIphyTopiaDao refSaActaIphyTopiaDao;
    protected RefZoneClimatiqueIphyTopiaDao refZoneClimatiqueIphyTopiaDao;
    protected RefCultureEdiGroupeCouvSolTopiaDao refCultureEdiGroupeCouvSolTopiaDao;
    protected RefCouvSolAnnuelleTopiaDao refCouvSolAnnuelleTopiaDao;
    protected RefCouvSolPerenneTopiaDao refCouvSolPerenneTopiaDao;

    public void setPracticedPlotTopiaDao(PracticedPlotTopiaDao practicedPlotTopiaDao) {
        this.practicedPlotTopiaDao = practicedPlotTopiaDao;
    }

    public void setPhytoProductInputTopiaDao(PhytoProductInputTopiaDao phytoProductInputTopiaDao) {
        this.phytoProductInputTopiaDao = phytoProductInputTopiaDao;
    }

    public void setRcesoCaseGroundWaterTopiaDao(RefRcesoCaseGroundWaterTopiaDao rcesoCaseGroundWaterTopiaDao) {
        this.rcesoCaseGroundWaterTopiaDao = rcesoCaseGroundWaterTopiaDao;
    }

    public void setRcesoFuzzySetGroundWaterTopiaDao(RefRcesoFuzzySetGroundWaterTopiaDao rcesoFuzzySetGroundWaterTopiaDao) {
        this.rcesoFuzzySetGroundWaterTopiaDao = rcesoFuzzySetGroundWaterTopiaDao;
    }

    public void setRcesoRulesGroundWaterTopiaDao(RefRcesoRulesGroundWaterTopiaDao rcesoRulesGroundWaterTopiaDao) {
        this.rcesoRulesGroundWaterTopiaDao = rcesoRulesGroundWaterTopiaDao;
    }

    public void setActaSubstanceActiveTopiaDao(RefActaSubstanceActiveTopiaDao actaSubstanceActiveTopiaDao) {
        this.actaSubstanceActiveTopiaDao = actaSubstanceActiveTopiaDao;
    }

    public void setPhytoSubstanceActiveIphyTopiaDao(RefPhytoSubstanceActiveIphyTopiaDao phytoSubstanceActiveIphyTopiaDao) {
        this.phytoSubstanceActiveIphyTopiaDao = phytoSubstanceActiveIphyTopiaDao;
    }

    public void setRcesuRunoffPotRulesParcTopiaDao(RefRcesuRunoffPotRulesParcTopiaDao rcesuRunoffPotRulesParcTopiaDao) {
        this.rcesuRunoffPotRulesParcTopiaDao = rcesuRunoffPotRulesParcTopiaDao;
    }

    public void setRefSaActaIphyTopiaDao(RefSaActaIphyTopiaDao refSaActaIphyTopiaDao) {
        this.refSaActaIphyTopiaDao = refSaActaIphyTopiaDao;
    }

    public void setRefZoneClimatiqueIphyTopiaDao(RefZoneClimatiqueIphyTopiaDao refZoneClimatiqueIphyTopiaDao) {
        this.refZoneClimatiqueIphyTopiaDao = refZoneClimatiqueIphyTopiaDao;
    }

    public void setRefCultureEdiGroupeCouvSolTopiaDao(RefCultureEdiGroupeCouvSolTopiaDao refCultureEdiGroupeCouvSolTopiaDao) {
        this.refCultureEdiGroupeCouvSolTopiaDao = refCultureEdiGroupeCouvSolTopiaDao;
    }

    public void setRefCouvSolAnnuelleTopiaDao(RefCouvSolAnnuelleTopiaDao refCouvSolAnnuelleTopiaDao) {
        this.refCouvSolAnnuelleTopiaDao = refCouvSolAnnuelleTopiaDao;
    }

    public void setRefCouvSolPerenneTopiaDao(RefCouvSolPerenneTopiaDao refCouvSolPerenneTopiaDao) {
        this.refCouvSolPerenneTopiaDao = refCouvSolPerenneTopiaDao;
    }

    protected MultiKeyMap<Object, Double[]> practicedGrowingSystemValues = new MultiKeyMap<Object, Double[]>();

    protected MultiKeyMap<Object, Double[]> practicedSystemValues = new MultiKeyMap<Object, Double[]>();

    public void computePracticed(IndicatorWriter writer, GrowingSystem growingSystem) {

        // multi map PracticedSystem, CroppingPlanEntryCode, CroppingPlanEntryCode > List<IPhyStep>
        MultiKeyMap<Object, List<IPhyStep>> practicedCroppingValues = new MultiKeyMap<Object, List<IPhyStep>>();

        List<PracticedSystem> practicedSystems = practicedSystemDao.forGrowingSystemEquals(growingSystem).findAll();

        for (PracticedSystem practicedSystem : practicedSystems) {
            String campaigns = practicedSystem.getCampaigns();

            // seasonal
            List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles = practicedSeasonalCropCycleDao.forPracticedSystemEquals(practicedSystem).findAll();
            for (PracticedSeasonalCropCycle cycle : practicedSeasonalCropCycles) {

                // stockage des iphystep par cycle
                Map<PracticedCropCycleConnection, List<IPhyStep>> cycleValues = Maps.newHashMap();

                Collection<PracticedCropCycleNode> nodes = cycle.getCropCycleNodes();
                if (CollectionUtils.isEmpty(nodes)) {
                    continue;
                }

                // recuperation des connections du cycle
                List<PracticedCropCycleConnection> cropCycleConnections = practicedCropCycleConnectionDao.forTargetIn(nodes).findAll();
                // recupération de toutes les interventions de toutes les connexions du cyle
                List<PracticedIntervention> interventions = practicedInterventionDAO.forPracticedCropCycleConnectionIn(cropCycleConnections).findAll();

                for (PracticedIntervention intervention : interventions) {
                    PracticedCropCycleConnection connection = intervention.getPracticedCropCycleConnection();

                    // get cropping plan entry and previous cropping plan entry
                    String croppingPlanEntryCode;
                    if (intervention.isIntermediateCrop() && !Strings.isNullOrEmpty(connection.getIntermediateCroppingPlanEntryCode())) {
                        croppingPlanEntryCode = connection.getIntermediateCroppingPlanEntryCode();
                    } else {
                        croppingPlanEntryCode = connection.getTarget().getCroppingPlanEntryCode();
                    }
                    String previousPlanEntryCode = connection.getSource().getCroppingPlanEntryCode();

                    IPhyStep step = manageIntervention(intervention, growingSystem, campaigns, croppingPlanEntryCode, previousPlanEntryCode, null);
                    if (step != null) {

                        // sortie résultat courant
                        Double[] interValues = newResult(step.rceso, step.rcesu_rd, step.rcesu_de, step.rcair_vo, step.rcair_de, step.iphyeso, step.iphyesu, step.iphyair, step.iphy);
                        CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
                        CroppingPlanEntry previousPlanEntry = croppingPlanEntryDao.forCodeEquals(previousPlanEntryCode).findAny();
                        List<AbstractAction> actions = abstractActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();
                        List<AbstractInput> inputs = abstractInputTopiaDao.findAllByPracticedIntervention(intervention);
                        for (int i = 0; i < interValues.length; i++) {
                            writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i),
                                    campaigns, growingSystem.getGrowingPlan().getDomain(), growingSystem, practicedSystem,
                                    croppingPlanEntry, previousPlanEntry, intervention, actions, inputs, interValues[i]);
                        }

                        // somme pour le cycle seulement
                        List<IPhyStep> previousValue = cycleValues.get(connection);
                        if (previousValue == null) {
                            previousValue = new ArrayList<IPhyStep>();
                            cycleValues.put(connection, previousValue);
                        }
                        previousValue.add(step);
                        
                        // somme pour Culture/precedent
                        List<IPhyStep> previous = practicedCroppingValues.get(practicedSystem, croppingPlanEntryCode, previousPlanEntryCode);
                        if (previous == null) {
                            previous = new ArrayList<IPhyStep>();
                            practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, previousPlanEntryCode, previous);
                        }
                        previous.add(step);
                    }
                }

                // get rank count and frequencies
                Map<PracticedCropCycleConnection, Double> cumulativeFrequencies = computeCumulativeFrequencies(cropCycleConnections);
                int campaignCount = getCampaignsCount(cropCycleConnections);
                Double[] result = null;
                // moyenne pour CA
                for (Map.Entry<PracticedCropCycleConnection, List<IPhyStep>> entry : cycleValues.entrySet()) {
                    PracticedCropCycleConnection conn = entry.getKey();
                    List<IPhyStep> values = entry.getValue();
                    //Cumul des substances actives selon l’algorithme ‘Christian Bockstaller’ (cumul des s.a. ‘CB’)
                    result = sumIPhyStep(values);
                    // application de la correction de fréquence cumulées
                    result = mults(result, cumulativeFrequencies.get(conn));
                }
                if (result != null && campaignCount > 0) {
                    result = divs(result, campaignCount);
                    // somme pour CA
                    Double[] previousValue = practicedGrowingSystemValues.get(campaigns, growingSystem);
                    if (previousValue == null) {
                        practicedGrowingSystemValues.put(campaigns, growingSystem, result);
                    } else {
                        Double[] sum = sum(previousValue, result);
                        practicedGrowingSystemValues.put(campaigns, growingSystem, sum);
                    }

                    Double[] psPreviousValue = practicedSystemValues.get(campaigns, growingSystem);
                    if (psPreviousValue == null) {
                        practicedGrowingSystemValues.put(campaigns, growingSystem, practicedSystem, result);
                    } else {
                        Double[] sum = sum(psPreviousValue, result);
                        practicedGrowingSystemValues.put(campaigns, growingSystem, practicedSystem, sum);
                    }
                }
            }

            // perenial
            List<PracticedPerennialCropCycle> practicedPerennialCropCycles = practicedPerennialCropCycleDao.forPracticedSystemEquals(practicedSystem).findAll();
            for (PracticedPerennialCropCycle cycle : practicedPerennialCropCycles) {
                Collection<PracticedCropCyclePhase> phases = cycle.getCropCyclePhases();
                List<PracticedIntervention> interventions = practicedInterventionDAO.forPracticedCropCyclePhaseIn(phases).findAll();
                for (PracticedIntervention intervention : interventions) {
                    String croppingPlanEntryCode = cycle.getCroppingPlanEntryCode();
                    PracticedCropCyclePhase phase = intervention.getPracticedCropCyclePhase();
                    IPhyStep step = manageIntervention(intervention, growingSystem, campaigns, croppingPlanEntryCode, null, phase);
                    if (step != null) {

                        // sortie résultat courant
                        CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
                        Double[] interValues = newResult(step.rceso, step.rcesu_rd, step.rcesu_de, step.rcair_vo, step.rcair_de, step.iphyeso, step.iphyesu, step.iphyair, step.iphy);
                        List<AbstractAction> actions = abstractActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();
                        List<AbstractInput> inputs = abstractInputTopiaDao.findAllByPracticedIntervention(intervention);
                        for (int i = 0; i < interValues.length; i++) {
                            writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i),
                                campaigns, growingSystem.getGrowingPlan().getDomain(), growingSystem, practicedSystem,
                                croppingPlanEntry, phase, intervention, actions, inputs, interValues[i]);
                        }

                        // somme pour CC
                        List<IPhyStep> previous = practicedCroppingValues.get(practicedSystem, croppingPlanEntryCode, phase);
                        if (previous == null) {
                            previous = new ArrayList<IndicatorIPhy.IPhyStep>();
                            practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, phase, previous);
                        }
                        previous.add(step);

                        // somme pour CA
                        Double[] previousValue = practicedGrowingSystemValues.get(campaigns, growingSystem);
                        if (previousValue == null) {
                            practicedGrowingSystemValues.put(campaigns, growingSystem, interValues);
                        } else {
                            Double[] sum = sum(previousValue, interValues);
                            practicedGrowingSystemValues.put(campaigns, growingSystem, sum);
                        }

                        Double[] psPreviousValue = practicedSystemValues.get(campaigns, growingSystem);
                        if (psPreviousValue == null) {
                            practicedGrowingSystemValues.put(campaigns, growingSystem, practicedSystem, interValues);
                        } else {
                            Double[] sum = sum(psPreviousValue, interValues);
                            practicedGrowingSystemValues.put(campaigns, growingSystem, practicedSystem, sum);
                        }
                        
                    }
                }
            }
        }

        // mise à l'echelle Intervention >> Culture (CC)
        for (Map.Entry<MultiKey<? extends Object>, List<IPhyStep>> entry : practicedCroppingValues.entrySet()) {
            PracticedSystem practicedSystem = (PracticedSystem)entry.getKey().getKey(0);
            String croppingPlanEntryCode = (String)entry.getKey().getKey(1);
            Object previousPlanEntryCodeOrPhase = entry.getKey().getKey(2);
            List<IPhyStep> steps = entry.getValue();
            Double[] values = sumIPhyStep(steps);
            CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
            if (previousPlanEntryCodeOrPhase instanceof String) {
                CroppingPlanEntry previousPlanEntry = croppingPlanEntryDao.forCodeEquals((String)previousPlanEntryCodeOrPhase).findAny();
                for (int i = 0; i < values.length; i++) {
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), practicedSystem.getCampaigns(),
                        growingSystem.getGrowingPlan().getDomain(),
                        growingSystem, practicedSystem, croppingPlanEntry, previousPlanEntry, values[i]);
                }
            } else {
                for (int i = 0; i < values.length; i++) {
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), practicedSystem.getCampaigns(),
                        growingSystem.getGrowingPlan().getDomain(),
                        growingSystem, practicedSystem, croppingPlanEntry, (PracticedCropCyclePhase)previousPlanEntryCodeOrPhase, values[i]);
                }
            }
        }
    }

    /**
     * Pour un ensemble de connexion, reconstitue le graph pour calculer les % cumulés de chaque
     * connexion depuis l'origine du graph.
     * 
     * @param cropCycleConnections
     * @return
     */
    protected Map<PracticedCropCycleConnection, Double> computeCumulativeFrequencies(List<PracticedCropCycleConnection> cropCycleConnections) {
        List<PracticedCropCycleConnection> localConnections = Lists.newArrayList(cropCycleConnections);

        // get precedence of each node
        MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection> precedenceMap = new MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection>();
        for (PracticedCropCycleConnection conn : localConnections) {
            precedenceMap.put(conn.getTarget(), conn);
        }

        // on calcul les frequences cumulées en aditionnant les frequences des précédences
        // et on multiplie le tout par la frequence de la connexion courante
        Map<PracticedCropCycleConnection, Double> result = Maps.newHashMap();
        for (PracticedCropCycleConnection conn : localConnections) {
            double frequency = getCumulativeFrequencies(precedenceMap, conn);
            result.put(conn, frequency);
        }
        return result;
    }

    /**
     * Retourne le pourcentage cumulé d'une connection récursivement jusqu'à ce qu'une connexion
     * n'ait plus de précédence.
     * 
     * @param precedenceMap precedence map
     * @param conn connection
     * @return cumulative frequency
     */
    protected double getCumulativeFrequencies(MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection> precedenceMap, PracticedCropCycleConnection conn) {
        double result;
        Collection<PracticedCropCycleConnection> precedences = precedenceMap.getCollection(conn.getSource());
        if (CollectionUtils.isEmpty(precedences)) {
            result = 1;
        } else {
            result = 0;
            for (PracticedCropCycleConnection precedence : precedences) {
                // manage loop connection on end cycle node
                if (precedence.getSource().getRank() == 0) {
                    // previous node = first node
                    // previous node's initial frequency * previous node's connection frequency
                    if (precedence.getSource().getInitNodeFrequency() == null) {
                        result = 1;
                    } else {
                        result = precedence.getSource().getInitNodeFrequency() / 100;
                    }
                    result *= precedence.getCroppingPlanEntryFrequency() / 100.0;
                // prevent infinite recursions
                } else if (precedence.getSource().getRank() < precedence.getTarget().getRank()) {
                    result += getCumulativeFrequencies(precedenceMap, precedence);
                }
            }
        }
        result *= conn.getCroppingPlanEntryFrequency() / 100.0;
        return result;
    }

    /**
     * Parmis les rangs définit sur le graph, détermine combien de campagnes existe réelement
     * dans les cas où des campagnes portent sur plusieurs rang.
     * 
     * @param cropCycleConnections cycle connections
     * @return campaigns count
     */
    protected int getCampaignsCount(List<PracticedCropCycleConnection> cropCycleConnections) {
        // les set permettent de comptes plusieurs fois les rangs, mais de ne les
        // prendre en compte qu'une seule fois au final
        Set<Integer> foundRanks = Sets.newHashSet();
        Set<Integer> sameCampaignRanks = Sets.newHashSet();

        for (PracticedCropCycleConnection cropCycleConnection : cropCycleConnections) {
            PracticedCropCycleNode sourceNode = cropCycleConnection.getSource();
            PracticedCropCycleNode targetNode = cropCycleConnection.getTarget();
            
            foundRanks.add(sourceNode.getRank());
            foundRanks.add(targetNode.getRank());
            if (targetNode.isSameCampaignAsPreviousNode()) {
                sameCampaignRanks.add(targetNode.getRank());
            }
        }
        foundRanks.removeAll(sameCampaignRanks);
        return foundRanks.size();
    }

    public void computePracticed(IndicatorWriter writer, Domain domain) {

        Map<String, Double[]> weightedValueSum = Maps.newLinkedHashMap();
        Map<String, Double[]> weightSum = Maps.newLinkedHashMap();
        for (Map.Entry<MultiKey<? extends Object>, Double[]> entry : practicedSystemValues.entrySet()) {
            String campaigns = (String)entry.getKey().getKey(0);
            GrowingSystem growingSystem = (GrowingSystem)entry.getKey().getKey(1);
            PracticedSystem practicedSystem = (PracticedSystem)entry.getKey().getKey(2);

            Double[] values = entry.getValue();

            // CA
            for (int i = 0; i < values.length; i++) {
                writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), campaigns, domain, growingSystem, practicedSystem, values[i]);
            }

            if (domain.getUsedAgriculturalArea() != null && growingSystem.getAffectedAreaRate() != null) {
                // somme des valeurs pondérée
                if (!weightedValueSum.containsKey(campaigns)) {
                    weightedValueSum.put(campaigns, newArray(values.length, 0.0));
                }
                for (int i = 0; i < values.length; i++) {
                    weightedValueSum.get(campaigns)[i] += values[i] * domain.getUsedAgriculturalArea() * growingSystem.getAffectedAreaRate();
                }

                // somme des poids
                if (!weightSum.containsKey(campaigns)) {
                    weightSum.put(campaigns, newArray(values.length, 0.0));
                }
                for (int i = 0; i < values.length; i++) {
                    weightSum.get(campaigns)[i] += domain.getUsedAgriculturalArea() * growingSystem.getAffectedAreaRate();
                }
            }
        }

        for (Map.Entry<String, Double[]> weightedValueEntry : weightedValueSum.entrySet()) {
            String campaigns = weightedValueEntry.getKey();
            Double[] values = weightedValueEntry.getValue();
            Double[] weightValues = weightSum.get(campaigns);
            for (int i = 0; i < values.length; i++) {
                if (weightValues[i] != 0) {
                    double weightedAverage = values[i] / weightValues[i];
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), campaigns, domain, weightedAverage);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't compute growing system scale with 0 weigth");
                    }
                }
            }
        }
    }

    /**
     * Reset state for domain.
     * 
     * @param domain domain to reset state
     */
    public void resetPracticed(Domain domain) {
        // cleanup for next iteration
        practicedGrowingSystemValues.clear();
    }

    protected MultiKeyMap<Object, List<IPhyStep>> effectiveCroppingValues = new MultiKeyMap<Object, List<IPhyStep>>();
    protected Map<Zone, List<IPhyStep>> effectiveZoneValues = Maps.newHashMap();
    protected Map<Plot, Double[]> effectivePlotValues = Maps.newHashMap();
    protected Map<GrowingSystem, Double[]> effectiveGrowingSystemValues = Maps.newHashMap();

    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone) {

        // perennial
        List<EffectivePerennialCropCycle> perennialCycles = effectivePerennialCropCycleTopiaDao.forZoneEquals(zone).findAll();
        for (EffectivePerennialCropCycle perennialCycle : perennialCycles) {
            EffectiveCropCyclePhase phase = perennialCycle.getPhase();
            List<EffectiveIntervention> interventions = effectiveInterventionTopiaDao.forEffectiveCropCyclePhaseEquals(phase).findAll();
            for (EffectiveIntervention intervention : interventions) {
                
                IPhyStep step = manageIntervention(writer, intervention, zone, perennialCycle.getCroppingPlanEntry(), null, phase);
                if (step != null) {

                    // sortie résultat courant
                    Double[] interValues = newResult(step.rceso, step.rcesu_rd, step.rcesu_de, step.rcair_vo, step.rcair_de, step.iphyeso, step.iphyesu, step.iphyair, step.iphy);
                    List<AbstractAction> actions = abstractActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();
                    List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
                    for (int i = 0; i < interValues.length; i++) {
                        writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i),
                            domain.getCampaign(), domain,
                            zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, perennialCycle.getCroppingPlanEntry(),
                            phase, intervention, actions, inputs, interValues[i]);
                    }

                    // somme pour CC
                    List<IPhyStep> previous = effectiveCroppingValues.get(perennialCycle.getCroppingPlanEntry(), phase);
                    if (previous == null) {
                        previous = new ArrayList<IPhyStep>();
                        effectiveCroppingValues.put(perennialCycle.getCroppingPlanEntry(), phase, previous);
                    }
                    previous.add(step);

                    // somme pour CA
                    List<IPhyStep> previousValue = effectiveZoneValues.get(zone);
                    if (previousValue == null) {
                        previousValue = new ArrayList<IPhyStep>();
                        effectiveZoneValues.put(zone, previousValue);
                    }
                    previousValue.add(step);
                }
            }
        }

        // seasonnal
        List<EffectiveSeasonalCropCycle> seasonnalCycles = effectiveSeasonalCropCycleTopiaDao.forZoneEquals(zone).findAll();
        for (EffectiveSeasonalCropCycle seasonnalCycle : seasonnalCycles) {
            Collection<EffectiveCropCycleNode> nodes = seasonnalCycle.getNodes();
            if (CollectionUtils.isEmpty(nodes)) {
                continue;
            }
            for (final EffectiveCropCycleNode node : nodes) {
                EffectiveCropCycleConnection connection = effectiveCropCycleConnectionTopiaDao.forTargetEquals(node).findUniqueOrNull();

                List<EffectiveIntervention> interventions = effectiveInterventionTopiaDao.forEffectiveCropCycleNodeEquals(node).findAll();

                for (EffectiveIntervention intervention : interventions) {
                    if (connection == null) {
                        // case of first node
                        CroppingPlanEntry croppingPlanEntry = node.getCroppingPlanEntry();
                        computeIphyStepIndicator(writer, domain, zone, node, intervention, croppingPlanEntry, null);
                    } else {
                        // get cropping plan entry and previous cropping plan entry
                        CroppingPlanEntry croppingPlanEntry;
                        if (intervention.isIntermediateCrop() && connection.getIntermediateCroppingPlanEntry() != null) {
                            croppingPlanEntry = connection.getIntermediateCroppingPlanEntry();
                        } else {
                            croppingPlanEntry = node.getCroppingPlanEntry();
                        }

                        // if source is null, it's a reference to last node of previous year
                        CroppingPlanEntry previousPlanEntry;
                        if (connection.getSource() == null) {
                            EffectiveCropCycleNode lastNode = effectiveCropCycleNodeTopiaDao.findLastNodeForPreviousCampaign(zone);
                            if (lastNode == null) {
                                previousPlanEntry = null;
                            } else {
                                // should not happen
                                previousPlanEntry = lastNode.getCroppingPlanEntry();
                            }
                        } else {
                            previousPlanEntry = connection.getSource().getCroppingPlanEntry();
                        }
                        if (previousPlanEntry == null) {
                            if (log.isWarnEnabled()) {
                                // should not happen
                                log.warn("Can't get previous cropping plan entry");
                            }
                        }
                        computeIphyStepIndicator(writer, domain, zone, node, intervention, croppingPlanEntry, previousPlanEntry);
                    }
                }

            }
        }

        // la map effectiveZoneValues est valuée plusieurs fois avec plusieurs zones par plusieurs
        // appel pour la mise à l'echelle de la methode suivante 
        // computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot)
        // on ne genere une ligne dans le fichier de sortie que pour la zone courante (s'il y a des valeurs)
        // et non pour toutes les zones
        List<IPhyStep> steps = effectiveZoneValues.get(zone);
        if (steps != null) {
            Double[] values = sumIPhyStep(steps);
            for (int i = 0; i < values.length; i++) {
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), zone.getPlot().getDomain().getCampaign(),
                    zone.getPlot().getDomain(), growingSystem, zone.getPlot(), zone, values[i]);
            }
        }
    }

    protected void computeIphyStepIndicator(IndicatorWriter writer, Domain domain, Zone zone, EffectiveCropCycleNode node, EffectiveIntervention intervention, CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry) {
        IPhyStep step = manageIntervention(writer, intervention, zone, node.getCroppingPlanEntry(), previousPlanEntry, null);

        if (step != null) {

            Double[] interValues = newResult(step.rceso, step.rcesu_rd, step.rcesu_de, step.rcair_vo, step.rcair_de, step.iphyeso, step.iphyesu, step.iphyair, step.iphy);

            // sortie résultat courant
            List<AbstractAction> actions = abstractActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();
            List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
            for (int i = 0; i < interValues.length; i++) {
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i),
                        domain.getCampaign(), domain,
                        zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, croppingPlanEntry, previousPlanEntry,
                        intervention, actions, inputs, interValues[i]);
            }

            // somme pour CC
            List<IPhyStep> previous = effectiveCroppingValues.get(croppingPlanEntry, previousPlanEntry);
            if (previous == null) {
                previous = new ArrayList<IPhyStep>();
                effectiveCroppingValues.put(croppingPlanEntry, previousPlanEntry, previous);
            }
            previous.add(step);

            // somme pour CA
            List<IPhyStep> previousValue = effectiveZoneValues.get(zone);
            if (previousValue == null) {
                previousValue = new ArrayList<IPhyStep>();
                effectiveZoneValues.put(zone, previousValue);
            }
            previousValue.add(step);
        }
    }

    /**
     * Zone &gt;&gt; Parcelle.
     * 
     * Pour les parcelles, on parcourt les valeurs pour les zones et on pondère par la surface
     * de la parcelle.
     */
    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot) {

        Double[] weightedValueSum = null;
        Double[] weightSum = null;

        // moyenne pondérée sur la surface
        for (Map.Entry<Zone, List<IPhyStep>> entry : effectiveZoneValues.entrySet()) {
            Zone zone = entry.getKey();
            List<IPhyStep> steps = entry.getValue();
            Double[] values = sumIPhyStep(steps);

            if (weightedValueSum == null) {
                weightedValueSum = newArray(values.length, 0.0);
                weightSum = newArray(values.length, 0.0);
            }
            for (int i = 0; i < values.length; i++) {
                weightedValueSum[i] += values[i] * zone.getArea();
                weightSum[i] += zone.getArea();
            }
        }

        if (weightSum != null) {
            effectivePlotValues.put(plot, newArray(weightedValueSum.length, 0.0));
            for (int i = 0; i < weightedValueSum.length; i++) {
                double plotValue = weightedValueSum[i] / weightSum[i];
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), plot.getDomain().getCampaign(),
                    plot.getDomain(), growingSystem, plot, plotValue);
                
                
                effectivePlotValues.get(plot)[i] = plotValue;
            }
        }
    }

    /**
     * Reset effective plot.
     * 
     * @param domain
     * @param growingSystem
     * @param plot
     */
    public void resetEffective(Domain domain, GrowingSystem growingSystem, Plot plot) {
        // cleanup for next iteration
        effectiveZoneValues.clear();
    }

    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem) {

        // toutes les interventions culturales d’une campagne culturale (CC)
        for (Map.Entry<MultiKey<? extends Object>, List<IPhyStep>> entry : effectiveCroppingValues.entrySet()) {
            List<IPhyStep> steps = entry.getValue();
            Double[] values = sumIPhyStep(steps);
            CroppingPlanEntry croppingPlanEntry = (CroppingPlanEntry)entry.getKey().getKey(0);
            Object previousPlanEntryOrPhase = entry.getKey().getKey(1);
            
            for (int i = 0; i < values.length; i++) {
                if (previousPlanEntryOrPhase instanceof CroppingPlanEntry) {
                    writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                        domain, growingSystem, croppingPlanEntry, (CroppingPlanEntry)previousPlanEntryOrPhase, values[i]);
                } else {
                    
                    writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                            domain, growingSystem, croppingPlanEntry, (EffectiveCropCyclePhase)previousPlanEntryOrPhase, values[i]);
                }
            }
        }

        // Parcelle >> SdC
        // plot: moyenne pondérée sur la surface
        Double[] weightedValueSum = null;
        Double[] weightSum = null;
        for (Map.Entry<Plot, Double[]> entry : effectivePlotValues.entrySet()) {
            Plot plot = entry.getKey();
            Double[] values = entry.getValue();

            if (weightedValueSum == null) {
                weightedValueSum = newArray(values.length, 0.0);
                weightSum = newArray(values.length, 0.0);
            }
            for (int i = 0; i < values.length; i++) {
                weightedValueSum[i] += values[i] * plot.getArea();
                weightSum[i] += plot.getArea();
            }
        }

        if (weightSum != null) {
            effectiveGrowingSystemValues.put(growingSystem, newArray(weightedValueSum.length, 0.0));
            for (int i = 0; i < weightedValueSum.length; i++) {
                double plotValue = weightedValueSum[i] / weightSum[i];
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                        domain, growingSystem, plotValue);
                
                effectiveGrowingSystemValues.get(growingSystem)[i] = plotValue;
            }
        }
    }

    public void resetEffective(Domain domain, GrowingSystem growingSystem) {
        // cleanup for next iteration
        effectiveCroppingValues.clear();
        // cleanup for next iteration
        effectivePlotValues.clear();
    }

    /**
     * Système de culture &gt;&gt; Domaine.
     * 
     * @param writer writer
     * @param domain domain
     */
    public void computeEffective(IndicatorWriter writer, Domain domain) {
        Double[] weightedValueSum = null;
        Double[] weightSum = null;

        // moyenne pondérée sur la surface
        for (Map.Entry<GrowingSystem, Double[]> entry : effectiveGrowingSystemValues.entrySet()) {
            GrowingSystem growingSystem = entry.getKey();
            Double[] values = entry.getValue();

            if (growingSystem.getAffectedAreaRate() != null) {
                if (weightedValueSum == null) {
                    weightedValueSum = newArray(values.length, 0.0);
                    weightSum = newArray(values.length, 0.0);
                }
                for (int i = 0; i < values.length; i++) {
                    weightedValueSum[i] += values[i] * growingSystem.getAffectedAreaRate();
                    weightSum[i] += growingSystem.getAffectedAreaRate();
                }
            }
        }

        if (weightSum != null) {
            for (int i = 0; i < weightedValueSum.length; i++) {
                if (weightSum[i] != 0) {
                    double domainValue = weightedValueSum[i] / weightSum[i];
                    writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                            domain, domainValue);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't compute growing system scale with 0 weigth");
                    }
                }
            }
        }
    }

    public void resetEffective(Domain domain) {
        
    }
    
    
    
    
    
    
    
    /**
     * Compute and return work time on intervention.
     * 
     * @param intervention
     * @param growingSystem
     * @param campaigns
     * @param croppingPlanEntryCode
     * @return
     */
    public IPhyStep manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {

        IPhyStep result = null;
        IPhyStep step = new IPhyStep();

        // calcul de la couverture de sol de la culture
        boolean applicable = computeCouvertureSol(step, null, intervention, null, null, croppingPlanEntryCode, previousPlanEntryCode);

        // get practiced plot
        PracticedPlot plot = null;
        if (applicable) {
            if (intervention.getPracticedCropCycleConnection() != null) {
                PracticedSystem practicedSystem = intervention.getPracticedCropCycleConnection().getTarget().getPracticedSeasonalCropCycle().getPracticedSystem();
                plot = practicedPlotTopiaDao.forPracticedSystemEquals(practicedSystem).findUniqueOrNull();
            } else {
                plot = practicedPlotTopiaDao.getPlotForPracticedCropCyclePhase(intervention.getPracticedCropCyclePhase());
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Impossible de déterminer la couverture de sol pour l'intervention " + intervention.getName());
            }
        }

        // get parameters
        if (plot != null) {
            List<PhytoProductInput> inputs = phytoProductInputTopiaDao.findAllByPracticedIntervention(intervention);

            // compute iphy
            applicable = applicable && computeIphyIndicator(step, plot, inputs, null, intervention);

            if (applicable) {
                result = step;
            }
        }

        return result;
    }

    public IPhyStep manageIntervention(IndicatorWriter writer, EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {

        IPhyStep result = null;
        IPhyStep step = new IPhyStep();

        // calcul de la couverture de sol de la culture
        boolean applicable = computeCouvertureSol(step, intervention, null, croppingPlanEntry, previousPlanEntry, null, null);

        if (applicable) {
            // get parameters
            Plot plot = zone.getPlot();
            List<PhytoProductInput> inputs = phytoProductInputTopiaDao.findAllByEffectiveIntervention(intervention);
    
            // compute iphy
            applicable = applicable && computeIphyIndicator(step, plot, inputs, intervention, null);
            
            if (applicable) {
                result = step;
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Impossible de déterminer la couverture de sol pour l'intervention " + intervention.getName());
            }
        }
        
        return result;
    }

    protected boolean computeIphyIndicator(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        boolean applicable = true;

        // Risque de contamination par voie de transfer
        applicable = applicable && computeRcesurdRceso(step, plot, inputs, effectiveIntervention, practicedIntervention);
        applicable = applicable && computeRcesude(step, plot, inputs, effectiveIntervention, practicedIntervention);
        applicable = applicable && computeRcairvo(step, plot, inputs, effectiveIntervention, practicedIntervention);
        applicable = applicable && computeRcairde(step, plot, inputs, effectiveIntervention, practicedIntervention);

        // Risque par compartiment
        applicable = applicable && computeIPhyeso(step, plot, inputs, effectiveIntervention, practicedIntervention);
        applicable = applicable && computeIPhyesu(step, plot, inputs, effectiveIntervention, practicedIntervention);
        applicable = applicable && computeIPhyair(step, plot, inputs, effectiveIntervention, practicedIntervention);

        // Agregation globale
        applicable = applicable && computeIPhy(step, plot, inputs, effectiveIntervention, practicedIntervention);

        return applicable;
    }

    /**
     * Calcul de la couverture au sol de la culture.
     * 
     * @param step step
     * @param effectiveIntervention effective intervention
     * @param practicedIntervention practiced intervention 
     * @param croppingPlanEntry cropping plan entry
     * @param previousPlanEntry previous cropping plan entry
     * @param croppingPlanEntryCode cropping plan entry code
     * @param previousPlanEntryCode previous cropping plan entry code
     * @return true if couverture sol can be computed
     */
    protected boolean computeCouvertureSol(IPhyStep step, EffectiveIntervention effectiveIntervention,
            PracticedIntervention practicedIntervention, CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry,
            String croppingPlanEntryCode, String previousPlanEntryCode) {
        Preconditions.checkArgument(effectiveIntervention != null ^ practicedIntervention != null);
        Preconditions.checkArgument(croppingPlanEntry != null ^ practicedIntervention != null);

        boolean result = false;

        // recherche de l'intervention qui contient l'action de semis
        EffectiveIntervention seendingEffectiveIntervention = null;
        PracticedIntervention seendingPracticedIntervention = null;
        if (effectiveIntervention != null) {
            if (effectiveIntervention.getEffectiveCropCycleNode() != null) {
                seendingEffectiveIntervention = effectiveInterventionTopiaDao.findFirstInterventionWithSeedingAction(
                    effectiveIntervention.getEffectiveCropCycleNode());
            } else {
                seendingEffectiveIntervention = effectiveInterventionTopiaDao.findFirstInterventionWithSeedingAction(
                        effectiveIntervention.getEffectiveCropCyclePhase());
            }
            if (seendingEffectiveIntervention == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find intervention of seeding action for " + effectiveIntervention.getName());
                }
                return result;
            }
        } else {
            if (practicedIntervention.getPracticedCropCycleConnection() != null) {
                seendingPracticedIntervention = practicedInterventionDAO.findFirstInterventionWithSeedingAction(
                    practicedIntervention.getPracticedCropCycleConnection());
            } else {
                seendingPracticedIntervention = practicedInterventionDAO.findFirstInterventionWithSeedingAction(
                        practicedIntervention.getPracticedCropCyclePhase());
            }
            if (seendingPracticedIntervention == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find intervention of seeding action for " + practicedIntervention.getName());
                }
                return result;
            }
        }

        // on cherche ensuite a déterminer la période de semis
        int monthIndex = -1;
        if (seendingEffectiveIntervention != null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(seendingEffectiveIntervention.getStartInterventionDate());
            monthIndex = calendar.get(Calendar.MONTH);
            // si c'est après le 15 octobre, on décale les mois
            if ((monthIndex == Calendar.OCTOBER && calendar.get(Calendar.DAY_OF_MONTH) >= 15) ||
                    monthIndex > Calendar.OCTOBER) {
                monthIndex++;
            }
        } else if (seendingPracticedIntervention != null) {
            Matcher matcher = PRACTICED_DATE_PATTERN.matcher(seendingPracticedIntervention.getStartingPeriodDate());
            if (matcher.find()) {
                monthIndex = Integer.parseInt(matcher.group(2)) - 1;
                if ((monthIndex == Calendar.OCTOBER && Integer.parseInt(matcher.group(1)) > 15) ||
                        monthIndex > Calendar.OCTOBER) {
                    monthIndex++;
                }
            }
        }
        PeriodeSemis periodSemis = PeriodeSemis.values()[monthIndex];

        // get cropping plan entry
        CroppingPlanEntry localCroppingPlanEntry = croppingPlanEntry;
        if (localCroppingPlanEntry == null) {
            localCroppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
        }

        // recherche de la couverture théorique max des especes
        List<CroppingPlanSpecies> croppingPlanSpecies = localCroppingPlanEntry.getCroppingPlanSpecies();
        if (CollectionUtils.isEmpty(croppingPlanSpecies)) {
            if (log.isWarnEnabled()) {
                log.warn("Can't find species for cropping plan entry " + localCroppingPlanEntry.getName());
            }
        }
        RefCultureEdiGroupeCouvSol refCultureEdiGroupeCouvSolMax = null;
        for (CroppingPlanSpecies species : croppingPlanSpecies) {
            RefEspece espece = species.getSpecies();
            RefCultureEdiGroupeCouvSol refCultureEdiGroupeCouvSol = refCultureEdiGroupeCouvSolTopiaDao.forProperties(
                    RefCultureEdiGroupeCouvSol.PROPERTY_CODE_ESPECE_BOTANIQUE, espece.getCode_espece_botanique(),
                    RefCultureEdiGroupeCouvSol.PROPERTY_CODE_QUALIFIANT_AEE, espece.getCode_qualifiant_AEE(),
                    RefCultureEdiGroupeCouvSol.PROPERTY_CODE_TYPE_SAISONNIER_AEE, espece.getCode_type_saisonnier_AEE(),
                    RefCultureEdiGroupeCouvSol.PROPERTY_CODE_DESTINATION_AEE, espece.getCode_destination_AEE()).findUniqueOrNull();
            if (refCultureEdiGroupeCouvSol == null) {
                if (log.isWarnEnabled()) {
                    log.warn(String.format("Can't find CultureEdiGroupeCouvSol for %s/%s/%s/%s", espece.getCode_espece_botanique(), espece.getCode_qualifiant_AEE(),
                            espece.getCode_type_saisonnier_AEE(), espece.getCode_destination_AEE()));
                }
                return result;
            }
            
            // conserve seulement l'espece qui a la plus grande couverture de sol max
            if (refCultureEdiGroupeCouvSolMax == null || refCultureEdiGroupeCouvSolMax.getTaux_couverture_max() < refCultureEdiGroupeCouvSol.getTaux_couverture_max()) {
                refCultureEdiGroupeCouvSolMax = refCultureEdiGroupeCouvSol;
            }
        }

        // info nécéssaires au calcul
        double couvSolMax = refCultureEdiGroupeCouvSolMax.getTaux_couverture_max();
        VitesseCouv vitesseCouv = refCultureEdiGroupeCouvSolMax.getVitesseCouv();
        int interventionStartMonth = 0;
        int interventionStartDay = 0;
        if (effectiveIntervention != null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(effectiveIntervention.getStartInterventionDate());
            interventionStartMonth = calendar.get(Calendar.MONTH) + 1;
            interventionStartDay = calendar.get(Calendar.DAY_OF_MONTH) + 1;
        } else {
            Matcher matcher = PRACTICED_DATE_PATTERN.matcher(practicedIntervention.getStartingPeriodDate());
            if (matcher.find()) {
                interventionStartMonth = Integer.parseInt(matcher.group(2));
                interventionStartDay = Integer.parseInt(matcher.group(1));
            }
        }

        switch (refCultureEdiGroupeCouvSolMax.getTypeCulture()) {
        case ANNUELLE:
        case PLURI_ANNUELLE:
            
            //Si on est en première année (culture précédente=une autre culture; ou présence d'une opération de semis sur la culture), on utilise le même principe que pour les annuelles 
            //On recherche la vitesse de couverture de la culture (CulturesEDI_Groupe-couv-sol), son taux de couverture max (couv_sol_max dans CulturesEDI_Groupe-couv-sol), la période de semis, la date du traitement
            //on recherche le taux de couverture (couv_sol_théorique) dans la table Couv-sol_Annuelles (en fonction de la vitesse de couverture de la culture, de la période de semis et de la date du traitement)
            //Couv_sol_réel=min(couv_sol_théorique;couv_sol_max)
            //Si on est en deuxième année ou plus (culture précédente=la même), alors Couv_sol_réel=100%  
            if (refCultureEdiGroupeCouvSolMax.getTypeCulture() == TypeCulture.PLURI_ANNUELLE && ((croppingPlanEntry != null && croppingPlanEntry.equals(previousPlanEntry)) ||
                    croppingPlanEntryCode != null && croppingPlanEntryCode.equals(previousPlanEntryCode))) {
                step.couvertureSol = 1;
                result = true;
            } else {

                //On recherche la vitesse de couverture de la culture (CulturesEDI_Groupe-couv-sol), son taux de couverture max (couv_sol_max dans CulturesEDI_Groupe-couv-sol), la période de semis, la date du traitement
                //on recherche le taux de couverture (couv_sol_théorique) dans la table Couv-sol_Annuelles (en fonction de la vitesse de couverture de la culture, de la période de semis et de la date du traitement)
                //Couv_sol_réel=min(couv_sol_théorique;couv_sol_max)
                
                // on recupére les 36 valeurs (3*12 à checker)
                List<RefCouvSolAnnuelle> refCouvSolAnnuelles = refCouvSolAnnuelleTopiaDao.forProperties(
                        RefCouvSolAnnuelle.PROPERTY_PERIODE_SEMIS, periodSemis,
                        RefCouvSolAnnuelle.PROPERTY_VITESSE_COUV, vitesseCouv).findAll();
                RefCouvSolAnnuelle matchingRefCouvSolAnnuelle = null;
                for (RefCouvSolAnnuelle refCouvSolAnnuelle : refCouvSolAnnuelles) {
                    Matcher debutInter = PRACTICED_DATE_PATTERN.matcher(refCouvSolAnnuelle.getDebut_inter());
                    Matcher finInter = PRACTICED_DATE_PATTERN.matcher(refCouvSolAnnuelle.getFin_inter());
                    if (debutInter.find() && finInter.find()) {
                        int debutInterMonth = Integer.parseInt(debutInter.group(2));
                        int debutInterDay = Integer.parseInt(debutInter.group(1));
                        int finInterMonth = Integer.parseInt(finInter.group(2));
                        int finInterDay = Integer.parseInt(finInter.group(1));
                    
                        if (debutInterMonth == interventionStartMonth && interventionStartMonth == finInterMonth &&
                                debutInterDay <= interventionStartDay && interventionStartDay <= finInterDay) {
                            matchingRefCouvSolAnnuelle = refCouvSolAnnuelle;
                        }
                    }
                }

                // ca peux se produire si vraiment les dates ne sont pas cohérentes
                if (matchingRefCouvSolAnnuelle != null) {
                    step.couvertureSol = Math.min(matchingRefCouvSolAnnuelle.getCouv(), couvSolMax);
                    result = true;
                }
            }

            break;
            // FIXME echatellier 20140220: il manque le cas des cultures intermédiaires
        case PERENNE:
            //On recherche la vitesse de couverture de la culture (CulturesEDI_Groupe-couv-sol), son taux de couverture max (couv_sol_max dans CulturesEDI_Groupe-couv-sol), la date du traitement
            //On recherche la phase de production et le type d'enherbement (Total/partiel/pas d'enherbement, cf onglet 'phase de production des cultures pérennes' dans Agrosyst)
            //on recherche le taux de couverture (couv_sol_théorique) dans la table Couv-sol_Pérennes (en fonction de la vitesse de couverture de la culture, de la phase de production et de la date du traitement)
            //Si type_enherbement="Total" alors Couv_sol_réel=100%
            //Si type_enherbement="Partiel" alors Couv_sol_réel=max[50%;min(couv_sol_max;couv_sol_théorique)]
            //Si type_enherbement="Pas d'enherbement" alors Couv_sol_réel=min(couv_sol_max;couv_sol_théorique)

            WeedType weedType;
            if (practicedIntervention != null) {
                PracticedPerennialCropCycle practicedPerennialCropCycle = practicedPerennialCropCycleDao.findPerennialCropCycleForIntervention(practicedIntervention);
                weedType = practicedPerennialCropCycle.getWeedType();
            } else {
                EffectivePerennialCropCycle effectivePerennialCropCycle = effectivePerennialCropCycleTopiaDao.findPerennialCropCycleForIntervention(effectiveIntervention);
                weedType = effectivePerennialCropCycle.getWeedType();
            }

            if (weedType == null) {
                result = false;
            }
            else if (weedType == WeedType.TOTAL) {
                step.couvertureSol = 1;
                result = true;
            } else {
                CropCyclePhaseType cropCyclePhaseType = null;
                if (practicedIntervention != null && practicedIntervention.getPracticedCropCyclePhase() != null) {
                    cropCyclePhaseType = practicedIntervention.getPracticedCropCyclePhase().getType();
                } else if (effectiveIntervention != null && effectiveIntervention.getEffectiveCropCyclePhase() != null) {
                    cropCyclePhaseType = effectiveIntervention.getEffectiveCropCyclePhase().getType();
                } else {
                    return result;
                }
    
                List<RefCouvSolPerenne> refCouvSolPerennes = refCouvSolPerenneTopiaDao.forProperties(
                        RefCouvSolPerenne.PROPERTY_PHASE, cropCyclePhaseType,
                        RefCouvSolPerenne.PROPERTY_VITESSE_COUV, vitesseCouv).findAll();
                RefCouvSolPerenne matchingRefCouvSolPerenne = null;
                for (RefCouvSolPerenne refCouvSolPerenne : refCouvSolPerennes) {
                    Matcher debutInter = PRACTICED_DATE_PATTERN.matcher(refCouvSolPerenne.getDebut_inter());
                    Matcher finInter = PRACTICED_DATE_PATTERN.matcher(refCouvSolPerenne.getFin_inter());
                    if (debutInter.find() && finInter.find()) {
                        int debutInterMonth = Integer.parseInt(debutInter.group(2));
                        int debutInterDay = Integer.parseInt(debutInter.group(1));
                        int finInterMonth = Integer.parseInt(finInter.group(2));
                        int finInterDay = Integer.parseInt(finInter.group(1));
                        
                        if (debutInterMonth == interventionStartMonth && interventionStartMonth == finInterMonth &&
                                debutInterDay <= interventionStartDay && interventionStartDay <= finInterDay) {
                            matchingRefCouvSolPerenne = refCouvSolPerenne;
                        }
                    }
                }

                // cela peut arriver si les saisies utilisateurs concernant les dates ne sont pas cohérentes
                if (matchingRefCouvSolPerenne != null) {
                    if (weedType == WeedType.PARTIEL) {
                        step.couvertureSol = Math.max(0.5, Math.min(couvSolMax, matchingRefCouvSolPerenne.getCouv()));
                    } else if (weedType == WeedType.PAS_ENHERBEMENT) {
                        step.couvertureSol = Math.min(couvSolMax, matchingRefCouvSolPerenne.getCouv());
                    }
                    result = true;
                }
            }

            break;
        }

        return result;
    }

    /**
     * rcesu_rd AND rceso.
     * 
     * This two part are linked when plot is drained.
     * Can't be calculated standalone
     */
    protected boolean computeRcesurdRceso(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        Preconditions.checkArgument(effectiveIntervention == null ^ practicedIntervention == null);

        // 1-rcesu_rd. Calcul du potentiel de ruissellement
        // Rechercher la texture, pente, battance, hydromorphie de la parcelle
        String soilTexture = null;
        if (plot.getSurfaceTexture() == null) {
            return false;
        }
        soilTexture = plot.getSurfaceTexture().getClasse_INDIGO();
        boolean battance = plot.isSolBattance();
        MaxSlope maxSlope = plot.getMaxSlope();
        if (maxSlope == null) {
            return false;
        }
        boolean hydromorphisms = plot.isSolHydromorphisms();

        // A partir de ces données rechercher le potentiel de ruissellement (Runoff potential)  dans la table "RunOff_Pot_Rules_Parc"
        // Table RunOff_Pot_Rules_Parc
        if (log.isDebugEnabled()) {
            log.debug(String.format("Search runoff potential for %s, %s, %s, %s", soilTexture, battance, hydromorphisms, maxSlope));
        }
        RefRcesuRunoffPotRulesParc rcesuRunoffPotRulesParc = rcesuRunoffPotRulesParcTopiaDao.forProperties(
                RefRcesuRunoffPotRulesParc.PROPERTY_ACTIVE, true,
                RefRcesuRunoffPotRulesParc.PROPERTY_SOIL_TEXTURE, soilTexture,
                RefRcesuRunoffPotRulesParc.PROPERTY_BATTANCE, battance,
                RefRcesuRunoffPotRulesParc.PROPERTY_HYDROMORPHISMS, hydromorphisms,
                RefRcesuRunoffPotRulesParc.PROPERTY_SLOPE, maxSlope
                ).findUniqueOrNull();

        if (rcesuRunoffPotRulesParc == null) {
            if (log.isWarnEnabled()) {
                log.warn(String.format(" Can't find %s with active %s, soilTexture %s, battance %s, hydromorphisms %s, slope %s", RefRcesuRunoffPotRulesParc.class.getName(), true, soilTexture, battance, hydromorphisms, maxSlope));
            }
            return false;
        }

        double runoffPotential = rcesuRunoffPotRulesParc != null ? rcesuRunoffPotRulesParc.getRunoff_potential() : 0d;

        // Rechercher le facteur correcteur
        //double facteurCorrecteur = 1; // FIXME comment on fait dans agrosyst
        // Rechercher le travail du sol de la parcelle
        // Table Soil_work (ou tillage)
        //double tillage = 1; // FIXME comment on fait dans agrosyst ?
        // A partir de ces données rechercher facteur correcteur tillage (Factor_tillage)
        // Table RunOff_Pot_Rules_Tillage
        double factorTillage = 1;

        //Rechercher la largeur de  bande enherbée
        //Table   Weeded_banded (ou buffer_strip)
        BufferStrip bufferStrip = plot.getBufferStrip();
        //A partir de ces données rechercher facteur correcteur bande enherbée (=Factor_bufferstrip)
        //Table   RunOffDrift_Pot_Rules_buffer
        double factorBufferstrip;
        if (bufferStrip == BufferStrip.NONE) {
            factorBufferstrip = 1.0;
        } else if (bufferStrip == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
            factorBufferstrip = 0.5;
        } else {
            factorBufferstrip = 0.25;
        }

        // Calcul de la variable Pot_Ruis
        // PotRuis = Runoff potential * Factor_tillage * Factor_bufferstrip
        double potRuis = runoffPotential * factorTillage * factorBufferstrip;



        //1-rceso. Recherche de données fixes
        //Rechercher période application = f(date application produit phyto)
        //01/09   31/12   Automne
        //31/12   31/03   Hiver
        //01/04   31/08   Printemps
        int month;
        if (effectiveIntervention != null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(effectiveIntervention.getStartInterventionDate());
            month = calendar.get(Calendar.MONTH) + 1;
        } else {
            String startDate = practicedIntervention.getStartingPeriodDate();
            String startMonth = StringUtils.substringAfterLast(startDate, "/");
            month = Integer.parseInt(startMonth);
        }
        String applicationPeriod;
        if (month < 4) {
            applicationPeriod = "Hiver";
        } else if (month < 9) {
            applicationPeriod = "Printemps";
        } else {
            applicationPeriod = "Automne";
        }

        //Rechercher zone climatique en fonction échelle
        RefLocation location = plot.getLocation();
        if (location == null) {
            if (log.isWarnEnabled()) {
                log.warn("No location found for plot " + plot.getName());
            }
            return false;
        }
        String departement = location.getDepartement().toUpperCase();
        RefZoneClimatiqueIphy zoneClimatiqueIphy = refZoneClimatiqueIphyTopiaDao.forProperties(
                RefZoneClimatiqueIphy.PROPERTY_ACTIVE, true,
                RefZoneClimatiqueIphy.PROPERTY_DEPARTEMENT, departement).findUniqueOrNull();

        if (zoneClimatiqueIphy == null) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("Can't find %s with active %s, departement %s",
                        RefZoneClimatiqueIphy.class.getName(), true, departement));
            }
            return false;
        }

        int zoneClimatique =zoneClimatiqueIphy.getZone_climatique();

        //Rechercher  texutre surface, texture subsoil
        //Transfomer texture surface, texture subsoil en 3 catégorie
        if (plot.getSurfaceTexture() == null) {
            if (log.isWarnEnabled()) {
                log.warn("No surface texture found for plot " + plot.getName());
            }
            return false;
        }
        String textureSurface = plot.getSurfaceTexture().getTexture_iphy();
        if (plot.getSubSoilTexture() == null) {
            if (log.isWarnEnabled()) {
                log.warn("No sub soil texture found for plot " + plot.getName());
            }
            return false;
        }
        String textureSubSoil = plot.getSubSoilTexture().getTexture_iphy();

        //Recherche le casenumber = f(zone climatique, texture surface, texture subsoil dans table parcelle)
        if (log.isDebugEnabled()) {
            log.debug(String.format("Searching caseNumber for %d,%s,%s,%s", zoneClimatique, applicationPeriod, textureSurface, textureSubSoil));
        }
        RefRcesoCaseGroundWater caseGroundWater = rcesoCaseGroundWaterTopiaDao.forProperties(
                RefRcesoCaseGroundWater.PROPERTY_ACTIVE, true,
                RefRcesoCaseGroundWater.PROPERTY_CLIMATE, zoneClimatique,
                RefRcesoCaseGroundWater.PROPERTY_APPLICATION_PERIOD, applicationPeriod,
                RefRcesoCaseGroundWater.PROPERTY_SURFACE_TEXTURE, textureSurface,
                RefRcesoCaseGroundWater.PROPERTY_SUBSOIL_TEXTURE, textureSubSoil).findUniqueOrNull();

        if (caseGroundWater == null) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("Can't find %s with active %s, climate %s, applicationPeriod %s, surfaceTexture %s, subsoilTexture %s",
                        RefRcesoCaseGroundWater.class.getName(), true, zoneClimatique, applicationPeriod, textureSurface, textureSubSoil));
            }
            return false;
        }


        int caseNumber = caseGroundWater.getCaseNumber();

        if (log.isDebugEnabled()) {
            log.debug(String.format(" caseNumber is %d", caseNumber));
        }

        //Rechercher variable taux MO et profondeur réelle dans table parcelle
        Double tauxMo = plot.getSolOrganicMaterialPercent();
        if (tauxMo == null) {
            if (log.isWarnEnabled()) {
                log.warn("No organic material percent found for plot " + plot.getName());
            }
            return false;
        }
        Integer depth = plot.getSolMaxDepth();
        if (depth == null) {
            if (plot.getSolDepth() == null) {
                if (log.isWarnEnabled()) {
                    log.warn("No default depth found for plot " + plot.getName());
                }
                return false;
            }
            depth = plot.getSolDepth().getProfondeur_par_defaut();
        }

        //Recherche valeur OrgC en fonciton taux M0
        //<1,5%   0,87
        //1,5%-5% 2,03
        //>5% 3,20
        double orgC;
        if (tauxMo > 5.0) {
            orgC = 3.2;
        } else if (tauxMo >= 1.5) {
            orgC = 2.03;
        } else {
            orgC = 0.87;
        }



        // 2. Recherche par traitement
        // Rechercher Produit, dose produit, couverture sol, % surface traitée, incorporation, Potruis, TpsJusquaRuis
        // Table   Traitement phyto
        step.rcesurds = new MultiKeyMap<Object, Double>();
        step.rcesos = new MultiKeyMap<Object, Double>();
        for (PhytoProductInput input : inputs) {
            RefActaTraitementsProduit produit = input.getPhytoProduct();
            if (produit == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find phyto product for input " + input.getProductName());
                }
                continue;
            }

            double couvertureSol = step.couvertureSol;
            // Incorporation dans AgroSyst : considérer systématiquement que incorporation = non
            boolean incorporation = false;
            //Dans Agrosyst : valeur non modifiable par l'utilisateur
            double tpsJusquaRuis = 3;


            //Rechercher nom substanceActive du produit commercial appliqué et concentration substance active
            //Table   DétailProduitPhyto
            String idProduit = produit.getId_produit();
            List<RefActaSubstanceActive> substanceActives = actaSubstanceActiveTopiaDao.forId_produitEquals(idProduit).findAll();

            // 3. Recherche par substance active
            // Pour chaque sustance active
            for (RefActaSubstanceActive substanceActive : substanceActives) {
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active ACTA : " + substanceActive.getNom_commun_sa());
                }
                RefSaActaIphy saActaIphy = refSaActaIphyTopiaDao.forNaturalId(substanceActive.getNom_commun_sa()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active IPhy : " + saActaIphy.getNom_sa_iphy());
                }
                RefPhytoSubstanceActiveIphy substanceActiveIphy = phytoSubstanceActiveIphyTopiaDao.forNom_saEquals(saActaIphy.getNom_sa_iphy()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la phyto substance active iphy " + substanceActiveIphy.getNom_sa());
                }

                // Rechercher  DT50, KOC
                // Table   matière active
                double dt50 = substanceActiveIphy.getDt50();
                double koc = substanceActiveIphy.getKoc();

                // Calculer dose substance active (Dose produit * concentration substance active)
                double doseSA = 0;// TODO DCossé 25/07/14 whet is default value in seedingProductInput ? 
                // Si traitement de semence pas de doseSa
                if (!(input instanceof SeedingProductInputImpl)) {
                    doseSA = input.getQtAvg() * substanceActive.getConcentration_valeur();
                }

                // Calcul disponibilité (Dispo) Dispo=exp(-Log(2)*TpsJusquaRuis/DT50)
                double dispo = Math.exp( -Math.log(2) * tpsJusquaRuis / dt50);
                // Si produit  = incorporé alors Dipo = Dispo/2
                if (incorporation) {
                    dispo = dispo / 2;
                }


                // 4-rcesu_rd. Calcul par substance active
                // Calcul des valeurs d'appartenance pour PotRuis et Dispo
                // Recherche des valeurs pt01, pt11, pt12, pt02 pour PotRuis et Dispo "noté X de manière générique
                //Variables   FussySet    pt01    pt11    pt12    pt02
                //PotRuis      D          0       1       1       1
                //PotRuis      F          0       0       0       1
                //Dispo        D          0       1       1       1
                //Dispo        F          0       0       0       1
                double potRuis_D_pt01 = 0, potRuis_D_pt11 = 1/*, potRuis_D_pt12 = 1, potRuis_D_pt02 = 1*/;
                double /*potRuis_F_pt01 = 0, potRuis_F_pt11 = 0,*/potRuis_F_pt12 = 0, potRuis_F_pt02 = 1;
                double dispo_D_pt01 = 0, dispo_D_pt11 = 1/*, dispo_D_pt12 = 1, dispo_D_pt02 = 1*/;
                double /*dispo_F_pt01 = 0, dispo_F_pt11 = 0, */dispo_F_pt12 = 0, dispo_F_pt02 = 1;

                // Si X<pt01 alors  MbshipD(X) = 0
                // Si X>pt11 alors  MbshipD(X) = 1
                // MbshipD(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
                // Si X<pt12 alors  MbshipF(X) = 1
                // Si X>pt02 alors  MbshipF(X) = 0
                // MbshipF(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
                double mbshipD_potRuis;
                if (potRuis < potRuis_D_pt01) {
                    mbshipD_potRuis = 0;
                } else if (potRuis > potRuis_D_pt11) {
                    mbshipD_potRuis = 1;
                } else {
                    mbshipD_potRuis = 0.5 + 0.5 * Math.sin(Math.PI * (potRuis - potRuis_D_pt01) / (potRuis_D_pt11 - potRuis_D_pt01) - 0.5);
                }
                double mbshipD_dispo;
                if (dispo < dispo_D_pt01) {
                    mbshipD_dispo = 0;
                } else if (dispo > dispo_D_pt11) {
                    mbshipD_dispo = 1;
                } else {
                    mbshipD_dispo = 0.5 + 0.5 * Math.sin(Math.PI * (dispo - dispo_D_pt01) / (dispo_D_pt11 - dispo_D_pt01) - 0.5);
                }
                double mbshipF_potRuis;
                if (potRuis < potRuis_F_pt12) {
                    mbshipF_potRuis = 1;
                } else if (potRuis > potRuis_F_pt02) {
                    mbshipF_potRuis = 0;
                } else {
                    mbshipF_potRuis = 0.5 + 0.5 * Math.cos(Math.PI * (potRuis - potRuis_F_pt12) / (potRuis_F_pt02 - potRuis_F_pt12));
                }
                double mbshipF_dispo;
                if (dispo < dispo_F_pt12) {
                    mbshipF_dispo = 1;
                } else if (dispo > dispo_F_pt02) {
                    mbshipF_dispo = 0;
                } else {
                    mbshipF_dispo = 0.5 + 0.5 * Math.cos(Math.PI * (dispo - dispo_F_pt12) / (dispo_F_pt02 - dispo_F_pt12));
                }

                //Calcul des poids pour les 4 règles
                //WeightDD=   Minimum(MbshipD(PotRuis) ,MbshipD(Dispo) )
                //WeightDF=   Minimum(MbshipD(PotRuis) ,MbshipF(Dispo) )
                //WeightFD=   Minimum(MbshipF(PotRuis) ,MbshipD(Dispo) )
                //WeightFF=   Minimum(MbshipF(PotRuis) ,MbshipF(Dispo) )
                double weightDD = Math.min(mbshipD_potRuis, mbshipD_dispo);
                double weightDF = Math.min(mbshipD_potRuis, mbshipF_dispo);
                double weightFD = Math.min(mbshipF_potRuis, mbshipD_dispo);
                double weightFF = Math.min(mbshipF_potRuis, mbshipF_dispo);

                //Calcul des conclusions pondérées pour les4 règles
                double zDD = 0.0;
                double zDF = 5.5;
                double zFD = 6.7;
                double zFF = 10.0;
                //WR_DD = WeightDD*z(DD)
                //to
                //WR_FF =     Weight FF *z(FF)
                double wr_DD = weightDD * zDD;
                double wr_DF = weightDF * zDF;
                double wr_FD = weightFD * zFD;
                double wr_FF = weightFF * zFF;

                // Calcul de la valeur de l'indicateur pour 1 kg
                // rcesu_rd1kg=    Somme des 4  conclusions pondérées (WR_DD to WR_FF)/Somme 4 poids (WeightDD to WeightFF)
                double rcesu_rd1kg = (weightDD + weightDF + weightFD + weightFF) / (wr_DD + wr_DF + wr_FD + wr_FF);


                //4-rceso. Calcul par substance active
                //Calcul pour les 8 lignes de la valeur d'appartenance en fonction de la valeur de la variable
                //Table   FussySet-GroundWater
                //fonction
                //Pour FussySet D     EXP(-((MAX(Valeur variable,c)-c)^2)/(2*sigma^2))
                //Pour FussySet F   EXP(-((MIN(Valeur variable,c)-c)^2)/(2*sigma^2))
                RefRcesoFuzzySetGroundWater dOrgc = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "OrgC", "D").findUnique();
                RefRcesoFuzzySetGroundWater fOrgc = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "OrgC", "F").findUnique();
                RefRcesoFuzzySetGroundWater dDepth = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "Depth", "D").findUnique();
                RefRcesoFuzzySetGroundWater fDepth = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "Depth", "F").findUnique();
                RefRcesoFuzzySetGroundWater dKoc = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "Koc", "D").findUnique();
                RefRcesoFuzzySetGroundWater fKoc = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "Koc", "F").findUnique();
                RefRcesoFuzzySetGroundWater dDt50 = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "DT50", "D").findUnique();
                RefRcesoFuzzySetGroundWater fDt50 = rcesoFuzzySetGroundWaterTopiaDao.forNaturalId(caseNumber, "DT50", "F").findUnique();

                double mbshipDOrgC = Math.exp(- Math.pow(Math.max(orgC, dOrgc.getC()) - dOrgc.getC(), 2) / ( 2 * Math.pow(dOrgc.getSigma(), 2)));
                double mbshipDDepth = Math.exp(- Math.pow(Math.max(depth, dDepth.getC()) - dDepth.getC(), 2) / ( 2 * Math.pow(dDepth.getSigma(), 2)));
                double mbshipDKOC = Math.exp(- Math.pow(Math.max(koc, dKoc.getC()) - dKoc.getC(), 2) / ( 2 * Math.pow(dKoc.getSigma(), 2)));
                double mbshipDDT50 = Math.exp(- Math.pow(Math.max(dt50, dDt50.getC()) - dDt50.getC(), 2) / ( 2 * Math.pow(dDt50.getSigma(), 2)));
                double mbshipFOrgC = Math.exp(- Math.pow(Math.min(orgC, fOrgc.getC()) - fOrgc.getC(), 2) / ( 2 * Math.pow(fOrgc.getSigma(), 2)));
                double mbshipFDepth = Math.exp(- Math.pow(Math.min(depth, fDepth.getC()) - fDepth.getC(), 2) / ( 2 * Math.pow(fDepth.getSigma(), 2)));
                double mbshipFKOC = Math.exp(- Math.pow(Math.min(koc, fKoc.getC()) - fKoc.getC(), 2) / ( 2 * Math.pow(fKoc.getSigma(), 2)));
                double mbshipFDT50 = Math.exp(- Math.pow(Math.min(dt50, fDt50.getC()) - fDt50.getC(), 2) / ( 2 * Math.pow(fDt50.getSigma(), 2)));


                // Calcul des poids pour les 16 règles
                // WeightDDDD = MshipD (OrgC)* MshipD (Depth) * MshipD (KOC) * MshipD(DT50)
                //to
                // WeightFFFF = MshipF (OrgC)* MshipF (Depth) * MshipF (KOC) * MshipF(DT50)
                double weightDDDD = mbshipDOrgC * mbshipDDepth * mbshipDKOC * mbshipDDT50;
                double weightDDDF = mbshipDOrgC * mbshipDDepth * mbshipDKOC * mbshipFDT50;
                double weightDDFD = mbshipDOrgC * mbshipDDepth * mbshipFKOC * mbshipDDT50;
                double weightDDFF = mbshipDOrgC * mbshipDDepth * mbshipFKOC * mbshipFDT50;
                double weightDFDD = mbshipDOrgC * mbshipFDepth * mbshipDKOC * mbshipDDT50;
                double weightDFDF = mbshipDOrgC * mbshipFDepth * mbshipDKOC * mbshipFDT50;
                double weightDFFD = mbshipDOrgC * mbshipFDepth * mbshipFKOC * mbshipDDT50;
                double weightDFFF = mbshipDOrgC * mbshipFDepth * mbshipFKOC * mbshipFDT50;
                double weightFDDD = mbshipFOrgC * mbshipDDepth * mbshipDKOC * mbshipDDT50;
                double weightFDDF = mbshipFOrgC * mbshipDDepth * mbshipDKOC * mbshipFDT50;
                double weightFDFD = mbshipFOrgC * mbshipDDepth * mbshipFKOC * mbshipDDT50;
                double weightFDFF = mbshipFOrgC * mbshipDDepth * mbshipFKOC * mbshipFDT50;
                double weightFFDD = mbshipFOrgC * mbshipFDepth * mbshipDKOC * mbshipDDT50;
                double weightFFDF = mbshipFOrgC * mbshipFDepth * mbshipDKOC * mbshipFDT50;
                double weightFFFD = mbshipFOrgC * mbshipFDepth * mbshipFKOC * mbshipDDT50;
                double weightFFFF = mbshipFOrgC * mbshipFDepth * mbshipFKOC * mbshipFDT50;

                // Calcul des conclusions pondérées pour les 16 règles
                // WR_DDDD =   WeightDDDD*z(DDDD)
                //to
                //WR_FFFF =   Weight FFFF *z(FFFF)
                double zDDDD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "D", "D", "D").findUnique().getZ();
                double zDDDF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "D", "D", "F").findUnique().getZ();
                double zDDFD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "D", "F", "D").findUnique().getZ();
                double zDDFF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "D", "F", "F").findUnique().getZ();
                double zDFDD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "F", "D", "D").findUnique().getZ();
                double zDFDF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "F", "D", "F").findUnique().getZ();
                double zDFFD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "F", "F", "D").findUnique().getZ();
                double zDFFF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "D", "F", "F", "F").findUnique().getZ();
                double zFDDD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "D", "D", "D").findUnique().getZ();
                double zFDDF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "D", "D", "F").findUnique().getZ();
                double zFDFD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "D", "F", "D").findUnique().getZ();
                double zFDFF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "D", "F", "F").findUnique().getZ();
                double zFFDD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "F", "D", "D").findUnique().getZ();
                double zFFDF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "F", "D", "F").findUnique().getZ();
                double zFFFD = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "F", "F", "D").findUnique().getZ();
                double zFFFF = rcesoRulesGroundWaterTopiaDao.forNaturalId(caseNumber, "F", "F", "F", "F").findUnique().getZ();

                double wrDDDD = weightDDDD * zDDDD;
                double wrDDDF = weightDDDF * zDDDF;
                double wrDDFD = weightDDFD * zDDFD;
                double wrDDFF = weightDDFF * zDDFF;
                double wrDFDD = weightDFDD * zDFDD;
                double wrDFDF = weightDFDF * zDFDF;
                double wrDFFD = weightDFFD * zDFFD;
                double wrDFFF = weightDFFF * zDFFF;
                double wrFDDD = weightFDDD * zFDDD;
                double wrFDDF = weightFDDF * zFDDF;
                double wrFDFD = weightFDFD * zFDFD;
                double wrFDFF = weightFDFF * zFDFF;
                double wrFFDD = weightFFDD * zFFDD;
                double wrFFDF = weightFFDF * zFFDF;
                double wrFFFD = weightFFFD * zFFFD;
                double wrFFFF = weightFFFF * zFFFF;

                // Calcul de la valeur de l'indicateur pour 1 kg
                // rcesokg = Somme des 16 conclusions pondérées (WR_DDDD to WR_FFFF)/Somme 16 poids (WeightDDDD to WeightFFFF)
                double rcesokg = (weightDDDD + weightDDDF + weightDDFD + weightDDFF +
                        weightDFDD + weightDFDF + weightDFFD + weightDFFF +
                        weightFDDD + weightFDDF + weightFDFD + weightFDFF +
                        weightFFDD + weightFFDF + weightFFFD + weightFFFF) /
                        (wrDDDD + wrDDDF + wrDDFD + wrDDFF +
                                wrDFDD + wrDFDF + wrDFFD + wrDFFF +
                                wrFDDD + wrFDDF + wrFDFD + wrFDFF +
                                wrFFDD + wrFFDF + wrFFFD + wrFFFF);




                // Cas particulier des parcelles drainées (rceso AND rcesu_rd)
                // Si profondeur = drainé alors
                // si rcesu_rd > rceso; rcesu_rd=rceso
                // receso=10
                if (plot.isDrainage()) {
                    if (rcesu_rd1kg > rcesokg) {
                        rcesu_rd1kg = rcesokg;
                        rcesokg = 10.0;
                    }
                }



                // 5-rcesu_rd. Calcul de la dose réelle substance active
                // DoseSA=(DoseSA*(1-couveture sol/100))/1000
                doseSA = (doseSA * (1 - couvertureSol )) / 1000;

                // Calcul de l'indicateur réel
                // Voir formule du document Excel
                double rcesu_rd = 0;
                if (doseSA > 1.0) {
                    if (rcesu_rd1kg == 0.0) {
                        rcesu_rd = 0;
                    } else if (doseSA <= 10) {
                        rcesu_rd = Math.max(rcesu_rd1kg - 2 * (doseSA - 1) / 9, 0);
                    } else if (doseSA <= 100){
                        rcesu_rd = Math.max(rcesu_rd1kg - 2 - 2 * (doseSA / 10 - 1) / 9, 0);
                    }
                } else {
                    if (doseSA >= 0.1) {
                        rcesu_rd = Math.min(rcesu_rd1kg + 2 - 2 * (doseSA * 10 - 1) / 9, 10);
                    } else if (doseSA >= 0.01) {
                        rcesu_rd = Math.min(rcesu_rd1kg + 4 - 2 * (doseSA * 100 - 1) / 9, 10);
                    } else if (doseSA >= 0.001) {
                        rcesu_rd = Math.min(rcesu_rd1kg + 6 - 2 * (doseSA * 1000 - 1) / 9, 10);
                    }
                }

                // compute pcsi for input
                double psci;
                if (effectiveIntervention != null) {
                    psci = getInputPSCi(effectiveIntervention, input);
                } else {
                    psci = getInputPSCi(practicedIntervention, input);
                }

                // Pondération par % surface traitée
                // rcesu_rd=10*(1-% surface traitée/100)+rcesu_rd*% surface traitée/100
                rcesu_rd = 10 * (1 - psci) + rcesu_rd * psci;

                //Stokage de  rcesu_rd    MbshipF(PotRuis)    MbshipF(Dispo)
                step.rcesurds.put(produit, substanceActiveIphy, rcesu_rd);





                // 5-rceso. Calcul de la dose réelle substance active
                // DoseSA=(DoseSA*(1-couveture sol/100))/1000
                doseSA = (doseSA * (1 - couvertureSol)) / 1000;

                // Calcul de l'indicateur réel
                // Voir formule du document Excel
                double rceso = 0;
                if (doseSA > 1.0) {
                    if (rcesokg == 0.0) {
                        rceso = 0;
                    } else if (doseSA <= 10) {
                        rceso = Math.max(rcesokg - 2 * (doseSA - 1) / 9, 0);
                    } else if (doseSA <= 100){
                        rceso = Math.max(rcesokg - 2 - 2 * (doseSA / 10 - 1) / 9, 0);
                    }
                } else {
                    if (doseSA >= 0.1) {
                        rceso = Math.min(rcesokg + 2 - 2 * (doseSA * 10 - 1) / 9, 10);
                    } else if (doseSA >= 0.01) {
                        rceso = Math.min(rcesokg + 4 - 2 * (doseSA * 100 - 1) / 9, 10);
                    } else if (doseSA >= 0.001) {
                        rceso = Math.min(rcesokg + 6 - 2 * (doseSA * 1000 - 1) / 9, 10);
                    }
                }

                //Pondération par % surface traitée
                //rceso=10*(1-% surface traitée/100)+rceso*% surface traitée/100
                //=resultat final par substance active
                rceso = 10 * (1 - psci / 100) + rceso * psci / 100;

                // sauvegarde dans la map par substance active
                step.rcesos.put(produit, substanceActiveIphy, rceso);
            }
        }

        // 6-rcesu_rd. Calcul pour l'ensemble de la zone phyto (programme de traitement)
        // Rechercher les résultats rcesu_rdi par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double rcesu_rd = scaleToTreatment(step.rcesurds.values());
        if (log.isDebugEnabled()) {
            log.debug("rcesu_rd = " + rcesu_rd);
        }
        step.rcesu_rd = rcesu_rd;



        // 6-rceso. Calcul pour l'ensemble de la zone phyto (programme de traitement)
        // Rechercher les résultats rcesoi par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double rceso = scaleToTreatment(step.rcesos.values());
        if (log.isDebugEnabled()) {
            log.debug("rceso = " + rceso);
        }
        step.rceso = rceso;

        return true;
    }

    /**
     * rcesu_de.
     */
    protected boolean computeRcesude(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        Preconditions.checkArgument(effectiveIntervention == null ^ practicedIntervention == null);

        //1. Calcul du potentiel de dérive
        //Rechercher la distance à la rivière, largeur de bande enherbée
        WaterFlowDistance distanceRiviere = plot.getWaterFlowDistance();
        BufferStrip bandeEnherbee = plot.getBufferStrip();

        //Table Parcels nom dans le logiciel INDIGO voir quelle table dans le programme
        //A partir de ces données rechercher le potentiel de dérive (Drift potential)
        //Table Drift_Pot_Rules
        double driftPotential = 0.0001;
        if (distanceRiviere == WaterFlowDistance.LESS_THAN_THREE) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.0209229411025554;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        } else if (distanceRiviere == WaterFlowDistance.THREE_TO_FIVE) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.000263232689183211;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        } else if (distanceRiviere == WaterFlowDistance.FIVE_TO_TEN) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.000160096648750328;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        }

        // Calcul de la variable Pot_Der
        // PotDer=100* DriftPotential
        double potDer1 = 100 * driftPotential;

        // 2. Recherche par traitement 
        // Rechercher Produit, dose produit, couverture sol, % surface traitée, incorporation, PotDer, BuseAntiDérive  
        // Table   Traitement phyto
        step.rcesudes = new MultiKeyMap<Object, Double>();
        for (PhytoProductInput input : inputs) {
            RefActaTraitementsProduit produit = input.getPhytoProduct();
            if (produit == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find phyto product for input " + input.getProductName());
                }
                continue;
            }
            double couvertureSol = step.couvertureSol;

            // "Présence de buse anti-dérive sur le pulvérisateur" au niveau
            // de l'action de type "Application de produits phytosanitaires" ?
            boolean buseAntiDerive = false;
            if (input instanceof PesticideProductInput) {
                buseAntiDerive = ((PesticideProductInput)input).getPesticidesSpreadingAction().isAntiDriftNozzle();
            }

            //Rechercher nom substanceActive du produit commercial appliqué et concentration substance active 
            //Table   DétailProduitPhyto
            String idProduit = produit.getId_produit();
            List<RefActaSubstanceActive> substanceActives = actaSubstanceActiveTopiaDao.forId_produitEquals(idProduit).findAll();
            
            // 3. Recherche par substance active   
            // Pour chaque sustance active 
            for (RefActaSubstanceActive substanceActive : substanceActives) {
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active ACTA : " + substanceActive.getNom_commun_sa());
                }
                RefSaActaIphy saActaIphy = refSaActaIphyTopiaDao.forNaturalId(substanceActive.getNom_commun_sa()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active IPhy : " + saActaIphy.getNom_sa_iphy());
                }
                RefPhytoSubstanceActiveIphy substanceActiveIphy = phytoSubstanceActiveIphyTopiaDao.forNom_saEquals(saActaIphy.getNom_sa_iphy()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la phyto substance active iphy " + substanceActiveIphy.getNom_sa());
                }

                // Calculer dose substance active (Dose produit * concentration substance active)
                double doseSA = 0;// TODO DCossé 25/07/14 whet is default value in seedingProductInput ?
                // Si traitement de semence pas de doseSa
                if (!(input instanceof SeedingProductInputImpl)) {
                    doseSA = input.getQtAvg() * substanceActive.getConcentration_valeur();
                }


                // 4. Calcul par substance active
                // Correction PotDer
                // A partir de l'utilisation ou non de Buse antidérive, rechercher le facteur correcteur potentiel de dérive (Drift_Correction)
                // Table Drift_Pot_Antideriv
                double driftCorrection = buseAntiDerive ? 0.01 : 1;
                // PotDer = PotDer * Drift_Correction
                double potDer = potDer1 * driftCorrection;

                
                //Calcul des valeurs d'appartenance pour PotDer   
                //Recherche des valeurs pt01, pt11, pt12, pt02 pour PotDer "noté X de manière générique
                //Variables   FussySet    pt01    pt11    pt12    pt02
                //Potder      D           0       1       1       1
                //Potder      F           0       0       0       1
                //double potDer_D_pt01 = 0, potDer_D_pt11 = 1, potDer_D_pt12 = 1, potDer_D_pt02 = 1;
                double /*potDer_F_pt01 = 0, potDer_F_pt11 = 0, */potDer_F_pt12 = 0, potDer_F_pt02 = 1;

                // Si X<pt01 alors  MbshipD(X) = 0
                // Si X>pt11 alors  MbshipD(X) = 1
                // MbshipD(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
                // Si X<pt12 alors  MbshipF(X) = 1
                // Si X>pt02 alors  MbshipF(X) = 0
                // MbshipF(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
                /*double mbshipD_potDer;
                if (potDer < potDer_D_pt01) {
                    mbshipD_potDer = 0;
                } else if (potDer > potDer_D_pt11) {
                    mbshipD_potDer = 1;
                } else {
                    mbshipD_potDer = 0.5 + 0.5 * Math.sin(Math.PI * (potDer - potDer_D_pt01) / (potDer_D_pt11 - potDer_D_pt01) - 0.5);
                }*/
                double mbshipF_potDer;
                if (potDer < potDer_F_pt12) {
                    mbshipF_potDer = 1;
                } else if (potDer > potDer_F_pt02) {
                    mbshipF_potDer = 0;
                } else {
                    mbshipF_potDer = 0.5 + 0.5 * Math.cos(Math.PI * (potDer - potDer_F_pt12) / (potDer_F_pt02 - potDer_F_pt12));
                }

                // Calcul de la valeur de l'indicateur pour 1 kg   
                // rcesu_de=   MbshipF(PotDer)  * 10
                double rcesu_de1kg = mbshipF_potDer * 10;
                
                // 5. Calcul de la dose réelle substance active
                // DoseSA=(DoseSA*(1-couveture sol/100))/1000
                doseSA = (doseSA * (1 - couvertureSol)) / 1000;

                // Calcul de l'indicateur réel
                double rcesu_de = 0;
                if (doseSA > 1.0) {
                    if (rcesu_de1kg == 0.0) {
                        rcesu_de = 0;
                    } else if (doseSA <= 10) {
                        rcesu_de = Math.max(rcesu_de1kg - 2 * (doseSA - 1) / 9, 0);
                    } else if (doseSA <= 100){
                        rcesu_de = Math.max(rcesu_de1kg - 2 - 2 * (doseSA / 10 - 1) / 9, 0);
                    }
                } else {
                    if (doseSA >= 0.1) {
                        rcesu_de = Math.min(rcesu_de1kg + 2 - 2 * (doseSA * 10 - 1) / 9, 10);
                    } else if (doseSA >= 0.01) {
                        rcesu_de = Math.min(rcesu_de1kg + 4 - 2 * (doseSA * 100 - 1) / 9, 10);
                    } else if (doseSA >= 0.001) {
                        rcesu_de = Math.min(rcesu_de1kg + 6 - 2 * (doseSA * 1000 - 1) / 9, 10);
                    }
                }

                // compute pcsi for input
                double psci;
                if (effectiveIntervention != null) {
                    psci = getInputPSCi(effectiveIntervention, input);
                } else {
                    psci = getInputPSCi(practicedIntervention, input);
                }

                // Pondération par % surface traitée
                // rcesu_rd=10*(1-% surface traitée/100)+rcesu_rd*% surface traitée/100
                rcesu_de = 10 * (1 - psci) + rcesu_de * psci;
                
                // Stokage de  rcesu_rd    MbshipF(PotRuis)    MbshipF(Dispo) 
                step.rcesudes.put(produit, substanceActiveIphy, rcesu_de);
            }
        }

        // 6. Calcul pour l'ensemble de la zone phyto (programme de traitement)
        // Rechercher les résultats rcesu_dei par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double rcesu_de = scaleToTreatment(step.rcesudes.values());
        if (log.isDebugEnabled()) {
            log.debug("rcesu_de = " + rcesu_de);
        }
        step.rcesu_de = rcesu_de;
        
        return true;
    }

    /**
     * rcair_vo.
     */
    protected boolean computeRcairvo(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        // 1. Recherche par traitement 
        // Rechercher Produit, dose produit, couverture sol, % surface traitée
        // Table   Traitement phyto
        step.rcairvos = new MultiKeyMap<Object, Double>();
        for (PhytoProductInput input : inputs) {
            RefActaTraitementsProduit produit = input.getPhytoProduct();
            if (produit == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find phyto product for input " + input.getProductName());
                }
                continue;
            }
            double couvertureSol = step.couvertureSol;

            //Rechercher nom substanceActive du produit commercial appliqué et concentration substance active 
            //Table   DétailProduitPhyto
            String idProduit = produit.getId_produit();
            List<RefActaSubstanceActive> substanceActives = actaSubstanceActiveTopiaDao.forId_produitEquals(idProduit).findAll();


            //3. Recherche par substance active
            //Pour chaque sustance active
            for (RefActaSubstanceActive substanceActive : substanceActives) {
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active ACTA : " + substanceActive.getNom_commun_sa());
                }
                RefSaActaIphy saActaIphy = refSaActaIphyTopiaDao.forNaturalId(substanceActive.getNom_commun_sa()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active IPhy : " + saActaIphy.getNom_sa_iphy());
                }
                RefPhytoSubstanceActiveIphy substanceActiveIphy = phytoSubstanceActiveIphyTopiaDao.forNom_saEquals(saActaIphy.getNom_sa_iphy()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la phyto substance active iphy " + substanceActiveIphy.getNom_sa());
                }

                //Rechercher  DT50, VP, Sol
                double koc = substanceActiveIphy.getKoc();
                double vp = substanceActiveIphy.getPressionVapeur();
                double sol = substanceActiveIphy.getSolubilite();

                //Calculer dose substance active (Dose produit * concentration substance active)
                double doseSA = 0;// TODO DCossé 25/07/14 whet is default value in seedingProductInput ?
                // Si traitement de semence pas de doseSa
                if (!(input instanceof SeedingProductInputImpl)) {
                    doseSA = input.getQtAvg() * substanceActive.getConcentration_valeur();
                }

                // 3. Calcul par substance active
                // Flux volatilisation plante      VolSol=e^(28,335+Ln(VP/(KOC*Sol))*1,6158)*(CouvSol/100)
                // Flux volatilisation sol     VolPlant=e^(11,779+Ln(VP)*0,85543)*(1-CouvSol/100)
                // Flux total      Vol=VolsSol+VolPlant
                double volatilisationSol = Math.pow(Math.E, (28.335 + Math.log(vp / (koc * sol)) * 1.6158)) * (couvertureSol);
                double volatilisationPlante = Math.pow(Math.E, (11.779 + Math.log(vp) * 0.85543)) * (1.0 - couvertureSol);
                double volatilisation = volatilisationPlante + volatilisationSol;
                
                // Transformation en un score entre 0 et 10 pour dose 1 kg
                //Recherche des limites entre 
                //pt01    pt11
                //0       200500
                double pt01 = 0;
                double pt11 = 200500;

                //Si X<pt01 alors  rcair_vol1kg=10
                //Si X>pt11 alors  rcair_vol1kg=0
                //rcair_vol = 0.5 + 0.5 * Sin(3.14159267 * ((Vol - pt01) / (pt11 - pt01) - 0.5))
                double rcair_vol1kg;
                if (volatilisation < pt01) {
                    rcair_vol1kg = 10;
                } else if (volatilisation > pt11) {
                    rcair_vol1kg = 0;
                } else {
                    rcair_vol1kg = 0.5 + 0.5 * Math.sin(Math.PI * (volatilisation - pt01) / (pt11 - pt01) - 0.5);
                }

                //4. Calcul de la dose réelle substance active
                //DoseSA=(DoseSA*(1-couveture sol/100))/1000
                doseSA = (doseSA * (1 - couvertureSol)) / 1000;

                // Calcul de l'indicateur réel
                double rcair_vol = 0;
                if (doseSA > 1.0) {
                    if (rcair_vol1kg == 0.0) {
                        rcair_vol = 0;
                    } else if (doseSA <= 10) {
                        rcair_vol = Math.max(rcair_vol1kg - 2 * (doseSA - 1) / 9, 0);
                    } else if (doseSA <= 100){
                        rcair_vol = Math.max(rcair_vol1kg - 2 - 2 * (doseSA / 10 - 1) / 9, 0);
                    }
                } else {
                    if (doseSA >= 0.1) {
                        rcair_vol = Math.min(rcair_vol1kg + 2 - 2 * (doseSA * 10 - 1) / 9, 10);
                    } else if (doseSA >= 0.01) {
                        rcair_vol = Math.min(rcair_vol1kg + 4 - 2 * (doseSA * 100 - 1) / 9, 10);
                    } else if (doseSA >= 0.001) {
                        rcair_vol = Math.min(rcair_vol1kg + 6 - 2 * (doseSA * 1000 - 1) / 9, 10);
                    }
                }

                // compute pcsi for input
                double psci;
                if (effectiveIntervention != null) {
                    psci = getInputPSCi(effectiveIntervention, input);
                } else {
                    psci = getInputPSCi(practicedIntervention, input);
                }

                // Pondération par % surface traitée
                // rcair_vol=10*(1-% surface traitée,/100)+rcair_vol*% surface traitée,/100
                rcair_vol = 10 * (1 - psci) + rcair_vol * psci;

                // Stokage de  rcair_vol
                step.rcairvos.put(produit, substanceActiveIphy, rcair_vol);
            }
        }

        // 6. Calcul pour le programme de traitement
        // Rechercher les résultats rcair_voli par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double rcair_vo = scaleToTreatment(step.rcairvos.values());
        if (log.isDebugEnabled()) {
            log.debug("rcair_vo = " + rcair_vo);
        }
        step.rcair_vo = rcair_vo;
        
        return true;
    }

    /**
     * rcair_de.
     */
    protected boolean computeRcairde(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        Preconditions.checkArgument(effectiveIntervention == null ^ practicedIntervention == null);

        //1 (bis). Calcul du potentiel de dérive
        //Rechercher la distance à la rivière, largeur de bande enherbée
        WaterFlowDistance distanceRiviere = plot.getWaterFlowDistance();
        BufferStrip bandeEnherbee = plot.getBufferStrip();

        //Table Parcels nom dans le logiciel INDIGO voir quelle table dans le programme
        //A partir de ces données rechercher le potentiel de dérive (Drift potential)
        //Table Drift_Pot_Rules
        double driftPotential = 0.0001;
        if (distanceRiviere == WaterFlowDistance.LESS_THAN_THREE) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.0209229411025554;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        } else if (distanceRiviere == WaterFlowDistance.THREE_TO_FIVE) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.000263232689183211;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        } else if (distanceRiviere == WaterFlowDistance.FIVE_TO_TEN) {
            if (bandeEnherbee == BufferStrip.NONE) {
                driftPotential = 0.000160096648750328;
            } else if (bandeEnherbee == BufferStrip.BUFFER_STRIP_NEXT_TO_WATER_FLOW) {
                driftPotential = 0.000160096648750328;
            }
        }

        // Calcul de la variable Pot_Der
        // PotDer=100* DriftPotential
        double potDer1 = 100 * driftPotential;

        // 1. Recherche par traitement 
        // Rechercher Produit, dose produit, couverture sol, % surface traitée, incorporation, PotDer, BuseAntiDérive
        // Table   Traitement phyto
        step.rcairdes = new MultiKeyMap<Object, Double>();
        for (PhytoProductInput input : inputs) {
            RefActaTraitementsProduit produit = input.getPhytoProduct();
            if (produit == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find phyto product for input " + input.getProductName());
                }
                continue;
            }
            double couvertureSol = step.couvertureSol;

            // "Présence de buse anti-dérive sur le pulvérisateur" au niveau
            // de l'action de type "Application de produits phytosanitaires" ?
            boolean buseAntiDerive = false;
            if (input instanceof PesticideProductInput) {
                buseAntiDerive = ((PesticideProductInput)input).getPesticidesSpreadingAction().isAntiDriftNozzle();
            }

            //Rechercher nom substanceActive du produit commercial appliqué et concentration substance active 
            //Table   DétailProduitPhyto
            String idProduit = produit.getId_produit();
            List<RefActaSubstanceActive> substanceActives = actaSubstanceActiveTopiaDao.forId_produitEquals(idProduit).findAll();

            //2. Recherche par substance active
            //Pour chaque sustance active
            for (RefActaSubstanceActive substanceActive : substanceActives) {
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active ACTA : " + substanceActive.getNom_commun_sa());
                }
                RefSaActaIphy saActaIphy = refSaActaIphyTopiaDao.forNaturalId(substanceActive.getNom_commun_sa()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la substance active IPhy : " + saActaIphy.getNom_sa_iphy());
                }
                RefPhytoSubstanceActiveIphy substanceActiveIphy = phytoSubstanceActiveIphyTopiaDao.forNom_saEquals(saActaIphy.getNom_sa_iphy()).findUnique();
                if (log.isDebugEnabled()) {
                    log.debug("Gestion de la phyto substance active iphy " + substanceActiveIphy.getNom_sa());
                }

                //Calculer dose substance active (Dose produit * concentration substance active)
                double doseSA = 0;// TODO DCossé 25/07/14 whet is default value in seedingProductInput ?
                // Si traitement de semence pas de doseSa
                if (!(input instanceof SeedingProductInputImpl)) {
                    doseSA = input.getQtAvg() * substanceActive.getConcentration_valeur();
                }
                
                // 3. Calcul par substance active
                // Correction PotDer
                // A partir de l'utilisation ou non de Buse antidérive, rechercher le facteur correcteur potentiel de dérive (Drift_Correction)
                // Table DriftAir_Pot_Rules
                double driftCorrection = buseAntiDerive ? 0.01 : 0.1;
                // PotDer = PotDer * Drift_Correction
                double potDer = potDer1 * driftCorrection;

                //Calcul des valeurs d'appartenance pour PotDer
                //Recherche des valeurs pt01, pt11, pt12, pt02 pour PotDer "noté X de manière générique   
                //Variables   FussySet    pt01    pt11    pt12    pt02
                //Potder      D           0       10      10      10
                //Potder      F           0       0       0       10
                /*double potDer_D_pt01 = 0, potDer_D_pt11 = 10, potDer_D_pt12 = 10, potDer_D_pt02 = 10;*/
                double /*potDer_F_pt01 = 0, potDer_F_pt11 = 0, */potDer_F_pt12 = 0, potDer_F_pt02 = 10;

                // Si X<pt01 alors  MbshipD(X) = 0
                // Si X>pt11 alors  MbshipD(X) = 1
                // MbshipD(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
                // Si X<pt12 alors  MbshipF(X) = 1
                // Si X>pt02 alors  MbshipF(X) = 0
                // MbshipF(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
                /*double mbshipD_potDer;
                if (potDer < potDer_D_pt01) {
                    mbshipD_potDer = 0;
                } else if (potDer > potDer_D_pt11) {
                    mbshipD_potDer = 1;
                } else {
                    mbshipD_potDer = 0.5 + 0.5 * Math.sin(Math.PI * (potDer - potDer_D_pt01) / (potDer_D_pt11 - potDer_D_pt01) - 0.5);
                }*/
                double mbshipF_potDer;
                if (potDer < potDer_F_pt12) {
                    mbshipF_potDer = 1;
                } else if (potDer > potDer_F_pt02) {
                    mbshipF_potDer = 0;
                } else {
                    mbshipF_potDer = 0.5 + 0.5 * Math.cos(Math.PI * (potDer - potDer_F_pt12) / (potDer_F_pt02 - potDer_F_pt12));
                }

                // Calcul de la valeur de l'indicateur pour 1 kg   
                // rcesu_de = MbshipF(PotDer) * 10
                double rcair_de1kg = mbshipF_potDer * 10;

                // 5. Calcul de la dose réelle substance active
                // DoseSA=(DoseSA*(1-couveture sol/100))/1000
                doseSA = (doseSA * (1 - couvertureSol)) / 1000;

                // Calcul de l'indicateur réel
                double rcair_de = 0;
                if (doseSA > 1.0) {
                    if (rcair_de1kg == 0.0) {
                        rcair_de = 0;
                    } else if (doseSA <= 10) {
                        rcair_de = Math.max(rcair_de1kg - 2 * (doseSA - 1) / 9, 0);
                    } else if (doseSA <= 100){
                        rcair_de = Math.max(rcair_de1kg - 2 - 2 * (doseSA / 10 - 1) / 9, 0);
                    }
                } else {
                    if (doseSA >= 0.1) {
                        rcair_de = Math.min(rcair_de1kg + 2 - 2 * (doseSA * 10 - 1) / 9, 10);
                    } else if (doseSA >= 0.01) {
                        rcair_de = Math.min(rcair_de1kg + 4 - 2 * (doseSA * 100 - 1) / 9, 10);
                    } else if (doseSA >= 0.001) {
                        rcair_de = Math.min(rcair_de1kg + 6 - 2 * (doseSA * 1000 - 1) / 9, 10);
                    }
                }

                // compute pcsi for input
                double psci;
                if (effectiveIntervention != null) {
                    psci = getInputPSCi(effectiveIntervention, input);
                } else {
                    psci = getInputPSCi(practicedIntervention, input);
                }

                // Pondération par % surface traitée
                // rcesu_de=10*(1-% surface traitée/100)+rcesu_de*% surface traitée/100
                rcair_de = 10 * (1 - psci) + rcair_de * psci;

                // Stokage de rcesu_air
                step.rcairdes.put(produit, substanceActiveIphy, rcair_de);
            }
        }
        
        // 6. Calcul pour le programme de traitement
        // Rechercher les résultats rcair_dei par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double rcair_de = scaleToTreatment(step.rcairdes.values());
        if (log.isDebugEnabled()) {
            log.debug("rcair_de = " + rcair_de);
        }
        step.rcair_de = rcair_de;
        
        return true;
    }

    /**
     * I-Phyeso.
     */
    protected boolean computeIPhyeso(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        // I-Phyeso
        // 1. Recherche par substance active   
        // Pour chaque sustance active 
        // Rechercher  DJA
        step.iphyesos = new MultiKeyMap<Object, Double>();
        for (Map.Entry<MultiKey<? extends Object>, Double> entry : step.rcesos.entrySet()) {
            RefActaTraitementsProduit produit = (RefActaTraitementsProduit)entry.getKey().getKey(0);
            RefPhytoSubstanceActiveIphy substanceActiveIphy = (RefPhytoSubstanceActiveIphy)entry.getKey().getKey(1);
            double rceso = entry.getValue();
            double dja = substanceActiveIphy.getDja();

            //2. Calcul par substance active  
            //Calcul des valeurs d'appartenance pour rceso et Log10(DJA)  

            //Rechercher les valeurs pt01, pt11, pt12, pt02 pour rceso et DJA  "noté X de manière générique   
            //Table   FussySet-I-Phyeso
            //Variables   FussySet    pt01     pt11     pt12     pt02
            //rceso       D           0        0          0       10
            //rceso       F           0        10         0       0
            //Log10(DJA)  D       -1000000000 -1000000000 -4      0
            //Log10(DJA)  F           -4        0    1000000000  1000000000
            double /*rceso_D_pt01 = 0, rceso_D_pt11 = 0, */rceso_D_pt12 = 0, rceso_D_pt02 = 0;
            double rceso_F_pt01 = 0, rceso_F_pt11 = 10/*, rceso_F_pt12 = 0, rceso_F_pt02 = 0*/;
            double /*log10dja_D_pt01 = -1000000000, log10dja_D_pt11 = -1000000000, */log10dja_D_pt12 = -4, log10dja_D_pt02 = 0;
            double log10dja_F_pt01 = -4, log10dja_F_pt11 = 0/*, log10dja_F_pt12 = 1000000000, log10dja_F_pt02 = 1000000000*/;

            //Fonction D  Si X<pt12 alors  MbshipD(X) = 1
            //            Si X>pt02 alors  MbshipD(X) = 0 
            //            MbshipD(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
            //Fonction F  Si X<pt01 alors  MbshipF(X) = 0
            //            Si X>pt11 alors  MbshipF(X) = 1
            //            MbshipF(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))

            double mbshipD_rceso;
            if (rceso < rceso_D_pt12) {
                mbshipD_rceso = 1;
            } else if (rceso > rceso_D_pt02) {
                mbshipD_rceso = 0;
            } else {
                mbshipD_rceso = 0.5 + 0.5 * Math.cos(Math.PI * (rceso - rceso_D_pt12) / (rceso_D_pt02 - rceso_D_pt12));
            }
            double mbshipF_rceso;
            if (rceso < rceso_F_pt01) {
                mbshipF_rceso = 0;
            } else if (rceso > rceso_F_pt11) {
                mbshipF_rceso = 1;
            } else {
                mbshipF_rceso = 0.5 + 0.5 * Math.sin(Math.PI * ((rceso - rceso_F_pt01) / (rceso_F_pt11 - rceso_F_pt01) - 0.5));
            }
            double mbshipD_log10dja;
            if (Math.log10(dja) < log10dja_D_pt12) {
                mbshipD_log10dja = 1;
            } else if (Math.log10(dja) > log10dja_D_pt02) {
                mbshipD_log10dja = 0;
            } else {
                mbshipD_log10dja = 0.5 + 0.5 * Math.cos(Math.PI * (Math.log10(dja) - log10dja_D_pt12) / (log10dja_D_pt02 - log10dja_D_pt12));
            }
            double mbshipF_log10dja;
            if (Math.log10(dja) < log10dja_F_pt01) {
                mbshipF_log10dja = 0;
            } else if (Math.log10(dja) > log10dja_F_pt11) {
                mbshipF_log10dja = 1;
            } else {
                mbshipF_log10dja = 0.5 + 0.5 * Math.sin(Math.PI * ((Math.log10(dja) - log10dja_F_pt01) / (log10dja_F_pt11 - log10dja_F_pt01) - 0.5));
            }

            // Calcul des poids pour les 4 règles  
            //WeightDD=   Minimum(MbshipD(rceso) ,MbshipD(Log10(DJA)) )
            //WeightDF=   Minimum(MbshipD(rceso) ,MbshipF(Log10(DJA)) )
            //WeightFD=   Minimum(MbshipF(rceso) ,MbshipD(Log10(DJA)) )
            //WeightFF=   Minimum(MbshipF(rceso) ,MbshipF(Log10(DJA)) )
            double weightDD = Math.min(mbshipD_rceso, mbshipD_log10dja);
            double weightDF = Math.min(mbshipD_rceso, mbshipF_log10dja);
            double weightFD = Math.min(mbshipF_rceso, mbshipD_log10dja);
            double weightFF = Math.min(mbshipF_rceso, mbshipF_log10dja);

            //Calcul des conclusions pondérées pour les 4 règles  
            //Rechercher les valeurs de conclusion z()    
            //Table   Rules-I-Phyeso
            //rceso   DJA z
            //D       D   0,0
            //D       F   4,0
            //F       D   6,0
            //F       F   10,0
            double zDD = 0.0;
            double zDF = 4.0;
            double zFD = 6.0;
            double zFF = 10.0;

            //WR_DD = WeightDD*z(DD)
            //to  
            //WR_FF =     Weight FF *z(FF)
            double wr_DD = weightDD * zDD;
            double wr_DF = weightDF * zDF;
            double wr_FD = weightFD * zFD;
            double wr_FF = weightFF * zFF;

            //Calcul de la valeur de l'indicateur 
            //I-Phyeso=   Somme des 4  conclusions pondérées (WR_DD to WR_FF)/Somme 4 poids (WeightDD to WeightFF)
            double iPhyeso = (weightDD + weightDF + weightFD + weightFF) / (wr_DD + wr_DF + wr_FD + wr_FF);

            //Stokage de  I-Phyeso    MbshipF(rceso)  MbshipF(DJA) 
            step.iphyesos.put(produit, substanceActiveIphy, iPhyeso);
        }

        //3. Calcul pour le programme de traitement
        //Rechercher les résultats I-phyesoi par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double iphyeso = scaleToTreatment(step.iphyesos.values());
        if (log.isDebugEnabled()) {
            log.debug("iphyeso = " + iphyeso);
        }
        step.iphyeso = iphyeso;
        
        return true;
    }

    /**
     * I-Phyesu.
     */
    protected boolean computeIPhyesu(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        //1. Recherche par substance active
        //Pour chaque sustance active 
        //Rechercher  DJA, Aquatox
        //Table   matière active
        step.iphyesus = new MultiKeyMap<Object, Double>();
        for (Map.Entry<MultiKey<? extends Object>, Double> entry : step.rcesudes.entrySet()) {
            RefActaTraitementsProduit produit = (RefActaTraitementsProduit)entry.getKey().getKey(0);
            RefPhytoSubstanceActiveIphy substanceActiveIphy = (RefPhytoSubstanceActiveIphy)entry.getKey().getKey(1);
            double dja = substanceActiveIphy.getDja();
            double aquatox = substanceActiveIphy.getAquatox();

            // Recherche   rcesu_rd, rcesu_de
            // Table   result_Iphy_SA
            double rceso_de = entry.getValue();
            double rceso_rd = step.rcesurds.get(produit, substanceActiveIphy);

            // 2. Calcul par substance active
            // Calcul de la variable Tox
            // Tox=min(Log10(DJA)+2; Log10(Aquatox))
            double tox = Math.min(Math.log10(dja) + 2, Math.log10(aquatox));

            // Calcul des valeurs d'appartenance pour rceso et Log10(DJA)  
            // Rechercher les valeurs pt01, pt11, pt12, pt02 pour rcesu_rd, rcesu_de et Tox  "noté X de manière générique
            // Variables   FussySet    pt01    pt11    pt12    pt02
            // rcesu_rd    D           0       0       0       10
            // rcesu_rd    F           0       10      0       0
            // rcesu_de    D           0       0       0       10
            // rcesu_de    F           0       10      0       0
            // Tox         D          -2      -2       -2      2
            // Tox         F          -2       2       -2      -2
            double rceso_rd_D_pt12 = 0, rceso_rd_D_pt02 = 10;
            double rceso_rd_F_pt01 = 0, rceso_rd_F_pt11 = 10;
            double rceso_de_D_pt12 = 0, rceso_de_D_pt02 = 10;
            double rceso_de_F_pt01 = 0, rceso_de_F_pt11 = 10;
            double tox_D_pt12 = -2, tox_D_pt02 = 2;
            double tox_F_pt01 = -2, tox_F_pt11 = 2;

            //Fonction D  Si X<pt12 alors  MbshipD(X) = 1
            //            Si X>pt02 alors  MbshipD(X) = 0 
            //            MbshipD(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
            //Fonction F  Si X<pt01 alors  MbshipF(X) = 0
            //            Si X>pt11 alors  MbshipF(X) = 1
            //            MbshipF(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
            double mbshipD_rceso_rd;
            if (rceso_rd < rceso_rd_D_pt12) {
                mbshipD_rceso_rd = 1;
            } else if (rceso_rd > rceso_rd_D_pt02) {
                mbshipD_rceso_rd = 0;
            } else {
                mbshipD_rceso_rd = 0.5 + 0.5 * Math.cos(Math.PI * (rceso_rd - rceso_rd_D_pt12) / (rceso_rd_D_pt02 - rceso_rd_D_pt12));
            }
            double mbshipF_rceso_rd;
            if (rceso_rd < rceso_rd_F_pt01) {
                mbshipF_rceso_rd = 0;
            } else if (rceso_rd > rceso_rd_F_pt11) {
                mbshipF_rceso_rd = 1;
            } else {
                mbshipF_rceso_rd = 0.5 + 0.5 * Math.sin(Math.PI * ((rceso_rd - rceso_rd_F_pt01) / (rceso_rd_F_pt11 - rceso_rd_F_pt01) - 0.5));
            }
            double mbshipD_rceso_de;
            if (rceso_de < rceso_de_D_pt12) {
                mbshipD_rceso_de = 1;
            } else if (rceso_de > rceso_de_D_pt02) {
                mbshipD_rceso_de = 0;
            } else {
                mbshipD_rceso_de = 0.5 + 0.5 * Math.cos(Math.PI * (rceso_de - rceso_de_D_pt12) / (rceso_de_D_pt02 - rceso_de_D_pt12));
            }
            double mbshipF_rceso_de;
            if (rceso_de < rceso_de_F_pt01) {
                mbshipF_rceso_de = 0;
            } else if (rceso_de > rceso_de_F_pt11) {
                mbshipF_rceso_de = 1;
            } else {
                mbshipF_rceso_de = 0.5 + 0.5 * Math.sin(Math.PI * ((rceso_de - rceso_de_F_pt01) / (rceso_de_F_pt11 - rceso_de_F_pt01) - 0.5));
            }
            double mbshipD_tox;
            if (tox < tox_D_pt12) {
                mbshipD_tox = 1;
            } else if (tox > tox_D_pt02) {
                mbshipD_tox = 0;
            } else {
                mbshipD_tox = 0.5 + 0.5 * Math.cos(Math.PI * (tox - tox_D_pt12) / (tox_D_pt02 - tox_D_pt12));
            }
            double mbshipF_tox;
            if (tox < tox_F_pt01) {
                mbshipF_tox = 0;
            } else if (tox > tox_F_pt11) {
                mbshipF_tox = 1;
            } else {
                mbshipF_tox = 0.5 + 0.5 * Math.sin(Math.PI * ((tox - tox_F_pt01) / (tox_F_pt11 - tox_F_pt01) - 0.5));
            }


            // Calcul des poids pour les 8 règles  
            //WeightDDD=  Minimum(MbshipD(rcesu_rd), MbshipD(rcesu_de), MbshipD(Tox))
            //WeightDDF=  Minimum(MbshipD(rcesu_rd), MbshipD(rcesu_de), MbshipF(Tox))
            //WeightDFD=  Minimum(MbshipD(rcesu_rd), MbshipF(rcesu_de), MbshipD(Tox))
            //WeightDFF=  Minimum(MbshipD(rcesu_rd), MbshipF(rcesu_de), MbshipF(Tox))
            //WeightFDD=  Minimum(MbshipF(rcesu_rd), MbshipD(rcesu_de), MbshipD(Tox))
            //WeightFDF=  Minimum(MbshipF(rcesu_rd), MbshipD(rcesu_de), MbshipF(Tox))
            //WeightFFD=  Minimum(MbshipF(rcesu_rd), MbshipF(rcesu_de), MbshipD(Tox))
            //WeightFFF=  Minimum(MbshipF(rcesu_rd), MbshipF(rcesu_de), MbshipF(Tox))
            double weightDDD = NumberUtils.min(mbshipD_rceso_rd, mbshipD_rceso_de, mbshipD_tox);
            double weightDDF = NumberUtils.min(mbshipD_rceso_rd, mbshipD_rceso_de, mbshipF_tox);
            double weightDFD = NumberUtils.min(mbshipD_rceso_rd, mbshipF_rceso_de, mbshipD_tox);
            double weightDFF = NumberUtils.min(mbshipD_rceso_rd, mbshipF_rceso_de, mbshipF_tox);
            double weightFDD = NumberUtils.min(mbshipF_rceso_rd, mbshipD_rceso_de, mbshipD_tox);
            double weightFDF = NumberUtils.min(mbshipF_rceso_rd, mbshipD_rceso_de, mbshipF_tox);
            double weightFFD = NumberUtils.min(mbshipF_rceso_rd, mbshipF_rceso_de, mbshipD_tox);
            double weightFFF = NumberUtils.min(mbshipF_rceso_rd, mbshipF_rceso_de, mbshipF_tox);

            // Calcul des conclusions pondérées pour les 8 règles
            // Rechercher les valeurs de conclusion z()
            // Table
            double zDDD = 0;
            double zDDF = 4;
            double zDFD = 1;
            double zDFF = 9;
            double zFDD = 1;
            double zFDF = 9;
            double zFFD = 6;
            double zFFF = 10;

            // WR_DDD = WeightDDD*z(DDD)
            //to  
            //WR_FFF =    Weight FFF *z(FFF)
            double wr_DDD = weightDDD * zDDD;
            double wr_DDF = weightDDF * zDDF;
            double wr_DFD = weightDFD * zDFD;
            double wr_DFF = weightDFF * zDFF;
            double wr_FDD = weightFDD * zFDD;
            double wr_FDF = weightFDF * zFDF;
            double wr_FFD = weightFFD * zFFD;
            double wr_FFF = weightFFF * zFFF;

            // Calcul de la valeur de l'indicateur 
            // I-Phyesu = Somme des 8 conclusions pondérées (WR_DDD to WR_FFF)/Somme 8 poids (WeightDDD to WeightFFF)
            double iphyesu = (wr_DDD + wr_DDF + wr_DFD + wr_DFF
                    + wr_FDD + wr_FDF + wr_FFD + wr_FFF) /
                    (weightDDD + weightDDF + weightDFD + weightDFF
                    + weightFDD + weightFDF + weightFFD + weightFFF);

            //Stokage de  I-Phyesu 
            step.iphyesus.put(produit, substanceActiveIphy, iphyesu);
        }

        //3. Calcul pour le programme de traitement
        //Rechercher les résultats I-phyesui par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double iphyesu = scaleToTreatment(step.iphyesus.values());
        if (log.isDebugEnabled()) {
            log.debug("iphyesu = " + iphyesu);
        }
        step.iphyesu = iphyesu;

        return true;
    }

    /**
     * I-Phyair.
     */
    protected boolean computeIPhyair(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {
        
        //1. Recherche par substance active    
        //Pour chaque sustance active 
        //Rechercher  DJA
        //Table   matière active
        step.iphyairs = new MultiKeyMap<Object, Double>();
        for (Map.Entry<MultiKey<? extends Object>, Double> entry : step.rcairvos.entrySet()) {
            RefActaTraitementsProduit produit = (RefActaTraitementsProduit)entry.getKey().getKey(0);
            RefPhytoSubstanceActiveIphy substanceActiveIphy = (RefPhytoSubstanceActiveIphy)entry.getKey().getKey(1);
            double dja = substanceActiveIphy.getDja();

            //Recherche   rcair_vol, rcair_de
            //Table   result_Iphy_SA
            double rcair_vo = entry.getValue();
            double rcair_de = step.rcairdes.get(produit, substanceActiveIphy);

            // 2. Calcul par substance active
            //Calcul de la variable Tox
            double tox = Math.log10(dja);
            
            //Calcul des valeurs d'appartenance pour rceso et Log10(DJA)  
            //Variables   FussySet    pt01    pt11    pt12    pt02
            //rcesu_rd    D           0       0       0       10
            //rcesu_rd    F           0       10      0       0
            //rcesu_de    D           0       0       0       10
            //rcesu_de    F           0       10      0       0
            //Tox         D           -4      -4      -4      0
            //Tox         F           -4      0       0       0
            double rcair_vo_D_pt12 = 0,rcair_vo_D_pt02 = 10;
            double rcair_vo_F_pt01 = 0, rcair_vo_F_pt11 = 10;
            double rcair_de_D_pt12 = 0, rcair_de_D_pt02 = 10;
            double rcair_de_F_pt01 = 0, rcair_de_F_pt11 = 10;
            double tox_D_pt12 = -4, tox_D_pt02 = -4;
            double tox_F_pt01 = -4, tox_F_pt11 = 0;

            //Fonction D  Si X<pt12 alors  MbshipD(X) = 1
            //            Si X>pt02 alors  MbshipD(X) = 0 
            //            MbshipD(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
            //Fonction F  Si X<pt01 alors  MbshipF(X) = 0
            //            Si X>pt11 alors  MbshipF(X) = 1
            //            MbshipF(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
            double mbshipD_rcair_vo;
            if (rcair_vo < rcair_vo_D_pt12) {
                mbshipD_rcair_vo = 1;
            } else if (rcair_vo > rcair_vo_D_pt02) {
                mbshipD_rcair_vo = 0;
            } else {
                mbshipD_rcair_vo = 0.5 + 0.5 * Math.cos(Math.PI * (rcair_vo - rcair_vo_D_pt12) / (rcair_vo_D_pt02 - rcair_vo_D_pt12));
            }
            double mbshipF_rcair_vo;
            if (rcair_vo < rcair_vo_F_pt01) {
                mbshipF_rcair_vo = 0;
            } else if (rcair_vo > rcair_vo_F_pt11) {
                mbshipF_rcair_vo = 1;
            } else {
                mbshipF_rcair_vo = 0.5 + 0.5 * Math.sin(Math.PI * ((rcair_vo - rcair_vo_F_pt01) / (rcair_vo_F_pt11 - rcair_vo_F_pt01) - 0.5));
            }
            double mbshipD_rcair_de;
            if (rcair_de < rcair_de_D_pt12) {
                mbshipD_rcair_de = 1;
            } else if (rcair_de > rcair_de_D_pt02) {
                mbshipD_rcair_de = 0;
            } else {
                mbshipD_rcair_de = 0.5 + 0.5 * Math.cos(Math.PI * (rcair_de - rcair_de_D_pt12) / (rcair_de_D_pt02 - rcair_de_D_pt12));
            }
            double mbshipF_rcair_de;
            if (rcair_de < rcair_de_F_pt01) {
                mbshipF_rcair_de = 0;
            } else if (rcair_de > rcair_de_F_pt11) {
                mbshipF_rcair_de = 1;
            } else {
                mbshipF_rcair_de = 0.5 + 0.5 * Math.sin(Math.PI * ((rcair_de - rcair_de_F_pt01) / (rcair_de_F_pt11 - rcair_de_F_pt01) - 0.5));
            }
            double mbshipD_tox;
            if (tox < tox_D_pt12) {
                mbshipD_tox = 1;
            } else if (tox > tox_D_pt02) {
                mbshipD_tox = 0;
            } else {
                mbshipD_tox = 0.5 + 0.5 * Math.cos(Math.PI * (tox - tox_D_pt12) / (tox_D_pt02 - tox_D_pt12));
            }
            double mbshipF_tox;
            if (tox < tox_F_pt01) {
                mbshipF_tox = 0;
            } else if (tox > tox_F_pt11) {
                mbshipF_tox = 1;
            } else {
                mbshipF_tox = 0.5 + 0.5 * Math.sin(Math.PI * ((tox - tox_F_pt01) / (tox_F_pt11 - tox_F_pt01) - 0.5));
            }


            // Calcul des poids pour les 8 règles  
            //WeightDDD=  Minimum(MbshipD(rcair_vo), MbshipD(rcair_de), MbshipD(Tox))
            //WeightDDF=  Minimum(MbshipD(rcair_vo), MbshipD(rcair_de), MbshipF(Tox))
            //WeightDFD=  Minimum(MbshipD(rcair_vo), MbshipF(rcair_de), MbshipD(Tox))
            //WeightDFF=  Minimum(MbshipD(rcair_vo), MbshipF(rcair_de), MbshipF(Tox))
            //WeightFDD=  Minimum(MbshipF(rcair_vo), MbshipD(rcair_de), MbshipD(Tox))
            //WeightFDF=  Minimum(MbshipF(rcair_vo), MbshipD(rcair_de), MbshipF(Tox))
            //WeightFFD=  Minimum(MbshipF(rcair_vo), MbshipF(rcair_de), MbshipD(Tox))
            //WeightFFF=  Minimum(MbshipF(rcair_vo), MbshipF(rcair_de), MbshipF(Tox))
            double weightDDD = NumberUtils.min(mbshipD_rcair_vo, mbshipD_rcair_de, mbshipD_tox);
            double weightDDF = NumberUtils.min(mbshipD_rcair_vo, mbshipD_rcair_de, mbshipF_tox);
            double weightDFD = NumberUtils.min(mbshipD_rcair_vo, mbshipF_rcair_de, mbshipD_tox);
            double weightDFF = NumberUtils.min(mbshipD_rcair_vo, mbshipF_rcair_de, mbshipF_tox);
            double weightFDD = NumberUtils.min(mbshipF_rcair_vo, mbshipD_rcair_de, mbshipD_tox);
            double weightFDF = NumberUtils.min(mbshipF_rcair_vo, mbshipD_rcair_de, mbshipF_tox);
            double weightFFD = NumberUtils.min(mbshipF_rcair_vo, mbshipF_rcair_de, mbshipD_tox);
            double weightFFF = NumberUtils.min(mbshipF_rcair_vo, mbshipF_rcair_de, mbshipF_tox);

            // Calcul des conclusions pondérées pour les 8 règles
            // Rechercher les valeurs de conclusion z()
            // Table
            double zDDD = 0;
            double zDDF = 4;
            double zDFD = 1;
            double zDFF = 9;
            double zFDD = 1;
            double zFDF = 9;
            double zFFD = 6;
            double zFFF = 10;

            // WR_DDD = WeightDDD*z(DDD)
            //to  
            //WR_FFF =    Weight FFF *z(FFF)
            double wr_DDD = weightDDD * zDDD;
            double wr_DDF = weightDDF * zDDF;
            double wr_DFD = weightDFD * zDFD;
            double wr_DFF = weightDFF * zDFF;
            double wr_FDD = weightFDD * zFDD;
            double wr_FDF = weightFDF * zFDF;
            double wr_FFD = weightFFD * zFFD;
            double wr_FFF = weightFFF * zFFF;

            // Calcul de la valeur de l'indicateur 
            // I-Phyair = Somme des 8 conclusions pondérées (WR_DDD to WR_FFF)/Somme 8 poids (WeightDDD to WeightFFF)
            double iphyair = (wr_DDD + wr_DDF + wr_DFD + wr_DFF
                    + wr_FDD + wr_FDF + wr_FFD + wr_FFF) /
                    (weightDDD + weightDDF + weightDFD + weightDFF
                    + weightFDD + weightFDF + weightFFD + weightFFF);

            //Stokage de  I-Phyair
            step.iphyairs.put(produit, substanceActiveIphy, iphyair);
        }
        
        //3. Calcul pour le programme de traitement
        //Rechercher les résultats I-phyairi par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double iphyair = scaleToTreatment(step.iphyairs.values());
        if (log.isDebugEnabled()) {
            log.debug("iphyesu = " + iphyair);
        }
        step.iphyesu = iphyair;

        return true;
    }

    /**
     * I-Phy.
     */
    protected boolean computeIPhy(IPhyStep step, BasicPlot plot, List<PhytoProductInput> inputs,
            EffectiveIntervention effectiveIntervention, PracticedIntervention practicedIntervention) {

        //1. Recherche par substance active
        step.iphys = new MultiKeyMap<Object, Double>();
        for (Map.Entry<MultiKey<? extends Object>, Double> entry : step.iphyesos.entrySet()) {
            RefActaTraitementsProduit produit = (RefActaTraitementsProduit)entry.getKey().getKey(0);
            RefPhytoSubstanceActiveIphy substanceActiveIphy = (RefPhytoSubstanceActiveIphy)entry.getKey().getKey(1);

            // Recherche   I-Phyeso, I-Phyesu, I-Phy-Air
            double iphyeso = entry.getValue();
            double iphyesu = step.iphyesus.get(produit, substanceActiveIphy);
            double iphyair = step.iphyairs.get(produit, substanceActiveIphy);

            // 2. Calcul par substance active

            //Calcul des valeurs d'appartenance pour rceso et Log10(DJA)  
            //Variables FussySet    pt01    pt11    pt12    pt02
            //I-Phyeso    D         0        0       0       10
            //I-Phyeso    F         0        10      0       0
            //I-Phyesu    D         0        0       0       10
            //I-Phyesu    F         0        10      0       0
            //I-Phyair    D         0        0       0       10
            //I-Phyair    F         0        10      0       0

            double iphyeso_D_pt12 = 0,iphyeso_D_pt02 = 10;
            double iphyeso_F_pt01 = 0, iphyeso_F_pt11 = 10;
            double iphyesu_D_pt12 = 0, iphyesu_D_pt02 = 10;
            double iphyesu_F_pt01 = 0, iphyesu_F_pt11 = 10;
            double iphyair_D_pt12 = -4, iphyair_D_pt02 = -4;
            double iphyair_F_pt01 = -4, iphyair_F_pt11 = 0;

            //Fonction D  Si X<pt12 alors  MbshipD(X) = 1
            //            Si X>pt02 alors  MbshipD(X) = 0 
            //            MbshipD(X) = 0.5 + 0.5 * Cos(3.14159267 * (X - pt12) / (pt02 - pt12))
            //Fonction F  Si X<pt01 alors  MbshipF(X) = 0
            //            Si X>pt11 alors  MbshipF(X) = 1
            //            MbshipF(X) = 0.5 + 0.5 * Sin(3.14159267 * ((X - pt01) / (pt11 - pt01) - 0.5))
            double mbshipD_iphyeso;
            if (iphyeso < iphyeso_D_pt12) {
                mbshipD_iphyeso = 1;
            } else if (iphyeso > iphyeso_D_pt02) {
                mbshipD_iphyeso = 0;
            } else {
                mbshipD_iphyeso = 0.5 + 0.5 * Math.cos(Math.PI * (iphyeso - iphyeso_D_pt12) / (iphyeso_D_pt02 - iphyeso_D_pt12));
            }
            double mbshipF_iphyeso;
            if (iphyeso < iphyeso_F_pt01) {
                mbshipF_iphyeso = 0;
            } else if (iphyeso > iphyeso_F_pt11) {
                mbshipF_iphyeso = 1;
            } else {
                mbshipF_iphyeso = 0.5 + 0.5 * Math.sin(Math.PI * ((iphyeso - iphyeso_F_pt01) / (iphyeso_F_pt11 - iphyeso_F_pt01) - 0.5));
            }
            double mbshipD_iphyesu;
            if (iphyesu < iphyesu_D_pt12) {
                mbshipD_iphyesu = 1;
            } else if (iphyesu > iphyesu_D_pt02) {
                mbshipD_iphyesu = 0;
            } else {
                mbshipD_iphyesu = 0.5 + 0.5 * Math.cos(Math.PI * (iphyesu - iphyesu_D_pt12) / (iphyesu_D_pt02 - iphyesu_D_pt12));
            }
            double mbshipF_iphyesu;
            if (iphyesu < iphyesu_F_pt01) {
                mbshipF_iphyesu = 0;
            } else if (iphyesu > iphyesu_F_pt11) {
                mbshipF_iphyesu = 1;
            } else {
                mbshipF_iphyesu = 0.5 + 0.5 * Math.sin(Math.PI * ((iphyesu - iphyesu_F_pt01) / (iphyesu_F_pt11 - iphyesu_F_pt01) - 0.5));
            }
            double mbshipD_iphyair;
            if (iphyair < iphyair_D_pt12) {
                mbshipD_iphyair = 1;
            } else if (iphyair > iphyair_D_pt02) {
                mbshipD_iphyair = 0;
            } else {
                mbshipD_iphyair = 0.5 + 0.5 * Math.cos(Math.PI * (iphyair - iphyair_D_pt12) / (iphyair_D_pt02 - iphyair_D_pt12));
            }
            double mbshipF_iphyair;
            if (iphyair < iphyair_F_pt01) {
                mbshipF_iphyair = 0;
            } else if (iphyair > iphyair_F_pt11) {
                mbshipF_iphyair = 1;
            } else {
                mbshipF_iphyair = 0.5 + 0.5 * Math.sin(Math.PI * ((iphyair - iphyair_F_pt01) / (iphyair_F_pt11 - iphyair_F_pt01) - 0.5));
            }


            // Calcul des poids pour les 8 règles  
            //WeightDDD=  Minimum(MbshipD(iphyeso), MbshipD(iphyesu), MbshipD(Tox))
            //WeightDDF=  Minimum(MbshipD(iphyeso), MbshipD(iphyesu), MbshipF(Tox))
            //WeightDFD=  Minimum(MbshipD(iphyeso), MbshipF(iphyesu), MbshipD(Tox))
            //WeightDFF=  Minimum(MbshipD(iphyeso), MbshipF(iphyesu), MbshipF(Tox))
            //WeightFDD=  Minimum(MbshipF(iphyeso), MbshipD(iphyesu), MbshipD(Tox))
            //WeightFDF=  Minimum(MbshipF(iphyeso), MbshipD(iphyesu), MbshipF(Tox))
            //WeightFFD=  Minimum(MbshipF(iphyeso), MbshipF(iphyesu), MbshipD(Tox))
            //WeightFFF=  Minimum(MbshipF(iphyeso), MbshipF(iphyesu), MbshipF(Tox))
            double weightDDD = NumberUtils.min(mbshipD_iphyeso, mbshipD_iphyesu, mbshipD_iphyair);
            double weightDDF = NumberUtils.min(mbshipD_iphyeso, mbshipD_iphyesu, mbshipF_iphyair);
            double weightDFD = NumberUtils.min(mbshipD_iphyeso, mbshipF_iphyesu, mbshipD_iphyair);
            double weightDFF = NumberUtils.min(mbshipD_iphyeso, mbshipF_iphyesu, mbshipF_iphyair);
            double weightFDD = NumberUtils.min(mbshipF_iphyeso, mbshipD_iphyesu, mbshipD_iphyair);
            double weightFDF = NumberUtils.min(mbshipF_iphyeso, mbshipD_iphyesu, mbshipF_iphyair);
            double weightFFD = NumberUtils.min(mbshipF_iphyeso, mbshipF_iphyesu, mbshipD_iphyair);
            double weightFFF = NumberUtils.min(mbshipF_iphyeso, mbshipF_iphyesu, mbshipF_iphyair);

            // Calcul des conclusions pondérées pour les 8 règles
            // Rechercher les valeurs de conclusion z()
            // Table
            double zDDD = 0;
            double zDDF = 2;
            double zDFD = 2;
            double zDFF = 7;
            double zFDD = 2;
            double zFDF = 7;
            double zFFD = 7;
            double zFFF = 10;

            // WR_DDD = WeightDDD*z(DDD)
            //to  
            //WR_FFF =    Weight FFF *z(FFF)
            double wr_DDD = weightDDD * zDDD;
            double wr_DDF = weightDDF * zDDF;
            double wr_DFD = weightDFD * zDFD;
            double wr_DFF = weightDFF * zDFF;
            double wr_FDD = weightFDD * zFDD;
            double wr_FDF = weightFDF * zFDF;
            double wr_FFD = weightFFD * zFFD;
            double wr_FFF = weightFFF * zFFF;

            // Calcul de la valeur de l'indicateur 
            // I-Phyair = Somme des 8 conclusions pondérées (WR_DDD to WR_FFF)/Somme 8 poids (WeightDDD to WeightFFF)
            double iphy = (wr_DDD + wr_DDF + wr_DFD + wr_DFF
                    + wr_FDD + wr_FDF + wr_FFD + wr_FFF) /
                    (weightDDD + weightDDF + weightDFD + weightDFF
                    + weightFDD + weightFDF + weightFFD + weightFFF);

            //Stokage de  I-Phy
            step.iphys.put(produit, substanceActiveIphy, iphy);
        }
        
        //3. Calcul pour le programme de traitement
        //Rechercher les résultats I-phy par substance active i de la zone phyto (phyto_area) du système, de la parcelle
        double iphy = scaleToTreatment(step.iphys.values());
        if (log.isDebugEnabled()) {
            log.debug("iphy = " + iphy);
        }
        step.iphy = iphy;
        
        return true;
    }
    
    /**
     * Applique l'algorithm de calcul de la valeur finale pour le programme de traitement
     * à partir de toutes les valeurs "par substance active".
     * 
     * @param savalues all subtances active values
     * @return la valeur pour le programme de traitement
     */
    protected double scaleToTreatment(Collection<Double> savalues) {
        double value = 0;
        double valuek = 0;
        Double minvalue = null;

        for (Double valuei : savalues) {
            if (minvalue == null) {
                minvalue = valuei;
            } else if (valuei < minvalue) {
                minvalue = valuei;
            }

            double value0_1 = 1 - valuei / 10;
            double k = 1.7175 * Math.pow(Math.E, -0.2913 * valuei);
            double kmin = 1.7175 * Math.pow(Math.E, -0.2913 * minvalue);
            valuek = valuek + k * value0_1;
            value = minvalue - valuek + (kmin * ( 1 - minvalue / 10));
        }

        // il est possibe que l'algo calcule une valeur négative, mais on
        // ne conserve qu'une valeur positive ici
        if (value < 0) {
            value = 0;
        }

        return value;
    }

    /**
     * Effectue le cumul des susbtances actives via l'algorithme {@link #scaleToTreatment(Collection)}
     * pour le passage à l'echelle.
     * 
     * @param iphysteps les iphystep à passer à l'echelle
     * @return tableau de toutes les valeurs à l'echelle
     */
    protected Double[] sumIPhyStep(List<IPhyStep> iphysteps) {
        // collection pour avoir les valeurs par type
        Collection<Double> rcesos = new ArrayList<Double>();
        Collection<Double> rcesurds = new ArrayList<Double>();
        Collection<Double> rcesudes = new ArrayList<Double>();
        Collection<Double> rcairvos = new ArrayList<Double>();
        Collection<Double> rcairdes = new ArrayList<Double>();
        Collection<Double> iphyesos = new ArrayList<Double>();
        Collection<Double> iphyesus = new ArrayList<Double>();
        Collection<Double> iphyairs = new ArrayList<Double>();
        Collection<Double> iphys = new ArrayList<Double>();
        
        // ajout des valeurs dans les types correspondants
        for (IPhyStep iphystep : iphysteps) {
            rcesos.addAll(iphystep.rcesos.values());
            rcesurds.addAll(iphystep.rcesurds.values());
            rcesudes.addAll(iphystep.rcesudes.values());
            rcairvos.addAll(iphystep.rcairvos.values());
            rcairdes.addAll(iphystep.rcairdes.values());
            iphyesos.addAll(iphystep.iphyesos.values());
            iphyesus.addAll(iphystep.iphyesus.values());
            iphyairs.addAll(iphystep.iphyairs.values());
            iphys.addAll(iphystep.iphys.values());
        }
        
        // aggregation des collections
        Double[] result = new Double[9];
        result[0] = scaleToTreatment(rcesos);
        result[1] = scaleToTreatment(rcesurds);
        result[2] = scaleToTreatment(rcesudes);
        result[3] = scaleToTreatment(rcairvos);
        result[4] = scaleToTreatment(rcairdes);
        result[5] = scaleToTreatment(iphyesos);
        result[6] = scaleToTreatment(iphyesus);
        result[7] = scaleToTreatment(iphyairs);
        result[8] = scaleToTreatment(iphys);
        
        return result;
    }
}
