package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorIFT.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorIFT.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Collection;
import java.util.List;
import java.util.Set;

import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.SeedingActionTopiaDao;
import fr.inra.agrosyst.api.entities.action.SeedingProductInputImpl;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectiveSpeciesStade;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedSpeciesStade;
import fr.inra.agrosyst.api.entities.referential.RefActaDosageSPC;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduit;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitsCateg;
import fr.inra.agrosyst.api.entities.referential.RefActaTraitementsProduitsCategTopiaDao;
import fr.inra.agrosyst.api.services.referential.ReferentialService;

public class IndicatorIFT extends AbstractIndicator {

    private static final Log log = LogFactory.getLog(IndicatorIFT.class);

    protected PhytoProductInputTopiaDao phytoProductInputTopiaDao;
    protected RefActaTraitementsProduitsCategTopiaDao refActaTraitementsProduitsCategTopiaDao;
    protected CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao;
    protected ReferentialService referentialService;
    protected SeedingActionTopiaDao seedingActionTopiaDao;

    protected static final String[] RESULTS = new String[] {
        "IFT total",       // IFT phytosanitaire total
        "IFT tot hts",     // IFT phytosanitaire total hors traitement de semence
        "IFT h",           // IFT herbicide
        "IFT f",           // IFT fongicide
        "IFT i",           // IFT insecticide
        "IFT ts",          // IFT traitement de semence
        "IFT a",           // IFT phytosanitaire autres
        "IFT hh",          // IFT phytosanitaire hors herbicide
        "IFT biocontrole", // IFT vert
        "IFT moyens biologiques", // IFT moyens biologiques
    };

    @Override
    public String getIndicatorCategory() {
        return "Indicateurs de pression d’utilisation des intrants";
    }

    @Override
    public String getIndicatorLabel(int i) {
        return RESULTS[i];
    }

    public void setSeedingActionTopiaDao(SeedingActionTopiaDao seedingActionTopiaDao) {
        this.seedingActionTopiaDao = seedingActionTopiaDao;
    }

    public void setPhytoProductInputTopiaDao(PhytoProductInputTopiaDao phytoProductInputTopiaDao) {
        this.phytoProductInputTopiaDao = phytoProductInputTopiaDao;
    }

    public void setRefActaTraitementsProduitsCategTopiaDao(RefActaTraitementsProduitsCategTopiaDao refActaTraitementsProduitsCategTopiaDao) {
        this.refActaTraitementsProduitsCategTopiaDao = refActaTraitementsProduitsCategTopiaDao;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public void setCroppingPlanSpeciesTopiaDao(CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao) {
        this.croppingPlanSpeciesTopiaDao = croppingPlanSpeciesTopiaDao;
    }

    /**
     * Formule de calcul:
     * 
     * <pre>
     *              DA
     * IFTi = sum ( -- * PSCi )
     *              DH
     * </pre>
     */
    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {

        Double[] result = newArray(RESULTS.length, 0.0);

        List<SeedingAction> seedingActions = seedingActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();
        computeSeedingActionIFT(result, seedingActions);

        List<PhytoProductInput> phytoProductInputs = phytoProductInputTopiaDao.findAllByPracticedIntervention(intervention);
        if (phytoProductInputs != null) {
            for (PhytoProductInput phytoProductInput : phytoProductInputs) {
                // L’onglet «ACTA_traitements_produits» indique quels produits sont concernés par le calcul de l’IFT biocontrôle (colonne F).
                RefActaTraitementsProduit phytoProduct = phytoProductInput.getPhytoProduct();
                if (phytoProduct == null) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't find phyto product for input " + phytoProductInput.getProductName());
                    }
                    continue;
                }
                boolean isSeedingPhytoProduct = phytoProductInput instanceof SeedingProductInputImpl;

                // le cas des actions actions de type semis a été traité plus haut
                if(!isSeedingPhytoProduct) {
                    int idTraitement = phytoProduct.getId_traitement();
                    boolean nodu = phytoProduct.isNodu();

                    // L’onglet « ACTA_traitements_prod_categ » indique quels traitements sont concernés par quelles catégories de l’IFT (colonnes G à O).
                    RefActaTraitementsProduitsCateg traitementProduitCateg = refActaTraitementsProduitsCategTopiaDao.forNaturalId(idTraitement).findUnique();

                    // DA : dose de chaque produit appliqué lors d’une intervention, exprimée en g ou kg par ha
                    // ou hl. Données saisies par l’utilisateur (l’utilisateur entre la dose appliquée et
                    // sélectionne l’unité dans une liste de valeurs). Si la dose est exprimée par hl, elle
                    // doit être multipliée par le volume de bouillie épandu par ha (donnée saisie par l’utilisateur).
                    Double da = phytoProductInput.getQtAvg();
                    if (da == null || phytoProductInput.getPhytoProductUnit() == null) {
                        continue;
                    }
                    if (phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.G_HL ||
                            phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.KG_HL ||
                            phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.L_HL) {
                        if (phytoProductInput instanceof PesticideProductInput) {
                            PesticideProductInput pesticideProductInput = (PesticideProductInput)phytoProductInput;
                            if (pesticideProductInput.getPesticidesSpreadingAction().getBoiledQuantity() != null) {
                                da *= pesticideProductInput.getPesticidesSpreadingAction().getBoiledQuantity();
                            }
                        } else {
                            if (log.isWarnEnabled()) {
                                log.warn("Phyto quantity defined as hl, but not on PesticideProductInput");
                            }
                        }
                    }

                    // DH : dose homologuée du produit appliqué, exprimée en g ou kg par ha ou hl (l’unité doit
                    // être identique à celle saisie par l’utilisateur pour DA). Donnée de référence. Si la dose
                    // appliquée est exprimée par hl, elle doit automatiquement être multipliée par 10 pour l’exprimer
                    // par ha (les doses d’homologation sont calculées sur une base d’un traitement à 10 hl de bouillie par hectare).
                    Collection<PracticedSpeciesStade> speciesStades = intervention.getSpeciesStades();
                    if (CollectionUtils.isNotEmpty(speciesStades)) {
                        // calcul de la valeur de l'ift
                        double ift;

                        Set<String> speciesIds = Sets.newHashSetWithExpectedSize(speciesStades.size());
                        for (PracticedSpeciesStade speciesStade : speciesStades) {
                            CroppingPlanSpecies species = croppingPlanSpeciesTopiaDao.forCodeEquals(speciesStade.getSpeciesCode()).findAny();
                            speciesIds.add(species.getSpecies().getTopiaId());
                        }
                        RefActaDosageSPC referenceDose = referentialService.computeActaReferenceDose(phytoProduct.getId_produit(), speciesIds);
                        double dh = 0.0;
                        if (referenceDose != null) {
                            dh = referenceDose.getDosage_spc_valeur();
                            if (referenceDose.getDosage_spc_unite() == PhytoProductUnit.G_HL ||
                                    referenceDose.getDosage_spc_unite() == PhytoProductUnit.KG_HL ||
                                    referenceDose.getDosage_spc_unite() == PhytoProductUnit.L_HL) {
                                dh *= 10;
                            }
                        } else if (log.isWarnEnabled()) {
                            log.warn(String.format("Can't find reference dose for %s/%s", phytoProduct.getNom_produit(), phytoProduct.getNom_traitement()));
                        }

                        // calcul de la valeur de l'ift
                        if (dh == 0.0) {
                            ift = 1 * getInputPSCi(intervention, phytoProductInput);
                        } else {
                            // calcul de la valeur de l'ift
                            ift = da / dh * getInputPSCi(intervention, phytoProductInput);
                        }

                        feedUpIftResults(result, nodu, traitementProduitCateg, ift);

                    } else if (log.isWarnEnabled()) {
                        log.warn("No species selected for intervention " + intervention.getName());
                    }
                }

            }
        }

        return newResult(result);
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone, CroppingPlanEntry croppingPlanEntry,
            CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {
        Double[] result = newArray(RESULTS.length, 0.0);

        List<SeedingAction> seedingActions = seedingActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();
        computeSeedingActionIFT(result, seedingActions);

        List<PhytoProductInput> phytoProductInputs = phytoProductInputTopiaDao.findAllByEffectiveIntervention(intervention);
        if (phytoProductInputs != null) {
            for (PhytoProductInput phytoProductInput : phytoProductInputs) {
                // L’onglet «ACTA_traitements_produits» indique quels produits sont concernés par le calcul de l’IFT biocontrôle (colonne F).
                RefActaTraitementsProduit phytoProduct = phytoProductInput.getPhytoProduct();
                if (phytoProduct == null) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't find phyto product for input " + phytoProductInput.getProductName());
                    }
                    continue;
                }
                boolean isSeedingPhytoProduct = phytoProductInput instanceof SeedingProductInputImpl;

                // le cas des actions actions de type semis a été traité plus haut
                if(!isSeedingPhytoProduct) {

                    int idTraitement = phytoProduct.getId_traitement();
                    boolean nodu = phytoProduct.isNodu();

                    // L’onglet « ACTA_traitements_prod_categ » indique quels traitements sont concernés par quelles catégories de l’IFT (colonnes G à O).
                    RefActaTraitementsProduitsCateg traitementProduitCateg = refActaTraitementsProduitsCategTopiaDao.forNaturalId(idTraitement).findUnique();

                    // DA : dose de chaque produit appliqué lors d’une intervention, exprimée en g ou kg par ha
                    // ou hl. Données saisies par l’utilisateur (l’utilisateur entre la dose appliquée et
                    // sélectionne l’unité dans une liste de valeurs). Si la dose est exprimée par hl, elle
                    // doit être multipliée par le volume de bouillie épandu par ha (donnée saisie par l’utilisateur).
                    Double da = phytoProductInput.getQtAvg();
                    if (da == null) {
                        continue;
                    }
                    if (phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.G_HL ||
                            phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.KG_HL ||
                            phytoProductInput.getPhytoProductUnit() == PhytoProductUnit.L_HL) {
                        if (phytoProductInput instanceof PesticideProductInput) {
                            PesticideProductInput pesticideProductInput = (PesticideProductInput)phytoProductInput;
                            if (pesticideProductInput.getPesticidesSpreadingAction().getBoiledQuantity() != null) {
                                da *= pesticideProductInput.getPesticidesSpreadingAction().getBoiledQuantity();
                            }
                        } else {
                            if (log.isWarnEnabled()) {
                                log.warn("Phyto quantity defined as hl, but not on PesticideProductInput");
                            }
                        }
                    }

                    // DH : dose homologuée du produit appliqué, exprimée en g ou kg par ha ou hl (l’unité doit
                    // être identique à celle saisie par l’utilisateur pour DA). Donnée de référence. Si la dose
                    // appliquée est exprimée par hl, elle doit automatiquement être multipliée par 10 pour l’exprimer
                    // par ha (les doses d’homologation sont calculées sur une base d’un traitement à 10 hl de bouillie par hectare).
                    Collection<EffectiveSpeciesStade> speciesStades = intervention.getSpeciesStades();
                    if (CollectionUtils.isNotEmpty(speciesStades)) {
                        // calcul de la valeur de l'ift
                        double ift;

                        Set<String> speciesIds = Sets.newHashSetWithExpectedSize(speciesStades.size());
                        for (EffectiveSpeciesStade speciesStade : speciesStades) {
                            CroppingPlanSpecies species = speciesStade.getCroppingPlanSpecies();
                            speciesIds.add(species.getSpecies().getTopiaId());
                        }
                        RefActaDosageSPC referenceDose = referentialService.computeActaReferenceDose(phytoProduct.getId_produit(), speciesIds);
                        double dh = 0.0;
                        if (referenceDose != null) {
                            dh = referenceDose.getDosage_spc_valeur();
                            if (referenceDose.getDosage_spc_unite() == PhytoProductUnit.G_HL ||
                                    referenceDose.getDosage_spc_unite() == PhytoProductUnit.KG_HL ||
                                    referenceDose.getDosage_spc_unite() == PhytoProductUnit.L_HL) {
                                dh *= 10;
                            }
                        } else if (log.isWarnEnabled()) {
                            log.warn(String.format("Can't find reference dose for %s/%s", phytoProduct.getNom_produit(), phytoProduct.getNom_traitement()));
                        }

                        // calcul de la valeur de l'ift
                        if (dh == 0.0) {
                            ift = 1 * getInputPSCi(intervention, phytoProductInput);
                        } else {
                            // calcul de la valeur de l'ift
                            ift = da / dh * getInputPSCi(intervention, phytoProductInput);
                        }

                        feedUpIftResults(result, nodu, traitementProduitCateg, ift);


                    } else if (log.isWarnEnabled()) {
                        log.warn("No species selected for intervention " + intervention.getName());
                    }
                }

            }
        }

        return newResult(result);
    }

    protected void feedUpIftResults(Double[] result, boolean nodu, RefActaTraitementsProduitsCateg traitementProduitCateg, double ift) {
        // Par exemple, pour le calcul de l'IFT Total, l'algorithme regardera dans la table
        // ACTA_traitements_produits si le produit est dans la liste NODU (colonne NODU O/N)
        // Si ce n'est pas le cas, il ira vérifier dans ACTA_traitement_prod_categ si le traitement
        // entre dans le calcul de l'IFT total (colonne IFT_total O/N). Si toutes les conditions
        // sont respectées, alors le produit rentre dans le calcul de l'IFT.

        // affectation de l'IFT au catégories concernées
        if (traitementProduitCateg.isIft_total() && !nodu) {
            result[0] += ift; // "IFT total"
        }
        if (traitementProduitCateg.isIft_tot_hts() && !nodu) {
            result[1] += ift; // "IFT tot hts"
        }
        if (traitementProduitCateg.isIft_h() && !nodu) {
            result[2] += ift; // "IFT h"
        }
        if (traitementProduitCateg.isIft_f() && !nodu) {
            result[3] += ift; // "IFT f"
        }
        if (traitementProduitCateg.isIft_i() && !nodu) {
            result[4] += ift; // "IFT i"
        }
        if (traitementProduitCateg.isIft_ts() && !nodu) {
            result[5] += ift; // "IFT ts"
        }
        if (traitementProduitCateg.isIft_a() && !nodu) {
            result[6] += ift; // "IFT a"
        }
        if (traitementProduitCateg.isIft_hh() && !nodu) {
            result[7] += ift; // "IFT hh"
        }
        if (nodu) {
            result[8] += ift; // "IFT biocontrole"
        }
        if (traitementProduitCateg.isIft_moy_bio()) {
            result[9] += ift; // "IFT moyens biologiques"
        }
    }

    protected void computeSeedingActionIFT(Double[] result, List<SeedingAction> seedingActions) {
        // cas des interventions contenant das actions de type semis
        // refs #5724
        // Compter 1 point d’IFT TS chaque fois qu’une intervention contient une action de type semis avec au moins une espèce traitée chimiquement
        // Compter 1 point d’IFT moy bio chaque fois qu’une intervention contient une action de type semis avec au moins une espèce avec inoculation biologique

        if (seedingActions != null) {
            double ift_ts = 0d;
            double ift_bio = 0d;

            for (SeedingAction seedingAction : seedingActions) {
                boolean isTreatment = false;
                boolean isBiologicalSeedInoculation = false;
                Collection<SeedingActionSpecies> seedingActionSpecieses = seedingAction.getSeedingSpecies();
                if (seedingActionSpecieses != null) {
                    for (SeedingActionSpecies seedingActionSpeciese : seedingActionSpecieses) {
                        if (seedingActionSpeciese.isTreatment()) {
                            isTreatment = true;
                        }
                        if (seedingActionSpeciese.isBiologicalSeedInoculation()) {
                            isBiologicalSeedInoculation = true;
                        }
                    }
                }

                ift_ts = isTreatment ? ift_ts + 1 : ift_ts;
                ift_bio = isBiologicalSeedInoculation ? ift_bio + 1 : ift_bio;
            }

            // prise en compte de la part de la culture dans la surface du SDC sur l'IFT
            result[0] = result[0] + ift_ts; // "IFT total"
            result[1] += 0; // "IFT tot hts"
            result[2] += 0; // "IFT h"
            result[3] += 0; // "IFT f"
            result[4] += 0; // "IFT i"
            result[5] += ift_ts; // "IFT ts"
            result[6] += 0; // "IFT a"
            result[7] += 0; // "IFT hh"
            result[8] += 0; // "IFT biocontrole"
            result[9] += ift_bio; // "IFT moyens biologiques"

        }
    }
}
