package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: IndicatorFuelConsumption.java 4393 2014-10-06 10:00:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/IndicatorFuelConsumption.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Collection;

import com.google.common.base.Strings;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.referential.RefMateriel;
import fr.inra.agrosyst.api.entities.referential.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referential.RefMaterielOutil;
import fr.inra.agrosyst.api.entities.referential.RefMaterielTraction;

/**
 * Consommation de carburant.
 * 
 * Présentation de l’indicateur
 * La consommation de carburant d’une combinaison d’outils, d’un automoteur ou d’un tracteur correspond à la quantité de carburant utilisée lors d’une intervention culturale. Les interventions concernées par cet indicateur sont toutes les interventions qui mobilisent une combinaison d’outils.
 * 
 * Formule de calcul
 *  Consommation(i) = Puissance * taux de charge * 0.22
 * 
 * Avec :
 * Consommation : consommation de carburant (l ha-1)
 * Puissance : puissance de l’automoteur ou du tracteur (ch). Donnée issue d’un référentiel.
 * Taux de charge : taux de charge du tracteur ou de l’automoteur (sans unité). Donnée issue d’un référentiel. Si un tracteur ou un automoteur est utilisé seul, le taux de charge correspond au taux de charge par défaut de ce tracteur ou de cet automoteur. Si un tracteur est utilisé avec un ou plusieurs outils, le taux de charge correspond au taux de charge de l’outil imposant le plus fort taux de charge.
 * 0.22 : valeur de référence pour la consommation de carburant par cheval pour un taux de charge de 100% (l ch-1), source BCMA.
 * Echelles de présentation de la variable calculée
 * Echelle de la variable d’intérêt :
 * La consommation de carburant peut être calculée pour :
 * 1 intervention
 * x interventions (permet les regroupements d’interventions de même type EDI : traction, entretien, travail du sol, semis, épandage, traitements phytosanitaires, opérations en vert, récolte, transport, stockage, conditionnement, irrigation, transformation, divers). 
 * 1 campagne culturale
 * 1 campagne agricole
 * toutes les campagnes agricoles
 * 
 * Echelle spatiale :
 * La consommation de carburant peut être présentée aux échelles spatiales de :
 * la zone,
 * la parcelle,
 * la sole du SdC
 * le domaine
 * 
 * Echelle temporelle :
 * La consommation de carburant peut être présentée aux échelles temporelles de :
 * 1 année
 * plusieurs années,
 * 
 * Les modalités de changement d’échelle sont présentées ici.
 * Données de référence
 * Les données de référence mobilisées dans le calcul de la consommation de carburant sont : la puissance du tracteur ou de l’automoteur ainsi que le taux de charge imposé par les outils de traction ou moyen pour un tracteur ou un automoteur utilisé sans outil. Ces données sont issues du référentiel BCMA. Onglets BCMA_xxx du fichier référentiel.
 * 
 * @author Eric Chatellier
 */
public class IndicatorFuelConsumption extends AbstractIndicator {

    private static final Log log = LogFactory.getLog(IndicatorFuelConsumption.class);

    @Override
    public String getIndicatorCategory() {
        return "Résultats socio-techniques";
    }

    @Override
    public String getIndicatorLabel(int i) {
        return "Consommation de carburant (l/ha)";
    }

    /**
     * Consommation(i) = Puissance * taux de charge * 0.22
     * 
     * Consommation : consommation de carburant (l ha-1)
     * 
     * Puissance : puissance de l’automoteur ou du tracteur (ch). Donnée issue d’un référentiel.
     * 
     * Taux de charge : taux de charge du tracteur ou de l’automoteur (sans unité).
     *  Donnée issue d’un référentiel. Si un tracteur ou un automoteur est utilisé seul, le taux de
     *  charge correspond au taux de charge par défaut de ce tracteur ou de cet automoteur.
     *  Si un tracteur est utilisé avec un ou plusieurs outils, le taux de charge correspond au
     *  taux de charge de l’outil imposant le plus fort taux de charge.
     * 
     * 0.22 : valeur de référence pour la consommation de carburant par cheval pour un taux de
     *  charge de 100% (l ch-1), source BCMA.
     */
    @Override
    public Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase) {

        Collection<String> toolsCouplingCodes = intervention.getToolsCouplingCodes();
        Double puissance = null;
        Double tauxDeCharge = null;
        Double[] result = null;

        if (toolsCouplingCodes != null) {

            for (String toolsCouplingCode : toolsCouplingCodes) {
                ToolsCoupling toolsCoupling = toolsCouplingDAO.forCodeEquals(toolsCouplingCode).findAny();
                Equipment tracteur = toolsCoupling.getTractor();
                
                //FIXME echatellier : que faire si plusieurs tracteur sont trouvés ?
                if (tracteur != null) {
                    RefMateriel materiel = tracteur.getRefMateriel();
                    if (materiel instanceof RefMaterielAutomoteur) {
                        puissance = ((RefMaterielAutomoteur)materiel).getPuissanceChISO();
                        String strTauxCharge = ((RefMaterielAutomoteur)materiel).getDonneesTauxDeChargeMoteur().replace(',', '.');
                        if (strTauxCharge.matches("\\d+(\\.\\d+)?")) {
                            tauxDeCharge = Double.parseDouble(strTauxCharge);
                        } else if (log.isWarnEnabled()) {
                            log.warn("Can't parse '" + strTauxCharge + "' as number");
                        }
                    } else if (materiel instanceof RefMaterielTraction) {
                        puissance = ((RefMaterielTraction)materiel).getPuissanceChIso();
                        String strTauxCharge = ((RefMaterielTraction)materiel).getDonneesTauxDeChargeMoteur().replace(',', '.');
                        if (strTauxCharge.matches("\\d+(\\.\\d+)?")) {
                            tauxDeCharge = Double.parseDouble(strTauxCharge);
                        } else if (Strings.isNullOrEmpty(strTauxCharge)) {
                            // TODO DCossé 20/05/14 retrouver l'origine de la valeur 1.0
                            tauxDeCharge = 1.0;
                        } else if (log.isWarnEnabled()) {
                            log.warn("Can't parse '" + strTauxCharge + "' as number");
                        }
                    }

                    tauxDeCharge = findHigherTauxDeCharge(tauxDeCharge, toolsCoupling);
                }

                // compute result
                result = computeFuelConsumption(puissance, tauxDeCharge, result);
            }
        }

        return result;
    }

    @Override
    public Double[] manageIntervention(EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase) {

        Double puissance = null;
        Double tauxDeCharge = null;
        Double[] result = null;

        Collection<ToolsCoupling> toolsCouplings = intervention.getToolCouplings();
        if (toolsCouplings != null) {
            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                Equipment tracteur = toolsCoupling.getTractor();

                //FIXME echatellier : que faire si plusieurs tracteur sont trouvés ?
                if (tracteur != null) {
                    RefMateriel materiel = tracteur.getRefMateriel();
                    if (materiel instanceof RefMaterielAutomoteur) {
                        puissance = ((RefMaterielAutomoteur)materiel).getPuissanceChISO();
                        String strTauxCharge = ((RefMaterielAutomoteur)materiel).getDonneesTauxDeChargeMoteur().replace(',', '.');
                        if (strTauxCharge.matches("\\d+(\\.\\d+)?")) {
                            tauxDeCharge = Double.parseDouble(strTauxCharge);
                        } else if (log.isWarnEnabled()) {
                            log.warn("Can't parse '" + strTauxCharge + "' as number");
                        }
                    } else if (materiel instanceof RefMaterielTraction) {
                        puissance = ((RefMaterielTraction)materiel).getPuissanceChIso();
                        String strTauxCharge = ((RefMaterielTraction)materiel).getDonneesTauxDeChargeMoteur().replace(',', '.');
                        if (strTauxCharge.matches("\\d+(\\.\\d+)?")) {
                            tauxDeCharge = Double.parseDouble(strTauxCharge);
                        } else if (Strings.isNullOrEmpty(strTauxCharge)) {
                            // TODO DCossé 20/05/14 retrouver l'origine de la valeur 1.0
                            tauxDeCharge = 1.0;
                        } else if (log.isWarnEnabled()) {
                            log.warn("Can't parse '" + strTauxCharge + "' as number");
                        }
                    }

                    tauxDeCharge = findHigherTauxDeCharge(tauxDeCharge, toolsCoupling);
                }

                // compute result
                result = computeFuelConsumption(puissance, tauxDeCharge, result);
            }
        }

        return result;
    }

    protected Double[] computeFuelConsumption(Double puissance, Double tauxDeCharge, Double[] result) {
        if (puissance != null && tauxDeCharge != null) {

            double fuelConsumption = puissance * tauxDeCharge * 0.22;
            result = newResult(fuelConsumption);
            /*if (previousPlanEntry != null) {
                writer.writeEffective("Résultats socio-techniques", "Consommation de carburant (l/ha)", zone.getPlot().getDomain().getCampaign(),
                    zone.getPlot().getDomain(), zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, croppingPlanEntry,
                    previousPlanEntry, intervention, result);
            } else {
                writer.writeEffective("Résultats socio-techniques", "Consommation de carburant (l/ha)", zone.getPlot().getDomain().getCampaign(),
                        zone.getPlot().getDomain(), zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, croppingPlanEntry,
                        phase , intervention, result);
            }*/
        }
        return result;
    }

    protected Double findHigherTauxDeCharge(Double tauxDeCharge, ToolsCoupling toolsCoupling) {
        if (tauxDeCharge != null && toolsCoupling.getEquipments() != null) {
            for (Equipment equipement : toolsCoupling.getEquipments()) {
                if (equipement.getRefMateriel() instanceof RefMaterielOutil) {
                    String strTauxCharge = ((RefMaterielOutil)equipement.getRefMateriel()).getDonnessTauxDeChargeMoteur().replace(',', '.');
                    if (strTauxCharge.matches("\\d+(\\.\\d+)?")) {
                        Double tauxDeCharge2 = Double.parseDouble(strTauxCharge);
                        if (tauxDeCharge2 > tauxDeCharge) {
                            tauxDeCharge = tauxDeCharge2;
                        }
                    } else if (log.isWarnEnabled()) {
                        log.warn("Can't parse '" + strTauxCharge + "' as number");
                    }
                }
            }
        }
        return tauxDeCharge;
    }
}
