package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: Indicator.java 4307 2014-09-10 10:00:58Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/Indicator.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Arrays;
import java.util.regex.Pattern;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.services.performance.IndicatorWriter;

public abstract class Indicator {

    /** Pour retourner une instance de resultat marquant l'indicateur comme "non applicable". */
    protected static final Double[] NOT_APPLICABLE = new Double[0];

    /** Utilisé pour les indicateurs définissant des résultats sur plusieurs mois. */
    protected static final String[] MONTHS = {"Janvier", "Février", "Mars", "Avril", "Mai", "Juin",
        "Juillet", "Août", "Septembre", "Octobre", "Novembre", "Décembre"};

    /** Pattern des dates de périodes des interventions ITK. */
    protected static final Pattern PRACTICED_DATE_PATTERN = Pattern.compile("(\\d\\d?)/(\\d\\d?)");

    /**
     * Retourne la categories de l'indicateur.
     * Par exemple "Résultats socio-techniques".
     * 
     * @return categorie de l'indicateur
     */
    public abstract String getIndicatorCategory();

    /**
     * Retourne le nom de l'indicateur.
     * 
     * Dans le cas ou un indicateur retourne de multiple résultat (Double[]), la méthode est appelée
     * plusieurs fois avec l'indice dans le tableau pour obtenir le nom spécific de l'indicateur
     * pour l'indice demandé.
     * 
     * @param i indice dans le cas de résultat multiple (peut être ingoré suivant l'indicateur)
     * @return indicateur label
     */
    public abstract String getIndicatorLabel(int i);

    public abstract void computePracticed(IndicatorWriter writer, GrowingSystem growingSystem);

    public abstract void computePracticed(IndicatorWriter writer, Domain domain);

    /**
     * Reset state for domain.
     * 
     * @param domain domain to reset state
     */
    public abstract void resetPracticed(Domain domain);

    public abstract void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone);

    /**
     * Zone &gt;&gt; Parcelle.
     * 
     * Pour les parcelles, on parcourt les valeurs pour les zones et on pondère par la surface
     * de la parcelle.
     */
    public abstract void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot);

    /**
     * Reset effective plot.
     * 
     * @param domain
     * @param growingSystem
     * @param plot
     */
    public abstract void resetEffective(Domain domain, GrowingSystem growingSystem, Plot plot);

    public abstract void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem);

    public abstract void resetEffective(Domain domain, GrowingSystem growingSystem);

    /**
     * Système de culture &gt;&gt; Domaine.
     * 
     * @param writer writer
     * @param domain domain
     */
    public abstract void computeEffective(IndicatorWriter writer, Domain domain);

    public abstract void resetEffective(Domain domain);

    /**
     * Pour le synthétisé PSCi est égal à :
     * <pre>PSCi = fréquence temporelle d'intervention * fréquence spatial d'intervention</pre>
     * 
     * @param intervention intervention
     * @return practiced PSCi
     */
    protected double getToolPSCi(PracticedIntervention intervention) {
        double result = intervention.getTemporalFrequency() * intervention.getSpatialFrequency();
        return result;
    }

    /**
     * Pour le synthétisé PSCi est égal à :
     * <pre>PSCi = fréquence temporelle d'intervention * fréquence spatial d'intervention * de la surface traité de l'action</pre>
     * pour les intrants de type "Lutte Biologique" et "Application de produit phyto sanitaire.
     * 
     * Sinon, les autres intrant ne sont pas pris en compte et la formule pour les autres intrants est celle du PSCi Outils:
     * <pre>PSCi = fréquence temporelle d'intervention * fréquence spatial d'intervention</pre>
     * 
     * @param intervention intervention
     * @param input intrant pouvant influencer (ou pas) le calcul du PSCi
     * @return practiced PSCi
     */
    protected double getInputPSCi(PracticedIntervention intervention, PhytoProductInput input) {
        double result = getToolPSCi(intervention);
        if (input instanceof PesticideProductInput) {
            PesticideProductInput pesticideProductInput = (PesticideProductInput)input;
            result *= pesticideProductInput.getPesticidesSpreadingAction().getProportionOfTreatedSurface() / 100;
        } else if (input instanceof BiologicalProductInput) {
            BiologicalProductInput biologicalProductInput = (BiologicalProductInput)input;
            result *= biologicalProductInput.getBiologicalControlAction().getProportionOfTreatedSurface() / 100;
        }
        return result;
    }

    /**
     * Pour le réalisé PSCi est égal à
     * <pre>PSCi = nombre de passage * fréquence spatial d'intervention</pre>
     * 
     * @param intervention intervention
     * @return effective PSCi
     */
    protected double getToolPSCi(EffectiveIntervention intervention) {
        double result = intervention.getTransitCount() * intervention.getSpatialFrequency();
        return result;
    }

    
    /**
     * Pour le réalisé PSCi est égal à :
     * <pre>PSCi = nombre de passage * fréquence spatial d'intervention * de la surface traité de l'action</pre>
     * pour les intrants de type "Lutte Biologique" et "Application de produit phyto sanitaire.
     * 
     * Sinon, les autres intrant ne sont pas pris en compte et la formule pour les autres intrants est celle du PSCi Outils:
     * <pre>PSCi = nombre de passage * fréquence spatial d'intervention</pre>
     * 
     * @param intervention intervention
     * @param input intrant pouvant influencer (ou pas) le calcul du PSCi
     * @return effective PSCi
     */
    protected double getInputPSCi(EffectiveIntervention intervention, PhytoProductInput input) {
        double result = getToolPSCi(intervention);
        if (input instanceof PesticideProductInput) {
            PesticideProductInput pesticideProductInput = (PesticideProductInput)input;
            result *= pesticideProductInput.getPesticidesSpreadingAction().getProportionOfTreatedSurface() / 100;
        } else if (input instanceof BiologicalProductInput) {
            BiologicalProductInput biologicalProductInput = (BiologicalProductInput)input;
            result *= biologicalProductInput.getBiologicalControlAction().getProportionOfTreatedSurface() / 100;
        }
        return result;
    }

    /**
     * Initialize new array with size and default value.
     * 
     * @param length size
     * @param defaultValue defalt value
     * @return initialized array
     */
    protected Double[] newArray(int length, double defaultValue) {
        Double[] array = new Double[length];
        Arrays.fill(array, defaultValue);
        return array;
    }

    /**
     * Sum two double arrays.
     * 
     * @param arr1
     * @param arr2
     * @return arr1
     */
    protected Double[] sum(Double[] arr1, Double[] arr2) {
        Double[] arr3 = new Double[arr1.length];
        for (int i = 0; i < Math.min(arr1.length, arr2.length); i++) {
            arr3[i] = arr1[i] + arr2[i];
        }
        return arr3;
    }

    /**
     * Multiplication du vecteur par un scalaire.
     * 
     * @param arr1 array
     * @param s scalar
     * @return result
     */
    protected Double[] mults(Double[] arr1, double s) {
        Double[] array = new Double[arr1.length];
        for (int i = 0; i < arr1.length; ++i) {
            array[i] = arr1[i] * s;
        }
        return array;
    }

    /**
     * Division du vecteur par un scalaire.
     * 
     * @param arr1 array
     * @param s scalar
     * @return result
     */
    protected Double[] divs(Double[] arr1, double s) {
        Double[] array = new Double[arr1.length];
        for (int i = 0; i < arr1.length; ++i) {
            array[i] = arr1[i] / s;
        }
        return array;
    }

    /**
     * Convert result into always array of double (even if there is nly one value).
     * 
     * @param val val (single double or multiple double, or array of double).
     * @return always array of double
     */
    protected Double[] newResult(Double... val) {
        return val;
    }
}