package fr.inra.agrosyst.services.performance.indicators;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AbstractIndicator.java 4855 2015-03-23 17:56:42Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/indicators/AbstractIndicator.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.ToolsCouplingTopiaDao;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractActionTopiaDao;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.AbstractInputTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnection;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNodeTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectiveInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemTopiaDao;
import fr.inra.agrosyst.services.performance.IndicatorWriter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.keyvalue.MultiKey;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.collections4.map.MultiValueMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

public abstract class AbstractIndicator extends Indicator {

    private static final Log log = LogFactory.getLog(AbstractIndicator.class);

    protected CroppingPlanEntryTopiaDao croppingPlanEntryDao;
    protected ToolsCouplingTopiaDao toolsCouplingDAO;
    protected AbstractActionTopiaDao abstractActionTopiaDao;
    protected AbstractInputTopiaDao abstractInputTopiaDao;

    protected PracticedSystemTopiaDao practicedSystemDao;
    protected PracticedSeasonalCropCycleTopiaDao practicedSeasonalCropCycleDao;
    protected PracticedPerennialCropCycleTopiaDao practicedPerennialCropCycleDao;
    protected PracticedCropCycleConnectionTopiaDao practicedCropCycleConnectionDao;
    protected PracticedInterventionTopiaDao practicedInterventionDAO;

    protected EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao;
    protected EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao;
    protected EffectiveInterventionTopiaDao effectiveInterventionTopiaDao;
    protected EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao;
    protected EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao;

    public void setCroppingPlanEntryDao(CroppingPlanEntryTopiaDao croppingPlanEntryDao) {
        this.croppingPlanEntryDao = croppingPlanEntryDao;
    }

    public void setToolsCouplingDAO(ToolsCouplingTopiaDao toolsCouplingDAO) {
        this.toolsCouplingDAO = toolsCouplingDAO;
    }

    public void setAbstractActionTopiaDao(AbstractActionTopiaDao abstractActionTopiaDao) {
        this.abstractActionTopiaDao = abstractActionTopiaDao;
    }

    public void setAbstractInputTopiaDao(AbstractInputTopiaDao abstractInputTopiaDao) {
        this.abstractInputTopiaDao = abstractInputTopiaDao;
    }

    public void setPracticedSystemDao(PracticedSystemTopiaDao practicedSystemDao) {
        this.practicedSystemDao = practicedSystemDao;
    }

    public void setPracticedSeasonalCropCycleDao(PracticedSeasonalCropCycleTopiaDao practicedSeasonalCropCycleDao) {
        this.practicedSeasonalCropCycleDao = practicedSeasonalCropCycleDao;
    }

    public void setPracticedPerennialCropCycleDao(PracticedPerennialCropCycleTopiaDao practicedPerennialCropCycleDao) {
        this.practicedPerennialCropCycleDao = practicedPerennialCropCycleDao;
    }

    public void setPracticedCropCycleConnectionDao(PracticedCropCycleConnectionTopiaDao practicedCropCycleConnectionDao) {
        this.practicedCropCycleConnectionDao = practicedCropCycleConnectionDao;
    }

    public void setPracticedInterventionDAO(PracticedInterventionTopiaDao practicedInterventionDAO) {
        this.practicedInterventionDAO = practicedInterventionDAO;
    }

    public void setEffectivePerennialCropCycleTopiaDao(EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao) {
        this.effectivePerennialCropCycleTopiaDao = effectivePerennialCropCycleTopiaDao;
    }

    public void setEffectiveInterventionTopiaDao(EffectiveInterventionTopiaDao effectiveInterventionTopiaDao) {
        this.effectiveInterventionTopiaDao = effectiveInterventionTopiaDao;
    }

    public void setEffectiveCropCycleConnectionTopiaDao(EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao) {
        this.effectiveCropCycleConnectionTopiaDao = effectiveCropCycleConnectionTopiaDao;
    }

    public EffectiveCropCycleConnectionTopiaDao getEffectiveCropCycleConnectionTopiaDao() {
        return effectiveCropCycleConnectionTopiaDao;
    }

    public void setEffectiveSeasonalCropCycleTopiaDao(EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao) {
        this.effectiveSeasonalCropCycleTopiaDao = effectiveSeasonalCropCycleTopiaDao;
    }

    public void setEffectiveCropCycleNodeTopiaDao(EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao) {
        this.effectiveCropCycleNodeTopiaDao = effectiveCropCycleNodeTopiaDao;
    }

    protected MultiKeyMap<Object, Double[]> practicedSystemValues = new MultiKeyMap<Object, Double[]>();

    protected MultiKeyMap<Object, Double[]> practicedSystemCycleValues = new MultiKeyMap<Object, Double[]>();

    @Override
    public void computePracticed(IndicatorWriter writer, GrowingSystem growingSystem) {

        // multi map PracticedSystem, CroppingPlanEntryCode, CroppingPlanEntryCode > Double[]
        MultiKeyMap<Object, Double[]> practicedCroppingValues = new MultiKeyMap<Object, Double[]>();

        List<PracticedSystem> practicedSystems = practicedSystemDao.forGrowingSystemEquals(growingSystem).findAll();

        for (PracticedSystem practicedSystem : practicedSystems) {
            String campaigns = practicedSystem.getCampaigns();

            // seasonal
            List<PracticedSeasonalCropCycle> practicedSeasonalCropCycles = practicedSeasonalCropCycleDao.forPracticedSystemEquals(practicedSystem).findAll();
            for (PracticedSeasonalCropCycle cycle : practicedSeasonalCropCycles) {

                // sum value on cycle and divide by campaigns count to get 
                Map<PracticedCropCycleConnection, Double[]> cycleValues = Maps.newHashMap();

                Collection<PracticedCropCycleNode> nodes = cycle.getCropCycleNodes();
                if (CollectionUtils.isEmpty(nodes)) {
                    continue;
                }

                // recuperation des connections du cycle
                List<PracticedCropCycleConnection> cropCycleConnections = practicedCropCycleConnectionDao.forTargetIn(nodes).findAll();
                // recupération de toutes les interventions de toutes les connexions du cyle
                List<PracticedIntervention> interventions = practicedInterventionDAO.forPracticedCropCycleConnectionIn(cropCycleConnections).findAll();

                // compute cumulative frequencies
                Map<PracticedCropCycleConnection, Double> cumulativeFrequencies = computeCumulativeFrequencies(cropCycleConnections);

                for (PracticedIntervention intervention : interventions) {
                    PracticedCropCycleConnection connection = intervention.getPracticedCropCycleConnection();

                    // get cropping plan entry and previous cropping plan entry
                    String croppingPlanEntryCode;
                    if (intervention.isIntermediateCrop() && !Strings.isNullOrEmpty(connection.getIntermediateCroppingPlanEntryCode())) {
                        croppingPlanEntryCode = connection.getIntermediateCroppingPlanEntryCode();
                    } else {
                        croppingPlanEntryCode = connection.getTarget().getCroppingPlanEntryCode();
                    }
                    String previousPlanEntryCode = connection.getSource().getCroppingPlanEntryCode();

                    Double[] interValues = manageIntervention(intervention, growingSystem, campaigns, croppingPlanEntryCode, previousPlanEntryCode, null);
                    if (interValues != null) {

                        // sortie résultat courant
                        CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
                        CroppingPlanEntry previousPlanEntry = croppingPlanEntryDao.forCodeEquals(previousPlanEntryCode).findAny();
                        List<AbstractAction> actions = abstractActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();
                        List<AbstractInput> inputs = abstractInputTopiaDao.findAllByPracticedIntervention(intervention);
                        for (int i = 0; i < interValues.length; i++) {
                            writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i),
                                    campaigns, growingSystem.getGrowingPlan().getDomain(),
                                    growingSystem, practicedSystem, croppingPlanEntry, previousPlanEntry, intervention,
                                    actions, inputs, interValues[i]);
                        }

                        // somme pour CC
                        Double[] previous = practicedCroppingValues.get(practicedSystem, croppingPlanEntryCode, previousPlanEntryCode);
                        if (previous == null) {
                            practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, previousPlanEntryCode, interValues);
                        } else {
                            Double[] sum = sum(previous, interValues);
                            practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, previousPlanEntryCode, sum);
                        }

                        // somme pour le cycle seulement
                        Double[] previousValue = cycleValues.get(connection);
                        if (previousValue == null) {
                            cycleValues.put(connection, interValues);
                        } else {
                            Double[] sum = sum(previousValue, interValues);
                            cycleValues.put(connection, sum);
                        }
                    }
                }

                // get rank count
                int campaignCount = getCampaignsCount(cropCycleConnections);
                Double[] result = null;
                // moyenne pour CA
                for (Map.Entry<PracticedCropCycleConnection, Double[]> entry : cycleValues.entrySet()) {
                    Double[] values = entry.getValue();
                    // application de la correction de fréquence cumulées
                    values = mults(values, cumulativeFrequencies.get(entry.getKey()));
                    result = result == null ? values : sum(result, values);
                }
                if (result != null && campaignCount > 0) {
                    result = divs(result, campaignCount);

                    Double[] pspreviousValue = practicedSystemValues.get(campaigns, growingSystem, practicedSystem);
                    if (pspreviousValue == null) {
                        practicedSystemValues.put(campaigns, growingSystem, practicedSystem, result);
                    } else {
                        Double[] sum = sum(pspreviousValue, result);
                        practicedSystemValues.put(campaigns, growingSystem, practicedSystem, sum);
                    }
                }


            }

            // perenial
            List<PracticedPerennialCropCycle> practicedPerennialCropCycles = practicedPerennialCropCycleDao.forPracticedSystemEquals(practicedSystem).findAll();

            if (!practicedPerennialCropCycles.isEmpty()) {
                // surface totale d'occupation des cultures pérennes sur le SDC, devrait-être égal à 100.
                Double totalSolOccupationPercent = 0d;
                for (PracticedPerennialCropCycle cycle : practicedPerennialCropCycles) {
                    totalSolOccupationPercent += cycle.getSolOccupationPercent();
                }
                totalSolOccupationPercent = totalSolOccupationPercent > 0d ? totalSolOccupationPercent : null;

                for (PracticedPerennialCropCycle cycle : practicedPerennialCropCycles) {

                    Collection<PracticedCropCyclePhase> phases = cycle.getCropCyclePhases();
                    List<PracticedIntervention> interventions = practicedInterventionDAO.forPracticedCropCyclePhaseIn(phases).findAll();
                    for (PracticedIntervention intervention : interventions) {
                        String croppingPlanEntryCode = cycle.getCroppingPlanEntryCode();
                        PracticedCropCyclePhase phase = intervention.getPracticedCropCyclePhase();
                        Double[] interValues = manageIntervention(intervention, growingSystem, campaigns, croppingPlanEntryCode, null, phase);
                        if (interValues != null) {

                            // sortie résultat courant
                            CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
                            List<AbstractAction> actions = abstractActionTopiaDao.forPracticedInterventionEquals(intervention).findAll();
                            List<AbstractInput> inputs = abstractInputTopiaDao.findAllByPracticedIntervention(intervention);
                            for (int i = 0; i < interValues.length; i++) {
                                writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i),
                                        campaigns, growingSystem.getGrowingPlan().getDomain(),
                                        growingSystem, practicedSystem, croppingPlanEntry, phase, intervention, actions, inputs, interValues[i]);
                            }

                            // somme pour CC
                            Double[] previous = practicedCroppingValues.get(practicedSystem, croppingPlanEntryCode, phase);
                            if (previous == null) {
                                practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, phase, interValues);
                            } else {
                                Double[] sum = sum(previous, interValues);
                                practicedCroppingValues.put(practicedSystem, croppingPlanEntryCode, phase, sum);
                            }

                            Double[] pspreviousValue = practicedSystemCycleValues.get(practicedSystem, cycle);
                            if (pspreviousValue == null) {
                                practicedSystemCycleValues.put(practicedSystem, cycle, interValues);
                            } else {
                                Double[] sum = sum(pspreviousValue, interValues);
                                practicedSystemCycleValues.put(practicedSystem, cycle, sum);
                            }
                        }
                    }
                }

                // mise à l'échelle vers Practiced System, on tient compte de la part des cultures perennes dans le sdc
                for (Map.Entry<MultiKey<? extends Object>, Double[]> entry : practicedSystemCycleValues .entrySet()) {
                    PracticedSystem ps = (PracticedSystem)entry.getKey().getKey(0);
                    PracticedPerennialCropCycle cycle = (PracticedPerennialCropCycle)entry.getKey().getKey(1);
                    Double[] psValues = practicedSystemCycleValues.get(ps, cycle);
                    Double[] psPartValues = addPerennialCropPart(cycle.getSolOccupationPercent(), totalSolOccupationPercent, psValues);

                    Double[] pspreviousValue = practicedSystemValues.get(campaigns, growingSystem, practicedSystem);
                    if (pspreviousValue == null) {
                        practicedSystemValues.put(campaigns, growingSystem, practicedSystem, psPartValues);
                    } else {
                        Double[] sum = sum(pspreviousValue, psPartValues);
                        practicedSystemValues.put(campaigns, growingSystem, practicedSystem, sum);
                    }
                }
            }
        }

        // mise à l'echelle Intervention >> Culture (CC)
        for (Map.Entry<MultiKey<? extends Object>, Double[]> entry : practicedCroppingValues.entrySet()) {
            PracticedSystem practicedSystem = (PracticedSystem)entry.getKey().getKey(0);
            String croppingPlanEntryCode = (String)entry.getKey().getKey(1);
            Object previousPlanEntryCodeOrPhase = entry.getKey().getKey(2);
            Double[] values = entry.getValue();
            CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forCodeEquals(croppingPlanEntryCode).findAny();
            if (previousPlanEntryCodeOrPhase instanceof String) {
                CroppingPlanEntry previousPlanEntry = croppingPlanEntryDao.forCodeEquals((String)previousPlanEntryCodeOrPhase).findAny();
                for (int i = 0; i < values.length; i++) {
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), practicedSystem.getCampaigns(),
                        growingSystem.getGrowingPlan().getDomain(),
                        growingSystem, practicedSystem, croppingPlanEntry, previousPlanEntry, values[i]);
                }
            } else {
                for (int i = 0; i < values.length; i++) {
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), practicedSystem.getCampaigns(),
                        growingSystem.getGrowingPlan().getDomain(),
                        growingSystem, practicedSystem, croppingPlanEntry, (PracticedCropCyclePhase)previousPlanEntryCodeOrPhase, values[i]);
                }
            }
        }
    }

    public Double[] addPerennialCropPart(Double solOccupationPercent, Double totalSolOccupationPercent, Double[] result) {
        Double[] r = new Double[result.length];
        // calcul de la part des culture pérennes dans la surface du SDC
        double part = solOccupationPercent == null ? 1 : solOccupationPercent/100;
        double sumPart = totalSolOccupationPercent == null ? 1 : totalSolOccupationPercent/100;

        if(sumPart == 0) {
            // resultat non valid on ignore le calcul des part
            sumPart = 1;
            part = 1;
        }

        for (int i = 0; i < r.length; i++) {
            Double aDouble = r[i];
            // prise en compte de la part de la culture dans la surface du SDC
            r[i] = (result[i] * part)/sumPart;
        }
        return r;
    }

    /**
     * Pour un ensemble de connexion, reconstitue le graph pour calculer les % cumulés de chaque
     * connexion depuis l'origine du graph.
     * 
     * @param cropCycleConnections
     * @return
     */
    protected Map<PracticedCropCycleConnection, Double> computeCumulativeFrequencies(List<PracticedCropCycleConnection> cropCycleConnections) {
        List<PracticedCropCycleConnection> localConnections = Lists.newArrayList(cropCycleConnections);

        // get precedence of each node
        MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection> precedenceMap = new MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection>();
        for (PracticedCropCycleConnection conn : localConnections) {
            precedenceMap.put(conn.getTarget(), conn);
        }

        // on calcul les frequences cumulées en aditionnant les frequences des précédences
        // et on multiplie le tout par la frequence de la connexion courante excepté dans le cas ou la source est sur un rand supérieur à la target
        Map<PracticedCropCycleConnection, Double> result = Maps.newHashMap();
        for (PracticedCropCycleConnection conn : localConnections) {
            double frequency = getCumulativeFrequencies(precedenceMap, conn);
            result.put(conn, frequency);
        }
        return result;
    }

    /**
     * Retourne le pourcentage cumulé d'une connection récursivement jusqu'à ce qu'une connexion
     * n'ait plus de précédence.
     * 
     * @param precedenceMap precedence map
     * @param conn connection
     * @return cumulative frequency
     */
    protected double getCumulativeFrequencies(MultiValueMap<PracticedCropCycleNode, PracticedCropCycleConnection> precedenceMap, PracticedCropCycleConnection conn) {
        double result;
        Collection<PracticedCropCycleConnection> precedences = precedenceMap.getCollection(conn.getSource());
        if (CollectionUtils.isEmpty(precedences)) {
            result = 1;
        } else {
            result = 0;
            for (PracticedCropCycleConnection precedence : precedences) {
                // manage loop connection on end cycle node
                if (precedence.getSource().isEndCycle() && (precedence.getSource().getRank() > precedence.getTarget().getRank())) {
                    // previous node = first node
                    // on retourne la valeur de la fréquence initiale
                    result = precedence.getTarget().getInitNodeFrequency() != null ? (precedence.getTarget().getInitNodeFrequency()/100) : 0;
                } else if (precedence.getSource().getRank() < precedence.getTarget().getRank()) {
                    // on fait la somme des fréquences cumulées
                    result += getCumulativeFrequencies(precedenceMap, precedence);
                }
            }
        }
        if (conn.getSource().getRank() < conn.getTarget().getRank()) {
            result *= conn.getCroppingPlanEntryFrequency() / 100.0;
        } else {
            // Fréquence d apparition d un couple culture∗précédent=fréquence initiale de la culture x ∏ fréquences de tous les liens entre cette 1ère culture et son précédent
            double initialFreq = conn.getTarget().getInitNodeFrequency() != null ? (conn.getTarget().getInitNodeFrequency()/100) : 1;
            result = result * initialFreq;
        }
        return result;
    }

    /**
     * Parmis les rangs définit sur le graph, détermine combien de campagnes existe réelement
     * dans les cas où des campagnes portent sur plusieurs rang.
     * 
     * @param cropCycleConnections cycle connections
     * @return campaigns count
     */
    protected int getCampaignsCount(List<PracticedCropCycleConnection> cropCycleConnections) {
        // les set permettent de comptes plusieurs fois les rangs, mais de ne les
        // prendre en compte qu'une seule fois au final
        Set<Integer> foundRanks = Sets.newHashSet();
        Set<Integer> sameCampaignRanks = Sets.newHashSet();

        for (PracticedCropCycleConnection cropCycleConnection : cropCycleConnections) {
            PracticedCropCycleNode sourceNode = cropCycleConnection.getSource();
            PracticedCropCycleNode targetNode = cropCycleConnection.getTarget();
            
            foundRanks.add(sourceNode.getRank());
            foundRanks.add(targetNode.getRank());
            if (targetNode.isSameCampaignAsPreviousNode()) {
                sameCampaignRanks.add(targetNode.getRank());
            }
        }
        foundRanks.removeAll(sameCampaignRanks);
        return foundRanks.size();
    }

    /*
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#computePracticed(fr.inra.agrosyst.services.performance.IndicatorWriter, fr.inra.agrosyst.api.entities.Domain)
     */
    @Override
    public void computePracticed(IndicatorWriter writer, Domain domain) {

        
        Map<String, Double[]> weightedValueSum = Maps.newLinkedHashMap();
        Map<String, Double[]> weightSum = Maps.newLinkedHashMap();


        for (Map.Entry<MultiKey<? extends Object>, Double[]> entry : practicedSystemValues .entrySet()) {
            String campaigns = (String)entry.getKey().getKey(0);
            GrowingSystem growingSystem = (GrowingSystem)entry.getKey().getKey(1);
            PracticedSystem practicedSystem = (PracticedSystem)entry.getKey().getKey(2);

            Double[] values = entry.getValue();

            // CA
            for (int i = 0; i < values.length; i++) {
                writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), campaigns, domain, growingSystem, practicedSystem, values[i]);
            }

            if (domain.getUsedAgriculturalArea() != null && growingSystem.getAffectedAreaRate() != null) {
                // somme des valeurs pondérée
                if (!weightedValueSum.containsKey(campaigns)) {
                    weightedValueSum.put(campaigns, newArray(values.length, 0.0));
                }
                for (int i = 0; i < values.length; i++) {
                    weightedValueSum.get(campaigns)[i] += values[i] * domain.getUsedAgriculturalArea() * growingSystem.getAffectedAreaRate();
                }

                // somme des poids
                if (!weightSum.containsKey(campaigns)) {
                    weightSum.put(campaigns, newArray(values.length, 0.0));
                }
                for (int i = 0; i < values.length; i++) {
                    weightSum.get(campaigns)[i] += domain.getUsedAgriculturalArea() * growingSystem.getAffectedAreaRate();
                }
            }

        }

        for (Map.Entry<String, Double[]> weightedValueEntry : weightedValueSum.entrySet()) {
            String campaigns = weightedValueEntry.getKey();
            Double[] values = weightedValueEntry.getValue();
            Double[] weightValues = weightSum.get(campaigns);
            for (int i = 0; i < values.length; i++) {
                if (weightValues[i] != 0) {
                    double weightedAverage = values[i] / weightValues[i];
                    writer.writePracticed(getIndicatorCategory(), getIndicatorLabel(i), campaigns, domain, weightedAverage);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't compute growing system scale with 0 weigth");
                    }
                }
            }
        }
    }

    /*
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#resetPracticed(fr.inra.agrosyst.api.entities.Domain)
     */
    @Override
    public void resetPracticed(Domain domain) {
        // cleanup for next iteration
        practicedSystemCycleValues.clear();
        practicedSystemValues.clear();
    }

    public abstract Double[] manageIntervention(PracticedIntervention intervention, GrowingSystem growingSystem,
            String campaigns, String croppingPlanEntryCode, String previousPlanEntryCode, PracticedCropCyclePhase phase);

    protected MultiKeyMap<Object, Double[]> effectiveCroppingValues = new MultiKeyMap<Object, Double[]>();
    protected Map<Zone, Double[]> effectiveZoneValues = Maps.newHashMap();
    protected Map<Plot, Double[]> effectivePlotValues = Maps.newHashMap();
    protected Map<GrowingSystem, Double[]> effectiveGrowingSystemValues = Maps.newHashMap();

    /*
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#computeEffective(fr.inra.agrosyst.services.performance.IndicatorWriter, fr.inra.agrosyst.api.entities.Domain, fr.inra.agrosyst.api.entities.GrowingSystem, fr.inra.agrosyst.api.entities.plot, fr.inra.agrosyst.api.entities.Zone)
     */
    @Override
    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone) {
       
        // perennial
        List<EffectivePerennialCropCycle> perennialCycles = effectivePerennialCropCycleTopiaDao.forZoneEquals(zone).findAll();
        for (EffectivePerennialCropCycle perennialCycle : perennialCycles) {
            EffectiveCropCyclePhase phase = perennialCycle.getPhase();
            List<EffectiveIntervention> interventions = effectiveInterventionTopiaDao.forEffectiveCropCyclePhaseEquals(phase).findAll();
            for (EffectiveIntervention intervention : interventions) {
                Double[] interValues = manageIntervention(intervention, zone, perennialCycle.getCroppingPlanEntry(), null, phase);

                if (interValues != null) {
                    // sortie résultat courant
                    List<AbstractAction> actions = abstractActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();
                    List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
                    for (int i = 0; i < interValues.length; i++) {
                        writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i),
                            domain.getCampaign(), domain,
                            zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, perennialCycle.getCroppingPlanEntry(),
                            phase, intervention, actions, inputs, interValues[i]);
                    }

                    // somme pour CC
                    Double[] previous = effectiveCroppingValues.get(perennialCycle.getCroppingPlanEntry(), phase);
                    if (previous == null) {
                        effectiveCroppingValues.put(perennialCycle.getCroppingPlanEntry(), phase, interValues);
                    } else {
                        Double[] sum = sum(previous, interValues);
                        effectiveCroppingValues.put(perennialCycle.getCroppingPlanEntry(), phase, sum);
                    }

                    // somme pour CA
                    Double[] previousValue = effectiveZoneValues.get(zone);
                    if (previousValue == null) {
                        effectiveZoneValues.put(zone, interValues);
                    } else {
                        Double[] sum = sum(previousValue, interValues);
                        effectiveZoneValues.put(zone, sum);
                    }
                }
            }
        }

        // seasonnal
        List<EffectiveSeasonalCropCycle> seasonalCycles = effectiveSeasonalCropCycleTopiaDao.forZoneEquals(zone).findAll();
        for (EffectiveSeasonalCropCycle seasonalCycle : seasonalCycles) {
            Collection<EffectiveCropCycleNode> nodes = seasonalCycle.getNodes();
            if (CollectionUtils.isEmpty(nodes)) {
                continue;
            }
            for (final EffectiveCropCycleNode node : nodes) {

                EffectiveCropCycleConnection connection = effectiveCropCycleConnectionTopiaDao.forTargetEquals(node).findUniqueOrNull();

                List<EffectiveIntervention> interventions = effectiveInterventionTopiaDao.forEffectiveCropCycleNodeEquals(node).findAll();

                for (EffectiveIntervention intervention : interventions) {
                    if (connection == null) {
                        // case of first node
                        CroppingPlanEntry croppingPlanEntry = node.getCroppingPlanEntry();
                        computeEffectiveInterventionSheet(writer, domain, zone, node, intervention, croppingPlanEntry, null);
                    } else {
                        // get cropping plan entry and previous cropping plan entry
                        CroppingPlanEntry croppingPlanEntry;
                        if (intervention.isIntermediateCrop() && connection.getIntermediateCroppingPlanEntry() != null) {
                            croppingPlanEntry = connection.getIntermediateCroppingPlanEntry();
                        } else {
                            croppingPlanEntry = node.getCroppingPlanEntry();
                        }

                        // if source is null, it's a reference to last node of previous year
                        CroppingPlanEntry previousPlanEntry;
                        if (connection.getSource() == null) {
                            EffectiveCropCycleNode lastNode = effectiveCropCycleNodeTopiaDao.findLastNodeForPreviousCampaign(zone);
                            if (lastNode == null) {
                                previousPlanEntry = null;
                            } else {
                                // should not happen
                                previousPlanEntry = lastNode.getCroppingPlanEntry();
                            }
                        } else {
                            previousPlanEntry = connection.getSource().getCroppingPlanEntry();
                        }
                        if (previousPlanEntry == null) {
                            // should not happen
                            if (log.isWarnEnabled()) {
                                log.warn("Can't get previous cropping plan entry");
                            }
                        }

                        computeEffectiveInterventionSheet(writer, domain, zone, node, intervention, croppingPlanEntry, previousPlanEntry);
                    }
                }
            }
        }

        // la map effectiveZoneValues est valuée plusieurs fois avec plusieurs zones par plusieurs
        // appel pour la mise à l'echelle de la methode suivante 
        // computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, plot plot)
        // on ne genere une ligne dans le fichier de sortie que pour la zone courante (s'il y a des valeurs)
        // et non pour toutes les zones
        Double[] values = effectiveZoneValues.get(zone);
        if (values != null) {
            for (int i = 0; i < values.length; i++) {
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), zone.getPlot().getDomain().getCampaign(),
                        zone.getPlot().getDomain(), growingSystem, zone.getPlot(), zone, values[i]);
            }
        }

    }

    protected void computeEffectiveInterventionSheet(IndicatorWriter writer, Domain domain, Zone zone, EffectiveCropCycleNode node, EffectiveIntervention intervention, CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry) {
        Double[] interValues = manageIntervention(intervention, zone, node.getCroppingPlanEntry(), previousPlanEntry, null);
        if (interValues != null) {

            // sortie résultat courant
            List<AbstractAction> actions = abstractActionTopiaDao.forEffectiveInterventionEquals(intervention).findAll();
            List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
            for (int i = 0; i < interValues.length; i++) {
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i),
                        domain.getCampaign(), domain,
                        zone.getPlot().getGrowingSystem(), zone.getPlot(), zone, croppingPlanEntry, previousPlanEntry,
                        intervention, actions, inputs, interValues[i]);
            }

            // somme pour CC
            Double[] previous = effectiveCroppingValues.get(croppingPlanEntry, previousPlanEntry);
            if (previous == null) {
                effectiveCroppingValues.put(croppingPlanEntry, previousPlanEntry, interValues);
            } else {
                Double[] sum = sum(previous, interValues);
                effectiveCroppingValues.put(croppingPlanEntry, previousPlanEntry, sum);
            }

            // somme pour CA
            Double[] previousValue = effectiveZoneValues.get(zone);
            if (previousValue == null) {
                effectiveZoneValues.put(zone, interValues);
            } else {
                Double[] sum = sum(previousValue, interValues);
                effectiveZoneValues.put(zone, sum);
            }
        }
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#computeEffective(fr.inra.agrosyst.services.performance.IndicatorWriter, fr.inra.agrosyst.api.entities.Domain, fr.inra.agrosyst.api.entities.GrowingSystem, fr.inra.agrosyst.api.entities.plot)
     */
    @Override
    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem, Plot plot) {

        Double[] weightedValueSum = null;
        Double[] weightSum = null;

        // moyenne pondérée sur la surface
        for (Map.Entry<Zone, Double[]> entry : effectiveZoneValues.entrySet()) {
            Zone zone = entry.getKey();
            Double[] values = entry.getValue();

            if (weightedValueSum == null) {
                weightedValueSum = newArray(values.length, 0.0);
                weightSum = newArray(values.length, 0.0);
            }
            for (int i = 0; i < values.length; i++) {
                weightedValueSum[i] += values[i] * zone.getArea();
                weightSum[i] += zone.getArea();
            }
        }

        if (weightSum != null) {
            effectivePlotValues.put(plot, newArray(weightedValueSum.length, 0.0));
            for (int i = 0; i < weightedValueSum.length; i++) {
                double plotValue = weightedValueSum[i] / weightSum[i];
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), plot.getDomain().getCampaign(),
                    plot.getDomain(), growingSystem, plot, plotValue);
                
                
                effectivePlotValues.get(plot)[i] = plotValue;
            }
        }
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#resetEffective(fr.inra.agrosyst.api.entities.Domain, fr.inra.agrosyst.api.entities.GrowingSystem, fr.inra.agrosyst.api.entities.plot)
     */
    @Override
    public void resetEffective(Domain domain, GrowingSystem growingSystem, Plot plot) {
        // cleanup for next iteration
        effectiveZoneValues.clear();
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#computeEffective(fr.inra.agrosyst.services.performance.IndicatorWriter, fr.inra.agrosyst.api.entities.Domain, fr.inra.agrosyst.api.entities.GrowingSystem)
     */
    @Override
    public void computeEffective(IndicatorWriter writer, Domain domain, GrowingSystem growingSystem) {

        // toutes les interventions culturales d’une campagne culturale (CC)
        for (Map.Entry<MultiKey<? extends Object>, Double[]> entry : effectiveCroppingValues.entrySet()) {
            Double[] values = entry.getValue();
            CroppingPlanEntry croppingPlanEntry = (CroppingPlanEntry)entry.getKey().getKey(0);
            Object previousPlanEntryOrPhase = entry.getKey().getKey(1);

            if (previousPlanEntryOrPhase != null) {
                for (int i = 0; i < values.length; i++) {
                    if (previousPlanEntryOrPhase instanceof CroppingPlanEntry) {
                        writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                                domain, growingSystem, croppingPlanEntry, (CroppingPlanEntry)previousPlanEntryOrPhase, values[i]);
                    } else {
                        writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                                domain, growingSystem, croppingPlanEntry, (EffectiveCropCyclePhase)previousPlanEntryOrPhase, values[i]);
                    }
                }
            }
        }

        // Parcelle >> SdC
        // plot: moyenne pondérée sur la surface
        Double[] weightedValueSum = null;
        Double[] weightSum = null;
        for (Map.Entry<Plot, Double[]> entry : effectivePlotValues.entrySet()) {
            Plot plot = entry.getKey();
            Double[] values = entry.getValue();

            if (weightedValueSum == null) {
                weightedValueSum = newArray(values.length, 0.0);
                weightSum = newArray(values.length, 0.0);
            }
            for (int i = 0; i < values.length; i++) {
                weightedValueSum[i] += values[i] * plot.getArea();
                weightSum[i] += plot.getArea();
            }
        }

        if (weightSum != null) {
            effectiveGrowingSystemValues.put(growingSystem, newArray(weightedValueSum.length, 0.0));
            for (int i = 0; i < weightedValueSum.length; i++) {
                double plotValue = weightedValueSum[i] / weightSum[i];
                writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                        domain, growingSystem, plotValue);
                
                effectiveGrowingSystemValues.get(growingSystem)[i] = plotValue;
            }
        }
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#resetEffective(fr.inra.agrosyst.api.entities.Domain, fr.inra.agrosyst.api.entities.GrowingSystem)
     */
    @Override
    public void resetEffective(Domain domain, GrowingSystem growingSystem) {
        // cleanup for next iteration
        effectiveCroppingValues.clear();
        // cleanup for next iteration
        effectivePlotValues.clear();
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#computeEffective(fr.inra.agrosyst.services.performance.IndicatorWriter, fr.inra.agrosyst.api.entities.Domain)
     */
    @Override
    public void computeEffective(IndicatorWriter writer, Domain domain) {
        Double[] weightedValueSum = null;
        Double[] weightSum = null;

        // moyenne pondérée sur la surface
        for (Map.Entry<GrowingSystem, Double[]> entry : effectiveGrowingSystemValues.entrySet()) {
            GrowingSystem growingSystem = entry.getKey();
            Double[] values = entry.getValue();

            if (growingSystem.getAffectedAreaRate() != null) {
                if (weightedValueSum == null) {
                    weightedValueSum = newArray(values.length, 0.0);
                    weightSum = newArray(values.length, 0.0);
                }
                for (int i = 0; i < values.length; i++) {
                    weightedValueSum[i] += values[i] * growingSystem.getAffectedAreaRate();
                    weightSum[i] += growingSystem.getAffectedAreaRate();
                }
            }
        }

        if (weightSum != null) {
            for (int i = 0; i < weightedValueSum.length; i++) {
                if (weightSum[i] != 0) {
                    double domainValue = weightedValueSum[i] / weightSum[i];
                    writer.writeEffective(getIndicatorCategory(), getIndicatorLabel(i), domain.getCampaign(),
                            domain, domainValue);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't compute growing system scale with 0 weigth");
                    }
                }
            }
        }
    }

    /* (non-Javadoc)
     * @see fr.inra.agrosyst.services.performance.indicators.Indicator#resetEffective(fr.inra.agrosyst.api.entities.Domain)
     */
    @Override
    public void resetEffective(Domain domain) {
        // cleanup for next iteration
        effectiveGrowingSystemValues.clear();
    }

    public abstract Double[] manageIntervention(EffectiveIntervention intervention, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveCropCyclePhase phase);

}
