package fr.inra.agrosyst.services.performance;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PerformanceServiceImpl.java 4855 2015-03-23 17:56:42Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/PerformanceServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.performance.PerformanceFile;
import fr.inra.agrosyst.api.entities.performance.PerformanceFileTopiaDao;
import fr.inra.agrosyst.api.entities.performance.PerformanceState;
import fr.inra.agrosyst.api.entities.performance.PerformanceTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.domain.PlotDto;
import fr.inra.agrosyst.api.services.domain.ZoneDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.performance.PerformanceDto;
import fr.inra.agrosyst.api.services.performance.PerformanceFilter;
import fr.inra.agrosyst.api.services.performance.PerformanceService;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.ServiceContext;
import fr.inra.agrosyst.services.performance.indicators.Indicator;
import fr.inra.agrosyst.services.performance.indicators.IndicatorIFT;
import fr.inra.agrosyst.services.performance.indicators.IndicatorSurfaceUTH;
import fr.inra.agrosyst.services.performance.indicators.IndicatorTransitCount;
import fr.inra.agrosyst.services.performance.indicators.IndicatorWorkTime;
import fr.inra.agrosyst.services.security.SecurityContext;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Hibernate;
import org.hibernate.Session;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class PerformanceServiceImpl extends AbstractAgrosystService implements PerformanceService {

    protected BusinessAuthorizationService authorizationService;
    protected AnonymizeService anonymizeService;
    protected DomainService domainService;

    protected PerformanceTopiaDao performanceDao;
    protected PerformanceFileTopiaDao performanceFileTopiaDao;
    protected GrowingSystemTopiaDao growingSystemDao;
    protected DomainTopiaDao domainDao;
    protected GrowingSystemTopiaDao growingSystemTopiaDao;
    protected PlotTopiaDao plotTopiaDao;
    protected ZoneTopiaDao zoneTopiaDao;
    protected AgrosystUserTopiaDao userDao;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setPerformanceDao(PerformanceTopiaDao performanceDao) {
        this.performanceDao = performanceDao;
    }

    public void setPerformanceFileTopiaDao(PerformanceFileTopiaDao performanceFileTopiaDao) {
        this.performanceFileTopiaDao = performanceFileTopiaDao;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    public void setGrowingSystemTopiaDao(GrowingSystemTopiaDao growingSystemTopiaDao) {
        this.growingSystemTopiaDao = growingSystemTopiaDao;
    }

    public void setZoneTopiaDao(ZoneTopiaDao zoneTopiaDao) {
        this.zoneTopiaDao = zoneTopiaDao;
    }

    public void setPlotTopiaDao(PlotTopiaDao plotTopiaDao) {
        this.plotTopiaDao = plotTopiaDao;
    }

    public void setUserDao(AgrosystUserTopiaDao userDao) {
        this.userDao = userDao;
    }

    @Override
    public ResultList<PerformanceDto> getFilteredPerformances(PerformanceFilter performanceFilter) {
        ResultList<Performance> performances = performanceDao.getFilteredPerformances(performanceFilter, getSecurityContext());
        ResultList<PerformanceDto> result = ResultList.transform(performances, anonymizeService.getPerformanceToDtoFunction());
        return result;
    }

    @Override
    public Performance getPerformance(String performanceTopiaId) {
        Performance result;
        if (StringUtils.isBlank(performanceTopiaId)) {
            result = performanceDao.newInstance();
        } else {
            authorizationService.checkPerformanceReadable(performanceTopiaId);
            result = performanceDao.forTopiaIdEquals(performanceTopiaId).findUnique();
        }
        return result;
    }

    @Override
    public List<PlotDto> getPlots(List<String> domainIds) {
        List<Plot> plots = plotTopiaDao.newQueryBuilder().addTopiaIdIn(Plot.PROPERTY_DOMAIN, domainIds).setOrderByArguments(Plot.PROPERTY_NAME).findAll();
        List<PlotDto> result = Lists.transform(plots, anonymizeService.getPlotToDtoFunction());
        return result;
    }

    @Override
    public List<ZoneDto> getZones(List<String> plotIds) {
        List<Zone> zones = zoneTopiaDao.newQueryBuilder().addTopiaIdIn(Zone.PROPERTY_PLOT, plotIds).setOrderByArguments(Zone.PROPERTY_NAME).findAll();
        List<ZoneDto> result = Lists.newArrayList(Iterables.transform(zones, anonymizeService.getZoneToDtoFunction(false)));
        return result;
    }

    @Override
    public Performance createOrUpdatePerformance(Performance performance, List<String> domainIds,
                                                 List<String> growingSystemIds, List<String> plotIds,
                                                 List<String> zoneIds) {
        return createOrUpdatePerformance(performance, domainIds, growingSystemIds, plotIds, zoneIds, true);
    }

    public Performance createOrUpdatePerformance(Performance performance, List<String> domainIds,
                                                 List<String> growingSystemIds, List<String> plotIds,
                                                 List<String> zoneIds, boolean runComputeThread) {

        authorizationService.checkCreateOrUpdatePerformance(performance.getTopiaId());

        Collection<Domain> domains;
        if (CollectionUtils.isNotEmpty(domainIds)) {
            domains = domainDao.forTopiaIdIn(domainIds).findAll();
        } else {
            domains = null;
        }
        performance.setDomains(domains);

        Collection<GrowingSystem> growingSystems;
        if (CollectionUtils.isNotEmpty(growingSystemIds)) {
            growingSystems = growingSystemDao.forTopiaIdIn(growingSystemIds).findAll();
        } else {
            growingSystems = null;
        }
        performance.setGrowingSystems(growingSystems);

        Collection<Plot> plots;
        if (CollectionUtils.isNotEmpty(plotIds)) {
            plots = plotTopiaDao.forTopiaIdIn(plotIds).findAll();
        } else {
            plots = null;
        }
        performance.setPlots(plots);

        Collection<Zone> zones;
        if (CollectionUtils.isNotEmpty(zoneIds)) {
            zones = zoneTopiaDao.forTopiaIdIn(zoneIds).findAll();
        } else {
            zones = null;
        }
        performance.setZones(zones);

        performance.setUpdateDate(context.getCurrentDate());
        performance.setComputeStatus(PerformanceState.GENERATING);
        
        if (performance.isPersisted()) {
            performance = performanceDao.update(performance);
        } else {
            String userId = getSecurityContext().getUserId();
            AgrosystUser user = userDao.forTopiaIdEquals(userId).findUnique();
            performance.setAuthor(user);
            performance = performanceDao.create(performance);
        }

        getTransaction().commit();

        if (runComputeThread) {
            // start a background thread for performance generation
            ServiceContext threadServiceContext = context.newServiceContext();
            PerformanceThread thread = new PerformanceThread(threadServiceContext, performance.getTopiaId());
            thread.start();
        }

        return performance;
    }

    @Override
    public void generatePerformanceFile(String performanceId) {

        // get performance
        Performance performance = performanceDao.forTopiaIdEquals(performanceId).findUnique();
        
        // build writer
        ByteArrayOutputStream writerStream = new ByteArrayOutputStream(4096);
        IndicatorWriter writer = new HssfIndicatorWriter(performance, writerStream);
        writer.init();

        // compute and output to writer
        convertToWriter(performance, writer);
        
        // write file
        InputStream inputStream;
        try {
            writer.finish();
            writerStream.close();
            inputStream = new ByteArrayInputStream(writerStream.toByteArray());

            // get performance file to create or update
            PerformanceFile performanceFile = performanceFileTopiaDao.forPerformanceEquals(performance).findUniqueOrNull();
            if (performanceFile == null) {
                performanceFile = performanceFileTopiaDao.newInstance();
                performanceFile.setPerformance(performance);
            }

            // store binary stream into database
            Session hibernateSession = getPersistenceContext().getHibernateSupport().getHibernateSession();
            Blob blob = Hibernate.getLobCreator(hibernateSession).createBlob(inputStream, inputStream.available());
            performanceFile.setContent(blob);
            

            // persist performance file
            if (performanceFile.isPersisted()) {
                performanceFileTopiaDao.update(performanceFile);
            } else {
                performanceFileTopiaDao.create(performanceFile);
            }

            // update status and commit
            performance.setComputeStatus(PerformanceState.SUCCESS);
            performanceDao.update(performance);
            getTransaction().commit();
        } catch (IOException ex) {
            // update status to failed
            performance.setComputeStatus(PerformanceState.FAILED);
            performanceDao.update(performance);
            getTransaction().commit();

            throw new AgrosystTechnicalException("Can't convert workbook to stream", ex);
        }
    }

    @Override
    public InputStream downloadPerformances(String performanceId) {

        Performance performance = performanceDao.forTopiaIdEquals(performanceId).findUnique();
        PerformanceFile performanceFile = performanceFileTopiaDao.forPerformanceEquals(performance).findUniqueOrNull();

        // write file
        InputStream inputStream;
        try {
            inputStream = performanceFile.getContent().getBinaryStream();
        } catch (SQLException ex) {
            throw new AgrosystTechnicalException("Can't get performance file stream", ex);
        }
        return inputStream;
    }

    /**
     * Compute performances indicators with all available indicators.
     * 
     * @param performance performance
     * @param writer writer
     */
    protected void convertToWriter(Performance performance, IndicatorWriter writer) {

        // build indicator list to compute
        List<Indicator> indicators = Lists.newArrayList();
        indicators.add(newInstance(IndicatorWorkTime.class));
        indicators.add(newInstance(IndicatorSurfaceUTH.class));
        indicators.add(newInstance(IndicatorTransitCount.class));
        // refs #5731 Certaines valeurs d'indicateurs sont erronées,
        // dans l'export des performances du réalisé et du synthétisé, il faudrait bloquer l'affichage des indicateurs suivants:
        //indicators.add(newInstance(IndicatorFuelConsumption.class));
        //indicators.add(newInstance(IndicatorYield.class));
        //indicators.add(newInstance(IndicatorIPhy.class));
        // refs #5731 Dans l'export des performances du synthétisé, il faudrait bloquer l'affichage des indicateurs suivants: les différents IFT
        // TODO DCossé 11/09/14 désactiver si le calcul n'est pas correct d'ici la fin de l'itération
        indicators.add(newInstance(IndicatorIFT.class));

        convertToWriter(performance, writer, indicators.toArray(new Indicator[indicators.size()]));
    }

    /**
     * Compute performances indicators with given indicators.
     * 
     * @param performance performance
     * @param writer writer
     * @param indicators indicators
     */
    public void convertToWriter(Performance performance, IndicatorWriter writer, Indicator... indicators) {

        // execute performance computing with performance author
        SecurityContext securityContext = getSecurityContextAsUser(performance.getAuthor().getTopiaId());

        // get domain again for security and anonymization
        Collection<Domain> domains = performance.getDomains();
        domains = anonymizeService.checkForDomainsAnonymization(domains);

        if (performance.isPracticed()) {
            if (CollectionUtils.isNotEmpty(domains)) {

                // filtre commun pour tous les domaines
                GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
                growingSystemFilter.setActive(true);
                growingSystemFilter.setAllPageSize();

                for (Domain domain : performance.getDomains()) {

                    // pour un domaine, on récupére l'ensemble des systèmes de cultures
                    // visibles par un utilisateur (sécuritycontext) et pour le domaine courant
                    NavigationContext navigationContext = new NavigationContext();
                    navigationContext.setDomains(Collections.singleton(domain.getTopiaId()));
                    growingSystemFilter.setNavigationContext(navigationContext);
                    List<GrowingSystem> growingSystems = growingSystemTopiaDao.getFilteredGrowingSystems(growingSystemFilter, securityContext).getElements();

                    // on détermine ensuite si l'utilisateur a demande specifiquement la génération
                    // des données pour certains GS en particulier
                    Collection<GrowingSystem> askedGrowingSystems = CollectionUtils.retainAll(growingSystems, CollectionUtils.emptyIfNull(performance.getGrowingSystems()));

                    // on génére soit les GS demandé, soit tous ceux du domaine
                    Collection<GrowingSystem> computedGrowingSystems = CollectionUtils.isNotEmpty(askedGrowingSystems) ? askedGrowingSystems : growingSystems;
                    for (GrowingSystem growingSystem : computedGrowingSystems) {
                        GrowingSystem anonGrowingSystem = anonymizeService.checkForGrowingSystemAnonymization(growingSystem);
                        for (Indicator indicator : indicators) {
                            indicator.computePracticed(writer, anonGrowingSystem);
                        }
                    }
                    // et si on a généré tous ceux du domaine, on réalisé en plus une mise à l'echelle
                    // (n'a de sens que pour toutes les données)
                    if (CollectionUtils.isEmpty(askedGrowingSystems)) {
                        for (Indicator indicator : indicators) {
                            indicator.computePracticed(writer, domain);
                        }
                    }
                    
                    // reset
                    for (Indicator indicator : indicators) {
                        indicator.resetPracticed(domain);
                    }
                }
            }
        } else {

            if (CollectionUtils.isNotEmpty(domains)) {

                // filtre commun pour tous les domaines (etend GrowingSystemFilter et PlotFilter
                // donc peut être utilisé pour tous les appels)
                GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
                growingSystemFilter.setActive(true);
                growingSystemFilter.setAllPageSize();

                for (Domain domain : domains) {

                    // pour un domaine, on récupére l'ensemble des systèmes de cultures
                    // visibles par un utilisateur (sécuritycontext) et pour le domaine courant
                    NavigationContext navigationContext = new NavigationContext();
                    navigationContext.setDomains(Collections.singleton(domain.getTopiaId()));
                    growingSystemFilter.setNavigationContext(navigationContext);
                    List<GrowingSystem> growingSystems = growingSystemTopiaDao.getFilteredGrowingSystems(growingSystemFilter, securityContext).getElements();

                    // on détermine ensuite si l'utilisateur a demande specifiquement la génération
                    // des données pour certains GS en particulier
                    Collection<GrowingSystem> askedGrowingSystems = CollectionUtils.retainAll(growingSystems, CollectionUtils.emptyIfNull(performance.getGrowingSystems()));

                    // on génére soit les GS demandé, soit tous ceux du domaine
                    Collection<GrowingSystem> computedGrowingSystems = CollectionUtils.isNotEmpty(askedGrowingSystems) ? askedGrowingSystems : growingSystems;
                    for (GrowingSystem growingSystem : computedGrowingSystems) {

                        GrowingSystem anonGrowingSystem = anonymizeService.checkForGrowingSystemAnonymization(growingSystem);

                        // on récupere les parcelles du système de cultures (sans sécurité)
                        Collection<Plot> plots = plotTopiaDao.forProperties(
                                Plot.PROPERTY_GROWING_SYSTEM, growingSystem,
                                Plot.PROPERTY_ACTIVE, true).findAll();

                        // on détermine ensuite si l'utilisateur a demande specifiquement la génération
                        // des données pour certaines parcelles en particulier
                        Collection<Plot> askedPlots = CollectionUtils.retainAll(plots, CollectionUtils.emptyIfNull(performance.getPlots()));

                        // on génére soit les Plot demandé, soit tous ceux du GS
                        Collection<Plot> computedPlots = CollectionUtils.isNotEmpty(askedPlots) ? askedPlots : plots;
                        for (Plot plot : computedPlots) {

                            Plot anonPlot = anonymizeService.checkForPlotAnonymization(plot);

                            // on récupere les zones de la parcelle (sans sécurité)
                            Collection<Zone> zones = zoneTopiaDao.forProperties(
                                    Zone.PROPERTY_PLOT, plot,
                                    Zone.PROPERTY_ACTIVE, true).findAll();

                            // on détermine ensuite si l'utilisateur a demande specifiquement la génération
                            // des données pour certaines zones en particulier
                            Collection<Zone> askedZones = CollectionUtils.retainAll(zones, CollectionUtils.emptyIfNull(performance.getZones()));

                            // on génére soit les zones demandées, soit tous ceux du GS
                            Collection<Zone> computedZones = CollectionUtils.isNotEmpty(askedZones) ? askedZones : zones;
                            for (Zone zone : computedZones) {
                                Zone anonZone = anonymizeService.checkForZoneAnonymization(zone);
                                for (Indicator indicator : indicators) {
                                    indicator.computeEffective(writer, domain, anonGrowingSystem, anonPlot, anonZone);
                                }
                            }

                            // et si on a généré tous ceux de la parcelle, on réalisé en plus une mise à l'echelle
                            // (n'a de sens que pour toutes les données)
                            if (CollectionUtils.isEmpty(askedZones)) {
                                for (Indicator indicator : indicators) {
                                    indicator.computeEffective(writer, domain, anonGrowingSystem, anonPlot);
                                }
                            }

                            // reset
                            for (Indicator indicator : indicators) {
                                indicator.resetEffective(domain, anonGrowingSystem, anonPlot);
                            }
                        }

                        // et si on a généré toutes les parcelles ceux du GS, on réalisé en plus une mise à l'echelle
                        // (n'a de sens que pour toutes les données)
                        if (CollectionUtils.isEmpty(askedPlots)) {
                            for (Indicator indicator : indicators) {
                                indicator.computeEffective(writer, domain, anonGrowingSystem);
                            }
                        }

                        // reset
                        for (Indicator indicator : indicators) {
                            indicator.resetEffective(domain, anonGrowingSystem);
                        }
                    }

                    // et si on a généré tous ceux du domaine, on réalisé en plus une mise à l'echelle
                    // (n'a de sens que pour toutes les données)
                    if (CollectionUtils.isEmpty(askedGrowingSystems)) {
                        for (Indicator indicator : indicators) {
                            indicator.computeEffective(writer, domain);
                        }
                    }
                    
                    // reset
                    for (Indicator indicator : indicators) {
                        indicator.resetEffective(domain);
                    }
                }
            }
        }
    }

    @Override
    public void deletePerformance(List<String> performanceTopiaIds) {
        List<Performance> performance = performanceDao.forTopiaIdIn(performanceTopiaIds).findAll();
        List<PerformanceFile> performanceFiles = performanceFileTopiaDao.forPerformanceIn(performance).findAll();
        performanceFileTopiaDao.deleteAll(performanceFiles);
        performanceDao.deleteAll(performance);
        getTransaction().commit();
    }
}
