package fr.inra.agrosyst.services.performance;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: HssfIndicatorWriter.java 4324 2014-09-11 13:06:43Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/performance/HssfIndicatorWriter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Row;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCyclePhase;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;

/**
 * Writer abstrait que les indicateurs peuvent utiliser qui délègue l'ecriture au feuille de calcul
 * excel.
 * 
 * Les onglets sont composés ainsi en réalisé:
 * <ul>
 * <li>Domaine_Exploitation</li>
 * <li>Système de culture</li>
 * <li>Culture &amp; précédent</li>
 * <li>Parcelle</li>
 * <li>Zone</li>
 * <li>Intervention</li>
 * </ul>
 * 
 * Et en pratiqué:
 * <ul>
 * <li>Domaine_Exploitation</li>
 * <li>Système de culture</li>
 * <li>Culture &amp; précédent</li>
 * <li>Intervention</li>
 * </ul>
 * 
 * @author Eric Chatellier
 */
public class HssfIndicatorWriter implements IndicatorWriter {

    protected Performance performance;
    protected OutputStream output;
    protected HSSFWorkbook workbook;
    protected HSSFSheet domainSheet;
    protected HSSFSheet growingSystemSheet;
    protected HSSFSheet practicedSystemSheet;
    protected HSSFSheet plotSystemSheet;
    protected HSSFSheet zoneSystemSheet;
    protected HSSFSheet croppingPlanSheet;
    protected HSSFSheet interventionSheet;

    /** XXX: Intervention type hard coded i18n. */
    protected static final Map<AgrosystInterventionType, String> INTERVENTION_TYPE_I18N = Maps.newHashMap();

    /** XXX: Crop cycle type hard coded i18n. */
    protected static final Map<CropCyclePhaseType, String> CROP_CYCLE_PHASE_TYPE_I18N = Maps.newHashMap();

    static {
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.RECOLTE, "Récolte");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.SEMIS, "Semis");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES, "Application de produits phytosanitaires");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX, "Application de produits minéraux");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.EPANDAGES_ORGANIQUES, "Épandage organique");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.IRRIGATION, "Irrigation");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.LUTTE_BIOLOGIQUE, "Lutte biologique");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.TRAVAIL_DU_SOL, "Travail du sol");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.ENTRETIEN_TAILLE_VIGNE_ET_VERGER, "Entretien/Taille de vigne et verger");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.AUTRE, "Autre");
        INTERVENTION_TYPE_I18N.put(AgrosystInterventionType.TRANSPORT, "Transport");

        CROP_CYCLE_PHASE_TYPE_I18N.put(CropCyclePhaseType.INSTALLATION, "Installation");
        CROP_CYCLE_PHASE_TYPE_I18N.put(CropCyclePhaseType.MONTEE_EN_PRODUCTION, "Montée en production");
        CROP_CYCLE_PHASE_TYPE_I18N.put(CropCyclePhaseType.PLEINE_PRODUCTION, "Pleine production");
        CROP_CYCLE_PHASE_TYPE_I18N.put(CropCyclePhaseType.DECLIN_DE_PRODUCTION, "Déclin de production");
    }

    public HssfIndicatorWriter(Performance performance, OutputStream output) {
        this.performance = performance;
        this.output = output;
        workbook = new HSSFWorkbook();
    }

    public void init() {
        if (performance.isPracticed()) {
            initPracticed();
        } else {
            initEffective();
        }
    }

    protected void initEffective() {
        domainSheet = workbook.createSheet("Domaine_Exploitation");
        addHeaders(workbook, domainSheet, "Domaine_Exploitation",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        growingSystemSheet = workbook.createSheet("Systèmes de cultures");
        addHeaders(workbook, growingSystemSheet, "Domaine_Exploitation", "Systèmes de culture",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        croppingPlanSheet = workbook.createSheet("Culture & précédent-phase");
        addHeaders(workbook, croppingPlanSheet, "Domaine_Exploitation", "Systèmes de culture", "Culture", "Précédent/Phase",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        plotSystemSheet = workbook.createSheet("Parcelles");
        addHeaders(workbook, plotSystemSheet, "Domaine_Exploitation", "Systèmes de culture", "Parcelle",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        zoneSystemSheet = workbook.createSheet("Zones");
        addHeaders(workbook, zoneSystemSheet, "Domaine_Exploitation", "Systèmes de culture", "Parcelle", "Zone",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        interventionSheet = workbook.createSheet("Interventions");
        addHeaders(workbook, interventionSheet, "Domaine_Exploitation", "Systèmes de culture", "Parcelle", "Zone", "Culture", "Précédent/Phase",
                "Type d'intervention", "Intervention", "Action(s)", "Intrant(s)", "Cible(s) de traitement",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");
    }

    protected void initPracticed() {
        domainSheet = workbook.createSheet("Domaine_Exploitation");
        addHeaders(workbook, domainSheet, "Domaine_Exploitation",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        practicedSystemSheet = workbook.createSheet("Système synthétisé");
        addHeaders(workbook, practicedSystemSheet, "Domaine_Exploitation", "Systèmes de culture", "Système synthétisé",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        croppingPlanSheet = workbook.createSheet("Culture & précédent-phase");
        addHeaders(workbook, croppingPlanSheet, "Domaine_Exploitation", "Systèmes de culture", "Système synthétisé", "Culture", "Précédent/Phase",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");

        interventionSheet = workbook.createSheet("Interventions");
        addHeaders(workbook, interventionSheet, "Domaine_Exploitation", "Systèmes de culture", "Système synthétisé", "Culture", "Précédent/Phase",
                "Type d'intervention", "Intervention", "Action(s)", "Intrant(s)", "Cible(s) de traitement",
                "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque");
    }

    /**
     * Write header line (bold font) in given SpeadSheed.
     * 
     * @param workbook workbook
     * @param sheet sheet
     * @param headers headers
     */
    protected void addHeaders(HSSFWorkbook workbook, HSSFSheet sheet, String... headers) {
        Row row0 = sheet.createRow(0);
        CellStyle style = workbook.createCellStyle(); // bold
        HSSFFont font = workbook.createFont();
        font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
        style.setFont(font);
        int header = 0;
        for (String displayName : headers) {
            Cell cell0 = row0.createCell(header++);
            cell0.setCellValue(displayName);
            cell0.setCellStyle(style);
        }
    }

    public void finish() {
        // autosize columns
        for (int s = 0; s < workbook.getNumberOfSheets(); s++) {
            HSSFSheet sheet = workbook.getSheetAt(s);
            HSSFRow row = sheet.getRow(0);
            for (int column = 0; column < row.getPhysicalNumberOfCells(); column++) {
                sheet.autoSizeColumn(column);
            }
        }

        // dump workbook to stream
        try {
            workbook.write(output);
        } catch (IOException ex) {
            throw new AgrosystTechnicalException("Can't write to output stream", ex);
        }
    }

    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, Double value) {
        writeToSheet(domainSheet, indicatorCategory, indicatorName, value, "", domain.getName(), campaigns);
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, Double value) {
        writeToSheet(domainSheet, indicatorCategory, indicatorName, value, "", domain.getName(), String.valueOf(campaign));
    }

    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, GrowingSystem growingSystem, PracticedSystem practicedSystem, Double value) {
        writeToSheet(practicedSystemSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(), practicedSystem.getName(), campaigns);
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem, Double value) {
        writeToSheet(growingSystemSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(), String.valueOf(campaign));
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem, Plot plot, Double value) {
        writeToSheet(plotSystemSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                plot.getName(), String.valueOf(campaign));
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone, Double value) {
        writeToSheet(zoneSystemSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                plot.getName(), zone.getName(), String.valueOf(campaign));
    }

    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, GrowingSystem growingSystem, PracticedSystem practicedSystem,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, Double value) {
        writeToSheet(croppingPlanSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                practicedSystem.getName(), croppingPlanEntry.getName(), previousPlanEntry != null ? previousPlanEntry.getName() : "", campaigns);
    }
    
    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, GrowingSystem growingSystem, PracticedSystem practicedSystem,
            CroppingPlanEntry croppingPlanEntry, PracticedCropCyclePhase phase, Double value) {
        writeToSheet(croppingPlanSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                practicedSystem.getName(), croppingPlanEntry.getName(), CROP_CYCLE_PHASE_TYPE_I18N.get(phase.getType()), campaigns);
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, Double value) {
        writeToSheet(croppingPlanSheet, indicatorCategory, indicatorName, value, "", domain.getName(),
                growingSystem.getName(), croppingPlanEntry != null ? croppingPlanEntry.getName() : "",
                previousPlanEntry.getName(), String.valueOf(campaign));
    }
    
    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem,
            CroppingPlanEntry croppingPlanEntry, EffectiveCropCyclePhase phase, Double value) {
        writeToSheet(croppingPlanSheet, indicatorCategory, indicatorName, value, "", domain.getName(),
                growingSystem.getName(), croppingPlanEntry != null ? croppingPlanEntry.getName() : "",
                phase != null ? CROP_CYCLE_PHASE_TYPE_I18N.get(phase.getType()) : "", String.valueOf(campaign));
    }

    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, GrowingSystem growingSystem, PracticedSystem practicedSystem,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, PracticedIntervention intervention,
            Collection<AbstractAction> actions, Collection<AbstractInput> inputs, Double value) {
        writeToSheet(interventionSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(), practicedSystem.getName(), croppingPlanEntry.getName(),
                previousPlanEntry.getName(), INTERVENTION_TYPE_I18N.get(intervention.getType()), intervention.getName(),
                getActionsToString(actions), getInputsToString(inputs), getTargetsToString(inputs), campaigns);
    }
    
    @Override
    public void writePracticed(String indicatorCategory, String indicatorName, String campaigns, Domain domain, GrowingSystem growingSystem, PracticedSystem practicedSystem,
            CroppingPlanEntry croppingPlanEntry, PracticedCropCyclePhase phase, PracticedIntervention intervention,
            Collection<AbstractAction> actions, Collection<AbstractInput> inputs, Double value) {
        writeToSheet(interventionSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(), practicedSystem.getName(), croppingPlanEntry.getName(),
                CROP_CYCLE_PHASE_TYPE_I18N.get(phase.getType()), INTERVENTION_TYPE_I18N.get(intervention.getType()), intervention.getName(),
                getActionsToString(actions), getInputsToString(inputs), getTargetsToString(inputs), campaigns);
    }

    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone,
            CroppingPlanEntry croppingPlanEntry, CroppingPlanEntry previousPlanEntry, EffectiveIntervention intervention,
            Collection<AbstractAction> actions, Collection<AbstractInput> inputs, Double value) {
        writeToSheet(interventionSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                plot.getName(), zone.getName(), croppingPlanEntry != null ? croppingPlanEntry.getName() : "",
                previousPlanEntry != null ? previousPlanEntry.getName() : "", INTERVENTION_TYPE_I18N.get(intervention.getType()), intervention.getName(),
                        getActionsToString(actions), getInputsToString(inputs), getTargetsToString(inputs), String.valueOf(campaign));
    }
    
    @Override
    public void writeEffective(String indicatorCategory, String indicatorName, int campaign, Domain domain, GrowingSystem growingSystem, Plot plot, Zone zone,
            CroppingPlanEntry croppingPlanEntry, EffectiveCropCyclePhase phase, EffectiveIntervention intervention,
            Collection<AbstractAction> actions, Collection<AbstractInput> inputs, Double value) {
        writeToSheet(interventionSheet, indicatorCategory, indicatorName, value, "", domain.getName(), growingSystem.getName(),
                plot.getName(), zone.getName(), croppingPlanEntry != null ? croppingPlanEntry.getName() : "",
                phase != null ? CROP_CYCLE_PHASE_TYPE_I18N.get(phase.getType()) : "", INTERVENTION_TYPE_I18N.get(intervention.getType()), intervention.getName(),
                getActionsToString(actions), getInputsToString(inputs), getTargetsToString(inputs), String.valueOf(campaign));
    }

    //"Domaine_Exploitation", "Systèmes de culture", "Parcelle", "Zone", "Année(s)", "Approche de calcul", "Catégorie d'indicateur", "Indicateur", "Valeur", "Remarque"
    protected void writeToSheet(HSSFSheet sheet, String indicatorCategory, String indicatorName, double value, String remarque, String... columns) {
        // new line
        int line = sheet.getPhysicalNumberOfRows();
        Row row = sheet.createRow(line);

        int column = 0;

        // dynamic cell
        for (String dynColumn : columns) {
            Cell cellx = row.createCell(column++);
            cellx.setCellValue(dynColumn);
        }

        // "Approche de calcul"
        if (performance.isPracticed()) {
            Cell cell0 = row.createCell(column++);
            cell0.setCellValue("Synthétisé");
        } else {
            Cell cell0 = row.createCell(column++);
            cell0.setCellValue("Réalisé");
        }

        // indicatorCategory
        Cell cell1 = row.createCell(column++);
        cell1.setCellValue(indicatorCategory);

        // indicatorName
        Cell cell2 = row.createCell(column++);
        cell2.setCellValue(indicatorName);

        // value
        Cell cellLast = row.createCell(column++);
        cellLast.setCellValue(value);

        // remarque
        Cell remCell = row.createCell(column);
        remCell.setCellValue(remarque);
    }

    /**
     * Join actions type into string.
     * 
     * @param actions actions
     * @return to string
     */
    protected String getActionsToString(Collection<AbstractAction> actions) {
        String result;
        if (actions != null) {
            result = Joiner.on(", ").join(Iterables.transform(actions, new Function<AbstractAction, String>() {
                @Override
                public String apply(AbstractAction input) {
                    return INTERVENTION_TYPE_I18N.get(input.getMainAction().getIntervention_agrosyst());
                }
            }));
        } else {
            result = "";
        }
        return result;
    }

    /**
     * Join input name into string.
     * 
     * @param inputs inputs
     * @return to string
     */
    protected String getInputsToString(Collection<AbstractInput> inputs) {
        String result;
        if (inputs != null) {
            result = Joiner.on(", ").join(Iterables.transform(inputs, new Function<AbstractInput, String>() {
                @Override
                public String apply(AbstractInput input) {
                    return getInputProductName(input);
                }
            }));
        } else {
            result = "";
        }
        return result;
    }

    /**
     * Get product name depending on input type.
     * 
     * @return product name
     */
    protected String getInputProductName(AbstractInput input) {
        String result = input.getProductName();
        if (Strings.isNullOrEmpty(result)) {
            switch (input.getInputType()) {
            case SEMIS:
            case LUTTE_BIOLOGIQUE:
            case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                PhytoProductInput phytoProductInput = (PhytoProductInput)input;
                if (phytoProductInput.getPhytoProduct() != null) {
                    result = phytoProductInput.getPhytoProduct().getNom_produit();
                }
                break;

            case EPANDAGES_ORGANIQUES:
                OrganicProductInput organicProductInput = (OrganicProductInput)input;
                result = "N:" + organicProductInput.getN() + ",P2O5:" + organicProductInput.getP2O5() + ",K2O:" + organicProductInput.getK2O();
                break;

            //case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX: // ferti min unifa n'a pas de nom

            default:
                break;
            }
        }

        // dans les autres cas, on affiche N/A
        if (Strings.isNullOrEmpty(result)) {
            result = "N/A";
        }
        return result;
    }

    /**
     * Join inputs targets into string.
     * 
     * @param inputs inputs
     * @return to string
     */
    protected String getTargetsToString(Collection<AbstractInput> inputs) {
        // get all targets from all inputs
        Set<RefBioAgressor> aggregates = Sets.newHashSet();
        if (inputs != null) {
            for (AbstractInput input : inputs) {
                if (input instanceof PhytoProductInput) {
                    PhytoProductInput phytoProductInput = (PhytoProductInput)input;
                    if (phytoProductInput.getTargets() != null) {
                        aggregates.addAll(phytoProductInput.getTargets());
                    }
                }
            }
        }

        // join all found targets
        String result = Joiner.on(", ").join(Iterables.transform(aggregates, new Function<RefBioAgressor, String>() {
            @Override
            public String apply(RefBioAgressor bioAgressor) {
                String result = bioAgressor.getLabel();
                return result;
            }
        }));
        return result;
    }
}
