package fr.inra.agrosyst.services.network;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: NetworkServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/network/NetworkServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkManager;
import fr.inra.agrosyst.api.entities.NetworkManagerTopiaDao;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserImpl;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.common.GrowingSystemsIndicator;
import fr.inra.agrosyst.api.services.network.NetworkConnectionDto;
import fr.inra.agrosyst.api.services.network.NetworkFilter;
import fr.inra.agrosyst.api.services.network.NetworkGraph;
import fr.inra.agrosyst.api.services.network.NetworkIndicators;
import fr.inra.agrosyst.api.services.network.NetworkManagerDto;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.network.SmallNetworkDto;
import fr.inra.agrosyst.api.services.referential.ImportResult;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.services.users.UserDto;
import fr.inra.agrosyst.services.AbstractAgrosystService;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.Import;

import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author cosse
 */
public class NetworkServiceImpl extends AbstractAgrosystService implements NetworkService {

    private static final Log log = LogFactory.getLog(NetworkServiceImpl.class);

    protected static final Comparator<SmallNetworkDto> ALPHABETICAL_NETWORK_COMPARATOR = new Comparator<SmallNetworkDto>() {
        @Override
        public int compare(SmallNetworkDto o1, SmallNetworkDto o2) {
            String label1 = Strings.nullToEmpty(o1.getLabel());
            String label2 = Strings.nullToEmpty(o2.getLabel());
            return label1.toLowerCase().compareTo(label2.toLowerCase());
        }
    };
    protected static final Predicate<Network> ACTIVE_NETWORK = new Predicate<Network>() {
        @Override
        public boolean apply(Network input) {
            return input.isActive();
        }
    };

    protected BusinessAuthorizationService authorizationService;

    protected NetworkTopiaDao networkDao;

    protected NetworkManagerTopiaDao networkManagerDao;

    protected AgrosystUserTopiaDao userDao;

    protected GrowingSystemTopiaDao growingSystemTopiaDao;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setNetworkDao(NetworkTopiaDao networkDao) {
        this.networkDao = networkDao;
    }

    public void setNetworkManagerDao(NetworkManagerTopiaDao networkManagerDao) {
        this.networkManagerDao = networkManagerDao;
    }

    public void setUserDao(AgrosystUserTopiaDao userDao) {
        this.userDao = userDao;
    }

    public void setGrowingSystemTopiaDao(GrowingSystemTopiaDao growingSystemTopiaDao) {
        this.growingSystemTopiaDao = growingSystemTopiaDao;
    }

    @Override
    public ResultList<Network> getFilteredNetworks(NetworkFilter filter) {

        List<NetworkManager> managers = null;
        if (filter != null) {
            String networkManagerName = filter.getNetworkManager();
            if (!Strings.isNullOrEmpty(networkManagerName)) {
                managers = networkManagerDao.getNameFilteredNetworkManager(networkManagerName);
            }
        }

        ResultList<Network> result = networkDao.getFilteredNetworks(filter, managers);

        getTransaction().commit();
        return result;
    }

    @Override
    public Network getNetwork(String networkTopiaId) {
        Network result = networkDao.forTopiaIdEquals(networkTopiaId).findUnique();
        return result;
    }

    @Override
    public Network newNetwork() {
        Network result = networkDao.newInstance();
        // A new network is active
        result.setActive(true);
        // ... with a manager which is the current user
        NetworkManager manager = networkManagerDao.newInstance();
        manager.setFromDate(getContext().getCurrentDate());
        manager.setActive(true);
        String userId = getSecurityContext().getUserId();
        AgrosystUserImpl agrosystUser = new AgrosystUserImpl();
        agrosystUser.setTopiaId(userId);
        manager.setAgrosystUser(agrosystUser);
        result.addManagers(manager);
        return result;
    }

    @Override
    public Network createOrUpdateNetwork(Network network, Collection<NetworkManagerDto> networkManagerDtos, List<String> parentsIds) {

        authorizationService.checkCreateOrUpdateNetwork(network.getTopiaId());

        // Add parents to Network
        if (parentsIds != null) {
            if (StringUtils.isBlank(network.getTopiaId())) {
                List<Network> parents = Lists.newArrayList();
                for (String parentTopiaId : parentsIds) {
                    parents.add(networkDao.forTopiaIdEquals(parentTopiaId).findUnique());
                }
                network.setParents(parents);
            } else {
                Collection<Network> parents = network.getParents();
                List<Network> parentsToKeep = Lists.newArrayList();
                if (parents == null) {
                    parents = Lists.newArrayList();
                    network.setParents(parents);
                }
                Map<String, Network> indexedNetworkParents = Maps.uniqueIndex(parents, Entities.GET_TOPIA_ID);
                for (String parentTopiaId : parentsIds) {
                    Network parent = indexedNetworkParents.get(parentTopiaId);
                    if (parent == null) {
                        parent = networkDao.forTopiaIdEquals(parentTopiaId).findUnique();
                        parents.add(parent);
                    }
                    parentsToKeep.add(parent);
                }
                parents.retainAll(parentsToKeep);
            }
        }

        Collection<NetworkManager> networkManagers;
        // add network managers to the network
        if (network.isPersisted()) {
            networkManagers = network.getManagers();
            // In case the list is persisted null ...
            if (networkManagers == null) {
                networkManagers = Lists.newArrayList();
                network.setManagers(networkManagers);
            }
        } else {
            networkManagers = Lists.newArrayList();
            network.setManagers(networkManagers);
        }

        Map<String, NetworkManager> indexedNetworkManagers = Maps.uniqueIndex(networkManagers, Entities.GET_TOPIA_ID);
        List<NetworkManager> networkManagerToKeep = Lists.newArrayList();

        for (NetworkManagerDto networkManagerDto : networkManagerDtos) {
            UserDto userDto = networkManagerDto.getUser();
            AgrosystUser user = userDao.forTopiaIdEquals(userDto.getTopiaId()).findUnique();

            String networkManagerTopiaId = networkManagerDto.getTopiaId();
            NetworkManager networkManager;
            if (StringUtils.isBlank(networkManagerTopiaId)) {
                networkManager = networkManagerDao.newInstance();
            } else {
                networkManager = indexedNetworkManagers.get(networkManagerTopiaId);
            }
            networkManager.setAgrosystUser(user);
            networkManager.setFromDate(networkManagerDto.getFromDate());
            networkManager.setToDate(networkManagerDto.getToDate());
            networkManager.setActive(networkManagerDto.isActive());

            NetworkManager persistedNetworkManager;
            if (StringUtils.isBlank(networkManager.getTopiaId())) {
                persistedNetworkManager = networkManagerDao.create(networkManager);
                networkManagers.add(persistedNetworkManager);
            } else {
                persistedNetworkManager = networkManagerDao.update(networkManager);
            }
            networkManagerToKeep.add(persistedNetworkManager);
        }
        networkManagers.retainAll(networkManagerToKeep);

        Network result;
        if (StringUtils.isBlank(network.getTopiaId())) {
            network.setActive(true);
            result = networkDao.create(network);

            authorizationService.networkCreated(result);
        } else {
            result = networkDao.update(network);

            authorizationService.networkUpdated(network);
        }

        getTransaction().commit();

        return result;
    }

    @Override
    public NetworkManager newNetworkManager() {

        NetworkManager networkManager = networkManagerDao.newInstance();
        return networkManager;
    }

    @Override
    public LinkedHashMap<String, String> searchNameFilteredActiveNetworks(String research, Integer nbResult,
                                                                          Set<String> exclusions, String selfNetworkId,
                                                                          boolean onNetworkEdition) {

        if (!onNetworkEdition) {
            // It is not allowed to specify a networkId while not on network edition
            Preconditions.checkArgument(Strings.isNullOrEmpty(selfNetworkId));
        }
        LinkedHashMap<String, String> networks = networkDao.getNameFilteredNetworks(research, nbResult, exclusions,
                selfNetworkId, onNetworkEdition);
        return networks;
    }

    @Override
    public Set<String> findNetworksByName(String name, String excludeNetworkId) {
        Iterable<Network> networks = networkDao.findAllByNameWithoutCase(name);
        Iterable<String> networkIds = Iterables.transform(networks, Entities.GET_TOPIA_ID);
        if (!Strings.isNullOrEmpty(excludeNetworkId)) {
            Iterables.removeIf(networkIds, Predicates.equalTo(excludeNetworkId));
        }
        Set<String> result = Sets.newHashSet(networkIds);
        return result;
    }

    @Override
    public NetworkGraph buildFullNetworkGraph() {

        NetworkGraph graph = new NetworkGraph();

        if (getConfig().isDemoModeEnabled()) {
            // find all active networks
            List<Network> allNetworks = networkDao.forActiveEquals(true).findAll();

            // build graph
            for (Network network : allNetworks) {
                graph.addNetwork(network);
            }

            escapeTopiaIds(graph);
            sortLevels(graph);
        }

        return graph;
    }

    protected void sortLevels(NetworkGraph graph) {
        for (List<SmallNetworkDto> smallNetworkDtos : graph.getNetworks()) {
            Collections.sort(smallNetworkDtos, ALPHABETICAL_NETWORK_COMPARATOR);
        }
    }

    protected void escapeTopiaIds(NetworkGraph graph) {
        for (List<SmallNetworkDto> smallNetworkDtos : graph.getNetworks()) {
            for (SmallNetworkDto smallNetworkDto : smallNetworkDtos) {
                smallNetworkDto.setId(Entities.ESCAPE_TOPIA_ID.apply(smallNetworkDto.getId()));
            }
        }
        for (NetworkConnectionDto networkConnectionDto : graph.getConnections()) {
            networkConnectionDto.setSourceId(Entities.ESCAPE_TOPIA_ID.apply(networkConnectionDto.getSourceId()));
            networkConnectionDto.setTargetId(Entities.ESCAPE_TOPIA_ID.apply(networkConnectionDto.getTargetId()));
        }
    }

    @Override
    public NetworkGraph buildNetworkGraph(String fromNetworkId) {

        NetworkGraph graph = new NetworkGraph();
        Network network = getNetwork(fromNetworkId);

        graph.addNetwork(network);

        escapeTopiaIds(graph);
        sortLevels(graph);

        return graph;
    }

    protected void findAllParents(Set<Network> result, Iterable<Network> networks) {
        for (Network network : networks) {
            result.add(network);
            Collection<Network> parents = network.getParents();
            findAllParents(result, parents);
        }
    }

    @Override
    public NetworkGraph buildGrowingSystemAndNetworkGraph(String growingSystemName, Set<String> parentNetworkIds) {

        NetworkGraph graph = new NetworkGraph();

        Set<Network> parents = Sets.newHashSet();

        for (String networkId : parentNetworkIds) {
            Network network = networkDao.forTopiaIdEquals(networkId).findUnique();
            parents.add(network);
        }

        graph.addGrowingSystem(growingSystemName, parents);

        Set<Network> parentsOfParents = Sets.newHashSet();
        findAllParents(parentsOfParents, parents);

        for (Network parentsOfParent : parentsOfParents) {
            graph.addNetwork(parentsOfParent);
        }

        escapeTopiaIds(graph);
        sortLevels(graph);

        return graph;
    }

    @Override
    public void unactivateNetworks(List<String> networkIds, boolean activate) {
        if (CollectionUtils.isNotEmpty(networkIds)) {

            for (String networkId : networkIds) {
                Network network = networkDao.forTopiaIdEquals(networkId).findUnique();
                network.setActive(activate);
                networkDao.update(network);
            }
            getTransaction().commit();
        }
    }

    @Override
    public long getNetworksCount(Boolean active) {
        if (active == null) {
            return networkDao.count();
        }
        return networkDao.forActiveEquals(active).count();
    }

    @Override
    public ImportResult importNetworks(InputStream networkStream) {

        Import<NetworkImportDto> importer = null;
        ImportResult result = new ImportResult();
        try {
            NetworkImportModel model = new NetworkImportModel();
            importer = Import.newImport(model, networkStream);

            Map<String, Network> networkCache = Maps.newHashMap();
            for (NetworkImportDto dto : importer) {

                Network network = networkDao.forNameEquals(dto.getName()).findAnyOrNull();
                if (network != null) {
                    // FIXME echatellier 20140131 manage already existing networks
                    result.incIgnored();
                } else {
                    // find responsible by email
                    AgrosystUser agrosystUser = userDao.forEmailEquals(dto.getResponsible()).findUniqueOrNull();

                    if (agrosystUser == null) {
                        String errorMessage = "Pas d'utilisateur trouvé avec l'email : " + dto.getResponsible();
                        result.addError(errorMessage);
                        if (log.isWarnEnabled()) {
                            log.warn(errorMessage);
                        }
                    } else {

                        // create network
                        network = networkDao.newInstance();
                        network.setName(dto.getName());
                        network.setActive(true);

                        // create network manager
                        NetworkManager manager = networkManagerDao.create(
                                NetworkManager.PROPERTY_AGROSYST_USER, agrosystUser,
                                NetworkManager.PROPERTY_ACTIVE, true);
                        network.addManagers(manager);

                        // set parents
                        String parents = dto.getParentNetworks();
                        if (parents != null) {
                            Iterable<String> parentNames = Splitter.on(',').omitEmptyStrings().trimResults().split(parents);
                            for (String parentName : parentNames) {
                                Network parent = networkCache.get(parentName);
                                if (parent == null) {
                                    parent = networkDao.forNameEquals(parentName).findAnyOrNull();
                                }
                                if (parent != null) {
                                    network.addParents(parent);
                                }
                            }
                        }

                        // persist
                        network = networkDao.create(network);
                        networkCache.put(network.getName(), network);

                        // promote responsible with security
                        authorizationService.networkCreated(network);

                        result.incCreated();
                    }
                }
            }

            getTransaction().commit();
        } finally {
            IOUtils.closeQuietly(importer);
        }

        return result;
    }

    @Override
    public NetworkIndicators getIndicators(String networkId) {
        NetworkIndicators result = new NetworkIndicators();
        if (!Strings.isNullOrEmpty(networkId)) {
            ImmutableSet<String> set = ImmutableSet.of(networkId);

            result.setDomainsCount(networkDao.getProjectionHelper().networksToDomains(set).size());
            result.setActiveDomainsCount(networkDao.getProjectionHelper().networksToDomains(set, true).size());

            result.setGrowingPlansCount(networkDao.getProjectionHelper().networksToGrowingPlans(set).size());
            result.setActiveGrowingPlansCount(networkDao.getProjectionHelper().networksToGrowingPlans(set, true).size());

            List<GrowingSystemsIndicator> gsIndicators = growingSystemTopiaDao.networksToGrowingSystemIndicators(set);
            result.setGrowingSystems(gsIndicators);

            Set<Network> subNetworks = networkDao.loadNetworksWithDescendantHierarchy(set);
            Iterables.removeIf(subNetworks, Predicates.compose(Predicates.equalTo(networkId), Entities.GET_TOPIA_ID));
            result.setSubNetworksCount(subNetworks.size());
            long activeSubNetworksCount = Iterables.size(Iterables.filter(subNetworks, ACTIVE_NETWORK));
            result.setActiveSubNetworksCount(activeSubNetworksCount);
        }

        return result;
    }

    @Override
    public List<Network> getNteworkWithName(String networkName) {
        Preconditions.checkNotNull(networkName);
        List<Network> result = networkDao.forNameEquals(networkName).findAll();
        return result;
    }

}
