package fr.inra.agrosyst.services.measurement;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: MeasurementServiceImpl.java 4783 2015-02-11 16:36:22Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/measurement/MeasurementServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.keyvalue.MultiKey;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.VariableType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.measure.HorizonType;
import fr.inra.agrosyst.api.entities.measure.Measure;
import fr.inra.agrosyst.api.entities.measure.MeasureImpl;
import fr.inra.agrosyst.api.entities.measure.MeasureType;
import fr.inra.agrosyst.api.entities.measure.Measurement;
import fr.inra.agrosyst.api.entities.measure.MeasurementSession;
import fr.inra.agrosyst.api.entities.measure.MeasurementSessionImpl;
import fr.inra.agrosyst.api.entities.measure.MeasurementSessionTopiaDao;
import fr.inra.agrosyst.api.entities.measure.MeasurementTopiaDao;
import fr.inra.agrosyst.api.entities.measure.MeasurementType;
import fr.inra.agrosyst.api.entities.measure.NitrogenMolecule;
import fr.inra.agrosyst.api.entities.measure.Observation;
import fr.inra.agrosyst.api.entities.measure.ObservationImpl;
import fr.inra.agrosyst.api.entities.referential.RefActaSubstanceActive;
import fr.inra.agrosyst.api.entities.referential.RefActaSubstanceActiveTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefAdventice;
import fr.inra.agrosyst.api.entities.referential.RefAdventiceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefMesure;
import fr.inra.agrosyst.api.entities.referential.RefMesureTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefNuisibleEDI;
import fr.inra.agrosyst.api.entities.referential.RefNuisibleEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefProtocoleVgObs;
import fr.inra.agrosyst.api.entities.referential.RefProtocoleVgObsTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefStadeEDI;
import fr.inra.agrosyst.api.entities.referential.RefStadeEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefStadeNuisibleEDI;
import fr.inra.agrosyst.api.entities.referential.RefStadeNuisibleEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSupportOrganeEDI;
import fr.inra.agrosyst.api.entities.referential.RefSupportOrganeEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefTypeNotationEDI;
import fr.inra.agrosyst.api.entities.referential.RefTypeNotationEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefUnitesQualifiantEDI;
import fr.inra.agrosyst.api.entities.referential.RefUnitesQualifiantEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefValeurQualitativeEDI;
import fr.inra.agrosyst.api.entities.referential.RefValeurQualitativeEDITopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.measurement.MeasurementService;
import fr.inra.agrosyst.api.services.measurement.ProtocoleVgObsFilter;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.EntityImporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportEntity;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementAdventicesBeanInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementGesInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementMainBeanInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementMeteoInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementNuisibleMaladiesPhysiologiquesAuxBeanInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementPlanteInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementSolInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementStadeCultureBeanInfo;
import fr.inra.agrosyst.services.measurement.export.MeasurementExportMetadata.MeasurementTransferDeSolutesInfo;


public class MeasurementServiceImpl extends AbstractAgrosystService implements MeasurementService {

    protected static final Predicate<Measurement> IS_PLANTE = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.PLANTE;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_SOL = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.SOL;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_TRANSFERT_SOLUTES = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.TRANSFERT_DE_SOLUTES;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_GES = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.GES;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_METEO = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.METEO;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_STADE_CULTURE = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.STADE_CULTURE;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_NUISIBLE_MALADIES_PHYSIOLOGIQUES_AUXILIAIRES = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.NUISIBLE_MALADIES_PHYSIOLOGIQUES_AUXILIAIRES;
            return result;
        }
    };

    protected static final Predicate<Measurement> IS_ADVENTICES = new Predicate<Measurement>() {
        @Override
        public boolean apply(Measurement input) {
            Boolean result = input.getMeasurementType() == MeasurementType.ADVENTICES;
            return result;
        }
    };

    protected AnonymizeService anonymizeService;

    protected MeasurementSessionTopiaDao measurementSessionDao;
    protected ZoneTopiaDao zoneDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryDao;
    protected CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao;
    protected RefMesureTopiaDao refMesureDao;
    protected MeasurementTopiaDao measurementDao;
    protected RefSupportOrganeEDITopiaDao refSupportOrganeEDIDao;
    protected RefActaSubstanceActiveTopiaDao refSubstanceActiveDao;
    protected RefAdventiceTopiaDao refAdventiceDao;
    protected RefStadeEDITopiaDao refStadeEDIDao;
    protected EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleDao;
    protected EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleDao;
    protected RefProtocoleVgObsTopiaDao refProtocoleVgObsDao;
    protected RefNuisibleEDITopiaDao refNuisiblesEdiDao;
    protected RefStadeNuisibleEDITopiaDao refStadeNuisibleEDIDao;
    protected RefTypeNotationEDITopiaDao refTypeNotationEDIDao;
    protected RefUnitesQualifiantEDITopiaDao refUnitesQualifiantEDIDao;
    protected RefValeurQualitativeEDITopiaDao refValeurQualitativeEDIDao;
    protected RefEspeceTopiaDao refEspeceTopiaDao;
    protected RefAdventiceTopiaDao refAdventiceTopiaDao;

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setMeasurementSessionDao(MeasurementSessionTopiaDao measurementSessionDao) {
        this.measurementSessionDao = measurementSessionDao;
    }

    public void setZoneDao(ZoneTopiaDao zoneDao) {
        this.zoneDao = zoneDao;
    }

    public void setCroppingPlanEntryDao(CroppingPlanEntryTopiaDao croppingPlanEntryDao) {
        this.croppingPlanEntryDao = croppingPlanEntryDao;
    }

    public void setCroppingPlanSpeciesTopiaDao(CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao) {
        this.croppingPlanSpeciesTopiaDao = croppingPlanSpeciesTopiaDao;
    }

    public void setRefMesureDao(RefMesureTopiaDao refMesureDao) {
        this.refMesureDao = refMesureDao;
    }

    public void setMeasurementDao(MeasurementTopiaDao measurementDao) {
        this.measurementDao = measurementDao;
    }

    public void setRefSupportOrganeEDIDao(RefSupportOrganeEDITopiaDao refSupportOrganeEDIDao) {
        this.refSupportOrganeEDIDao = refSupportOrganeEDIDao;
    }

    public void setRefSubstanceActiveDao(RefActaSubstanceActiveTopiaDao refSubstanceActiveDao) {
        this.refSubstanceActiveDao = refSubstanceActiveDao;
    }

    public void setRefAdventiceDao(RefAdventiceTopiaDao refAdventiceDao) {
        this.refAdventiceDao = refAdventiceDao;
    }

    public void setRefStadeEDIDao(RefStadeEDITopiaDao refStadeEDIDao) {
        this.refStadeEDIDao = refStadeEDIDao;
    }

    public void setEffectiveSeasonalCropCycleDao(EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleDao) {
        this.effectiveSeasonalCropCycleDao = effectiveSeasonalCropCycleDao;
    }

    public void setRefAdventiceTopiaDao(RefAdventiceTopiaDao refAdventiceTopiaDao) {
        this.refAdventiceTopiaDao = refAdventiceTopiaDao;
    }

    public void setEffectivePerennialCropCycleDao(EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleDao) {
        this.effectivePerennialCropCycleDao = effectivePerennialCropCycleDao;
    }

    public void setRefProtocoleVgObsDao(RefProtocoleVgObsTopiaDao refProtocoleVgObsDao) {
        this.refProtocoleVgObsDao = refProtocoleVgObsDao;
    }

    public void setRefNuisiblesEdiDao(RefNuisibleEDITopiaDao refNuisiblesEdiDao) {
        this.refNuisiblesEdiDao = refNuisiblesEdiDao;
    }

    public void setRefStadeNuisibleEDIDao(RefStadeNuisibleEDITopiaDao refStadeNuisibleEDIDao) {
        this.refStadeNuisibleEDIDao = refStadeNuisibleEDIDao;
    }

    public void setRefTypeNotationEDIDao(RefTypeNotationEDITopiaDao refTypeNotationEDIDao) {
        this.refTypeNotationEDIDao = refTypeNotationEDIDao;
    }

    public void setRefUnitesQualifiantEDIDao(RefUnitesQualifiantEDITopiaDao refUnitesQualifiantEDIDao) {
        this.refUnitesQualifiantEDIDao = refUnitesQualifiantEDIDao;
    }

    public void setRefValeurQualitativeEDIDao(RefValeurQualitativeEDITopiaDao refValeurQualitativeEDIDao) {
        this.refValeurQualitativeEDIDao = refValeurQualitativeEDIDao;
    }

    public void setRefEspeceTopiaDao(RefEspeceTopiaDao refEspeceTopiaDao) {
        this.refEspeceTopiaDao = refEspeceTopiaDao;
    }

    @Override
    public List<MeasurementSession> getZoneMeasurementSessions(Zone zone) {
        return measurementSessionDao.forZoneEquals(zone).setOrderByArguments(MeasurementSession.PROPERTY_START_DATE).findAll();
    }

    @Override
    public void updateMeasurementSessions(Zone zone, Collection<MeasurementSession> inputSessions) {

        // copy binder
        Binder<MeasurementSession, MeasurementSession> sessionBinder = BinderFactory.newBinder(MeasurementSession.class);

        List<MeasurementSession> currentSessions = measurementSessionDao.forZoneEquals(zone).findAll();
        List<MeasurementSession> toDeleteSessions = Lists.newArrayList(currentSessions);
        Map<String, MeasurementSession> idToSession = Maps.uniqueIndex(currentSessions, TopiaEntities.getTopiaIdFunction());
        for (MeasurementSession inputSession : inputSessions) {

            Preconditions.checkNotNull(inputSession.getStartDate());

            MeasurementSession currentSession;
            if (StringUtils.isEmpty(inputSession.getTopiaId())) {
                currentSession = measurementSessionDao.newInstance();
                currentSession.setZone(zone);
            } else {
                currentSession = idToSession.get(inputSession.getTopiaId());
                toDeleteSessions.remove(currentSession);
            }

            // il faut aider topia pour la composition sur une entité abstraite (inverse = true)
            sessionBinder.copyExcluding(inputSession, currentSession,
                    MeasurementSession.PROPERTY_ZONE,
                    MeasurementSession.PROPERTY_MEASUREMENTS);

            if (currentSession.isPersisted()) {
                currentSession = measurementSessionDao.update(currentSession);
            } else {
                currentSession = measurementSessionDao.create(currentSession);
            }

            Collection<Measurement> currentMeasurements = currentSession.getMeasurements();
            if (currentMeasurements == null) {
                currentMeasurements = Lists.newArrayList();
            }
            if (inputSession.getMeasurements() != null) {
                List<Measurement> toDeleteMeasurements = Lists.newArrayList(currentMeasurements);
                Map<String, Measurement> idToMeasurement = Maps.uniqueIndex(currentMeasurements, TopiaEntities.getTopiaIdFunction());
                for (Measurement inputMeasurement : inputSession.getMeasurements()) {

                    // c'est un cas possible si la deserialisation gson n'a pas pu déterminer
                    // le type de mesure dans le cas ou l'utilisateur n'a pas sélectionné
                    // de type de mesure (c'est normalement un cas non permis)
                    if (inputMeasurement == null) {
                        continue;
                    } else {
                        Preconditions.checkNotNull(inputMeasurement.getMeasurementType());
                    }

                    Measurement currentMeasurement;
                    if (StringUtils.isEmpty(inputMeasurement.getTopiaId())) {
                        currentSession.addMeasurements(inputMeasurement);
                        currentMeasurement = measurementDao.create(inputMeasurement);
                    } else {
                        currentMeasurement = idToMeasurement.get(inputMeasurement.getTopiaId());
                        toDeleteMeasurements.remove(currentMeasurement);
                    }

                    // dynamic binder for inheritance
                    Binder<Measurement, Measurement> measurementBinder = (Binder<Measurement, Measurement>)BinderFactory.newBinder(currentMeasurement.getClass());
                    measurementBinder.copyExcluding(inputMeasurement, currentMeasurement,
                            Measurement.PROPERTY_MEASUREMENT_SESSION);

                    if (currentMeasurement.isPersisted()) {
                        measurementDao.update(currentMeasurement);
                    } else {
                        measurementDao.create(currentMeasurement);
                    }
                }
                currentMeasurements.removeAll(toDeleteMeasurements);
                measurementDao.deleteAll(toDeleteMeasurements);
            }
        }
        measurementSessionDao.deleteAll(toDeleteSessions);

        getTransaction().commit();
    }

    @Override
    public Zone getZone(String zoneTopiaId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        Zone result = anonymizeService.checkForZoneAnonymization(zone);
        return result;
    }

    @Override
    public Set<CroppingPlanEntry> getZoneCroppingPlanEntries(Zone zone) {
        Set<CroppingPlanEntry> result = Sets.newHashSet();
        List<EffectiveSeasonalCropCycle> cycleS = effectiveSeasonalCropCycleDao.forZoneEquals(zone).findAll();
        List<EffectivePerennialCropCycle> cycleP = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();
        
        for (EffectiveSeasonalCropCycle cycle : cycleS) {
            if (cycle.getNodes() != null) {
                for (EffectiveCropCycleNode node : cycle.getNodes()) {
                    result.add(node.getCroppingPlanEntry());
                }
            }
        }
        for (EffectivePerennialCropCycle cycle : cycleP) {
            result.add(cycle.getCroppingPlanEntry());
        }
        return result;
    }

    @Override
    public List<VariableType> findAllVariableTypes(MeasurementType measurementType) {
        return refMesureDao.findTypeVariableValues(measurementType);
    }

    @Override
    public List<RefMesure> findAllVariables(MeasurementType measurementType, VariableType variableType) {
        return refMesureDao.findVariables(measurementType, variableType);
    }

    @Override
    public List<RefSupportOrganeEDI> findAllSupportOrganeEDI() {
        List<RefSupportOrganeEDI> result = refSupportOrganeEDIDao.forActiveEquals(true)
                .setOrderByArguments(RefSupportOrganeEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<RefActaSubstanceActive> findDistinctSubstanceActives() {
        List<RefActaSubstanceActive> result = refSubstanceActiveDao.findDistinctSubtanceActive();
        return result;
    }

    @Override
    public List<RefAdventice> findAllAdventices() {
        List<RefAdventice> result = refAdventiceDao.forActiveEquals(true)
                .setOrderByArguments(RefAdventice.PROPERTY_ADVENTICE).findAll();
        return result;
    }

    @Override
    public List<RefStadeEDI> findAllStadeEdi(String cropFamily, String vegetativeProfile) {
        List<RefStadeEDI> result;
        
        if (StringUtils.isNotEmpty(cropFamily)) {
            result = refStadeEDIDao.forFamille_de_cultureEquals(cropFamily).setOrderByArguments(RefStadeEDI.PROPERTY_COLONNE2).findAll();
        } else if (StringUtils.isNumeric(vegetativeProfile)) {
            Integer pv = Integer.valueOf(vegetativeProfile);
            result = refStadeEDIDao.forProfil_vegetatifEquals(pv).setOrderByArguments(RefStadeEDI.PROPERTY_COLONNE2).findAll();
        } else {
            result = Collections.emptyList();
        }

        return result;
    }

    @Override
    public List<String> findAllProtocoleVgObsLabels() {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_PROTOCOLE_LIBELLE, null);
        return result;
    }
    
    @Override
    public List<String> findAllProtocoleVgObsPests(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_LIGNE_ORGANISME_VIVANT, filter);
        return result;
    }
    
    @Override
    public List<String> findAllProtocoleVgObsStades(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_LIGNE_STADES_DEVELOPPEMENT, filter);
        return result;
    }
    
    @Override
    public List<String> findAllProtocoleVgObsSupports(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_LIGNE_SUPPORTS_ORGANES, filter);
        return result;
    }
    
    @Override
    public List<String> findAllProtocoleVgObsObservations(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_LIGNE_TYPE_OBSERVATION, filter);
        return result;
    }
    
    @Override
    public List<String> findAllProtocoleVgObsQualitatives(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_CLASSE_VALEUR_QUALITATIVE, filter);
        return result;
    }

    @Override
    public List<String> findAllProtocoleVgObsUnits(ProtocoleVgObsFilter filter) {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_RELEVE_UNITE, filter);
        return result;
    }

    @Override
    public List<RefProtocoleVgObs> findAllProtocoleVgObsQualifiers(ProtocoleVgObsFilter filter) {
        List<RefProtocoleVgObs> result = refProtocoleVgObsDao.findAllProperties(null, filter);
        return result;
    }

    @Override
    public List<BioAgressorType> findAllEdiPestTypes() {
        List<BioAgressorType> result = refNuisiblesEdiDao.findAllActiveParam();
        return result;
    }

    @Override
    public List<RefNuisibleEDI> findAllEdiPests(BioAgressorType pestType) {
        List<RefNuisibleEDI> result = refNuisiblesEdiDao.forReference_paramEquals(pestType)
                .setOrderByArguments(RefNuisibleEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<RefStadeNuisibleEDI> findAllEdiPestStades() {
        List<RefStadeNuisibleEDI> result = refStadeNuisibleEDIDao.newQueryBuilder()
                .setOrderByArguments(RefStadeNuisibleEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<String> findAllVgObsUnits() {
        List<String> result = refProtocoleVgObsDao.findAllProperties(RefProtocoleVgObs.PROPERTY_RELEVE_QUALIFIANT_UNITE_MESURE, null);
        return result;
    }

    @Override
    public List<RefTypeNotationEDI> findAllEdiNotations() {
        List<RefTypeNotationEDI> result = refTypeNotationEDIDao.newQueryBuilder()
                .setOrderByArguments(RefTypeNotationEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<RefValeurQualitativeEDI> findAllEdiQualitatives() {
        List<RefValeurQualitativeEDI> result = refValeurQualitativeEDIDao.newQueryBuilder()
                .setOrderByArguments(RefValeurQualitativeEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<RefUnitesQualifiantEDI> findAllEdiQualifiantUnits() {
        List<RefUnitesQualifiantEDI> result = refUnitesQualifiantEDIDao.newQueryBuilder()
                .setOrderByArguments(RefUnitesQualifiantEDI.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public InputStream exportEffectiveMeasurementsAsXlsStream(List<String> zoneIds) {
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        // get all possible tab info
        MeasurementMainBeanInfo measurementMainBeanInfo = newInstance(MeasurementMainBeanInfo.class);
        MeasurementStadeCultureBeanInfo measurementStadeCultureBeanInfo = newInstance(MeasurementStadeCultureBeanInfo.class);
        MeasurementAdventicesBeanInfo measurementAdventicesBeanInfo = newInstance(MeasurementAdventicesBeanInfo.class);
        MeasurementNuisibleMaladiesPhysiologiquesAuxBeanInfo measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo = newInstance(MeasurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.class);
        MeasurementPlanteInfo measurementPlanteInfo = newInstance(MeasurementPlanteInfo.class);
        MeasurementSolInfo measurementSolInfo = newInstance(MeasurementSolInfo.class);
        MeasurementTransferDeSolutesInfo measurementTransferDeSolutesInfo = newInstance(MeasurementTransferDeSolutesInfo.class);
        MeasurementGesInfo measurementGesInfo = newInstance(MeasurementGesInfo.class);
        MeasurementMeteoInfo measurementMeteoInfo = newInstance(MeasurementMeteoInfo.class);

        // add all in sheet
        ExportUtils.addAllBeanInfo(sheet, measurementMainBeanInfo, measurementStadeCultureBeanInfo, measurementAdventicesBeanInfo,
                measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo, measurementPlanteInfo, measurementSolInfo,
                measurementTransferDeSolutesInfo, measurementGesInfo, measurementMeteoInfo);

        // NuisibleMaladiesPhysiologiquesAux trop compliqué
        List<MeasurementExportEntity> nuisibleEntities = (List<MeasurementExportEntity>)sheet.get(measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo);

        try {
            if (CollectionUtils.isNotEmpty(zoneIds)) {
                Iterable<Zone> zones = zoneDao.forTopiaIdIn(zoneIds).findAll();
                for (Zone zone : zones) {

                    Iterable<MeasurementSession> measurementSessions = measurementSessionDao.forZoneEquals(zone).findAllLazy(100);

                    // anonymize plot
                    zone = anonymizeService.checkForZoneAnonymization(zone);

                    for (MeasurementSession measurementSession : measurementSessions) {
                        // Common data for all tabs
    
                        MeasurementExportEntity model = new MeasurementExportEntity();
                        model.setZoneName(zone.getName());
                        model.setPlotName(zone.getPlot().getName());
                        if (zone.getPlot().getGrowingSystem() != null) {
                            model.setGrowingSystemName(zone.getPlot().getGrowingSystem().getName());
                            model.setGrowingPlanName(zone.getPlot().getGrowingSystem().getGrowingPlan().getName());
                        }
                        model.setDomainName(zone.getPlot().getDomain().getName());
                        model.setCampaign(zone.getPlot().getDomain().getCampaign());
                        model.setStartDate(measurementSession.getStartDate());
                        model.setEndDate(measurementSession.getEndDate());
    
                        Collection<Measurement> measurements = measurementSession.getMeasurements();
    
                        // main tab
                        ExportUtils.export(sheet, model, measurements, measurementMainBeanInfo);
                        // StadeCulture
                        List<Measurement> tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_STADE_CULTURE), measurementStadeCultureBeanInfo);
                        // Adventices
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_ADVENTICES), measurementAdventicesBeanInfo);
                        // NuisibleMaladiesPhysiologiquesAux
                        tmp = Lists.newArrayList(measurements);
                        Iterable<Measurement> nuisibles = Iterables.filter(tmp, IS_NUISIBLE_MALADIES_PHYSIOLOGIQUES_AUXILIAIRES);
                        for (Measurement nuisible : nuisibles) {
                            Observation observation = (Observation)nuisible;
                            MeasurementExportEntity export = (MeasurementExportEntity) model.clone();
                            ExportUtils.copyFields(model, export,
                                    measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.getCustomFormatters(),
                                    measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.getCommonColumns().keySet());
                            if (observation.isProtocolVgObs()) {
                                ExportUtils.setExtraField(export, Observation.PROPERTY_REPETITION_NUMBER, observation.getRepetitionNumber());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_COMMENT, observation.getComment());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_PROTOCOL, observation.getProtocol().getProtocole_libelle());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_NUMBER_OBSERVED, observation.getCropNumberObserved());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_PEST, observation.getProtocol().getLigne_organisme_vivant());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_ORGANISM_STAGE, observation.getProtocol().getLigne_stades_developpement());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_ORGAN_SUPPORT, observation.getProtocol().getLigne_supports_organes());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_NOTATION_TYPE, observation.getProtocol().getLigne_type_observation());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_QUALITITIVE_VALUE, observation.getProtocol().getClasse_valeur_qualitative());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_QUANTITATIVE_VALUE, observation.getQuantitativeValue());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_UNIT_EDI, observation.getProtocol().getReleve_unite());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_CROP_UNIT_QUALIFIER, observation.getProtocol().getReleve_qualifiant_unite_mesure());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_UNIT_OTHER, observation.getUnitOther());
                                ExportUtils.setExtraField(export, Observation.PROPERTY_OTHER_QUALIFIER, observation.getOtherQualifier());
                            } else {
                                ExportUtils.copyFields(observation, export,
                                        measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.getCustomFormatters(),
                                        measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.getExtraColumns().keySet());
                            }
                            nuisibleEntities.add(export);
                        }
                        // Plante
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_PLANTE), measurementPlanteInfo);
                        // Sol
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_SOL), measurementSolInfo);
                        // TransferDeSolutes
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_TRANSFERT_SOLUTES), measurementTransferDeSolutesInfo);
                        // Ges
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_GES), measurementGesInfo);
                        // Meteo
                        tmp = Lists.newArrayList(measurements);
                        ExportUtils.export(sheet, model, Iterables.filter(tmp, IS_METEO), measurementMeteoInfo);
    
                    }
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    @Override
    public void importEffectiveMeasurementsForXlsStream(InputStream is) {

        // get all possible bean info
        MeasurementStadeCultureBeanInfo measurementStadeCultureBeanInfo = newInstance(MeasurementStadeCultureBeanInfo.class);
        MeasurementAdventicesBeanInfo measurementAdventicesBeanInfo = newInstance(MeasurementAdventicesBeanInfo.class);
        MeasurementNuisibleMaladiesPhysiologiquesAuxBeanInfo measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo = newInstance(MeasurementNuisibleMaladiesPhysiologiquesAuxBeanInfo.class);
        MeasurementPlanteInfo measurementPlanteInfo = newInstance(MeasurementPlanteInfo.class);
        MeasurementSolInfo measurementSolInfo = newInstance(MeasurementSolInfo.class);
        MeasurementTransferDeSolutesInfo measurementTransferDeSolutesInfo = newInstance(MeasurementTransferDeSolutesInfo.class);
        MeasurementGesInfo measurementGesInfo = newInstance(MeasurementGesInfo.class);
        MeasurementMeteoInfo measurementMeteoInfo = newInstance(MeasurementMeteoInfo.class);

        // technical import
        EntityImporter importer = new EntityImporter();
        Map<EntityExportTabInfo, List<MeasurementExportEntity>> datas = importer.importFromStream(is,
                MeasurementExportEntity.class,
                //measurementMainBeanInfo,
                measurementStadeCultureBeanInfo,
                measurementAdventicesBeanInfo,
                measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo,
                measurementPlanteInfo,
                measurementSolInfo,
                measurementTransferDeSolutesInfo,
                measurementGesInfo,
                measurementMeteoInfo);

        // measurement session cache (multimap)
        // zonename, plotname, domainname, campaign, begindate, enddate > session
        MultiKeyMap<Object, MeasurementSession> sessionCache = new MultiKeyMap<Object, MeasurementSession>();
        
        // custom import (stade de culture)
        List<MeasurementExportEntity> stadeBeanInfo = datas.get(measurementStadeCultureBeanInfo);
        for (MeasurementExportEntity beanInfo : stadeBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            // create observation session
            Observation observation = new ObservationImpl();
            observation.setMeasurementType(MeasurementType.STADE_CULTURE);
            // common
            observation.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                observation.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            observation.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            // species
            String espece = beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE);
            if (StringUtils.isNotBlank(espece)) {
                RefEspece refEspece = refEspeceTopiaDao.forProperties(
                        RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE, espece,
                        RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE, Strings.nullToEmpty(beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE)),
                        RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE, Strings.nullToEmpty(beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE)),
                        RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE, Strings.nullToEmpty(beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE))).findUnique();
                CroppingPlanSpecies croppingPlanSpecies = croppingPlanSpeciesTopiaDao.forProperties(
                        CroppingPlanSpecies.PROPERTY_SPECIES, refEspece,
                        CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN,
                            session.getZone().getPlot().getDomain()).findUnique();
                observation.setCroppingPlanSpecies(croppingPlanSpecies);
            }
            // stade
            String stadeMin = beanInfo.getExtraAsString(Observation.PROPERTY_CROP_STAGE_MIN);
            if (StringUtils.isNoneEmpty(stadeMin)) {
                RefStadeEDI refStadeEDI = refStadeEDIDao.forColonne2Equals(stadeMin).findUnique();
                observation.setCropStageMin(refStadeEDI);
            }
            String stadeMax = beanInfo.getExtraAsString(Observation.PROPERTY_CROP_STAGE_MAX);
            if (StringUtils.isNoneEmpty(stadeMax)) {
                RefStadeEDI refStadeEDI = refStadeEDIDao.forColonne2Equals(stadeMax).findUnique();
                observation.setCropStageMax(refStadeEDI);
            }
            String stadeMedium = beanInfo.getExtraAsString(Observation.PROPERTY_CROP_STAGE_MEDIUM);
            if (StringUtils.isNoneEmpty(stadeMedium)) {
                RefStadeEDI refStadeEDI = refStadeEDIDao.forColonne2Equals(stadeMedium).findUnique();
                observation.setCropStageMedium(refStadeEDI);
            }
            String stadeAvg = beanInfo.getExtraAsString(Observation.PROPERTY_CROP_STAGE_AVERAGE);
            if (StringUtils.isNoneEmpty(stadeAvg)) {
                RefStadeEDI refStadeEDI = refStadeEDIDao.forColonne2Equals(stadeAvg).findUnique();
                observation.setCropStageAverage(refStadeEDI);
            }
            // specific
            String cropNumberObserved = beanInfo.getExtraAsString(Observation.PROPERTY_CROP_NUMBER_OBSERVED);
            if (StringUtils.isNotBlank(cropNumberObserved)) {
                observation.setCropNumberObserved(Integer.parseInt(cropNumberObserved));
            }
            // save measurement
            session.addMeasurements(observation);
        }

        // custom import (adventices)
        List<MeasurementExportEntity> adventiceBeanInfo = datas.get(measurementAdventicesBeanInfo);
        for (MeasurementExportEntity beanInfo : adventiceBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Observation observation = new ObservationImpl();
            observation.setMeasurementType(MeasurementType.ADVENTICES);
            // common
            observation.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                observation.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            observation.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            // specific
            String adventice = beanInfo.getExtraAsString(Observation.PROPERTY_MEASURED_ADVENTICE);
            if (StringUtils.isNoneBlank(adventice)) {
                RefAdventice refAdventice = refAdventiceTopiaDao.forAdventiceEquals(adventice).findUnique();
                observation.setMeasuredAdventice(refAdventice);
            }
            String adventiceStade = beanInfo.getExtraAsString(Observation.PROPERTY_ADVENTICE_STAGE);
            if (StringUtils.isNoneBlank(adventiceStade)) {
                RefStadeEDI refStadeEDI = refStadeEDIDao.forColonne2Equals(adventiceStade).findUnique();
                observation.setAdventiceStage(refStadeEDI);
            }
            try {
                ExportUtils.copyFields(beanInfo, observation,
                        Observation.PROPERTY_ADVENTICE_AVERAGE,
                        Observation.PROPERTY_ADVENTICE_MIN,
                        Observation.PROPERTY_ADVENTICE_MAX,
                        Observation.PROPERTY_ADVENTICE_MEDIAN);
            } catch (Exception ex) {
                throw new AgrosystTechnicalException("Can't copy fields", ex);
            }
            // save measurement
            session.addMeasurements(observation);
        }

        // custom import (nuisible)
        List<MeasurementExportEntity> nuisibleBeanInfo = datas.get(measurementNuisibleMaladiesPhysiologiquesAuxBeanInfo);
        for (MeasurementExportEntity beanInfo : nuisibleBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Observation measure = new ObservationImpl();
            measure.setMeasurementType(MeasurementType.NUISIBLE_MALADIES_PHYSIOLOGIQUES_AUXILIAIRES);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));

            session.addMeasurements(measure);
        }

        // custom import (plante)
        List<MeasurementExportEntity> planteBeanInfo = datas.get(measurementPlanteInfo);
        for (MeasurementExportEntity beanInfo : planteBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Measure measure = new MeasureImpl();
            measure.setMeasurementType(MeasurementType.PLANTE);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measure.PROPERTY_SAMPLING));
            // specific
            measure.setEffectiveOrAreaTaken(beanInfo.getExtraAsString(Measure.PROPERTY_EFFECTIVE_OR_AREA_TAKEN));
            String variableMesuree = beanInfo.getExtraAsString(RefMesure.PROPERTY_VARIABLE_MESUREE);
            if (StringUtils.isNotBlank(variableMesuree)) {
                RefMesure refMesure = refMesureDao.forProperties(
                        RefMesure.PROPERTY_VARIABLE_MESUREE, variableMesuree).findUnique();
                measure.setRefMesure(refMesure);
            }
            String measureType = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_TYPE);
            if (StringUtils.isNotEmpty(measureType)) {
                measure.setMeasureType(MeasureType.valueOf(measureType));
            }
            String mesureValue = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_VALUE);
            if (StringUtils.isNotBlank(mesureValue)) {
                measure.setMeasureValue(mesureValue);
            }
            measure.setMeasureUnit(beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_UNIT));
            session.addMeasurements(measure);
        }

        // custom import (sol)
        List<MeasurementExportEntity> solBeanInfo = datas.get(measurementSolInfo);
        for (MeasurementExportEntity beanInfo : solBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);
            
            Measure measure = new MeasureImpl();
            measure.setMeasurementType(MeasurementType.SOL);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measure.PROPERTY_SAMPLING));
            // specific
            String measureHorizon = beanInfo.getExtraAsString(Measure.PROPERTY_HORIZON_TYPE);
            if (StringUtils.isNotEmpty(measureHorizon)) {
                measure.setHorizonType(HorizonType.valueOf(measureHorizon));
            }
            measure.setSampling(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            String variableMesuree = beanInfo.getExtraAsString(RefMesure.PROPERTY_VARIABLE_MESUREE);
            if (StringUtils.isNotBlank(variableMesuree)) {
                RefMesure refMesure = refMesureDao.forProperties(
                        RefMesure.PROPERTY_VARIABLE_MESUREE, variableMesuree).findUnique();
                measure.setRefMesure(refMesure);
            }
            String measureType = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_TYPE);
            if (StringUtils.isNotEmpty(measureType)) {
                measure.setMeasureType(MeasureType.valueOf(measureType));
            }
            String mesureValue = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_VALUE);
            if (StringUtils.isNotBlank(mesureValue)) {
                measure.setMeasureValue(mesureValue);
            }
            measure.setMeasureUnit(beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_UNIT));
 
            session.addMeasurements(measure);
        }

        // custom import (transfer)
        List<MeasurementExportEntity> tranferBeanInfo = datas.get(measurementTransferDeSolutesInfo);
        for (MeasurementExportEntity beanInfo : tranferBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Measure measure = new MeasureImpl();
            measure.setMeasurementType(MeasurementType.TRANSFERT_DE_SOLUTES);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            // specific
            measure.setSampling(beanInfo.getExtraAsString(Measure.PROPERTY_SAMPLING));
            String variableMesuree = beanInfo.getExtraAsString(RefMesure.PROPERTY_VARIABLE_MESUREE);
            if (StringUtils.isNotBlank(variableMesuree)) {
                RefMesure refMesure = refMesureDao.forProperties(
                        RefMesure.PROPERTY_VARIABLE_MESUREE, variableMesuree).findUnique();
                measure.setRefMesure(refMesure);
            }
            String activeSubtance = beanInfo.getExtraAsString(Measure.PROPERTY_ACTIVE_SUBSTANCE);
            if (StringUtils.isNotBlank(activeSubtance)) {
                RefActaSubstanceActive refActaSubstanceActive = refSubstanceActiveDao.forProperties(
                        RefActaSubstanceActive.PROPERTY_NOM_PRODUIT, activeSubtance).findUnique();
                measure.setActiveSubstance(refActaSubstanceActive);
            }
            String nitrogenMolecule = beanInfo.getExtraAsString(Measure.PROPERTY_NITROGEN_MOLECULE);
            if (StringUtils.isNotEmpty(nitrogenMolecule)) {
                measure.setNitrogenMolecule(NitrogenMolecule.valueOf(nitrogenMolecule));
            }
            String measureType = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_TYPE);
            if (StringUtils.isNotEmpty(measureType)) {
                measure.setMeasureType(MeasureType.valueOf(measureType));
            }
            String mesureValue = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_VALUE);
            if (StringUtils.isNotBlank(mesureValue)) {
                measure.setMeasureValue(mesureValue);
            }
            measure.setMeasureUnit(beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_UNIT));
            session.addMeasurements(measure);
        }

        // custom import (ges)
        List<MeasurementExportEntity> gesBeanInfo = datas.get(measurementGesInfo);
        for (MeasurementExportEntity beanInfo : gesBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Measure measure = new MeasureImpl();
            measure.setMeasurementType(MeasurementType.GES);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            // specific
            measure.setSampling(beanInfo.getExtraAsString(Measure.PROPERTY_SAMPLING));
            String variableMesuree = beanInfo.getExtraAsString(RefMesure.PROPERTY_VARIABLE_MESUREE);
            if (StringUtils.isNotBlank(variableMesuree)) {
                RefMesure refMesure = refMesureDao.forProperties(
                        RefMesure.PROPERTY_VARIABLE_MESUREE, variableMesuree).findUnique();
                measure.setRefMesure(refMesure);
            }
            String measureType = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_TYPE);
            if (StringUtils.isNotEmpty(measureType)) {
                measure.setMeasureType(MeasureType.valueOf(measureType));
            }
            String mesureValue = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_VALUE);
            if (StringUtils.isNotBlank(mesureValue)) {
                measure.setMeasureValue(mesureValue);
            }
            measure.setMeasureUnit(beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_UNIT));
            session.addMeasurements(measure);
        }

        // custom import (meteo)
        List<MeasurementExportEntity> meteoBeanInfo = datas.get(measurementMeteoInfo);
        for (MeasurementExportEntity beanInfo : meteoBeanInfo) {
            MeasurementSession session = getMeasurementSession(sessionCache, beanInfo);

            Measure measure = new MeasureImpl();
            measure.setMeasurementType(MeasurementType.METEO);
            // common
            measure.setMeasuringProtocol(beanInfo.getExtraAsString(Measurement.PROPERTY_MEASURING_PROTOCOL));
            String repetitionNumber = beanInfo.getExtraAsString(Measurement.PROPERTY_REPETITION_NUMBER);
            if (StringUtils.isNotBlank(repetitionNumber)) {
                measure.setRepetitionNumber(Integer.parseInt(repetitionNumber));
            }
            measure.setComment(beanInfo.getExtraAsString(Measurement.PROPERTY_COMMENT));
            // specific
            measure.setSampling(beanInfo.getExtraAsString(Measure.PROPERTY_SAMPLING));
            String variableMesuree = beanInfo.getExtraAsString(RefMesure.PROPERTY_VARIABLE_MESUREE);
            if (StringUtils.isNotBlank(variableMesuree)) {
                RefMesure refMesure = refMesureDao.forProperties(
                        RefMesure.PROPERTY_VARIABLE_MESUREE, variableMesuree).findUnique();
                measure.setRefMesure(refMesure);
            }
            String measureType = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_TYPE);
            if (StringUtils.isNotEmpty(measureType)) {
                measure.setMeasureType(MeasureType.valueOf(measureType));
            }
            String mesureValue = beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_VALUE);
            if (StringUtils.isNotBlank(mesureValue)) {
                measure.setMeasureValue(mesureValue);
            }
            measure.setMeasureUnit(beanInfo.getExtraAsString(Measure.PROPERTY_MEASURE_UNIT));
            session.addMeasurements(measure);
        }
        
        // persist all session in cache
        for (MeasurementSession session : sessionCache.values()) {
            if (session.isPersisted()) {
                measurementSessionDao.update(session);
            } else {
                measurementSessionDao.create(session);
            }
        }
        
        getTransaction().commit();
    }
    
    protected MeasurementSession getMeasurementSession(MultiKeyMap<Object, MeasurementSession> sessionCache, final MeasurementExportEntity beanInfo) {
        // search in cache first
        MultiKey<Object> key = new MultiKey<Object>(new Object[] {
                beanInfo.getZoneName(),
                beanInfo.getPlotName(),
                beanInfo.getDomainName(),
                beanInfo.getCampaign(),
                beanInfo.getStartDate(),
                beanInfo.getEndDate()
        });
        MeasurementSession result;
        if (sessionCache.containsKey(key)) {
            result = sessionCache.get(key);
        } else {
            // find measurement zone
            Zone zone = zoneDao.forProperties(
                    Zone.PROPERTY_NAME, beanInfo.getZoneName(),
                    Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_NAME, beanInfo.getPlotName(),
                    Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_DOMAIN + "." + Domain.PROPERTY_NAME, beanInfo.getDomainName(),
                    Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, beanInfo.getCampaign()).findUnique();
            // find measurement session for zone
            List<MeasurementSession> sessions = measurementSessionDao.forZoneEquals(zone).findAll();
            Optional<MeasurementSession> session = Iterables.tryFind(sessions, new Predicate<MeasurementSession>() {
                @Override
                public boolean apply(MeasurementSession input) {
                    Date startDate = DateUtils.truncate(input.getStartDate(), Calendar.DATE);
                    Date endDate = DateUtils.truncate(input.getEndDate(), Calendar.DATE);
                    return startDate.equals(beanInfo.getStartDate()) && endDate.equals(beanInfo.getEndDate());
                }
            });
    
            // return or create new measurement session
            
            if (session.isPresent()) {
                result = session.get();
            } else {
                result = new MeasurementSessionImpl();
                result.setStartDate(beanInfo.getStartDate());
                result.setEndDate(beanInfo.getEndDate());
                result.setZone(zone);
            }
            
            // put in cache
            sessionCache.put(key, result);
        }
        return result;
    }
}
