package fr.inra.agrosyst.services.managementmode;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ManagementModeServiceImpl.java 4279 2014-08-29 15:16:22Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/managementmode/ManagementModeServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.InputStream;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.util.PagerBean;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.BioAgressorType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRule;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRuleTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeCategory;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.Section;
import fr.inra.agrosyst.api.entities.managementmode.SectionTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.SectionType;
import fr.inra.agrosyst.api.entities.managementmode.Strategy;
import fr.inra.agrosyst.api.entities.managementmode.StrategyTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefAdventiceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressorTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefNuisibleEDITopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.common.HistoryItem;
import fr.inra.agrosyst.api.services.common.HistoryType;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.managementmode.DecisionRuleFilter;
import fr.inra.agrosyst.api.services.managementmode.DecisionRulesDto;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeDto;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeFilter;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.api.services.managementmode.SectionDto;
import fr.inra.agrosyst.api.services.managementmode.StrategyDto;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.managementmode.export.DecisionRuleExportEntity;
import fr.inra.agrosyst.services.managementmode.export.ManagementModeExportEntity;
import fr.inra.agrosyst.services.managementmode.export.ManagementModeExportMetadata.ManagementModeMainBeanInfo;
import fr.inra.agrosyst.services.managementmode.export.ManagementModeExportMetadata.ManagementModeSectionBeanInfo;
import fr.inra.agrosyst.services.managementmode.export.DecisionRuleExportMetadata.DecisionRuleMainBeanInfo;

public class ManagementModeServiceImpl extends AbstractAgrosystService implements ManagementModeService {

    protected BusinessAuthorizationService authorizationService;
    protected AnonymizeService anonymizeService;

    protected DecisionRuleTopiaDao decisionRuleTopiaDao;
    protected RefNuisibleEDITopiaDao refNuisibleEDITopiaDao;
    protected RefAdventiceTopiaDao refAdventiceTopiaDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao;
    protected GrowingSystemTopiaDao growingSystemTopiaDao;
    protected DomainTopiaDao domainTopiaDao;
    protected ManagementModeTopiaDao managementModeTopiaDao;
    protected RefBioAgressorTopiaDao refBioAgressorTopiaDao;
    protected SectionTopiaDao sectionTopiaDao;
    protected StrategyTopiaDao strategyTopiaDao;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setDecisionRuleTopiaDao(DecisionRuleTopiaDao decisionRuleTopiaDao) {
        this.decisionRuleTopiaDao = decisionRuleTopiaDao;
    }

    public void setRefNuisibleEDITopiaDao(RefNuisibleEDITopiaDao refNuisibleEDITopiaDao) {
        this.refNuisibleEDITopiaDao = refNuisibleEDITopiaDao;
    }

    public void setRefAdventiceTopiaDao(RefAdventiceTopiaDao refAdventiceTopiaDao) {
        this.refAdventiceTopiaDao = refAdventiceTopiaDao;
    }

    public void setCroppingPlanEntryTopiaDao(CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao) {
        this.croppingPlanEntryTopiaDao = croppingPlanEntryTopiaDao;
    }

    public void setDomainTopiaDao(DomainTopiaDao domainTopiaDao) {
        this.domainTopiaDao = domainTopiaDao;
    }

    public void setGrowingSystemTopiaDao(GrowingSystemTopiaDao growingSystemTopiaDao) {
        this.growingSystemTopiaDao = growingSystemTopiaDao;
    }

    public void setManagementModeTopiaDao(ManagementModeTopiaDao managementModeTopiaDao) {
        this.managementModeTopiaDao = managementModeTopiaDao;
    }

    public void setRefBioAgressorTopiaDao(RefBioAgressorTopiaDao refBioAgressorTopiaDao) {
        this.refBioAgressorTopiaDao = refBioAgressorTopiaDao;
    }

    public void setSectionTopiaDao(SectionTopiaDao sectionTopiaDao) {
        this.sectionTopiaDao = sectionTopiaDao;
    }

    public void setStrategyTopiaDao(StrategyTopiaDao strategyTopiaDao) {
        this.strategyTopiaDao = strategyTopiaDao;
    }

    @Override
    public ResultList<DecisionRulesDto> getFilteredDecisionRules(DecisionRuleFilter filter) {
        ResultList<DecisionRule> decisionRules = decisionRuleTopiaDao.getFilteredDecisionRules(filter, getSecurityContext());

        PagerBean pager = decisionRules.getPager();
        List<DecisionRulesDto> decisionRulesDtos = Lists.newArrayListWithCapacity(decisionRules.getElements().size());
        Binder<DecisionRule, DecisionRulesDto> binder = BinderFactory.newBinder(DecisionRule.class, DecisionRulesDto.class);

        Function<Domain, DomainDto> domainToDtoFunction = anonymizeService.getDomainToDtoFunction(false);
        for (DecisionRule decisionRule : decisionRules) {

            // copy each result into a DTO to link a decision rule
            DecisionRulesDto decisionRulesDto = new DecisionRulesDto();

            // search max version rule
            DecisionRule maxVersionRule = decisionRuleTopiaDao.forProperties(
                    DecisionRule.PROPERTY_ACTIVE, decisionRule.isActive(),
                    DecisionRule.PROPERTY_CODE, decisionRule.getCode()).setOrderByArguments(DecisionRule.PROPERTY_VERSION_NUMBER + " DESC").findFirst();
            binder.copy(maxVersionRule, decisionRulesDto);

            // link to a real domain via DecisionRule#domainCode code
            Domain relatedRulesDomain = domainTopiaDao.forProperties(
                    Domain.PROPERTY_ACTIVE, true,
                    Domain.PROPERTY_CODE, decisionRule.getDomainCode()).findAnyOrNull();

            if(relatedRulesDomain != null) {
                decisionRulesDto.setDomain(domainToDtoFunction.apply(relatedRulesDomain));

                decisionRulesDtos.add(decisionRulesDto);
            }
        }

        ResultList<DecisionRulesDto> result = ResultList.of(decisionRulesDtos, pager);
        return result;
    }

    @Override
    public DecisionRule newDecisionRule() {
        DecisionRule decisionRule = decisionRuleTopiaDao.newInstance();

        // active by default
        decisionRule.setActive(true);
        // add defaut solution template
        decisionRule.setSolution("<p><strong>si</strong>&nbsp;</p><p><strong>alors</strong>&nbsp;</p><p><strong>sinon</strong>&nbsp;</p>");
        // add a code
        decisionRule.setCode(UUID.randomUUID().toString());
        // add a version number
        decisionRule.setVersionNumber(1);

        return decisionRule;
    }

    @Override
    public DecisionRule getDecisionRule(String decisionRuleTopiaId) {
        DecisionRule result = decisionRuleTopiaDao.forTopiaIdEquals(decisionRuleTopiaId).findUnique();
        return result;
    }

    @Override
    public DecisionRule getLastDecisionRuleVersion(String decisionRuleCode) {
        DecisionRule result = decisionRuleTopiaDao.getLastRuleVersion(decisionRuleCode);
        return result;
    }

    @Override
    public List<BioAgressorType> getBioAgressorTypes() {

        List<BioAgressorType> result = Lists.newArrayList();
        result.add(BioAgressorType.ADVENTICE);
        result.addAll(refNuisibleEDITopiaDao.findAllActiveParam());
        return result;
    }

    @Override
    public <E extends RefBioAgressor> List<E> getBioAgressors(BioAgressorType bioAgressorType) {
        List<E> result;

        if (bioAgressorType == BioAgressorType.ADVENTICE) {
            result = (List<E>) refAdventiceTopiaDao.forActiveEquals(true).findAll();
        } else {
            result = (List<E>) refNuisibleEDITopiaDao.forReference_paramEquals(bioAgressorType).findAll();
        }

        return result;
    }

    /**
     * @deprecated duplicated method from DomainServiceImpl.
     */
    @Deprecated
    protected List<CroppingPlanEntry> getCroppingPlan0(String domainId) {

        String propertyDomainId = CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
        return croppingPlanEntryTopiaDao.forProperties(propertyDomainId, domainId).findAll();
    }

    @Override
    public List<CroppingPlanEntry> getDomainCodeCroppingPlanEntries(String domainCode) {
        List<CroppingPlanEntry> result = managementModeTopiaDao.getCroppingPlanEntryWithDomainCode(domainCode);
        return result;
    }

    @Override
    public List<CroppingPlanEntry> getGrowingSystemCroppingPlanEntries(String growingSystemTopiaId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(growingSystemTopiaId));
        GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemTopiaId).findUnique();
        List<CroppingPlanEntry> result = getCroppingPlan0(growingSystem.getGrowingPlan().getDomain().getTopiaId());
        return result;
    }

    @Override
    public CroppingPlanEntry getCroppingPlanEntries(String croppingPlanEntryTopiaId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(croppingPlanEntryTopiaId));
        CroppingPlanEntry croppingPlanEntry = croppingPlanEntryTopiaDao.forTopiaIdEquals(croppingPlanEntryTopiaId).findUnique();
        return croppingPlanEntry;
    }

    @Override
    public Collection<DecisionRule> getGrowingSystemDecisionRules(String growingSystemTopiaId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(growingSystemTopiaId));
        GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemTopiaId).findUnique();
        List<DecisionRule> decisionRules = decisionRuleTopiaDao.forActiveEquals(true).
                addEquals(DecisionRule.PROPERTY_DOMAIN_CODE,
                        growingSystem.getGrowingPlan().getDomain().getCode()).
                setOrderByArguments(DecisionRule.PROPERTY_CODE, DecisionRule.PROPERTY_VERSION_NUMBER + " DESC")
                .findAll();
        Map<String, DecisionRule> lastVersionDecisionRules = Maps.newHashMap();
        for (DecisionRule decisionRule: decisionRules) {
            if (lastVersionDecisionRules.get(decisionRule.getCode()) == null) {
                lastVersionDecisionRules.put(decisionRule.getCode(), decisionRule);
            }
        }

        return lastVersionDecisionRules.values();
    }

    @Override
    public DecisionRule createOrUpdateDecisionRule(DecisionRule decisionRule, String domainCode, String croppingPlanEntryCode, String bioAgressorTopiaId) {

        authorizationService.checkCreateOrUpdateDecisionRule(decisionRule.getTopiaId());
        DecisionRule originalDecisionRule = null;

        // growing system id
        if (decisionRule.getTopiaId() == null) {
            if (StringUtils.isBlank(domainCode)) {
                throw new AgrosystTechnicalException("Domain can't be null");
            } else {
                Preconditions.checkState(domainTopiaDao.forCodeEquals(domainCode).count() > 0);
                decisionRule.setDomainCode(domainCode);
            }
        } else {
            originalDecisionRule = decisionRuleTopiaDao.forTopiaIdEquals(decisionRule.getTopiaId()).findUnique();
        }

        // croping plan entry code
        decisionRule.setCroppingPlanEntryCode(croppingPlanEntryCode);

        // bio agressor
        RefBioAgressor bioAgressor = null;
        if (StringUtils.isNotBlank(bioAgressorTopiaId)) {
            bioAgressor = refBioAgressorTopiaDao.forTopiaIdEquals(bioAgressorTopiaId).findUnique();
        }
        decisionRule.setBioAgressor(bioAgressor);

        DecisionRule result;
        // persist decision rule
        if (originalDecisionRule != null) {

            Binder<DecisionRule, DecisionRule> binder = BinderFactory.newBinder(DecisionRule.class);

            // it's always a copy of the original rule that we are using in the JSP
            // in the case of a modification of the rule push the modification to the original one.
            binder.copyExcluding(decisionRule, originalDecisionRule,
                    DecisionRule.PROPERTY_TOPIA_ID);
            result = decisionRuleTopiaDao.update(originalDecisionRule);
        } else {
            result = decisionRuleTopiaDao.create(decisionRule);
        }

        // On met à jour le nom sur toutes les règles
        final String newName = result.getName();
        List<DecisionRule> relatedDecisionRules = getRelatedDecisionRules(result.getCode());
        Iterable<DecisionRule> rulesWithDifferentName = Iterables.filter(relatedDecisionRules, new Predicate<DecisionRule>() {
            @Override
            public boolean apply(DecisionRule related) {
                return !newName.equals(related.getName());
            }
        });
        for (DecisionRule related : rulesWithDifferentName) {
            related.setName(newName);
            decisionRuleTopiaDao.update(related);
        }

        getTransaction().commit();
        return result;
    }

    @Override
    public DecisionRule createNewDecisionRule(
            AgrosystInterventionType interventionType,
            String growingSystemTopiaId,
            BioAgressorType bioAgressorType,
            String bioAgressorTopiaId,
            String croppingPlanEntryId,
            String name) {

        Preconditions.checkNotNull(growingSystemTopiaId);
        Preconditions.checkNotNull(interventionType);
        Preconditions.checkNotNull(name);

        GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemTopiaId).findUnique();

        RefBioAgressor bioAgressor = null;
        if (!Strings.isNullOrEmpty(bioAgressorTopiaId)) {
            bioAgressor = refBioAgressorTopiaDao.forTopiaIdEquals(bioAgressorTopiaId).findUnique();
        }

        String croppingPlanEntryCode = null;
        if (!Strings.isNullOrEmpty(croppingPlanEntryId)) {
            croppingPlanEntryCode = croppingPlanEntryTopiaDao.forTopiaIdEquals(croppingPlanEntryId).findUnique().getCode();
        }

        DecisionRule decisionRule = newDecisionRule();
        decisionRule.setDomainCode(growingSystem.getGrowingPlan().getDomain().getCode());
        decisionRule.setBioAgressorType(bioAgressorType);
        decisionRule.setBioAgressor(bioAgressor);
        decisionRule.setName(name);
        decisionRule.setInterventionType(interventionType);
        decisionRule.setCroppingPlanEntryCode(croppingPlanEntryCode);
        DecisionRule result = decisionRuleTopiaDao.create(decisionRule);
        getTransaction().commit();
        return result;
    }

    @Override
    public ResultList<ManagementModeDto> getFilteredManagementModeDtos(ManagementModeFilter managementModeFilter) {
        Pair<Map<GrowingSystem, ManagementModeDto>, PagerBean> mmList = managementModeTopiaDao.getFilteredManagementModeDtos(managementModeFilter, getSecurityContext());
        Set<Map.Entry<GrowingSystem,ManagementModeDto>> entries = mmList.getLeft().entrySet();
        final Function<GrowingSystem,GrowingSystemDto> growingSystemToDtoFunction = anonymizeService.getGrowingSystemToDtoFunction();
        Iterable<ManagementModeDto> list = Iterables.transform(entries, new Function<Map.Entry<GrowingSystem, ManagementModeDto>, ManagementModeDto>() {
            @Override
            public ManagementModeDto apply(Map.Entry<GrowingSystem, ManagementModeDto> input) {
                GrowingSystemDto growingSystemDto = growingSystemToDtoFunction.apply(input.getKey());
                ManagementModeDto result = input.getValue();
                result.setGrowingSystem(growingSystemDto);
                return result;
            }
        });
        ResultList<ManagementModeDto> result = ResultList.of(Lists.newArrayList(list), mmList.getRight());
        return result;
    }

    @Override
    public ManagementMode newManagementMode() {
        ManagementMode managementMode = managementModeTopiaDao.newInstance();

        // active by default
        managementMode.setActive(true);

        return managementMode;
    }

    @Override
    public ManagementMode getManagementMode(String managementModeTopiaId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(managementModeTopiaId));
        ManagementMode managementMode = managementModeTopiaDao.forTopiaIdEquals(managementModeTopiaId).findUnique();
        ManagementMode result = anonymizeService.checkForManagementModeAnonymization(managementMode);
        return result;
    }

    @Override
    public ManagementMode createOrUpdateManagementMode(ManagementMode managementMode, String growingSystemTopiaId, Collection<SectionDto> sections) {

        authorizationService.checkCreateOrUpdateManagementMode(managementMode.getTopiaId());

        Preconditions.checkNotNull(managementMode.getCategory());

        // growing system id
        if (!managementMode.isPersisted()) {
            if (StringUtils.isBlank(growingSystemTopiaId)) {
                throw new AgrosystTechnicalException("Growing system can't be null");
            } else {
                GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemTopiaId).findUnique();
                managementMode.setGrowingSystem(growingSystem);
            }
        }

        updateManagementModeSections(managementMode, sections);

        // update management mode
        if (managementMode.isPersisted()) {
            managementMode = managementModeTopiaDao.update(managementMode);
        } else {

            List<HistoryItem> histories = Lists.newArrayList();
            Gson gson = context.getGson();

            // historic
            histories.add(createNewManagementModeHistory(managementMode));
            managementMode.setHistorical(gson.toJson(histories));

            managementMode = managementModeTopiaDao.create(managementMode);
        }

        getTransaction().commit();
        return managementMode;
    }

    /**
     * Update management mode's section with provided DTO.
     *
     * @param managementMode management mode containing section
     * @param sections       dto
     */
    protected void updateManagementModeSections(ManagementMode managementMode, Collection<SectionDto> sections) {
        // sections
        if (sections != null) {
            Date transactionDate = context.getCurrentDate();

            String previousHistory = managementMode.getHistorical();

            List<HistoryItem> histories;
            Gson gson = context.getGson();
            Type type = new TypeToken<List<HistoryItem>>() {}.getType();

            if (previousHistory != null) {
                histories = gson.fromJson(previousHistory,type);
            } else {
                histories = Lists.newArrayList();
            }

            Collection<Section> currentSections = managementMode.getSections();
            List<Section> nonDeletedSection = Lists.newArrayList();
            if (currentSections == null) {
                currentSections = Lists.newArrayList();
                managementMode.setSections(currentSections);
            }
            Map<String, Section> sectionsIndex = Maps.uniqueIndex(currentSections, Entities.GET_TOPIA_ID);
            for (SectionDto sectionDto : sections) {
                String topiaId = sectionDto.getTopiaId();
                Section section;

                RefBioAgressor bioAgressor = null;
                if (StringUtils.isNotBlank(sectionDto.getBioAgressorTopiaId())) {
                    bioAgressor = refBioAgressorTopiaDao.forTopiaIdEquals(sectionDto.getBioAgressorTopiaId()).findUnique();
                }
                if (StringUtils.isBlank(topiaId)) {
                    section = sectionTopiaDao.newInstance();

                    HistoryItem historyItem = createHistoryItemSectionAdd(sectionDto, bioAgressor, transactionDate);
                    histories.add(historyItem);

                } else {
                    section = sectionsIndex.get(topiaId);
                }
                section.setAgronomicObjective(sectionDto.getAgronomicObjective());
                section.setBioAgressorType(sectionDto.getBioAgressorType());
                section.setExpectedResult(sectionDto.getExpectedResult());
                section.setSectionType(sectionDto.getSectionType());
                section.setCategoryObjective(sectionDto.getCategoryObjective());
                section.setCategoryStrategy(sectionDto.getCategoryStrategy());

                section.setBioAgressor(bioAgressor);

                if (StringUtils.isBlank(topiaId)) {
                    currentSections.add(section);
                }
                nonDeletedSection.add(section);

                List<HistoryItem> strategiesHistories = updateSectionStrategies(section, sectionDto.getStrategiesDto());
                histories.addAll(strategiesHistories);
            }
            Collection<Section> removedSections = CollectionUtils.subtract(currentSections,nonDeletedSection);

            if (removedSections != null) {
                for (Section section:removedSections){
                    HistoryItem historyItem = createHistoryItemSectionRemove(section, section.getBioAgressor(), transactionDate);
                    histories.add(historyItem);
                }
            }
            managementMode.setHistorical(gson.toJson(histories));

            currentSections.retainAll(nonDeletedSection);
        }
    }

    protected HistoryItem createHistoryItemSectionAdd(SectionDto sectionDto, RefBioAgressor bioAgressor, Date transactionDate) {
        HistoryItem historyItem = createHistoryItemSection(sectionDto.getSectionType(), sectionDto.getBioAgressorType(), bioAgressor, transactionDate, HistoryType.SECTION_ADD);
        return historyItem;
    }

    protected HistoryItem createHistoryItemSectionRemove(Section section, RefBioAgressor bioAgressor, Date transactionDate) {
        HistoryItem historyItem = createHistoryItemSection(section.getSectionType(), section.getBioAgressorType(), bioAgressor, transactionDate, HistoryType.SECTION_REMOVE);
        return historyItem;
    }

    protected HistoryItem createHistoryItemSection(SectionType sectionType, BioAgressorType bioAgressorType, RefBioAgressor bioAgressor, Date transactionDate, HistoryType type) {
        HistoryItem historyItem = new HistoryItem();
        historyItem.setDate(transactionDate);
        historyItem.setType(type);
        List<String> args = historyItem.getArgs();

        args.add(sectionType.name());

        if (bioAgressorType != null) {
            args.add(bioAgressorType.name());
        } else {
            args.add(null);
        }
        if (bioAgressor != null) {
            args.add(bioAgressor.getTopiaId());
        } else {
            args.add(null);
        }
        return historyItem;
    }

    /**
     * Update management mode's section with provided DTO.
     *
     * @param section    section containing strategies
     * @param strategies dto
     */
    protected List<HistoryItem> updateSectionStrategies(Section section, List<StrategyDto> strategies) {

        List<HistoryItem> historiesItems = Lists.newArrayList();
        // sections
        if (strategies != null) {
            List<Strategy> currentStrategies = section.getStrategies();
            List<Strategy> nonDeletedStrategies = Lists.newArrayList();
            if (currentStrategies == null) {
                currentStrategies = Lists.newArrayList();
                section.setStrategies(currentStrategies);
            }
            Map<String, Strategy> strategiesIndex = Maps.uniqueIndex(currentStrategies, Entities.GET_TOPIA_ID);

            for (StrategyDto strategyDto : strategies) {
                String topiaId = strategyDto.getTopiaId();
                Strategy strategy;
                if (StringUtils.isBlank(topiaId)) {
                    strategy = strategyTopiaDao.newInstance();
                    historiesItems.add(createStrategyHistoryAdd(strategyDto));
                } else {
                    strategy = strategiesIndex.get(topiaId);
                }
                strategy.setStrategyType(strategyDto.getStrategyType());
                strategy.setExplanation(strategyDto.getExplanation());
                strategy.setCroppingPlanManagmentName(strategyDto.getCroppingPlanManagmentName());
                strategy.setMultiannual(strategyDto.isMultiannual());

                CroppingPlanEntry croppingPlanEntry = null;
                if (StringUtils.isNotBlank(strategyDto.getCroppingPlanEntryId())) {
                    croppingPlanEntry = croppingPlanEntryTopiaDao.forTopiaIdEquals(strategyDto.getCroppingPlanEntryId()).findUnique();
                }
                strategy.setCroppingPlanEntry(croppingPlanEntry);

                // for rules
                Collection<DecisionRule> currentRules = strategy.getRules();
                if (currentRules == null) {
                    currentRules = Lists.newArrayList();
                    strategy.setRules(currentRules);
                }
                Map<String, DecisionRule> indexedRules = Maps.newHashMap(Maps.uniqueIndex(currentRules, Entities.GET_TOPIA_ID));
                currentRules.clear();
                if (strategyDto.getDecisionRuleIds() != null) {
                    for (String decisionRuleId : strategyDto.getDecisionRuleIds()) {
                        DecisionRule decisionRule = indexedRules.remove(decisionRuleId);
                        if (decisionRule == null) {
                            decisionRule = decisionRuleTopiaDao.forTopiaIdEquals(decisionRuleId).findUnique();
                        }
                        currentRules.add(decisionRule);
                    }
                }

                historiesItems.addAll(createRulesHistoryAdd(currentRules));
                historiesItems.addAll(createRulesHistoryRemove(indexedRules.values()));

                if (StringUtils.isBlank(topiaId)) {
                    currentStrategies.add(strategy);
                }
                nonDeletedStrategies.add(strategy);
            }
            List<Strategy> removedStrategies = ListUtils.subtract(currentStrategies,nonDeletedStrategies);
            historiesItems.addAll(createStrategyHistoryRemove(removedStrategies));

            currentStrategies.retainAll(nonDeletedStrategies);
        }

        return historiesItems;
    }

    protected HistoryItem createStrategyHistoryAdd(StrategyDto strategyDto) {
        //(Boolean isMultiannual, String explanation, CroppingPlanEntry croppingPlanEntry, HistoryType type)
        HistoryItem item = createStrategyHistory(
                strategyDto.isMultiannual(),
                strategyDto.getExplanation(),
                strategyDto.getCroppingPlanEntryId(),
                HistoryType.STRATEGY_ADD);
        return item;
    }

    protected List<HistoryItem> createStrategyHistoryRemove(List<Strategy> strategies) {
        List<HistoryItem> result = Lists.newArrayListWithCapacity(strategies.size());
        for(Strategy strategy:strategies) {
            String croppingPanEntryId = strategy.getCroppingPlanEntry() == null ? null : strategy.getCroppingPlanEntry().getTopiaId();
            result.add(createStrategyHistory(
                    strategy.isMultiannual(),
                    strategy.getExplanation(),
                    croppingPanEntryId,
                    HistoryType.STRATEGY_REMOVE));
        }
        return result;
    }

    protected List<HistoryItem> createRulesHistoryAdd(Collection<DecisionRule> rules) {
        List<HistoryItem> result = Lists.newArrayListWithCapacity(rules.size());
        for (DecisionRule decisionRule : rules) {
            result.add(createRuleHistory(decisionRule, HistoryType.RULE_ADD));
        }
        return result;
    }

    protected List<HistoryItem> createRulesHistoryRemove(Collection<DecisionRule> rules) {
        List<HistoryItem> result = Lists.newArrayListWithCapacity(rules.size());
        for (DecisionRule decisionRule : rules) {
            result.add(createRuleHistory(decisionRule, HistoryType.RULE_REMOVE));
        }
        return result;
    }

    protected HistoryItem createRuleHistory(DecisionRule decisionRule, HistoryType type) {
        HistoryItem item = new HistoryItem();
        item.setDate(context.getCurrentDate());
        item.setType(type);
        List<String> args = Lists.newArrayList(decisionRule.getTopiaId(), String.valueOf(decisionRule.getVersionNumber()));
        item.setArgs(args);
        return item;
    }

    protected HistoryItem createStrategyHistory(Boolean isMultiannual, String explanation, String croppingPlanEntryId, HistoryType type) {
        HistoryItem item = new HistoryItem();
        item.setDate(context.getCurrentDate());
        item.setType(type);
        String multiannual = isMultiannual ? "true" : "false";
        List<String> args = Lists.newArrayList(multiannual , explanation, croppingPlanEntryId);
        item.setArgs(args);
        return item;
    }

    @Override
    public ManagementMode getManagementModeByGrowingSystem(GrowingSystem growingSystem) {
        ManagementMode managementMode = managementModeTopiaDao.forGrowingSystemEquals(growingSystem).findAnyOrNull();
        return managementMode;
    }

    @Override
    public ManagementMode extendManagementMode(ExtendContext extendContext, ManagementMode managementMode, GrowingSystem clonedGrowingSystem) {

        Preconditions.checkNotNull(managementMode);

        // perform clone
        Binder<ManagementMode, ManagementMode> managementModeBinder = BinderFactory.newBinder(ManagementMode.class);
        ManagementMode clonedManagementMode = managementModeTopiaDao.newInstance();
        managementModeBinder.copyExcluding(managementMode, clonedManagementMode,
                ManagementMode.PROPERTY_TOPIA_ID,
                ManagementMode.PROPERTY_TOPIA_CREATE_DATE,
                ManagementMode.PROPERTY_SECTIONS,
                ManagementMode.PROPERTY_CATEGORY,
                ManagementMode.PROPERTY_GROWING_SYSTEM,
                ManagementMode.PROPERTY_HISTORICAL);

        // natural id (groing system/category)
        clonedManagementMode.setCategory(ManagementModeCategory.PLANNED);
        clonedManagementMode.setGrowingSystem(clonedGrowingSystem);

        // historic
        List<HistoryItem> histories = Lists.newArrayList();
        Gson gson = context.getGson();
        histories.add(createManagementModeExtendHistory(managementMode, clonedManagementMode));
        clonedManagementMode.setHistorical(gson.toJson(histories));

        // clone section
        cloneSections(managementMode, clonedManagementMode);

        // persist mode
        ManagementMode result = managementModeTopiaDao.create(clonedManagementMode);

        return result;
    }

    protected HistoryItem createNewManagementModeHistory(ManagementMode managementMode) {
        HistoryItem result = createManagementModeHistory(
                HistoryType.MANAGEMENT_MODE_ADD,
                null,
                managementMode.getCategory().name());
        return result;
    }

    protected HistoryItem createManagementModeCopyHistory(ManagementMode fromManagementMode, ManagementMode toManagementMode) {
        HistoryItem result = createManagementModeHistory(
                HistoryType.MANAGEMENT_MODE_COPY,
                fromManagementMode.getCategory().name(),
                toManagementMode.getCategory().name());
        return result;
    }

    protected HistoryItem createManagementModeExtendHistory(ManagementMode fromManagementMode, ManagementMode toManagementMode) {
        HistoryItem result = createManagementModeHistory(
                HistoryType.MANAGEMENT_MODE_EXTEND,
                fromManagementMode.getCategory().name(),
                toManagementMode.getCategory().name());
        return result;
    }

    protected HistoryItem createManagementModeHistory(HistoryType type, String fromManagementModeCategoryName, String toManagementModeCategoryName) {
        HistoryItem item = new HistoryItem();
        item.setDate(context.getCurrentDate());
        item.setType(type);
        List<String> args = Lists.newArrayList();
        if (!Strings.isNullOrEmpty(fromManagementModeCategoryName)) {
            args.add(fromManagementModeCategoryName);
        }
        args.add(toManagementModeCategoryName);
        item.setArgs(args);
        return item;
    }

    protected void cloneSections(ManagementMode managementMode, ManagementMode clonedManagementMode){
        if (managementMode != null && managementMode.getSections() != null) {
            for (Section section : managementMode.getSections()) {
                Binder<Section, Section> sectionModeBinder = BinderFactory.newBinder(Section.class);
                Section clonedSection = sectionTopiaDao.newInstance();
                sectionModeBinder.copyExcluding(section, clonedSection,
                        Section.PROPERTY_TOPIA_ID,
                        Section.PROPERTY_STRATEGIES);

                clonedManagementMode.addSections(clonedSection);

                // clone strategies
                if (section.getStrategies() != null) {
                    for (Strategy strategy : section.getStrategies()) {
                        Binder<Strategy, Strategy> strategyModeBinder = BinderFactory.newBinder(Strategy.class);
                        Strategy clonedStrategy = strategyTopiaDao.newInstance();
                        strategyModeBinder.copyExcluding(strategy, clonedStrategy,
                                Strategy.PROPERTY_TOPIA_ID,
                                Strategy.PROPERTY_RULES);

                        clonedSection.addStrategies(clonedStrategy);

                        // clone strategies
                        if (strategy.getRules() != null) {
                            clonedStrategy.setRules(Lists.newArrayList(strategy.getRules()));
                        }
                    }
                }
            }
        }
    }

    @Override
    public List<DecisionRule> getRelatedDecisionRules(String code) {
        List<DecisionRule> result = decisionRuleTopiaDao.findAllRelatedDecisionRules(code);
        return result;
    }

    @Override
    public DecisionRule createNewDecisonRuleVersion(String decisionRuleCode, String comment) {
        DecisionRule decisionRule = decisionRuleTopiaDao.getLastRuleVersion(decisionRuleCode);
        authorizationService.checkCreateOrUpdateDecisionRule(decisionRule.getTopiaId());

        Binder<DecisionRule, DecisionRule> binder = BinderFactory.newBinder(DecisionRule.class);

        DecisionRule newDecisionRuleVersion = decisionRuleTopiaDao.newInstance();
        binder.copyExcluding(decisionRule, newDecisionRuleVersion,
                DecisionRule.PROPERTY_TOPIA_ID,
                DecisionRule.PROPERTY_TOPIA_CREATE_DATE,
                DecisionRule.PROPERTY_VERSION_NUMBER,
                DecisionRule.PROPERTY_VERSION_REASON);
        newDecisionRuleVersion.setVersionReason(comment);
        newDecisionRuleVersion.setVersionNumber(decisionRule.getVersionNumber() + 1);

        DecisionRule created = decisionRuleTopiaDao.create(newDecisionRuleVersion);

        getTransaction().commit();

        return created;
    }

    @Override
    public List<ManagementModeCategory> getAvailableManagementModeCategories(String growingSystemId){
        List<ManagementModeCategory> usedManagementMode;
        if (!Strings.isNullOrEmpty(growingSystemId)) {
            usedManagementMode = managementModeTopiaDao.findManagementModeCategories(growingSystemId);
        } else {
            usedManagementMode = Lists.newArrayListWithCapacity(0);
        }
        List<ManagementModeCategory>  result = ListUtils.subtract(Arrays.asList(ManagementModeCategory.values()), usedManagementMode);
        return result;
    }

    @Override
    public List<ManagementMode> getRelatedManagementModes(GrowingSystem growingSystem) {
        List<ManagementMode> result = managementModeTopiaDao.forGrowingSystemEquals(growingSystem).
                setOrderByArguments(ManagementMode.PROPERTY_CATEGORY + " DESC").findAll();
        return result;
    }

    @Override
    public ManagementMode copyManagementMode(
            String growingSystemId,
            ManagementModeCategory category,
            String mainChangesFromPlanned,
            String changeReasonFromPlanned) {

        Preconditions.checkNotNull(growingSystemId);
        Preconditions.checkNotNull(category);

        GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemId).findUnique();
        ManagementMode newManagementMode = managementModeTopiaDao.newInstance();

        List<HistoryItem> histories = Lists.newArrayList();
        Gson gson = context.getGson();

        ManagementMode plannedManagementMode = managementModeTopiaDao.forGrowingSystemEquals(growingSystem).findUniqueOrNull();

        if (plannedManagementMode != null) {
            Binder<ManagementMode, ManagementMode> binder = BinderFactory.newBinder(ManagementMode.class);

            binder.copyExcluding(plannedManagementMode, newManagementMode,
                    ManagementMode.PROPERTY_TOPIA_ID,
                    ManagementMode.PROPERTY_TOPIA_CREATE_DATE,
                    ManagementMode.PROPERTY_CATEGORY,
                    ManagementMode.PROPERTY_SECTIONS,
                    ManagementMode.PROPERTY_HISTORICAL);

            newManagementMode.setCategory(ManagementModeCategory.OBSERVED);
            newManagementMode.setChangeReasonFromPlanned(changeReasonFromPlanned);
            newManagementMode.setMainChangesFromPlanned(mainChangesFromPlanned);

            // historic
            histories.add(createManagementModeCopyHistory(plannedManagementMode, newManagementMode));
            newManagementMode.setHistorical(gson.toJson(histories));
        }

        // clone section
        cloneSections(plannedManagementMode, newManagementMode);

        ManagementMode created = managementModeTopiaDao.create(newManagementMode);

        getTransaction().commit();

        return created;
    }

    @Override
    public List<GrowingSystem> getGrowingSystemsForManagementMode(NavigationContext navigationContext){
        List<GrowingSystem> result = growingSystemTopiaDao.getGrowingSystemsForManagementMode(navigationContext, getSecurityContext());
        return result;
    }

    @Override
    public List<GrowingSystem> getAvailableGsForDuplication(String growingSystemId, NavigationContext navigationContext) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(growingSystemId));

        GrowingSystem managementModeGrowingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemId).findUnique();
        // Dans un premier temps on filtre sur les systèmes de cultures sur lesquels on a le droit en écriture
        GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
        growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
        growingSystemFilter.setNavigationContext(navigationContext);
        ResultList<GrowingSystem> growingSystems = growingSystemTopiaDao.getFilteredGrowingSystems(growingSystemFilter, getSecurityContext());
        // dans un second temps on ne garde que les sdc pointant vers un domaine du même code que celui du mode de gestion
        //  et sur lesquels il n'y a pas de mode de gestion.
        List<GrowingSystem> result = growingSystemTopiaDao.findAllAvailableForManagementModeDuplication(growingSystems.getElements(), managementModeGrowingSystem);
        return result;
    }

    @Override
    public void unactivateDecisionRules(List<String> decisionRuleIds, boolean activate) {
        if (CollectionUtils.isNotEmpty(decisionRuleIds)) {

            for (String decisionRuleId : decisionRuleIds) {
                DecisionRule decisionRule = decisionRuleTopiaDao.forTopiaIdEquals(decisionRuleId).findUnique();

                // active/desactive toutes les regles qui ont le même code de duplication
                Collection<DecisionRule> decisionRules = decisionRuleTopiaDao.forCodeEquals(decisionRule.getCode()).findAll();
                for (DecisionRule decisionRule2 : decisionRules) {
                    decisionRule2.setActive(activate);
                    decisionRuleTopiaDao.update(decisionRule2);
                }
            }
            getTransaction().commit();
        }
    }

    @Override
    public DecisionRule duplicateDecisionRule(String decisionRuleId) {
        DecisionRule decisionRule = decisionRuleTopiaDao.forTopiaIdEquals(decisionRuleId).findUnique();
        DecisionRule duplicatedDecisionRule = duplicateDecisionRule(new ExtendContext(true), decisionRule);
        getTransaction().commit();
        return duplicatedDecisionRule;
    }

    @Override
    public ManagementMode duplicateManagementModes(String plannedManagementModeId, String observedManagementModeId, String duplicateManagementModeGrowingSystemId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(duplicateManagementModeGrowingSystemId));
        // If it exist the 'Planned Management Mode' is return otherwise it's the 'Observed' one.
        ManagementMode result = null;
        if(!Strings.isNullOrEmpty(observedManagementModeId)) {
            result = duplicateManagementMode(observedManagementModeId, duplicateManagementModeGrowingSystemId);
        }
        if(!Strings.isNullOrEmpty(plannedManagementModeId)) {
            result = duplicateManagementMode(plannedManagementModeId, duplicateManagementModeGrowingSystemId);
        }
        getTransaction().commit();

        return result;
    }

    @Override
    public List<DecisionRule> getAllDecisionRules(List<String> decisionRulesIds) {
        List<DecisionRule> result = decisionRuleTopiaDao.forIn(DecisionRule.PROPERTY_TOPIA_ID, decisionRulesIds).findAll();
        return result;
    }

    protected ManagementMode duplicateManagementMode(String managementModeId, String growingSystemId){
        ManagementMode originalManagementMode = managementModeTopiaDao.forTopiaIdEquals(managementModeId).findUnique();
        ManagementMode duplicatedManagementMode = managementModeTopiaDao.newInstance();
        Binder<ManagementMode, ManagementMode> managementModeManagementModeBinder = BinderFactory.newBinder(ManagementMode.class);
        managementModeManagementModeBinder.copyExcluding(originalManagementMode, duplicatedManagementMode,
                ManagementMode.PROPERTY_TOPIA_CREATE_DATE,
                ManagementMode.PROPERTY_TOPIA_ID,
                ManagementMode.PROPERTY_TOPIA_VERSION,
                ManagementMode.PROPERTY_GROWING_SYSTEM,
                ManagementMode.PROPERTY_SECTIONS,
                ManagementMode.PROPERTY_HISTORICAL
        );
        GrowingSystem growingSystem = growingSystemTopiaDao.forTopiaIdEquals(growingSystemId).findUnique();
        duplicatedManagementMode.setGrowingSystem(growingSystem);

        // clone section
        cloneSections(originalManagementMode, duplicatedManagementMode);

        ManagementMode result = managementModeTopiaDao.create(duplicatedManagementMode);
        return result;
    }

    protected DecisionRule duplicateDecisionRule(ExtendContext extendContext, DecisionRule decisionRule) {

        // perform clone
        Binder<DecisionRule, DecisionRule> decisionRuleBinder = BinderFactory.newBinder(DecisionRule.class);
        DecisionRule clonedDecisionRule = decisionRuleTopiaDao.newInstance();
        decisionRuleBinder.copyExcluding(decisionRule, clonedDecisionRule,
                DecisionRule.PROPERTY_TOPIA_ID,
                DecisionRule.PROPERTY_TOPIA_CREATE_DATE,
                DecisionRule.PROPERTY_TOPIA_VERSION);

        // if duplication, break code
        if (extendContext.isDuplicateOnly()) {
            clonedDecisionRule.setCode(UUID.randomUUID().toString());
            clonedDecisionRule.setVersionNumber(1);
        }

        // persist clone
        clonedDecisionRule = decisionRuleTopiaDao.create(clonedDecisionRule);

        return clonedDecisionRule;
    }

    private ManagementMode getLastManagementModeVersion(GrowingSystem growingSystem) {
        ManagementMode managementMode = managementModeTopiaDao.forGrowingSystemEquals(growingSystem).setOrderByArguments(ManagementMode.PROPERTY_VERSION_NUMBER + " DESC").findFirst();
        return managementMode;
    }

    @Override
    public InputStream exportManagementModesAsXlsStream(Collection<ManagementModeDto> managementModeDtos) {

        // contains all tabs
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        ManagementModeMainBeanInfo mainTab = new ManagementModeMainBeanInfo();
        ManagementModeSectionBeanInfo sectionTab = new ManagementModeSectionBeanInfo();

        ExportUtils.addAllBeanInfo(sheet, mainTab, sectionTab);

        List<ManagementModeExportEntity> sectionEntities = (List<ManagementModeExportEntity>)sheet.get(sectionTab);

        try {
            if (CollectionUtils.isNotEmpty(managementModeDtos)) {
                for (ManagementModeDto managementModeDto : managementModeDtos) {
    
                    if (StringUtils.isNotBlank(managementModeDto.getPlannedManagementModeId())) {
                        ManagementMode plannedManagementMode = managementModeTopiaDao.forTopiaIdEquals(managementModeDto.getPlannedManagementModeId()).findUnique();
    
                        // anonymize management mode
                        plannedManagementMode = anonymizeService.checkForManagementModeAnonymization(plannedManagementMode);
    
                        exportManagementMode(sheet, mainTab, sectionEntities, plannedManagementMode);
                    }
    
                    if (StringUtils.isNotBlank(managementModeDto.getObservedManagementModeId())) {
                        ManagementMode observedManagementMode = managementModeTopiaDao.forTopiaIdEquals(managementModeDto.getObservedManagementModeId()).findUnique();
    
                        // anonymize management mode
                        observedManagementMode = anonymizeService.checkForManagementModeAnonymization(observedManagementMode);
    
                        exportManagementMode(sheet, mainTab, sectionEntities, observedManagementMode);
                    }
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }
        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    protected void exportManagementMode(Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet, ManagementModeMainBeanInfo mainTab, List<ManagementModeExportEntity> sectionEntities, ManagementMode managementMode) throws CloneNotSupportedException, IllegalAccessException, NoSuchMethodException, java.lang.reflect.InvocationTargetException {
        // common filds
        ManagementModeExportEntity model = new ManagementModeExportEntity();
        model.setCategory(managementMode.getCategory());
        model.setGrowingSystemName(managementMode.getGrowingSystem().getName());
        model.setGrowingPlanName(managementMode.getGrowingSystem().getGrowingPlan().getName());
        model.setDomainName(managementMode.getGrowingSystem().getGrowingPlan().getDomain().getName());
        model.setCampaign(managementMode.getGrowingSystem().getGrowingPlan().getDomain().getCampaign());

        //main tab
        ExportUtils.export(sheet, model, managementMode, mainTab);

        //sections
        Collection<Section> sections = managementMode.getSections();
        if (sections != null && !sections.isEmpty()) {
            for (Section section : sections) {
                List<Strategy> strategies = section.getStrategies();
                // strategies
                if (strategies != null && !strategies.isEmpty()) {
                    for (Strategy strategy : strategies) {
                        ManagementModeExportEntity export = (ManagementModeExportEntity) model.clone();

                        ExportUtils.copyFields(section, model, null,
                                Section.PROPERTY_SECTION_TYPE,
                                Section.PROPERTY_AGRONOMIC_OBJECTIVE,
                                Section.PROPERTY_CATEGORY_OBJECTIVE,
                                Section.PROPERTY_EXPECTED_RESULT,
                                Section.PROPERTY_BIO_AGRESSOR,
                                Section.PROPERTY_CATEGORY_STRATEGY);

                        RefBioAgressor bioAgressor = section.getBioAgressor();
                        String bioagrssorLabel = null;
                        if (bioAgressor != null){
                            bioagrssorLabel = bioAgressor.getLabel();
                        }
                        ExportUtils.setExtraField(export, Section.PROPERTY_BIO_AGRESSOR_TYPE, bioagrssorLabel);

                        ExportUtils.setExtraField(export, "StrategyMultiannual", strategy.getStrategyType());
                        ExportUtils.setExtraField(export, "StrategyExplanation", strategy.getExplanation());
                        String cropName = null;
                        CroppingPlanEntry crop = strategy.getCroppingPlanEntry();
                        if (crop != null) {
                            cropName = crop.getName();
                        }
                        ExportUtils.setExtraField(export, "StrategyCroppingPlanEntry", cropName);
                        sectionEntities.add(export);
                    }

                } else {
                    ManagementModeExportEntity export = (ManagementModeExportEntity) model.clone();

                    ExportUtils.copyFields(section, model, null,
                            Section.PROPERTY_SECTION_TYPE,
                            Section.PROPERTY_AGRONOMIC_OBJECTIVE,
                            Section.PROPERTY_CATEGORY_OBJECTIVE,
                            Section.PROPERTY_EXPECTED_RESULT,
                            Section.PROPERTY_BIO_AGRESSOR,
                            Section.PROPERTY_CATEGORY_STRATEGY);

                    RefBioAgressor bioAgressor = section.getBioAgressor();
                    String bioagrssorLabel = null;
                    if (bioAgressor != null){
                        bioagrssorLabel = bioAgressor.getLabel();
                    }
                    ExportUtils.setExtraField(export, Section.PROPERTY_BIO_AGRESSOR_TYPE, bioagrssorLabel);
                    sectionEntities.add(export);
                }
            }
        }
    }

    @Override
    public InputStream exportDecisionRulesAsXlsStream(Collection<String> decisionRuleIds) {

        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        DecisionRuleMainBeanInfo decisionRuleMainBeanInfo = newInstance(DecisionRuleMainBeanInfo.class);

        ExportUtils.addAllBeanInfo(sheet, decisionRuleMainBeanInfo);
        try {
            if (decisionRuleIds != null && !decisionRuleIds.isEmpty()) {
                Iterable<DecisionRule> decisionRules = decisionRuleTopiaDao.forTopiaIdIn(decisionRuleIds).findAllLazy(100);
                for (DecisionRule decisionRule:decisionRules) {
                    DecisionRuleExportEntity model = new DecisionRuleExportEntity();
                    Domain domain = domainTopiaDao.forCodeEquals(decisionRule.getDomainCode()).findAny();
                    domain = anonymizeService.checkForDomainAnonymization(domain);
                    model.setDomainName(domain.getName());

                    // main tab
                    ExportUtils.export(sheet, model, decisionRule, decisionRuleMainBeanInfo);
                }
            }

        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }
        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }
}
