package fr.inra.agrosyst.services.input;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: InputServiceImpl.java 5120 2015-10-29 15:21:51Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/input/InputServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.AbstractInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.BiologicalControlAction;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInput;
import fr.inra.agrosyst.api.entities.action.BiologicalProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.HarvestingAction;
import fr.inra.agrosyst.api.entities.action.IrrigationAction;
import fr.inra.agrosyst.api.entities.action.MaintenancePruningVinesAction;
import fr.inra.agrosyst.api.entities.action.MineralFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.OrganicFertilizersSpreadingAction;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.OtherAction;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.OtherProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.PesticideProductInput;
import fr.inra.agrosyst.api.entities.action.PesticideProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.PesticidesSpreadingAction;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.SeedingAction;
import fr.inra.agrosyst.api.entities.action.SeedingActionSpecies;
import fr.inra.agrosyst.api.entities.action.SeedingProductInput;
import fr.inra.agrosyst.api.entities.action.SeedingProductInputTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.referential.RefBioAgressor;
import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.common.PricesService;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.input.InputService;
import fr.inra.agrosyst.api.services.practiced.DuplicateCropCyclesContext;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.commons.gson.InputAdapter;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author David Cossé
 */
public class InputServiceImpl extends AbstractAgrosystService implements InputService {

    protected PricesService pricesService;
    protected ReferentialService referentialService;
    protected AbstractInputTopiaDao inputDao;
    protected MineralProductInputTopiaDao mineralProductInputDao;
    protected OrganicProductInputTopiaDao organicProductInputDao;
    protected SeedingProductInputTopiaDao seedingProductInputDao;
    protected BiologicalProductInputTopiaDao biologicalProductInputDao;
    protected PesticideProductInputTopiaDao pesticideProductInputDao;
    protected OtherProductInputTopiaDao otherProductInputDao;

    protected Map<AgrosystInterventionType, List<String>> actaTreatmentProductTypes;


    public void setPricesService(PricesService pricesService) {
        this.pricesService = pricesService;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;

    }

    protected Map<AgrosystInterventionType, List<String>> getActaTreatmentProductTypes() {
        if (actaTreatmentProductTypes == null || actaTreatmentProductTypes.isEmpty()) {
            actaTreatmentProductTypes = referentialService.getAllActiveActaTreatmentProductTypes();
        }
        return actaTreatmentProductTypes;
    }

    public void setAbstractInputTopiaDao(AbstractInputTopiaDao inputTopiaDao) {
        this.inputDao = inputTopiaDao;
    }

    public void setMineralProductInputDao(MineralProductInputTopiaDao mineralProductInputDao) {
        this.mineralProductInputDao = mineralProductInputDao;
    }

    public void setOrganicProductInputDao(OrganicProductInputTopiaDao organicProductInputDao) {
        this.organicProductInputDao = organicProductInputDao;
    }

    public void setSeedingProductInputDao(SeedingProductInputTopiaDao seedingProductInputDao) {
        this.seedingProductInputDao = seedingProductInputDao;
    }

    public void setBiologicalProductInputDao(BiologicalProductInputTopiaDao biologicalProductInputDao) {
        this.biologicalProductInputDao = biologicalProductInputDao;
    }

    public void setPesticideProductInputDao(PesticideProductInputTopiaDao pesticideProductInputDao) {
        this.pesticideProductInputDao = pesticideProductInputDao;
    }

    public void setOtherProductInputDao(OtherProductInputTopiaDao otherProductInputDao) {
        this.otherProductInputDao = otherProductInputDao;
    }

    public void setInputDao(AbstractInputTopiaDao inputDao) {
        this.inputDao = inputDao;
    }

    @Override
    public void updateInterventionInputs(PracticedIntervention practicedIntervention, EffectiveIntervention effectiveIntervention, List<AbstractInput> inputs, Map<String, AbstractAction> actions) {
        Collection<AbstractInput> originalInputs;
        if (practicedIntervention != null) {
            originalInputs = inputDao.findAllByPracticedIntervention(practicedIntervention);
        } else if (effectiveIntervention != null) {
            originalInputs = inputDao.findAllByEffectiveIntervention(effectiveIntervention);
        } else {
            throw new UnsupportedOperationException("At least a practiced of effective intervention must be provided");
        }
        createOrUpdateInterventionInputs(originalInputs, inputs, actions);
    }

    @Override
    public void createInterventionInputs(PracticedIntervention practicedIntervention, EffectiveIntervention effectiveIntervention, List<AbstractInput> inputs, Map<String, AbstractAction> actions) {
        createOrUpdateInterventionInputs(null, inputs, actions);
    }

    protected void createOrUpdateInterventionInputs(
            Collection<AbstractInput> originalInputs,
            List<AbstractInput> inputs,
            Map<String, AbstractAction> actions) {

        if (originalInputs == null) {
            originalInputs = Lists.newArrayList();
        }

        ImmutableMap<String,AbstractInput> inputsIndex = Maps.uniqueIndex(originalInputs, Entities.GET_TOPIA_ID);
        Map<String, AbstractInput> indexedOriginalInputs = Maps.newHashMap(inputsIndex);

        if (inputs != null) {
            try {
                for (AbstractInput input : inputs) {
                    AgrosystInterventionType inputType = InputAdapter.GET_INPUT_TYPE.apply(input);
                    input.setInputType(inputType);

                    if (input instanceof MineralProductInput) {
                        MineralProductInput mineralProductInput = (MineralProductInput) input;
                        RefFertiMinUNIFA product = mineralProductInput.getMineralProduct();
                        Preconditions.checkState(product != null, "A mineralProduct is required for 'MineralProductInput'");
                        mineralProductInput.setMineralProduct(referentialService.createOrUpdateRefMineralProductToInput(product));

                        Preconditions.checkState(mineralProductInput.getMineralFertilizersSpreadingAction() != null, "Aucune action associée à l'intrant de type 'APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX'");
                        AbstractAction action = actions.get(mineralProductInput.getMineralFertilizersSpreadingAction().getTopiaId());

                        Preconditions.checkState(action != null, "Aucune action associée à l'intrant de type 'APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX'");
                        mineralProductInput.setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);
                        if (mineralProductInput.isPersisted()) {
                            MineralProductInput modifiedMineralProductInput = bindMineralProductInputToPersistedOne(indexedOriginalInputs, mineralProductInput);
                            mineralProductInputDao.update(modifiedMineralProductInput);
                        } else {
                            mineralProductInputDao.create(mineralProductInput);
                        }

                    } else if (input instanceof OrganicProductInput) {
                        OrganicProductInput organicProductInput = (OrganicProductInput) input;
                        Preconditions.checkState(organicProductInput.getOrganicFertilizersSpreadingAction() != null, "Aucune action associée à l'intrant de type 'EPANDAGES_ORGANIQUES'");
                        AbstractAction action = actions.get(organicProductInput.getOrganicFertilizersSpreadingAction().getTopiaId());
                        Preconditions.checkState(action != null, "Aucune action associée à l'intrant de type 'EPANDAGES_ORGANIQUES'");

                        organicProductInput.setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);
                        if (organicProductInput.isPersisted()) {
                            OrganicProductInput modifiedOrganicProductInput = bindOrganicProductInputToPersitedOne(indexedOriginalInputs, organicProductInput);
                            organicProductInputDao.update(modifiedOrganicProductInput);
                        } else {
                            organicProductInputDao.create(organicProductInput);
                        }

                    } else if (AgrosystInterventionType.SEMIS.equals(inputType) && input instanceof SeedingProductInput) {
                        SeedingProductInput currentSeedingProductInput = (SeedingProductInput) input;
                        Preconditions.checkState(currentSeedingProductInput.getSeedingAction() != null, "Aucune action associée à l'intrant de type 'SEMIS'");

                        SeedingAction seedingAction = (SeedingAction) actions.get(currentSeedingProductInput.getSeedingAction().getTopiaId());
                        Preconditions.checkState(seedingAction != null, "Aucune action associée à l'intrant de type 'SEMIS'");
                        validSeedingInputSeedingAction(currentSeedingProductInput, seedingAction);

                        SeedingProductInput seedingProductInput = bindSeedingProductInput(currentSeedingProductInput, indexedOriginalInputs, seedingAction);

                        if (seedingProductInput.isPersisted()) {
                            seedingProductInputDao.update(seedingProductInput);
                        } else {
                            seedingProductInputDao.create(seedingProductInput);
                        }
                    } else if (AgrosystInterventionType.LUTTE_BIOLOGIQUE.equals(inputType) && input instanceof BiologicalProductInput) {
                        BiologicalProductInput phytoProductInput = (BiologicalProductInput) input;
                        Preconditions.checkState(phytoProductInput.getBiologicalControlAction() != null, "Aucune action associée à l'intrant de type 'LUTTE_BIOLOGIQUE'");

                        AbstractAction action = actions.get(phytoProductInput.getBiologicalControlAction().getTopiaId());
                        Preconditions.checkState(action != null, "Aucune action associée à l'intrant de type 'LUTTE_BIOLOGIQUE'");
                        if (phytoProductInput.isPersisted()) {
                            BiologicalProductInput biologicalProductInput;
                            biologicalProductInput = getPersistedBiologicalProductInput(indexedOriginalInputs, phytoProductInput);
                            bindBiologicalProductInput((BiologicalControlAction) action, phytoProductInput, biologicalProductInput);
                            biologicalProductInputDao.update(biologicalProductInput);
                        } else {
                            phytoProductInput.setBiologicalControlAction((BiologicalControlAction) action);
                            biologicalProductInputDao.create(phytoProductInput);
                        }

                    } else if (AgrosystInterventionType.APPLICATION_DE_PRODUITS_PHYTOSANITAIRES.equals(inputType) && input instanceof PesticideProductInput) {
                        PesticideProductInput phytoProductInput = (PesticideProductInput) input;
                        Preconditions.checkState(phytoProductInput.getPesticidesSpreadingAction() != null, "Aucune action associée à l'intrant de type 'APPLICATION_DE_PRODUITS_PHYTOSANITAIRES'");
                        AbstractAction action = actions.get(phytoProductInput.getPesticidesSpreadingAction().getTopiaId());
                        Preconditions.checkState(action != null, "Aucune action associée à l'intrant de type 'APPLICATION_DE_PRODUITS_PHYTOSANITAIRES'");
                        if (phytoProductInput.isPersisted()) {
                            PesticideProductInput pesticideProductInput = getPersistedPesticideProductInput(indexedOriginalInputs, phytoProductInput);
                            bindToPesticideProductInput((PesticidesSpreadingAction) action, phytoProductInput, pesticideProductInput);
                            pesticideProductInputDao.update(pesticideProductInput);
                        } else {
                            phytoProductInput.setPesticidesSpreadingAction((PesticidesSpreadingAction) action);
                            pesticideProductInputDao.create(phytoProductInput);

                        }
                    } else if (input instanceof OtherProductInput) {
                        OtherProductInput otherProductInput = (OtherProductInput) input;

                        boolean isAction = false;
                        if (otherProductInput.getOtherAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getOtherAction().getTopiaId());
                            otherProductInput.setOtherAction((OtherAction) action);
                            isAction = true;
                        } else if (otherProductInput.getHarvestingAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getHarvestingAction().getTopiaId());
                            otherProductInput.setHarvestingAction((HarvestingAction) action);
                            isAction = true;
                        } else if (otherProductInput.getMaintenancePruningVinesAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getMaintenancePruningVinesAction().getTopiaId());
                            otherProductInput.setMaintenancePruningVinesAction((MaintenancePruningVinesAction) action);
                            isAction = true;
                        }  else if (otherProductInput.getIrrigationAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getIrrigationAction().getTopiaId());
                            otherProductInput.setIrrigationAction((IrrigationAction) action);
                            isAction = true;
                        }  else if (otherProductInput.getMineralFertilizersSpreadingAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getMineralFertilizersSpreadingAction().getTopiaId());
                            otherProductInput.setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) action);
                            isAction = true;
                        }  else if (otherProductInput.getOrganicFertilizersSpreadingAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getOrganicFertilizersSpreadingAction().getTopiaId());
                            otherProductInput.setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) action);
                            isAction = true;
                        }  else if (otherProductInput.getPesticidesSpreadingAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getPesticidesSpreadingAction().getTopiaId());
                            otherProductInput.setPesticidesSpreadingAction((PesticidesSpreadingAction) action);
                            isAction = true;
                        }  else if (otherProductInput.getBiologicalControlAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getBiologicalControlAction().getTopiaId());
                            otherProductInput.setBiologicalControlAction((BiologicalControlAction) action);
                            isAction = true;
                        } else if (otherProductInput.getSeedingAction() != null) {
                            AbstractAction action = actions.get(otherProductInput.getSeedingAction().getTopiaId());
                            otherProductInput.setSeedingAction((SeedingAction) action);
                            isAction = true;
                        }

                        Preconditions.checkState(isAction, "Aucune action associée à l'intrant de type 'Autre'");

                        Preconditions.checkState(StringUtils.isNotBlank(otherProductInput.getProductName()), "Aucun nom de produit n'est définit pour l'intrant de type 'AUTRE'");

                        if (otherProductInput.isPersisted()) {
                            OtherProductInput modifiedOtherProductInput = bindToPersistedOtherProductInput(indexedOriginalInputs, otherProductInput);
                            otherProductInputDao.update(modifiedOtherProductInput);
                        } else {
                            otherProductInputDao.create(otherProductInput);
                        }
                    } else {
                        throw new UnsupportedOperationException("Unsupported input type: " + input.getClass());
                    }
                }
            } catch (Exception e) {
                throw new AgrosystTechnicalException("Echec de sauvegarde des intrants", e);
            }
        }
        inputDao.deleteAll(indexedOriginalInputs.values());
    }

    protected OtherProductInput bindToPersistedOtherProductInput(Map<String, AbstractInput> indexedOriginalInputs, OtherProductInput otherProductInput) {
        OtherProductInput modifiedOtherProductInput = (OtherProductInput) indexedOriginalInputs.remove(otherProductInput.getTopiaId());
        if (modifiedOtherProductInput == null) {
            modifiedOtherProductInput = otherProductInputDao.forTopiaIdEquals(otherProductInput.getTopiaId()).findUnique();
        }
        Binder<OtherProductInput, OtherProductInput> binder = BinderFactory.newBinder(OtherProductInput.class);
        binder.copyExcluding(otherProductInput, modifiedOtherProductInput,
                OtherProductInput.PROPERTY_TOPIA_ID,
                OtherProductInput.PROPERTY_TOPIA_CREATE_DATE,
                OtherProductInput.PROPERTY_TOPIA_VERSION);
        return modifiedOtherProductInput;
    }

    protected void bindToPesticideProductInput(PesticidesSpreadingAction action, PesticideProductInput phytoProductInput, PesticideProductInput pesticideProductInput) {
        Binder<PesticideProductInput, PesticideProductInput> binder = BinderFactory.newBinder(PesticideProductInput.class, PesticideProductInput.class);
        binder.copyExcluding(phytoProductInput, pesticideProductInput,
                PhytoProductInput.PROPERTY_TOPIA_ID,
                PhytoProductInput.PROPERTY_TOPIA_CREATE_DATE,
                PhytoProductInput.PROPERTY_TOPIA_VERSION);
        pesticideProductInput.setPesticidesSpreadingAction(action);
    }

    protected PesticideProductInput getPersistedPesticideProductInput(Map<String, AbstractInput> indexedOriginalInputs, PhytoProductInput phytoProductInput) {
        PesticideProductInput pesticideProductInput;
        pesticideProductInput = (PesticideProductInput) indexedOriginalInputs.remove(phytoProductInput.getTopiaId());
        if (pesticideProductInput == null) {
            pesticideProductInput = pesticideProductInputDao.forTopiaIdEquals(phytoProductInput.getTopiaId()).findUnique();
        }
        return pesticideProductInput;
    }

    protected void bindBiologicalProductInput(BiologicalControlAction action, BiologicalProductInput phytoProductInput, BiologicalProductInput biologicalProductInput) {
        Binder<BiologicalProductInput, BiologicalProductInput> binder = BinderFactory.newBinder(BiologicalProductInput.class, BiologicalProductInput.class);
        binder.copyExcluding(phytoProductInput, biologicalProductInput,
                PhytoProductInput.PROPERTY_TOPIA_ID,
                PhytoProductInput.PROPERTY_TOPIA_CREATE_DATE,
                PhytoProductInput.PROPERTY_TOPIA_VERSION);
        biologicalProductInput.setBiologicalControlAction(action);
    }

    protected BiologicalProductInput getPersistedBiologicalProductInput(Map<String, AbstractInput> indexedOriginalInputs, PhytoProductInput phytoProductInput) {
        BiologicalProductInput biologicalProductInput;
        biologicalProductInput = (BiologicalProductInput) indexedOriginalInputs.remove(phytoProductInput.getTopiaId());
        if (biologicalProductInput == null) {
            biologicalProductInput = biologicalProductInputDao.forTopiaIdEquals(phytoProductInput.getTopiaId()).findUnique();
        }
        return biologicalProductInput;
    }

    protected OrganicProductInput bindOrganicProductInputToPersitedOne(Map<String, AbstractInput> indexedOriginalInputs, OrganicProductInput organicProductInput) {
        OrganicProductInput modifiedOrganicProductInput = (OrganicProductInput) indexedOriginalInputs.remove(organicProductInput.getTopiaId());
        if (modifiedOrganicProductInput == null) {
            modifiedOrganicProductInput = organicProductInputDao.forTopiaIdEquals(organicProductInput.getTopiaId()).findUnique();
        }
        Binder<OrganicProductInput, OrganicProductInput> binder = BinderFactory.newBinder(OrganicProductInput.class);
        binder.copyExcluding(organicProductInput, modifiedOrganicProductInput,
                OrganicProductInput.PROPERTY_TOPIA_ID,
                OrganicProductInput.PROPERTY_TOPIA_CREATE_DATE,
                OrganicProductInput.PROPERTY_TOPIA_VERSION);
        return modifiedOrganicProductInput;
    }

    protected MineralProductInput bindMineralProductInputToPersistedOne(Map<String, AbstractInput> indexedOriginalInputs, MineralProductInput mineralProductInput) {
        MineralProductInput modifiedMineralProductInput = (MineralProductInput) indexedOriginalInputs.remove(mineralProductInput.getTopiaId());
        if (modifiedMineralProductInput == null) {
            modifiedMineralProductInput = mineralProductInputDao.forTopiaIdEquals(mineralProductInput.getTopiaId()).findUnique();
        }
        Binder<MineralProductInput, MineralProductInput> binder = BinderFactory.newBinder(MineralProductInput.class);
        binder.copyExcluding(mineralProductInput, modifiedMineralProductInput,
                MineralProductInput.PROPERTY_TOPIA_ID,
                MineralProductInput.PROPERTY_TOPIA_CREATE_DATE,
                MineralProductInput.PROPERTY_TOPIA_VERSION);

        return modifiedMineralProductInput;
    }

    protected SeedingProductInput bindSeedingProductInput(SeedingProductInput seedingProductInput, Map<String, AbstractInput> indexedOriginalInputs, SeedingAction seedingAction) {
        SeedingProductInput inputToSave;
        if (seedingProductInput.isPersisted()) {
            inputToSave = (SeedingProductInput) indexedOriginalInputs.remove(seedingProductInput.getTopiaId());
            if (inputToSave == null) {
                inputToSave = seedingProductInputDao.forTopiaIdEquals(seedingProductInput.getTopiaId()).findUnique();
            }
        } else {
            inputToSave = seedingProductInputDao.newInstance();
        }
        Binder<SeedingProductInput, SeedingProductInput> binder = BinderFactory.newBinder(SeedingProductInput.class, SeedingProductInput.class);
        binder.copyExcluding(seedingProductInput, inputToSave,
                PhytoProductInput.PROPERTY_TOPIA_ID,
                PhytoProductInput.PROPERTY_TOPIA_CREATE_DATE,
                PhytoProductInput.PROPERTY_TOPIA_VERSION);
        inputToSave.setSeedingAction(seedingAction);
        return inputToSave;
    }

    protected void validSeedingInputSeedingAction(SeedingProductInput input, SeedingAction seedingAction) {
        Collection<SeedingActionSpecies> specieses = seedingAction.getSeedingSpecies();
        Preconditions.checkState(CollectionUtils.isNotEmpty(specieses), String.format("Aucune espèce de l'action 'Semis' n'a de traitement sélectionné pour ce type d'intrant '%s' !", input.getProductType()));

        if (StringUtils.isNotBlank(input.getProductType())) {
            List<String> productTypes = getActaTreatmentProductTypes().get(AgrosystInterventionType.SEMIS);
            Set<String> allowedProductTypes = Sets.newHashSet();

            for (SeedingActionSpecies species : specieses) {
                if (species.isTreatment()) {
                    //Fongicides
                    allowedProductTypes.add(productTypes.get(0));
                    //Insecticides
                    allowedProductTypes.add(productTypes.get(2));
                }
                if (species.isBiologicalSeedInoculation()) {
                    // "Inoculation biologique de semences ou plants"
                    allowedProductTypes.add(productTypes.get(1));
                }
            }

            Preconditions.checkState(allowedProductTypes.contains(input.getProductType()), String.format("Aucune espèce de l'action 'Semis' n'a de traitement sélectionné pour ce type d'intrant '%s' !", input.getProductType()));
        }
    }

    @Override
    public void deleteInputForActions(Collection<AbstractAction> actions) {
        if (actions != null) {
            for (AbstractAction action : actions) {
                if (action instanceof SeedingAction) {
                    List<SeedingProductInput> seedingProductInputs = seedingProductInputDao.forSeedingActionEquals((SeedingAction) action).findAll();
                    seedingProductInputDao.deleteAll(seedingProductInputs);
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forSeedingActionEquals((SeedingAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof MineralFertilizersSpreadingAction) {
                    List<MineralProductInput> mineralProductInputs = mineralProductInputDao.forMineralFertilizersSpreadingActionEquals((MineralFertilizersSpreadingAction) action).findAll();
                    mineralProductInputDao.deleteAll(mineralProductInputs);
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forMineralFertilizersSpreadingActionEquals((MineralFertilizersSpreadingAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof OrganicFertilizersSpreadingAction) {
                    List<OrganicProductInput> organicProductInputs = organicProductInputDao.forOrganicFertilizersSpreadingActionEquals((OrganicFertilizersSpreadingAction) action).findAll();
                    organicProductInputDao.deleteAll(organicProductInputs);
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forOrganicFertilizersSpreadingActionEquals((OrganicFertilizersSpreadingAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof BiologicalControlAction) {
                    List<BiologicalProductInput> biologicalProductInputs = biologicalProductInputDao.forBiologicalControlActionEquals((BiologicalControlAction) action).findAll();
                    biologicalProductInputDao.deleteAll(biologicalProductInputs);
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forBiologicalControlActionEquals((BiologicalControlAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof PesticidesSpreadingAction) {
                    List<PesticideProductInput> pesticideProductInputs = pesticideProductInputDao.forPesticidesSpreadingActionEquals((PesticidesSpreadingAction) action).findAll();
                    pesticideProductInputDao.deleteAll(pesticideProductInputs);
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forPesticidesSpreadingActionEquals((PesticidesSpreadingAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof OtherAction) {
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forOtherActionEquals((OtherAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof IrrigationAction) {
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forIrrigationActionEquals((IrrigationAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof MaintenancePruningVinesAction) {
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forMaintenancePruningVinesActionEquals((MaintenancePruningVinesAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                } else if (action instanceof HarvestingAction) {
                    List<OtherProductInput> otherProductInputs = otherProductInputDao.forHarvestingActionEquals((HarvestingAction) action).findAll();
                    otherProductInputDao.deleteAll(otherProductInputs);
                }
            }
        }
    }

    @Override
    public void duplicateInputs(DuplicateCropCyclesContext duplicateContext, PracticedIntervention practicedIntervention,
                                EffectiveIntervention effectiveIntervention) {
        Preconditions.checkArgument(practicedIntervention != null ^ effectiveIntervention != null);

        List<AbstractInput> inputs;
        if (effectiveIntervention != null) {
            inputs = inputDao.findAllByEffectiveIntervention(effectiveIntervention);
        } else {
            inputs = inputDao.findAllByPracticedIntervention(practicedIntervention);
        }

        Map<AbstractInput, AbstractInput> inputCache = Maps.newHashMap();
        for (AbstractInput input : inputs) {
            Map<AbstractAction, AbstractAction> actionsToDuplicatedActions = duplicateContext.getActionCache();
            AbstractInput inputClone = getDuplicatedAbstractInput(actionsToDuplicatedActions, input);

            // persist it
            inputClone = inputDao.create(inputClone);

            // save it into input clone cache
            inputCache.put(input, inputClone);

            // add duplicated input in cache to then duplicate prices
            duplicateContext.getInputCache().put(input, inputClone);
        }
    }

    @Override
    public void migrateInputSpeciesToTargetedSpecies(EffectiveInterventionDto interventionDto) {
        List<AbstractInput> newInputs = Lists.newArrayList();
        Collection<AbstractInput> inputs = interventionDto.getInputs();
        if (inputs != null) {
            for (AbstractInput input : inputs) {
                if (input instanceof SeedingProductInput){
                    SeedingAction inputAction = ((SeedingProductInput) input).getSeedingAction();
                    if (inputAction.getSeedingSpecies() != null && !inputAction.getSeedingSpecies().isEmpty()) {
                        newInputs.add(input);
                    }
                } else {
                    newInputs.add(input);
                }
            }
        }
        interventionDto.setInputs(newInputs);
    }

    @Override
    public AbstractInput getDuplicatedAbstractInput(Map<AbstractAction, AbstractAction> actionsToDuplicatedActions, AbstractInput input) {
        AbstractInput inputClone;

        if (input instanceof MineralProductInput) {
            MineralProductInput mineralProductInput = (MineralProductInput) input;
            MineralProductInput duplicateMineralProductInput = mineralProductInputDao.newInstance();
            Binder<MineralProductInput, MineralProductInput> binder = BinderFactory.newBinder(MineralProductInput.class);
            binder.copyExcluding(mineralProductInput, duplicateMineralProductInput,
                    MineralProductInput.PROPERTY_TOPIA_ID,
                    MineralProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    MineralProductInput.PROPERTY_TOPIA_VERSION,
                    MineralProductInput.PROPERTY_MINERAL_FERTILIZERS_SPREADING_ACTION);
            duplicateMineralProductInput.setMineralFertilizersSpreadingAction(
                    (MineralFertilizersSpreadingAction) actionsToDuplicatedActions.get(mineralProductInput.getMineralFertilizersSpreadingAction()));
            inputClone = duplicateMineralProductInput;

        } else if (input instanceof OrganicProductInput) {
            OrganicProductInput organicProductInput = (OrganicProductInput) input;
            OrganicProductInput newOrganicProductInput = organicProductInputDao.newInstance();
            Binder<OrganicProductInput, OrganicProductInput> binder = BinderFactory.newBinder(OrganicProductInput.class);
            binder.copyExcluding(organicProductInput, newOrganicProductInput,
                    OrganicProductInput.PROPERTY_TOPIA_ID,
                    OrganicProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    OrganicProductInput.PROPERTY_TOPIA_VERSION,
                    OrganicProductInput.PROPERTY_ORGANIC_FERTILIZERS_SPREADING_ACTION);
            newOrganicProductInput.setOrganicFertilizersSpreadingAction(
                    (OrganicFertilizersSpreadingAction) actionsToDuplicatedActions.get(organicProductInput.getOrganicFertilizersSpreadingAction()));
            inputClone = newOrganicProductInput;

        } else if (input instanceof SeedingProductInput) {
            SeedingProductInput phytoProductInput = (SeedingProductInput) input;
            SeedingProductInput newPhytoProductInput = seedingProductInputDao.newInstance();
            Binder<SeedingProductInput, SeedingProductInput> binder = BinderFactory.newBinder(SeedingProductInput.class);
            binder.copyExcluding(phytoProductInput, (SeedingProductInput)newPhytoProductInput,
                    SeedingProductInput.PROPERTY_TOPIA_ID,
                    SeedingProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    SeedingProductInput.PROPERTY_TOPIA_VERSION,
                    SeedingProductInput.PROPERTY_SEEDING_ACTION,
                    SeedingProductInput.PROPERTY_TARGETS);
            newPhytoProductInput.setSeedingAction(
                    (SeedingAction) actionsToDuplicatedActions.get(phytoProductInput.getSeedingAction()));
            if (phytoProductInput.getTargets() != null) {
                newPhytoProductInput.setTargets(new ArrayList<RefBioAgressor>(phytoProductInput.getTargets()));
            }
            inputClone = newPhytoProductInput;

        } else if (input instanceof BiologicalProductInput) {
            BiologicalProductInput phytoProductInput = (BiologicalProductInput) input;
            BiologicalProductInput newPhytoProductInput = biologicalProductInputDao.newInstance();
            Binder<BiologicalProductInput, BiologicalProductInput> binder = BinderFactory.newBinder(BiologicalProductInput.class);
            binder.copyExcluding(phytoProductInput, newPhytoProductInput,
                    BiologicalProductInput.PROPERTY_TOPIA_ID,
                    BiologicalProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    BiologicalProductInput.PROPERTY_TOPIA_VERSION,
                    BiologicalProductInput.PROPERTY_BIOLOGICAL_CONTROL_ACTION,
                    BiologicalProductInput.PROPERTY_TARGETS);
            newPhytoProductInput.setBiologicalControlAction(
                    (BiologicalControlAction) actionsToDuplicatedActions.get(phytoProductInput.getBiologicalControlAction()));
            if (phytoProductInput.getTargets() != null) {
                newPhytoProductInput.setTargets(new ArrayList<RefBioAgressor>(phytoProductInput.getTargets()));
            }
            inputClone = newPhytoProductInput;

        } else if (input instanceof PesticideProductInput) {
            PesticideProductInput phytoProductInput = (PesticideProductInput) input;
            PesticideProductInput newPhytoProductInput = pesticideProductInputDao.newInstance();
            Binder<PesticideProductInput, PesticideProductInput> binder = BinderFactory.newBinder(PesticideProductInput.class);
            binder.copyExcluding(phytoProductInput, newPhytoProductInput,
                    PesticideProductInput.PROPERTY_TOPIA_ID,
                    PesticideProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    PesticideProductInput.PROPERTY_TOPIA_VERSION,
                    PesticideProductInput.PROPERTY_PESTICIDES_SPREADING_ACTION,
                    PesticideProductInput.PROPERTY_TARGETS);
            newPhytoProductInput.setPesticidesSpreadingAction(
                    (PesticidesSpreadingAction) actionsToDuplicatedActions.get(phytoProductInput.getPesticidesSpreadingAction()));
            if (phytoProductInput.getTargets() != null) {
                newPhytoProductInput.setTargets(new ArrayList<RefBioAgressor>(phytoProductInput.getTargets()));
            }
            inputClone = newPhytoProductInput;

        } else if (input instanceof OtherProductInput) {
            OtherProductInput otherProductInput = (OtherProductInput) input;
            OtherProductInput newOtherProductInput = otherProductInputDao.newInstance();
            Binder<OtherProductInput, OtherProductInput> binder = BinderFactory.newBinder(OtherProductInput.class);
            binder.copyExcluding(otherProductInput, newOtherProductInput,
                    OtherProductInput.PROPERTY_TOPIA_ID,
                    OtherProductInput.PROPERTY_TOPIA_CREATE_DATE,
                    OtherProductInput.PROPERTY_TOPIA_VERSION,
                    OtherProductInput.PROPERTY_OTHER_ACTION,
                    OtherProductInput.PROPERTY_HARVESTING_ACTION,
                    OtherProductInput.PROPERTY_MAINTENANCE_PRUNING_VINES_ACTION,
                    OtherProductInput.PROPERTY_IRRIGATION_ACTION,
                    OtherProductInput.PROPERTY_MINERAL_FERTILIZERS_SPREADING_ACTION,
                    OtherProductInput.PROPERTY_ORGANIC_FERTILIZERS_SPREADING_ACTION,
                    OtherProductInput.PROPERTY_PESTICIDES_SPREADING_ACTION,
                    OtherProductInput.PROPERTY_BIOLOGICAL_CONTROL_ACTION,
                    OtherProductInput.PROPERTY_SEEDING_ACTION);

            newOtherProductInput.setOtherAction((OtherAction) actionsToDuplicatedActions.get(otherProductInput.getOtherAction()));
            newOtherProductInput.setHarvestingAction((HarvestingAction) actionsToDuplicatedActions.get(otherProductInput.getHarvestingAction()));
            newOtherProductInput.setMaintenancePruningVinesAction((MaintenancePruningVinesAction) actionsToDuplicatedActions.get(otherProductInput.getMaintenancePruningVinesAction()));
            newOtherProductInput.setIrrigationAction((IrrigationAction) actionsToDuplicatedActions.get(otherProductInput.getIrrigationAction()));
            newOtherProductInput.setMineralFertilizersSpreadingAction((MineralFertilizersSpreadingAction) actionsToDuplicatedActions.get(otherProductInput.getMineralFertilizersSpreadingAction()));
            newOtherProductInput.setOrganicFertilizersSpreadingAction((OrganicFertilizersSpreadingAction) actionsToDuplicatedActions.get(otherProductInput.getOrganicFertilizersSpreadingAction()));
            newOtherProductInput.setPesticidesSpreadingAction((PesticidesSpreadingAction) actionsToDuplicatedActions.get(otherProductInput.getPesticidesSpreadingAction()));
            newOtherProductInput.setBiologicalControlAction((BiologicalControlAction) actionsToDuplicatedActions.get(otherProductInput.getBiologicalControlAction()));
            newOtherProductInput.setSeedingAction((SeedingAction) actionsToDuplicatedActions.get(otherProductInput.getSeedingAction()));
            inputClone = newOtherProductInput;
        } else {
            throw new UnsupportedOperationException("Unsupported input type: " + input.getClass());
        }
        return inputClone;
    }

}
