package fr.inra.agrosyst.services.growingsystem;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: GrowingSystemServiceImpl.java 5071 2015-08-26 17:01:54Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/growingsystem/GrowingSystemServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.managementmode.DecisionRuleTopiaDao;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeCategory;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystemTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefTraitSdC;
import fr.inra.agrosyst.api.entities.referential.RefTraitSdCTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgriculture;
import fr.inra.agrosyst.api.entities.referential.RefTypeAgricultureTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.growingSystem.GrowingSystemAndDependencies;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.CacheService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.growingsystem.export.ExportGrowingSystemEntity;
import fr.inra.agrosyst.services.growingsystem.export.ExportGrowingSystemMetadata.GrowingSystemCaracteristicsBeanInfo;
import fr.inra.agrosyst.services.growingsystem.export.ExportGrowingSystemMetadata.GrowingSystemMainBeanInfo;
import fr.inra.agrosyst.services.growingsystem.export.ExportGrowingSystemMetadata.GrowingSystemPlotsBeanInfo;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * Growing system service impl.
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 * @author Eric Chatellier
 */
public class GrowingSystemServiceImpl extends AbstractAgrosystService implements GrowingSystemService {

    private static final Log log = LogFactory.getLog(GrowingSystemServiceImpl.class);


    protected PlotService plotService;
    protected ManagementModeService managementModeService;
    protected AnonymizeService anonymizeService;
    protected BusinessAuthorizationService authorizationService;
    protected CacheService cacheService;

    protected GrowingSystemTopiaDao growingSystemDao;
    protected NetworkTopiaDao networkDao;
    protected DecisionRuleTopiaDao decisionRuleDao;
    protected ManagementModeTopiaDao managementModeDao;
    protected RefTypeAgricultureTopiaDao refTypeAgricultureDao;
    protected PracticedSystemTopiaDao practicedSystemDao;
    protected RefTraitSdCTopiaDao refTraitSdCDao;

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    public void setManagementModeService(ManagementModeService managementModeService) {
        this.managementModeService = managementModeService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    public void setNetworkDao(NetworkTopiaDao networkDao) {
        this.networkDao = networkDao;
    }

    public void setDecisionRuleDao(DecisionRuleTopiaDao decisionRuleDao) {
        this.decisionRuleDao = decisionRuleDao;
    }

    public void setManagementModeDao(ManagementModeTopiaDao managementModeDao) {
        this.managementModeDao = managementModeDao;
    }

    public void setRefTypeAgricultureDao(RefTypeAgricultureTopiaDao refTypeAgricultureDao) {
        this.refTypeAgricultureDao = refTypeAgricultureDao;
    }

    public void setPracticedSystemDao(PracticedSystemTopiaDao practicedSystemDao) {
        this.practicedSystemDao = practicedSystemDao;
    }

    public void setRefTraitSdCDao(RefTraitSdCTopiaDao refTraitSdCDao) {
        this.refTraitSdCDao = refTraitSdCDao;
    }

    @Override
    public ResultList<GrowingSystem> getFilteredGrowingSystems(GrowingSystemFilter filter) {
        ResultList<GrowingSystem> result = growingSystemDao.getFilteredGrowingSystems(filter, getSecurityContext());
        return result;
    }

    @Override
    public ResultList<GrowingSystemDto> getFilteredGrowingSystemsDto(GrowingSystemFilter filter) {
        ResultList<GrowingSystem> growingSystems = getFilteredGrowingSystems(filter);
        ResultList<GrowingSystemDto> result = ResultList.transform(growingSystems, anonymizeService.getGrowingSystemToDtoFunction());
        for (GrowingSystemDto gs : result.getElements()) {
            boolean canValidate = authorizationService.isGrowingSystemValidable(gs.getTopiaId());
            gs.setUserCanValidate(canValidate);
        }
        return result;
    }

    @Override
    public List<GrowingSystemDto> getGrowingSystems(Collection<String> growingSystemIds) {
        List<GrowingSystem> growingSystems = growingSystemDao.forTopiaIdIn(growingSystemIds).findAll();
        List<GrowingSystemDto> result = Lists.transform(growingSystems, anonymizeService.getGrowingSystemToDtoFunction());
        return result;
    }

    @Override
    public GrowingSystem getGrowingSystem(String growingSystemId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(growingSystemId));
        GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
        GrowingSystem result = anonymizeService.checkForGrowingSystemAnonymization(growingSystem);
        return result;
    }

    @Override
    public GrowingSystem newGrowingSystem() {
        GrowingSystem result = growingSystemDao.newInstance();
        return result;
    }

    @Override
    public GrowingSystem createOrUpdateGrowingSystem(GrowingSystem growingSystem, List<String> growingSystemNetworkIds,
                                                     Collection<String> selectedPlotsIds, String typeAgricultureTopiaId, List<String> growingSystemCharacteristicIds) {
        String growingSystemId = growingSystem.getTopiaId();
        authorizationService.checkCreateOrUpdateGrowingSystem(growingSystemId);

        GrowingSystem result = createOrUpdateGrowingSystemWithoutCommit(growingSystem, growingSystemNetworkIds, selectedPlotsIds, typeAgricultureTopiaId, growingSystemCharacteristicIds);

        addUserAuthorization(growingSystemId, result);

        getTransaction().commit();

        return result;
    }

    protected void addUserAuthorization(String growingSystemId, GrowingSystem result) {
        if (StringUtils.isBlank(growingSystemId)) {
            authorizationService.growingSystemCreated(result);
        }
    }

    protected GrowingSystem createOrUpdateGrowingSystemWithoutCommit(GrowingSystem growingSystem, List<String> growingSystemNetworkIds, Collection<String> selectedPlotsIds, String typeAgricultureId, List<String> growingSystemCharacteristicIds) {

        if (growingSystemNetworkIds != null) {
            List<Network> networks = networkDao.forTopiaIdIn(growingSystemNetworkIds).findAll();
            Preconditions.checkArgument(networks.size() == growingSystemNetworkIds.size(), "Réseau non retrouvé !");
            growingSystem.setNetworks(networks);
        } else {
            growingSystem.clearNetworks();
        }
        cacheService.clear();

        /**
         * GrowingSystemCharacteristic
         * Simple scenario, so the GrowingSystemCharacteristics list is erased and re-completed each time (for more complexity see plots and zones)
         */
        if (growingSystemCharacteristicIds != null) {
            List<RefTraitSdC> growingSystemCharacteristics = refTraitSdCDao.forTopiaIdIn(growingSystemCharacteristicIds).findAll();
            Preconditions.checkArgument(growingSystemCharacteristics.size() == growingSystemCharacteristicIds.size(), "Charactéristiques non trouvées !");
            growingSystem.setCharacteristics(growingSystemCharacteristics);
        }

        GrowingSystem result;
        growingSystem.setUpdateDate(context.getCurrentDate());
        if (StringUtils.isBlank(growingSystem.getTopiaId())) {
            growingSystem.setActive(true);
            // create a random growingSystem code, used to link growingSystems each other
            setGrowingSystemCode(growingSystem);
            result = growingSystemDao.create(growingSystem);
        } else {
            result = growingSystemDao.update(growingSystem);
        }

        RefTypeAgriculture typeAgriculture = null;
        if (StringUtils.isNotBlank(typeAgricultureId)) {
            typeAgriculture = refTypeAgricultureDao.forTopiaIdEquals(typeAgricultureId).findUnique();
        }
        growingSystem.setTypeAgriculture(typeAgriculture);

        // save selected plots
        plotService.updatePlotsGrowingSystemRelationship(growingSystem, selectedPlotsIds);
        return result;
    }

    private void setGrowingSystemCode(GrowingSystem growingSystem) {
        if (StringUtils.isBlank(growingSystem.getCode())) {
            growingSystem.setCode(UUID.randomUUID().toString());
        }
    }

    @Override
    public List<GrowingSystem> findAllByGrowingPlan(GrowingPlan growingPlan) {
        List<GrowingSystem> result = growingSystemDao.forGrowingPlanEquals(growingPlan).findAll();
        return result;
    }

    @Override
    public GrowingSystem duplicateGrowingSystem(ExtendContext extendContext, GrowingPlan clonedGrowingPlan, GrowingSystem growingSystem) {

        // perform clone
        Binder<GrowingSystem, GrowingSystem> binder = BinderFactory.newBinder(GrowingSystem.class);
        GrowingSystem clonedGrowingSystem = growingSystemDao.newInstance();
        binder.copyExcluding(growingSystem, clonedGrowingSystem,
                GrowingSystem.PROPERTY_TOPIA_ID,
                GrowingSystem.PROPERTY_GROWING_PLAN,
                GrowingSystem.PROPERTY_NETWORKS,
                GrowingSystem.PROPERTY_UPDATE_DATE,
                GrowingSystem.PROPERTY_VALIDATION_DATE);
        clonedGrowingSystem.setGrowingPlan(clonedGrowingPlan);
        clonedGrowingSystem.setValidated(false);

        // break code in duplication case
        if (extendContext.isDuplicateOnly()) {
            clonedGrowingSystem.setCode(UUID.randomUUID().toString());
        }

        // force collection copy instead of collection reference copy
        if (growingSystem.getNetworks() != null) {
            clonedGrowingSystem.setNetworks(Lists.newArrayList(growingSystem.getNetworks()));
        }

        // same for growingSystemCharacteristics
        if (growingSystem.getCharacteristics() != null) {
            clonedGrowingSystem.setCharacteristics(Lists.newArrayList(growingSystem.getCharacteristics()));
        }

        // persist clone
        clonedGrowingSystem.setUpdateDate(context.getCurrentDate());
        GrowingSystem clonedGrowingSystemResult = growingSystemDao.create(clonedGrowingSystem);

        // extends management mode
        ManagementMode managementMode = managementModeDao.forGrowingSystemEquals(growingSystem).
                addEquals(ManagementMode.PROPERTY_ACTIVE, true).
                addEquals(ManagementMode.PROPERTY_CATEGORY, ManagementModeCategory.OBSERVED).findUniqueOrNull();

        if (managementMode == null) {
            managementMode = managementModeDao.forGrowingSystemEquals(growingSystem).
                    addEquals(ManagementMode.PROPERTY_ACTIVE, true).
                    addEquals(ManagementMode.PROPERTY_CATEGORY, ManagementModeCategory.PLANNED).findUniqueOrNull();
        }

        if (managementMode != null) {
            managementModeService.extendManagementMode(extendContext, managementMode, clonedGrowingSystem);
        }

        return clonedGrowingSystemResult;
    }

    @Override
    public void unactivateGrowingSystems(List<String> growingSystemsIds, boolean activate) {
        if (CollectionUtils.isNotEmpty(growingSystemsIds)) {

            for (String growingSystemId : growingSystemsIds) {
                GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
                growingSystem.setActive(activate);
                Date now = context.getCurrentDate();
                if (growingSystem.getEndingDate() == null) {
                    growingSystem.setEndingDate(now);
                }
                growingSystem.setUpdateDate(now);
                growingSystemDao.update(growingSystem);
            }
            getTransaction().commit();
        }
    }

    @Override
    public List<GrowingSystem> findAllActiveByDomainForPlotEdit(Domain domain) {
        List<GrowingSystem> result = growingSystemDao.findAllActiveWritableByDomain(domain, getSecurityContext());
        return result;
    }

    @Override
    public LinkedHashMap<Integer, String> getRelatedGrowingSystems(String growingSystemCode) {
        LinkedHashMap<Integer, String> result = growingSystemDao.findAllRelatedGrowingSystemsByCampaigns(growingSystemCode);
        return result;
    }

    @Override
    public long getGrowingSystemsCount(Boolean active) {
        if (active == null) {
            return growingSystemDao.count();
        }
        return growingSystemDao.forActiveEquals(active).count();
    }

    @Override
    public List<GrowingSystem> findAllByNetwork(Network network) {
        List<GrowingSystem> result = growingSystemDao.forNetworksContains(network).findAll();
        return result;
    }

    @Override
    public GrowingSystem validate(String growingSystemId) {

        authorizationService.checkValidateGrowingSystem(growingSystemId);

        // AThimel 09/12/13 Perform DB update is more powerful
        growingSystemDao.validateGrowingSystem(growingSystemId, context.getCurrentDate());

        getTransaction().commit();

        // AThimel 09/12/13 The next line makes sure that cache state is synchronized with database state
        getPersistenceContext().getHibernateSupport().getHibernateSession().clear();

        GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUnique();
        return growingSystem;
    }

    @Override
    public List<RefTypeAgriculture> findAllTypeAgricultures() {
        List<RefTypeAgriculture> result = refTypeAgricultureDao.newQueryBuilder().setOrderByArguments(RefTypeAgriculture.PROPERTY_REFERENCE_LABEL).findAll();
        return result;
    }

    @Override
    public List<PracticedSystem> getPracticedSystems(String growingSystemCode) {
        List<PracticedSystem> result = practicedSystemDao.forProperties(PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_CODE, growingSystemCode)
                .setOrderByArguments(PracticedSystem.PROPERTY_CAMPAIGNS, PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_NAME).findAll();
        return result;
    }

    @Override
    public List<ManagementMode> getManagementModes(String growingSystemId) {
        List<ManagementMode> result = managementModeDao.forProperties(ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID, growingSystemId)
                .setOrderByArguments(ManagementMode.PROPERTY_VERSION_NUMBER, ManagementMode.PROPERTY_CATEGORY).findAll();
        return result;
    }

    @Override
    public List<Integer> getGrowingSystemCampaigns(String growingSystemCode) {
        List<Integer> result = growingSystemDao.findCampaigns(growingSystemCode);
        return result;
    }

    @Override
    public List<GrowingSystem> getAvailableGsForDuplication(String practicedSystemId, NavigationContext navigationContext) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(practicedSystemId));

        PracticedSystem practicedSystem = practicedSystemDao.forTopiaIdEquals(practicedSystemId).findUnique();
        // Dans un premier temps on filtre sur les systèmes de cultures sur lesquels on a le droit en écriture
        GrowingSystemFilter growingSystemFilter = new GrowingSystemFilter();
        growingSystemFilter.setPageSize(GrowingSystemFilter.ALL_PAGE_SIZE);
        growingSystemFilter.setActive(true);
        growingSystemFilter.setNavigationContext(navigationContext);

        // dans un second temps on ne garde que les sdc pointant vers un domaine du même code que celui du mode de gestion
        //  et sur lesquels il n'y a pas de mode de gestion.
        List<GrowingSystem> result = growingSystemDao.findAllAvailableFoPracticedSystemDuplication(growingSystemFilter, practicedSystem.getGrowingSystem());
        return result;
    }

    @Override
    public List<GrowingSystem> getGrowingSystemWithNameAndCampaign(String growingSystemName, Integer campaign) {
        Preconditions.checkNotNull(growingSystemName);
        List<GrowingSystem> result;
        if (campaign != null) {
            String campaignProperty = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;
            result = growingSystemDao.forProperties(GrowingSystem.PROPERTY_NAME, growingSystemName, campaignProperty , campaign).findAll();
        } else {
            result = growingSystemDao.forProperties(GrowingSystem.PROPERTY_NAME, growingSystemName).findAll();
        }
        return result;
    }

    @Override
    public InputStream exportGrowingSystemsAsXlsStream(List<String> growingSystemIds) {
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        // get all possible bean infos
        GrowingSystemMainBeanInfo mainBeanTab = new GrowingSystemMainBeanInfo();
        GrowingSystemCaracteristicsBeanInfo characteristicTab = new GrowingSystemCaracteristicsBeanInfo();
        GrowingSystemPlotsBeanInfo plotBeanTab = new GrowingSystemPlotsBeanInfo();

        // add all bean infos
        ExportUtils.addAllBeanInfo(sheet, mainBeanTab, characteristicTab, plotBeanTab);

        List<ExportGrowingSystemEntity> characteristicsEntities = (List<ExportGrowingSystemEntity>)sheet.get(characteristicTab);
        List<ExportGrowingSystemEntity> plotEntities = (List<ExportGrowingSystemEntity>)sheet.get(plotBeanTab);
        try {
            if (growingSystemIds != null && !growingSystemIds.isEmpty()) {
                Iterable<GrowingSystem> growingSystems = growingSystemDao.forTopiaIdIn(growingSystemIds).findAllLazy(100);
                for (GrowingSystem growingSystem : growingSystems) {
                    // anonymize growing system if necessary
                    growingSystem = anonymizeService.checkForGrowingSystemAnonymization(growingSystem);

                    ExportGrowingSystemEntity model = new ExportGrowingSystemEntity();
                    model.setGrowingSystemName(growingSystem.getName());
                    model.setSector(growingSystem.getSector());
                    model.setStartingDate(growingSystem.getStartingDate());
                    model.setGrowingPlanName(growingSystem.getGrowingPlan().getName());
                    model.setDomainName(growingSystem.getGrowingPlan().getDomain().getName());
                    model.setCampaign(growingSystem.getGrowingPlan().getDomain().getCampaign());

                    // main tab
                    ExportUtils.export(sheet, model, growingSystem, mainBeanTab);

                    // characteristics tab
                    Collection<RefTraitSdC> characteristics = growingSystem.getCharacteristics();
                    if (characteristics != null && !characteristics.isEmpty()) {
                        for (RefTraitSdC characteristic : characteristics) {
                            ExportGrowingSystemEntity export = (ExportGrowingSystemEntity)model.clone();
                            ExportUtils.copyFields(characteristic, export, null,
                                    "nom_trait",
                                    "type_trait_sdc");
                            if (growingSystem.getTypeAgriculture() != null) {
                                ExportUtils.setExtraField(export, "typeAgriculture", growingSystem.getTypeAgriculture().getReference_label());
                            }
                            ExportUtils.setExtraField(export, "categoryStrategy", growingSystem.getCategoryStrategy());

                            String characteristicComment = null;
                            if (StringUtils.isNotBlank(growingSystem.getCycleManagementComment())) {
                                characteristicComment = growingSystem.getCycleManagementComment();
                            } else if (StringUtils.isNotBlank(growingSystem.getCultureManagementComment())) {
                                characteristicComment = growingSystem.getCultureManagementComment();
                            } else if (StringUtils.isNotBlank(growingSystem.getGroundWorkComment())) {
                                characteristicComment = growingSystem.getGroundWorkComment();
                            } else if (StringUtils.isNotBlank(growingSystem.getParcelsManagementComment())) {
                                characteristicComment = growingSystem.getParcelsManagementComment();
                            }
                            ExportUtils.setExtraField(export, "characteristicComment", characteristicComment);
                            characteristicsEntities.add(export);
                        }
                    }

                    // Plots tab
                    List<Plot> plots = plotService.getAllGrowingSystemPlot(growingSystem);
                    if (plots != null) {
                        for (Plot plot : plots) {
                            ExportGrowingSystemEntity export = (ExportGrowingSystemEntity)model.clone();
                            ExportUtils.copyFields(plot, export, null,
                                    Plot.PROPERTY_NAME,
                                    Plot.PROPERTY_PAC_ILOT_NUMBER);
                            ExportUtils.setExtraField(export, "plotOutComment", growingSystem.getPlotOutputReason());
                            plotEntities.add(export);
                        }
                    }
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;

    }

    @Override
    public LinkedHashMap<Integer, List<String>> getGSCodeGPCodeDomainCodeByCampaignForGrowingSystemDephyId(String dephy) {
        LinkedHashMap<Integer, List<String>> result = null;
        if (StringUtils.isNotBlank(dephy)) {
            result = growingSystemDao.findAllDephyRelatedGSCodesGPCodesDomainCodesByCampaigns(dephy);
        }
        return result;
    }

    @Override
    public void importPZ0GrowingSystems(Map<Class, ImportResults> allResults) {
        ImportResults growingSystemsImportResults = allResults.remove(GrowingSystem.class);
        if (growingSystemsImportResults != null && growingSystemsImportResults.getIgnoredRecords() == 0) {
            try {
                Map<String, EntityAndDependencies> entitiesAndDependencies = growingSystemsImportResults.getEntityAndDepsByCsvIds();
                int count = 1;
                int total = entitiesAndDependencies.values().size();
                for (EntityAndDependencies entityAndDependencies : entitiesAndDependencies.values()) {
                    GrowingSystemAndDependencies growingSystemAndDependencies = (GrowingSystemAndDependencies) entityAndDependencies;
                    GrowingSystem growingSystem = (GrowingSystem) growingSystemAndDependencies.getEntity();

                    long start = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info(String.format("Début sauvegarde du SDC %s - %d/%d.", growingSystem.getName(), count++, total));
                    }

                    String refTypeAgricultureTopiaId = growingSystemAndDependencies.getRefTypeAgricultureId();
                    List<String> linkedNetworks = growingSystemAndDependencies.getNetworksId();
                    GrowingSystem persistedGrowingSystem = createOrUpdateGrowingSystemWithoutCommit(growingSystem, linkedNetworks, null, refTypeAgricultureTopiaId, null);
                    entityAndDependencies.setEntity(persistedGrowingSystem);

                    long p1 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Fin de sauvegarde du SDC, traitement réalisé en:" + (p1- start));
                    }
                }
            } catch (Exception e) {
                throw (new AgrosystImportException("Echec de persistance des systèmes de culture", e));
            }
        }
    }
}
