package fr.inra.agrosyst.services.growingplan;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: GrowingPlanServiceImpl.java 5028 2015-07-10 06:33:13Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/growingplan/GrowingPlanServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanImpl;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanDto;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.EntityImporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.growingplan.export.GrowingPlanExportEntity;
import fr.inra.agrosyst.services.growingplan.export.GrowingPlanExportMetadata.GrowingPlanMainBeanInfo;
import fr.inra.agrosyst.services.referential.csv.AbstractAgrosystModel;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class GrowingPlanServiceImpl extends AbstractAgrosystService implements GrowingPlanService {

    private static final Log log = LogFactory.getLog(GrowingPlanServiceImpl.class);

    protected GrowingSystemService growingSystemService;
    protected AnonymizeService anonymizeService;
    protected BusinessAuthorizationService authorizationService;

    protected GrowingPlanTopiaDao growingPlanDao;
    protected GrowingSystemTopiaDao growingSystemDao;
    protected DomainTopiaDao domainDao;

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    public void setGrowingPlanDao(GrowingPlanTopiaDao growingPlanDao) {
        this.growingPlanDao = growingPlanDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    @Override
    public ResultList<GrowingPlan> getFilteredGrowingPlans(GrowingPlanFilter filter) {
        ResultList<GrowingPlan> result = growingPlanDao.getFilteredGrowingPlans(filter, getSecurityContext());
        return result;
    }

    @Override
    public ResultList<GrowingPlanDto> getFilteredGrowingPlansDto(GrowingPlanFilter filter) {
        ResultList<GrowingPlan> growingPlans = getFilteredGrowingPlans(filter);
        ResultList<GrowingPlanDto> result = ResultList.transform(growingPlans, anonymizeService.getGrowingPlanToDtoFunction(true));
        return result;
    }

    @Override
    public void unactivateGrowingPlans(List<String> growingPlanIds, boolean activate) {
        if (CollectionUtils.isNotEmpty(growingPlanIds)) {

            for (String growingPlanId : growingPlanIds) {
                GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
                growingPlan.setActive(activate);
                growingPlan.setUpdateDate(context.getCurrentDate());
                growingPlanDao.update(growingPlan);
            }
            getTransaction().commit();
        }
    }

    @Override
    public GrowingPlan getGrowingPlan(String growingPlanId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(growingPlanId));
        GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
        GrowingPlan result = anonymizeService.checkForGrowingPlanAnonymization(growingPlan);
        return result;
    }

    @Override
    public GrowingPlan createOrUpdateGrowingPlan(GrowingPlan growingPlan) {
        String growingPlanId = growingPlan.getTopiaId();

        authorizationService.checkCreateOrUpdateGrowingPlan(growingPlanId);

        GrowingPlan result = createOrUpdateGrowingPlanWithoutCommit(growingPlan);

        addUserAuthorization(growingPlanId, result);

        // Now validation is automatic https://forge.codelutin.com/issues/4549
        result = validateAndCommit(result.getTopiaId());

        return result;
    }

    protected void addUserAuthorization(String growingPlanId, GrowingPlan result) {
        if (StringUtils.isBlank(growingPlanId)) {
            authorizationService.growingPlanCreated(result);
        }
    }

    protected GrowingPlan createOrUpdateGrowingPlanWithoutCommit(GrowingPlan growingPlan) {
        GrowingPlan result;

        growingPlan.setUpdateDate(context.getCurrentDate());
        if (StringUtils.isBlank(growingPlan.getTopiaId())) {

            // create a random growingPlan code, used to link growingPlans each other
            if (StringUtils.isBlank(growingPlan.getCode())) {
                String code = UUID.randomUUID().toString();
                growingPlan.setCode(code);
            }

            growingPlan.setActive(true);
            result = growingPlanDao.create(growingPlan);

        } else {
            result = growingPlanDao.update(growingPlan);
        }
        return result;
    }

    @Override
    public GrowingPlan newGrowingPlan() {
        GrowingPlan result = growingPlanDao.newInstance();
        return result;
    }

    @Override
    public List<GrowingPlan> findAllByDomain(Domain domain) {
        List<GrowingPlan> result = growingPlanDao.forDomainEquals(domain).findAll();
        return result;
    }

    @Override
    public GrowingPlan duplicateGrowingPlan(String growingPlanId, String duplicateDomainId, boolean duplicateGrowingSystems) {

        Domain domain = domainDao.forTopiaIdEquals(duplicateDomainId).findUnique();
        GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();

        GrowingPlan duplicatedGrowingPlan = duplicateGrowingPlan(new ExtendContext(true), domain, growingPlan, duplicateGrowingSystems);
        duplicatedGrowingPlan.setUpdateDate(context.getCurrentDate());
        growingPlanDao.update(duplicatedGrowingPlan);

        getTransaction().commit();

        return duplicatedGrowingPlan;
    }

    @Override
    public GrowingPlan duplicateGrowingPlan(ExtendContext extendContext, Domain clonedDomain, GrowingPlan growingPlan, boolean duplicateGrowingSystems) {

        // perform clone
        Binder<GrowingPlan, GrowingPlan> growingPlanBinder = BinderFactory.newBinder(GrowingPlan.class);
        GrowingPlan clonedGrowingPlan = growingPlanDao.newInstance();
        growingPlanBinder.copyExcluding(growingPlan, clonedGrowingPlan,
                GrowingPlan.PROPERTY_TOPIA_ID,
                GrowingPlan.PROPERTY_DOMAIN);
        clonedGrowingPlan.setDomain(clonedDomain);

        // if duplication, break code
        if (extendContext.isDuplicateOnly()) {
            clonedGrowingPlan.setCode(UUID.randomUUID().toString());
        }

        // Now the new growing plan is automatically validated https://forge.codelutin.com/issues/4549
        Date currentDate = context.getCurrentDate();
        clonedGrowingPlan.setUpdateDate(currentDate);
        clonedGrowingPlan.setValidated(true);
        clonedGrowingPlan.setValidationDate(currentDate);

        // persist clone
        clonedGrowingPlan = growingPlanDao.create(clonedGrowingPlan);

        // others entities to clone
        if (duplicateGrowingSystems) {
            Map<GrowingSystem, GrowingSystem> growingSystemCache = Maps.newHashMap();
            List<GrowingSystem> growingSystems = growingSystemDao.forGrowingPlanEquals(growingPlan).addEquals(GrowingSystem.PROPERTY_ACTIVE, true).findAll();
            for (GrowingSystem growingSystem : growingSystems) {
                GrowingSystem clone = growingSystemService.duplicateGrowingSystem(extendContext, clonedGrowingPlan, growingSystem);
                growingSystemCache.put(growingSystem, clone);
            }
            extendContext.setGrowingSystemCache(growingSystemCache);
        }

        return clonedGrowingPlan;
    }

    @Override
    public LinkedHashMap<Integer, String> getRelatedGrowingPlans(String growingPlanCode) {

        LinkedHashMap<Integer, String> result = growingPlanDao.findAllRelatedGrowingPlans(growingPlanCode);
        return result;
    }

    @Override
    public long getGrowingPlansCount(Boolean active) {
        if (active == null) {
            return growingPlanDao.count();
        }
        return growingPlanDao.forActiveEquals(active).count();
    }

    @Override
    public GrowingPlan validateAndCommit(String growingPlanId) {

        authorizationService.checkValidateGrowingPlan(growingPlanId);

        // AThimel 09/12/13 Perform DB update is more powerful
        growingPlanDao.validateGrowingPlan(growingPlanId, context.getCurrentDate());

        getTransaction().commit();

        // AThimel 09/12/13 The next line makes sure that cache state is synchronized with database state
        getPersistenceContext().getHibernateSupport().getHibernateSession().clear();

        GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUnique();
        return growingPlan;
    }

    @Override
    public List<GrowingSystem> getGrowingPlanGrowingSystems(String growingPlanId) {
        List<GrowingSystem> result = growingSystemDao.forProperties(GrowingSystem.PROPERTY_GROWING_PLAN + "." +
                GrowingPlan.PROPERTY_TOPIA_ID, growingPlanId).setOrderByArguments(GrowingPlan.PROPERTY_NAME).findAll();
        return result;
    }

    @Override
    public List<GrowingPlan> getGrowingPlanWithName(String growingPlanName) {
        Preconditions.checkNotNull(growingPlanName);
        List<GrowingPlan> result = growingPlanDao.forNameEquals(growingPlanName).findAll();
        return result;
    }


    @Override
    public InputStream exportGrowingPlanAsXlsStream(List<String> growingPlanIds) {
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        // get all possible bean info
        GrowingPlanMainBeanInfo growingPlanMainBeanInfo = newInstance(GrowingPlanMainBeanInfo.class);
        //GrowingPlanGrowingSystemBeanInfo growingPlanGrowingSystemBeanInfo = newInstance(GrowingPlanGrowingSystemBeanInfo.class);

        ExportUtils.addAllBeanInfo(sheet, growingPlanMainBeanInfo);

        // complexe export
        //List<GrowingPlanExportEntity> growingSystemTab = sheet.get(growingPlanGrowingSystemBeanInfo);

        try {
            if (CollectionUtils.isNotEmpty(growingPlanIds)) {
                Iterable<GrowingPlan> growingPlans = growingPlanDao.forTopiaIdIn(growingPlanIds).findAllLazy(100);
                for (GrowingPlan gp : growingPlans) {

                    // anonymize growing plan
                    gp = anonymizeService.checkForGrowingPlanAnonymization(gp);

                    GrowingPlanExportEntity model = new GrowingPlanExportEntity();
                    model.setDomainName(gp.getDomain().getName());
                    model.setCampaign(gp.getDomain().getCampaign());
                    model.setGrowingPlanName(gp.getName());
                    model.setTypeName(ExportUtils.TYPE_DEPHY_FORMATTER.format(gp.getType()));

                    // main tab
                    ExportUtils.export(sheet, model, gp, growingPlanMainBeanInfo);

                    // growing systems tab
                    /*List<GrowingSystem> gss = getGrowingPlanGrowingSystems(gp.getTopiaId());
                    if (!sheet.containsKey(growingPlanGrowingSystemBeanInfo)) {
                        sheet.put(growingPlanGrowingSystemBeanInfo, growingSystemTab);
                    }
                    if (gss != null) {
                        for (GrowingSystem gs:gss) {
                            Collection<Network> networks = gs.getNetworks();
    
                            if (networks != null && !networks.isEmpty()) {
                                for(Network network:networks) {
                                    GrowingPlanExportEntity export = (GrowingPlanExportEntity) model.clone();
                                    ExportUtils.copyFields(gs, export, null,
                                            GrowingSystem.PROPERTY_NAME);
                                    ExportUtils.setExtraField(export, "networks",network.getName());
                                    growingSystemTab.add(export);
                                }
                            } else {
                                GrowingPlanExportEntity export = (GrowingPlanExportEntity) model.clone();
                                ExportUtils.copyFields(gs, export, null,
                                        GrowingSystem.PROPERTY_NAME);
                                growingSystemTab.add(export);
                            }
                        }
                    }*/
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    @Override
    public void importGrowingPlanForXlsStream(InputStream is) {
        // get all possible bean info
        GrowingPlanMainBeanInfo growingPlanMainBeanInfo = newInstance(GrowingPlanMainBeanInfo.class);

        // technical import
        EntityImporter importer = new EntityImporter();
        Map<EntityExportTabInfo, List<GrowingPlanExportEntity>> datas = importer.importFromStream(is,
                GrowingPlanExportEntity.class,
                growingPlanMainBeanInfo);

        try {
            // custom import (main)
            List<GrowingPlanExportEntity> mainBeanInfo = datas.get(growingPlanMainBeanInfo);
            for (GrowingPlanExportEntity beanInfo : mainBeanInfo) {
                // get domain
                Domain domain = domainDao.forProperties(
                        Domain.PROPERTY_NAME, beanInfo.getDomainName(),
                        Domain.PROPERTY_CAMPAIGN, beanInfo.getCampaign()).findUnique();

                GrowingPlan growingPlan = new GrowingPlanImpl();
                growingPlan.setDomain(domain);
                growingPlan.setName(beanInfo.getGrowingPlanName());
                growingPlan.setType(AbstractAgrosystModel.TYPE_DEPHY_PARSER.parse(beanInfo.getTypeName()));

                ExportUtils.copyFields(beanInfo, growingPlan,
                        GrowingPlan.PROPERTY_DESCRIPTION,
                        GrowingPlan.PROPERTY_GOALS,
                        GrowingPlan.PROPERTY_PROTOCOL_REFERENCE,
                        GrowingPlan.PROPERTY_INSTITUTIONAL_STRUCTURE);

                growingPlan.setActive(true);
                growingPlan.setCode(UUID.randomUUID().toString());
                growingPlan.setUpdateDate(getContext().getCurrentDate());
                growingPlanDao.create(growingPlan);
            }

            getTransaction().commit();
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy fields", ex);
        }
    }

    @Override
    public void importPZ0GrowingPlans(Map<Class, ImportResults> allResults) {
        ImportResults growingPlanResult = allResults.remove(GrowingPlan.class);
        if (growingPlanResult != null && growingPlanResult.getIgnoredRecords() == 0) {
            try {
                Map<String, EntityAndDependencies> entityAndDependencies = growingPlanResult.getEntityAndDepsByCsvIds();
                int count = 1;
                int total = entityAndDependencies.values().size();
                for (EntityAndDependencies growingPlanAndDependencies : entityAndDependencies.values()) {
                    GrowingPlan growingPlan = (GrowingPlan) growingPlanAndDependencies.getEntity();

                    long start = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info(String.format("Début sauvegarde du dipositif %s - %d/%d.", growingPlan.getName(), count++, total));
                    }


                    growingPlan = createOrUpdateGrowingPlanWithoutCommit(growingPlan);
                    growingPlanAndDependencies.setEntity(growingPlan);

                    long p1 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Fin de sauvegarde du dispositif, traitement réalisé en:" + (p1- start));
                    }
                }
            }  catch (Exception e) {
                throw (new AgrosystImportException("Echec de persistance des dispositifs", e));
            }
        }
    }
}
