package fr.inra.agrosyst.services.generic;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: GenericEntityServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/generic/GenericEntityServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.support.TopiaJpaSupport;
import org.nuiton.util.PagerBean;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.referential.ReferentialEntity;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.generic.GenericEntityService;
import fr.inra.agrosyst.api.services.generic.GenericFilter;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class GenericEntityServiceImpl extends AbstractAgrosystService implements GenericEntityService {

    private static final Log log = LogFactory.getLog(GenericEntityServiceImpl.class);

    protected static final String PROPERTY_VALEUR = "Valeur";

    protected static final Function<Object, Map<String, String>> ENUM_NAME_AS_MAP = new Function<Object, Map<String, String>>() {
        @Override
        public Map<String, String> apply(Object enumConstant) {
            Enum<?> anEnum = (Enum<?>) enumConstant;
            LinkedHashMap<String, String> result = Maps.newLinkedHashMap();
            result.put(PROPERTY_VALEUR, anEnum.name());
            return result;
        }
    };

    protected static final Function<String, Class<?>> GET_CLASS = new Function<String, Class<?>>() {
        @Override
        public Class<?> apply(String input) {
            try {
                return Class.forName(input);
            } catch (ClassNotFoundException e) {
                if (log.isErrorEnabled()) {
                    log.error("An exception occurred", e);
                }
                throw new AgrosystTechnicalException("Class not found", e);
            }
        }
    };

    protected BusinessAuthorizationService authorizationService;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    @Override
    public ResultList<?> listEntities(Class<?> klass, GenericFilter filter) {
        authorizationService.checkIsAdmin();
        Preconditions.checkArgument(!AgrosystUser.class.equals(klass), "Forbidden entity: " + klass);

        ResultList<?> result;

        if (klass.isEnum()) {

            Object[] enumConstants = klass.getEnumConstants();

            PagerBean pager = DaoUtils.getPager(0, enumConstants.length, enumConstants.length);

            Iterable<Map<String, String>> transformed = Iterables.transform(Lists.newArrayList(enumConstants), ENUM_NAME_AS_MAP);
            result = ResultList.of(Lists.newArrayList(transformed), pager);

        } else if (TopiaEntity.class.isAssignableFrom(klass)) {

            Class<? extends TopiaEntity> entityKlass = (Class<? extends TopiaEntity>) klass;
            TopiaDao dao = context.getDaoSupplier().getDao(entityKlass);

            StringBuilder query = new StringBuilder(String.format(" FROM %s t ", dao.getEntityClass().getName()));
            query.append(" WHERE 1 = 1 ");
            Map<String, Object> args = Maps.newLinkedHashMap();

            if (filter != null) {
                Map<String, String> propertyNamesAndValues = filter.getPropertyNamesAndValues();
                if (propertyNamesAndValues != null) {
                    for (Map.Entry<String, String> entry : propertyNamesAndValues.entrySet()) {
                        String propertyName = entry.getKey();
                        String value = entry.getValue();
                        if (value.isEmpty()) {
                            continue;
                        }
                        Class returnType = getReturnType(entityKlass, propertyName);
                        if (returnType != null) {
                            try {
                                if (returnType.isAssignableFrom(String.class) || returnType.isEnum()) {
                                    String subQuery = DaoUtils.andAttributeLike("t", propertyName, args, value);
                                    query.append(subQuery);
                                } else {
                                    Object arg = ConvertUtils.convert(value, returnType);
                                    String subQuery = DaoUtils.andAttributeEquals("t", propertyName, args, arg);
                                    query.append(subQuery);
                                }
                            } catch (NumberFormatException ex) {
                                if (log.isDebugEnabled()) {
                                    log.debug("Invalid value for numeric field", ex);
                                }
                            }
                        } else {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't get return type for property " + propertyName + " on " + entityKlass.getName());
                            }
                        }
                    }
                }

                // active
                String subQuery = DaoUtils.andAttributeEquals("t", "active", args, filter.getActive());
                query.append(subQuery);
            }

            int page = filter != null ? filter.getPage() : 0;
            int count = filter != null ? filter.getPageSize() : 10;
            int startIndex = page * count;
            int endIndex = page * count + count - 1;

            TopiaJpaSupport jpaSupport = getPersistenceContext().getJpaSupport();
            List<?> entities = jpaSupport.find(query.toString() + " ORDER BY t." + TopiaEntity.PROPERTY_TOPIA_CREATE_DATE
                    + ", t." + TopiaEntity.PROPERTY_TOPIA_ID, startIndex, endIndex, args);
            long totalCount = ((Number) jpaSupport.findUnique("SELECT COUNT(*) " + query.toString(), args)).longValue();

            // build result bean
            PagerBean pager = DaoUtils.getPager(page, count, totalCount);

            result = ResultList.of(entities, pager);
        } else {
            throw new UnsupportedOperationException("This class cannot be used in this service: " + klass);
        }

        return result;
    }

    /**
     * Recursive getReturn type method that get method return type by checking inhéritance too.
     *
     * @return return type, or null if undetermined
     */
    protected Class getReturnType(Class klass, String methodName) {
        Class result = null;
        try {
            Method method = klass.getDeclaredMethod("get" + StringUtils.capitalize(methodName));
            result = method.getReturnType();
        } catch (NoSuchMethodException ex) {
            // not found here, try on super interfaces
            Class[] superKlass = klass.getInterfaces();
            for (int i = 0; i < superKlass.length && result == null; i++) {
                result = getReturnType(superKlass[i], methodName);
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get method return type", ex);
            }
        }
        return result;
    }

    @Override
    public Map<String, Long> countEntities(Class<?>... classes) {
        List<Class<?>> list = Arrays.asList(classes);
        Map<String, Long> result = countEntities0(list);
        return result;
    }

    protected Map<String, Long> countEntities0(Iterable<Class<?>> classes) {
        authorizationService.checkIsAdmin();
        Map<String, Long> result = Maps.newLinkedHashMap();

        for (Class<?> klass : classes) {

            if (klass.isEnum()) {

                Object[] enumConstants = klass.getEnumConstants();
                long count = enumConstants == null ? 0L : enumConstants.length;
                result.put(klass.getName(), count);
            } else if (TopiaEntity.class.isAssignableFrom(klass)) {

                Class<? extends TopiaEntity> entityKlass = (Class<? extends TopiaEntity>) klass;
                TopiaDao dao = context.getDaoSupplier().getDao(entityKlass);

                long count = dao.count();
                result.put(klass.getName(), count);
            } else {
                throw new UnsupportedOperationException("This class cannot be used in this service: " + klass);
            }
        }

        return result;
    }

    @Override
    public List<String> getProperties(Class<?> klass) {
        authorizationService.checkIsAdmin();

        List<String> result = Lists.newArrayList();

        if (klass.isEnum()) {

            result.add(PROPERTY_VALEUR);

        } else if (TopiaEntity.class.isAssignableFrom(klass)) {

            Class<? extends TopiaEntity> entityKlass = (Class<? extends TopiaEntity>) klass;

            Set<Class<? extends TopiaEntity>> allClasses = getAllClassesExceptTopiaEntity(entityKlass);
            for (Class<? extends TopiaEntity> aClass : allClasses) {
                for (Field declaredField : aClass.getDeclaredFields()) {
                    if (declaredField.getName().startsWith("PROPERTY_")) {
                        try {
                            String value = (String) declaredField.get(null);
                            result.add(value);
                        } catch (IllegalAccessException e) {
                            if (log.isErrorEnabled()) {
                                log.error("Un exception occured", e);
                            }
                        }
                    }
                }
            }
        } else {
            throw new UnsupportedOperationException("This class cannot be used in this service: " + klass);
        }
        return result;
    }

    protected Set<Class<? extends TopiaEntity>> getAllClassesExceptTopiaEntity(Class<? extends TopiaEntity> entityClass) {
        Set<Class<? extends TopiaEntity>> result = Sets.newLinkedHashSet();
        Class<?>[] interfaces = entityClass.getInterfaces();
        if (interfaces != null) {
            for (Class<?> aClass : interfaces) {
                if (!TopiaEntity.class.equals(aClass) && TopiaEntity.class.isAssignableFrom(aClass)) {
                    Class<? extends TopiaEntity> parentClass = (Class<? extends TopiaEntity>) aClass;
                    Set<Class<? extends TopiaEntity>> parentClasses = getAllClassesExceptTopiaEntity(parentClass);
                    result.addAll(parentClasses);
                }
            }
        }
        result.add(entityClass); // Add at the end in order to get always the same order
        return result;
    }

    @Override
    public void unactivateEntities(Class<?> klass, List<String> entityIds, boolean activate) {
        authorizationService.checkIsAdmin();

        if (ReferentialEntity.class.isAssignableFrom(klass)) {

            Class<? extends ReferentialEntity> entityKlass = (Class<? extends ReferentialEntity>) klass;
            TopiaDao dao = context.getDaoSupplier().getDao(entityKlass);

            for (String entityId : entityIds) {

                TopiaEntity entity = dao.forTopiaIdEquals(entityId).findUnique();
                ((ReferentialEntity) entity).setActive(activate);
                dao.update(entity);
            }
            getTransaction().commit();
        } else {
            throw new UnsupportedOperationException("This class cannot be used in this service: " + klass);
        }

    }

    @Override
    public Map<String, Long> countEntitiesFromString(List<String> classesList) {
        Iterable<Class<?>> classes = Iterables.transform(classesList, GET_CLASS);
        Map<String, Long> result = countEntities0(classes);
        return result;
    }

    @Override
    public ResultList<?> listEntitiesFromString(String className, GenericFilter filter) {
        ResultList<?> result = listEntities(GET_CLASS.apply(className), filter);
        return result;
    }

    @Override
    public List<String> getPropertiesFromString(String className) {
        List<String> result = getProperties(GET_CLASS.apply(className));
        return result;
    }

    @Override
    public void unactivateEntitiesFromString(String className, List<String> entityIds, boolean activate) {
        unactivateEntities(GET_CLASS.apply(className), entityIds, activate);
    }
}
