package fr.inra.agrosyst.services.effective;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EffectiveCropCycleServiceImpl.java 5199 2015-12-11 20:16:33Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/effective/EffectiveCropCycleServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Functions;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryImpl;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.ToolsCouplingTopiaDao;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractActionTopiaDao;
import fr.inra.agrosyst.api.entities.action.AbstractInput;
import fr.inra.agrosyst.api.entities.action.AbstractInputTopiaDao;
import fr.inra.agrosyst.api.entities.action.MineralProductInput;
import fr.inra.agrosyst.api.entities.action.MineralProductUnit;
import fr.inra.agrosyst.api.entities.action.OrganicProductInput;
import fr.inra.agrosyst.api.entities.action.OrganicProductUnit;
import fr.inra.agrosyst.api.entities.action.OtherProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductInput;
import fr.inra.agrosyst.api.entities.action.PhytoProductUnit;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnection;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNodeTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhase;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCyclePhaseTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleSpecies;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectiveInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSpeciesStade;
import fr.inra.agrosyst.api.entities.effective.EffectiveSpeciesStadeTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrape;
import fr.inra.agrosyst.api.entities.referential.RefClonePlantGrapeTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDI;
import fr.inra.agrosyst.api.entities.referential.RefOrientationEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefStadeEDI;
import fr.inra.agrosyst.api.entities.referential.RefStadeEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.entities.referential.RefVarieteTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.common.PricesService;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.domain.Equipments;
import fr.inra.agrosyst.api.services.domain.ToolsCouplingDto;
import fr.inra.agrosyst.api.services.domain.ZoneDto;
import fr.inra.agrosyst.api.services.effective.CopyPasteZoneByCampaigns;
import fr.inra.agrosyst.api.services.effective.CopyPasteZoneDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleNodeDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleSpeciesDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycles;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveZoneFilter;
import fr.inra.agrosyst.api.services.effective.TargetedZones;
import fr.inra.agrosyst.api.services.input.InputService;
import fr.inra.agrosyst.api.services.itk.SpeciesStadeDto;
import fr.inra.agrosyst.api.services.practiced.PracticedSystems;
import fr.inra.agrosyst.api.services.practiced.RefStadeEdiDto;
import fr.inra.agrosyst.api.services.pz0.effective.EffectiveCropCycleAndDependencies;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.services.pz0.effective.Pz0EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.domain.DomainServiceImpl;
import fr.inra.agrosyst.services.effective.export.EffectiveCropCycleExportEntity;
import fr.inra.agrosyst.services.effective.export.EffectiveCropCycleExportMetadata.EffectiveCropCycleBeanInfo;
import fr.inra.agrosyst.services.effective.export.EffectiveCropCycleExportMetadata.EffectiveITKBeanInfo;
import fr.inra.agrosyst.services.effective.export.EffectiveCropCycleExportMetadata.EffectivePerennialCropCycleSpeciesBeanInfo;
import fr.inra.agrosyst.services.effective.export.EffectiveSeasonalCropCycleExport;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public class EffectiveCropCycleServiceImpl extends AbstractAgrosystService implements EffectiveCropCycleService {

    public static final String NEW_NODE_PREFIX = "new-node-";
    private static final Log log = LogFactory.getLog(EffectiveCropCycleServiceImpl.class);

    protected BusinessAuthorizationService authorizationService;
    protected ActionService actionService;
    protected InputService inputService;
    protected PricesService pricesService;
    protected DomainService domainService;
    protected AnonymizeService anonymizeService;
    protected ReferentialService referentialService;

    protected ZoneTopiaDao zoneDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryDao;
    protected CroppingPlanSpeciesTopiaDao croppingPlanSpeciesDao;
    protected ToolsCouplingTopiaDao toolsCouplingDao;
    protected EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleDao;
    protected EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleDao;
    protected EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionDao;
    protected EffectiveCropCyclePhaseTopiaDao effectiveCropCyclePhaseDao;
    protected EffectiveCropCycleSpeciesTopiaDao effectiveCropCycleSpeciesDao;
    protected EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeDao;
    protected EffectiveInterventionTopiaDao effectiveInterventionDao;
    protected EffectiveSpeciesStadeTopiaDao effectiveSpeciesStadeDao;
    protected AbstractActionTopiaDao abstractActionDao;
    protected AbstractInputTopiaDao abstractInputTopiaDao;

    protected RefClonePlantGrapeTopiaDao refClonePlantGrapeDao;
    protected RefVarieteTopiaDao refVarieteDao;
    protected RefStadeEDITopiaDao refStadeEDIDao;
    protected RefOrientationEDITopiaDao refOrientationEDIDao;

    protected static final Function<EffectiveSeasonalCropCycle, Collection<EffectiveCropCycleNode>> EFFECTIVE_SEASONAL_CROP_CYCLE_TO_NODE_FUNCTION = new Function<EffectiveSeasonalCropCycle, Collection<EffectiveCropCycleNode>>() {
        @Override
        public Collection<EffectiveCropCycleNode> apply(EffectiveSeasonalCropCycle cycle) {
            Collection<EffectiveCropCycleNode> result = cycle.getNodes();
            return result;
        }
    };

    protected static final Function<EffectiveCropCycleConnection, String> EFFECTIVE_CONNECTION_BY_TARGET_ID = new Function<EffectiveCropCycleConnection, String>() {
        @Override
        public String apply(EffectiveCropCycleConnection input) {
            String targetId = input.getTarget().getTopiaId();
            return targetId;
        }
    };

    protected final Function<EffectivePerennialCropCycle, CopyPasteZoneDto> EFFECTIVE_PERENNIAL_TO_CROP_CYCLE_CROP_TARGET_FUNCTION = new Function<EffectivePerennialCropCycle, CopyPasteZoneDto>() {
        @Override
        public CopyPasteZoneDto apply(EffectivePerennialCropCycle epcc) {
            CopyPasteZoneDto result = new CopyPasteZoneDto();
            result.setCroppingPlanEntryId(epcc.getCroppingPlanEntry().getTopiaId());
            result.setCroppingPlanEntryName(epcc.getCroppingPlanEntry().getName());

            EffectiveCropCyclePhase phase = epcc.getPhase();
            result.setPhaseId(phase.getTopiaId());
            result.setPhaseType(phase.getType());
            result.setNbInterventions(effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).count());
            return result;
        }
    };

    protected final Function<EffectiveCropCycleNode, CopyPasteZoneDto> EFFECTIVE_CROP_CYCLE_NODE_TO_CROP_TARGET_FUNCTION = new Function<EffectiveCropCycleNode, CopyPasteZoneDto>() {
        @Override
        public CopyPasteZoneDto apply(EffectiveCropCycleNode node) {
            long nbNonIntermediateInterventions = effectiveInterventionDao.forProperties(EffectiveIntervention.PROPERTY_EFFECTIVE_CROP_CYCLE_NODE, node, EffectiveIntervention.PROPERTY_INTERMEDIATE_CROP, false).count();
            long nbIntermediateInterventions = effectiveInterventionDao.forProperties(EffectiveIntervention.PROPERTY_EFFECTIVE_CROP_CYCLE_NODE, node, EffectiveIntervention.PROPERTY_INTERMEDIATE_CROP, true).count();

            CopyPasteZoneDto result = new CopyPasteZoneDto();
            result.setCroppingPlanEntryId(node.getCroppingPlanEntry().getTopiaId());
            result.setCroppingPlanEntryName(node.getCroppingPlanEntry().getName());
            result.setRank(node.getRank());

            result.setNbInterventions(nbNonIntermediateInterventions);
            result.setNbIntermediateCropInterventions(nbIntermediateInterventions);
            result.setNodeId(node.getTopiaId());
            return result;
        }
    };

    public static final Function<CroppingPlanSpecies, String> GET_CROPPING_PLAN_SPECIES_CODE = new Function<CroppingPlanSpecies, String>() {
        @Override
        public String apply(CroppingPlanSpecies species) {
            String code = species.getCode();
            return code;
        }
    };

    protected static final Function<SpeciesStadeDto, String> GET_SPECIES_STADE_BY_SPECIES_CODE = new Function<SpeciesStadeDto, String>() {
        @Override
        public String apply(SpeciesStadeDto speciesStadeDto) {
            String code = speciesStadeDto.getSpeciesCode();
            return code;
        }
    };

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setActionService(ActionService actionService) {
        this.actionService = actionService;
    }

    public void setInputService(InputService inputService) {
        this.inputService = inputService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setAbstractInputTopiaDao(AbstractInputTopiaDao abstractInputTopiaDao) {
        this.abstractInputTopiaDao = abstractInputTopiaDao;
    }

    public void setPricesService(PricesService pricesService) {
        this.pricesService = pricesService;
    }

    public void setZoneDao(ZoneTopiaDao zoneDao) {
        this.zoneDao = zoneDao;
    }

    public void setCroppingPlanEntryDao(CroppingPlanEntryTopiaDao croppingPlanEntryDao) {
        this.croppingPlanEntryDao = croppingPlanEntryDao;
    }

    public void setCroppingPlanSpeciesDao(
            CroppingPlanSpeciesTopiaDao croppingPlanSpeciesDao) {
        this.croppingPlanSpeciesDao = croppingPlanSpeciesDao;
    }

    public void setToolsCouplingDao(ToolsCouplingTopiaDao toolsCouplingDao) {
        this.toolsCouplingDao = toolsCouplingDao;
    }

    public void setEffectivePerennialCropCycleDao(EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleDao) {
        this.effectivePerennialCropCycleDao = effectivePerennialCropCycleDao;
    }

    public void setEffectiveSeasonalCropCycleDao(EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleDao) {
        this.effectiveSeasonalCropCycleDao = effectiveSeasonalCropCycleDao;
    }

    public void setEffectiveCropCycleConnectionDao(EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionDao) {
        this.effectiveCropCycleConnectionDao = effectiveCropCycleConnectionDao;
    }

    public void setEffectiveCropCyclePhaseDao(
            EffectiveCropCyclePhaseTopiaDao effectiveCropCyclePhaseDao) {
        this.effectiveCropCyclePhaseDao = effectiveCropCyclePhaseDao;
    }

    public void setEffectiveCropCycleSpeciesDao(
            EffectiveCropCycleSpeciesTopiaDao effectiveCropCycleSpeciesDao) {
        this.effectiveCropCycleSpeciesDao = effectiveCropCycleSpeciesDao;
    }

    public void setEffectiveCropCycleNodeDao(
            EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeDao) {
        this.effectiveCropCycleNodeDao = effectiveCropCycleNodeDao;
    }

    public void setRefClonePlantGrapeDao(RefClonePlantGrapeTopiaDao refClonePlantGrapeDao) {
        this.refClonePlantGrapeDao = refClonePlantGrapeDao;
    }

    public void setRefVarieteDao(RefVarieteTopiaDao refVarieteDao) {
        this.refVarieteDao = refVarieteDao;
    }

    public void setEffectiveInterventionDao(
            EffectiveInterventionTopiaDao effectiveInterventionDao) {
        this.effectiveInterventionDao = effectiveInterventionDao;
    }

    public void setRefStadeEDIDao(RefStadeEDITopiaDao refStadeEDIDao) {
        this.refStadeEDIDao = refStadeEDIDao;
    }

    public void setEffectiveSpeciesStadeDao(EffectiveSpeciesStadeTopiaDao effectiveSpeciesStadeDao) {
        this.effectiveSpeciesStadeDao = effectiveSpeciesStadeDao;
    }

    public void setAbstractActionDao(AbstractActionTopiaDao abstractActionDao) {
        this.abstractActionDao = abstractActionDao;
    }

    public void setRefOrientationEDIDao(RefOrientationEDITopiaDao refOrientationEDIDao) {
        this.refOrientationEDIDao = refOrientationEDIDao;
    }

    @Override
    public Zone getZone(String zoneTopiaId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        Zone result = anonymizeService.checkForZoneAnonymization(zone);
        return result;
    }

    @Override
    public ResultList<Zone> getFilteredZones(EffectiveZoneFilter filter) {
        ResultList<Zone> result = zoneDao.getFilteredZones(filter, getSecurityContext());
        return result;
    }

    @Override
    public ResultList<ZoneDto> getFilteredZonesDto(EffectiveZoneFilter filter) {
        ResultList<Zone> zones = getFilteredZones(filter);
        ResultList<ZoneDto> result = ResultList.transform(zones, anonymizeService.getZoneToDtoFunction());
        return result;
    }

    protected String getEffectivePerennialCropCycleInfo(String zoneTopiaId) {
        StringBuilder result = new StringBuilder();
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();

        for (EffectivePerennialCropCycle cycle : cycles) {

            result.append(cycle.getCroppingPlanEntry().getName());
            EffectiveCropCyclePhase phase = cycle.getPhase();
            if (phase != null) {
                result.append(" (");
                result.append(effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).count());
                result.append(") ");
            }
        }


        return result.toString();
    }

    protected String getEffectiveCropCycleNodesInfo(Collection<EffectiveCropCycleNode> allEffectiveCropCycleNodes) {
        StringBuilder result = new StringBuilder();

        if (allEffectiveCropCycleNodes != null && !allEffectiveCropCycleNodes.isEmpty()) {
            for (EffectiveCropCycleNode effectiveCropCycleNode:allEffectiveCropCycleNodes) {

                EffectiveCropCycleConnection effectiveCropCycleConnection = effectiveCropCycleConnectionDao.forTargetEquals(effectiveCropCycleNode).findUniqueOrNull();
                List<EffectiveIntervention> allEffectiveInterventions = effectiveInterventionDao.forEffectiveCropCycleNodeEquals(effectiveCropCycleNode).findAll();
                int interventionCount = allEffectiveInterventions.size();
                int intermediateCropInterventionCount = 0;

                // test if the node gets a connection (the first node doesn't) and if the connection gets an intermediateCroppingPlanEntry before getting infos
                if (effectiveCropCycleConnection != null && effectiveCropCycleConnection.getIntermediateCroppingPlanEntry() != null) {
                    intermediateCropInterventionCount = getIntermediateCropIntervetionCount(allEffectiveInterventions);
                    buildIntermediateInterventionsResult(result, intermediateCropInterventionCount);
                }

                // add principal croppingPlanEntry Info
                buildCropInfoResult(result, effectiveCropCycleNode, interventionCount, intermediateCropInterventionCount);

            }
        }
        return result.toString();
    }

    private void buildCropInfoResult(StringBuilder result, EffectiveCropCycleNode effectiveCropCycleNode, int interventionCount, int intermediateCropInterventionCount) {
        result.append(" ");
        result.append(effectiveCropCycleNode.getCroppingPlanEntry().getName());
        result.append(" (");
        result.append(interventionCount-intermediateCropInterventionCount);
        result.append(") -");
    }

    private void buildIntermediateInterventionsResult(StringBuilder result, int intermediateCropInterventionCount) {
        result.append(" CI (");
        result.append(intermediateCropInterventionCount);
        result.append(") ");
    }

    private int getIntermediateCropIntervetionCount(List<EffectiveIntervention> allEffectiveInterventions) {
        int intermediateCropInterventionCount = 0;
        if (!allEffectiveInterventions.isEmpty()) {
            for (EffectiveIntervention effectiveIntervention : allEffectiveInterventions) {
                if (effectiveIntervention.isIntermediateCrop()) {
                    intermediateCropInterventionCount++;
                }
            }
        }
        return intermediateCropInterventionCount;
    }

    protected String getEffectiveSeasonalCropCyclesInfo(String zoneTopiaId) {
        String result = "";
        // get all SeasonnalCycles Dtos.
        // Theoretically, there is only one seasonalcycle per zone but in case of a future update, it works even for several seasonalcycles per zone
        List<EffectiveSeasonalCropCycle> allEffectiveSeasonalCropCycles =  effectiveSeasonalCropCycleDao.forZoneEquals(zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique()).findAll();

        if (allEffectiveSeasonalCropCycles != null && !allEffectiveSeasonalCropCycles.isEmpty()) {
            StringBuilder resultBuilder = new StringBuilder();
            for (EffectiveSeasonalCropCycle effectiveSeasonalCropCycle:allEffectiveSeasonalCropCycles) {
                Collection<EffectiveCropCycleNode> allEffectiveCropCycleNodes = effectiveSeasonalCropCycle.getNodes();
                resultBuilder.append(getEffectiveCropCycleNodesInfo(allEffectiveCropCycleNodes));
            }
            result = resultBuilder.toString();
            // this should not be done if there was EffectiveSeasonalCropCycle
            if (result.length() > 2) {
                result = result.substring(0, result.length()-2);
            }
        }
        return result;
    }

    @Override
    public ResultList<ZoneDto> getFilteredZonesAndCroppingPlanInfosDto(EffectiveZoneFilter filter) {

        ResultList<ZoneDto> result = getFilteredZonesDto(filter);
        // loop among Zone DTOs
        for (ZoneDto zoneDto:result) {

            String croppingPlanInfo;

            // for each PerennialCropCycle String with: The crop name (the number of interventions)
            croppingPlanInfo = getEffectivePerennialCropCycleInfo(zoneDto.getTopiaId());
            // for each SeasonalCropCycle String with: The crop name (the number of interventions) CI (the number of interventions for intermediate crop)
            croppingPlanInfo = croppingPlanInfo + getEffectiveSeasonalCropCyclesInfo(zoneDto.getTopiaId());

            zoneDto.setCroppingPlanInfo(croppingPlanInfo);
        }
        return result;
    }

    @Override
    public List<CroppingPlanEntry> getZoneCroppingPlanEntries(Zone zone) {
        List<CroppingPlanEntry> entries = croppingPlanEntryDao.forDomainEquals(zone.getPlot().getDomain()).findAll();
        return entries;
    }

    @Override
    public List<CroppingPlanEntry> getZoneCroppingPlanEntriesWithoutDomain(Zone zone) {
        List<CroppingPlanEntry> entries = croppingPlanEntryDao.forDomainEquals(zone.getPlot().getDomain()).setOrderByArguments(CroppingPlanEntry.PROPERTY_NAME).findAll();
        List<CroppingPlanEntry> croppingPlanEntries = Lists.newArrayListWithCapacity(entries.size());
        Binder<CroppingPlanEntry, CroppingPlanEntry> croppingPlanEntryCroppingPlanEntryBinder = BinderFactory.newBinder(CroppingPlanEntry.class, CroppingPlanEntry.class);
        for(CroppingPlanEntry croppingPlanEntry:entries) {
            CroppingPlanEntry lightCroppingPlanEntry = new CroppingPlanEntryImpl();
            croppingPlanEntryCroppingPlanEntryBinder.copyExcluding(croppingPlanEntry, lightCroppingPlanEntry,
                    CroppingPlanEntry.PROPERTY_DOMAIN);
            croppingPlanEntries.add(lightCroppingPlanEntry);
        }
        return croppingPlanEntries;
    }

    @Override
    public CroppingPlanEntry getPreviousCampaignCroppingPlanEntry(String zoneTopiaId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();

        EffectiveCropCycleNode node = effectiveCropCycleNodeDao.findLastNodeForPreviousCampaign(zone);
        CroppingPlanEntry result = null;
        if (node != null) {
            result = node.getCroppingPlanEntry();
        }
        return result;
    }

    @Override
    public List<EffectivePerennialCropCycleDto> getAllEffectivePerennialCropCyclesDtos(String zoneTopiaId) {

        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();
        List<EffectivePerennialCropCycleDto> result = convertPerennialCropCyclesToDto(cycles, zone);
        return result;
    }

    protected List<EffectivePerennialCropCycle> getAllEffectivePerennialCropCycles(String zoneTopiaId) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();
        return cycles;
    }

    protected List<EffectivePerennialCropCycleDto> convertPerennialCropCyclesToDto(
            List<EffectivePerennialCropCycle> cycles, Zone zone) {
        List<EffectivePerennialCropCycleDto> result = Lists.newArrayListWithCapacity(cycles.size());

        for (EffectivePerennialCropCycle cycle : cycles) {
            String domainId = cycle.getZone().getPlot().getDomain().getTopiaId();

            EffectivePerennialCropCycleDto dto = EffectiveCropCycles.PERENNIAL_CROP_CYCLE_TO_DTO.apply(cycle);

            EffectiveCropCyclePhase phase = cycle.getPhase();
            if (phase != null) {
                EffectiveCropCyclePhaseDto phaseDto = EffectiveCropCycles.CROP_CYCLE_PHASE_TO_DTO.apply(phase);
                // interventions
                List<EffectiveIntervention> interventions = effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).findAll();
                List<EffectiveInterventionDto> interventionDtos = Lists.newArrayListWithCapacity(interventions.size());
                for (EffectiveIntervention intervention : interventions) {
                    EffectiveInterventionDto interventionDto = EffectiveCropCycles.INTERVENTION_TO_DTO.apply(intervention);
                    List<AbstractAction> abstractActions = abstractActionDao.forEffectiveInterventionEquals(intervention).findAll();
                    interventionDto.setActions(abstractActions);
                    List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
                    interventionDto.setInputs(inputs);
                    interventionDto.setDomainId(domainId);
                    interventionDtos.add(interventionDto);
                }

                phaseDto.setInterventions(interventionDtos);
                dto.setPhaseDtos(Lists.newArrayList(phaseDto));
            }

            String croppingPlanEntryCode = cycle.getCroppingPlanEntry().getCode();
            List<EffectiveCropCycleSpeciesDto> species = getCropCyclePerennialSpecies(
                    croppingPlanEntryCode, cycle, zone.getPlot().getDomain().getCampaign());
            dto.setSpeciesDtos(species);

            result.add(dto);
        }
        return result;
    }

    /**
     * Méthode qui charge la liste des espèces disponibles pour le cycle passé en paramètre et remplit les Dto avec les
     * informations déjà saisie dans ce cycle
     *
     * @param croppingPlanEntryCode l'identifiant de la culture à traiter
     * @param cycle                 le cycle pérenne
     * @param campaign              la campagne concernée (issue de la zone)
     * @return la liste complète des espèces complétée des informations liées au cycle
     */
    protected List<EffectiveCropCycleSpeciesDto> getCropCyclePerennialSpecies(
            String croppingPlanEntryCode, EffectivePerennialCropCycle cycle, Integer campaign) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(croppingPlanEntryCode));
        Preconditions.checkArgument(campaign != null);

        final Map<String, EffectiveCropCycleSpecies> speciesCodeToCycleEntity;

        if (cycle == null || cycle.getSpecies() == null) {
            speciesCodeToCycleEntity = Maps.newHashMap();
        } else {
            speciesCodeToCycleEntity = Maps.uniqueIndex(cycle.getSpecies(), EffectiveCropCycles.GET_CROP_CYCLE_PERENNIAL_SPECIES_CODE);
        }

        // Chargement de la liste des espèces (sans doublon)
        Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> entryAndSpeciesFromCode =
                domainService.getEntryAndSpeciesFromCode(croppingPlanEntryCode, Sets.newHashSet(campaign));

        // Dto -> Decoration
        Function<CroppingPlanSpeciesDto, EffectiveCropCycleSpeciesDto> decorateFunction = new Function<CroppingPlanSpeciesDto, EffectiveCropCycleSpeciesDto>() {
            @Override
            public EffectiveCropCycleSpeciesDto apply(CroppingPlanSpeciesDto input) {
                String croppingPlanSpeciesCode = input.getCode();
                EffectiveCropCycleSpeciesDto result = new EffectiveCropCycleSpeciesDto(input);
                EffectiveCropCycleSpecies cropCyclePerennialSpecies = speciesCodeToCycleEntity.get(croppingPlanSpeciesCode);
                if (cropCyclePerennialSpecies != null) {
                    result.setOverGraftDate(cropCyclePerennialSpecies.getOverGraftDate());
                    result.setPlantsCertified(cropCyclePerennialSpecies.isPlantsCertified());
                    result.setGraftSupport(PracticedSystems.REF_VARIETE_TO_GRAPH_DTO.apply(cropCyclePerennialSpecies.getGraftSupport()));
                    result.setGraftClone(PracticedSystems.REF_CLONE_TO_GRAPH_DTO.apply(cropCyclePerennialSpecies.getGraftClone()));
                }
                return result;
            }
        };
        // La transformation se fait en 2 temps : Entity -> Dto -> Decoration
        Function<CroppingPlanSpecies, EffectiveCropCycleSpeciesDto> transformFunction = Functions.compose(decorateFunction, CroppingPlans.CROPPING_PLAN_SPECIES_TO_DTO);
        Iterable<EffectiveCropCycleSpeciesDto> transformed = Iterables.transform(entryAndSpeciesFromCode.getRight().values(), transformFunction);

        List<EffectiveCropCycleSpeciesDto> result = Lists.newArrayList(transformed);

        return result;
    }

    @Override
    public List<EffectiveSeasonalCropCycleDto> getAllEffectiveSeasonalCropCyclesDtos(String zoneTopiaId) {

        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        List<EffectiveSeasonalCropCycle> cycles = effectiveSeasonalCropCycleDao.forZoneEquals(zone).findAll();
        List<EffectiveSeasonalCropCycleDto> result = Lists.newArrayListWithCapacity(cycles.size());
        for (EffectiveSeasonalCropCycle cycle : cycles) {
            String domainId = cycle.getZone().getPlot().getDomain().getTopiaId();
            EffectiveSeasonalCropCycleDto dto = EffectiveCropCycles.SEASONNAL_CROP_CYCLE_TO_DTO.apply(cycle);

            if (cycle.getNodes() != null) {
                List<EffectiveCropCycleNodeDto> nodeDtos = Lists.newArrayListWithCapacity(cycle.getNodes().size());
                for (EffectiveCropCycleNode node : cycle.getNodes()) {
                    EffectiveCropCycleNodeDto nodeDto = EffectiveCropCycles.CROP_CYCLE_NODE_TO_DTO.apply(node);
                    nodeDtos.add(nodeDto);

                    // interventions
                    List<EffectiveIntervention> interventions = effectiveInterventionDao.forEffectiveCropCycleNodeEquals(node).findAll();
                    List<EffectiveInterventionDto> interventionDtos = Lists.newArrayListWithCapacity(interventions.size());
                    for (EffectiveIntervention intervention : interventions) {
                        EffectiveInterventionDto interventionDto = EffectiveCropCycles.INTERVENTION_TO_DTO.apply(intervention);
                        List<AbstractAction> abstractActions = abstractActionDao.forEffectiveInterventionEquals(intervention).findAll();
                        interventionDto.setActions(abstractActions);
                        List<AbstractInput> inputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
                        interventionDto.setInputs(inputs);
                        interventionDto.setDomainId(domainId);

                        interventionDtos.add(interventionDto);
                    }
                    nodeDto.setInterventions(interventionDtos);
                }
                dto.setNodeDtos(nodeDtos);
            }
            List<EffectiveCropCycleConnection> connections = effectiveCropCycleConnectionDao.findAllByEffectiveSeasonalCropCycle(cycle);
            dto.setConnectionDtos(Lists.newArrayList(Iterables.transform(connections, EffectiveCropCycles.CROP_CYCLE_CONNECTION_TO_DTO)));

            result.add(dto);
        }

        return result;
    }

    protected List<EffectiveSeasonalCropCycle> getAllEffectiveSeasonalCropCycles(String zoneTopiaId) {

        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        List<EffectiveSeasonalCropCycle> cycles = effectiveSeasonalCropCycleDao.forZoneEquals(zone).findAll();
        return cycles;
    }

    @Override
    public void updateEffectiveCropCycles(String zoneId,
                                          List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles,
                                          List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles,
                                          List<Price> prices) {

        authorizationService.checkCreateOrUpdateEffectiveCropCycles(zoneId);

        // Note: plot is not updated here
        updateEffectiveCropCyclesWithoutCommit(zoneId, effectiveSeasonalCropCycles, effectivePerennialCropCycles, prices);

        getTransaction().commit();
    }

    protected void updateEffectiveCropCyclesWithoutCommit(String zoneId,
                                                          List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles,
                                                          List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles,
                                                          List<Price> prices) {

        // Note: plot is not updated here
        updateEffectiveSeasonalCropCycles(zoneId, effectiveSeasonalCropCycles);
        udpateEffectivePerennialCropCycles(zoneId, effectivePerennialCropCycles);

        updateEffectivePrices(zoneId, prices);
    }

    /**
     * Update seasonal crop cycles linked to zone.
     *
     * @param zoneId                      the identifier of the zone on which the crop cycle is about
     * @param effectiveSeasonalCropCycles seasonal crop cycles
     */
    protected void updateEffectiveSeasonalCropCycles(String zoneId, List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycles) {

        if (effectiveSeasonalCropCycles != null) { // null can be valid only for test

            Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();
            Domain domain = zone.getPlot().getDomain();

            List<CroppingPlanEntry> targetedZoneCroppingPlanEntries = croppingPlanEntryDao.forDomainEquals(domain).findAll();

            List<EffectiveSeasonalCropCycle> cycles = effectiveSeasonalCropCycleDao.forZoneEquals(zone).findAll();
            Set<EffectiveSeasonalCropCycle> toDeleteCycles = Sets.newHashSet(cycles);
            Map<String, EffectiveSeasonalCropCycle> idToCycles = Maps.uniqueIndex(cycles, Entities.GET_TOPIA_ID);

            for (EffectiveSeasonalCropCycleDto cycleDto : effectiveSeasonalCropCycles) {

                String cycleId = cycleDto.getTopiaId();
                EffectiveSeasonalCropCycle cycle = getEffectiveSeasonalCropCycle(zone, toDeleteCycles, idToCycles, cycleId);

                // manage cycle nodes
                updateCycleNodesAndConnections(domain, targetedZoneCroppingPlanEntries, cycleDto, cycle);

                // save or remove cycle (if cycle hasn't got any node then it must be removed)
                if (CollectionUtils.isNotEmpty(cycle.getNodes())) {
                    if (StringUtils.isBlank(cycleId)) {
                        effectiveSeasonalCropCycleDao.create(cycle);
                    } else {
                        effectiveSeasonalCropCycleDao.update(cycle);
                    }
                } else if (StringUtils.isNotBlank(cycleId)){
                    // remove cycle that has no more nodes
                    toDeleteCycles.add(cycle);
                }

            }
            removeEffectiveSeasonalCropCycles(toDeleteCycles);
        }
    }

    private EffectiveSeasonalCropCycle getEffectiveSeasonalCropCycle(Zone zone, Set<EffectiveSeasonalCropCycle> toDeleteCycles, Map<String, EffectiveSeasonalCropCycle> idToCycles, String cycleId) {
        EffectiveSeasonalCropCycle cycle;
        if (StringUtils.isBlank(cycleId)) {
            cycle = effectiveSeasonalCropCycleDao.newInstance();
            cycle.setZone(zone);
        } else {
            cycle = idToCycles.get(cycleId);
            toDeleteCycles.remove(cycle);
        }
        return cycle;
    }

    private void manageCycleConnections(List<CroppingPlanEntry> targetedZoneCroppingPlanEntries,
                                        EffectiveSeasonalCropCycleDto cycleDto,
                                        Map<String, EffectiveCropCycleNode> nodeIdToEntity,
                                        EffectiveSeasonalCropCycle cycle) {
        Collection<EffectiveCropCycleConnection> cycleConnections = effectiveCropCycleConnectionDao.findAllByEffectiveSeasonalCropCycle(cycle);
        Map<String, EffectiveCropCycleConnection> connectionByTarget = Maps.newHashMap(Maps.uniqueIndex(cycleConnections, EFFECTIVE_CONNECTION_BY_TARGET_ID));

        if (CollectionUtils.isNotEmpty(cycleDto.getConnectionDtos())) {
            for (EffectiveCropCycleConnectionDto connectionDto : cycleDto.getConnectionDtos()) {
                String sourceId = null;
                if (connectionDto.getSourceId() != null) {
                    if (EffectiveCropCycleNodeDto.NODE_BEFORE_ID.equals(connectionDto.getSourceId())) {
                        sourceId = EffectiveCropCycleNodeDto.NODE_BEFORE_ID;
                    } else {
                        sourceId = Entities.UNESCAPE_TOPIA_ID.apply(connectionDto.getSourceId());
                    }
                }

                String targetId = Entities.UNESCAPE_TOPIA_ID.apply(connectionDto.getTargetId());
                Preconditions.checkArgument(StringUtils.isNotBlank(targetId), "La connexion n'a pas de noeud cible");

                EffectiveCropCycleNode sourceNode = nodeIdToEntity.get(sourceId);
                EffectiveCropCycleNode targetNode = nodeIdToEntity.get(targetId);

                if (targetNode == null) {
                    // if targeted node is null we continue to remove the un valid connection
                    continue;
                }

                EffectiveCropCycleConnection connection = connectionByTarget.remove(targetId);

                if (connection == null) {
                    connection = effectiveCropCycleConnectionDao.newInstance();
                }

                connection.setEdaplosIssuerId(connectionDto.getEdaplosIssuerId());
                connection.setSource(sourceNode);
                connection.setTarget(targetNode);
                CroppingPlanEntry intermediateCrops = null;
                if (StringUtils.isNotBlank(connectionDto.getIntermediateCroppingPlanEntryId())) {
                    intermediateCrops = croppingPlanEntryDao.forTopiaIdEquals(connectionDto.getIntermediateCroppingPlanEntryId()).findUnique();
                }
                if (intermediateCrops == null) {
                    connection.setIntermediateCroppingPlanEntry(null);
                }
                if (intermediateCrops != null && targetedZoneCroppingPlanEntries.contains(intermediateCrops)) {
                    connection.setIntermediateCroppingPlanEntry(intermediateCrops);
                }
                if (connection.isPersisted()) {
                    effectiveCropCycleConnectionDao.update(connection);
                } else {
                    effectiveCropCycleConnectionDao.create(connection);
                }
            }
        }

        effectiveCropCycleConnectionDao.deleteAll(connectionByTarget.values());
    }

    private void updateCycleNodesAndConnections(Domain domain, List<CroppingPlanEntry> targetedZoneCroppingPlanEntries,
                                                EffectiveSeasonalCropCycleDto cycleDto, EffectiveSeasonalCropCycle cycle) {
        // map to link connection to nodes
        Map<String, EffectiveCropCycleNode> nodeIdToEntity = Maps.newHashMap();

        Collection<EffectiveCropCycleNode> result = cycle.getNodes();
        if (result == null) {
            result = Lists.newArrayList();
            cycle.setNodes(result);
        }


        List<EffectiveCropCycleNodeDto> nodeDtos = cycleDto.getNodeDtos();

        if (nodeDtos != null) {

            Map<String, CroppingPlanEntry> nodesCropsByIds = getNodesCropsByIds(nodeDtos);

            for (EffectiveCropCycleNodeDto nodeDto : nodeDtos) {//new-node-b486e417-08ec-45c1-f4a1-4dca84cd37c2
                String nodeId = Entities.UNESCAPE_TOPIA_ID.apply(nodeDto.getNodeId());
                Preconditions.checkNotNull(nodeId);

                // this node is boring, just ignore him ;)
                if (nodeId.equalsIgnoreCase(EffectiveCropCycleNodeDto.NODE_BEFORE_ID)) {
                    continue;
                }

                EffectiveCropCycleNode node;
                if (nodeId.startsWith(NEW_NODE_PREFIX)) {
                    node = effectiveCropCycleNodeDao.newInstance();
                } else {
                    node = TopiaEntities.findByTopiaId(cycle.getNodes(), nodeId);
                }

                node.setRank(nodeDto.getX());
                node.setEdaplosIssuerId(nodeDto.getEdaplosIssuerId());

                CroppingPlanEntry croppingPlanEntry;
                if (StringUtils.isNotBlank(nodeDto.getCroppingPlanEntryId())) {
                    croppingPlanEntry = nodesCropsByIds.get(nodeDto.getCroppingPlanEntryId());
                } else {
                    continue;
                }

                Map<String, CroppingPlanEntry> targetedZoneCroppingPlanEntriesByCodes = Maps.uniqueIndex(targetedZoneCroppingPlanEntries, DomainServiceImpl.GET_CROPPING_PLAN_ENTRY_CODE);
                CroppingPlanEntry targetedCroppingPlanEntry = targetedZoneCroppingPlanEntriesByCodes.get(croppingPlanEntry.getCode());
                if ((targetedCroppingPlanEntry == null)){
                    continue;
                }

                node.setCroppingPlanEntry(targetedCroppingPlanEntry);

                if (!node.isPersisted()) {
                    result.add(node);
                }
                nodeIdToEntity.put(nodeId, node);

                // node must be persisted for intervention saving
                if (node.isPersisted()) {
                    node = effectiveCropCycleNodeDao.update(node);
                } else {
                    node = effectiveCropCycleNodeDao.create(node);
                }

                // node's interventions
                updateEffectiveInterventions(nodeDto.getInterventions(), cycleDto, Pair.of(nodeId, node), null, null, domain);
            }
        }
        // manage cycle connections
        manageCycleConnections(targetedZoneCroppingPlanEntries, cycleDto, nodeIdToEntity, cycle);

        Collection<EffectiveCropCycleNode> nodesToDelete = CollectionUtils.subtract(result, nodeIdToEntity.values());

        // delete all intervention for node to delete
        removeNodesChilds(nodesToDelete);

        // delete node (after connections)
        result.retainAll(nodeIdToEntity.values());

        //effectiveCropCycleNodeDao.deleteAll(nodesToDelete);
    }

    protected Map<String, CroppingPlanEntry> getNodesCropsByIds(List<EffectiveCropCycleNodeDto> nodeDtos) {
        Set<String> cropIds = Sets.newHashSet();
        for (EffectiveCropCycleNodeDto nodeDto : nodeDtos) {
            if (StringUtils.isNotBlank(nodeDto.getCroppingPlanEntryId())) {
                cropIds.add(nodeDto.getCroppingPlanEntryId());
            }
        }
        List<CroppingPlanEntry> nodesCrops = croppingPlanEntryDao.forTopiaIdIn(cropIds).findAll();
        return Maps.uniqueIndex(nodesCrops, Entities.GET_TOPIA_ID);
    }

    protected void removeEffectiveSeasonalCropCycles(Set<EffectiveSeasonalCropCycle> cycles) {
        if (cycles != null) {
            List<EffectiveCropCycleConnection> connectionsToRemove = Lists.newArrayList();
            for (EffectiveSeasonalCropCycle cycle : cycles) {
                Collection<EffectiveCropCycleNode> nodes = cycle.getNodes();
                connectionsToRemove.addAll(getConnectionsToRemove(nodes));
            }
            effectiveCropCycleConnectionDao.deleteAll(connectionsToRemove);
            effectiveSeasonalCropCycleDao.deleteAll(cycles);// nodes are remove from compo.
        }

    }

    protected void removeNodesChilds(Collection<EffectiveCropCycleNode> nodesToRemove) {
        List<EffectiveCropCycleConnection>  connectionsToRemove = getConnectionsToRemove(nodesToRemove);
        if (connectionsToRemove != null && !connectionsToRemove.isEmpty()) {
            effectiveCropCycleConnectionDao.deleteAll(connectionsToRemove);
        }
    }

    protected List<EffectiveCropCycleConnection> getConnectionsToRemove(Collection<EffectiveCropCycleNode> nodes) {
        List<EffectiveCropCycleConnection> result = Lists.newArrayList();
        if (nodes != null && !nodes.isEmpty()) {
            Collection<EffectiveIntervention> effectiveInterventions = effectiveInterventionDao.forEffectiveCropCycleNodeIn(nodes).findAll();
            removeEffectiveInterventions(effectiveInterventions);

            for (EffectiveCropCycleNode node : nodes) {
                EffectiveCropCycleConnection target = effectiveCropCycleConnectionDao.forTargetEquals(node).findUniqueOrNull();
                if (target != null) {
                    result.add(target);
                }
            }
        }
        return result;
    }

    /**
     * Update perennial crop cycles linked to plot.
     *
     * @param zoneId                       the identifier of the zone on which the crop cycle is about
     * @param effectivePerennialCropCycles perennial crop cycles
     */
    protected void udpateEffectivePerennialCropCycles(String zoneId, List<EffectivePerennialCropCycleDto> effectivePerennialCropCycles) {

        if (effectivePerennialCropCycles != null) { // null can be valid only for test

            Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();
            Domain domain = zone.getPlot().getDomain();

            Domain targetedDomain = zone.getPlot().getDomain();
            List<CroppingPlanEntry> targetedZoneCroppingPlanEntries = croppingPlanEntryDao.forDomainEquals(targetedDomain).findAll();

            // a cycle is related to a cropping plan. If there are no cropping plan for the targeted zone so cycle can not be attached.
            Preconditions.checkArgument(!targetedZoneCroppingPlanEntries.isEmpty());

            Map<String, CroppingPlanSpecies> availableCroppingPlanSpecies = getCroppingPlanSpecies(targetedZoneCroppingPlanEntries);

            List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();
            List<EffectivePerennialCropCycle> toDelete = Lists.newArrayList(cycles);
            Map<String, EffectivePerennialCropCycle> idToCycles = Maps.uniqueIndex(cycles, Entities.GET_TOPIA_ID);
            for (EffectivePerennialCropCycleDto cycleDto : effectivePerennialCropCycles) {

                CroppingPlanEntry croppingPlanEntry = croppingPlanEntryDao.forTopiaIdEquals(cycleDto.getCroppingPlanEntryId()).findUnique();
                if (!targetedZoneCroppingPlanEntries.contains(croppingPlanEntry)) {
                    // cycle cropping plan does not append to the targeted zone so it can no be affected.
                    continue;
                }

                // current cycle
                String dtoId = cycleDto.getTopiaId();
                EffectivePerennialCropCycle cycle = getEffectivePerennialCropCycle(zone, toDelete, idToCycles, dtoId);

                Binder<EffectivePerennialCropCycleDto, EffectivePerennialCropCycle> binder = BinderFactory.newBinder(EffectivePerennialCropCycleDto.class, EffectivePerennialCropCycle.class);
                binder.copyExcluding(cycleDto, cycle,
                        EffectivePerennialCropCycle.PROPERTY_ZONE,
                        EffectivePerennialCropCycle.PROPERTY_TOPIA_ID,
                        EffectivePerennialCropCycle.PROPERTY_ORIENTATION,
                        EffectivePerennialCropCycle.PROPERTY_PHASE,
                        EffectivePerennialCropCycle.PROPERTY_SPECIES,
                        EffectivePerennialCropCycle.PROPERTY_CROPPING_PLAN_ENTRY);
                cycle.setCroppingPlanEntry(croppingPlanEntry);

                RefOrientationEDI orientation = loadRefOrientationEDI(cycleDto);
                cycle.setOrientation(orientation);

                Preconditions.checkState(cycleDto.getPhaseDtos().size() == 1);
                // cycle's phases
                managePhases(domain, cycleDto, cycle);

                // cycle's species
                managePhaseSpecies(availableCroppingPlanSpecies, cycleDto, cycle);

                // persist cycle
                if (StringUtils.isBlank(dtoId)) {
                    effectivePerennialCropCycleDao.create(cycle);
                } else {
                    effectivePerennialCropCycleDao.update(cycle);
                }
            }
            removeEffectivePerennialCropCycles(toDelete);
        }
    }

    protected Map<String, CroppingPlanSpecies> getCroppingPlanSpecies(List<CroppingPlanEntry> targetedZoneCroppingPlanEntries) {
        Map<String, CroppingPlanSpecies> availableCroppingPlanSpecies = new HashMap<String, CroppingPlanSpecies>();
        for (CroppingPlanEntry croppingPlanEntry : targetedZoneCroppingPlanEntries) {
            if (croppingPlanEntry != null) {
                if (CollectionUtils.isNotEmpty(croppingPlanEntry.getCroppingPlanSpecies())) {
                    for (CroppingPlanSpecies croppingPlanSpecies : croppingPlanEntry.getCroppingPlanSpecies()) {
                        if (croppingPlanSpecies != null) {
                            availableCroppingPlanSpecies.put(croppingPlanSpecies.getTopiaId(), croppingPlanSpecies);
                        }
                    }
                }
            }
        }
        return availableCroppingPlanSpecies;
    }

    private void managePhaseSpecies(Map<String, CroppingPlanSpecies> availableCroppingPlanSpecies, EffectivePerennialCropCycleDto cycleDto, EffectivePerennialCropCycle cycle) {
        if (cycleDto.getSpeciesDtos() != null) {
            Collection<EffectiveCropCycleSpecies> cycleSpeciesList = cycle.getSpecies();
            if (cycleSpeciesList == null) {
                cycleSpeciesList = Lists.newArrayList();
                cycle.setSpecies(cycleSpeciesList);
            }
            Collection<EffectiveCropCycleSpecies> nonDeletedSpecies = Lists.newArrayList();
            Map<String, EffectiveCropCycleSpecies> idToSpecies = Maps.uniqueIndex(cycleSpeciesList, Entities.GET_TOPIA_ID);
            for (EffectiveCropCycleSpeciesDto speciesDto : cycleDto.getSpeciesDtos()) {
                String cycleSpeciesId = speciesDto.getTopiaId();
                EffectiveCropCycleSpecies cycleSpecies;
                if (StringUtils.isBlank(cycleSpeciesId)) {
                    cycleSpecies = effectiveCropCycleSpeciesDao.newInstance();
                } else {
                    cycleSpecies = idToSpecies.get(cycleSpeciesId);
                }

                String targetedCroppingPlanSpeciesId = speciesDto.getCroppingPlanSpeciesId();
                if (StringUtils.isNotBlank(targetedCroppingPlanSpeciesId) && availableCroppingPlanSpecies.containsKey(targetedCroppingPlanSpeciesId)) {
                    CroppingPlanSpecies croppingPlanSpecies = croppingPlanSpeciesDao.forTopiaIdEquals(targetedCroppingPlanSpeciesId).findUnique();
                    cycleSpecies.setCroppingPlanSpecies(croppingPlanSpecies);
                } else {
                    // EffectiveCropCycleSpecies related to a species than not belong to the zone
                    continue;
                }

                cycleSpecies.setPlantsCertified(speciesDto.isPlantsCertified());
                cycleSpecies.setOverGraftDate(speciesDto.getOverGraftDate());

                RefClonePlantGrape cloneGrape = null;
                if (speciesDto.getGraftClone() != null && StringUtils.isNotBlank(speciesDto.getGraftClone().getTopiaId())) {
                    cloneGrape = refClonePlantGrapeDao.forTopiaIdEquals(speciesDto.getGraftClone().getTopiaId()).findUnique();
                }
                cycleSpecies.setGraftClone(cloneGrape);

                RefVariete variete = null;
                if (speciesDto.getGraftSupport() != null && StringUtils.isNotBlank(speciesDto.getGraftSupport().getTopiaId())) {
                    variete = refVarieteDao.forTopiaIdEquals(speciesDto.getGraftSupport().getTopiaId()).findUnique();
                }
                cycleSpecies.setGraftSupport(variete);

                if (StringUtils.isBlank(cycleSpeciesId)) {
                    cycle.addSpecies(cycleSpecies);
                }
                nonDeletedSpecies.add(cycleSpecies);
            }
            cycleSpeciesList.retainAll(nonDeletedSpecies);
        }
    }

    private void managePhases(Domain domain, EffectivePerennialCropCycleDto cycleDto, EffectivePerennialCropCycle cycle) {
        EffectiveCropCyclePhase phase = cycle.getPhase();
        if (phase == null) {
            phase = effectiveCropCyclePhaseDao.newInstance();
        }
        for (EffectiveCropCyclePhaseDto phaseDto : cycleDto.getPhaseDtos()) {// for the moment there is only one phase
            phase.setDuration(phaseDto.getDuration());
            phase.setType(phaseDto.getType());

            // phases must be persisted for intervention saving
            if (phase.isPersisted()) {
                phase = effectiveCropCyclePhaseDao.update(phase);
            } else {
                phase = effectiveCropCyclePhaseDao.create(phase);
            }
            cycle.setPhase(phase);
            // phase's interventions
            updateEffectiveInterventions(phaseDto.getInterventions(), null, null, cycle, phase, domain);
        }
    }

    private RefOrientationEDI loadRefOrientationEDI(EffectivePerennialCropCycleDto cycleDto) {
        RefOrientationEDI orientation = null;
        if (cycleDto.getOrientationId() != null) {
            orientation = refOrientationEDIDao.forTopiaIdEquals(cycleDto.getOrientationId()).findUnique();
        }
        return orientation;
    }

    private EffectivePerennialCropCycle getEffectivePerennialCropCycle(Zone zone, List<EffectivePerennialCropCycle> toDelete, Map<String, EffectivePerennialCropCycle> idToCycles, String dtoId) {
        EffectivePerennialCropCycle cycle;
        if (StringUtils.isBlank(dtoId)) {
            cycle = effectivePerennialCropCycleDao.newInstance();
            cycle.setZone(zone);
        } else {
            cycle = idToCycles.get(dtoId);
            toDelete.remove(cycle);
        }
        return cycle;
    }


    protected void removeEffectivePerennialCropCycles(List<EffectivePerennialCropCycle> effectivePerennialCropCycles) {
        if (effectivePerennialCropCycles != null) {
            for (EffectivePerennialCropCycle effectivePerennialCropCycle : effectivePerennialCropCycles) {
                EffectiveCropCyclePhase phase = effectivePerennialCropCycle.getPhase();
                removeEffectiveCropCyclePhaseChildrenObjects(phase);
            }
            effectivePerennialCropCycleDao.deleteAll(effectivePerennialCropCycles);
        }
    }

    protected void removeEffectiveCropCyclePhaseChildrenObjects(EffectiveCropCyclePhase phase) {
        List<EffectiveIntervention> effectiveInterventions = effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).findAll();
        removeEffectiveInterventions(effectiveInterventions);
    }

    private void removeEffectiveInterventions(Collection<EffectiveIntervention> effectiveInterventions) {
        if (effectiveInterventions != null) {
            List<AbstractInput> allInputsToRemove = Lists.newArrayList();
            List<AbstractAction> allActionsToRemove = Lists.newArrayList();
            for (EffectiveIntervention intervention : effectiveInterventions) {
                List<AbstractInput> abstractInputs = abstractInputTopiaDao.findAllByEffectiveIntervention(intervention);
                if (abstractInputs != null) {
                    allInputsToRemove.addAll(abstractInputs);
                }

                List<AbstractAction> abstractActions = abstractActionDao.forEffectiveInterventionEquals(intervention).findAll();
                // ne peut-être null
                allActionsToRemove.addAll(abstractActions);

            }
            abstractInputTopiaDao.deleteAll(allInputsToRemove);
            effectiveInterventionDao.deleteAll(effectiveInterventions);
            abstractActionDao.deleteAll(allActionsToRemove);
        }
    }

    protected void updateEffectivePrices(String zoneId, List<Price> prices) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();
        Domain domain = zone.getPlot().getDomain();

        pricesService.updatePrices(prices, domain, null);
    }

    /**
     * Persist interventions on node OR phase.
     */
    protected void updateEffectiveInterventions(List<EffectiveInterventionDto> interventionDtos, EffectiveSeasonalCropCycleDto seasonalCropCycle,
                                                Pair<String, EffectiveCropCycleNode> nodeIdToNode, EffectivePerennialCropCycle perennialCropCycle, EffectiveCropCyclePhase phase, Domain domain) {

        // either one of node or phase MUST be null (but not both)
        Preconditions.checkArgument(nodeIdToNode == null ^ phase == null);

        interventionDtos = interventionDtos != null ? interventionDtos : new ArrayList<EffectiveInterventionDto>();

        // remove new interventionDtos witch domainId doesn't match the domain topiaId (to avoid copy/paste of intervention from another domain).
        filterOnDomain(interventionDtos, domain);

        List<EffectiveIntervention> interventions;
        if (nodeIdToNode != null && nodeIdToNode.getRight().isPersisted()) {
            interventions = effectiveInterventionDao.forEffectiveCropCycleNodeEquals(nodeIdToNode.getRight()).findAll();
        } else if (phase != null && phase.isPersisted()){ // can be null from test
            interventions = effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).findAll();
        } else {
            interventions = new ArrayList<EffectiveIntervention>();
        }

        Map<String, EffectiveIntervention> idsToInterventions = Maps.newHashMap(Maps.uniqueIndex(interventions, Entities.GET_TOPIA_ID));

        List<EffectiveCropCycleConnectionDto> connectionDtos = seasonalCropCycle != null ? seasonalCropCycle.getConnectionDtos() : null;
        interventionDtosToInterventions(new InterventionDtoToInterventionUnivers(interventionDtos, connectionDtos, nodeIdToNode, perennialCropCycle, phase, domain, idsToInterventions));

        effectiveInterventionDao.deleteAll(idsToInterventions.values());
    }

    protected void filterOnDomain(List<EffectiveInterventionDto> interventionDtos, Domain domain) {
        Iterator<EffectiveInterventionDto> validInterventionDtos = interventionDtos.iterator();
        while(validInterventionDtos.hasNext()) {
            EffectiveInterventionDto interventionDto = validInterventionDtos.next();
            if (Strings.isNullOrEmpty(interventionDto.getTopiaId()) && !interventionDto.getDomainId().equals(domain.getTopiaId())) {
                validInterventionDtos.remove();
            }
        }
    }

    private void interventionDtosToInterventions(InterventionDtoToInterventionUnivers interventionDtoToInterventionUnivers) {
        for (EffectiveInterventionDto interventionDto : interventionDtoToInterventionUnivers.getInterventionDtos()) {
            String interventionId = interventionDto.getTopiaId();
            // create new intervention instance
            EffectiveCropCycleNode node = interventionDtoToInterventionUnivers.getNodeIdToNode() != null ? interventionDtoToInterventionUnivers.getNodeIdToNode().getRight() : null;
            EffectiveIntervention intervention = getEffectiveIntervention(node, interventionDtoToInterventionUnivers.getPhase(), interventionDtoToInterventionUnivers.getIdsToInterventions(), interventionId);

            Map<String, CroppingPlanSpecies> phaseOrNodeSpeciesMap = getPhaseOrNodeCroppingPlanSpeciesByCode(interventionDto.isIntermediateCrop(), interventionDtoToInterventionUnivers.getNodeIdToNode(), interventionDtoToInterventionUnivers.getConnectionDtos(), interventionDtoToInterventionUnivers.getPerennialCropCycle());
            Set<String> speciesCodes = phaseOrNodeSpeciesMap.keySet();

            bindDataToIntervention(interventionDtoToInterventionUnivers.getDomain(), interventionDto, intervention, phaseOrNodeSpeciesMap);

            if (!intervention.isPersisted()) {
                EffectiveIntervention persistedIntervention = effectiveInterventionDao.create(intervention);
                Map<String, AbstractAction> actions = actionService.createEffectiveInterventionActions(persistedIntervention, interventionDto.getActions(), speciesCodes);
                inputService.createInterventionInputs(null, persistedIntervention, interventionDto.getInputs(), actions);
            } else {
                Map<String, AbstractAction> actions = actionService.updateEffectiveInterventionActions(intervention, interventionDto.getActions(), speciesCodes);
                inputService.updateInterventionInputs(null, intervention, interventionDto.getInputs(), actions);
            }
        }
    }

    private void bindDataToIntervention(Domain domain, EffectiveInterventionDto interventionDto, EffectiveIntervention intervention, Map<String, CroppingPlanSpecies> phaseOrNodeSpeciesMap) {
        Binder<EffectiveInterventionDto, EffectiveIntervention> binder = BinderFactory.newBinder(EffectiveInterventionDto.class, EffectiveIntervention.class);

        binder.copyExcluding(interventionDto, intervention,
                EffectiveIntervention.PROPERTY_TOPIA_ID,
                EffectiveIntervention.PROPERTY_TOOL_COUPLINGS,
                EffectiveIntervention.PROPERTY_SPECIES_STADES,
                EffectiveIntervention.PROPERTY_EFFECTIVE_CROP_CYCLE_NODE,
                EffectiveIntervention.PROPERTY_EFFECTIVE_CROP_CYCLE_PHASE);

        // intervention tools couplings
        bindToolsCouplingToIntervention(domain, interventionDto, intervention);

        // intervention speciesStade
        bindSpeciesStadesToIntervention(interventionDto, intervention, phaseOrNodeSpeciesMap);
    }

    // DCossé valider cette méthode
    private void bindSpeciesStadesToIntervention(EffectiveInterventionDto interventionDto, EffectiveIntervention intervention, Map<String, CroppingPlanSpecies> phaseOrNodeSpeciesMap0) {
        Collection<EffectiveSpeciesStade> speciesStades = intervention.getSpeciesStades();
        Map<String, CroppingPlanSpecies> phaseOrNodeSpeciesMap = Maps.newHashMap(phaseOrNodeSpeciesMap0);
        if (speciesStades == null) {
            speciesStades = Lists.newArrayList();
            intervention.setSpeciesStades(speciesStades);
        }
        if (interventionDto.getSpeciesStadesDtos() != null) {
            Collection<EffectiveSpeciesStade> nonDeletedStades = Lists.newArrayList();
            Map<String, EffectiveSpeciesStade> idToSpeciesStade = Maps.uniqueIndex(speciesStades, Entities.GET_TOPIA_ID);

            for (SpeciesStadeDto speciesStadeDto : interventionDto.getSpeciesStadesDtos()) {
                String speciesStadeId = speciesStadeDto.getTopiaId();

                CroppingPlanSpecies croppingPlanSpecies = null;
                String speciesCode = speciesStadeDto.getSpeciesCode();
                if (speciesCode != null) {
                    croppingPlanSpecies = phaseOrNodeSpeciesMap.remove(speciesCode);
                }

                // if the croppingPlanSpecies was not found on the available one, SpeciesStade is not valid.
                if (croppingPlanSpecies == null) {
                    continue;
                }

                EffectiveSpeciesStade effectiveSpeciesStade = getEffectiveSpeciesStade(speciesStades, idToSpeciesStade, speciesStadeDto, speciesStadeId, croppingPlanSpecies);
                nonDeletedStades.add(effectiveSpeciesStade);
            }
            speciesStades.retainAll(nonDeletedStades);
        } else {
            // in case of no species stades where defined, default ones are created with the crop species.
            for (CroppingPlanSpecies croppingPlanSpecies : phaseOrNodeSpeciesMap.values()) {
                EffectiveSpeciesStade effectiveSpeciesStade = effectiveSpeciesStadeDao.newInstance();
                effectiveSpeciesStade.setCroppingPlanSpecies(croppingPlanSpecies);
                speciesStades.add(effectiveSpeciesStade);
            }
        }
    }

    protected EffectiveSpeciesStade getEffectiveSpeciesStade(Collection<EffectiveSpeciesStade> speciesStades, Map<String, EffectiveSpeciesStade> idToSpeciesStade, SpeciesStadeDto speciesStadeDto, String speciesStadeId, CroppingPlanSpecies croppingPlanSpecies) {
        EffectiveSpeciesStade effectiveSpeciesStade = null;
        if (StringUtils.isNotBlank(speciesStadeId)) {
            effectiveSpeciesStade = idToSpeciesStade.get(speciesStadeId);

        }
        if (effectiveSpeciesStade == null) {
            effectiveSpeciesStade = effectiveSpeciesStadeDao.newInstance();
        }

        effectiveSpeciesStade.setCroppingPlanSpecies(croppingPlanSpecies);

        RefStadeEDI minStade = null;
        RefStadeEdiDto stadeMinFromDto = speciesStadeDto != null ? speciesStadeDto.getStadeMin() : null;
        if (stadeMinFromDto != null) {
            minStade = refStadeEDIDao.forTopiaIdEquals(stadeMinFromDto.getTopiaId()).findUnique();
        }
        effectiveSpeciesStade.setMinStade(minStade);
        RefStadeEDI maxStade = null;
        RefStadeEdiDto stadeMaxFromDto = speciesStadeDto != null ? speciesStadeDto.getStadeMax() : null;
        if (stadeMaxFromDto != null) {
            maxStade = refStadeEDIDao.forTopiaIdEquals(stadeMaxFromDto.getTopiaId()).findUnique();
        }
        effectiveSpeciesStade.setMaxStade(maxStade);

        if (StringUtils.isBlank(speciesStadeId)) {
            speciesStades.add(effectiveSpeciesStade);
        }
        return effectiveSpeciesStade;
    }

    private void bindToolsCouplingToIntervention(Domain domain, EffectiveInterventionDto interventionDto, EffectiveIntervention intervention) {
        if (interventionDto.getToolsCouplingCodes() != null) {
            Collection<ToolsCoupling> toolsCouplings = intervention.getToolCouplings();
            if (toolsCouplings == null) {
                toolsCouplings = Lists.newArrayList();
                intervention.setToolCouplings(toolsCouplings);
            }
            toolsCouplings.clear();
            for (String toolsCouplingCodes : interventionDto.getToolsCouplingCodes()) {
                ToolsCoupling toolsCoupling = toolsCouplingDao.forProperties(ToolsCoupling.PROPERTY_CODE, toolsCouplingCodes, ToolsCoupling.PROPERTY_DOMAIN, domain).findAnyOrNull();
                if (toolsCoupling != null) {
                    toolsCouplings.add(toolsCoupling);
                }
            }
        }
    }

    private Map<String, CroppingPlanSpecies> getPhaseOrNodeCroppingPlanSpeciesByCode(boolean isIntermediate, Pair<String, EffectiveCropCycleNode> nodeIdToNode, List<EffectiveCropCycleConnectionDto> connectionDtos, EffectivePerennialCropCycle perennialCropCycle) {
        // find all species from cycle crop
        Set<CroppingPlanSpecies> allowedSpecies = Sets.newHashSet();
        if (nodeIdToNode != null) {

            // add species from intermediate crops
            if (isIntermediate && connectionDtos != null) {
                if (CollectionUtils.isNotEmpty(connectionDtos)) {
                    Map<String, CroppingPlanEntry> loadedCPEcach = Maps.newHashMap();

                    for (EffectiveCropCycleConnectionDto connectionDto : connectionDtos) {
                        String targetedNodeId = connectionDto.getIntermediateCroppingPlanEntryId() != null ? Entities.UNESCAPE_TOPIA_ID.apply(connectionDto.getTargetId()) : null;
                        if (targetedNodeId != null && targetedNodeId.equals(nodeIdToNode.getLeft())){
                            CroppingPlanEntry cpe = loadedCPEcach.get(connectionDto.getIntermediateCroppingPlanEntryId());
                            if (cpe == null) {
                                cpe = croppingPlanEntryDao.forTopiaIdEquals(connectionDto.getIntermediateCroppingPlanEntryId()).findUnique();
                                loadedCPEcach.put(cpe.getTopiaId(), cpe);
                            }
                            if (cpe.getCroppingPlanSpecies() != null) {
                                allowedSpecies.addAll(cpe.getCroppingPlanSpecies());
                            }
                            break;
                        }
                    }
                }
            } else {
                // all species from node
                allowedSpecies = Sets.newHashSet(nodeIdToNode.getRight().getCroppingPlanEntry().getCroppingPlanSpecies());
            }
        } else {
            if (perennialCropCycle != null) {
                allowedSpecies = Sets.newHashSet(perennialCropCycle.getCroppingPlanEntry().getCroppingPlanSpecies());
            }
        }
        Collection<CroppingPlanSpecies> filtratedAllowedSpecies = Collections2.filter(allowedSpecies, Predicates.notNull());
        Map<String, CroppingPlanSpecies> result = CollectionUtils.isNotEmpty(filtratedAllowedSpecies) ? Maps.uniqueIndex(filtratedAllowedSpecies, GET_CROPPING_PLAN_SPECIES_CODE) : new HashMap<String, CroppingPlanSpecies>(0);
        return result;
    }

    private EffectiveIntervention getEffectiveIntervention(EffectiveCropCycleNode node, EffectiveCropCyclePhase phase, Map<String, EffectiveIntervention> phaseOrNodeInterventionsByIds, String interventionId) {
        EffectiveIntervention intervention;
        if (StringUtils.isBlank(interventionId)) {
            intervention = effectiveInterventionDao.newInstance();
            if (node != null) {
                intervention.setEffectiveCropCycleNode(node);
            } else {
                intervention.setEffectiveCropCyclePhase(phase);
            }
        } else {
            intervention = phaseOrNodeInterventionsByIds.remove(interventionId);
        }
        return intervention;
    }

    @Override
    public List<Price> getEffectivePrices(String zoneId, List<String> objectIds) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(zoneId));
        Zone zone = zoneDao.forTopiaIdEquals(zoneId).findUnique();
        String domainId = zone.getPlot().getDomain().getTopiaId();
        List<Price> result = pricesService.getDomainPrices(domainId, objectIds);
        return result;
    }

    @Override
    public List<ToolsCouplingDto> getToolsCouplingModel(String zoneTopiaId, final AgrosystInterventionType interventionType) {
        Zone zone = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        Domain domain = zone.getPlot().getDomain();
        List<ToolsCoupling> toolsCouplings = toolsCouplingDao.forDomainEquals(domain).findAll();
        Iterable<ToolsCoupling> filtered = Iterables.filter(toolsCouplings, new Predicate<ToolsCoupling>() {
            @Override
            public boolean apply(ToolsCoupling toolsCoupling) {
                return interventionType == null || Iterables.tryFind(toolsCoupling.getMainsActions(), new Predicate<RefInterventionAgrosystTravailEDI>() {
                    @Override
                    public boolean apply(RefInterventionAgrosystTravailEDI mainAction) {
                        return interventionType.equals(mainAction.getIntervention_agrosyst());
                    }
                }).isPresent();
            }
        });
        List<ToolsCouplingDto> result = Lists.newArrayList(Iterables.transform(filtered, Equipments.TOOLS_COUPLING_TO_TOOLS_COUPLING_DTO));
        return result;
    }

    @Override
    public void duplicateEffectiveCropCycles(String fromZoneId, String toZoneId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(fromZoneId));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(toZoneId));

        List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycleDtos = getAllEffectiveSeasonalCropCyclesDtos(fromZoneId);
        List<EffectivePerennialCropCycleDto> effectivePerennialCropCycleDtos = getAllEffectivePerennialCropCyclesDtos(fromZoneId);

        for (EffectiveSeasonalCropCycleDto seasonalCropCycleDto : effectiveSeasonalCropCycleDtos) {
            seasonalCropCycleDto.setTopiaId(null);
            List<EffectiveCropCycleNodeDto> nodesDto = seasonalCropCycleDto.getNodeDtos();

            Map<String, UUID> mappingUUIDs = Maps.newHashMap();
            for (EffectiveCropCycleNodeDto nodeDto : nodesDto) {
                UUID uuid = UUID.randomUUID();
                mappingUUIDs.put(nodeDto.getNodeId(), uuid);
                nodeDto.setTopiaId(null);
                nodeDto.setNodeId(NEW_NODE_PREFIX + uuid.toString());
                List<EffectiveInterventionDto> interventionDtos = nodeDto.getInterventions();
                nodeDto.setInterventions(toNewInterventionDtos(interventionDtos));
            }

            List<EffectiveCropCycleConnectionDto> connectionDtos = seasonalCropCycleDto.getConnectionDtos();
            for (EffectiveCropCycleConnectionDto cycleConnectionDto:connectionDtos) {
                cycleConnectionDto.setTopiaId(null);
                String sourceId = StringUtils.isBlank(cycleConnectionDto.getSourceId()) ? null : NEW_NODE_PREFIX + mappingUUIDs.get(cycleConnectionDto.getSourceId());
                cycleConnectionDto.setSourceId(sourceId);
                cycleConnectionDto.setTargetId(NEW_NODE_PREFIX + mappingUUIDs.get(cycleConnectionDto.getTargetId()));
            }

        }

        for (EffectivePerennialCropCycleDto effectivePerennialCropCycleDto:effectivePerennialCropCycleDtos) {
            effectivePerennialCropCycleDto.setTopiaId(null);
            List<EffectiveCropCyclePhaseDto> phases = effectivePerennialCropCycleDto.getPhaseDtos();
            for (EffectiveCropCyclePhaseDto phase:phases) {
                phase.setTopiaId(null);
                List<EffectiveInterventionDto> interventionDtos = phase.getInterventions();
                phase.setInterventions(toNewInterventionDtos(interventionDtos));
            }
        }
        updateEffectiveSeasonalCropCycles(toZoneId, effectiveSeasonalCropCycleDtos);
        udpateEffectivePerennialCropCycles(toZoneId, effectivePerennialCropCycleDtos);

        getTransaction().commit();
    }

    protected List<EffectiveInterventionDto> toNewInterventionDtos(List<EffectiveInterventionDto> interventionDtos) {
        List<EffectiveInterventionDto> newInterventionDtos = Lists.newArrayListWithCapacity(interventionDtos.size());
        for (EffectiveInterventionDto interventionDto:interventionDtos) {
            newInterventionDtos.add(toNewInterventionDto(interventionDto));
        }
        return newInterventionDtos;
    }

    protected EffectiveInterventionDto toNewInterventionDto(EffectiveInterventionDto interventionDto) {
        EffectiveInterventionDto newEffectiveInterventionDto = new EffectiveInterventionDto();

        bindEffectiveInterventionData(interventionDto, newEffectiveInterventionDto);

        setSpeciesStadeAsNonPersisted(newEffectiveInterventionDto);

        Map<AbstractAction, AbstractAction> actionsToDuplicatedActions = cloneActions(newEffectiveInterventionDto);

        cloneInputs(newEffectiveInterventionDto, actionsToDuplicatedActions);

        return newEffectiveInterventionDto;
    }

    protected Map<AbstractAction, AbstractAction> cloneActions(EffectiveInterventionDto newEffectiveInterventionDto) {
        Map<AbstractAction, AbstractAction> actionsToDuplicatedActions = Maps.newHashMap();
        for(AbstractAction action:newEffectiveInterventionDto.getActions()) {
            actionsToDuplicatedActions.put(action, actionService.getClonedAbstractAction(action));
        }
        newEffectiveInterventionDto.setActions(actionsToDuplicatedActions.values());
        return actionsToDuplicatedActions;
    }

    protected void cloneInputs(EffectiveInterventionDto newEffectiveInterventionDto, Map<AbstractAction, AbstractAction> actionsToDuplicatedActions) {
        List<AbstractInput> inputs = newEffectiveInterventionDto.getInputs();
        List<AbstractInput> duplicatedInputs = Lists.newArrayListWithCapacity(inputs.size());
        for(AbstractInput input:inputs) {
            duplicatedInputs.add(inputService.getDuplicatedAbstractInput(actionsToDuplicatedActions, input));
        }
        newEffectiveInterventionDto.setInputs(duplicatedInputs);
    }

    protected void setSpeciesStadeAsNonPersisted(EffectiveInterventionDto newEffectiveInterventionDto) {
        List<SpeciesStadeDto> speciesStadeDtos = newEffectiveInterventionDto.getSpeciesStadesDtos();
        if (speciesStadeDtos != null) {
            for (SpeciesStadeDto speciesStadeDto:speciesStadeDtos) {
                speciesStadeDto.setTopiaId(null);
            }
        }
    }

    protected void bindEffectiveInterventionData(EffectiveInterventionDto interventionDto, EffectiveInterventionDto newEffectiveInterventionDto) {
        Binder<EffectiveInterventionDto, EffectiveInterventionDto> binder = BinderFactory.newBinder(EffectiveInterventionDto.class);
        binder.copy(interventionDto, newEffectiveInterventionDto);
        newEffectiveInterventionDto.setTopiaId(null);
    }


    @Override
    public List<ZoneDto> getZones(Collection<String> zoneIds) {
        List<Zone> plots = zoneDao.forTopiaIdIn(zoneIds).findAll();
        List<ZoneDto> result = Lists.transform(plots, anonymizeService.getZoneToDtoFunction(false));
        return result;
    }

    @Override
    public InputStream exportEffectiveCropCyclesAsXlsStream(List<String> effectiveCropCycleIds) {
        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        // get all possible bean infos
        EffectiveCropCycleBeanInfo effectiveCropCycleBeanTab = newInstance(EffectiveCropCycleBeanInfo.class);
        EffectivePerennialCropCycleSpeciesBeanInfo perennialSpeciesTab = newInstance(EffectivePerennialCropCycleSpeciesBeanInfo.class);
        EffectiveITKBeanInfo effectiveITKBeanTab = newInstance(EffectiveITKBeanInfo.class);

        // add all bean infos
        ExportUtils.addAllBeanInfo(sheet, effectiveCropCycleBeanTab, perennialSpeciesTab, effectiveITKBeanTab);

        // too complex to be simplified
        List<EffectiveCropCycleExportEntity> perennialEntities = (List<EffectiveCropCycleExportEntity>)sheet.get(perennialSpeciesTab);
        List<EffectiveCropCycleExportEntity> itkEntities = (List<EffectiveCropCycleExportEntity>)sheet.get(effectiveITKBeanTab);

        try {
            if (CollectionUtils.isNotEmpty(effectiveCropCycleIds)) {
                Iterable<Zone> zones = zoneDao.forTopiaIdIn(effectiveCropCycleIds).findAll();
                for (Zone zone : zones) {

                    // anonymize zone
                    zone = anonymizeService.checkForZoneAnonymization(zone);

                    Domain domain = zone.getPlot().getDomain();
                    EffectiveCropCycleExportEntity model = new EffectiveCropCycleExportEntity();
                    model.setZoneName(zone.getName());
                    model.setPlotName(zone.getPlot().getName());
                    if (zone.getPlot().getGrowingSystem() != null) {
                        model.setGrowingSystemName(zone.getPlot().getGrowingSystem().getName());
                        model.setGrowingPlanName(zone.getPlot().getGrowingSystem().getGrowingPlan().getName());
                    }
                    model.setDomainName(domain.getName());
                    model.setCampaign(domain.getCampaign());

                    List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycleDtos = this.getAllEffectiveSeasonalCropCyclesDtos(zone.getTopiaId());
                    List<EffectivePerennialCropCycleDto> effectivePerennialCropCycleDtos = this.getAllEffectivePerennialCropCyclesDtos(zone.getTopiaId());

                    // cycle tab
                    exportEffectiveSeasonalCropCycles(effectiveSeasonalCropCycleDtos, sheet, effectiveCropCycleBeanTab, model);
                    ExportUtils.export(sheet, model, effectivePerennialCropCycleDtos, effectiveCropCycleBeanTab);

                    // perennial species tab
                    if (effectivePerennialCropCycleDtos != null) {
                        for (EffectivePerennialCropCycleDto perennialCropCycleDto : effectivePerennialCropCycleDtos) {
                            List<EffectiveCropCycleSpeciesDto> speciesDtos = perennialCropCycleDto.getSpeciesDtos();
                            if (!speciesDtos.isEmpty()) {
                                EffectiveCropCycleExportEntity export = (EffectiveCropCycleExportEntity) model.clone();
                                ExportUtils.copyFields(perennialCropCycleDto, export, null,
                                        "croppingPlanEntryName");
                                ExportUtils.setExtraField(export, "phase", perennialCropCycleDto.getPhaseDtos().get(0).getType());

                                for (EffectiveCropCycleSpeciesDto speciesDto : speciesDtos) {
                                    ExportUtils.copyFields(speciesDto, export, null,
                                            "speciesEspece",
                                            "speciesQualifiant",
                                            "speciesTypeSaisonnier",
                                            "speciesDestination",
                                            "varietyLibelle",
                                            "plantsCertified",
                                            "overGraftDate"
                                    );
                                    String graftSupport = speciesDto.getGraftSupport() !=null ? speciesDto.getGraftSupport().getLabel() : null;
                                    ExportUtils.setExtraField(export, "graftSupport",graftSupport);

                                    String graftClone = speciesDto.getGraftClone() != null ? speciesDto.getGraftClone().getLabel() : null;
                                    ExportUtils.setExtraField(export, "graftClone",graftClone);
                                    perennialEntities.add(export);
                                }
                            }
                        }
                    }

                    // intervention tab
                    if (effectiveSeasonalCropCycleDtos != null) {
                        for (EffectiveSeasonalCropCycleDto seasonalCycle : effectiveSeasonalCropCycleDtos){
                            List<EffectiveCropCycleNodeDto> nodes = seasonalCycle.getNodeDtos();
                            if (nodes != null) {
                                for (EffectiveCropCycleNodeDto node:nodes) {
                                    List<EffectiveInterventionDto> itks = node.getInterventions();
                                    if (itks != null) {
                                        for (EffectiveInterventionDto itk : itks) {
                                            model = exportCommonInterventionFields(node, null, null, itk, model);
                                            // add tools couplings, species, actions, inputs fields
                                            exportToolsCouplingsSpeciesActionsInputsFields(itkEntities, domain, model, itk);
                                        }
                                    }
                                }
                            }
                        }
                    }

                    if (effectivePerennialCropCycleDtos != null) {
                        for (EffectivePerennialCropCycleDto cycle : effectivePerennialCropCycleDtos){
                            List<EffectiveCropCyclePhaseDto> phases = cycle.getPhaseDtos();
                            if (phases != null) {
                                for (EffectiveCropCyclePhaseDto phase : phases){
                                    List<EffectiveInterventionDto> itks = phase.getInterventions();
                                    if (itks != null) {
                                        for (EffectiveInterventionDto itk : itks) {
                                            model = exportCommonInterventionFields(null,phase, cycle, itk, model);
                                            // add tools couplings, species, actions, inputs fields
                                            exportToolsCouplingsSpeciesActionsInputsFields(itkEntities, domain, model, itk);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    protected void exportEffectiveSeasonalCropCycles(List<EffectiveSeasonalCropCycleDto> seasonalCropCycleDtos, Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet, EffectiveCropCycleBeanInfo effectiveCropCycleBeanInfo, EffectiveCropCycleExportEntity model) throws CloneNotSupportedException, IllegalAccessException, NoSuchMethodException, InvocationTargetException {
        for (EffectiveSeasonalCropCycleDto seasonalCropCycleDto : seasonalCropCycleDtos) {

            List<EffectiveCropCycleNodeDto> nodeDtos = seasonalCropCycleDto.getNodeDtos();
            List<EffectiveCropCycleConnectionDto> connectionDtos = seasonalCropCycleDto.getConnectionDtos();

            // create export entity for seasonal crop cycle
            Map<String, EffectiveSeasonalCropCycleExport> seasonalCropCycleExports = Maps.newHashMap();
            for(EffectiveCropCycleNodeDto nodeDto:nodeDtos) {
                EffectiveSeasonalCropCycleExport nodeExport = new EffectiveSeasonalCropCycleExport();
                nodeExport.setCropName(nodeDto.getLabel());
                nodeExport.setRank(nodeDto.getX());

                seasonalCropCycleExports.put(nodeDto.getNodeId(), nodeExport);
            }
            for(EffectiveCropCycleConnectionDto connectionDto:connectionDtos) {
                // set intermediate crop cycle name if necessary
                if (StringUtils.isNotBlank(connectionDto.getIntermediateCroppingPlanEntryName())) {
                    EffectiveSeasonalCropCycleExport nodeExport = seasonalCropCycleExports.get(connectionDto.getTargetId());
                    nodeExport.setIntermediateCropName(connectionDto.getIntermediateCroppingPlanEntryName());
                }
            }

            List<EffectiveCropCycleExportEntity> seasonalCropCycles = (List<EffectiveCropCycleExportEntity>)sheet.get(effectiveCropCycleBeanInfo);
            if (!sheet.containsKey(effectiveCropCycleBeanInfo)) {
                sheet.put(effectiveCropCycleBeanInfo, seasonalCropCycles);
            }

            List<EffectiveSeasonalCropCycleExport> nodeExports = Lists.newArrayList(seasonalCropCycleExports.values());
            if (!CollectionUtils.isEmpty(nodeExports)) {
                for (EffectiveSeasonalCropCycleExport seasonalCropCycleExport : nodeExports) {
                    EffectiveCropCycleExportEntity export = (EffectiveCropCycleExportEntity) model.clone();
                    ExportUtils.copyFields(seasonalCropCycleExport, export, null,
                            "cropName",
                            "intermediateCropName");
                    Integer rank = seasonalCropCycleExport.getRank();
                    ExportUtils.setExtraField(export, "rank", rank == null ? null : rank + 1);
                    seasonalCropCycles.add(export);
                }
            }

        }
    }

    protected void exportToolsCouplingsSpeciesActionsInputsFields(List<EffectiveCropCycleExportEntity> itkEntities, Domain domain, EffectiveCropCycleExportEntity model, EffectiveInterventionDto itk) throws CloneNotSupportedException, InvocationTargetException, IllegalAccessException, NoSuchMethodException {
        //EffectiveCropCycleExportEntity export;
        if (itk.getToolsCouplingCodes() != null && !itk.getToolsCouplingCodes().isEmpty()) {
            Set<String> tcCodes = itk.getToolsCouplingCodes();
            for (String tcCode : tcCodes) {
                ToolsCoupling toolsCoupling = toolsCouplingDao.forProperties(ToolsCoupling.PROPERTY_CODE, tcCode ,ToolsCoupling.PROPERTY_DOMAIN, domain).findAny();
                ExportUtils.setExtraField(model, "toolsCouplings", toolsCoupling.getToolsCouplingName());
                // add species, actions, inputs fields
                exportSpeciesActionsInputsFields(itkEntities, itk, model);
            }

        } else {
            // add species, actions, inputs fields
            exportSpeciesActionsInputsFields(itkEntities, itk, model);

        }
    }

    protected void exportSpeciesActionsInputsFields(List<EffectiveCropCycleExportEntity> itkEntities, EffectiveInterventionDto itk, EffectiveCropCycleExportEntity model) throws CloneNotSupportedException {
        if (itk.getSpeciesStadesDtos() != null && !itk.getSpeciesStadesDtos().isEmpty()) {
            for(SpeciesStadeDto stade : itk.getSpeciesStadesDtos()) {
                // add actions and inputs fields
                Collection<AbstractAction> actions = itk.getActions();
                for (AbstractAction action:actions) {
                    exportSpeciesStadeFields(stade, model);
                    exportActionFields(model, action);

                    // inputs
                    exportInputsFields(itkEntities, itk, model);

                }
            }
        } else {
            // add actions and inputs fields
            Collection<AbstractAction> actions = itk.getActions();
            for (AbstractAction action:actions) {
                exportActionFields(model, action);
                // inputs
                exportInputsFields(itkEntities, itk, model);
            }
        }
    }

    protected void exportSpeciesStadeFields(SpeciesStadeDto speciesStadeDto, EffectiveCropCycleExportEntity model) {
        ExportUtils.setExtraField(model, "species", speciesStadeDto.getSpeciesName());
        String stadeMin = speciesStadeDto.getStadeMin() != null ? speciesStadeDto.getStadeMin().getLabel() : null;
        ExportUtils.setExtraField(model, "stadeMin", stadeMin);
        String stadeMax = speciesStadeDto.getStadeMax() != null ? speciesStadeDto.getStadeMax().getLabel() : null;
        ExportUtils.setExtraField(model, "stadeMax",stadeMax);
    }

    protected void exportActionFields(EffectiveCropCycleExportEntity export, AbstractAction action) {
        ExportUtils.setExtraField(export, "actionType", action.getMainAction().getIntervention_agrosyst());
        ExportUtils.setExtraField(export, "action", action.getMainAction().getReference_label());
        ExportUtils.setExtraField(export, "mainAction", StringUtils.isNotBlank(action.getToolsCouplingCode()));
    }

    protected void exportInputsFields(List<EffectiveCropCycleExportEntity> itkEntities, EffectiveInterventionDto itk, EffectiveCropCycleExportEntity model) throws CloneNotSupportedException {
        List<AbstractInput> inputs = itk.getInputs();
        if (inputs != null && !inputs.isEmpty()) {
            for (AbstractInput input: inputs) {
                EffectiveCropCycleExportEntity export = (EffectiveCropCycleExportEntity) model.clone();
                String unit = "";
                AgrosystInterventionType actionType = input.getInputType();
                switch (actionType) {
                    case APPLICATION_DE_PRODUITS_FERTILISANTS_MINERAUX:
                        MineralProductUnit mpu = ((MineralProductInput)input).getMineralProductUnit();
                        if (mpu != null) {
                            unit = mpu.name();
                        }
                        break;
                    case EPANDAGES_ORGANIQUES:
                        OrganicProductUnit opu = ((OrganicProductInput)input).getOrganicProductUnit();
                        if (opu != null) {
                            unit = opu.name();
                        }
                        break;
                    case APPLICATION_DE_PRODUITS_PHYTOSANITAIRES:
                    case LUTTE_BIOLOGIQUE:
                    case SEMIS:
                        PhytoProductUnit ppu = ((PhytoProductInput)input).getPhytoProductUnit();
                        if (ppu != null) {
                            unit = ppu.name();
                        }
                        break;
                    case AUTRE:
                        unit = ((OtherProductInput)input).getOtherProductQtUnit();
                        break;
                    default:
                        throw new UnsupportedOperationException("Unsupported input type: " + actionType);
                }

                ExportUtils.setExtraField(export, "inputType", input.getInputType());
                ExportUtils.setExtraField(export, "product", input.getProductName());
                ExportUtils.setExtraField(export, "qtMin", input.getQtMin());
                ExportUtils.setExtraField(export, "qtAvg", input.getQtAvg());
                ExportUtils.setExtraField(export, "qtMed", input.getQtMed());
                ExportUtils.setExtraField(export, "qtMax", input.getQtMax());
                ExportUtils.setExtraField(export, "unit", unit);

                itkEntities.add(export);
            }
        } else {
            EffectiveCropCycleExportEntity export = (EffectiveCropCycleExportEntity) model.clone();
            itkEntities.add(export);
        }
    }

    protected EffectiveCropCycleExportEntity exportCommonInterventionFields(EffectiveCropCycleNodeDto node, EffectiveCropCyclePhaseDto phase, EffectivePerennialCropCycleDto perennialCropCycle, EffectiveInterventionDto itk, EffectiveCropCycleExportEntity model) throws CloneNotSupportedException, InvocationTargetException, IllegalAccessException, NoSuchMethodException {
        EffectiveCropCycleExportEntity export;
        export = (EffectiveCropCycleExportEntity) model.clone();
        String cycle = node != null ? "Assolé" : "Pérenne";
        ExportUtils.setExtraField(export, "itk_cycle", cycle);

        String cropField = node != null ?  node.getLabel() : perennialCropCycle.getCroppingPlanEntryName();
        ExportUtils.setExtraField(export, "itk_crop", cropField);

        Integer rank = node != null ? node.getX() : null;
        ExportUtils.setExtraField(export, "itk_rank", rank == null ? null : rank + 1);

        CropCyclePhaseType phaseType = phase != null ? phase.getType() : null;
        ExportUtils.setExtraField(export, "itk_phase", phaseType);

        ExportUtils.copyFields(itk, export, null,
                EffectiveIntervention.PROPERTY_NAME,
                EffectiveIntervention.PROPERTY_TYPE,
                EffectiveIntervention.PROPERTY_START_INTERVENTION_DATE,
                EffectiveIntervention.PROPERTY_END_INTERVENTION_DATE,
                EffectiveIntervention.PROPERTY_TRANSIT_COUNT,
                EffectiveIntervention.PROPERTY_PROGRESSION_SPEED,
                EffectiveIntervention.PROPERTY_INVOLVED_PEOPLE_COUNT,
                EffectiveIntervention.PROPERTY_WORK_RATE,
                EffectiveIntervention.PROPERTY_SPATIAL_FREQUENCY,
                EffectiveIntervention.PROPERTY_INTERMEDIATE_CROP,
                EffectiveIntervention.PROPERTY_COMMENT);
        Double workRate = itk.getWorkRate();
        Double spendingTime = (workRate != null && workRate != 0) ? ((1/workRate) * 1000)/1000 : null;

        Double psci = itk.getTransitCount() * itk.getSpatialFrequency();
        ExportUtils.setExtraField(export, "PSCi", psci);

        ExportUtils.setExtraField(export, "spendingTime", spendingTime);

        return export;
    }

    protected List<EffectivePerennialCropCycle> getAllEffectivePerennialCropCycles(Zone zone) {
        List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forZoneEquals(zone).findAll();
        return cycles;
    }

    protected List<EffectivePerennialCropCycle> getAllEffectivePerennialCropCyclesWithSameCrop(Zone zone, CroppingPlanEntry cp) {
        List<EffectivePerennialCropCycle> cycles = effectivePerennialCropCycleDao.forProperties(EffectivePerennialCropCycle.PROPERTY_ZONE, zone, EffectivePerennialCropCycle.PROPERTY_CROPPING_PLAN_ENTRY, cp).findAll();
        return cycles;
    }

    protected List<EffectiveSeasonalCropCycle> getAllEffectiveSeasonalCropCycles(Zone zone) {
        List<EffectiveSeasonalCropCycle> cycles = effectiveSeasonalCropCycleDao.forZoneEquals(zone).findAll();
        return cycles;
    }

    protected List<EffectiveCropCycleNode> getAllEffectiveCropCycleNodes(Zone zone) {
        List<EffectiveCropCycleNode> result = Lists.newArrayList();

        List<EffectiveSeasonalCropCycle> cycles = getAllEffectiveSeasonalCropCycles(zone);
        if (cycles != null) {
            for (EffectiveSeasonalCropCycle cycle : cycles) {
                Collection<EffectiveCropCycleNode> nodes = EFFECTIVE_SEASONAL_CROP_CYCLE_TO_NODE_FUNCTION.apply(cycle);
                if (nodes != null) {
                    result.addAll(nodes);
                }
            }
        }

        return result;
    }

    protected List<EffectiveSeasonalCropCycle> getAllEffectiveSeasonalCropCycleWithSameCrop(List<EffectiveSeasonalCropCycle> escc, CroppingPlanEntry cp) {
        List<EffectiveSeasonalCropCycle> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(escc)){
            for (EffectiveSeasonalCropCycle effectiveSeasonalCropCycle : escc) {
                Collection<EffectiveCropCycleNode> nodes = effectiveSeasonalCropCycle.getNodes();
                for (EffectiveCropCycleNode node : nodes) {
                    if (node.getCroppingPlanEntry() == cp){
                        result.add(effectiveSeasonalCropCycle);
                        break;
                    }
                }
                break;
            }
        }
        return result;
    }


    @Override
    public CopyPasteZoneByCampaigns getAvailableZonesForCopy(String zoneTopiaId) {
        Zone zoneFrom = zoneDao.forTopiaIdEquals(zoneTopiaId).findUnique();
        Domain domain = zoneFrom.getPlot().getDomain();
        List<Zone> zones = zoneDao.findZonesFromDomainCode(domain.getCode(), getSecurityContext());


        HashMap<Integer, List<CopyPasteZoneDto>> copyPasteZoneDtoByCampaign = new HashMap<Integer, List<CopyPasteZoneDto>>();
        CopyPasteZoneByCampaigns result =  new CopyPasteZoneByCampaigns();
        result.setZonesByCampaigns(copyPasteZoneDtoByCampaign);

        for (Zone zone:zones) {

            List<EffectivePerennialCropCycle> allEffectivePerennialCropCycles = getAllEffectivePerennialCropCycles(zone.getTopiaId());
            List<EffectiveSeasonalCropCycle> allEffectiveSeasonalCropCycles = getAllEffectiveSeasonalCropCycles(zone.getTopiaId());

            List<CopyPasteZoneDto> cropTargets = Lists.newArrayList();
            if (allEffectivePerennialCropCycles != null && !allEffectivePerennialCropCycles.isEmpty()) {
                cropTargets.addAll(Lists.newArrayList(Iterables.transform(allEffectivePerennialCropCycles, EFFECTIVE_PERENNIAL_TO_CROP_CYCLE_CROP_TARGET_FUNCTION)));
            }

            if (allEffectiveSeasonalCropCycles != null && !allEffectiveSeasonalCropCycles.isEmpty()) {
                for (EffectiveSeasonalCropCycle allEffectiveSeasonalCropCycle : allEffectiveSeasonalCropCycles) {
                    Collection<EffectiveCropCycleNode> nodes = allEffectiveSeasonalCropCycle.getNodes();
                    if (nodes != null) {
                        // it must have at least one crop
                        cropTargets.addAll(Lists.newArrayList(Iterables.transform(nodes, EFFECTIVE_CROP_CYCLE_NODE_TO_CROP_TARGET_FUNCTION)));
                    }
                }
            }

            String zoneId = zone.getTopiaId();
            String zoneName = zone.getName();
            double zoneArea = zone.getArea();
            String plotName = zone.getPlot().getName();
            double plotArea = zone.getPlot().getArea();
            String domainName = zone.getPlot().getDomain().getName();
            int domainCampaign = zone.getPlot().getDomain().getCampaign();
            GrowingSystem gs = zone.getPlot().getGrowingSystem();
            String gsName = "";
            String gpName = "";
            if (gs != null) {
                gsName = gs.getName();
                gpName = gs.getGrowingPlan().getName();
            }

            for (CopyPasteZoneDto cropTarget : cropTargets) {
                cropTarget.setZoneId(zoneId);
                cropTarget.setZoneName(zoneName);
                cropTarget.setZoneArea(zoneArea);
                cropTarget.setPlotName(plotName);
                cropTarget.setPlotArea(plotArea);
                cropTarget.setDomainName(domainName);
                cropTarget.setGrowingPlanName(gpName);
                cropTarget.setGrowingSystemName(gsName);
                cropTarget.setCampaign(domainCampaign);

                addCropTocopyPasteZoneDtoByCampaign(copyPasteZoneDtoByCampaign, cropTarget);
            }

        }
        return result;
    }

    protected void addCropTocopyPasteZoneDtoByCampaign(HashMap<Integer, List<CopyPasteZoneDto>> copyPasteZoneDtoByCampaign, CopyPasteZoneDto cropTarget) {
        List<CopyPasteZoneDto> copyPasteZoneDtosForCampaing = copyPasteZoneDtoByCampaign.get(cropTarget.getCampaign());
        if (copyPasteZoneDtosForCampaing == null) {
            copyPasteZoneDtosForCampaing = new ArrayList<CopyPasteZoneDto>();
            copyPasteZoneDtoByCampaign.put(cropTarget.getCampaign(), copyPasteZoneDtosForCampaing);
        }
        copyPasteZoneDtosForCampaing.add(cropTarget);
    }

    protected void copyInterventionsToTargetedNodes(List<EffectiveInterventionDto> interventionDtos, Zone targetedZone, List<String> targetedNodeIds) {
        if (CollectionUtils.isNotEmpty(targetedNodeIds)) {
            // find all nodes from there ids
            List<EffectiveCropCycleNode> targetedNodes = getNodesFromJspIds(targetedNodeIds);

            // two strategies are used to find the targeted crop code.
            // 1st try to find same matching crop code on targeted zone
            // 2nd try to find equivalent crop (same RefEspece) on targeted zone
            List<CroppingPlanSpecies> fromZoneSpecies = getFromCroppingPlanSpecieses(interventionDtos);

            Map<String, String> fromSpeciesKeyToSpeciesCode = referentialService.getCroppingPlanSpeciesCodeByRefEspeceAndVarietyKey(fromZoneSpecies);

            for (EffectiveCropCycleNode targetedNode : targetedNodes) {
                // intervention need to be clear for each iteration as action and input from DTO are same as entity one
                // and have to be recreated for each intervention (and not modified otherwise)
                List<EffectiveInterventionDto> copiedInterventionDtos = toNewInterventionDtos(interventionDtos);

                EffectiveCropCycleConnection targetedConnection = effectiveCropCycleConnectionDao.forTargetEquals(targetedNode).findUniqueOrNull();

                Map<String, String> targetedSpeciesKeyToSpeciesCode = getSpeciesKeyToSpeciesCode(targetedNode, targetedConnection);
                Map<String, String> fromSpeciesCodeToSpeciesCode = getFromSpeciesCodeToSpeciesCode(fromSpeciesKeyToSpeciesCode, targetedSpeciesKeyToSpeciesCode);

                CroppingPlanEntry targetedIntermediateCrop = targetedConnection != null ? targetedConnection.getIntermediateCroppingPlanEntry() : null;
                CroppingPlanEntry targetedCrop = targetedNode.getCroppingPlanEntry();

                validIntermediateStatus(copiedInterventionDtos, targetedIntermediateCrop);

                doSeedingActionsAndInputsMigration(copiedInterventionDtos, targetedCrop, targetedIntermediateCrop, fromSpeciesCodeToSpeciesCode);

                addCroppingPanSpeciesSpeciesStades(copiedInterventionDtos, targetedCrop, targetedIntermediateCrop, fromSpeciesCodeToSpeciesCode);

                Map<String, EffectiveIntervention> idToIntervention = getEffectiveSeasonalInterventionById(targetedNode);

                List<EffectiveCropCycleConnectionDto> connectionDtos = targetedConnection != null ? Lists.newArrayList(EffectiveCropCycles.CROP_CYCLE_CONNECTION_TO_DTO.apply(targetedConnection)) : null;
                InterventionDtoToInterventionUnivers interventionDtoToInterventionUnivers = new InterventionDtoToInterventionUnivers(copiedInterventionDtos, connectionDtos, Pair.of(targetedNode.getTopiaId(), targetedNode), null, null, targetedZone.getPlot().getDomain(), idToIntervention);
                interventionDtosToInterventions(interventionDtoToInterventionUnivers);
            }
        }
    }

    protected void validIntermediateStatus(List<EffectiveInterventionDto> interventionDtos, CroppingPlanEntry targetedIntermediateCrop) {
        for (EffectiveInterventionDto interventionDto : interventionDtos) {
            boolean isIntermediate = interventionDto.isIntermediateCrop() && targetedIntermediateCrop != null;
            interventionDto.setIntermediateCrop(isIntermediate);
        }
    }

    protected Map<String, EffectiveIntervention> getEffectiveSeasonalInterventionById(EffectiveCropCycleNode targetedNode) {
        List<EffectiveIntervention> interventions = Lists.newArrayList();
        interventions.addAll(effectiveInterventionDao.forEffectiveCropCycleNodeEquals(targetedNode).findAll());
        return Maps.newHashMap(Maps.uniqueIndex(interventions, Entities.GET_TOPIA_ID));
    }

    protected List<EffectiveCropCycleNode> getNodesFromJspIds(List<String> targetedNodeIds) {
        List<EffectiveCropCycleNode> nodes = Lists.newArrayList();

        if (CollectionUtils.isNotEmpty(targetedNodeIds)) {
            List<String> nodeIds = Lists.newArrayListWithExpectedSize(targetedNodeIds.size());
            for (String nodeId : targetedNodeIds) {
                if (StringUtils.isNotBlank(nodeId)) {
                    nodeIds.add(Entities.UNESCAPE_TOPIA_ID.apply(nodeId));
                }
            }
            // nodes to copy intervention to.
            List<EffectiveCropCycleNode> zoneNodes = effectiveCropCycleNodeDao.forTopiaIdIn(nodeIds).findAll();
            nodes.addAll(zoneNodes);
        }
        return nodes;
    }

    protected Map<String, String> getSpeciesKeyToSpeciesCode(EffectiveCropCycleNode node, EffectiveCropCycleConnection connection) {
        Map<String, String> targetedSpeciesIdToCode = Maps.newHashMap();

        if (node != null) {
            CroppingPlanEntry croppingPlanEntry = node.getCroppingPlanEntry();
            targetedSpeciesIdToCode = doCroppingPlanEntrySpeciesKeyToSpeciesCode(croppingPlanEntry);
        }

        if (connection != null) {
            CroppingPlanEntry croppingPlanEntry = connection.getIntermediateCroppingPlanEntry();
            targetedSpeciesIdToCode.putAll(doCroppingPlanEntrySpeciesKeyToSpeciesCode(croppingPlanEntry));
        }

        return targetedSpeciesIdToCode;
    }

    protected Map<String, String> doCroppingPlanEntrySpeciesKeyToSpeciesCode(CroppingPlanEntry croppingPlanEntry) {
        Map<String, String> targetedSpeciesKeyToSpeciesCode = Maps.newHashMap();
        if (croppingPlanEntry != null) {
            Collection<CroppingPlanSpecies> croppingPlanSpecies = croppingPlanEntry.getCroppingPlanSpecies();
            targetedSpeciesKeyToSpeciesCode.putAll(referentialService.getCroppingPlanSpeciesCodeByRefEspeceAndVarietyKey(croppingPlanSpecies));
        }
        return targetedSpeciesKeyToSpeciesCode;
    }

    protected void copyInterventionsToTargetedPhases(List<EffectiveInterventionDto> interventionDtos, Zone zone, List<String> targetedPhaseIds) {
        if (CollectionUtils.isNotEmpty(targetedPhaseIds)){
            List<EffectiveCropCyclePhase> phases = Lists.newArrayList();

            phases.addAll(effectiveCropCyclePhaseDao.forTopiaIdIn(targetedPhaseIds).findAll());

            // two strategies are used to find the targeted crop code.
            // 1st try to find same matching crop code on targeted zone
            // 2nd try to find equivalent crop (same RefEspece) on targeted zone
            List<CroppingPlanSpecies> fromZoneSpecies = getFromCroppingPlanSpecieses(interventionDtos);

            Map<String, String> fromSpeciesKeyToSpeciesCode = referentialService.getRefEspeceAndVarietyKeyByCroppingPlanSpeciesCode(fromZoneSpecies);

            for (EffectiveCropCyclePhase phase : phases) {
                EffectivePerennialCropCycle targetedPerennialCropCycle = effectivePerennialCropCycleDao.forPhaseEquals(phase).findUnique();
                CroppingPlanEntry toZoneCrop = targetedPerennialCropCycle.getCroppingPlanEntry();

                // intervention need to be clear for each iteration as action and input from DTO are same as entity one
                // and have to be recreated for each intervention (and not modified otherwise)
                List<EffectiveInterventionDto> copiedInterventionDtos = toNewInterventionDtos(interventionDtos);

                Map<String, String> fromSpeciesCodeToSpeciesCode = getFromSpeciesCodesToSpeciesCodes(fromSpeciesKeyToSpeciesCode, toZoneCrop);

                doSeedingActionsAndInputsMigration(copiedInterventionDtos, toZoneCrop, null, fromSpeciesCodeToSpeciesCode);

                addCroppingPanSpeciesSpeciesStades(copiedInterventionDtos, toZoneCrop, null, fromSpeciesCodeToSpeciesCode);

                Map<String, EffectiveIntervention> idToIntervention = getPerennialEffectiveInterventionsByIds(phase);

                interventionDtosToInterventions(new InterventionDtoToInterventionUnivers(copiedInterventionDtos, null, null, targetedPerennialCropCycle, phase, zone.getPlot().getDomain(), idToIntervention));
            }
        }
    }

    private Map<String, String> getFromSpeciesCodesToSpeciesCodes(Map<String, String> fromSpeciesKeyToSpeciesCode, CroppingPlanEntry toZoneCrop) {
        Collection<CroppingPlanSpecies> toZoneSpecies = toZoneCrop.getCroppingPlanSpecies();
        Map<String, String> targetedSpeciesKeyToSpeciesCode = referentialService.getCroppingPlanSpeciesCodeByRefEspeceAndVarietyKey(toZoneSpecies);
        return getFromSpeciesCodeToSpeciesCode(fromSpeciesKeyToSpeciesCode, targetedSpeciesKeyToSpeciesCode);
    }

    private Map<String, EffectiveIntervention> getPerennialEffectiveInterventionsByIds(EffectiveCropCyclePhase phase) {
        List<EffectiveIntervention> interventions = Lists.newArrayList();
        interventions.addAll(effectiveInterventionDao.forEffectiveCropCyclePhaseEquals(phase).findAll());
        return Maps.newHashMap(Maps.uniqueIndex(interventions, Entities.GET_TOPIA_ID));
    }

    protected Map<String, String> getFromSpeciesCodeToSpeciesCode(Map<String, String> fromSpeciesKeyToSpeciesCode, Map<String, String> targetedSpeciesKeyToSpeciesCode) {
        Map<String, String> fromSpeciesCodeToSpeciesCode = Maps.newHashMap();
        for (Map.Entry<String, String> targetedSpeciesKeyToCode : targetedSpeciesKeyToSpeciesCode.entrySet()) {
            String fromSpeciesCode = fromSpeciesKeyToSpeciesCode.get(targetedSpeciesKeyToCode.getKey());
            if (StringUtils.isNotBlank(fromSpeciesCode)) {
                fromSpeciesCodeToSpeciesCode.put(fromSpeciesCode, targetedSpeciesKeyToCode.getValue());
            }
        }
        return fromSpeciesCodeToSpeciesCode;
    }

    protected List<CroppingPlanSpecies> getFromCroppingPlanSpecieses(List<EffectiveInterventionDto> interventionDtos) {
        String fromDomainId = interventionDtos.get(0).getDomainId();
        Set<String> speciesCodes = Sets.newHashSet();
        for (EffectiveInterventionDto interventionDto : interventionDtos) {
            for (SpeciesStadeDto speciesStadeDto : interventionDto.getSpeciesStadesDtos()) {
                speciesCodes.add(speciesStadeDto.getSpeciesCode());
            }
        }
        List<CroppingPlanSpecies> interventionSpecies = croppingPlanSpeciesDao.getCroppingPlanSpeciesForCodeAndDomainId(speciesCodes, fromDomainId);
        return interventionSpecies;
    }

    @Override
    public void copyInterventions(List<TargetedZones> zonesDto, List<EffectiveInterventionDto> interventionDtos) {
        if (CollectionUtils.isNotEmpty(interventionDtos) && CollectionUtils.isNotEmpty(zonesDto)) {

            Iterable<String> zoneIds = getZoneIdsFromZoneDtos(zonesDto);

            List<Zone> zones = zoneDao.forTopiaIdIn(Sets.newHashSet(zoneIds)).findAll();

            Map<String, Zone> indexedZones = Maps.uniqueIndex(zones, Entities.GET_TOPIA_ID);

            for (TargetedZones targetedZoneDto : zonesDto) {
                Zone targetedZone = indexedZones.get(targetedZoneDto.getZoneId());

                if (targetedZone != null) {
                    copyInterventionsToTargetedNodes(interventionDtos, targetedZone, targetedZoneDto.getNodes());
                    copyInterventionsToTargetedPhases(interventionDtos, targetedZone, targetedZoneDto.getPhases());
                }
            }
            getTransaction().commit();
        }
    }

    protected Iterable<String> getZoneIdsFromZoneDtos(List<TargetedZones> zonesDto) {
        return Iterables.transform(zonesDto, new Function<TargetedZones, String>() {
            @Override
            public String apply(TargetedZones input) {
                return input.getZoneId();
            }
        });
    }


    public static final Function<Pz0EffectiveSeasonalCropCycle, EffectiveSeasonalCropCycleDto> EFFECTIVE_PZ0_SEASONAL_CROP_CYCLE_TO_EFFECTIVE_PZ0_SEASONAL_CROP_CYCLE = new Function<Pz0EffectiveSeasonalCropCycle, EffectiveSeasonalCropCycleDto>() {
        @Override
        public EffectiveSeasonalCropCycleDto apply(Pz0EffectiveSeasonalCropCycle cycle) {
            EffectiveSeasonalCropCycleDto result = cycle.getSeasonalCropCycle();

            return result;
        }
    };

    @Override
    public void importPZ0EffectiveCropCycles(Collection<EffectiveCropCycleAndDependencies> effectiveCropCycleAndDependencies) {
        try {
            int count = 1;
            int total = effectiveCropCycleAndDependencies.size();
            for (EffectiveCropCycleAndDependencies entityAndDependencies : effectiveCropCycleAndDependencies) {
                String zoneId = entityAndDependencies.getZoneId();

                long start = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info(String.format("Début sauvegarde du cycle de cultures réalisé '%s' - %d/%d.", zoneId, count++, total));
                }

                List<EffectiveSeasonalCropCycleDto> seasonalCropCycles = null;
                if (entityAndDependencies.getEffectivePz0SeasonalCropCyclesByCsvId() != null) {
                    Collection<Pz0EffectiveSeasonalCropCycle> seasonalCropCycleDtos = entityAndDependencies.getEffectivePz0SeasonalCropCyclesByCsvId().values();
                    seasonalCropCycles = Lists.newArrayList(Iterables.transform(seasonalCropCycleDtos, EFFECTIVE_PZ0_SEASONAL_CROP_CYCLE_TO_EFFECTIVE_PZ0_SEASONAL_CROP_CYCLE));
                }

                List<EffectivePerennialCropCycleDto> effectivePerennialCropCycleDtos = null;
                if (entityAndDependencies.getEffectivePz0PerennialCropCyclesByCsvId() != null) {
                    Map<String, Pz0EffectivePerennialCropCycle> effectivePerennialCropCyclesByCsvId = entityAndDependencies.getEffectivePz0PerennialCropCyclesByCsvId();
                    Collection<Pz0EffectivePerennialCropCycle> pz0EffectivePerennialCropCycles = effectivePerennialCropCyclesByCsvId.values();
                    List<EffectivePerennialCropCycleDto> perennialCropCycleDtos = Lists.newArrayListWithCapacity(pz0EffectivePerennialCropCycles.size());
                    for (Pz0EffectivePerennialCropCycle pz0EffectivePerennialCropCycle : pz0EffectivePerennialCropCycles) {
                        perennialCropCycleDtos.add(pz0EffectivePerennialCropCycle.getPerennialCropCycleDto());
                    }
                    effectivePerennialCropCycleDtos = Lists.newArrayList(perennialCropCycleDtos);
                }

                List<Price> prices = null;
                if (!entityAndDependencies.getPrices().isEmpty()) {
                    prices = Lists.newArrayList(entityAndDependencies.getPrices().values());
                }

                updateEffectiveCropCyclesWithoutCommit(zoneId, seasonalCropCycles, effectivePerennialCropCycleDtos, prices);
                long p1 = System.currentTimeMillis();
                if (log.isInfoEnabled()) {
                    log.info("Fin de sauvegarde du cycle de cultures réalisé en:" + (p1- start));
                }
            }
        } catch (Exception e) {
            throw (new AgrosystImportException("Echec de persistance des cycles de culture du réalisé", e));
        }
    }

    /**
     * only add the seeding action species that match the targeted species
     */
    protected void doSeedingActionsAndInputsMigration(
            List<EffectiveInterventionDto> interventionDtos,
            CroppingPlanEntry targetedCrop,
            CroppingPlanEntry targetedIntermediateCrop,
            Map<String, String> fromSpeciesCodeToSpeciesCode) {

        for (EffectiveInterventionDto interventionDto : interventionDtos) {
            Collection<CroppingPlanSpecies> targetedSpecieses = interventionDto.isIntermediateCrop() ? targetedIntermediateCrop.getCroppingPlanSpecies():  targetedCrop.getCroppingPlanSpecies();
            Map<String, CroppingPlanSpecies> targetedSpeciesByCodes = Maps.newHashMap();
            for (CroppingPlanSpecies targetedSpeciese : targetedSpecieses) {
                targetedSpeciesByCodes.put(GET_CROPPING_PLAN_SPECIES_CODE.apply(targetedSpeciese), targetedSpeciese);
            }

            actionService.migrateActionsSpeciesToTargetedSpecies(interventionDto.getActions(), targetedSpeciesByCodes, fromSpeciesCodeToSpeciesCode);
            inputService.migrateInputSpeciesToTargetedSpecies(interventionDto);
        }
    }


    protected void addCroppingPanSpeciesSpeciesStades(List<EffectiveInterventionDto> interventionDtos,
                                                      CroppingPlanEntry targetedNodeOrPhaseCrop,
                                                      CroppingPlanEntry targetedIntermediateCrop,
                                                      Map<String, String> fromSpeciesCodeToSpeciesCode) {
        for (EffectiveInterventionDto interventionDto : interventionDtos) {

            Collection<CroppingPlanSpecies> targetedSpecieses = interventionDto.isIntermediateCrop() ? targetedIntermediateCrop.getCroppingPlanSpecies() : targetedNodeOrPhaseCrop.getCroppingPlanSpecies();
            Map<String, CroppingPlanSpecies> targetedSpeciesByCodes = Maps.uniqueIndex(targetedSpecieses, GET_CROPPING_PLAN_SPECIES_CODE);

            setSpeciesStadesBySpeciesCode(targetedSpeciesByCodes, fromSpeciesCodeToSpeciesCode, interventionDto);
        }
    }

    protected void setSpeciesStadesBySpeciesCode(
            Map<String, CroppingPlanSpecies> targetedSpeciesByCodes,
            Map<String, String> fromSpeciesCodeToSpeciesCode,
            EffectiveInterventionDto interventionDto) {

        List<SpeciesStadeDto> fromSpeciesStadeDtos = interventionDto.getSpeciesStadesDtos();
        if (fromSpeciesStadeDtos == null) {
            fromSpeciesStadeDtos = Lists.newArrayList();
        }
        List<SpeciesStadeDto> validSpeciesStadeDtos = Lists.newArrayList();
        for (SpeciesStadeDto fromSpeciesStadeDto : fromSpeciesStadeDtos) {
            String speciesCode = targetedSpeciesByCodes.get(fromSpeciesStadeDto.getSpeciesCode()) != null ? fromSpeciesStadeDto.getSpeciesCode() : fromSpeciesCodeToSpeciesCode.get(fromSpeciesStadeDto.getSpeciesCode());
            if (StringUtils.isNotBlank(speciesCode)) {
                fromSpeciesStadeDto.setSpeciesCode(speciesCode);
                validSpeciesStadeDtos.add(fromSpeciesStadeDto);
            }
        }
        // set as null as default behaviours is to create new intervention species stades if none.
        validSpeciesStadeDtos = validSpeciesStadeDtos.isEmpty() ? null : validSpeciesStadeDtos;
        interventionDto.setSpeciesStadesDtos(validSpeciesStadeDtos);
    }
}
