package fr.inra.agrosyst.services.edaplos;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EdaplosServiceImpl.java 4906 2015-04-24 09:00:23Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/edaplos/EdaplosServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.zip.ZipInputStream;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.BufferStrip;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.MaxSlope;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.WaterFlowDistance;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.ZoneType;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNode;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.edaplos.EdaplosDomainDto;
import fr.inra.agrosyst.api.services.edaplos.EdaplosParsingResult;
import fr.inra.agrosyst.api.services.edaplos.EdaplosParsingStatus;
import fr.inra.agrosyst.api.services.edaplos.EdaplosPlotDto;
import fr.inra.agrosyst.api.services.edaplos.EdaplosService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleNodeDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.referential.ImportResult;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * Services to upload eDaplos
 * 
 * @author eancelet@orleans.inra.fr
 * 
 * TODO 2015-01-27 eancelet : replace the use of domainDAO by domainService ? And use plotService instead of plotDAO ?
 * TODO 2015-03-17 eancelet : gérer si il y a des erreurs remontées par edaplosPlotDto.soilOccErrorMessages
 * 
 */
public class EdaplosServiceImpl extends AbstractAgrosystService implements EdaplosService {

    private static final Log log = LogFactory.getLog(EdaplosServiceImpl.class);

    public static final int BUFFER_SIZE = 1024;

    
    protected DomainTopiaDao domainDao;
    protected PlotTopiaDao plotDao;
    protected ZoneTopiaDao zoneDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao;
    protected DomainService domainService;
    protected PlotService plotService;
    protected EffectiveCropCycleService effectiveCropCycleService;
    
    
    
    public DomainTopiaDao getDomainDao() {
        return domainDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }


    public PlotTopiaDao getPlotDao() {
        return plotDao;
    }

    public void setPlotDao(PlotTopiaDao plotDao) {
        this.plotDao = plotDao;
    }
    
    public ZoneTopiaDao getZoneDao() {
        return zoneDao;
    }

    public void setZoneDao(ZoneTopiaDao zoneDao) {
        this.zoneDao = zoneDao;
    }   
    
    public CroppingPlanEntryTopiaDao getCroppingPlanEntryTopiaDao() {
        return croppingPlanEntryTopiaDao;
    }

    public void setCroppingPlanEntryTopiaDao(
            CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao) {
        this.croppingPlanEntryTopiaDao = croppingPlanEntryTopiaDao;
    }

    public DomainService getDomainService() {
        return domainService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public PlotService getPlotService() {
        return plotService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    
    
    public EffectiveCropCycleService getEffectiveCropCycleService() {
        return effectiveCropCycleService;
    }

    public void setEffectiveCropCycleService(EffectiveCropCycleService effectiveCropCycleService) {
        this.effectiveCropCycleService = effectiveCropCycleService;
    }

    /**
     * return the result of Edaplos parsing
     * @param inputStream the Edaplos file
     * @return the result of Edaplos parsing
     */
    @Override
    public List<EdaplosParsingResult> validEdaplosData(InputStream inputStream) {

        List<EdaplosParsingResult> results = Lists.newArrayList();
        List<InputStream> streams = extractZipOrRegularFile(inputStream);

        try {
            // loop among all files in zip
            for (InputStream stream : streams) {
                SAXReader reader = new SAXReader();
                // don't use reader.read(String);
                // don't work on windows because of : in path
                // Document doc = reader.read(xmlFile);
                reader.setEncoding("utf-8");
                Document doc = reader.read(stream);

                EdaplosDocumentParser parser = newInstance(EdaplosDocumentParser.class);
                EdaplosParsingResult result = parser.parse(doc);
                results.add(result);
                IOUtils.closeQuietly(stream);
            }
        } catch (DocumentException ex) {
            throw new AgrosystTechnicalException("Can't parse xml file", ex);
        } finally {
            IOUtils.closeQuietly(inputStream);
        }

        return results;
    }

    protected List<InputStream> extractZipOrRegularFile(InputStream inputStream) {

        ZipInputStream zipInputStream = null;
        // use to be sure to not load old streams.

        List<InputStream> inputStreams = Lists.newArrayList();

        try {
            zipInputStream = new ZipInputStream(inputStream);

            while ((zipInputStream.getNextEntry()) != null) {

                ByteArrayOutputStream out = null;
                try {
                    byte[] buffer = new byte[BUFFER_SIZE];
                    out = new ByteArrayOutputStream();

                    int n;
                    while ((n = zipInputStream.read(buffer, 0, BUFFER_SIZE)) > -1) {
                        out.write(buffer, 0, n);
                    }
                    InputStream is = new ByteArrayInputStream(out.toByteArray());
                    inputStreams.add(is);

                    zipInputStream.closeEntry();
                } finally {
                    IOUtils.closeQuietly(out);
                }
            }

        } catch (Exception e) {
            throw new AgrosystTechnicalException("Can't parse xml file", e);
        } finally {
            IOUtils.closeQuietly(zipInputStream);
        }
        return inputStreams;
    }


    /**
     * Import parse data
     * @param edaplosParsingResults parsing data
     * @return import result as class name, result
     */
    
    @Override
    public Map<Class, ImportResult> importEdaplos(List<EdaplosParsingResult> edaplosParsingResults) {

        Map<Class, ImportResult> result = Maps.newHashMap();

        if (edaplosParsingResults != null){
            // each eDaplos message return one edaplosParsingResult
            for (EdaplosParsingResult edaplosParsingResult : edaplosParsingResults) {
                // import Datas only if general status = SUCCESS
                if (edaplosParsingResult.getEdaplosParsingStatus() == EdaplosParsingStatus.SUCCESS){
                    ImportResult domainsImportResult = result.get(Domain.class);
                    if (domainsImportResult == null) {
                        domainsImportResult = new ImportResult();
                        result.put(Domain.class, domainsImportResult);
                    }

                    List<EdaplosDomainDto> eDomains = edaplosParsingResult.getDomains();
                    for (EdaplosDomainDto eDomain : eDomains) {
                        Domain domain;
                        
                        if (StringUtils.isNotBlank(eDomain.getTopiaId())){ // si le domaine a un topiaId alors il est déjà persisté
                            domain = domainDao.forTopiaIdEquals(eDomain.getTopiaId()).findUnique();
                        } else {
                            domain = domainService.newDomain();
                            domain.setCampaign(Integer.parseInt(eDomain.getCampaign()));
                            domain.setCode(UUID.randomUUID().toString());
                            // TODO ajouter un test pour remonter une erreur s'il n'y a pas de SIRET
                            domain.setSiret(eDomain.getSiret());
                        }
                        
                        domain.setName(eDomain.getName());
                        domain.setLocation(eDomain.getLocation());
                        domain.setMainContact(eDomain.getMainContact());
                        // eancelet le 03/12/2014 : tous les domaines créés sont de type Exploitation Agricole
                        DomainType domainType = DomainType.EXPLOITATION_AGRICOLE;
                        domain.setType(domainType);                      

                        
                        
                        domain.setUpdateDate(new Date());

                        domain = domainService.createOrUpdateDomain(domain, eDomain.getLocation().getTopiaId(), null, null, eDomain.getCroppingPlanEntryDtos(), null, null, null, null, null, null);
                        /*
                        if (domain.isPersisted()) {
                            domain = domainDao.update(domain);
                            domainsImportResult.incUpdated();
                        } else {
                            domain = domainDao.create(domain);
                            domainsImportResult.incCreated();
                        }*/

                        if (eDomain.getPlots() != null && !eDomain.getPlots().isEmpty()) {
                            ImportResult plotImportResult = result.get(Plot.class);
                            if (plotImportResult == null) {
                                plotImportResult = new ImportResult();
                                result.put(Plot.class, plotImportResult);
                            }

                            for (EdaplosPlotDto ePlot : eDomain.getPlots()) {
                                Plot plot;
                                List<Zone> zones = new ArrayList<Zone>();                                
                                if (StringUtils.isNotBlank(ePlot.getTopiaId())){
                                    plot = plotDao.forTopiaIdEquals(ePlot.getTopiaId()).findUnique();
                                } else {
                                    plot = plotDao.newInstance();
                                    
                                    plot.setName(ePlot.getPlotName());
                                    plot.setActive(true);                                    
                                 
                                    // required info are unknown, set some default values for plot.
                                    plot.setWaterFlowDistance(WaterFlowDistance.LESS_THAN_THREE);
                                    plot.setBufferStrip(BufferStrip.NONE);
                                    plot.setMaxSlope(MaxSlope.ZERO);
                                    
                                    // add default zone                                    
                                    Zone defaultZone = zoneDao.newInstance();
                                    // TODO eancelet 03/12/2014 passer le nom en paramètre comme dans PlotServiceImpl
                                    defaultZone.setName("Zone principale");
                                    defaultZone.setType(ZoneType.PRINCIPALE);
                                    defaultZone.setActive(true);
                                    defaultZone.setCode(UUID.randomUUID().toString());
                                    zones.add(defaultZone);
                                        
                                    
                                    
                                }
                                plot.seteDaplosIssuerId(ePlot.getIssuerId());
                                if(!Strings.isNullOrEmpty(ePlot.getArea())){
                                    plot.setArea(Double.parseDouble(ePlot.getArea()));
                                }

                                if (plot.isPersisted()) {
                                    plotDao.update(plot);
                                    plotImportResult.incUpdated();
                                } else {
                                    // set to plot the same location as domain
                                    plot = plotService.createOrUpdatePlot(plot, domain.getTopiaId(), domain.getLocation().getTopiaId(), null, null, null, null, null, null, null, zones, null);
                                    plotImportResult.incCreated();
                                }
                                
                                
                                // Import plotSoilOccupations
                                if (ePlot.getSeasonalCycleDto() != null && ! CollectionUtils.isEmpty(ePlot.getSeasonalCycleDto().getNodeDtos())) {
                                    // TODO 2015-01-28 eancelet : add ImportResult stuff 
                                    
                                    String myZoneId = plotService.getPlotZones(plot).get(0).getTopiaId();
                                    
                                        
                                    // if a node is linked to a new entry we need to switch the node.croppingPlanEntryId from "new-entry-" to the persisted one
                                    for (EffectiveCropCycleNodeDto cropCycleNode : ePlot.getSeasonalCycleDto().getNodeDtos()) {
                                        String entryId = cropCycleNode.getCroppingPlanEntryId();                                                
                                        if (entryId.startsWith("new-entry-")) {
                                            List<CroppingPlanEntryDto> entriesFromEDomain = eDomain.getCroppingPlanEntryDtos();
                                            for (CroppingPlanEntryDto entryFromEDomain : entriesFromEDomain) {                                     
                                                if (entryId.equals(entryFromEDomain.getTopiaId())) {
                                                    String entryCode = entryFromEDomain.getCode();
                                                    String realEntryTopiaId = croppingPlanEntryTopiaDao.forProperties(CroppingPlanEntry.PROPERTY_CODE, entryCode, CroppingPlanEntry.PROPERTY_DOMAIN, domain).findUnique().getTopiaId();
                                                    cropCycleNode.setCroppingPlanEntryId(realEntryTopiaId);
                                                }
                                            }
                                        }  
                                    }
                                    
                                    // same with connection
                                    for (EffectiveCropCycleConnectionDto cropCycleConn : ePlot.getSeasonalCycleDto().getConnectionDtos()) {
                                        String entryId = cropCycleConn.getIntermediateCroppingPlanEntryId();
                                        if (entryId.startsWith("new-entry")){
                                            List<CroppingPlanEntryDto> entriesFromDomain = eDomain.getCroppingPlanEntryDtos();
                                            for (CroppingPlanEntryDto entryFromEDomain : entriesFromDomain) {
                                                if (entryId.equals(entryFromEDomain.getTopiaId())){
                                                    String entryCode = entryFromEDomain.getCode();
                                                    String realEntryTopiaId = croppingPlanEntryTopiaDao.forProperties(CroppingPlanEntry.PROPERTY_CODE, entryCode, CroppingPlanEntry.PROPERTY_DOMAIN, domain).findUnique().getTopiaId();
                                                    cropCycleConn.setIntermediateCroppingPlanEntryId(realEntryTopiaId);
                                                }
                                            }
                                        }
                                    }
                                    
                                    // TODO 2015-03-18 eancelet : same with connection ? 
                                    for (EffectivePerennialCropCycleDto perennialCycleDto : ePlot.getPerennialCycleDtos()) {
                                        String entryId = perennialCycleDto.getCroppingPlanEntryId();
                                        if (entryId.startsWith("new-entry-")) {
                                            List<CroppingPlanEntryDto> entriesFromEDomain = eDomain.getCroppingPlanEntryDtos();
                                            for (CroppingPlanEntryDto entryFromEDomain : entriesFromEDomain) {                                     
                                                if (entryId.equals(entryFromEDomain.getTopiaId())) {
                                                    String entryCode = entryFromEDomain.getCode();
                                                    String realEntryTopiaId = croppingPlanEntryTopiaDao.forProperties(CroppingPlanEntry.PROPERTY_CODE, entryCode, CroppingPlanEntry.PROPERTY_DOMAIN, domain).findUnique().getTopiaId();
                                                    perennialCycleDto.setCroppingPlanEntryId(realEntryTopiaId);
                                                }
                                            }
                                        }
                                    }
                                    
                                    List<EffectiveSeasonalCropCycleDto> effSeasonalCycleDtos = new ArrayList<EffectiveSeasonalCropCycleDto>();
                                    effSeasonalCycleDtos.add(ePlot.getSeasonalCycleDto());
                                                                        
                                    // TODO 2015-03-18 : handle prices
                                    effectiveCropCycleService.updateEffectiveCropCycles(myZoneId, effSeasonalCycleDtos, ePlot.getPerennialCycleDtos(), null);
                                    
                                    
                                    
                                    
                                }
                                
                                
                                
                            }
                        }
                        getTransaction().commit();
                    }
                }
            }
        }


        return result;
    }
    
    
    
    /*
    @Override
    public Map<Class, ImportResult> importEdaplos(List<EdaplosParsingResult> edaplosParsingResults) {

        Map<Class, ImportResult> result = Maps.newHashMap();

        if (edaplosParsingResults != null){
            // each eDaplos message return one edaplosParsingResult
            for (EdaplosParsingResult edaplosParsingResult : edaplosParsingResults) {
                // import Datas only if general status = SUCCESS
                if (edaplosParsingResult.getEdaplosParsingStatus() == EdaplosParsingStatus.SUCCESS){
                    ImportResult domainsImportResult = result.get(Domain.class);
                    if (domainsImportResult == null) {
                        domainsImportResult = new ImportResult();
                        result.put(Domain.class, domainsImportResult);
                    }

                    List<EdaplosDomainDto> eDomains = edaplosParsingResult.getDomains();
                    for (EdaplosDomainDto eDomain : eDomains) {
                        Domain domain;
                        
                        
                        if (StringUtils.isNotBlank(eDomain.getTopiaId())){ // si le domaine a un topiaId alors il est déjà persisté
                            domain = domainDao.forTopiaIdEquals(eDomain.getTopiaId()).findUnique();
                        } else { // sinon on le crée
                            domain = domainDao.newInstance();
                            domain.setName(eDomain.getName());
                            domain.setCampaign(Integer.parseInt(eDomain.getCampaign()));
                            domain.setCode(UUID.randomUUID().toString());
                            domain.setActive(true);
                            domain.setLocation(eDomain.getLocation());
                            domain.setMainContact(eDomain.getMainContact());
                            // eancelet le 03/12/2014 : tous les domaines créés sont de type Exploitation Agricole
                            DomainType domainType = DomainType.EXPLOITATION_AGRICOLE;
                            domain.setType(domainType);
                            

                            
                        }
                        
                        List<CroppingPlanEntryDto> eCroppingPlanEntries = eDomain.getNewCroppingPlanEntryDtos();
                        if (eCroppingPlanEntries != null && !eCroppingPlanEntries.isEmpty()) {
                            for (CroppingPlanEntryDto eCroppingPlanEntry : eCroppingPlanEntries) {
                                //croppingPlanEntry = 
                            }
                        }
                        
                        if (StringUtils.isNotBlank(eDomain.getSiret())) {
                            domain.setSiret(eDomain.getSiret());
                        }
                        // TODO ajouter un test pour remonter une erreur s'il n'y a pas de SIRET
                        
                        domain.setUpdateDate(new Date());

                        if (domain.isPersisted()) {
                            domain = domainDao.update(domain);
                            domainsImportResult.incUpdated();
                        } else {
                            domain = domainDao.create(domain);
                            domainsImportResult.incCreated();
                        }

                        if (eDomain.getPlots() != null && !eDomain.getPlots().isEmpty()) {
                            ImportResult plotImportResult = result.get(Plot.class);
                            if (plotImportResult == null) {
                                plotImportResult = new ImportResult();
                                result.put(Plot.class, plotImportResult);
                            }

                            for (EdaplosPlotDto ePlot : eDomain.getPlots()) {
                                Plot plot;
                                List<Zone> zones = new ArrayList<Zone>();                                
                                if (StringUtils.isNotBlank(ePlot.getTopiaId())){
                                    plot = plotDao.forTopiaIdEquals(ePlot.getTopiaId()).findUnique();
                                } else {
                                    plot = plotDao.newInstance();
                                    
                                    plot.setName(ePlot.getPlotName());
                                    plot.setActive(true);                                    
                                 
                                    // required info are unknown, set some default values for plot.
                                    plot.setWaterFlowDistance(WaterFlowDistance.LESS_THAN_THREE);
                                    plot.setBufferStrip(BufferStrip.NONE);
                                    plot.setMaxSlope(MaxSlope.ZERO);
                                    
                                    // add default zone                                    
                                    Zone defaultZone = zoneDao.newInstance();
                                    // TODO eancelet 03/12/2014 passer le nom en paramètre comme dans PlotServiceImpl
                                    defaultZone.setName("Zone principale");
                                    defaultZone.setType(ZoneType.PRINCIPALE);
                                    defaultZone.setActive(true);
                                    defaultZone.setCode(UUID.randomUUID().toString());
                                    zones.add(defaultZone);
                                        
                                    
                                    
                                }
                                plot.seteDaplosIssuerId(ePlot.getIssuerId());
                                if(!Strings.isNullOrEmpty(ePlot.getArea())){
                                    plot.setArea(Double.parseDouble(ePlot.getArea()));
                                }

                                if (plot.isPersisted()) {
                                    plotDao.update(plot);
                                    plotImportResult.incUpdated();
                                } else {
                                    // set to plot the same location as domain
                                    plot = plotService.createOrUpdatePlot(plot, domain.getTopiaId(), domain.getLocation().getTopiaId(), null, null, null, null, null, null, null, zones, null);
                                    plotImportResult.incCreated();
                                }
                                
                                
                                // Import plotSoilOccupations
                                if (ePlot.getSoilOcuppations() != null && !ePlot.getSoilOcuppations().isEmpty()) {
                                    // TODO 2015-01-28 eancelet : add ImportResult stuff 
                                    
                                    for (EdaplosPlotSoilOccupationDto ePlotSoilOccupation : ePlot.getSoilOcuppations()) {
                                        String myZone = plotService.getPlotZones(plot).get(0).getTopiaId();
                                        effectiveCropCycleService.updateEffectiveCropCycles(myZone, ePlotSoilOccupation.getEffectiveSeasonalCropCycleDtos(), null, null);
                                        
                                    }
                                    
                                    
                                }
                                
                                
                                
                            }
                        }
                        getTransaction().commit();
                    }
                }
            }
        }


        return result;
    }
    */
}
