package fr.inra.agrosyst.services.edaplos;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EdaplosDocumentParser.java 4906 2015-04-24 09:00:23Z eancelet $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/edaplos/EdaplosDocumentParser.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.Element;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.CropCyclePhaseType;
import fr.inra.agrosyst.api.entities.CroppingEntryType;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.WeedType;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleConnectionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleNodeTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycleTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefCultureEdiGroupeCouvSol;
import fr.inra.agrosyst.api.entities.referential.RefCultureEdiGroupeCouvSolTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVarieteGeves;
import fr.inra.agrosyst.api.entities.referential.RefVarieteGevesTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVarietePlantGrapeTopiaDao;
import fr.inra.agrosyst.api.entities.referential.TypeCulture;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.edaplos.EdaplosDomainDto;
import fr.inra.agrosyst.api.services.edaplos.EdaplosDomainImportStatus;
import fr.inra.agrosyst.api.services.edaplos.EdaplosParsingResult;
import fr.inra.agrosyst.api.services.edaplos.EdaplosParsingStatus;
import fr.inra.agrosyst.api.services.edaplos.EdaplosPlotDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleConnectionDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleNodeDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCyclePhaseDto;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.effective.EffectiveInterventionDto;
import fr.inra.agrosyst.api.services.effective.EffectivePerennialCropCycleDto;
import fr.inra.agrosyst.api.services.effective.EffectiveSeasonalCropCycleDto;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.common.CommonService;

/**
 * Parser dom d'un document eDaplos.
 * 
 * @author Eric Chatellier, eancelet
 * 
 * TODO : assurer la filiation des domaines et parcelles (avec ce qui existe déjà en base et ce qui est importé lors du même import)
 * TODO : assurer la filiation des cultures ?
 * TODO : tester que des infos ne sont pas rentrées dans d'autres fichiers du même zip comme pour les PZ0
 * TODO : empêcher la suite des évènement si la parcelle trouvée a plus d'une zone 
 * TODO : persister le eDaplosIssuerId si la parcelle est retrouvée avec le nom + l'air 
 * TODO : ajouter un responsable domaine / responsable dispositif ?
 * TODO : modifier le comportement du découpage de parcelle : ne pas copier le eDaplosIssuerId
 * TODO : ajouter id du domaine + campagne dans tous les messages d'erreur (sera utile lors de l'import de plusieurs fichiers en même temps)
 * TODO : vérifier les droits sur les SdC
 * 
 */
public class EdaplosDocumentParser {

    private static final Log log = LogFactory.getLog(EdaplosDocumentParser.class);

    // error messages
    protected static String ERROR_SIRET = "Le SIRET du domaine est une information obligatoire hors il est manquant dans le document.";
    protected static String ERROR_INVALID_CAMPAIGN = "Le format de la campagne (%s) pour le domaine ''%s'' n'est pas valide";// campagne, numéro siret
    protected static String ERROR_INVALID_CAMPAIGN_FOR_PLOT = "La campagne pour la parcelle ''%s'' (%s, domaine %s) n'est pas valide";// GUID parcelle, nom parcelle, siret
    protected static String ERROR_INVALID_IDENTIFIER_FOR_PLOT = "L'identifiant pour la parcelle ''%s'' (domaine %s) n'est pas valide";// nom parcelle, siret
    protected static String ERROR_NO_CAMPAIGN_FOR_DOMAIN = "Aucune campagne n'est définie pour le domaine ''%s'', de siret ''%s''";
    protected static String DOMAIN_CREATE="Le domaine ayant le numéro de siret :%s, sera créé pour la campagne : %s";
    protected static String DOMAIN_FORBIDDEN="Vous n'avez pas les droits d'écriture sur le domain avec numéro de siret : %s, et la campagne : %s";
    protected static String DOMAIN_UPDATE="Le domaine de siret :%s, et pour la campagne : %s est déjà dans Agrosyst avec le nom : %s. Des informations seront rajoutées";
    protected static final String PLOT_NO_CAMPAIGN_DEFINED = "Il n'y a pas de campagne de renseigné pour cette parcelle: %s";
    protected static final String PLOT_OUT_OF_RANGE_CAMPAIGN = "Campagne %d non valide pour cette parcelle: %s";
    protected static final String PLOT_UNKNOWN = "Une parcelle n'a pas d'identifiant, elle ne sera pas importée";
    protected static final String PLOT_UNKNOWN_ISSUER = "IssuerInternalReference inconnue pour la parcelle %s";
    protected static final String PLOT_NO_INDENTIFICATION = "Impossible de chercher la parcelle ayant pour identifiant :%s. Problème : les informations suivantes sont incomplètes: nom :%s, surface:%s." +
            " La parcelle ne sera pas créée et l'itinéraire technique ne sera pas complété.";
    protected static final String PLOT_INCORRECT_AREA_TYPE = "La surface de la parcelle doit être de type \"A17 - Surface de la parcelle culturale\"  ou \"ABE05 - Surface de la parcelle pérenne\", " +
            "pour la parcelle ayant pour identifiant :%s. Impossible de chercher la parcelle, l'itinéraire technique ne sera pas complété.";
    protected static final String TO_MANY_PLOTS_FOUND = "Trop de parcelles avec comme nom:%s et comme surface:%s dans le domaine :%s ! Les informations ne seront pas ajoutées.";
    protected static final String PLOT_CREATED = "Aucune parcelle trouvée pour cette campagne ! Elle sera créée";
    protected static final String ERROR_SOIL_OCCUPATION_NO_ISSUERID = "La parcelle ayant pour identifiant (%s) possède une occupation du sol sans identifiant. Aucune donnée ne pourra être ajoutée";
    protected static final String ERROR_SOIL_OCCUPATION_NO_TYPE = "L'occupation du sol (%s) sur la parcelle (%s), ne possède pas de type. Cette information est obligatoire.";
    protected static final String ERROR_SOIL_OCCUPATION_WRONG_TYPE = "L'occupation du sol (%s) sur la parcelle (%s) est de type (%s). Ce type d'occupation du sol n'est pas géré par Agrosyst. Contactez l'équipe Agrosyst.";
    protected static final String ERROR_SOIL_OCCUPATION_WRONG_RANK_FORMAT = "Le rang de l'occupation du sol ayant pour identifiant : %s, n'a pas le bon format (il faut un entier ou une valeur nulle)"; 
    protected static final String ERROR_SOIL_OCCUPATION_RANK_BELOW_ZERO = "Le rang de l'occupation du sol ayant pour identifiant : %s (parcelle : %s), n'a pas le bon format (il faut un entier positif)";
    protected static final String ERROR_SOIL_OCCUPATION_RANK_MUST_BE_ZERO = "Le rang de l'occupation du sol ayant pour identifiant : %s, doit être zero (il est de %s)";
    protected static final String ERROR_SOIL_OCCUPATION_NO_SPECIES = "L'occupation du sol ayant pour identifiant : %s, ne possède pas de cultures";
    protected static final String ERROR_SOIL_OCCUPATION_CONNECTION_WITHOUT_NODE = "Une succession de cultures assolées possède des cultures intermédiaires mais aucune culture principale (occupation du sol : %s, rang : %s)";
    protected static final String ERROR_SOIL_OCCUPATION_NO_NODE = "Succession de cultures assolées : il n'est pas possible d'ajouter une culture intermédiaire si aucune culture principale n'existe (occupation du sol : %s, rang : %s)";
    protected static final String ERROR_SOIL_OCCUPATION_NO_TARGET = "Occupation du sol (%s) sur la parcelle (%s), impossible d'importer la culture intermédiaire (%s) car la culture principale n'existe pas";
    protected static final String ERROR_SOIL_OCCUPATION_NO_SOURCE = "Occupation du sol (%s) sur la parcelle (%s), impossible d'importer la culture intermédiaire (%s) car la culture précédente n'existe pas";
    protected static final String ERROR_SOIL_OCCUPATION_WRONG_CROPPING_TYPE = "Sur la parcelle (%s), l'occupation du sol (%s) possède une culture pérenne (%s) associée à une culture dérobée ou intermédaire. Ceci empêche l'import de se dérouler convenablement.";
    protected static final String ERROR_SOIL_OCCUPATION_TOO_MANY_PROD_CYCLE = "Sur la parcelle (%s), l'occupation du sol (S) possède plus d'un cycle de production.";
    protected static final String ERROR_SPECIES_BOTANICALCODE_MISSING = "L'occupation du sol ayant pour identifiant : %s, possède une culture sans identifiant d'espèce botanique";
    protected static final String ERROR_NO_EXISTING_SPECIES = "L'occupation du sol ayant pour identifiant : %s, possède une espèce qui n'a pas été retrouvée dans le référentiel des espèces Agrosyst. Voici les caractéristiques de cette culture : Espèce botanique (%s), Qualifiant (%s), Type saisonnier (%s)";
    protected static final String ERROR_NO_PLANT_COVER = "L'occupation du sol ayant pour identifiant : %s, possède une espèce à laquelle il n'est pas possible d'attribuer un type (assolée ou pérenne). Contactez l'équipe Agrosyst avec les caractéristiques de cette culture : Espèce botanique (%s), Qualifiant (%s), Type saisonnier (%s)";
    protected static final String ERROR_NO_VALID_SPECIES = "L'occupation du sol ayant pour identifiant : %s, ne possède aucune espèce valide permettant de créer une culture"; 
    protected static final String ERROR_GEVESCODE_NOT_A_NUMBER = "L'occupation du sol ayant pour identifiant : %s, possède une variété dont l'identifiant n'est pas un nombre. Cet identifiant est %s";
    protected static final String ERROR_GEVESCODE_MISSING = "L'occupation du sol ayant pour identifiant : %s, possède une variété dont l'identifiant n'est pas présent dans le référentiel variété d'Agrosyst. Contactez l'équipe Agrosyst avec l'identifiant de cette variété %s";
    protected static final String ERROR_VARIETY_NAME_MISSING = "L'occupation du sol ayant pour identifiant : %s, possède une variété sans libellé (Espèce botanique - %s, Qualifiant - %s, Type saisonnier - %s). Cette donnée est obligatoire";
    // TODO 2015-02-09 eancelet : ajouter le nom de l'espèce dans le message ci-dessous
    protected static final String ERROR_VARIETY_MISSING = "L'occupation du sol ayant pour identifiant : %s, possède une variété dont le nom n'est pas présent dans le référentiel variété d'Agrosyst. Contactez l'équipe Agrosyst avec le nom de cette variété %s";
    protected static final String ERROR_MORE_THAN_ONE_VARIETY = "L'occupation du sol ayant pour identifiant : %s, possède une culture reliée à plusieurs variétés (maximum 1 variété). Cette culture est la suivante : Espèce botanique (%s), Qualifiant (%s), Type saisonnier (%s)";
    protected static final String ERROR_INTERVENTION_NO_GUID = "Sur la parcelle (%s), occupation du sol (%s), une intervention ne possède pas d'identifiant. Import Impossible.";
    protected static final String ERROR_INTERVENTION_NO_EDI_TYPE = "Sur la parcelle (%s), occupation du sol (%s), l'intervention (%s) n'a pas de type renseigné. Import impossible.";
    protected static final String ERROR_INTERVENTION_NO_SUBORDINATE_TYPE = "Sur la parcelle (%s), occupation du sol (%s), l'intervention (%s) n'a pas de qualifiant renseigné. Import impossible.";
    protected static final String ERROR_INTERVENTION_NOT_ONE_PERIOD = "Sur la parcelle (%s), occupation du sol (%s), l'intervention (%s) doit avoir une et une seule fenêtre temporelle";
    protected static final String ERROR_INTERVENTION_NO_STARTING_DATE = "Sur la parcelle (%s), occupation du sol (%s), l'intervention (%s) n'a pas de date de début. Import impossible.";
    protected static final String ERROR_INTERVENTION_TO_MANY_REASONS = "Sur la parcelle (%s), occupation du sol (%s), l'intervention (%s) a trop de commentaires. Import impossible.";
    
    // info messages
    protected static String INFO_PLOT_EMPTY = "Pas de parcelles pour le domaine de Siret %s, aucune information ne sera importée";
    protected static String INFO_NO_PLOT_SOIL_OCCUPATION = "Il n'y a pas d'occupation du sol pour la parcelle ayant pour identifiant %s. Aucune information ne sera ajoutée";

    protected static final String VALID_TYPE_CODE = "415";

    protected DomainTopiaDao domainTopiaDao;
    
    protected DomainService domainService;

    protected PlotTopiaDao plotTopiaDao;
    
    protected ZoneTopiaDao zoneTopiaDao;

    protected RefLocationTopiaDao locationTopiaDao;
    
    protected EffectiveCropCycleService effectiveCropCycleService;
    
    protected EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao;
    
    protected EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao;
    
    protected EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao;
    
    protected EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao;
    
    protected RefCultureEdiGroupeCouvSolTopiaDao refCultureEdiGroupeCouvSolTopiaDao;
    
    protected RefEspeceTopiaDao refEspeceTopiaDao;
    
    protected RefVarieteGevesTopiaDao refVarieteGevesTopiaDao;
    
    protected RefVarietePlantGrapeTopiaDao refVarietePlantGrapeTopiaDao;

    protected BusinessAuthorizationService authorizationService;
    
    protected EdaplosParsingResult edaplosParsingResult;    
    

    public void setDomainTopiaDao(DomainTopiaDao domainTopiaDao) {
        this.domainTopiaDao = domainTopiaDao;
    }

    public void setPlotTopiaDao(PlotTopiaDao plotTopiaDao) {
        this.plotTopiaDao = plotTopiaDao;
    }

    public void setZoneTopiaDao(ZoneTopiaDao zoneTopiaDao) {
        this.zoneTopiaDao = zoneTopiaDao;
    }

    public void setEffectiveSeasonalCropCycleTopiaDao(
            EffectiveSeasonalCropCycleTopiaDao effectiveSeasonalCropCycleTopiaDao) {
        this.effectiveSeasonalCropCycleTopiaDao = effectiveSeasonalCropCycleTopiaDao;
    }

    public void setLocationTopiaDao(RefLocationTopiaDao locationTopiaDao) {
        this.locationTopiaDao = locationTopiaDao;
    }
    
    public void setEffectiveCropCycleService(EffectiveCropCycleService effectiveCropCycleService) {
        this.effectiveCropCycleService = effectiveCropCycleService;
    }

    public void setEffectiveCropCycleNodeTopiaDao(EffectiveCropCycleNodeTopiaDao effectiveCropCycleNodeTopiaDao) {
        this.effectiveCropCycleNodeTopiaDao = effectiveCropCycleNodeTopiaDao;
    }

    public void setEffectiveCropCycleConnectionTopiaDao(EffectiveCropCycleConnectionTopiaDao effectiveCropCycleConnectionTopiaDao) {
        this.effectiveCropCycleConnectionTopiaDao = effectiveCropCycleConnectionTopiaDao;
    }

    public void setEffectivePerennialCropCycleTopiaDao(EffectivePerennialCropCycleTopiaDao effectivePerennialCropCycleTopiaDao) {
        this.effectivePerennialCropCycleTopiaDao = effectivePerennialCropCycleTopiaDao;
    }   

    public void setRefCultureEdiGroupeCouvSolTopiaDao(RefCultureEdiGroupeCouvSolTopiaDao refCultureEdiGroupeCouvSolTopiaDao) {
        this.refCultureEdiGroupeCouvSolTopiaDao = refCultureEdiGroupeCouvSolTopiaDao;
    }
    
    public void setRefEspeceTopiaDao(RefEspeceTopiaDao refEspeceTopiaDao) {
        this.refEspeceTopiaDao = refEspeceTopiaDao;
    }    

    public void setRefVarieteGevesTopiaDao(RefVarieteGevesTopiaDao refVarieteGevesTopiaDao) {
        this.refVarieteGevesTopiaDao = refVarieteGevesTopiaDao;
    }    

    public void setRefVarietePlantGrapeTopiaDao(RefVarietePlantGrapeTopiaDao refVarietePlantGrapeTopiaDao) {
        this.refVarietePlantGrapeTopiaDao = refVarietePlantGrapeTopiaDao;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public EdaplosParsingResult parse(Document document) {
        edaplosParsingResult = processEdaplosFileParsing(document);
        return edaplosParsingResult;
    }
    

    /**
     * Process document
     */
    protected EdaplosParsingResult processEdaplosFileParsing(Document document) {
        
        // Dto to transfert info on domain and plots (to be created, to be completed ...)
        EdaplosParsingResult eDaplosParsingResult = new EdaplosParsingResult();

        boolean endParsing = false;
        while (!endParsing) {
            Element rootElement = document.getRootElement();

            if (rootElement != null) {
                // extract metadata part
                Element metadataElement = rootElement.element("CropDataSheetDocument");

                if(metadataElement == null){
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage("Le document ne contient pas les informations nécessaires comme le TypeCode et l'Identification.");
                    break;
                }

                String typeCode = metadataElement.elementText("TypeCode");
                String documentId = metadataElement.elementText("Identification");

                if (typeCode == null || documentId == null) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage("Le document ne contient pas les informations nécessaires comme le TypeCode et l'Identification.");
                    break;
                }

                if (!VALID_TYPE_CODE.equalsIgnoreCase(typeCode)) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage(String.format(ERROR_SIRET, documentId));
                    break;
                }

                // get message issuer Info (= domain)
                Element issuerCropDataSheetParty = metadataElement.element("IssuerCropDataSheetParty");

                if (issuerCropDataSheetParty == null) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage("Le document ne contient pas les informations nécessaires pour l'émetteur.");
                    break;
                }

                String siret = issuerCropDataSheetParty.elementText("Identification");
                String name = issuerCropDataSheetParty.elementText("Name");
                Element address = issuerCropDataSheetParty.element("SpecifiedUnstructuredAddress");

                if (address == null) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage("Le document ne contient pas les informations à la localisation du domain (balise SpecifiedUnstructuredAddress).");
                    break;
                }
                String cityName = address.elementText("CityName");
                String postcodeCode = address.elementText("PostcodeCode");

                if (StringUtils.isBlank(cityName) || StringUtils.isBlank(postcodeCode)) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage("Le document ne contient pas les informations à la localisation du domain (Code postal).");
                    break;
                }

                // first try find exact same city otherwise try find one from postcode
                RefLocation location = locationTopiaDao.forProperties(RefLocation.PROPERTY_CODE_POSTAL, postcodeCode, RefLocation.PROPERTY_COMMUNE, cityName).findAnyOrNull();
                if (location == null) {
                    List<RefLocation> locations = locationTopiaDao.forProperties(RefLocation.PROPERTY_CODE_POSTAL, postcodeCode).findAll();
                }

                if (location == null) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage(String.format("Le nom de la commune %s et son code postal %s ne permettent pas de retouver son emplacement", cityName, postcodeCode));
                    break;
                }

                Element recipientCropDataSheetParty = metadataElement.element("RecipientCropDataSheetParty");
                if (recipientCropDataSheetParty == null) {
                    eDaplosParsingResult.addErrorMessage("Les information concernant le responsable du domaine ne peuvent être retrouvées (balise RecipientCropDataSheetParty).");
                    break;
                }

                Element contact = recipientCropDataSheetParty.element("DefinedPartyContact");
                if (contact == null) {
                    eDaplosParsingResult.addErrorMessage("Les information concernant le responsable du domaine ne peuvent être retrouvées (balise DefinedPartyContact).");
                    break;
                }

                String mainContact = contact.elementText("PersonName");
                if (mainContact == null) {
                    eDaplosParsingResult.addErrorMessage("Les information concernant le responsable du domaine ne peuvent être retrouvées (balise PersonName).");
                    break;
                }


                if (StringUtils.isBlank(siret)) {
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    eDaplosParsingResult.addErrorMessage(String.format(ERROR_SIRET, documentId));
                    break;
                }

                // get plots info to get campaigns. There can be several plots per message
                List<Element> plotElements = rootElement.elements("AgriculturalPlot");

                if (plotElements == null || plotElements.isEmpty()) {
                    eDaplosParsingResult.addInfoMessage(String.format(INFO_PLOT_EMPTY, siret));
                    break;
                }


                Set<String> stCampaigns = new HashSet<String>();
                // get campaigns in plot markups
                for (Element plot : plotElements){
                    String harvestedYear = plot.elementText("HarvestedYear");
                    if (StringUtils.isNotBlank(harvestedYear)){
                        stCampaigns.add(harvestedYear);
                    }
                }

                if (stCampaigns.isEmpty()) { // si aucune campagne détectée
                    eDaplosParsingResult.addErrorMessage(String.format(ERROR_NO_CAMPAIGN_FOR_DOMAIN, name, siret));
                    eDaplosParsingResult.setEdaplosParsingStatus(EdaplosParsingStatus.FAIL);
                    break;
                }

                // test if domain*campaign exist
                for (String stCampaign : stCampaigns) {
                    EdaplosDomainDto edaplosDomainDto = new EdaplosDomainDto();
                    edaplosDomainDto.setSiret(siret);
                    edaplosDomainDto.setMainContact(mainContact);
                    edaplosDomainDto.setCampaign(stCampaign);
                    edaplosDomainDto.setName(name);
                    edaplosDomainDto.setCityName(cityName);
                    edaplosDomainDto.setPostcodeCode(postcodeCode);
                    edaplosDomainDto.setLocation(location);

                    if (!CommonService.ARE_CAMPAIGNS_VALIDS(stCampaign)) {
                        String invalidCampaign = String.format(ERROR_INVALID_CAMPAIGN, stCampaign, siret);
                        edaplosDomainDto.addErrorMessage(invalidCampaign);
                        eDaplosParsingResult.addDomain(edaplosDomainDto);
                        continue;
                    }

                    Integer campaign = Integer.valueOf(stCampaign);

                    // extract domains by siret and campaign
                    // TODO gérer exceptions si plusieurs domaines sont remontés
                    Domain domain = domainTopiaDao.forProperties(Domain.PROPERTY_SIRET, siret, Domain.PROPERTY_CAMPAIGN, campaign).findUniqueOrNull();

                    if (domain == null) {
                        edaplosDomainDto.setStatus(EdaplosDomainImportStatus.CREATE);
                        edaplosDomainDto.addInfoMessage(String.format(DOMAIN_CREATE, siret, stCampaign));

                    } else if (!authorizationService.isDomainWritable(domain.getTopiaId())) {
                        edaplosDomainDto.setTopiaId(domain.getTopiaId());
                        edaplosDomainDto.setCampaign(stCampaign);
                        edaplosDomainDto.setStatus(EdaplosDomainImportStatus.FORBIDDEN);
                        edaplosDomainDto.addInfoMessage(String.format(DOMAIN_FORBIDDEN, siret, stCampaign));
                        eDaplosParsingResult.addDomain(edaplosDomainDto);
                        continue;
                    } else {
                        edaplosDomainDto.setTopiaId(domain.getTopiaId());
                        edaplosDomainDto.setCampaign(stCampaign);
                        // get the persisted croppingplanentries
                        List<CroppingPlanEntryDto> persistedCroppingPlanEntries = domainService.getCroppingPlanDtos(domain.getTopiaId());
                        edaplosDomainDto.setCroppingPlanEntryDtos(Lists.newArrayList(persistedCroppingPlanEntries)); // eancelet 2015-03-03 : we need to set a new arrayList otherwise persistedCroppingPlanEntries cannot be modified                  
                        
                        edaplosDomainDto.setStatus(EdaplosDomainImportStatus.UPDATE);
                        edaplosDomainDto.addInfoMessage(String.format(DOMAIN_UPDATE, siret, stCampaign, domain.getName()));
                        // TODO : ajouter 
                    }

                    // parse plots
                    processPlotImport(edaplosDomainDto, plotElements, domain, campaign);
                    eDaplosParsingResult.addDomain(edaplosDomainDto);
                }
                endParsing = true;
            }

        }
        return eDaplosParsingResult;
    }

    /**
     * 
     * Analyse les parcelles
     * Required fields for plot are:
     *    - name
     *    - waterFlowDistance
     *    - bufferStrip
     *    - maxSlope
     *    - domain
     *    a zone is required to set plot in
     */
    protected void processPlotImport(EdaplosDomainDto edaplosDomainDto, List<Element> plotList, Domain domain, Integer campaign) {
        
        for (Element plotElement : plotList){
            
            EdaplosPlotDto edaplosPlotDto = new EdaplosPlotDto();
            
            // Test plot name
            String plotName = plotElement.elementText("Identification");
            if (StringUtils.isBlank(plotName)){
                edaplosPlotDto.addPlotErrorMessage("Un nom de parcelle est manquant sur le domaine : " + edaplosDomainDto.getSiret());
                edaplosDomainDto.addPlot(edaplosPlotDto);
                continue;
            }
            edaplosPlotDto.setPlotName(plotName);
            
            // Test plot identifier
            String issuerPlotId = plotElement.elementText("IssuerInternalReference");
            if (StringUtils.isBlank(issuerPlotId)) {
                String invalidIdentifier = String.format(ERROR_INVALID_IDENTIFIER_FOR_PLOT, plotName, edaplosDomainDto.getSiret());
                edaplosPlotDto.addPlotErrorMessage(invalidIdentifier);
                edaplosDomainDto.addPlot(edaplosPlotDto);
                continue;
            }
            edaplosPlotDto.setIssuerId(issuerPlotId);
            
            // Test plot campaign
            String plotCampaign = plotElement.elementText("HarvestedYear");
            if (StringUtils.isEmpty(plotCampaign) || !CommonService.ARE_CAMPAIGNS_VALIDS(plotCampaign)) {
                String invalidPlotCampaign = String.format(ERROR_INVALID_CAMPAIGN_FOR_PLOT, issuerPlotId, plotName, edaplosDomainDto.getSiret());
                edaplosPlotDto.addPlotErrorMessage(invalidPlotCampaign);
                edaplosDomainDto.addPlot(edaplosPlotDto);
                continue;
            }
            // 
            Integer plotCampaignInt = Integer.valueOf(plotElement.elementText("HarvestedYear"));
      
            
            if (!plotCampaignInt.equals(campaign)){
                // TODO Si la campagne de la parcelle n'est pas la même que celle étudiée alors on passe à la parcelle suivante
                // edaplosPlotDto.addPlotErrorMessage(String.format(PLOT_OUT_OF_RANGE_CAMPAIGN, campaign, plotIssuerId));
                // edaplosDomainDto.addPlot(edaplosPlotDto);
                continue;
            }


            Plot plot = null;
            Zone zone = null;
            if (domain != null && domain.isPersisted()) { // 2 ways to find a plot in database : by name + area + campaign or by issuerPlotId
                
                // 1st solution : find plot by issuerPlotId (eDaplosIssuerId)
                if (StringUtils.isNotBlank(issuerPlotId)){
                    plot = plotTopiaDao.forProperties(Plot.PROPERTY_E_DAPLOS_ISSUER_ID, issuerPlotId, Plot.PROPERTY_DOMAIN, domain).findUniqueOrNull();
                    
                    // eancelet le 02/12/2014 : le code ci-dessous n'est plus valable 
                    //edaplosPlotDto.addPlotErrorMessage(String.format(PLOT_UNKNOWN_ISSUER, plotId));
                    //edaplosDomainDto.addPlot(edaplosPlotDto);
                    //continue;
                }                

                // 2nd solution : find plot by Name and Area
                if (plot == null) {
                    Element includedAgriculturalArea = plotElement.element("IncludedAgriculturalArea");

                    // si le type de l'air n'est pas le bon alors l'import n'est pas possible
                    if (includedAgriculturalArea != null) {
                        if (includedAgriculturalArea.elementText("Type") == null || (!includedAgriculturalArea.elementText("Type").equals("A17") && !includedAgriculturalArea.elementText("Type").equals("ABE05"))) {
                            edaplosPlotDto.addPlotErrorMessage(String.format(PLOT_INCORRECT_AREA_TYPE, issuerPlotId));
                            edaplosDomainDto.addPlot(edaplosPlotDto);
                            continue;
                        }
                    }

                    // s'il manque la surface ou le nom de la parcelle, on ne peut pas la chercher
                    if (StringUtils.isBlank(plotName) || includedAgriculturalArea == null || StringUtils.isBlank(includedAgriculturalArea.elementText("ActualMeasure"))) {
                        plotName = StringUtils.isBlank(plotName) ? "NOM INCONNU" : plotName;
                        edaplosPlotDto.setPlotName(plotName);
                        String area = (includedAgriculturalArea == null || StringUtils.isBlank(includedAgriculturalArea.elementText("ActualMeasure"))) ? "SURFACE INCONNUE" : includedAgriculturalArea.elementText("ActualMeasure");
                        edaplosPlotDto.setArea(area);
                        edaplosPlotDto.addPlotErrorMessage(String.format(PLOT_NO_INDENTIFICATION, issuerPlotId, plotName, area));
                        edaplosDomainDto.addPlot(edaplosPlotDto);
                        continue;
                    }

                    double plotArea = Double.valueOf(includedAgriculturalArea.elementText("ActualMeasure"));
                    // on cherche la parcelle en fonction de son domaine, son nom et sa surface
                    List<Plot> plots = plotTopiaDao.forProperties(Plot.PROPERTY_DOMAIN, domain, Plot.PROPERTY_NAME, plotName, Plot.PROPERTY_AREA, plotArea).findAll();

                    if (plots.size()>1){
                        edaplosPlotDto.setStatus("MANY");
                        edaplosPlotDto.addPlotErrorMessage(String.format(TO_MANY_PLOTS_FOUND, plotName, plotArea, domain.getName()));
                        edaplosDomainDto.addPlot(edaplosPlotDto);
                        continue;
                    }  else if (plots.size() == 1) {
                        plot = plots.get(0);
                        edaplosPlotDto.setIssuerId(issuerPlotId);
                    }
                }
            }


            if (plot == null){
                // add test to know if this plot exists on other campaigns
                // TODO DCossé 22/10/14 pourquoi ? eancelet : pour assurer la filiation
                edaplosPlotDto.setIssuerId(issuerPlotId);
                edaplosPlotDto.setStatus("CREATE");
                edaplosPlotDto.addPlotInfoMessage(PLOT_CREATED);
            } else {
                log.warn("Parcelle trouvée !");
                // add info to EdaplosImportStatusDto
                zone = zoneTopiaDao.forPlotEquals(plot).findUnique();
                edaplosPlotDto.setTopiaId(plot.getTopiaId());
                edaplosPlotDto.setStatus("OK");
                
                
                List<EffectiveSeasonalCropCycleDto> effectiveSeasonalCropCycleDtos = effectiveCropCycleService.getAllEffectiveSeasonalCropCyclesDtos(zone.getTopiaId());
                if (effectiveSeasonalCropCycleDtos != null && !effectiveSeasonalCropCycleDtos.isEmpty()) {
                    edaplosPlotDto.setSeasonalCycleDto(effectiveSeasonalCropCycleDtos.get(0));
                }
                
                List<EffectivePerennialCropCycleDto> effectivePerennialCropCycleDtos = effectiveCropCycleService.getAllEffectivePerennialCropCyclesDtos(zone.getTopiaId());
                if (effectivePerennialCropCycleDtos != null && !effectivePerennialCropCycleDtos.isEmpty()) {
                    edaplosPlotDto.setPerennialCycleDtos(effectivePerennialCropCycleDtos);
                }
                
                // TODO eancelet 2015-03-05 : traiter les prix                

            }

            // TODO DCossé 22/10/14  les champs suivant sont absents
            /**
             *
                 protected WaterFlowDistance waterFlowDistance;
                 protected BufferStrip bufferStrip;
                 protected MaxSlope maxSlope;
             */
            
            // get plotSoilOccupations
            List<Element> soilOccupationElements = plotElement.elements("AppliedPlotSoilOccupation");
            
            // if no plotSoilOccupation then no need to go further for this plot 
            if (soilOccupationElements == null || soilOccupationElements.isEmpty()) {
                edaplosPlotDto.addPlotInfoMessage(String.format(INFO_PLOT_EMPTY, issuerPlotId));
            } else { // else plotSoilOccupations analysis
                processPlotSoilOccupation(edaplosPlotDto,soilOccupationElements, zone, edaplosDomainDto);
            }
                        
            
            edaplosDomainDto.addPlot(edaplosPlotDto);
        }
    }
    
    /**
     * Analyse des occupations du sol
     * L'équivalent de l'occupation du sol dans Agrosyst est :
     * * * pour les espèces assolées : effectivecropcyclenode pour les cultures principales et effectivecropcycleconnection pour les intermédiaires
     * * * pour les espèces pérennes : effectiveperennialcropcycle
     */
    protected void processPlotSoilOccupation(EdaplosPlotDto edaplosPlotDto, List<Element> soilOccupationList, Zone zone, EdaplosDomainDto eDomainDto){
        for (Element soilOccupationElement : soilOccupationList){
            // Premise : only one effectiveseasonalcropcycle per zone (so per plot, as there is only one zone per plot using edaplos), so it is easy to get the effectiveseasonalcropcycle from the plot
            
            //EdaplosPlotSoilOccupationDto edaplosSoilOccupationDto = new EdaplosPlotSoilOccupationDto();
            String soilOccupationIssuerId = soilOccupationElement.elementText("Identification");
            // TODO 2015-02-04 eancelet : enlever la variable Identification de EdaplosSoilOccupationDto ?
            // edaplosSoilOccupationDto.setIdentification(soilOccupationIssuerId); // needed in constructSpeciesSet()
            
            if (StringUtils.isBlank(soilOccupationIssuerId)){
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_ISSUERID, edaplosPlotDto.getIssuerId()));
                continue;
            }
                        
            String soilOccupationType = soilOccupationElement.elementText("TypeCode");
            if (StringUtils.isBlank(soilOccupationType)){
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_TYPE, soilOccupationIssuerId, edaplosPlotDto.getIssuerId()));
                continue;
            }
            
            // only one culture per soiloccupation (made of one or more species)
            CroppingPlanEntryDto croppingPlanEntryDto = new CroppingPlanEntryDto();            
            
            // set croppingPlanEntryType
            CroppingEntryType croppingEntryType; 
            if (soilOccupationType.equals("ABA00")) {
                croppingPlanEntryDto.setType(CroppingEntryType.MAIN);
            } else if (soilOccupationType.equals("ABA09")) {
                croppingPlanEntryDto.setType(CroppingEntryType.CATCH);
            } else if (soilOccupationType.equals("ABA03") || soilOccupationType.equals("ABA04")) {
                croppingPlanEntryDto.setType(CroppingEntryType.INTERMEDIATE);                
            } else {
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_WRONG_TYPE, soilOccupationIssuerId, edaplosPlotDto.getIssuerId(), soilOccupationType));
                continue;                
            }   
            
            // Determine whether the culture is seasonal or perennial in order to know which type of cropcycle to create (seasonalcropcycle or perennialcrop cycle ?) 
            // Premise : we take only the first species
            List<Element> speciesOnSoilOccupation = soilOccupationElement.elements("SownAgriculturalCrop");
            
            if (speciesOnSoilOccupation == null || speciesOnSoilOccupation.isEmpty()) {
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_SPECIES, soilOccupationIssuerId));
                continue;
            }  
            
            // test species format and populate CroppingPlanEntryDto                
            addSpeciesToCroppingPlanEntry(speciesOnSoilOccupation, edaplosPlotDto, croppingPlanEntryDto, soilOccupationIssuerId);
            if (croppingPlanEntryDto.getSpecies() == null || croppingPlanEntryDto.getSpecies().isEmpty()){
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_NO_VALID_SPECIES, soilOccupationIssuerId));
                continue;                
            }            
            
            CroppingPlanSpeciesDto speciesToGetCycle = croppingPlanEntryDto.getSpecies().iterator().next();
            
            // HashMap<Object,Object> CycleTypeMap = new HashMap<Object, Object>();
            TypeCulture soilOccType = getCycleTypeFromSpecies(speciesToGetCycle);
            if (soilOccType == null) {
                // TODO
                //edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_NO_PLANT_COVER, soilOccupationIssuerId, species, speciesSupplementaryBotanicalCode, speciesSowingPeriodCode));
                continue;
            } 
            
            if (CollectionUtils.isNotEmpty(eDomainDto.getCroppingPlanEntryDtos())) { // domain already has croppingPlanEntry, we need to compare this new CroppingPlanEntry to the already existing ones 
                // for each croppingplanentry in domain
                getRightCroppingPlanEntry(croppingPlanEntryDto, eDomainDto.getCroppingPlanEntryDtos());
            }
            
            if (croppingPlanEntryDto.getTopiaId() == null) {
                croppingPlanEntryDto.setTopiaId("new-entry-"+UUID.randomUUID().toString());
                String agrosystCode = UUID.randomUUID().toString();
                croppingPlanEntryDto.setCode(agrosystCode);
                eDomainDto.addCroppingPlanEntryDto(croppingPlanEntryDto);
            }
            
            // Now we get our croppingPlanEntryDto and we know which cycle to construct  
            
            
            String soilOccupationSequence = soilOccupationElement.elementText("SequenceNumeric");
            // the PlotSoilOccupation.SequenceNumeric has to be an integer or ' ' or null
            if (!StringUtils.isNumericSpace(soilOccupationSequence)) {
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_WRONG_RANK_FORMAT, soilOccupationIssuerId));
                continue;
            }                        
            int soilOccupationRank = 1;
            if (!StringUtils.isBlank(soilOccupationSequence)){ // if sequenceNumeric is Blank then it is 1
                soilOccupationRank = Integer.parseInt(soilOccupationSequence)-1; // numbering start with 1 in eDaplos while 0 in Agrosyst
            }
            if (soilOccupationRank < 0) {
                edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_RANK_BELOW_ZERO, soilOccupationIssuerId, edaplosPlotDto.getIssuerId()));
                continue;
            }
            
            
            List<Element> prodCycles = soilOccupationElement.elements("SpecifiedAgriculturalCropProductionCycle");
           
            if (soilOccType.equals(TypeCulture.PERENNE) || soilOccType.equals(TypeCulture.PLURI_ANNUELLE)) {
                
                if (! croppingPlanEntryDto.getType().equals(CroppingEntryType.MAIN)) {
                    edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_WRONG_CROPPING_TYPE, soilOccupationIssuerId, edaplosPlotDto.getIssuerId(), croppingPlanEntryDto.getName()));
                    continue;
                }
                List<EffectivePerennialCropCycleDto> effPerennialCycleDtos = edaplosPlotDto.getPerennialCycleDtos();
                // effPerennialCycleDtos can't be null as it is initialised when the object is created
                EffectivePerennialCropCycleDto perennialCycleDto = null;
                // serach if perennialCycleDto exists. Serach first by edaplosIssuerId then by croppingPlanEntry.topiaId
                for (EffectivePerennialCropCycleDto existingPerennialCycleDto : effPerennialCycleDtos){
                    if (existingPerennialCycleDto.getEdaplosIssuerId() != null && existingPerennialCycleDto.getEdaplosIssuerId().equals(soilOccupationIssuerId)) {
                        perennialCycleDto = existingPerennialCycleDto;
                        break;
                    } else if (existingPerennialCycleDto.getCroppingPlanEntryId().equals(croppingPlanEntryDto.getTopiaId())) {
                        perennialCycleDto = existingPerennialCycleDto;
                        perennialCycleDto.setEdaplosIssuerId(soilOccupationIssuerId);
                        break;
                    }
                }
                if (perennialCycleDto == null) {
                    perennialCycleDto = new EffectivePerennialCropCycleDto();
                    perennialCycleDto.setCroppingPlanEntryId(croppingPlanEntryDto.getTopiaId());
                    List<EffectiveCropCyclePhaseDto> phaseDtos = new ArrayList<EffectiveCropCyclePhaseDto>();
                    EffectiveCropCyclePhaseDto phaseDto = new EffectiveCropCyclePhaseDto();
                    phaseDto.setType(CropCyclePhaseType.PLEINE_PRODUCTION);
                    phaseDto.setInterventions(new ArrayList<EffectiveInterventionDto>());
                    phaseDtos.add(phaseDto);
                    perennialCycleDto.setPhaseDtos(phaseDtos);
                    perennialCycleDto.setWeedType(WeedType.PARTIEL);
                    perennialCycleDto.setEdaplosIssuerId(soilOccupationIssuerId);
                    effPerennialCycleDtos.add(perennialCycleDto);                    
                }
                
                
            } else { // seasonal species                
                EffectiveSeasonalCropCycleDto effSeasonalCropCycleDto = edaplosPlotDto.getSeasonalCycleDto();
                if (effSeasonalCropCycleDto == null) {
                    effSeasonalCropCycleDto = new EffectiveSeasonalCropCycleDto();
                    edaplosPlotDto.setSeasonalCycleDto(effSeasonalCropCycleDto);
                }                
                if (croppingPlanEntryDto.getType().equals(CroppingEntryType.CATCH) || croppingPlanEntryDto.getType().equals(CroppingEntryType.MAIN)) { // Main and catch croppingPlanEntries 
                    
                    // tests
                    if (CollectionUtils.isEmpty(edaplosPlotDto.getSeasonalCycleDto().getNodeDtos()) && ! CollectionUtils.isEmpty(edaplosPlotDto.getSeasonalCycleDto().getConnectionDtos())) {
                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_CONNECTION_WITHOUT_NODE, soilOccupationIssuerId, soilOccupationRank));
                        continue;
                    }                    
                    if (CollectionUtils.isEmpty(edaplosPlotDto.getSeasonalCycleDto().getNodeDtos()) && soilOccupationRank > 0){
                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_RANK_MUST_BE_ZERO, soilOccupationIssuerId, soilOccupationRank));
                        continue;
                    }
                    
                    // get or construct nodes
                    List<EffectiveCropCycleNodeDto> effNodeDtos = effSeasonalCropCycleDto.getNodeDtos();
                    if (effNodeDtos == null) {
                        effNodeDtos = new ArrayList<EffectiveCropCycleNodeDto>();
                        effSeasonalCropCycleDto.setNodeDtos(effNodeDtos);
                    }                    
                    EffectiveCropCycleNodeDto effNodeDto = null;
                    // TODO eancelet 2015-03-05 : traiter les prix                        
                    Iterator i = effNodeDtos.iterator();                    
                    while (effNodeDto == null && i.hasNext()) { // test if node exists (with edaplosIssuerId or with rank)
                        EffectiveCropCycleNodeDto eNodeDto = (EffectiveCropCycleNodeDto) i.next();
                        if (eNodeDto.getEdaplosIssuerId().equals(soilOccupationIssuerId) || eNodeDto.getX() == soilOccupationRank) {
                            effNodeDto = eNodeDto;
                            if (effNodeDto.getEdaplosIssuerId() == null || effNodeDto.getEdaplosIssuerId().isEmpty()) {
                                // TODO ? eancelet 2015-03-05 : ajouter un test pour voir si on a la même culture en base que dans eDaplos pour contrôle ?
                                effNodeDto.setEdaplosIssuerId(soilOccupationIssuerId);
                            }
                            // ("UPDATE");
                            // eancelet 2015-03-05 : pour le moment on ne fait rien de plus, après il faudra rajouter les interventions
                        } 
                    } 
                    if (effNodeDto == null) { // the node needs to be created
                        // ("CREATE")
                        effNodeDto = createNodeDto(soilOccupationIssuerId, croppingPlanEntryDto.getTopiaId(), soilOccupationRank);
                        // 2015-03-12 eancelet : à priori pas besoin de contrôler si le numéro de rang est déjà utilisé car déjà testé juste au-dessus 
                        effNodeDtos.add(effNodeDto); 
                    }
                    if (effSeasonalCropCycleDto.getConnectionDtos() == null) {
                        List<EffectiveCropCycleConnectionDto> connectionDtos = new ArrayList<EffectiveCropCycleConnectionDto>();
                        effSeasonalCropCycleDto.setConnectionDtos(connectionDtos);
                    }   
                 /*   
                    // Handle interventions
                    if (!CollectionUtils.isEmpty(prodCycles)){
                        if (prodCycles.size() == 1) {
                            List<Element> eInterventions = prodCycles.get(0).elements("ApplicablePlotAgriculturalProcess");
                            List<EffectiveInterventionDto> existingInterDtos = effNodeDto.getInterventions(); 
                            
                            // if intervention already exists then keep it, else create it
                            for (Element eIntervention : eInterventions) { 
                                String eInterventionId = eIntervention.elementText("Identification");
                                if (StringUtils.isEmpty(eInterventionId)) {
                                    edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_NO_GUID, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId));
                                    continue;

                                }                              
                                
                                EffectiveInterventionDto eInterDto = null;
                                for (EffectiveInterventionDto existingInterDto : existingInterDtos) {    
                                    if (eInterventionId.equals(existingInterDto.getEdaplosIssuerId())) {
                                        break;
                                    }
                                }
                                if (eInterDto == null) {
                                    eInterDto = new EffectiveInterventionDto();
                                    String interType = eIntervention.elementText("TypeCode");
                                    if (StringUtils.isEmpty(interType)){
                                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_NO_EDI_TYPE, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId, eInterventionId));
                                        continue;
                                    }
                                    String interSubordinateType = eIntervention.elementText("SubordinateTypeCode");
                                    if (StringUtils.isEmpty(interSubordinateType)){
                                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_NO_SUBORDINATE_TYPE, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId, eInterventionId));
                                        continue;                                    
                                    }
                                    // TODO balises à traiter (eancelet 20150422, quelles balises ?)
                                    
                                    // handle dates
                                    if (CollectionUtils.isEmpty(eIntervention.elements("OccurrenceDelimitedPeriod")) || eIntervention.elements("OccurrenceDelimitedPeriod").size() > 1) {
                                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_NOT_ONE_PERIOD, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId, eInterventionId));
                                        continue;                                         
                                    }
                                    String startingDateString = ((Element) eIntervention.elements("OccurrenceDelimitedPeriod").get(0)).elementText("StartDateTime");
                                    if (StringUtils.isEmpty(startingDateString)) {
                                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_NO_STARTING_DATE, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId, eInterventionId));
                                        continue; 
                                    }
                                    Date startingDate = null;
                                    SimpleDateFormat format = new SimpleDateFormat("yyyy-mm-dd'T'hh:mm:ss'Z'");
                                    try {
                                        startingDate = format.parse(startingDateString);
                                    } catch (ParseException e) {
                                        // TODO Auto-generated catch block
                                        e.printStackTrace();
                                    }   
                                    eInterDto.setStartInterventionDate(startingDate);
                                    String endDateString = ((Element) eIntervention.elements("OccurrenceDelimitedPeriod").get(0)).elementText("EndDateTime");
                                    Date endDate = null;
                                    if (StringUtils.isEmpty(endDateString)) {
                                        endDate = startingDate;
                                    } else {
                                        try {
                                            endDate = format.parse(endDateString);
                                        } catch (ParseException e) {
                                            // TODO Auto-generated catch block
                                            e.printStackTrace();
                                        }
                                    }
                                    eInterDto.setEndInterventionDate(endDate);
                                    
                                    // handle comment
                                    if (CollectionUtils.isNotEmpty(eIntervention.elements("SpecifiedAgriculturalProcessReason"))) {
                                        if (eIntervention.elements("SpecifiedAgriculturalProcessReason").size() > 1) {
                                            edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_INTERVENTION_TO_MANY_REASONS, edaplosPlotDto.getIssuerId(), soilOccupationIssuerId, eInterventionId));
                                            continue;                                         
                                        }
                                        else {
                                            String interventionComment = ((Element) eIntervention.elements("SpecifiedAgriculturalProcessReason").get(0)).elementText("Description");
                                            if (StringUtils.isNotEmpty(interventionComment)) {
                                                eInterDto.setComment(interventionComment);
                                            }
                                        }
                                    }
                                    
                                    // TODO eancelet 2015-04-22 : add cropstage handling 
                                    
                                    Collection<AbstractAction> actionTests = new ArrayList<AbstractAction>();
                                    eInterDto.setActions(actionTests);
                                    eInterDto.setName("Inter test");
                                    eInterDto.setSpatialFrequency(1);
                                    eInterDto.setTransitCount(1);                                         
                                    effNodeDto.addIntervention(eInterDto);
                                    
                                }
                            }
                            
                            
                            
                            
                        } else {
                            edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_TOO_MANY_PROD_CYCLE, soilOccupationIssuerId, edaplosPlotDto.getIssuerId()));
                            continue;   
                        }
                    }
                    
                    */
                    
                    
                } else if (croppingPlanEntryDto.getType().equals(CroppingEntryType.INTERMEDIATE)) { // Intermediate croppingPlanEntries
                    if (CollectionUtils.isEmpty(edaplosPlotDto.getSeasonalCycleDto().getNodeDtos())) { 
                        edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_NODE, soilOccupationIssuerId, soilOccupationRank));
                        continue;                        
                    }
                    List<EffectiveCropCycleConnectionDto> connectionDtos = effSeasonalCropCycleDto.getConnectionDtos();
                    if (connectionDtos == null) {
                        connectionDtos = new ArrayList<EffectiveCropCycleConnectionDto>(); 
                        effSeasonalCropCycleDto.setConnectionDtos(connectionDtos);
                    }
                    EffectiveCropCycleConnectionDto connDto = null;
                    // search if connection already exists
                    for (EffectiveCropCycleConnectionDto connectionDto : connectionDtos) {
                        // search with edaplosIssuerId
                        if (connectionDto.getEdaplosIssuerId() != null && connectionDto.getEdaplosIssuerId().equals(soilOccupationIssuerId)) { 
                            connDto = connectionDto;
                            break;
                        } else { // search with rank
                            for (EffectiveCropCycleNodeDto eNode : effSeasonalCropCycleDto.getNodeDtos()){
                                if (connectionDto.getTargetId().equals(eNode.getNodeId())) {
                                    if (eNode.getX() == soilOccupationRank) { // connection found
                                        connDto = connectionDto;
                                        connDto.setEdaplosIssuerId(soilOccupationIssuerId);
                                        break;
                                        // TODO 2015-03-17 eancelet : add test to control if the croppingplanentry is still the same ?
                                    }
                                }
                            }
                        }
                    }
                    if (connDto == null) { // connection doesn't exist yet                        
                        // get target and source
                        String connectionTarget = null;
                        String connectionSource = null;
                        for (EffectiveCropCycleNodeDto eNode : effSeasonalCropCycleDto.getNodeDtos()){
                            if(eNode.getX() == soilOccupationRank) {
                                connectionTarget = eNode.getNodeId();
                            } if (soilOccupationRank > 0 && eNode.getX() == (soilOccupationRank-1)) {
                                connectionSource = eNode.getNodeId();
                            } 
                            if (connectionTarget != null && connectionSource!= null) {
                                break;
                            }
                        }                        
                        if (connectionTarget == null) {
                            edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_TARGET, soilOccupationIssuerId, edaplosPlotDto.getIssuerId(), croppingPlanEntryDto.getName()));
                            continue;  
                        }
                        if (soilOccupationRank > 0 && connectionSource == null) {
                            edaplosPlotDto.addSoilOccErrorMessage(String.format(ERROR_SOIL_OCCUPATION_NO_SOURCE, soilOccupationIssuerId, edaplosPlotDto.getIssuerId(), croppingPlanEntryDto.getName()));
                            continue;                         
                        }
                                           
                        connDto = createConnectionDto(soilOccupationIssuerId, connectionSource, connectionTarget,  croppingPlanEntryDto.getTopiaId());
                        connectionDtos.add(connDto);
                    }                        
                    if (soilOccupationRank == 0 && connDto.getSourceId() == null) {
                        connDto.setSourceId(EffectiveCropCycleNodeDto.NODE_BEFORE_ID);
                    } 
                }
            }
        }        
    }

    /**
     * 
     * Get the croppingPlanEntry from domain if already existing (matching made with botanical code species, suplementary botanical code species, sowing period code and croppingPlanEntry type) 
     * 
     * @param entryDto : croppingPlanEntry we try to import
     * @param entryDtoList : list of domain's croppingPlanEntries
     */
    
    protected void getRightCroppingPlanEntry(CroppingPlanEntryDto entryDto, List<CroppingPlanEntryDto> entryDtoList) {
        for (CroppingPlanEntryDto persistedCroppingPlanEntry : entryDtoList) {
            Collection<CroppingPlanSpeciesDto> persistedCroppingPlanSpecies = Lists.newArrayList(persistedCroppingPlanEntry.getSpecies()); 
            Collection<CroppingPlanSpeciesDto> eSpecies = Lists.newArrayList(entryDto.getSpecies());
            // get persistedSpecies list without duplicates
            Map<String, CroppingPlanSpeciesDto> persistedSpeciesWithoutDuplicate = Maps.newHashMap(); 
            for (CroppingPlanSpeciesDto persistedSpecies : persistedCroppingPlanSpecies) {
                String eKey = persistedSpecies.getSpeciesEspece()+persistedSpecies.getSpeciesQualifiant()+persistedSpecies.getSpeciesTypeSaisonnier()+persistedSpecies.getVarietyId()+persistedCroppingPlanEntry.getType();
                persistedSpeciesWithoutDuplicate.put(eKey, persistedSpecies);
            }
            // get edaplos Species list without duplicates
            Map<String, CroppingPlanSpeciesDto> eSpeciesWithoutDuplicate = Maps.newHashMap(); 
            for (CroppingPlanSpeciesDto eSoilOccupationSpecies : eSpecies) { 
                String eKey = eSoilOccupationSpecies.getSpeciesEspece()+eSoilOccupationSpecies.getSpeciesQualifiant()+eSoilOccupationSpecies.getSpeciesTypeSaisonnier()+eSoilOccupationSpecies.getVarietyId()+entryDto.getType();
                eSpeciesWithoutDuplicate.put(eKey, eSoilOccupationSpecies); // if key already stored (in case of duplicate species) then element is replaced
            }                         
            if (persistedSpeciesWithoutDuplicate.size() != eSpeciesWithoutDuplicate.size()) {
                continue;
            }                                
            Collection<CroppingPlanSpeciesDto> equalSpecies = new ArrayList<CroppingPlanSpeciesDto>(); // list matching species in 2 entries
            for (Map.Entry<String, CroppingPlanSpeciesDto> persistedSpeciesElement : persistedSpeciesWithoutDuplicate.entrySet()) {                            
                for (Map.Entry<String, CroppingPlanSpeciesDto> eSoilOccupationSpeciesElement : eSpeciesWithoutDuplicate.entrySet()) {
                    if (persistedSpeciesElement.getKey().equals(eSoilOccupationSpeciesElement.getKey())) {
                        equalSpecies.add(persistedSpeciesElement.getValue());
                    }
                }                         
            }
            if (eSpeciesWithoutDuplicate.size() == equalSpecies.size() && entryDto.getTopiaId() == null) { // we found a matching croppingPlanEntry in domain
                entryDto.setTopiaId(persistedCroppingPlanEntry.getTopiaId());
                // break; // eancelet 2015-03-04 : if uncommented, lazyloading problem in fr.inra.agrosyst.services.domain.DomainServiceImpl.validPreconditions() : Preconditions.checkNotNull(speciesDto.getSpeciesId());  
            }
        }
    }
    
    
    /**
     * Construct the croppingPlanSpeciesDto list and add it to the CroppingPlanEntry 
     * eSoilOccupation and eEntry are modified
     * 
     */
    
    protected void addSpeciesToCroppingPlanEntry(List<Element> speciesXmlList, EdaplosPlotDto ePlotDto, CroppingPlanEntryDto eEntry, String eIssuerId) {
        List<CroppingPlanSpeciesDto> eSpeciesSet = new ArrayList<CroppingPlanSpeciesDto>();
        for (Element speciesElement : speciesXmlList) {
            String speciesBotanicalCode =  speciesElement.elementText("BotanicalSpeciesCode");
            String speciesSupplementaryBotanicalCode = speciesElement.elementText("SupplementaryBotanicalSpeciesCode");
            String speciesSowingPeriodCode = speciesElement.elementText("SowingPeriodCode");
            if (StringUtils.isBlank(speciesBotanicalCode)){
                ePlotDto.addSoilOccErrorMessage(String.format(ERROR_SPECIES_BOTANICALCODE_MISSING, eIssuerId));
                continue;
            }                      
            if (speciesSupplementaryBotanicalCode == null) {
                speciesSupplementaryBotanicalCode = "";
            }                        
            if (speciesSowingPeriodCode == null) {
                speciesSowingPeriodCode = "";
            }  
            // TODO 2015-02-06 eancelet : que faire du champ ligne 224, Description de l'espèce ?
            // TODO 2015-02-06 eancelet : filtrer le code de l'espèce associée dans GEVES/Plantgrape avec le référentiel RefEspeceToVariete
            // if the species is not found in referential, then error
            RefEspece refEspece = refEspeceTopiaDao.forProperties(RefEspece.PROPERTY_CODE_ESPECE_BOTANIQUE, speciesBotanicalCode, RefEspece.PROPERTY_CODE_QUALIFIANT__AEE, speciesSupplementaryBotanicalCode, RefEspece.PROPERTY_CODE_TYPE_SAISONNIER__AEE, speciesSowingPeriodCode).findAnyOrNull();
            if (refEspece == null) {
                ePlotDto.addSoilOccErrorMessage(String.format(ERROR_NO_EXISTING_SPECIES, eIssuerId, speciesBotanicalCode, speciesSupplementaryBotanicalCode, speciesSowingPeriodCode));
                continue;
            }
            // Process varieties
            List<Element> varieties = speciesElement.elements("SownCropSpeciesVariety");
            String varietyName = null;
            String varietyCode = null;
            if (varieties != null && !varieties.isEmpty()) {
                if (varieties.size()>1) { // if more than one variety per species --> error
                    Element variety = varieties.get(0);                            
                    // variety name is mandatory
                    varietyName = variety.elementText("Description");
                    if (StringUtils.isBlank(varietyName)) {
                        ePlotDto.addSoilOccErrorMessage(String.format(ERROR_VARIETY_NAME_MISSING, eIssuerId, speciesBotanicalCode, speciesSupplementaryBotanicalCode, speciesSowingPeriodCode));
                        continue;                                
                    }      
                    varietyCode = variety.elementText("Type");
                    if (StringUtils.isNotBlank(varietyCode)) {
                        if (!StringUtils.isNumeric(varietyCode)) {
                            ePlotDto.addSoilOccErrorMessage(String.format(ERROR_GEVESCODE_NOT_A_NUMBER, eIssuerId, varietyCode));
                            continue;
                        }
                        Integer varietyCodeNum = Integer.parseInt(varietyCode);
                        RefVarieteGeves refVarieteGeves = refVarieteGevesTopiaDao.forNum_DossierEquals(varietyCodeNum).findUniqueOrNull();
                        if (refVarieteGeves == null) {
                            ePlotDto.addSoilOccErrorMessage(String.format(ERROR_GEVESCODE_MISSING, eIssuerId, varietyCode));
                            continue;
                        }      
                    } else { // variety code can be null. In that case, the fitting is done according to the variety name (filtered with species code)
                        // TODO 2015-02-09 eancelet - Attention : variétés de l'espèce seulement
                        // tables refespecetovariete et refvarietegeves. Rechercher la denomination en fonction du code_espece_edi
                        // RefVarieteGeves refVarieteGeves = refVarieteGevesTopiaDao.forProperties(RefVarieteGeves.PROPERTY_DENOMINATION, propertyValue, RefVarieteGeves.PROPERTY_NUM__ESPECE__BOTANIQUE, propertyValue ).findUniqueOrNull();
                        
                        ePlotDto.addSoilOccErrorMessage(String.format(ERROR_VARIETY_MISSING, eIssuerId, varietyName));
                        continue;                                
                    }                        
                } else {
                    ePlotDto.addSoilOccErrorMessage(String.format(ERROR_MORE_THAN_ONE_VARIETY, eIssuerId, speciesBotanicalCode, speciesSupplementaryBotanicalCode, speciesSowingPeriodCode));
                    continue;                            
                }                        
            }                    
            
            // TODO 2015-02-06 eancelet : on crée le DTO ici, on se chargera dans EdaplosSericeImpl de tester si les combinaisons d'sp (= culture) existent déjà
            CroppingPlanSpeciesDto croppingPlanSpeciesDto = new CroppingPlanSpeciesDto ();
            croppingPlanSpeciesDto.setSpeciesId(refEspece.getTopiaId());
            croppingPlanSpeciesDto.setSpeciesEspece(refEspece.getLibelle_espece_botanique()); // the DTO is waiting for the label, not the code
            croppingPlanSpeciesDto.setSpeciesQualifiant(refEspece.getLibelle_qualifiant_AEE());
            croppingPlanSpeciesDto.setSpeciesTypeSaisonnier(refEspece.getLibelle_type_saisonnier_AEE());
            // Impossible to set Destination as there is no destination in edaplos input
            if (eEntry.getName() == null || eEntry.getName().isEmpty()) {
                String speciesBotanicalLabel = refEspece.getLibelle_espece_botanique();
                String speciesSupplementaryBotanicalLabel = (refEspece.getLibelle_qualifiant_AEE().isEmpty()) ? "" : ", " + refEspece.getLibelle_qualifiant_AEE();
                String speciesSowingPeriodLabel = (refEspece.getLibelle_type_saisonnier_AEE().isEmpty()) ? "" : ", " + refEspece.getLibelle_type_saisonnier_AEE();
                eEntry.setName(speciesBotanicalLabel+speciesSupplementaryBotanicalLabel+speciesSowingPeriodLabel);
            }
            if (varietyCode != null) {
                croppingPlanSpeciesDto.setVarietyId(varietyCode);
                croppingPlanSpeciesDto.setVarietyLibelle(varietyName); // TODO : voir si ce champ est vraiment nécessaire
            }                    
            eSpeciesSet.add(croppingPlanSpeciesDto);  
        }
        eEntry.setSpecies(eSpeciesSet);
    }


    
    /**
     * Return cycle type (perennial or seasonal (annuel, biannuel)) from one species
     * 
     */
    protected TypeCulture getCycleTypeFromSpecies(CroppingPlanSpeciesDto eSpecies){
        RefEspece refEspece = refEspeceTopiaDao.forTopiaIdEquals(eSpecies.getSpeciesId()).findUnique();
        RefCultureEdiGroupeCouvSol refCultureEdiGroupeCouvSol = refCultureEdiGroupeCouvSolTopiaDao.forProperties(
                RefCultureEdiGroupeCouvSol.PROPERTY_CODE_ESPECE_BOTANIQUE, refEspece.getCode_espece_botanique(), 
                RefCultureEdiGroupeCouvSol.PROPERTY_CODE_QUALIFIANT_AEE, refEspece.getCode_qualifiant_AEE(), 
                RefCultureEdiGroupeCouvSol.PROPERTY_CODE_TYPE_SAISONNIER_AEE, refEspece.getCode_type_saisonnier_AEE())
                .findAnyOrNull();
        if (refCultureEdiGroupeCouvSol == null) {
            return null;
        } else {
            return refCultureEdiGroupeCouvSol.getTypeCulture();
        }
    }
    
    
    
    /**
     *  
     */
    
    protected boolean isPlotExisting(EdaplosPlotDto plot) {
        if (plot.getTopiaId() == null || plot.getTopiaId().isEmpty()) {
            return false;
        }
        return true;                
    }
    
    /**
     *  
     */
    
    protected EffectiveCropCycleNodeDto createNodeDto(String soilOccupationIssuerId, String croppingPlanEntryTopiaId, int rank) {
        EffectiveCropCycleNodeDto nodeDto = new EffectiveCropCycleNodeDto();
        nodeDto.setEdaplosIssuerId(soilOccupationIssuerId); 
        nodeDto.setCroppingPlanEntryId(croppingPlanEntryTopiaId);
        nodeDto.setNodeId("new-node-"+UUID.randomUUID().toString());
        nodeDto.setInterventions(new ArrayList<EffectiveInterventionDto>());                                     
        nodeDto.setX(rank);
        return nodeDto;
    }
    
    
    /**
     * 
     */
    
    protected EffectiveCropCycleConnectionDto createConnectionDto(String issuerID, String sourceId, String targetId, String croppingPlanEntryId) {
        EffectiveCropCycleConnectionDto connectionDto = new EffectiveCropCycleConnectionDto();
        connectionDto.setEdaplosIssuerId(issuerID);
        connectionDto.setSourceId(sourceId);
        connectionDto.setTargetId(targetId);
        connectionDto.setIntermediateCroppingPlanEntryId(croppingPlanEntryId);        
        return connectionDto;        
    }
    
    
}