package fr.inra.agrosyst.services.domain.export;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DomainExportMetadata.java 4005 2014-04-11 14:52:55Z bleny $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/domain/export/DomainExportMetadata.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Map;

import org.apache.commons.collections4.BidiMap;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.CroppingEntryType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.GeoPoint;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatusTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefMateriel;
import fr.inra.agrosyst.api.entities.referential.RefMaterielTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefOTEX;
import fr.inra.agrosyst.api.entities.referential.RefOTEXTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalis;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalisTopiaDao;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.common.export.UniqueDualLinkedHashMap;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class DomainExportMetadata {

    /**
     * Current bean info.
     */
    public static abstract class DomainCommonBeanInfo implements EntityExportTabInfo {

        @Override
        public BidiMap<String, String> getCommonColumns() {
            BidiMap<String, String> result = new UniqueDualLinkedHashMap<String, String>();
            result.put("domainType", "Type de domaine");
            result.put("departement", "Département");
            result.put("postalCode", "Code Postal");
            result.put("domainName", "Nom du domaine");
            result.put("campaign", "Campagne");
            result.put("mainContact", "Nom de l'interlocuteur principal");
            return result;
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> result = new UniqueDualLinkedHashMap<String, String>();
            return result;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = Maps.newLinkedHashMap();
            return result;
        }

        @Override
        public Map<String, Function<String, Object>> getCustomParsers() {
            Map<String, Function<String, Object>> result = Maps.newHashMap();
            return result;
        }

        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = Maps.newHashMap();
            result.put("domainType", ExportUtils.allStringOf(DomainType.class));
            return result;
        }
    }

    /**
     * Current bean info.
     */
    public static class DomainMainBeanInfo extends DomainCommonBeanInfo {
        protected RefLegalStatusTopiaDao refLegalStatusTopiaDao;

        public void setRefLegalStatusTopiaDao(RefLegalStatusTopiaDao refLegalStatusTopiaDao) {
            this.refLegalStatusTopiaDao = refLegalStatusTopiaDao;
        }

        @Override
        public String getBeanTitle() {
            return "Généralités";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Domain.PROPERTY_DESCRIPTION, "Description");
            columns.put(Domain.PROPERTY_LEGAL_STATUS, "Statut juridique ou institutionnel");
            columns.put("petiteRegionAgricole", "Petite région agricole");
            columns.put(Domain.PROPERTY_ZONING, "Zonage du domaine");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();

            result.put("petiteRegionAgricole", new Function<K, Object>() {
                @Override
                public Object apply(K input) {
                    Preconditions.checkArgument(input instanceof Domain);
                    Domain domain = (Domain) input;
                    return domain.getLocation().getPetiteRegionAgricoleNom();
                }
            });

            Function<K, Object> getLegalStatus = ExportUtils.ifNotNull(Domain.PROPERTY_LEGAL_STATUS, new Function<RefLegalStatus, Object>() {
                @Override
                public Object apply(RefLegalStatus input) {
                    return input.getLibelle_INSEE();
                }
            });
            result.put(Domain.PROPERTY_LEGAL_STATUS, getLegalStatus);

            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(Domain.PROPERTY_LEGAL_STATUS, refLegalStatusTopiaDao.findPropertyDistinctValues(RefLegalStatus.PROPERTY_LIBELLE__INSEE));
            result.put(Domain.PROPERTY_ZONING, ExportUtils.allStringOf(Zoning.class));
            return result;
        }
    }

    /**
     * Current bean info.
     */
    public static class DomainCroppingPlanEntryBeanInfo extends DomainCommonBeanInfo {
        protected RefEspeceTopiaDao refEspeceTopiaDao;
        public void setRefEspeceTopiaDao(RefEspeceTopiaDao refEspeceTopiaDao) {
            this.refEspeceTopiaDao = refEspeceTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Assolements";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(CroppingPlanEntry.PROPERTY_NAME, "Culture");
            columns.put(CroppingPlanEntry.PROPERTY_SELLING_PRICE, "Prix de vente (€)");
            columns.put(CroppingPlanEntry.PROPERTY_TYPE, "Type");
            columns.put(RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE, "Espèce");
            columns.put(RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE, "Qualifiant");
            columns.put(RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE, "Type saisonnier");
            columns.put(RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE, "Destination");
            columns.put("label", "Cépage/Variété");
            return columns;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = super.getCustomDropDownList();
            result.put(CroppingPlanEntry.PROPERTY_TYPE, ExportUtils.allStringOf(CroppingEntryType.class));
            result.put(RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE, refEspeceTopiaDao.findPropertyDistinctValues(RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE));
            result.put(RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE, refEspeceTopiaDao.findPropertyDistinctValues(RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE));
            result.put(RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE, refEspeceTopiaDao.findPropertyDistinctValues(RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE));
            result.put(RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE, refEspeceTopiaDao.findPropertyDistinctValues(RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE));
            return result;
        }
    }

    /**
     * Current bean info.
     */
    public static class DomainGpsDataBeanInfo extends DomainCommonBeanInfo {
        @Override
        public String getBeanTitle() {
            return "Données GPS";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(GeoPoint.PROPERTY_NAME, "Nom du centre");
            columns.put(GeoPoint.PROPERTY_LATITUDE, "Latitude");
            columns.put(GeoPoint.PROPERTY_LONGITUDE, "Longitude");
            columns.put(GeoPoint.PROPERTY_DESCRIPTION, "Description");
            return columns;
        }
    }

    /**
     * Current bean info.
     */
    public static class DomainMaterielBeanInfo extends DomainCommonBeanInfo {
        protected RefMaterielTopiaDao refMaterielTopiaDao;
        public void setRefMaterielTopiaDao(RefMaterielTopiaDao refMaterielTopiaDao) {
            this.refMaterielTopiaDao = refMaterielTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Materiels";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(RefMateriel.PROPERTY_TYPE_MATERIEL1, "Matériel type 1");
            columns.put(RefMateriel.PROPERTY_TYPE_MATERIEL2, "Matériel type 2");
            columns.put(RefMateriel.PROPERTY_TYPE_MATERIEL3, "Matériel type 3");
            columns.put(RefMateriel.PROPERTY_TYPE_MATERIEL4, "Matériel type 4");
            columns.put(RefMateriel.PROPERTY_UNITE, "Unité");
            columns.put(RefMateriel.PROPERTY_UNITE_PAR_AN, "Unité par an");
            columns.put(Equipment.PROPERTY_NAME, "Nom");
            columns.put(Equipment.PROPERTY_DESCRIPTION, "Description");
            columns.put(Equipment.PROPERTY_MATERIEL_ETA, "Materiel ETA / CUMA");

            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getTypeMateriel1();
                    }
                });
                result.put(RefMateriel.PROPERTY_TYPE_MATERIEL1, getter);
            }

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getTypeMateriel2();
                    }
                });
                result.put(RefMateriel.PROPERTY_TYPE_MATERIEL2, getter);
            }

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getTypeMateriel3();
                    }
                });
                result.put(RefMateriel.PROPERTY_TYPE_MATERIEL3, getter);
            }

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getTypeMateriel4();
                    }
                });
                result.put(RefMateriel.PROPERTY_TYPE_MATERIEL4, getter);
            }

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getUnite();
                    }
                });
                result.put(RefMateriel.PROPERTY_UNITE, getter);
            }

            {
                Function<K, Object> getter = ExportUtils.ifNotNull(Equipment.PROPERTY_REF_MATERIEL, new Function<RefMateriel, Object>() {
                    @Override
                    public Object apply(RefMateriel input) {
                        return input.getUniteParAn();
                    }
                });
                result.put(RefMateriel.PROPERTY_UNITE_PAR_AN, getter);
            }

            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = Maps.newHashMap();
            result.put(RefMateriel.PROPERTY_TYPE_MATERIEL1, refMaterielTopiaDao.findPropertyDistinctValues(RefMateriel.PROPERTY_TYPE_MATERIEL1));
            result.put(RefMateriel.PROPERTY_TYPE_MATERIEL2, refMaterielTopiaDao.findPropertyDistinctValues(RefMateriel.PROPERTY_TYPE_MATERIEL2));
            result.put(RefMateriel.PROPERTY_TYPE_MATERIEL3, refMaterielTopiaDao.findPropertyDistinctValues(RefMateriel.PROPERTY_TYPE_MATERIEL3));
            result.put(RefMateriel.PROPERTY_TYPE_MATERIEL4, refMaterielTopiaDao.findPropertyDistinctValues(RefMateriel.PROPERTY_TYPE_MATERIEL4));
            // FIXME echatellier 20140330 : double value are throws class cast exception
            // result.put(RefMateriel.PROPERTY_UNITE_PAR_AN, ExportUtils.allDaoStringProperty(refMaterielTopiaDao, RefMateriel.PROPERTY_UNITE_PAR_AN));
            result.put(RefMateriel.PROPERTY_UNITE, refMaterielTopiaDao.findPropertyDistinctValues(RefMateriel.PROPERTY_UNITE));
            return result;
        }
    }

    public static class DomainToolsCouplingBeanInfo extends DomainCommonBeanInfo{

        @Override
        public String getBeanTitle() {
            return "Combinaison d'outils";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(ToolsCoupling.PROPERTY_TOOLS_COUPLING_NAME, "Nom");
            columns.put(ToolsCoupling.PROPERTY_WORKFORCE, "Nombre de personnes mobilisées");
            columns.put(ToolsCoupling.PROPERTY_COMMENT, "Commentaire");
            columns.put("agrosystInterventionTypes", "Type d'action");
            columns.put("mainsActions", "Actions principales");
            columns.put("tractor", "Matériel de Traction ou Automoteur");
            columns.put("equipments", "Outils ou matériel d'irrigation");
            return columns;
        }

    }

    /**
     * Current bean info.
     */
    public static class DomainSolBeanInfo extends DomainCommonBeanInfo {
        protected RefSolArvalisTopiaDao refSolArvalisTopiaDao;
        public void setRefSolArvalisTopiaDao(RefSolArvalisTopiaDao refSolArvalisTopiaDao) {
            this.refSolArvalisTopiaDao = refSolArvalisTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Sols";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put("region", "Region");
            columns.put("solArvalis", "Type de sol Arvalis");
            columns.put(Ground.PROPERTY_NAME, "Nom local");
            columns.put(Ground.PROPERTY_IMPORTANCE, "Importance relative du type de sol sur le domaine");
            columns.put(Ground.PROPERTY_COMMENT, "Commentaire");
            return columns;
        }

        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();

            Function<K, Object> getSolRegion = ExportUtils.ifNotNull(Ground.PROPERTY_REF_SOL_ARVALIS, new Function<RefSolArvalis, Object>() {
                @Override
                public Object apply(RefSolArvalis input) {
                    return input.getSol_region();
                }
            });
            result.put("region", getSolRegion);

            Function<K, Object> getSolNom = ExportUtils.ifNotNull(Ground.PROPERTY_REF_SOL_ARVALIS, new Function<RefSolArvalis, Object>() {
                @Override
                public Object apply(RefSolArvalis input) {
                    return input.getSol_nom();
                }
            });
            result.put("solArvalis", getSolNom);

            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = Maps.newHashMap();
            result.put("region", refSolArvalisTopiaDao.findPropertyDistinctValues(RefSolArvalis.PROPERTY_SOL_REGION));
            result.put("solArvalis", refSolArvalisTopiaDao.findPropertyDistinctValues(RefSolArvalis.PROPERTY_SOL_NOM));
            return result;
        }
    }

    /**
     * Current bean info.
     */
    public static class DomainStatusBeanInfo extends DomainCommonBeanInfo {
        protected RefOTEXTopiaDao refOTEXTopiaDao;
        public void setRefOTEXTopiaDao(RefOTEXTopiaDao refOTEXTopiaDao) {
            this.refOTEXTopiaDao = refOTEXTopiaDao;
        }
        @Override
        public String getBeanTitle() {
            return "Statut et Main d'Oeuvre";
        }

        @Override
        public BidiMap<String, String> getExtraColumns() {
            BidiMap<String, String> columns = super.getExtraColumns();
            columns.put(Domain.PROPERTY_STATUS_COMMENT, "Commentaires du statut");
            columns.put(Domain.PROPERTY_PARTNERS_NUMBER, "Nombre d'associé");
            columns.put(Domain.PROPERTY_OTHER_WORK_FORCE, "Main d'oeuvre familiale ou main d'oeuvre des associés");
            columns.put(Domain.PROPERTY_FAMILY_WORK_FORCE_WAGE, "Rémunération de la main d'oeuvre Familiale");
            columns.put(Domain.PROPERTY_PERMANENT_EMPLOYEES_WORK_FORCE, "Main d'oeuvre salariés permanents");
            columns.put(Domain.PROPERTY_TEMPORARY_EMPLOYEES_WORK_FORCE, "Main d'oeuvre salariés temporaires");
            columns.put(Domain.PROPERTY_CROPS_WORK_FORCE, "Main d'oeuvre totale affectée à la conduite des cultures");
            columns.put(Domain.PROPERTY_WAGE_COSTS, "Charges salariales");
            columns.put(Domain.PROPERTY_WORKFORCE_COMMENT, "Commentaire sur la main d'oeuvre");
            columns.put(Domain.PROPERTY_ORIENTATION, "Description de l'orientation");
            columns.put(Domain.PROPERTY_OTEX18, "Orientation technico-économique de l'exploitation agricole OTEX");
            columns.put(Domain.PROPERTY_OTEX70, "Précision OTEX");
            columns.put(Domain.PROPERTY_USED_AGRICULTURAL_AREA, "SAU totale");
            columns.put(Domain.PROPERTY_EXPERIMENTAL_AGRICULTURAL_AREA, "SAU mobilisée pour l'expérimentation");
            columns.put(Domain.PROPERTY_HOMOGENIZATION_EXPERIMENTAL_AGRICULTURAL_AREA, "SAU mobilisée pour l'expérimentation dont SAU en homogénéisation");
            columns.put(Domain.PROPERTY_MSA_FEE, "Cotisations MSA");
            columns.put(Domain.PROPERTY_AVERAGE_TENANT_FARMING, "Fermage moyen");
            columns.put(Domain.PROPERTY_DECOUPLED_ASSISTANCE, "Aides découplées");

            return columns;
        }


        @Override
        public <K> Map<String, Function<K, Object>> getCustomFormatters() {
            Map<String, Function<K, Object>> result = super.getCustomFormatters();

            Function<K, Object> getOtex18 = ExportUtils.ifNotNull(Domain.PROPERTY_OTEX18, new Function<RefOTEX, Object>() {
                @Override
                public Object apply(RefOTEX input) {
                    return input.getLibelle_OTEX_18_postes();
                }
            });
            result.put(Domain.PROPERTY_OTEX18, getOtex18);

            Function<K, Object> getOtex70 = ExportUtils.ifNotNull(Domain.PROPERTY_OTEX70, new Function<RefOTEX, Object>() {
                @Override
                public Object apply(RefOTEX input) {
                    return input.getLibelle_OTEX_70_postes();
                }
            });
            result.put(Domain.PROPERTY_OTEX70, getOtex70);

            return result;
        }
        
        @Override
        public Map<String, Iterable<String>> getCustomDropDownList() {
            Map<String, Iterable<String>> result = Maps.newHashMap();
            result.put(Domain.PROPERTY_OTEX18, refOTEXTopiaDao.findPropertyDistinctValues(RefOTEX.PROPERTY_LIBELLE__OTEX_18_POSTES));
            result.put(Domain.PROPERTY_OTEX70, refOTEXTopiaDao.findPropertyDistinctValues(RefOTEX.PROPERTY_LIBELLE__OTEX_70_POSTES));
            return result;
        }
    }

}
