package fr.inra.agrosyst.services.domain;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DomainServiceImpl.java 5112 2015-10-27 09:09:48Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/domain/DomainServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystInterventionType;
import fr.inra.agrosyst.api.entities.CroppingEntryType;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesImpl;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.DomainType;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.Equipment;
import fr.inra.agrosyst.api.entities.EquipmentTopiaDao;
import fr.inra.agrosyst.api.entities.GeoPoint;
import fr.inra.agrosyst.api.entities.GeoPointTopiaDao;
import fr.inra.agrosyst.api.entities.Ground;
import fr.inra.agrosyst.api.entities.GroundTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.PlotTopiaDao;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.ToolsCouplingTopiaDao;
import fr.inra.agrosyst.api.entities.WeatherStation;
import fr.inra.agrosyst.api.entities.WeatherStationTopiaDao;
import fr.inra.agrosyst.api.entities.Zoning;
import fr.inra.agrosyst.api.entities.action.AbstractAction;
import fr.inra.agrosyst.api.entities.action.AbstractActionTopiaDao;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.effective.EffectiveInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedInterventionTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefEspece;
import fr.inra.agrosyst.api.entities.referential.RefEspeceTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDI;
import fr.inra.agrosyst.api.entities.referential.RefInterventionAgrosystTravailEDITopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatus;
import fr.inra.agrosyst.api.entities.referential.RefLegalStatusTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.entities.referential.RefLocationTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefMateriel;
import fr.inra.agrosyst.api.entities.referential.RefMaterielAutomoteur;
import fr.inra.agrosyst.api.entities.referential.RefMaterielIrrigation;
import fr.inra.agrosyst.api.entities.referential.RefMaterielTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefOTEX;
import fr.inra.agrosyst.api.entities.referential.RefOTEXTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalis;
import fr.inra.agrosyst.api.entities.referential.RefSolArvalisTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefStationMeteo;
import fr.inra.agrosyst.api.entities.referential.RefStationMeteoTopiaDao;
import fr.inra.agrosyst.api.entities.referential.RefVariete;
import fr.inra.agrosyst.api.entities.referential.RefVarieteTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystImportException;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.common.PricesService;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlanSpeciesDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.domain.DomainExtendException;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.domain.ExtendContext;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.pz0.EntityAndDependencies;
import fr.inra.agrosyst.api.services.pz0.ImportResults;
import fr.inra.agrosyst.api.services.pz0.domains.DomainAndDependencies;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;
import fr.inra.agrosyst.services.common.CommonService;
import fr.inra.agrosyst.services.common.EntityUsageService;
import fr.inra.agrosyst.services.common.export.EntityExportExtra;
import fr.inra.agrosyst.services.common.export.EntityExportTabInfo;
import fr.inra.agrosyst.services.common.export.EntityExporter;
import fr.inra.agrosyst.services.common.export.EntityImporter;
import fr.inra.agrosyst.services.common.export.ExportUtils;
import fr.inra.agrosyst.services.domain.export.DomainExportEntity;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainCroppingPlanEntryBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainGpsDataBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainMainBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainMaterielBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainSolBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainStatusBeanInfo;
import fr.inra.agrosyst.services.domain.export.DomainExportMetadata.DomainToolsCouplingBeanInfo;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.map.MultiKeyMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public class DomainServiceImpl extends AbstractAgrosystService implements DomainService {

    private static final Log log = LogFactory.getLog(DomainServiceImpl.class);

    public static final int BUFFER_SIZE = 1024;

    protected static final Predicate<CroppingPlanSpeciesDto> IS_CROPPING_SPECIES_EMPTY = new Predicate<CroppingPlanSpeciesDto>() {
        @Override
        public boolean apply(CroppingPlanSpeciesDto input) {
            String speciesId = input.getSpeciesId();
            boolean result = Strings.isNullOrEmpty(speciesId);
            return result;
        }
    };

    protected static final Function<CroppingPlanSpeciesDto, String> GET_SPECIES_TOPIA_ID = new Function<CroppingPlanSpeciesDto, String>() {
        @Override
        public String apply(CroppingPlanSpeciesDto input) {
            return Strings.nullToEmpty(input.getTopiaId());
        }
    };

    protected static final Predicate<? super CroppingPlanEntryDto> IS_CROPPING_PLAN_ENTRY_EMPTY = new Predicate<CroppingPlanEntryDto>() {
        @Override
        public boolean apply(CroppingPlanEntryDto input) {
            boolean result = true;
            if (input != null) {
                result = Strings.isNullOrEmpty(input.getName())
                        && input.getSellingPrice() == 0d
                        && (input.getSpecies() == null || input.getSpecies().isEmpty());
            }
            return result;
        }
    };

    public static final Function<CroppingPlanEntry, String> GET_CROPPING_PLAN_ENTRY_CODE = new Function<CroppingPlanEntry, String>() {
        @Override
        public String apply(CroppingPlanEntry input) {
            String result = input.getCode();
            return result;
        }
    };

    protected static final Function<CroppingPlanSpecies, String> GET_SPECIES_CODE = new Function<CroppingPlanSpecies, String>() {
        @Override
        public String apply(CroppingPlanSpecies input) {
            String result = input.getCode();
            return result;
        }
    };

    protected GrowingPlanService growingPlanService;
    protected PlotService plotService;
    protected BusinessAuthorizationService authorizationService;
    protected PricesService pricesService;
    protected AnonymizeService anonymizeService;
    protected EntityUsageService entityUsageService;
    protected ReferentialService referentialService;

    protected DomainTopiaDao domainDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected GeoPointTopiaDao geoPointDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryDao;
    protected CroppingPlanSpeciesTopiaDao croppingPlanSpeciesDao;
    protected EquipmentTopiaDao equipmentDao;
    protected GroundTopiaDao groundDao;
    protected ToolsCouplingTopiaDao toolsCouplingDao;
    protected WeatherStationTopiaDao weatherStationDao;
    protected PlotTopiaDao plotDao;

    protected RefLegalStatusTopiaDao refLegalStatusDao;
    protected RefLocationTopiaDao locationDao;
    protected RefOTEXTopiaDao refOTEXDao;
    protected RefEspeceTopiaDao refEspeceDao;
    protected RefVarieteTopiaDao refVarieteDao;
    protected RefStationMeteoTopiaDao refStationMeteoDao;
    protected RefSolArvalisTopiaDao refSolArvalisDao;
    protected RefMaterielTopiaDao refMaterielTopiaDao;
    protected RefInterventionAgrosystTravailEDITopiaDao refInterventionAgrosystTravailEDITopiaDao;
    protected AbstractActionTopiaDao abstractActionDao;
    protected PracticedInterventionTopiaDao practicedInterventionDao;
    protected EffectiveInterventionTopiaDao effectiveInterventionDao;

    public void setGrowingPlanService(GrowingPlanService growingPlanService) {
        this.growingPlanService = growingPlanService;
    }

    public void setPlotService(PlotService plotService) {
        this.plotService = plotService;
    }

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setPricesService(PricesService pricesService) {
        this.pricesService = pricesService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setEntityUsageService(EntityUsageService entityUsageService) {
        this.entityUsageService = entityUsageService;
    }

    public void setReferentialService(ReferentialService referentialService) {
        this.referentialService = referentialService;
    }

    public void setGrowingPlanDao(GrowingPlanTopiaDao growingPlanDao) {
        this.growingPlanDao = growingPlanDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    public void setRefLegalStatusDao(RefLegalStatusTopiaDao refLegalStatusDao) {
        this.refLegalStatusDao = refLegalStatusDao;
    }

    public void setLocationDao(RefLocationTopiaDao locationDao) {
        this.locationDao = locationDao;
    }

    public void setRefOTEXDao(RefOTEXTopiaDao refOTEXDao) {
        this.refOTEXDao = refOTEXDao;
    }

    public void setRefEspeceDao(RefEspeceTopiaDao refEspeceDao) {
        this.refEspeceDao = refEspeceDao;
    }

    public void setRefVarieteDao(RefVarieteTopiaDao refVarieteDao) {
        this.refVarieteDao = refVarieteDao;
    }

    public void setCroppingPlanEntryDao(CroppingPlanEntryTopiaDao croppingPlanEntryDao) {
        this.croppingPlanEntryDao = croppingPlanEntryDao;
    }

    public void setCroppingPlanSpeciesDao(CroppingPlanSpeciesTopiaDao croppingPlanSpeciesDao) {
        this.croppingPlanSpeciesDao = croppingPlanSpeciesDao;
    }

    public void setEquipmentDao(EquipmentTopiaDao equipmentDao) {
        this.equipmentDao = equipmentDao;
    }

    public void setToolsCouplingDao(ToolsCouplingTopiaDao toolsCouplingDao) {
        this.toolsCouplingDao = toolsCouplingDao;
    }

    public void setWeatherStationDao(WeatherStationTopiaDao weatherStationDao) {
        this.weatherStationDao = weatherStationDao;
    }

    public void setRefStationMeteoDao(RefStationMeteoTopiaDao refStationMeteoDao) {
        this.refStationMeteoDao = refStationMeteoDao;
    }

    public void setAbstractActionDao(AbstractActionTopiaDao abstractActionDao) {
        this.abstractActionDao = abstractActionDao;
    }

    public void setPracticedInterventionDao(PracticedInterventionTopiaDao practicedInterventionDao) {
        this.practicedInterventionDao = practicedInterventionDao;
    }

    public void setEffectiveInterventionDao(EffectiveInterventionTopiaDao effectiveInterventionDao) {
        this.effectiveInterventionDao = effectiveInterventionDao;
    }

    public void setPlotDao(PlotTopiaDao plotDao) {
        this.plotDao = plotDao;
    }

    public void setGeoPointDao(GeoPointTopiaDao geoPointDao) {
        this.geoPointDao = geoPointDao;
    }

    public void setGroundDao(GroundTopiaDao groundDao) {
        this.groundDao = groundDao;
    }

    public void setRefSolArvalisDao(RefSolArvalisTopiaDao refSolArvalisDao) {
        this.refSolArvalisDao = refSolArvalisDao;
    }

    public void setRefMaterielTopiaDao(RefMaterielTopiaDao refMaterielTopiaDao) {
        this.refMaterielTopiaDao = refMaterielTopiaDao;
    }
    
    public void setRefInterventionAgrosystTravailEDITopiaDao(RefInterventionAgrosystTravailEDITopiaDao refInterventionAgrosystTravailEDITopiaDao) {
        this.refInterventionAgrosystTravailEDITopiaDao = refInterventionAgrosystTravailEDITopiaDao;
    }

    @Override
    public List<Domain> getAllDomains() {
        List<Domain> domains = domainDao.findAll();
        return domains;
    }

    @Override
    public Domain getDomain(String domainId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domainId));
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        Domain result = anonymizeService.checkForDomainAnonymization(domain);
        return result;
    }

    @Override
    public Domain newDomain() {
        Domain result = domainDao.newInstance();

        // Le domain est actif à sa création
        result.setActive(true);

        return result;
    }

    @Override
    public GeoPoint newGpsData() {
        GeoPoint result = geoPointDao.newInstance();
        return result;
    }

    @Override
    public WeatherStation newWeatherStation() {
        return weatherStationDao.newInstance();
    }

    @Override
    public DomainDto getDomainByCode(String domainCode) {
        Domain domain = domainDao.forCodeEquals(domainCode).findAny();
        DomainDto result = anonymizeService.getDomainToDtoFunction(false).apply(domain);
        return result;
    }

    @Override
    public Domain createOrUpdateDomain(final Domain domain, String locationId, String legalStatusId, List<GeoPoint> geoPoints,
                                       List<CroppingPlanEntryDto> croppingPlan, Integer otex18code, Integer otex70code,
                                       List<Ground> grounds, List<Equipment> equipments, List<ToolsCoupling> toolsCouplings, List<Price> prices) {
        String domainId = domain.getTopiaId();
        authorizationService.checkCreateOrUpdateDomain(domainId);

        final Domain persistedDomain = createOrUpdateDomainWithoutCommit(domain, locationId, legalStatusId, geoPoints, croppingPlan, otex18code, otex70code, grounds, equipments, toolsCouplings, prices);

        addUserAuthorization(domainId, persistedDomain);

        // Now validation is automatic https://forge.codelutin.com/issues/4549
        Domain validatedResult = validateAndCommit(persistedDomain.getTopiaId());

        return validatedResult;
    }

    protected void addUserAuthorization(String domainId, Domain persistedDomain) {
        if (StringUtils.isBlank(domainId)) {
            authorizationService.domainCreated(persistedDomain);
        }
    }

    protected Domain createOrUpdateDomainWithoutCommit(Domain domain, String locationId, String legalStatusId, List<GeoPoint> geoPoints, Collection<CroppingPlanEntryDto> croppingPlanDtos, Integer otex18code, Integer otex70code, List<Ground> grounds, List<Equipment> equipments, List<ToolsCoupling> toolsCouplings, List<Price> prices) {
        final String domainId = domain.getTopiaId();

        validPreconditions(domain, locationId, geoPoints, toolsCouplings, croppingPlanDtos);

        // Le statut légal n'est défini que pour les types FERME
        if (!DomainType.EXPLOITATION_AGRICOLE.equals(domain.getType()) || !DomainType.FERME_DE_LYCEE_AGRICOLE.equals(domain.getType())) {
            domain.setLegalStatus(null);
        }


        // set location before create (not null)
        setLocation(domain, locationId);

        domain.setUpdateDate(context.getCurrentDate());

        final Domain persistedDomain = saveDomain(domain, domainId);

        addDomainLegalStatus(legalStatusId, persistedDomain);

        addDomainOtex18(otex18code, persistedDomain);

        addDomainOtex70(otex70code, persistedDomain);

        bindDomainToGrounds(grounds, persistedDomain, domainId);

        bindDomainToGeoPoint(geoPoints, persistedDomain, domainId);

        manageEquipmentAndToolsCoupling(persistedDomain, equipments, toolsCouplings, domainId);

        manageCrops(croppingPlanDtos, persistedDomain, domainId);

        pricesService.updatePrices(prices, persistedDomain, null);
        return persistedDomain;
    }

    private void setLocation(Domain domain, String locationId) {
        RefLocation location = locationDao.forTopiaIdEquals(locationId).findUnique();
        domain.setLocation(location);
    }

    private List<CroppingPlanEntry> findExistingCroppingPlans(String domainId) {
        List<CroppingPlanEntry> existingCroppingPlan;
        if (StringUtils.isNotBlank(domainId)) {
            existingCroppingPlan = getCroppingPlan0(domainId);
        } else {
            existingCroppingPlan = Lists.newArrayList();
        }
        return existingCroppingPlan;
    }

    private Domain saveDomain(final Domain domain, final String domainId) {
        Domain result;
        if (StringUtils.isBlank(domainId)) {
            // create a random domain code, used to link domains each other
            setDomainCode(domain);

            result = domainDao.create(domain);

        } else {
            result = domainDao.update(domain);
        }
        return result;
    }

    private void setDomainCode(Domain domain) {
        if (StringUtils.isBlank(domain.getCode())) {
            domain.setCode(UUID.randomUUID().toString());
        }
    }

    private void manageCrops(Collection<CroppingPlanEntryDto> croppingPlanDtos, Domain persistedDomain, String domainId) {
        if (croppingPlanDtos != null) {

            Iterables.removeIf(croppingPlanDtos, IS_CROPPING_PLAN_ENTRY_EMPTY);

            final Map<String, RefEspece> especesIndex = Maps.newHashMap();
            final Map<String, RefVariete> varietesIndex = Maps.newHashMap();

            for (CroppingPlanEntryDto entry : croppingPlanDtos) {
                if (entry.getSpecies() != null) {
                    for (CroppingPlanSpeciesDto species : entry.getSpecies()) {
                        String speciesId = species.getSpeciesId();
                        especesIndex.put(speciesId, refEspeceDao.forTopiaIdEquals(speciesId).findUnique());

                        String varietyId = species.getVarietyId();
                        if (!Strings.isNullOrEmpty(varietyId)) {
                            varietesIndex.put(varietyId, refVarieteDao.forTopiaIdEquals(varietyId).findUnique());
                        }
                    }
                }
            }

            List<CroppingPlanEntry> existingCroppingPlan = findExistingCroppingPlans(domainId);
            List<CroppingPlanEntry> croppingPlanEntriesToDelete = Lists.newArrayList(existingCroppingPlan);
            Map<String, CroppingPlanEntry> entityEntriesIndex = Maps.uniqueIndex(existingCroppingPlan, Entities.GET_TOPIA_ID);
            for (CroppingPlanEntryDto dtoEntry : croppingPlanDtos) {
                String cpEntryId = dtoEntry.getTopiaId();
                CroppingPlanEntry entityEntry = entityEntriesIndex.get(cpEntryId);
                if (entityEntry == null) {
                    entityEntry = croppingPlanEntryDao.newInstance();
                    setCropCode(dtoEntry, entityEntry);
                } else {
                    croppingPlanEntriesToDelete.remove(entityEntry);
                }
                entityEntry.setDomain(persistedDomain);
                entityEntry.setName(dtoEntry.getName());
                entityEntry.setSellingPrice(dtoEntry.getSellingPrice());
                entityEntry.setType(dtoEntry.getType());

                // Remove useless dtos
                Collection<CroppingPlanSpeciesDto> dtoSpeciesList = dtoEntry.getSpecies();
                if (dtoSpeciesList == null) {
                    dtoSpeciesList = Lists.newArrayList();
                    dtoEntry.setSpecies(dtoSpeciesList);
                }
                List<CroppingPlanSpecies> species = removeUnusedCroppingPlanSpecies(entityEntry, dtoSpeciesList);

                // Make an index of existing entities
                Map<String, CroppingPlanSpecies> speciesById = Maps.uniqueIndex(species, Entities.GET_TOPIA_ID);

                for (CroppingPlanSpeciesDto dtoSpecies : dtoSpeciesList) {
                    final String speciesTopiaId = dtoSpecies.getTopiaId();
                    CroppingPlanSpecies entitySpecies;
                    if (Strings.isNullOrEmpty(speciesTopiaId)) {
                        entitySpecies = croppingPlanSpeciesDao.newInstance();
                        String speciesCode = StringUtils.isBlank(dtoSpecies.getCode()) ? UUID.randomUUID().toString() : dtoSpecies.getCode();
                        entitySpecies.setCode(speciesCode);
                        entityEntry.addCroppingPlanSpecies(entitySpecies);
                    } else {
                        entitySpecies = speciesById.get(speciesTopiaId);
                        Preconditions.checkState(entitySpecies != null, "CroppingPlanSpecies non trouvée: " + speciesTopiaId);
                    }

                    String especeId = dtoSpecies.getSpeciesId();
                    RefEspece refEspece = especesIndex.get(especeId);
                    Preconditions.checkState(refEspece != null, "Espece non trouvée: " + especeId);
                    entitySpecies.setSpecies(refEspece);

                    String varietyId = dtoSpecies.getVarietyId();
                    if (!Strings.isNullOrEmpty(varietyId)) {
                        RefVariete refVariete = varietesIndex.get(varietyId);
                        Preconditions.checkState(refVariete != null, "Variété non trouvée: " + varietyId);
                        entitySpecies.setVariety(refVariete);
                        Preconditions.checkArgument(referentialService.validVarietesFromCodeEspeceEdi(refVariete, refEspece.getCode_espece_botanique()), String.format("Variété %s non valide pour l'espèce %s", refVariete.getLabel(), refEspece.getLibelle_espece_botanique()));
                    }
                }

                if (entityEntry.isPersisted()) {
                    croppingPlanEntryDao.update(entityEntry);
                } else {
                    croppingPlanEntryDao.create(entityEntry);
                }
            }

            // All cropping plan species belonging to any cropping plan entry to delete has to be deleted
            for (CroppingPlanEntry croppingPlanEntryToDelete : croppingPlanEntriesToDelete) {
                List<CroppingPlanSpecies> croppingPlanSpeciesToDelete = croppingPlanEntryToDelete.getCroppingPlanSpecies();
                if (croppingPlanSpeciesToDelete != null) {
                    croppingPlanEntryToDelete.clearCroppingPlanSpecies();
                    croppingPlanSpeciesDao.deleteAll(croppingPlanSpeciesToDelete);
                }
            }
            // Now species are deleted, delete cropping plan entry
            croppingPlanEntryDao.deleteAll(croppingPlanEntriesToDelete);

        }
    }

    protected void setCropCode(CroppingPlanEntryDto dtoEntry, CroppingPlanEntry entityEntry) {
        if (StringUtils.isBlank(dtoEntry.getCode())) {
            entityEntry.setCode(UUID.randomUUID().toString());
        } else {
            entityEntry.setCode(dtoEntry.getCode());
        }
    }

    protected List<CroppingPlanSpecies> removeUnusedCroppingPlanSpecies(CroppingPlanEntry entityEntry, Collection<CroppingPlanSpeciesDto> dtoSpeciesList) {
        Iterables.removeIf(dtoSpeciesList, IS_CROPPING_SPECIES_EMPTY);

        // Make an index for entities having an id
        final ImmutableListMultimap<String, CroppingPlanSpeciesDto> dtoSpeciesIndex = Multimaps.index(dtoSpeciesList, GET_SPECIES_TOPIA_ID);

        // Remove unused entities
        List<CroppingPlanSpecies> entitySpeciesList = entityEntry.getCroppingPlanSpecies();
        if (entitySpeciesList == null) {
            entitySpeciesList = Lists.newArrayList();
        }
        Iterables.removeIf(entitySpeciesList, new Predicate<CroppingPlanSpecies>() {
            @Override
            public boolean apply(final CroppingPlanSpecies species) {
                String speciesTopiaId = species.getTopiaId();
                boolean result = !dtoSpeciesIndex.containsKey(speciesTopiaId);
                return result;
            }
        });
        return entitySpeciesList;
    }

    protected void manageEquipmentAndToolsCoupling(final Domain persistedDomain, List<Equipment> equipments, List<ToolsCoupling> toolsCouplings, String domainId) {
        final Map<String, Equipment> equipmentsCache = Maps.newHashMap();

        List<Equipment> existingEquipments;
        List<ToolsCoupling> existingToolsCouplings;
        if (StringUtils.isBlank(domainId)) {
            existingEquipments = new ArrayList<Equipment>();
            existingToolsCouplings = new ArrayList<ToolsCoupling>();
        } else {
            existingEquipments = equipmentDao.forDomainEquals(persistedDomain).findAll();
            existingToolsCouplings = toolsCouplingDao.forDomainEquals(persistedDomain).findAll();
        }

        List<Equipment> equipmentsToDelete = bindDomainToEquipments(persistedDomain, equipments, equipmentsCache, existingEquipments);

        List<ToolsCoupling> toolsCouplingsToDelete = bindDomainToToolsCouplings(persistedDomain, toolsCouplings, equipmentsCache, existingToolsCouplings);

        // Now delete the toolsCoupling and equipments in the correct order
        deleteNotUsedToolsCouplings(toolsCouplingsToDelete);

        equipmentDao.deleteAll(equipmentsToDelete);
    }

    protected void deleteNotUsedToolsCouplings(List<ToolsCoupling> toolsCouplingsToDelete) {
        for (ToolsCoupling toolsCouplingsToDel : toolsCouplingsToDelete) {
            // XXX for ticket #4812: Problèmes à la suppression d'un matériel
            toolsCouplingsToDel.setTractor(null);
            toolsCouplingDao.delete(toolsCouplingsToDel);
        }
    }

    // Precondition have to be validated before
    protected List<ToolsCoupling> bindDomainToToolsCouplings(final Domain persistedDomain, List<ToolsCoupling> toolsCouplings, final Map<String, Equipment> equipmentsCache, List<ToolsCoupling> existingToolsCouplings) {
        return easyBindNoDelete(toolsCouplingDao, existingToolsCouplings, toolsCouplings, new Function<ToolsCoupling, Void>()
            {
                @Override
                public Void apply(ToolsCoupling input) {
                    if (StringUtils.isBlank(input.getCode())) {
                        input.setCode(UUID.randomUUID().toString());
                    }
                    input.setDomain(persistedDomain);
                    return null;
                }
            }, new Function<ToolsCoupling, Void>() {
                @Override
                public Void apply(ToolsCoupling input) {
                    // Replace with the correct tractor
                    Equipment tractor = input.getTractor();
                    if (tractor != null) {
                        String tractorId = input.getTractor().getTopiaId();
                        Equipment domainTractor = equipmentsCache.get(tractorId);
                        input.setTractor(domainTractor);
                    }

                    // Inject the good topiaId instead
                    Collection<Equipment> equipments = input.getEquipments();
                    if (equipments != null) {
                        for (Equipment equipment : equipments) {
                            String equipmentId = equipment.getTopiaId();
                            Equipment fromCache = equipmentsCache.get(equipmentId);

                            if (fromCache != null) {
                                equipment.setTopiaId(fromCache.getTopiaId());
                            }
                        }
                    }

                    return null;
                }
            }, ToolsCoupling.PROPERTY_DOMAIN
        );
    }

    protected List<Equipment> bindDomainToEquipments(final Domain persistedDomain, List<Equipment> equipments, final Map<String, Equipment> equipmentsCache, List<Equipment> existingEquipments) {
        return easyBindNoDelete(equipmentDao, existingEquipments, equipments, new Function<Equipment, Void>() {
                        @Override
                        public Void apply(Equipment input) {
                            if (StringUtils.isBlank(input.getCode())) {
                                input.setCode(UUID.randomUUID().toString());
                            }
                            input.setDomain(persistedDomain);
                            return null;
                        }
                    }, new Function<Equipment, Void>() {
                        @Override
                        public Void apply(Equipment input) {
                            String topiaId = input.getTopiaId();
                            equipmentsCache.put(topiaId, input);
                            if (topiaId.startsWith(NEW_EQUIPMENT)) {
                                input.setTopiaId(null);
                            }
                            return null;
                        }
                    }, Equipment.PROPERTY_DOMAIN
            );
    }

    protected void bindDomainToGeoPoint(List<GeoPoint> geoPoints, final Domain persistedDomain, final  String domainId) {
        List<GeoPoint> existingGeoPoints = StringUtils.isBlank(domainId) ? new ArrayList<GeoPoint>() : geoPointDao.forDomainEquals(persistedDomain).findAll();
        easyBind(geoPointDao, existingGeoPoints, geoPoints, new Function<GeoPoint, Void>() {
            @Override
            public Void apply(GeoPoint input) {
                input.setDomain(persistedDomain);
                return null;
            }
        }, GeoPoint.PROPERTY_DOMAIN);
    }

    protected void bindDomainToGrounds(List<Ground> grounds, final Domain persistedDomain, final String domainId) {
        List<Ground> existingGrounds = StringUtils.isBlank(domainId) ? new ArrayList<Ground>() : groundDao.forDomainEquals(persistedDomain).findAll();
        easyBind(groundDao, existingGrounds, grounds, new Function<Ground, Void>() {
            @Override
            public Void apply(Ground input) {
                input.setDomain(persistedDomain);
                return null;
            }
        }, Ground.PROPERTY_DOMAIN);
    }

    protected void addDomainOtex70(Integer otex70code, Domain persistedDomain) {
        if (otex70code != null) {
            RefOTEX refOtex70 = refOTEXDao.forCode_OTEX_70_postesEquals(otex70code).findAny();
            persistedDomain.setOtex70(refOtex70);
        } else {
            persistedDomain.setOtex70(null);
        }
    }

    protected void addDomainOtex18(Integer otex18code, Domain persistedDomain) {
        if (otex18code != null) {
            RefOTEX refOtex18 = refOTEXDao.forCode_OTEX_18_postesEquals(otex18code).findAny();
            persistedDomain.setOtex18(refOtex18);
        } else {
            persistedDomain.setOtex18(null);
        }
    }

    protected void addDomainLegalStatus(String legalStatusId, Domain persistedDomain) {
        if (!StringUtils.isBlank(legalStatusId) && (persistedDomain.getType().equals(DomainType.EXPLOITATION_AGRICOLE) || persistedDomain.getType().equals(DomainType.FERME_DE_LYCEE_AGRICOLE))) {
            RefLegalStatus legalStatus = refLegalStatusDao.forTopiaIdEquals(legalStatusId).findUnique();
            persistedDomain.setLegalStatus(legalStatus);
        }
    }

    protected void validPreconditions(Domain domain, String locationId, List<GeoPoint> geoPoints,  List<ToolsCoupling> toolsCouplings,Collection<CroppingPlanEntryDto> croppingPlanEntryDtos) {
        if (geoPoints != null) {
            for (GeoPoint geoPoint : geoPoints) {
                Preconditions.checkArgument(StringUtils.isNotBlank(geoPoint.getName()), "Le nom du centre est obligatoire sur une coordonées d'un centre opérationnels");
            }
        }
        if (toolsCouplings != null) {
            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                Preconditions.checkArgument(StringUtils.isNoneBlank(toolsCoupling.getToolsCouplingName()), "Aucun nom de définie sur la combinaison d'outil");
                Preconditions.checkArgument(CollectionUtils.isNotEmpty(toolsCoupling.getMainsActions()), "Aucune action principale de définie");
                if (toolsCoupling.isManualIntervention()){
                    Preconditions.checkArgument(toolsCoupling.getTractor() == null);
                    Preconditions.checkArgument(CollectionUtils.isEmpty(toolsCoupling.getEquipments()));
                } else {
                    Collection<Equipment>  tcEquipments = toolsCoupling.getEquipments();
                    boolean irrigationEquipmentsPresent = false;
                    boolean nonIrrigationEquipmentsPresent = false;
                    if (tcEquipments != null && tcEquipments.size() > 0) {
                        for (Equipment tcEquipment : tcEquipments) {
                            Preconditions.checkArgument(StringUtils.isNotBlank(tcEquipment.getName()), "Aucun nom de définit sur l'eqipement");
                            boolean isIrrigationEquipment = tcEquipment.getRefMateriel() != null && tcEquipment.getRefMateriel() instanceof RefMaterielIrrigation;
                            if (isIrrigationEquipment) {
                                irrigationEquipmentsPresent = true;
                            } else {
                                nonIrrigationEquipmentsPresent = true;
                            }
                        }
                    }

                    Equipment tractor = toolsCoupling.getTractor();
                    if (tractor == null) {
                        Preconditions.checkArgument(irrigationEquipmentsPresent,"Combinaison d'outils non valide, aucun tracteur/automoteur/matériel d'irrigation présent");
                        Preconditions.checkArgument(!nonIrrigationEquipmentsPresent,"Combinaison d'outils non valide, du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                    } else {
                        // if tractor is not auto-motor's type, it must be associated to 1+n equipment
                        if (tractor.getRefMateriel() == null || !(tractor.getRefMateriel() instanceof RefMaterielAutomoteur)) {
                            Preconditions.checkArgument(CollectionUtils.isNotEmpty(tcEquipments), "Combinaison d'outils non valide, aucun équipement présent avec le tracteur");
                        }
                        if (CollectionUtils.isNotEmpty(tcEquipments)) {
                            Preconditions.checkArgument(!(irrigationEquipmentsPresent && nonIrrigationEquipmentsPresent), "Combinaison d'outils non valide, du matériel d'irrigation ne peut être associé avec du matériel autre qu'un tracteur ou un automoteur");
                        }
                    }
                }
            }
        }
        if (croppingPlanEntryDtos != null) {
            for (CroppingPlanEntryDto croppingPlanEntryDto : croppingPlanEntryDtos) {
                Preconditions.checkArgument(StringUtils.isNotBlank(croppingPlanEntryDto.getName()));
                Preconditions.checkNotNull(croppingPlanEntryDto.getType(), "Type de culture non défini");
                Collection<CroppingPlanSpeciesDto> speciesDtos = croppingPlanEntryDto.getSpecies();
                if (CollectionUtils.isNotEmpty(speciesDtos)) {
                    for (CroppingPlanSpeciesDto speciesDto : speciesDtos) {
                        Preconditions.checkNotNull(speciesDto.getSpeciesId(), "Identifiant de l'espèce non renseigné");
                    }
                }
            }
        }

        Preconditions.checkArgument(CommonService.ARE_CAMPAIGNS_VALIDS(Integer.toString(domain.getCampaign())), "Campagne non valide pour le domaine");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domain.getMainContact()), "Aucun contact de renseigné");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domain.getName()), "Le nom du domaine est obligatoire");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(locationId), "Commune non renseignée");
        Preconditions.checkNotNull(domain.getType(), "Le type de domaine est obligatoire");
    }

    @Override
    public ResultList<Domain> getFilteredDomains(DomainFilter filter) {
        ResultList<Domain> result = getFilteredDomains0(filter);
        result = anonymizeService.checkForDomainsAnonymization(result);
        return result;
    }

    @Override
    public List<Domain> getDomainWithName(String name) {
        Preconditions.checkNotNull(name);
        List<Domain> domains = domainDao.forNameEquals(name).findAll();
        return domains;
    }

    protected ResultList<Domain> getFilteredDomains0(DomainFilter filter) {
        ResultList<Domain> result = domainDao.getFilteredDomains(filter, getSecurityContext());
        return result;
    }

    @Override
    public ResultList<DomainDto> getFilteredDomainsDto(DomainFilter filter) {
        ResultList<Domain> domains = getFilteredDomains0(filter);
        ResultList<DomainDto> result = ResultList.transform(domains, anonymizeService.getDomainToDtoFunction(true));
        return result;
    }

    @Override
    public List<DomainDto> getDomains(Collection<String> domainIds) {
        List<Domain> domains = domainDao.forTopiaIdIn(domainIds).findAll();
        List<DomainDto> result = Lists.transform(domains, anonymizeService.getDomainToDtoFunction(true));
        return result;
    }

    @Override
    public void unactivateDomains(List<String> domainIds, boolean activate) {
        if (domainIds != null && !domainIds.isEmpty()) {

            for (String domainId : domainIds) {
                Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
                domain.setActive(activate);
                domain.setUpdateDate(context.getCurrentDate());
                domainDao.update(domain);
            }
            getTransaction().commit();
        }
    }

    @Override
    public List<RefLegalStatus> getAllRefLegalStatus() {
        List<RefLegalStatus> refAllLegalStatus = refLegalStatusDao.findAll();

        return refAllLegalStatus;
    }

    @Override
    public RefStationMeteo findRefStationMeteoByTopiaId(String topiaId) {
        return refStationMeteoDao.forTopiaIdEquals(topiaId).findUnique();
    }

    @Override
    public Equipment newMateriel() {
        Equipment result = equipmentDao.newInstance();
        return result;
    }

    @Override
    public Ground newSol() {
        Ground result = groundDao.newInstance();
        return result;
    }

    @Override
    public Domain extendDomain(String domainTopiaId, int extendCampaign) throws DomainExtendException {

        Domain clonedDomainDraft;

        // get domain to duplicate
        Domain sourceDomain = domainDao.forTopiaIdEquals(domainTopiaId).findUnique();

        // get all related domains
        // find the last one
        // check that the extend campaign year is valid
        LinkedHashMap<Integer, String> relatedDomains = domainDao.findAllRelatedDomains(sourceDomain.getCode());

        if (relatedDomains != null && !relatedDomains.isEmpty()) {
            Integer closestLowerCampaign = null;
            for (Map.Entry<Integer, String> relatedDomain : relatedDomains.entrySet()) {
                Integer currentCampaign = relatedDomain.getKey();
                // The campaign must not by copy on a already copied campaign.
                if (currentCampaign == extendCampaign) {
                    throw new DomainExtendException("The domain is already extended for this campaign");

                    // The closest lower campaign from the required one is chosen.
                } else if (currentCampaign < extendCampaign && ((closestLowerCampaign != null && (currentCampaign > closestLowerCampaign)) || (closestLowerCampaign == null))) {
                    closestLowerCampaign = currentCampaign;
                }
            }
            // case where all campaign are > to the required extend campaign
            //  in this case the first next campaign is return.
            Set<Integer> relatedCampaigns = relatedDomains.keySet();
            if (closestLowerCampaign == null) {
                closestLowerCampaign = Iterables.getLast(relatedCampaigns);
            }
            String lastDomainId = relatedDomains.get(closestLowerCampaign);
            sourceDomain = domainDao.forTopiaIdEquals(lastDomainId).findUnique();
        } else if (sourceDomain.getCampaign() == extendCampaign) {
            throw new DomainExtendException("The domain is already extended for this campaign");
        }

        Map<Ground, Ground> groundCache = Maps.newHashMap();
        try {
            // perform clone
            Binder<Domain, Domain> domainBinder = BinderFactory.newBinder(Domain.class);
            clonedDomainDraft = domainDao.newInstance();
            domainBinder.copyExcluding(sourceDomain, clonedDomainDraft,
                    Domain.PROPERTY_TOPIA_ID,
                    Domain.PROPERTY_TOPIA_VERSION,
                    Domain.PROPERTY_TOPIA_CREATE_DATE,
                    Domain.PROPERTY_WEATHER_STATIONS,
                    Domain.PROPERTY_DEFAULT_WEATHER_STATION);
            clonedDomainDraft.setTopiaId(null);
            clonedDomainDraft.setCampaign(extendCampaign);

            if (sourceDomain.getWeatherStations() != null) {
                final Binder<WeatherStation, WeatherStation> weatherStationBinder = BinderFactory.newBinder(WeatherStation.class);
                for (WeatherStation ws : sourceDomain.getWeatherStations()) {
                    WeatherStation clone = weatherStationDao.newInstance();
                    weatherStationBinder.copyExcluding(ws, clone,
                            WeatherStation.PROPERTY_TOPIA_ID);
                    clonedDomainDraft.addWeatherStations(clone);
                    // default
                    if (ws.equals(sourceDomain.getDefaultWeatherStation())) {
                        clonedDomainDraft.setDefaultWeatherStation(clone);
                    }
                }
            }

            // Now the new domain is automatically validated https://forge.codelutin.com/issues/4549
            Date currentDate = context.getCurrentDate();
            clonedDomainDraft.setUpdateDate(currentDate);
            clonedDomainDraft.setValidated(true);
            clonedDomainDraft.setValidationDate(currentDate);

            // persist clone
            final Domain clonedDomain = domainDao.create(clonedDomainDraft);

            // duplicate collections
            List<GeoPoint> geoPoints = geoPointDao.forDomainEquals(sourceDomain).findAll();
            if (geoPoints != null && !geoPoints.isEmpty()) {
                final Binder<GeoPoint, GeoPoint> gpsDataBinder = BinderFactory.newBinder(GeoPoint.class);
                for (GeoPoint geoPoint : geoPoints) {

                    GeoPoint clone = geoPointDao.newInstance();
                    gpsDataBinder.copyExcluding(geoPoint, clone,
                            GeoPoint.PROPERTY_TOPIA_ID,
                            GeoPoint.PROPERTY_TOPIA_VERSION,
                            GeoPoint.PROPERTY_TOPIA_CREATE_DATE);
                    clone.setDomain(clonedDomain);
                    geoPointDao.create(clone);
                }
            }

            List<Ground> grounds = groundDao.forDomainEquals(sourceDomain).findAll();
            if (grounds != null && !grounds.isEmpty()) {
                Binder<Ground, Ground> solBinder = BinderFactory.newBinder(Ground.class);
                for (Ground sol : grounds) {
                    Ground cloneSol = groundDao.newInstance();
                    solBinder.copyExcluding(sol, cloneSol,
                            Ground.PROPERTY_TOPIA_ID,
                            Ground.PROPERTY_TOPIA_VERSION,
                            Ground.PROPERTY_TOPIA_CREATE_DATE);
                    cloneSol.setDomain(clonedDomain);
                    cloneSol = groundDao.create(cloneSol);
                    groundCache.put(sol, cloneSol);
                }
            }

            // The map's key is the topiaId of the equipement which has been used as source for duplication
            final Map<String, Equipment> equipmentClones = new HashMap<String, Equipment>();
            List<Equipment> equipments = equipmentDao.forDomainEquals(sourceDomain).findAll();
            if (equipments != null && !equipments.isEmpty()) {
                final Binder<Equipment, Equipment> binder = BinderFactory.newBinder(Equipment.class);
                for (Equipment equipment : equipments) {

                    Equipment clone = equipmentDao.newInstance();
                    binder.copyExcluding(equipment, clone,
                            Equipment.PROPERTY_TOPIA_ID,
                            Equipment.PROPERTY_TOPIA_VERSION,
                            Equipment.PROPERTY_TOPIA_CREATE_DATE);
                    clone.setDomain(clonedDomain);
                    clone = equipmentDao.create(clone);
                    equipmentClones.put(equipment.getTopiaId(), clone);
                }
            }

            List<ToolsCoupling> toolsCouplings = toolsCouplingDao.forDomainEquals(sourceDomain).findAll();
            if (toolsCouplings != null && !toolsCouplings.isEmpty()) {
                for (ToolsCoupling toolsCoupling : toolsCouplings) {
                    Binder<ToolsCoupling, ToolsCoupling> binder = BinderFactory.newBinder(ToolsCoupling.class);
                    ToolsCoupling clonedToolsCoupling = toolsCouplingDao.newInstance();
                    binder.copyExcluding(toolsCoupling, clonedToolsCoupling,
                            ToolsCoupling.PROPERTY_TOPIA_ID,
                            ToolsCoupling.PROPERTY_TOPIA_VERSION,
                            ToolsCoupling.PROPERTY_TOPIA_CREATE_DATE,
                            ToolsCoupling.PROPERTY_MAINS_ACTIONS,
                            ToolsCoupling.PROPERTY_TRACTOR,
                            ToolsCoupling.PROPERTY_EQUIPMENTS);

                    clonedToolsCoupling.setDomain(clonedDomain);
                    if (toolsCoupling.getTractor() != null) {
                        clonedToolsCoupling.setTractor(equipmentClones.get(toolsCoupling.getTractor().getTopiaId()));
                    }
                    List<RefInterventionAgrosystTravailEDI> clonedMainsActions = Lists.newArrayList(toolsCoupling.getMainsActions());
                    clonedToolsCoupling.setMainsActions(clonedMainsActions);

                    if (toolsCoupling.getEquipments() != null) {
                        List<Equipment> clonedEquipements = Lists.newArrayList();
                        for (Equipment equipement : toolsCoupling.getEquipments()) {
                            clonedEquipements.add(equipmentClones.get(equipement.getTopiaId()));
                        }
                        clonedToolsCoupling.setEquipments(clonedEquipements);
                    }

                    toolsCouplingDao.create(clonedToolsCoupling);
                }
            }

            // others entities to clone
            ExtendContext extendContext = new ExtendContext();
            extendContext.setGroundCache(groundCache);
            List<GrowingPlan> growingPlans = growingPlanDao.forDomainEquals(sourceDomain).addEquals(GrowingPlan.PROPERTY_ACTIVE, true).findAll();
            for (GrowingPlan growingPlan : growingPlans) {
                growingPlanService.duplicateGrowingPlan(extendContext, clonedDomain, growingPlan, true);
            }

            List<Plot> plots = plotDao.forDomainEquals(sourceDomain).addEquals(Plot.PROPERTY_ACTIVE, true).findAll();
            for (Plot plot : plots) {
                Plot clone = plotService.extendPlot(extendContext, clonedDomain, plot);

                // defined in GrowingPlanService#duplicateGrowingPlan
                Map<GrowingSystem, GrowingSystem> growingSystemCache = extendContext.getGrowingSystemCache();
                if (growingSystemCache != null) {
                    GrowingSystem cloneGrowingSystem = growingSystemCache.get(plot.getGrowingSystem());
                    clone.setGrowingSystem(cloneGrowingSystem);
                }
            }

            // Assolement (croppingPlan)
            List<CroppingPlanEntry> croppingPlan = getCroppingPlan0(sourceDomain.getTopiaId());
            if (croppingPlan != null) {
                Binder<CroppingPlanEntry, CroppingPlanEntry> binder = BinderFactory.newBinder(CroppingPlanEntry.class);
                final Binder<CroppingPlanSpecies, CroppingPlanSpecies> speciesBinder = BinderFactory.newBinder(CroppingPlanSpecies.class);
                for (CroppingPlanEntry cpEntry : croppingPlan) {
                    final CroppingPlanEntry clonedCpEntry = croppingPlanEntryDao.newInstance();
                    binder.copyExcluding(cpEntry, clonedCpEntry,
                            CroppingPlanEntry.PROPERTY_TOPIA_ID,
                            CroppingPlanEntry.PROPERTY_TOPIA_VERSION,
                            CroppingPlanEntry.PROPERTY_TOPIA_CREATE_DATE,
                            CroppingPlanEntry.PROPERTY_CROPPING_PLAN_SPECIES);
                    clonedCpEntry.setDomain(clonedDomain);
                    croppingPlanEntryDao.create(clonedCpEntry);
                    if (cpEntry.getCroppingPlanSpecies() != null) {
                        clonedCpEntry.setCroppingPlanSpecies(ImmutableList.copyOf(Lists.transform(cpEntry.getCroppingPlanSpecies(), new Function<CroppingPlanSpecies, CroppingPlanSpecies>() {
                            @Override
                            public CroppingPlanSpecies apply(CroppingPlanSpecies input) {
                                CroppingPlanSpecies clone = croppingPlanSpeciesDao.newInstance();
                                speciesBinder.copyExcluding(input, clone,
                                        CroppingPlanSpecies.PROPERTY_TOPIA_ID,
                                        CroppingPlanSpecies.PROPERTY_TOPIA_VERSION,
                                        CroppingPlanSpecies.PROPERTY_TOPIA_CREATE_DATE);
                                clone.setCroppingPlanEntry(clonedCpEntry);
                                CroppingPlanSpecies result = croppingPlanSpeciesDao.create(clone);
                                return result;
                            }
                        })));
                    }
                }
            }

            getTransaction().commit();
            return clonedDomain;
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't clone object", ex);
        }

    }

    @Override
    public LinkedHashMap<Integer, String> getRelatedDomains(String domainCode) {
        LinkedHashMap<Integer, String> result = domainDao.findAllRelatedDomains(domainCode);
        return result;
    }

    @Override
    public ToolsCoupling newToolsCoupling() {
        ToolsCoupling result = toolsCouplingDao.newInstance();
        return result;
    }

    protected List<CroppingPlanEntry> getCroppingPlan0(String domainId) {

        String propertyDomainId = CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
        List<CroppingPlanEntry> result = croppingPlanEntryDao.forProperties(propertyDomainId, domainId).setOrderByArguments(CroppingPlanEntry.PROPERTY_NAME).findAll();
        return result;
    }

    protected List<CroppingPlanSpecies> getCroppingPlanSpecies(String domainId) {

        String propertyDomainId = CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
        List<CroppingPlanSpecies> result = croppingPlanSpeciesDao.forProperties(propertyDomainId, domainId).findAll();
        return result;
    }

    @Override
    public InputStream exportDomainAsXlsStream(List<String> domainIds) {

        Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet = Maps.newLinkedHashMap();

        DomainMainBeanInfo domainMainTabInfo = newInstance(DomainMainBeanInfo.class);
        DomainGpsDataBeanInfo geoPointTabInfo = newInstance(DomainGpsDataBeanInfo.class);
        DomainSolBeanInfo groundTabInfo = newInstance(DomainSolBeanInfo.class);
        DomainStatusBeanInfo domainStatusTabInfo = newInstance(DomainStatusBeanInfo.class);
        DomainMaterielBeanInfo equipmentTabInfo = newInstance(DomainMaterielBeanInfo.class);
        DomainToolsCouplingBeanInfo domainToolsCouplingBeanInfo = newInstance(DomainToolsCouplingBeanInfo.class);
        DomainCroppingPlanEntryBeanInfo croppingPlanTabInfo = newInstance(DomainCroppingPlanEntryBeanInfo.class);

        // add all sheet (necessary for empty export : model)
        ExportUtils.addAllBeanInfo(sheet, domainMainTabInfo, geoPointTabInfo, groundTabInfo, domainStatusTabInfo, equipmentTabInfo,
                domainToolsCouplingBeanInfo, croppingPlanTabInfo);

        // Cropping plan entries are too complex to be simplified
        List<DomainExportEntity> croppingPlans = (List<DomainExportEntity>)sheet.get(croppingPlanTabInfo);
        List<DomainExportEntity> toolsCouplingsExports = (List<DomainExportEntity>)sheet.get(domainToolsCouplingBeanInfo);

        try {
            if (CollectionUtils.isNotEmpty(domainIds)) {
                Iterable<Domain> domains = domainDao.forTopiaIdIn(domainIds).findAllLazy(100);
                for (Domain domain : domains) {

                    // anonymize domain
                    domain = anonymizeService.checkForDomainAnonymization(domain);

                    // Common data for all tabs
                    DomainExportEntity model = new DomainExportEntity();
                    model.setDomainType(domain.getType());
                    model.setDomainName(domain.getName());
                    model.setCampaign(domain.getCampaign());
                    model.setPostalCode(domain.getLocation().getCodePostal());
                    model.setMainContact(domain.getMainContact());
                    model.setDepartement(domain.getLocation().getDepartement());
    
                    // main tab
                    ExportUtils.export(sheet, model, domain, domainMainTabInfo);
    
                    // gps data tab
                    List<GeoPoint> domainGeoPoints = geoPointDao.forDomainEquals(domain).findAll();
                    ExportUtils.export(sheet, model, domainGeoPoints, geoPointTabInfo);
    
                    // sol data tab
                    List<Ground> grounds = groundDao.forDomainEquals(domain).findAll();
                    ExportUtils.export(sheet, model, grounds, groundTabInfo);
    
                    // status data tab
                    ExportUtils.export(sheet, model, domain, domainStatusTabInfo);
    
                    // materiel tab
                    List<Equipment> domainEquipments = equipmentDao.forDomainEquals(domain).findAll();
                    ExportUtils.export(sheet, model, domainEquipments, equipmentTabInfo);
    
                    // toolsCouplings tab
                    List<ToolsCoupling> toolsCouplings = toolsCouplingDao.forDomainEquals(domain).findAll();
                    for (ToolsCoupling toolsCoupling : toolsCouplings) {
                        Collection<RefInterventionAgrosystTravailEDI> mainsActions = toolsCoupling.getMainsActions();
                        for (RefInterventionAgrosystTravailEDI mainAction : mainsActions) {
                            Collection<Equipment> equipments = toolsCoupling.getEquipments();
                            if (equipments.isEmpty()) {
                                DomainExportEntity export = (DomainExportEntity) model.clone();
                                ExportUtils.copyFields(toolsCoupling, export, null,
                                        ToolsCoupling.PROPERTY_TOOLS_COUPLING_NAME,
                                        ToolsCoupling.PROPERTY_WORKFORCE,
                                        ToolsCoupling.PROPERTY_COMMENT);
                                ExportUtils.setExtraField(export, "agrosystInterventionTypes", mainAction.getIntervention_agrosyst());
                                ExportUtils.setExtraField(export, "mainsActions", mainAction.getReference_label());
                                if (toolsCoupling.getTractor() != null) {
                                    ExportUtils.setExtraField(export, "tractors", toolsCoupling.getTractor().getName());
                                }
                                toolsCouplingsExports.add(export);
                            } else {
                                for (Equipment equipment:equipments) {
                                    DomainExportEntity export = (DomainExportEntity) model.clone();
                                    ExportUtils.copyFields(toolsCoupling, export, null,
                                            ToolsCoupling.PROPERTY_TOOLS_COUPLING_NAME,
                                            ToolsCoupling.PROPERTY_WORKFORCE,
                                            ToolsCoupling.PROPERTY_COMMENT);
                                    ExportUtils.setExtraField(export, "agrosystInterventionTypes", mainAction.getIntervention_agrosyst());
                                    ExportUtils.setExtraField(export, "mainsActions", mainAction.getReference_label());
                                    if (toolsCoupling.getTractor() != null) {
                                        ExportUtils.setExtraField(export, "tractor", toolsCoupling.getTractor().getName());
                                    }
                                    ExportUtils.setExtraField(export, "equipments", equipment.getName());
                                    toolsCouplingsExports.add(export);
                                }
                            }
                        }
                    }
    
                    // assolement tab
                    List<CroppingPlanEntry> croppingPlan = getCroppingPlan0(domain.getTopiaId());
                    for (CroppingPlanEntry croppingPlanEntry : croppingPlan) {
                        List<CroppingPlanSpecies> species = croppingPlanEntry.getCroppingPlanSpecies();
                        if (CollectionUtils.isEmpty(species)) {
                            DomainExportEntity export = (DomainExportEntity) model.clone();
                            ExportUtils.copyFields(croppingPlanEntry, export, null,
                                    CroppingPlanEntry.PROPERTY_NAME,
                                    CroppingPlanEntry.PROPERTY_SELLING_PRICE,
                                    CroppingPlanEntry.PROPERTY_TYPE
                            );
                            croppingPlans.add(export);
                        } else {
                            for (CroppingPlanSpecies aSpecies : species) {
                                DomainExportEntity export = (DomainExportEntity) model.clone();
                                ExportUtils.copyFields(croppingPlanEntry, export, null,
                                        CroppingPlanEntry.PROPERTY_NAME,
                                        CroppingPlanEntry.PROPERTY_SELLING_PRICE,
                                        CroppingPlanEntry.PROPERTY_TYPE
                                );
    
                                ExportUtils.copyFields(aSpecies.getSpecies(), export, null,
                                        RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE,
                                        RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE,
                                        RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE,
                                        RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE
                                );
                                if (aSpecies.getVariety() != null) {
                                    ExportUtils.setExtraField(export, "label", aSpecies.getVariety().getLabel());
                                }
                                croppingPlans.add(export);
                            }
                        }
                    }
    
                }
            }
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't copy properties", ex);
        }

        // technical export
        EntityExporter exporter = new EntityExporter();
        InputStream stream = exporter.exportAsXlsStream(sheet);

        return stream;
    }

    @Override
    public void importDomainForXlsStream(InputStream is) {

        // get all possible bean infos
        DomainMainBeanInfo domainMainTabInfo = newInstance(DomainMainBeanInfo.class);
        DomainGpsDataBeanInfo geoPointTabInfo = newInstance(DomainGpsDataBeanInfo.class);
        DomainSolBeanInfo groundTabInfo = newInstance(DomainSolBeanInfo.class);
        DomainStatusBeanInfo domainStatusTabInfo = newInstance(DomainStatusBeanInfo.class);
        DomainMaterielBeanInfo equipmentTabInfo = newInstance(DomainMaterielBeanInfo.class);
        DomainToolsCouplingBeanInfo domainToolsCouplingBeanInfo = newInstance(DomainToolsCouplingBeanInfo.class);
        DomainCroppingPlanEntryBeanInfo croppingPlanTabInfo = newInstance(DomainCroppingPlanEntryBeanInfo.class);

        // technical import
        EntityImporter importer = new EntityImporter();
        Map<EntityExportTabInfo, List<DomainExportEntity>> datas = importer.importFromStream(is,
                DomainExportEntity.class,
                domainMainTabInfo,
                geoPointTabInfo,
                groundTabInfo,
                domainStatusTabInfo,
                equipmentTabInfo,
                domainToolsCouplingBeanInfo,
                croppingPlanTabInfo);
        
        // custom import (main)
        List<DomainExportEntity> mainBeanInfo = datas.get(domainMainTabInfo);
        MultiKeyMap<Object, Domain> domainCache = new MultiKeyMap<Object, Domain>();
        for (DomainExportEntity beanInfo : mainBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            // si on a déjà lu la ligne, on ne recreer pas un noveau domaine
            if (!domainCache.containsKey(domainName, domainCampaign)) {
                Domain domain = domainDao.newInstance();
                // common
                domain.setType(beanInfo.getDomainType());
                domain.setCampaign(domainCampaign);
                domain.setName(domainName);
                domain.setLocation(locationDao.forCodePostalEquals(beanInfo.getPostalCode()).findAny()); // FIXME echatellier : any ?
                domain.setMainContact(beanInfo.getMainContact());
                // main
                domain.setDescription((String)beanInfo.getExtras().get(Domain.PROPERTY_DESCRIPTION));
                domain.setLegalStatus(refLegalStatusDao.forLibelle_INSEEEquals((String)beanInfo.getExtras().get(Domain.PROPERTY_LEGAL_STATUS)).findUniqueOrNull());
                domain.setZoning((Zoning)beanInfo.getExtras().get(Domain.PROPERTY_ZONING));
                
                // mandatory fields
                domain.setActive(true);
                domain.setCode(UUID.randomUUID().toString());
                domain.setUpdateDate(getContext().getCurrentDate());

                domain = domainDao.create(domain);
                domainCache.put(domainName, domainCampaign, domain);
            }
        }

        // custom import gps data
        List<DomainExportEntity> geoBeanInfo = datas.get(geoPointTabInfo);
        for (DomainExportEntity beanInfo : geoBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new gps data
            GeoPoint geoPoint = geoPointDao.newInstance();
            geoPoint.setName(beanInfo.getExtraAsString(GeoPoint.PROPERTY_NAME));
            String latitude = beanInfo.getExtraAsString(GeoPoint.PROPERTY_LATITUDE);
            String longitude = beanInfo.getExtraAsString(GeoPoint.PROPERTY_LONGITUDE);
            geoPoint.setLatitude(Double.parseDouble(latitude.replace(',', '.')));
            geoPoint.setLongitude(Double.parseDouble(longitude.replace(',', '.')));
            geoPoint.setDomain(domain);
            geoPoint.setDescription((String)beanInfo.getExtras().get(GeoPoint.PROPERTY_DESCRIPTION));
            geoPoint = geoPointDao.create(geoPoint);
        }

        // custom import sol
        List<DomainExportEntity> groundBeanInfo = datas.get(groundTabInfo);
        for (DomainExportEntity beanInfo : groundBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new ground
            Ground ground = groundDao.newInstance();
            ground.setName(beanInfo.getExtraAsString(Ground.PROPERTY_NAME));
            ground.setImportance(Double.parseDouble(beanInfo.getExtraAsString(Ground.PROPERTY_IMPORTANCE)));
            ground.setComment(beanInfo.getExtraAsString(Ground.PROPERTY_COMMENT));
            String nomSol = beanInfo.getExtraAsString("solArvalis");
            String regionSol = beanInfo.getExtraAsString("region");
            RefSolArvalis refSolArvalis = refSolArvalisDao.forProperties(RefSolArvalis.PROPERTY_SOL_NOM, nomSol,
                    RefSolArvalis.PROPERTY_SOL_REGION, regionSol).findUnique();
            ground.setRefSolArvalis(refSolArvalis);
            ground.setDomain(domain);
            ground = groundDao.create(ground);
        }

        // custom import status
        List<DomainExportEntity> statusBeanInfo = datas.get(domainStatusTabInfo);
        for (DomainExportEntity beanInfo : statusBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new ground
            try {
                ExportUtils.copyFields(beanInfo, domain,
                        Domain.PROPERTY_STATUS_COMMENT,
                        Domain.PROPERTY_PARTNERS_NUMBER,
                        Domain.PROPERTY_OTHER_WORK_FORCE,
                        Domain.PROPERTY_FAMILY_WORK_FORCE_WAGE,
                        Domain.PROPERTY_PERMANENT_EMPLOYEES_WORK_FORCE,
                        Domain.PROPERTY_TEMPORARY_EMPLOYEES_WORK_FORCE,
                        Domain.PROPERTY_CROPS_WORK_FORCE,
                        Domain.PROPERTY_WAGE_COSTS,
                        Domain.PROPERTY_WORKFORCE_COMMENT,
                        Domain.PROPERTY_ORIENTATION,
                        Domain.PROPERTY_OTEX18,
                        Domain.PROPERTY_OTEX70,
                        Domain.PROPERTY_USED_AGRICULTURAL_AREA,
                        Domain.PROPERTY_EXPERIMENTAL_AGRICULTURAL_AREA,
                        Domain.PROPERTY_HOMOGENIZATION_EXPERIMENTAL_AGRICULTURAL_AREA,
                        Domain.PROPERTY_MSA_FEE,
                        Domain.PROPERTY_AVERAGE_TENANT_FARMING,
                        Domain.PROPERTY_DECOUPLED_ASSISTANCE);
            } catch (Exception ex) {
                throw new AgrosystTechnicalException("Can't copy fields", ex);
            }
        }

        // custom import materiel
        List<DomainExportEntity> materielBeanInfo = datas.get(equipmentTabInfo);
        Map<String, Equipment> equipmentCache = Maps.newHashMap();
        for (DomainExportEntity beanInfo : materielBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new materiel
            Equipment equipment = equipmentDao.newInstance();
            equipment.setName(beanInfo.getExtraAsString(Equipment.PROPERTY_NAME));
            equipment.setDescription(beanInfo.getExtraAsString(Equipment.PROPERTY_DESCRIPTION));
            equipment.setMaterielETA("oui".equalsIgnoreCase(beanInfo.getExtraAsString(Equipment.PROPERTY_MATERIEL_ETA)));
            // get ref materiel
            String materielType1 = beanInfo.getExtraAsString(RefMateriel.PROPERTY_TYPE_MATERIEL1);
            if (StringUtils.isNoneBlank(materielType1)) { // sinon c'est autre
                String materielType2 = beanInfo.getExtraAsString(RefMateriel.PROPERTY_TYPE_MATERIEL2);
                String materielType3 = beanInfo.getExtraAsString(RefMateriel.PROPERTY_TYPE_MATERIEL3);
                String materielType4 = beanInfo.getExtraAsString(RefMateriel.PROPERTY_TYPE_MATERIEL4);
                String uniteParAn = beanInfo.getExtraAsString(RefMateriel.PROPERTY_UNITE_PAR_AN);
                double uniteParAnd = Double.parseDouble(uniteParAn.replace(',', '.'));
                RefMateriel refMateriel = refMaterielTopiaDao.forProperties(
                        RefMateriel.PROPERTY_TYPE_MATERIEL1, materielType1,
                        RefMateriel.PROPERTY_TYPE_MATERIEL2, materielType2,
                        RefMateriel.PROPERTY_TYPE_MATERIEL3, materielType3,
                        RefMateriel.PROPERTY_TYPE_MATERIEL4, materielType4,
                        RefMateriel.PROPERTY_UNITE_PAR_AN, uniteParAnd).findUnique();
                equipment.setRefMateriel(refMateriel);
            }
            equipment.setDomain(domain);
            equipment.setCode(UUID.randomUUID().toString());
            equipment = equipmentDao.create(equipment);
            // add in cache
            equipmentCache.put(equipment.getName(), equipment);
        }

        // custom import tools coupling
        List<DomainExportEntity> toolsCouplingBeanInfo = datas.get(domainToolsCouplingBeanInfo);
        Map<String, ToolsCoupling> toolsCouplingCache = Maps.newHashMap();
        for (DomainExportEntity beanInfo : toolsCouplingBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new tool coupling
            String toolsCouplingName = beanInfo.getExtraAsString(ToolsCoupling.PROPERTY_TOOLS_COUPLING_NAME);
            ToolsCoupling toolCoupling;
            if (toolsCouplingCache.containsKey(toolsCouplingName)) {
                toolCoupling = toolsCouplingCache.get(toolsCouplingName);
            } else {
                toolCoupling = toolsCouplingDao.newInstance();
                toolCoupling.setToolsCouplingName(toolsCouplingName);
                toolCoupling.setComment(beanInfo.getExtraAsString(ToolsCoupling.PROPERTY_COMMENT));
                String workForce = beanInfo.getExtraAsString(ToolsCoupling.PROPERTY_WORKFORCE);
                if (StringUtils.isNoneBlank(workForce)) {
                    toolCoupling.setWorkforce(Integer.parseInt(workForce));
                }
                toolCoupling.setDomain(domain);
                toolCoupling.setCode(UUID.randomUUID().toString());
                toolsCouplingCache.put(toolsCouplingName, toolCoupling);
            }

            // manage extra fields
            RefInterventionAgrosystTravailEDI interventionAgrosystTravailEDI = refInterventionAgrosystTravailEDITopiaDao
                    .forIntervention_agrosystEquals(AgrosystInterventionType.valueOf(beanInfo.getExtraAsString("agrosystInterventionTypes"))).findUnique();
            if (toolCoupling.getMainsActions() == null || !toolCoupling.getMainsActions().contains(interventionAgrosystTravailEDI)) {
                toolCoupling.addMainsActions(interventionAgrosystTravailEDI);
            }

            String tractor = beanInfo.getExtraAsString("tractor");
            if (!Strings.isNullOrEmpty(tractor)) {
                toolCoupling.setTractor(equipmentCache.get(tractor));
            }
            String equipment = beanInfo.getExtraAsString("equipments");
            if (Strings.isNullOrEmpty(equipment)) {
                toolCoupling.addEquipments(equipmentCache.get(equipment));
            }

            // persist
            if (toolCoupling.isPersisted()) {
                toolCoupling = toolsCouplingDao.update(toolCoupling);
            } else {
                toolCoupling = toolsCouplingDao.create(toolCoupling);
            }
        }

        // custom import cropping plan entry
        List<DomainExportEntity> croppingPlanBeanInfo = datas.get(croppingPlanTabInfo);
        Map<String, CroppingPlanEntry> croppingPlanEntryCache = Maps.newHashMap();
        for (DomainExportEntity beanInfo : croppingPlanBeanInfo) {
            String domainName = beanInfo.getDomainName();
            int domainCampaign = beanInfo.getCampaign();
            Domain domain = domainCache.get(domainName, domainCampaign);
            // create new tool cropping plan entry
            String croppingPlanEntryName = beanInfo.getExtraAsString(CroppingPlanEntry.PROPERTY_NAME);
            CroppingPlanEntry croppingPlanEntry;
            if (croppingPlanEntryCache.containsKey(croppingPlanEntryName)) {
                croppingPlanEntry = croppingPlanEntryCache.get(croppingPlanEntryName);
            } else {
                croppingPlanEntry = croppingPlanEntryDao.newInstance();
                croppingPlanEntry.setDomain(domain);
                croppingPlanEntry.setName(croppingPlanEntryName);
                croppingPlanEntry.setCode(UUID.randomUUID().toString());
                croppingPlanEntry.setType(CroppingEntryType.valueOf(beanInfo.getExtraAsString(CroppingPlanEntry.PROPERTY_TYPE)));
                String price = beanInfo.getExtraAsString(CroppingPlanEntry.PROPERTY_SELLING_PRICE);
                if (StringUtils.isNotBlank(price)) {
                    croppingPlanEntry.setSellingPrice(Double.parseDouble(price.replace(',', '.')));
                }
            }
            // create new species
            String species = beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE);
            if (StringUtils.isNotBlank(species)) {
                String qualifiant = beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE);
                String type = beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE);
                String destination = beanInfo.getExtraAsString(RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE);
                RefEspece refEspece = refEspeceDao.forProperties(
                        RefEspece.PROPERTY_LIBELLE_ESPECE_BOTANIQUE, species,
                        RefEspece.PROPERTY_LIBELLE_QUALIFIANT__AEE, Strings.nullToEmpty(qualifiant),
                        RefEspece.PROPERTY_LIBELLE_TYPE_SAISONNIER__AEE, Strings.nullToEmpty(type),
                        RefEspece.PROPERTY_LIBELLE_DESTINATION__AEE, Strings.nullToEmpty(destination)).findUnique();
                CroppingPlanSpecies croppingPlanSpecies = new CroppingPlanSpeciesImpl();
                croppingPlanSpecies.setCode(UUID.randomUUID().toString());
                croppingPlanSpecies.setSpecies(refEspece);
                croppingPlanEntry.addCroppingPlanSpecies(croppingPlanSpecies);
            }
            // persist croppign plan
            if (croppingPlanEntry.isPersisted()) {
                croppingPlanEntryDao.update(croppingPlanEntry);
            } else {
                croppingPlanEntryDao.create(croppingPlanEntry);
            }
        }

        getTransaction().commit();
    }

    @Override
    public boolean checkDomainExistence(String domainName) {
        long domain = domainDao.countDomainWithName(domainName);
        boolean result = domain > 0;
        return result;
    }

    @Override
    public long getDomainsCount(Boolean active) {
        if (active == null) {
            return domainDao.count();
        }
        return domainDao.forActiveEquals(active).count();
    }

    @Override
    public List<Price> getDomainPrices(String domainTopiaId) {
        List<Price> result = pricesService.getDomainPrices(domainTopiaId, null);
        return result;
    }

    @Override
    public void deleteToolsCouplings(Collection<ToolsCoupling> toolsCouplings) {
        // intervention to remove
        Set<PracticedIntervention> allPracticedInterventionsToRemove = Sets.newHashSet();
        Set<EffectiveIntervention> allEffectiveInterventionsToRemove = Sets.newHashSet();

        // all action related to the tools coupling
        Set<AbstractAction> allActionsToRemove = Sets.newHashSet();

        for (ToolsCoupling toolsCoupling : toolsCouplings) {

            // all intervention related to the tools coupling
            List<PracticedIntervention> practicedInterventions = practicedInterventionDao.forToolsCouplingCodesContains(toolsCoupling.getCode()).findAll();
            // all action related to the tools coupling
            List<AbstractAction> toolsCouplingAbstractActions = abstractActionDao.forToolsCouplingCodeEquals(toolsCoupling.getCode()).findAll();
            allActionsToRemove.addAll(toolsCouplingAbstractActions);

            for (PracticedIntervention practicedIntervention : practicedInterventions) {
                List<AbstractAction> interventionAbstractActions = abstractActionDao.forPracticedInterventionEquals(practicedIntervention).findAll();
                // if all actions or related to this toolsCoupling the intervention has to be deleted
                if (interventionAbstractActions.equals(toolsCouplingAbstractActions)) {
                    allPracticedInterventionsToRemove.add(practicedIntervention);
                } else {
                    Collection<String> toolCouplingsCodes = practicedIntervention.getToolsCouplingCodes();
                    toolCouplingsCodes.remove(toolsCoupling.getCode());
                }
            }

            List<EffectiveIntervention> effectiveInterventions = effectiveInterventionDao.forToolCouplingsContains(toolsCoupling).findAll();
            for (EffectiveIntervention effectiveIntervention : effectiveInterventions) {
                List<AbstractAction> interventionAbstractActions = abstractActionDao.forEffectiveInterventionEquals(effectiveIntervention).findAll();
                if (interventionAbstractActions.equals(toolsCouplingAbstractActions)) {
                    allEffectiveInterventionsToRemove.add(effectiveIntervention);
                } else {
                    effectiveIntervention.getToolCouplings().remove(toolsCoupling);
                }
            }

        }
        practicedInterventionDao.deleteAll(allPracticedInterventionsToRemove);
        effectiveInterventionDao.deleteAll(allEffectiveInterventionsToRemove);
        abstractActionDao.deleteAll(allActionsToRemove);
    }

    @Override
    public void deleteUnusedMainAction(String toolsCouplingCode, Collection<RefInterventionAgrosystTravailEDI> originalMainsActions, Collection<RefInterventionAgrosystTravailEDI> actualMainsActions) {
        Collection<RefInterventionAgrosystTravailEDI> mainsActionsToRemove = CollectionUtils.subtract(originalMainsActions, actualMainsActions);
        // all interventions related to the tools coupling.
        List<AbstractAction> interventionAbstractActions = abstractActionDao.forToolsCouplingCodeEquals(toolsCouplingCode).findAll();

        Set<PracticedIntervention> allPracticedInterventionsToRemove = Sets.newHashSet();
        Set<EffectiveIntervention> allEffectiveInterventionsToRemove = Sets.newHashSet();
        Set<AbstractAction> allActionsToRemove = Sets.newHashSet();

        for (AbstractAction action : interventionAbstractActions) {
            AbstractAction actionToRemove = null;
            // looking for the action related to the mainAction (RefInterventionAgrosystTravailEDI)
            for (RefInterventionAgrosystTravailEDI mainAction : mainsActionsToRemove) {
                if (action.getMainAction().getTopiaId().equals(mainAction.getTopiaId())) {
                    actionToRemove = action;
                    allActionsToRemove.add(actionToRemove);
                    break;
                }
            }
            if (actionToRemove != null) {
                PracticedIntervention practicedIntervention = actionToRemove.getPracticedIntervention();
                // if it's the only action the intervention is deleted
                List<AbstractAction> interventionActions = abstractActionDao.forPracticedInterventionEquals(practicedIntervention).findAll();
                if (interventionActions != null && interventionActions.size() == 1) {
                    allPracticedInterventionsToRemove.add(practicedIntervention);
                }

                EffectiveIntervention effectiveIntervention = actionToRemove.getEffectiveIntervention();
                // if it's the only action the intervention is deleted
                interventionActions = abstractActionDao.forEffectiveInterventionEquals(effectiveIntervention).findAll();
                if (interventionActions != null && interventionActions.size() == 1) {
                    allEffectiveInterventionsToRemove.add(effectiveIntervention);
                }
            }

        }
        practicedInterventionDao.deleteAll(allPracticedInterventionsToRemove);
        effectiveInterventionDao.deleteAll(allEffectiveInterventionsToRemove);
        abstractActionDao.deleteAll(allActionsToRemove);

    }

    @Override
    public Domain validateAndCommit(String domainId) {

        authorizationService.checkValidateDomain(domainId);

        // AThimel 09/12/13 Perform DB update is more powerful
        domainDao.validateDomain(domainId, context.getCurrentDate());

        getTransaction().commit();

        // AThimel 09/12/13 The next line makes sure that cache state is synchronized with database state
        getPersistenceContext().getHibernateSupport().getHibernateSession().clear();

        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        return domain;
    }

    @Override
    public List<GeoPoint> getGeoPoints(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<GeoPoint> result = geoPointDao.forDomainEquals(domain).findAll();
        result = anonymizeService.checkForGeoPointAnonymization(result);
        return result;
    }

    @Override
    public List<Equipment> getEquipments(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<Equipment> result = equipmentDao.forDomainEquals(domain).findAll();
        return result;
    }

    @Override
    public List<Ground> getGrounds(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<Ground> result = groundDao.forDomainEquals(domain).findAll();
        return result;
    }

    @Override
    public List<ToolsCoupling> getToolsCouplings(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<ToolsCoupling> result = toolsCouplingDao.forDomainEquals(domain).findAll();
        return result;
    }

    @Override
    public UsageList<ToolsCoupling> getToolsCouplingAndUsage(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<ToolsCoupling> elements = toolsCouplingDao.forDomainEquals(domain).findAll();
        UsageList<ToolsCoupling> result = entityUsageService.getToolsCouplingUsageList(elements, domain.getCampaign());
        return result;
    }

    @Override
    public List<String> getCroppingPlanCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns) {
        List<String> result = domainDao.findCroppingPlanCodeForDomainsAndCampaigns(code, campaigns);
        return result;
    }

    @Override
    public List<String> getCroppingPlanSpeciesCodeForCropCodeAndCampaigns(String cropCode, Set<Integer> campaigns) {
        List<String> result = domainDao.findCroppingPlanSpeciesCodeForCropCodeAndCampaigns(cropCode, campaigns);
        return result;
    }

    @Override
    public List<String> getToolsCouplingCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns) {
        List<String> result = domainDao.findToolsCouplingCodeForDomainsAndCampaigns(code, campaigns);
        return result;
    }

    @Override
    public List<ToolsCoupling> getToolsCouplingsForDomainCodeAndCampaigns(String code, Set<Integer> campaigns) {
        List<ToolsCoupling> result = domainDao.findToolsCouplingsForDomainCodeAndCampaigns(code, campaigns);
        return result;
    }
    @Override
    public List<CroppingPlanEntryDto> getCroppingPlanDtos(String domainId) {
        List<CroppingPlanEntry> entities = getCroppingPlan0(domainId);
        List<CroppingPlanEntryDto> result = Lists.transform(entities, CroppingPlans.CROPPING_PLAN_ENTRY_TO_DTO);
        return result;
    }

    @Override
    public UsageList<CroppingPlanEntryDto> getCroppingPlanEntryDtoAndUsage(String domainId) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(domainId));
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<CroppingPlanEntry> elements = getCroppingPlan0(domain.getTopiaId());
        UsageList<CroppingPlanEntryDto> result = entityUsageService.getCroppingPlanEntryDtoAndUsage(elements, domain.getCampaign());
        return result;
    }

    @Override
    public Map<String, Boolean> getCroppingPlanSpeciesUsage(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        List<CroppingPlanSpecies> elements = getCroppingPlanSpecies(domain.getTopiaId());
        Map<String, Boolean> result = entityUsageService.getCroppingPlanSpeciesUsageMap(elements, domain.getCampaign(), domain.getCode());
        return result;
    }

    @Override
    public Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> getEntryAndSpeciesFromCode(String croppingPlanEntryCode, Set<Integer> campaignsSet) {
        Preconditions.checkArgument(campaignsSet != null && !campaignsSet.isEmpty());

        // Maybe we have multiple entries, merge them
        List<CroppingPlanEntry> entries = croppingPlanEntryDao.findEntriesFromCode(croppingPlanEntryCode, campaignsSet);
        Preconditions.checkState(entries != null && !entries.isEmpty(), "No CroppingPlanEntry found with code: " + croppingPlanEntryCode);

        // Get all species
        Map<String, CroppingPlanSpecies> species = Maps.newLinkedHashMap();
        for (CroppingPlanEntry entry : entries) {
            if (entry.getCroppingPlanSpecies() != null) {
                species.putAll(Maps.uniqueIndex(entry.getCroppingPlanSpecies(), CroppingPlans.GET_SPECIES_CODE));
            }
        }

        CroppingPlanEntry entry = entries.iterator().next();
        Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> result = Pair.of(entry, species);
        return result;
    }

    @Override
    public Map<String, Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>>> findSpeciesCodeByCropCodeForCampaigns(List<String> croppingPlanEntryCodes, Set<Integer> campaignsSet) {
        Map<String, Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>>> result = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(croppingPlanEntryCodes) && CollectionUtils.isNotEmpty(campaignsSet)) {

            for (String croppingPlanEntryCode : croppingPlanEntryCodes) {
                // une Pair <Culture, Map<code espece, Espece>>
                Pair<CroppingPlanEntry, Map<String, CroppingPlanSpecies>> pair = getEntryAndSpeciesFromCode(croppingPlanEntryCode, campaignsSet);
                if (pair != null && pair.getValue() != null) {
                    Map<String, CroppingPlanSpecies> cpss = pair.getValue();
                    result.put(pair.getKey().getCode(), pair);
                }
            }
        }

        return result;
    }

    @Override
    public String getDomainCodeForGrowingSystem(String growingSystemId) {
        String result = domainDao.getDomainCodeForGrowingSystem(growingSystemId);
        return result;
    }

    @Override
    public List<Domain> getActiveWritableDomainsForDecisionRuleCreation() {
        List<Domain> result = domainDao.getActiveWritableDomainsForDecisionRuleCreation(getSecurityContext());
        return result;
    }

    @Override
    public List<String> findDomainIdsForCodeAndCampaigns(String code, Set<Integer> campaigns, boolean includeCropsFromInactiveDomains) {
        List<String> result = domainDao.findDomainIdsForCampaigns(code, campaigns, includeCropsFromInactiveDomains);
        return result;
    }
  

    protected Boolean validateEquipment(List<Equipment> equipments) {
        Boolean result = true;
        // validation des materiels
        if (equipments != null) {
            equipments.removeAll(Collections.singleton(null));
            for (Equipment equipment : equipments) {
                if (StringUtils.isBlank(equipment.getName())) {
                    result = false;
                }
            }
        }
        return result;
    }

    protected Boolean validateToolsCoupling(List<ToolsCoupling> toolsCouplings) {
        Boolean result = true;

        // validation des attelages
        if (toolsCouplings != null) {
            toolsCouplings.removeAll(Collections.singleton(null));
            for (ToolsCoupling toolsCoupling : toolsCouplings) {
                if (toolsCoupling.getMainsActions() == null || toolsCoupling.getMainsActions().isEmpty()) {
                    result = false;
                }
                if (StringUtils.isBlank(toolsCoupling.getToolsCouplingName())) {
                    result = false;
                }
            }
        }
        return result;
    }

    @Override
    public List<String> copyTools(String fromDomain, List<String> toDomainIds, Boolean includeToolCouplings, List<Equipment> equipmentsToCopy, List<ToolsCoupling> toolsCouplingsToCopy) {
        // List of "domain name - campaign"
        List<String> result = Lists.newArrayListWithExpectedSize(toDomainIds.size());

        Preconditions.checkArgument(validateEquipment(equipmentsToCopy));
        if (includeToolCouplings) {
            Preconditions.checkArgument(validateToolsCoupling(toolsCouplingsToCopy));
        }

        Domain domainFrom = domainDao.forTopiaIdEquals(fromDomain).findUnique();
        Collection<Domain> toDomains = domainDao.forTopiaIdIn(toDomainIds).findAll();

        if (equipmentsToCopy == null) {
            equipmentsToCopy = Lists.newArrayList();
        }
        Map indexedEquipments = Maps.uniqueIndex(equipmentsToCopy, Entities.GET_TOPIA_ID);

        if(toDomains != null && !toDomains.isEmpty()) {
            Binder<Equipment, Equipment> binder = BinderFactory.newBinder(Equipment.class);

            for (Domain toDomain : toDomains) {
                Map<String, Equipment> copiedEquipments = Maps.newHashMapWithExpectedSize(indexedEquipments.size());
                // copy equipments to the domain
                if (toDomain.getCode().equals(domainFrom.getCode())) {
                    // if same domain on other campaign Equipment Code is preserve to keep relation
                    // avoid duplicated equipment
                    for (Equipment equipment : equipmentsToCopy) {
                        Equipment toEquipment = null;
                        if (StringUtils.isNotBlank(equipment.getCode())) {
                            toEquipment = equipmentDao.forProperties(Equipment.PROPERTY_DOMAIN, toDomain, Equipment.PROPERTY_CODE, equipment.getCode()).findUniqueOrNull();
                        }
                        if (toEquipment == null) {
                            String originalId = equipment.getTopiaId();
                            Equipment copiedEquipment = equipmentDao.newInstance();
                            binder.copyExcluding(equipment, copiedEquipment,
                                    Equipment.PROPERTY_TOPIA_ID,
                                    Equipment.PROPERTY_DOMAIN);
                            // case of new equipment
                            if (Strings.isNullOrEmpty(copiedEquipment.getCode())) {
                                copiedEquipment.setCode(UUID.randomUUID().toString());
                            }
                            copiedEquipment.setDomain(toDomain);
                            copiedEquipment = equipmentDao.create(copiedEquipment);
                            copiedEquipments.put(originalId, copiedEquipment);
                        } else {

                            // the equipment already exists but it is added to the list to use it with tools couplings
                            copiedEquipments.put(equipment.getTopiaId(), toEquipment);
                        }
                    }
                } else {
                    for(Equipment equipment:equipmentsToCopy) {
                        Equipment copiedEquipment = equipmentDao.newInstance();
                        binder.copyExcluding(equipment, copiedEquipment,
                                Equipment.PROPERTY_TOPIA_ID,
                                Equipment.PROPERTY_CODE,
                                Equipment.PROPERTY_DOMAIN);
                        copiedEquipment.setCode(UUID.randomUUID().toString());
                        copiedEquipment.setDomain(toDomain);
                        copiedEquipment = equipmentDao.create(copiedEquipment);
                        copiedEquipments.put(equipment.getTopiaId(), copiedEquipment);
                    }
                }

                //copy toolsCoupling to domain
                if(includeToolCouplings && toolsCouplingsToCopy != null && !toolsCouplingsToCopy.isEmpty()){
                    List<ToolsCoupling> copiedToolsCouplings = Lists.newArrayList();

                        Binder<ToolsCoupling, ToolsCoupling> toolsCouplingBinder = BinderFactory.newBinder(ToolsCoupling.class);

                        if (toDomain.getCode().equals(domainFrom.getCode())) {
                            // if same domain on other campaign ToolsCoupling Code is preserve to keep relation
                            for (ToolsCoupling toolsCoupling : toolsCouplingsToCopy) {
                                // avoid duplicated toolsCoupling
                                ToolsCoupling toToolsCoupling = null;
                                if (StringUtils.isNotBlank(toolsCoupling.getCode())) {
                                    toToolsCoupling = toolsCouplingDao.forProperties(ToolsCoupling.PROPERTY_DOMAIN, toDomain, ToolsCoupling.PROPERTY_CODE, toolsCoupling.getCode()).findUniqueOrNull();
                                }
                                if (toToolsCoupling == null) {
                                    ToolsCoupling copiedToolsCoupling = toolsCouplingDao.newInstance();
                                    toolsCouplingBinder.copyExcluding(toolsCoupling, copiedToolsCoupling,
                                            ToolsCoupling.PROPERTY_TOPIA_ID,
                                            ToolsCoupling.PROPERTY_DOMAIN,
                                            ToolsCoupling.PROPERTY_TRACTOR,
                                            ToolsCoupling.PROPERTY_EQUIPMENTS,
                                            ToolsCoupling.PROPERTY_MAINS_ACTIONS
                                    );
                                    // case of new tools coupling
                                    if (Strings.isNullOrEmpty(copiedToolsCoupling.getCode())) {
                                        copiedToolsCoupling.setCode(UUID.randomUUID().toString());
                                    }
                                    copyRelatedEntitiesToToolsCoupling(copiedEquipments, copiedToolsCouplings, copiedToolsCoupling, toolsCoupling, toDomain);
                                }
                            }
                        } else {
                            for (ToolsCoupling toolsCoupling : toolsCouplingsToCopy) {
                                ToolsCoupling copiedToolsCoupling = toolsCouplingDao.newInstance();
                                toolsCouplingBinder.copyExcluding(toolsCoupling, copiedToolsCoupling,
                                        ToolsCoupling.PROPERTY_TOPIA_ID,
                                        ToolsCoupling.PROPERTY_DOMAIN,
                                        ToolsCoupling.PROPERTY_TRACTOR,
                                        ToolsCoupling.PROPERTY_EQUIPMENTS,
                                        ToolsCoupling.PROPERTY_MAINS_ACTIONS,
                                        ToolsCoupling.PROPERTY_CODE
                                );
                                copiedToolsCoupling.setCode(UUID.randomUUID().toString());
                                copyRelatedEntitiesToToolsCoupling(copiedEquipments, copiedToolsCouplings, copiedToolsCoupling, toolsCoupling, toDomain);
                            }
                        }

                    toolsCouplingDao.createAll(copiedToolsCouplings);
                }
                result.add(toDomain.getName() + " (" + toDomain.getCampaign() + ")");
            }
        }
        getTransaction().commit();
        return result;
    }

    /**
     * Set tractor, equipments and actions to the copied tools coupling
     * @param copiedEquipments map of original equipment topiaId to copied equipment
     * @param copiedToolsCouplings target domain's tools coupling
     * @param copiedToolsCoupling new copied tools coupling
     * @param toolsCoupling original tools coupling
     * @param toDomain targeted domain
     */
    protected void copyRelatedEntitiesToToolsCoupling(Map<String, Equipment> copiedEquipments, List<ToolsCoupling> copiedToolsCouplings, ToolsCoupling copiedToolsCoupling, ToolsCoupling toolsCoupling, Domain toDomain) {
        // find tractor from new Equipment
        if (toolsCoupling.getTractor() != null) {
            copiedToolsCoupling.setTractor(copiedEquipments.get(toolsCoupling.getTractor().getTopiaId()));
        }

        // find equipments from new Equipements
        if (toolsCoupling.getEquipments() != null && !toolsCoupling.getEquipments().isEmpty()) {
            List<Equipment> copiedTCEquipments = Lists.newArrayListWithExpectedSize(toolsCoupling.getEquipments().size());
            for (Equipment toolsCouplingEquipment : toolsCoupling.getEquipments()) {
                copiedTCEquipments.add(copiedEquipments.get(toolsCouplingEquipment.getTopiaId()));
            }
            copiedToolsCoupling.setEquipments(copiedTCEquipments);
        }
        List<RefInterventionAgrosystTravailEDI> clonedMainsActions = Lists.newArrayList(toolsCoupling.getMainsActions());
        copiedToolsCoupling.setMainsActions(clonedMainsActions);
        copiedToolsCoupling.setDomain(toDomain);

        copiedToolsCouplings.add(copiedToolsCoupling);
    }


    @Override
    public double getDomainSAUArea(String domainId) {
        Domain domain = domainDao.forTopiaIdEquals(domainId).findUnique();
        Double result = domain.getUsedAgriculturalArea();
        if (result == null) {
            result = plotDao.getDomainPlotTotalArea(domainId);
        }
        return result;
    }

    @Override
    public void importPZ0Domains(Map<Class, ImportResults> allResults) {
        ImportResults domainsImportResults = allResults.remove(Domain.class);
        if (domainsImportResults != null && domainsImportResults.getIgnoredRecords() == 0) {
            try {
                Map<String, EntityAndDependencies> entitiesAndDependencies = domainsImportResults.getEntityAndDepsByCsvIds();

                Collection<EntityAndDependencies> domainsAndDependencies = entitiesAndDependencies.values();
                int nbDomains = domainsAndDependencies.size();
                int i = 1;
                for (EntityAndDependencies entityAndDependencies : domainsAndDependencies) {
                    DomainAndDependencies domainAndDependencies = (DomainAndDependencies) entityAndDependencies;
                    Domain domain = (Domain) domainAndDependencies.getEntity();

                    Collection<CroppingPlanEntryDto> crops = domainAndDependencies.getCroppingPlanEntryDtosByPZ0CsvId() == null ? null : domainAndDependencies.getCroppingPlanEntryDtosByPZ0CsvId().values();
                    List<Equipment> equipments = domainAndDependencies.getEquipments() == null ? new ArrayList<Equipment>(): Lists.newArrayList(domainAndDependencies.getEquipments());
                    List<ToolsCoupling> toolsCouplings = domainAndDependencies.getToolsCouplings() == null ? new ArrayList<ToolsCoupling>(): Lists.newArrayList(domainAndDependencies.getToolsCouplings());
                    List<Ground> grounds = domainAndDependencies.getGrounds() == null ? null: Lists.newArrayList(domainAndDependencies.getGrounds());

                    long start = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info(String.format("Début sauvegarde du domaine:'%s', campagne:%d - %d/%d.", domain.getName(), domain.getCampaign(), i++, nbDomains));
                        log.info("Nb crops:" + (crops == null ? 0 : crops.size()));
                        log.info("Nb tc:" + (toolsCouplings.size()));
                    }
                    domain = createOrUpdateDomainWithoutCommit(
                            domain,
                            domainAndDependencies.getLocationId(),
                            domainAndDependencies.getLegalStatusTopiaId(),
                            null,
                            crops,
                            domainAndDependencies.getOtex18(),
                            domainAndDependencies.getOtex70(),
                            grounds,
                            equipments,
                            toolsCouplings,
                            null);
                    domainAndDependencies.setEntity(domain);

                    long p1 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Sauvegarde du domaine en:" + (p1- start));
                        log.info("Mise à jour cultures du domaine.");
                    }
                    List<CroppingPlanEntry> croppingPlanEntries = getCroppingPlan0(domain.getTopiaId());

                    if (crops != null) {
                        Map<String, CroppingPlanEntryDto> croppingPlanEntryDtoByCsvId = domainAndDependencies.getCroppingPlanEntryDtosByPZ0CsvId();
                        if (croppingPlanEntryDtoByCsvId != null && !croppingPlanEntryDtoByCsvId.isEmpty()) {
                            Map<String, CroppingPlanEntry> persistedCroppingPlanEntryByCode = Maps.uniqueIndex(croppingPlanEntries, GET_CROPPING_PLAN_ENTRY_CODE);
                            for (Map.Entry<String, CroppingPlanEntryDto> cropToSaveIter : croppingPlanEntryDtoByCsvId.entrySet()) {
                                CroppingPlanEntryDto dto = cropToSaveIter.getValue();
                                CroppingPlanEntry persistedCroppingPlanEntry = persistedCroppingPlanEntryByCode.get(dto.getCode());
                                dto.setTopiaId(persistedCroppingPlanEntry.getTopiaId());

                                Collection<CroppingPlanSpeciesDto> speciesToSaves = dto.getSpecies();
                                if (CollectionUtils.isNotEmpty(speciesToSaves)) {
                                    List<CroppingPlanSpecies> persistedSpecies = persistedCroppingPlanEntry.getCroppingPlanSpecies();
                                    Map<String, CroppingPlanSpecies> persistedSpeciesByCode = Maps.uniqueIndex(persistedSpecies, GET_SPECIES_CODE);
                                    for (CroppingPlanSpeciesDto speciesToSave : speciesToSaves) {
                                        speciesToSave.setTopiaId(persistedSpeciesByCode.get(speciesToSave.getCode()).getTopiaId());
                                    }
                                }
                            }
                        }
                    }

                    long p2 = System.currentTimeMillis();
                    if (log.isInfoEnabled()) {
                        log.info("Mise à jour des culture en:" + (p2- p1));
                        log.info("Fin de sauvegarde du domaine, traitement réalisé en:" + (p2- start));
                    }
                }

            }  catch (Exception e) {
                throw (new AgrosystImportException("Echec de persistance des domaines", e));
            }
        }
    }

}
