package fr.inra.agrosyst.services.context;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: NavigationContextServiceImpl.java 4483 2014-10-28 13:31:15Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/context/NavigationContextServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntity;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingPlanTopiaDao;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.context.NavigationContextService;
import fr.inra.agrosyst.api.services.domain.DomainDto;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanDto;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemDto;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class NavigationContextServiceImpl extends AbstractAgrosystService implements NavigationContextService {

    private static final Log log = LogFactory.getLog(NavigationContextServiceImpl.class);

    protected NetworkService networkService;
    protected DomainService domainService;
    protected GrowingPlanService growingPlanService;
    protected GrowingSystemService growingSystemService;
    protected AnonymizeService anonymizeService;

    protected NetworkTopiaDao networkDao;
    protected DomainTopiaDao domainDao;
    protected GrowingPlanTopiaDao growingPlanDao;
    protected GrowingSystemTopiaDao growingSystemDao;

    public void setNetworkService(NetworkService networkService) {
        this.networkService = networkService;
    }

    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }

    public void setGrowingPlanService(GrowingPlanService growingPlanService) {
        this.growingPlanService = growingPlanService;
    }

    public void setGrowingSystemService(GrowingSystemService growingSystemService) {
        this.growingSystemService = growingSystemService;
    }

    public void setAnonymizeService(AnonymizeService anonymizeService) {
        this.anonymizeService = anonymizeService;
    }

    public void setNetworkDao(NetworkTopiaDao networkDao) {
        this.networkDao = networkDao;
    }

    public void setDomainDao(DomainTopiaDao domainDao) {
        this.domainDao = domainDao;
    }

    public void setGrowingPlanDao(GrowingPlanTopiaDao growingPlanDao) {
        this.growingPlanDao = growingPlanDao;
    }

    public void setGrowingSystemDao(GrowingSystemTopiaDao growingSystemDao) {
        this.growingSystemDao = growingSystemDao;
    }

    @Override
    public List<Integer> getAllCampaigns() {
        List<Integer> result = domainDao.getAllCampaigns();
        return result;
    }

    @Override
    public ResultList<Network> getAllNetworks() {
        ResultList<Network> result = networkService.getFilteredNetworks(null);
        return result;
    }

    @Override
    public ResultList<DomainDto> getAllDomainsForCampaign(Set<Integer> campaigns, Set<String> networkIds) {

        NavigationContext navigationContext = new NavigationContext();
        navigationContext.setCampaigns(campaigns);
        navigationContext.setNetworks(networkIds);

        DomainFilter filter = new DomainFilter();
        filter.setNavigationContext(navigationContext);

        ResultList<DomainDto> result = domainService.getFilteredDomainsDto(filter);
        return result;
    }

    @Override
    public ResultList<GrowingPlanDto> getAllGrowingPlansForDomains(Set<Integer> campaigns, Set<String> domainIds, Set<String> networkIds) {

        NavigationContext navigationContext = new NavigationContext();
        navigationContext.setCampaigns(campaigns);
        navigationContext.setNetworks(networkIds);
        navigationContext.setDomains(domainIds);

        GrowingPlanFilter filter = new GrowingPlanFilter();
        filter.setNavigationContext(navigationContext);

        ResultList<GrowingPlanDto> result = growingPlanService.getFilteredGrowingPlansDto(filter);
        return result;
    }

    @Override
    public ResultList<GrowingSystemDto> getAllGrowingSystemsForGrowingPlans(
            Set<Integer> campaigns, Set<String> domainIds, Set<String> growingPlansIds, Set<String> networkIds) {

        NavigationContext navigationContext = new NavigationContext();
        navigationContext.setCampaigns(campaigns);
        navigationContext.setNetworks(networkIds);
        navigationContext.setDomains(domainIds);
        navigationContext.setGrowingPlans(growingPlansIds);

        GrowingSystemFilter filter = new GrowingSystemFilter();
        filter.setNavigationContext(navigationContext);

        ResultList<GrowingSystemDto> result = growingSystemService.getFilteredGrowingSystemsDto(filter);
        return result;
    }

    @Override
    public NavigationContext verify(NavigationContext navigationContext) {
        Preconditions.checkArgument(navigationContext != null);
        NavigationContext result = new NavigationContext();

        final Set<Integer> campaigns = Sets.newHashSet();
        final Set<String> networks = result.getNetworks();
        final Set<String> domains = result.getDomains();
        final Set<String> growingPlans = result.getGrowingPlans();
        final Set<String> growingSystems = result.getGrowingSystems();

        final Predicate<Domain> isValidDomain = Predicates.and(Predicates.notNull(), new Predicate<Domain>() {
            @Override
            public boolean apply(Domain domain) {
                // TODO AThimel 17/09/13 Manage networks
                boolean result = campaigns.isEmpty() || campaigns.contains(domain.getCampaign());
                return result;
            }
        });

        final Predicate<GrowingPlan> isValidGrowingPlan = Predicates.and(Predicates.notNull(), new Predicate<GrowingPlan>() {
            @Override
            public boolean apply(GrowingPlan growingPlan) {
                // TODO AThimel 17/09/13 Manage networks
                Domain domain = growingPlan.getDomain();
                boolean result = isValidDomain.apply(domain);
                result &= domains.isEmpty() || domains.contains(domain.getTopiaId());
                return result;
            }
        });

        final Predicate<GrowingSystem> isValidGrowingSystem = Predicates.and(Predicates.notNull(), new Predicate<GrowingSystem>() {
            @Override
            public boolean apply(GrowingSystem growingSystem) {
                // TODO AThimel 17/09/13 Manage networks
                GrowingPlan growingPlan = growingSystem.getGrowingPlan();
                boolean result = isValidGrowingPlan.apply(growingPlan);
                result &= growingPlans.isEmpty() || growingPlans.contains(growingPlan.getTopiaId());
                return result;
            }
        });

        if (navigationContext.getCampaigns() != null) {
            campaigns.addAll(navigationContext.getCampaigns());
        }
        result.setCampaigns(campaigns);

        if (navigationContext.getNetworksCount() > 0) {
            for (String networkId : navigationContext.getNetworks()) {
                Network network = networkDao.forTopiaIdEquals(networkId).findUniqueOrNull();
                if (Predicates.notNull().apply(network)) {
                    networks.add(networkId);
                } else if (log.isWarnEnabled()) {
                    log.warn(String.format("The network (%s) does not exist.", networkId));
                }
            }
        }

        if (navigationContext.getDomainsCount() > 0) {
            for (String domainId : navigationContext.getDomains()) {
                Domain domain = domainDao.forTopiaIdEquals(domainId).findUniqueOrNull();
                if (isValidDomain.apply(domain)) {
                    domains.add(domainId);
                } else if (log.isWarnEnabled()) {
                    log.warn(String.format("The domain (%s) does not exist or does not match any of the campaigns %s.", domainId, campaigns));
                }
            }
        }

        if (navigationContext.getGrowingPlansCount() > 0) {
            for (String growingPlanId : navigationContext.getGrowingPlans()) {
                GrowingPlan growingPlan = growingPlanDao.forTopiaIdEquals(growingPlanId).findUniqueOrNull();
                if (isValidGrowingPlan.apply(growingPlan)) {
                    growingPlans.add(growingPlanId);
                } else if (log.isWarnEnabled()) {
                    log.warn(String.format("The growingPlan (%s) does not exist or does not match any of the campaigns %s or domains %s.", growingPlanId, campaigns, domains));
                }
            }
        }

        if (navigationContext.getGrowingSystemsCount() > 0) {
            for (String growingSystemId : navigationContext.getGrowingSystems()) {
                GrowingSystem growingSystem = growingSystemDao.forTopiaIdEquals(growingSystemId).findUniqueOrNull();
                if (isValidGrowingSystem.apply(growingSystem)) {
                    growingSystems.add(growingSystemId);
                } else if (log.isWarnEnabled()) {
                    log.warn(String.format("The growingSystem (%s) does not exist or does not match any of the campaigns %s, domains %s or growingPlans %s.", growingSystemId, campaigns, domains, growingPlans));
                }
            }
        }

        return result;
    }

    @Override
    public NavigationContext verify(NavigationContext navigationContext, TopiaEntity newEntity) {
        Preconditions.checkArgument(navigationContext != null);
        String entityId = newEntity.getTopiaId();
        Preconditions.checkNotNull(entityId);
        Integer campaign = null;
        if (newEntity instanceof Network) {
            if (navigationContext.getNetworksCount() >= 1) {
                navigationContext.getNetworks().add(entityId);
            }
        } else if (newEntity instanceof Domain) {
            if (navigationContext.getDomainsCount() >= 1) {
                navigationContext.getDomains().add(entityId);
            }
            campaign = ((Domain) newEntity).getCampaign();
        } else if (newEntity instanceof GrowingPlan) {
            if (navigationContext.getGrowingPlansCount() >= 1) {
                navigationContext.getGrowingPlans().add(entityId);
            }
            campaign = ((GrowingPlan) newEntity).getDomain().getCampaign();

        } else if (newEntity instanceof GrowingSystem) {
            if (navigationContext.getGrowingSystemsCount() >= 1) {
                navigationContext.getGrowingSystems().add(entityId);
            }
            campaign = ((GrowingSystem) newEntity).getGrowingPlan().getDomain().getCampaign();
        } else {
            String message = "This method is supposed to be used with new " +
                    "Domain/GrowingPlan/GrowingSystem/Network entities only";
            throw new UnsupportedOperationException(message);
        }
        if (campaign != null && navigationContext.getCampaignsCount() >= 1) {
            navigationContext.getCampaigns().add(campaign);
        }
        NavigationContext result = verify(navigationContext);
        return result;
    }

    @Override
    public Map<String, String> getNetworks(Set<String> networkIds, int maxCount) {

        Map<String, String> result = Maps.newLinkedHashMap();
        if (networkIds != null && !networkIds.isEmpty()) {
            int max = maxCount >= 0 ? maxCount : Integer.MAX_VALUE;
            List<Network> networks = networkDao.forTopiaIdIn(networkIds).setOrderByArguments(Network.PROPERTY_NAME).find(0, max);

            for (Network network : networks) {
                result.put(network.getTopiaId(), network.getName());
            }
        }
        return result;
    }

    @Override
    public Map<String, String> getDomains(Set<String> domainsIds, int maxCount) {
        // TODO AThimel 26/02/14 Introduce cache (user dependent)

        Map<String, String> result = Maps.newLinkedHashMap();
        if (domainsIds != null && !domainsIds.isEmpty()) {
            int max = maxCount >= 0 ? maxCount : Integer.MAX_VALUE;
            List<Domain> domains = domainDao.forTopiaIdIn(domainsIds).setOrderByArguments(Domain.PROPERTY_NAME).find(0, max);

            Map<String, String> domainsMap = anonymizeService.getDomainsAsMap(domains);
            result.putAll(domainsMap);
        }
        return result;
    }

    @Override
    public Map<String, String> getGrowingPlans(Set<String> growingPlansIds, int maxCount) {
        // TODO AThimel 26/02/14 Introduce cache (user dependent)

        Map<String, String> result = Maps.newLinkedHashMap();
        if (growingPlansIds != null && !growingPlansIds.isEmpty()) {
            int max = maxCount >= 0 ? maxCount : Integer.MAX_VALUE;
            List<GrowingPlan> growingPlans = growingPlanDao.forTopiaIdIn(growingPlansIds).setOrderByArguments(GrowingPlan.PROPERTY_NAME).find(0, max);

            Map<String, String> growingPlansMap = anonymizeService.getGrowingPlansAsMap(growingPlans);
            result.putAll(growingPlansMap);
        }
        return result;
    }

    @Override
    public Map<String, String> getGrowingSystems(Set<String> growingSystemsIds, int maxCount) {
        // TODO AThimel 26/02/14 Introduce cache (user dependent)

        Map<String, String> result = Maps.newLinkedHashMap();

        if (growingSystemsIds != null && !growingSystemsIds.isEmpty()) {
            int max = maxCount >= 0 ? maxCount : Integer.MAX_VALUE;
            List<GrowingSystem> growingSystems = growingSystemDao.forTopiaIdIn(growingSystemsIds).setOrderByArguments(GrowingSystem.PROPERTY_NAME).find(0, max);

            for (GrowingSystem growingSystem : growingSystems) {
                result.put(growingSystem.getTopiaId(), growingSystem.getName());
            }
        }
        return result;
    }

}
