package fr.inra.agrosyst.services.common.export;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ExportUtils.java 4664 2014-12-16 13:19:57Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/export/ExportUtils.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.TypeDEPHY;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.csv.ValueFormatter;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.beans.BeanUtil;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class ExportUtils {

    private static final Log log = LogFactory.getLog(ExportUtils.class);

    protected static final String DATE_FORMAT = "dd/MM/yyyy";

    public static <K> void copyFields(K source, EntityExportExtra destination,
                                      Map<String, Function<K, Object>> customTransformers, Iterable<String> fields)
            throws InvocationTargetException, IllegalAccessException, NoSuchMethodException {
        for (final String field : fields) {

            // Get value
            Object value;
            if (customTransformers != null && customTransformers.containsKey(field)) {
                Function<K, Object> function = customTransformers.get(field);
                value = function.apply(source);
            } else {
                value = PropertyUtils.getProperty(source, field);
            }

            if (value != null && value instanceof TopiaEntity) {
                if (log.isWarnEnabled()) {
                    String message = String.format("A transformer is probably missing for property %s#%s",
                            source.getClass().getName(), field);
                    log.warn(message);
                }
            }

            // Search for setter
            Set<PropertyDescriptor> descriptors = BeanUtil.getDescriptors(
                    destination.getClass(),
                    Predicates.and(BeanUtil.IS_WRITE_DESCRIPTOR,
                            new Predicate<PropertyDescriptor>() {
                                @Override
                                public boolean apply(PropertyDescriptor input) {
                                    return field.equals(input.getName());
                                }
                            }
                    )
            );

            // Set value
            if (descriptors != null && !descriptors.isEmpty()) {
                Preconditions.checkState(descriptors.size() == 1,
                        "Unexpected size for field " + destination.getClass().getName() + "#" + field + "; size=" + descriptors.size());
                PropertyDescriptor descriptor = descriptors.iterator().next();
                descriptor.getWriteMethod().invoke(destination, value);
            } else {
                setExtraField(destination, field, value);
            }
        }
    }

    public static <K> void copyFields(EntityExportExtra source, K destination, String... fields) throws InvocationTargetException,
            IllegalAccessException, NoSuchMethodException, ParseException {
        SimpleDateFormat dateFormat = new SimpleDateFormat(ExportUtils.DATE_FORMAT);
        for (final String field : fields) {

            // Search for getter
            Set<PropertyDescriptor> descriptors = BeanUtil.getDescriptors(source.getClass(),
                    Predicates.and(BeanUtil.IS_READ_DESCRIPTOR, new Predicate<PropertyDescriptor>() {
                        @Override
                        public boolean apply(PropertyDescriptor input) {
                            return field.equals(input.getName());
                        }
                    }));

            Object value;
            if (!descriptors.isEmpty()) {
                Preconditions.checkState(descriptors.size() == 1, "Unexpected size for field " + destination.getClass().getName() + "#" + field
                        + "; size=" + descriptors.size());
                PropertyDescriptor descriptor = descriptors.iterator().next();
                // dans le cas de commons property, l'import la deja converti dans le bon type
                value = descriptor.getReadMethod().invoke(destination);
            } else {
                value = source.getExtra(field);

                // ici, c'est forcement une valeur String, on peut tenter de la convertir ici
                // car on dispose du type via "destination#field"
                String stringValue = (String)value;
                if (StringUtils.isNotBlank(stringValue)) {
                    // FIXME duplicated code with EntityImporter#importFromStream
                    Class type = PropertyUtils.getPropertyType(destination, field);
                    if (boolean.class.isAssignableFrom(type) || Boolean.class.isAssignableFrom(type)) {
                        value = "oui".equalsIgnoreCase(stringValue);
                    } else if (Date.class.isAssignableFrom(type)) {
                        value = dateFormat.parse(stringValue);
                    } else if (Double.class.isAssignableFrom(type) || double.class.isAssignableFrom(type)) {
                        value = Double.parseDouble(stringValue.replace(',', '.'));
                    } else if (Integer.class.isAssignableFrom(type) || int.class.isAssignableFrom(type)) {
                        value = Integer.parseInt(stringValue);
                    } else if (type.isEnum()) {
                        value = Enum.valueOf(type, stringValue);
                    }
                }
            }

            // set value into bean
            PropertyUtils.setProperty(destination, field, value);
        }
    }
    
    public static <K> void copyFields(K source, EntityExportExtra destination,
                                      Map<String, Function<K, Object>> customTransformers, String... fields)
            throws InvocationTargetException, IllegalAccessException, NoSuchMethodException {
        List<String> list = Lists.newArrayList(fields);
        copyFields(source, destination, customTransformers, list);
    }

    public static void setExtraField(EntityExportExtra destination, String field, Object value) {
        Map<String, Object> extra = destination.getExtras();
        if (extra == null) {
            extra = Maps.newHashMap();
            destination.setExtras(extra);
        }
        Object replaced = extra.put(field, value);
        if (replaced != null && log.isWarnEnabled()) {
            log.warn("Valeur remplacée (" + field + "): " + replaced);
        }
    }

    public static <R extends EntityExportExtra, K> void export(
            Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet, R model, K entity,
            EntityExportTabInfo tabInfo) throws CloneNotSupportedException, IllegalAccessException,
            NoSuchMethodException, InvocationTargetException {
        export(sheet, model, ImmutableList.of(entity), tabInfo);
    }

    public static <R extends EntityExportExtra, K> void export(
            Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet, R model, Iterable<K> entities,
            EntityExportTabInfo tabInfo) throws CloneNotSupportedException, IllegalAccessException,
            NoSuchMethodException, InvocationTargetException {

        // look for the list where to put export transformed entity
        List<R> list = (List<R>) sheet.get(tabInfo);

        // Do transform
        Set<String> tabFields = tabInfo.getExtraColumns().keySet();
        Map<String, Function<K, Object>> customTransformers = tabInfo.getCustomFormatters();
        for (K entity : entities) {
            R transformed = (R) model.clone();
            copyFields(entity, transformed, customTransformers, tabFields);
            list.add(transformed);
        }
    }

    /**
     * Add all possible tabInfo into map. This is needed to do empty export.
     * 
     * @param sheet sheet data
     * @param tabInfos tab infos
     */
    public static <R extends EntityExportExtra> void addAllBeanInfo(Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheet, EntityExportTabInfo... tabInfos) {
        for (EntityExportTabInfo tabInfo : tabInfos) {
            List<? extends EntityExportExtra> list = Lists.newArrayList();
            sheet.put(tabInfo, list);
        }
    }

    public static <K, M> Function<K, Object> ifNotNull(final String field, final Function<M, Object> realFunction) {
        Preconditions.checkNotNull(field);
        Preconditions.checkNotNull(realFunction);
        return new Function<K, Object>() {
            @Override
            public Object apply(K input) {
                M value;
                try {
                    value = (M) PropertyUtils.getProperty(input, field);
                } catch (Exception eee) {
                    throw new AgrosystTechnicalException("Invalid attribute", eee);
                }
                Object result = null;
                if (value != null) {
                    result = realFunction.apply(value);
                }
                return result;
            }
        };
    }

    /**
     * Return a iterable list of String representation of all enum element for {@code enumClass}.
     * 
     * A simplier way is Arrays.asList(XXX.values());
     * But this method could handle real translation later.
     * 
     * @param enumClazz enum class
     * @return list of string
     */
    public static <E extends Enum<E>> Iterable<String> allStringOf(Class<E> enumClazz) {
        return Iterables.transform(EnumSet.allOf(enumClazz), new Function<E, String>() {
            @Override
            public String apply(E input) {
                // XXX: toString() for now, could be replaced by translation later
                return input.name();
            }
        });
    }

    public static <E extends Enum<E>> Iterable<String> typeDephyToString() {
        return Iterables.transform(EnumSet.allOf(TypeDEPHY.class), new Function<TypeDEPHY, String>() {
            @Override
            public String apply(TypeDEPHY input) {
                String result = null;
                // XXX: toString() for now, could be replaced by translation later
                switch (input) {
                    case DEPHY_EXPE:
                        result = "DEPHY-EXPE";
                        break;
                    case DEPHY_FERME:
                        result = "DEPHY-FERME";
                        break;
                    case NOT_DEPHY:
                        result = "Hors DEPHY";
                        break;
                    default:
                        break;
                }
                return result;
            }
        });
    }

    public static final ValueFormatter<TypeDEPHY> TYPE_DEPHY_FORMATTER = new ValueFormatter<TypeDEPHY>() {
        @Override
        public String format(TypeDEPHY input) {
            String result = null;
            // XXX: toString() for now, could be replaced by translation later
            switch (input) {
                case DEPHY_EXPE:
                    result = "DEPHY-EXPE";
                    break;
                case DEPHY_FERME:
                    result = "DEPHY-FERME";
                    break;
                case NOT_DEPHY:
                    result = "Hors DEPHY";
                    break;
                default:
                    break;
            }
            return result;
        }
    };

}
