/*
 * #%L
 * Agrosyst :: Services
 * $Id: EntityImporter.java 4567 2014-11-26 14:16:20Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/export/EntityImporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */
package fr.inra.agrosyst.services.common.export;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;

/**
 * XLS importer utility.
 * 
 * @author Eric Chatellier
 */
public class EntityImporter {

    private static final Log log = LogFactory.getLog(EntityImporter.class);

    public <T extends EntityExportExtra> Map<EntityExportTabInfo, List<T>> importFromStream(InputStream is, Class<T> entityClass, EntityExportTabInfo... tabInfos) {

        Map<EntityExportTabInfo, List<T>> result = Maps.newHashMap();

        SimpleDateFormat dateFormat = new SimpleDateFormat(ExportUtils.DATE_FORMAT);

        try {
            HSSFWorkbook workbook = new HSSFWorkbook(is);

            // manage each sheet into tab info
            for (EntityExportTabInfo tabInfo : tabInfos) {
                HSSFSheet sheet = workbook.getSheet(tabInfo.getBeanTitle());

                // read header
                HSSFRow rowHeader = sheet.getRow(0);
                List<String> headers = Lists.newArrayList();
                for (int cellIndex = 0; cellIndex < rowHeader.getLastCellNum(); cellIndex++) {
                    HSSFCell cell = rowHeader.getCell(cellIndex);
                    String name = cell.getStringCellValue();
                    headers.add(name);
                }

                // read next lines
                List<T> entities = Lists.newArrayList();
                for (int rowIndex = 1; rowIndex < sheet.getPhysicalNumberOfRows(); rowIndex++) {
                    HSSFRow row = sheet.getRow(rowIndex);

                    T entity = entityClass.newInstance();
                    Map<String, Object> extras = Maps.newHashMap();
                    for (int cellIndex = 0; cellIndex < row.getLastCellNum(); cellIndex++) {
                        String headerName = headers.get(cellIndex);
                        HSSFCell cell = row.getCell(cellIndex);
                        String value = cell.getStringCellValue();

                        // ignore empty cells
                        if (Strings.isNullOrEmpty(value)) {
                            continue;
                        }

                        // search for bean associated property name
                        if (tabInfo.getCommonColumns().containsValue(headerName)) {
                            String property = tabInfo.getCommonColumns().getKey(headerName);

                            // Small hack to convert value into required type
                            Class type = PropertyUtils.getPropertyType(entity, property);
                            Object objectValue = value;
                            if (boolean.class.isAssignableFrom(type) || Boolean.class.isAssignableFrom(type)) {
                                objectValue = "oui".equalsIgnoreCase(value);
                            } else if (Date.class.isAssignableFrom(type)) {
                                objectValue = dateFormat.parse(value);
                            } else if (Double.class.isAssignableFrom(type) || double.class.isAssignableFrom(type)) {
                                objectValue = Double.parseDouble(value.replace(',', '.'));
                            } else if (Integer.class.isAssignableFrom(type) || int.class.isAssignableFrom(type)) {
                                objectValue = Integer.parseInt(value);
                            } else if (type.isEnum()) {
                                objectValue = Enum.valueOf(type, value);
                            }

                            // set value
                            PropertyUtils.setProperty(entity, property, objectValue);
                        } else if (tabInfo.getExtraColumns().containsValue(headerName)) {
                            String key = tabInfo.getExtraColumns().getKey(headerName);
                            extras.put(key, value);
                        } else {
                            throw new IllegalArgumentException("Unknown column " + headerName);
                        }
                    }
                    entity.setExtras(extras);
                    entities.add(entity);
                }
                result.put(tabInfo, entities);
            }

        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't read spread sheet from stream", ex);
        }
        
        return result;
    }
}
