package fr.inra.agrosyst.services.common.export;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EntityExporter.java 4567 2014-11-26 14:16:20Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/export/EntityExporter.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.poi.hssf.usermodel.DVConstraint;
import org.apache.poi.hssf.usermodel.HSSFDataValidation;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataValidation;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.ss.util.WorkbookUtil;
import org.nuiton.util.beans.BeanUtil;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;

/**
 * XLS exporter utility.
 *
 * @author Eric Chatellier
 */
public class EntityExporter {

    /**
     * Export map as single list data per xls sheet.
     *
     * @param sheetData map "Sheet Name", Data bean (one bean per line)
     * @return input on generated file
     */
    public InputStream exportAsXlsStream(Map<EntityExportTabInfo, List<? extends EntityExportExtra>> sheetData) {
        InputStream inputStream;

        try {
            // Get the workbook instance for XLS file
            HSSFWorkbook workbook = new HSSFWorkbook();

            CellStyle boldStyle = workbook.createCellStyle(); // bold
            HSSFFont font = workbook.createFont();
            font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
            boldStyle.setFont(font);

            SimpleDateFormat dateFormat = new SimpleDateFormat(ExportUtils.DATE_FORMAT);
            for (Map.Entry<EntityExportTabInfo, List<? extends EntityExportExtra>> sheetDatum : sheetData.entrySet()) {

                // Get first sheet from the workbook
                EntityExportTabInfo info = sheetDatum.getKey();
                HSSFSheet sheet = workbook.createSheet(info.getBeanTitle());

                List<? extends EntityExportExtra> values = sheetDatum.getValue();
                ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
                Map<String, String> declaredColumns = builder.putAll(info.getCommonColumns()).putAll(info.getExtraColumns()).build();
                Map<String, Iterable<String>> customDropDownList = info.getCustomDropDownList();

                {   // write headers
                    Row headerRow = sheet.createRow(0);
                    int column = 0;
                    for (Map.Entry<String, String> displayEntry : declaredColumns.entrySet()) {
                        String key = displayEntry.getKey();
                        Cell headerCell = headerRow.createCell(column);
                        headerCell.setCellValue(displayEntry.getValue());
                        headerCell.setCellStyle(boldStyle);

                        // add drop down list data validation if property support it
                        Iterable<String> dropDownList = customDropDownList.get(key);
                        if (dropDownList != null) {
                            int rowIndex = 0;
                            // create this into a hidden sheet (if not already exists)
                            String sheetName = "_" + key;
                            sheetName = WorkbookUtil.createSafeSheetName(sheetName);
                            if (workbook.getSheet(sheetName) == null) {
                                HSSFSheet sheetZonage = workbook.createSheet(sheetName);
                                int index = workbook.getSheetIndex(sheetName);
                                workbook.setSheetHidden(index, Workbook.SHEET_STATE_HIDDEN);
                                // add values into it
                                for (String value : dropDownList) {
                                    Row valueRow = sheetZonage.createRow(rowIndex);
                                    Cell valueCell = valueRow.createCell(0);
                                    valueCell.setCellValue(value);
                                    rowIndex++;
                                }
                            } else {
                                // le referentiel a déja été ajouté une fois, on suppose
                                // qu'il a la même taille et on le réutilise
                                rowIndex = Iterables.size(dropDownList);
                            }
                            
                            // add column validation
                            if (rowIndex >= 1) {
                                String range = sheetName + "!$A$1:$A$" + rowIndex;
                                DVConstraint dvConstraint = DVConstraint.createFormulaListConstraint(range);
                                CellRangeAddressList list = new CellRangeAddressList(1, -1, column, column); // -1 = all column
                                DataValidation dataValidation = new HSSFDataValidation(list, dvConstraint);
                                sheet.addValidationData(dataValidation);
                            }
                        }

                        column++;
                    }
                }

                for (int i = 0; i < values.size(); i++) {

                    EntityExportExtra entity = values.get(i);

                    // write data
                    Row row = sheet.createRow(i + 1); // + header
                    int column = 0;
                    for (final String columnName : declaredColumns.keySet()) {
                        Cell cell = row.createCell(column++);

                        Set<PropertyDescriptor> descriptors = BeanUtil.getDescriptors(
                                entity.getClass(),
                                Predicates.and(BeanUtil.IS_READ_DESCRIPTOR,
                                        new Predicate<PropertyDescriptor>() {
                                            @Override
                                            public boolean apply(PropertyDescriptor input) {
                                                return columnName.equals(input.getName());
                                            }
                                        }
                                )
                        );

                        Object value = null;
                        if (!descriptors.isEmpty()) {
                            Preconditions.checkState(descriptors.size() == 1,
                                    "Unexpected size for field " + entity.getClass().getName() + "#" + columnName + "; size=" + descriptors.size());
                            PropertyDescriptor descriptor = descriptors.iterator().next();
                            value = descriptor.getReadMethod().invoke(entity);
                        } else if (entity.getExtras() != null){
                            value = entity.getExtras().get(columnName);
                        }

                        if (value != null) {
                            if (value instanceof Boolean) {
                                if ((Boolean) value) {
                                    cell.setCellValue("Oui");
                                } else {
                                    cell.setCellValue("Non");
                                }
                            } else if (value instanceof Date){
                                cell.setCellValue(dateFormat.format(value));
                            } else {
                                cell.setCellValue(value.toString());
                            }
                        }
                    }
                }

                // autosize all columns
                for (int column = 0; column < declaredColumns.size(); column++) {
                    sheet.autoSizeColumn(column);
                }
            }

            // write file
            ByteArrayOutputStream out = new ByteArrayOutputStream(4096);
            workbook.write(out);
            out.close();
            inputStream = new ByteArrayInputStream(out.toByteArray());

        } catch (IOException ex) {
            throw new AgrosystTechnicalException("Can't write XLS file", ex);
        } catch (Exception ex) {
            throw new AgrosystTechnicalException("Can't generate file content", ex);
        }

        return inputStream;
    }
}
