package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ProjectionHelper.java 4948 2015-05-21 08:43:01Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/ProjectionHelper.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.AgrosystTopiaDaoSupplier;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.entities.Network;
import fr.inra.agrosyst.api.entities.NetworkTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.utils.DaoUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.support.TopiaJpaSupport;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class ProjectionHelper {

    private static final Log log = LogFactory.getLog(ProjectionHelper.class);

    protected static final String PROPERTY_GROWING_PLAN_ACTIVE = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_ACTIVE;
    protected static final String PROPERTY_GROWING_PLAN_DOMAIN_ACTIVE = GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_ACTIVE;
    protected static final String PROPERTY_GROWING_PLAN_CODE = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_CODE;
    protected static final String PROPERTY_GROWING_PLAN_DOMAIN_CODE = GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_CODE;

    protected AgrosystTopiaDaoSupplier topiaDaoSupplier;
    protected TopiaJpaSupport topiaJpaSupport;
    protected CacheService cacheService;

    public static void GET_NETWORK_SUB_QUERY(StringBuilder query, Map<String, Object> args, Set<Network> networks) {
        int index = 0;
        for (Network network : networks) {
            String key = "network" + index++;
            query.append(String.format(" OR :%s IN ELEMENTS ( gs.%s ) ", key, GrowingSystem.PROPERTY_NETWORKS));
            args.put(key, network);
        }
    }

    public ProjectionHelper(AgrosystTopiaDaoSupplier topiaDaoSupplier, TopiaJpaSupport topiaJpaSupport, CacheService cacheService) {
        this.topiaDaoSupplier = topiaDaoSupplier;
        this.topiaJpaSupport = topiaJpaSupport;
        this.cacheService = cacheService;
    }

    protected Set<String> networksProjection(Set<String> networkIds, String projection) {
        return networksProjection(networkIds, projection, null, null);
    }

    protected Set<String> networksProjection(final Set<String> networkIds, final String projection,
                                             final String filterProperty, final Object filterValue) {
        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                StringBuilder query = new StringBuilder(String.format("SELECT DISTINCT gs.%s FROM %s gs", projection, GrowingSystem.class.getName()));
                query.append(" WHERE ( 1 = 0");
                Map<String, Object> args = Maps.newLinkedHashMap();

                NetworkTopiaDao networkDAO = topiaDaoSupplier.getDao(Network.class, NetworkTopiaDao.class);
                Set<Network> networks = networkDAO.loadNetworksWithDescendantHierarchy(networkIds);
                GET_NETWORK_SUB_QUERY(query, args, networks);

                query.append(")");

                query.append(DaoUtils.andAttributeEquals("gs", filterProperty, args, filterValue));

                List<String> projectedList = topiaJpaSupport.findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(projectedList);

                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s", networkIds, projection, filterProperty, filterValue);
        Set<String> result = cacheService.get("networksProjection", key, loader);
        return result;
    }

    public Set<String> networksToGrowingSystems(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID);
        return result;
    }

    public Set<String> networksToGrowingSystemsCode(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, GrowingSystem.PROPERTY_CODE);
        return result;
    }

    public Set<String> networksToGrowingPlans(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_ID);
        return result;
    }

    public Set<String> networksToGrowingPlans(Set<String> networkIds, boolean active) {
        Set<String> result = networksProjection(networkIds, GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_ID, PROPERTY_GROWING_PLAN_ACTIVE, active);
        return result;
    }

    public Set<String> networksToGrowingPlansCode(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, PROPERTY_GROWING_PLAN_CODE);
        return result;
    }

    public Set<String> networksToGrowingPlansCode(Set<String> networkIds, boolean active) {
        Set<String> result = networksProjection(networkIds, PROPERTY_GROWING_PLAN_CODE, PROPERTY_GROWING_PLAN_ACTIVE, active);
        return result;
    }

    public Set<String> networksToDomains(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_DOMAIN_ID);
        return result;
    }

    public Set<String> networksToDomains(Set<String> networkIds, boolean active) {
        Set<String> result = networksProjection(networkIds, GrowingSystemTopiaDao.PROPERTY_GROWING_PLAN_DOMAIN_ID, PROPERTY_GROWING_PLAN_DOMAIN_ACTIVE, active);
        return result;
    }

    public Set<String> networksToDomainsCode(Set<String> networkIds, boolean active) {
        Set<String> result = networksProjection(networkIds, PROPERTY_GROWING_PLAN_DOMAIN_CODE, PROPERTY_GROWING_PLAN_DOMAIN_ACTIVE, true);
        return result;
    }

    public Set<String> networksToDomainsCode(Set<String> networkIds) {
        Set<String> result = networksProjection(networkIds, PROPERTY_GROWING_PLAN_DOMAIN_CODE);
        return result;
    }

    public LinkedHashSet<String> domainsToGrowingSystemsCode(Set<String> domainCodes) {
        Preconditions.checkArgument(domainCodes != null && !domainCodes.isEmpty());

        String domainCodeProperty = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE;

        StringBuilder query = new StringBuilder(String.format("SELECT distinct gs.%s FROM %s gs", GrowingSystem.PROPERTY_CODE, GrowingSystem.class.getName()));
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // domains
        if (domainCodes.size() == 1) {
            query.append(DaoUtils.andAttributeEquals("gs", domainCodeProperty, args, domainCodes.iterator().next()));
        } else {
            query.append(DaoUtils.andAttributeIn("gs", domainCodeProperty, args, domainCodes));
        }

        List<String> growingSystemCodes = topiaJpaSupport.findAll(query.toString(), args);
        LinkedHashSet<String> result = Sets.newLinkedHashSet(growingSystemCodes);

        return result;
    }

    public LinkedHashSet<String> growingPlansToGrowingSystemsCode(Set<String> growingPlanCodes) {
        Preconditions.checkArgument(growingPlanCodes != null && !growingPlanCodes.isEmpty());

        String growingPlanCodeProperty = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_CODE;

        StringBuilder query = new StringBuilder(String.format("SELECT distinct gs.%s FROM %s gs", GrowingSystem.PROPERTY_CODE, GrowingSystem.class.getName()));
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // domains
        if (growingPlanCodes.size() == 1) {
            query.append(DaoUtils.andAttributeEquals("gs", growingPlanCodeProperty, args, growingPlanCodes.iterator().next()));
        } else {
            query.append(DaoUtils.andAttributeIn("gs", growingPlanCodeProperty, args, growingPlanCodes));
        }

        List<String> growingSystemCodes = topiaJpaSupport.findAll(query.toString(), args);
        LinkedHashSet<String> result = Sets.newLinkedHashSet(growingSystemCodes);

        return result;
    }

    public Set<String> domainsToDomainCodes(final Set<String> domainIds, final boolean active) {

        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                StringBuilder query = new StringBuilder("SELECT DISTINCT d." + Domain.PROPERTY_CODE + " FROM " + Domain.class.getName() + " d WHERE 1 = 1");

                Map<String, Object> args = Maps.newLinkedHashMap();

                // domain TopiaIds
                if(domainIds == null) {
                    // nothing to add
                } else if (domainIds.size() == 1) {
                    query.append(DaoUtils.andAttributeEquals("d", Domain.PROPERTY_TOPIA_ID, args, domainIds));
                } else {
                    query.append(DaoUtils.andAttributeIn("d", Domain.PROPERTY_TOPIA_ID, args, domainIds));
                }

                // domain Active
                query.append(DaoUtils.andAttributeEquals("d", Domain.PROPERTY_ACTIVE, args, active));

                List<String> domainCodes = topiaJpaSupport.findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(domainCodes);
                return result;
            }
        };

        Set<String> result;
        if (domainIds == null) {
            try {
                result = loader.call();
            } catch (Exception ex) {
                throw new AgrosystTechnicalException("Unable to load domain's code",ex);
            }
        } else {
            String key = String.format("%s_%s_%s_%s", domainIds, Domain.PROPERTY_CODE, Domain.PROPERTY_TOPIA_ID, active);
            result = cacheService.get("domainsToDomainCodesProjection", key, loader);
        }

        return result;
    }

    public Set<String> growingPlansToDomainCodes(final Set<String> growingPlanTopiaIds, final boolean active) {
        Preconditions.checkArgument(growingPlanTopiaIds != null && !growingPlanTopiaIds.isEmpty());

        final String growingPlanToDomainCodeProperty = GrowingPlan.PROPERTY_DOMAIN + "." +  Domain.PROPERTY_CODE;
        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {

                StringBuilder query = new StringBuilder(String.format("SELECT distinct gp.%s FROM %s gp", growingPlanToDomainCodeProperty, GrowingPlan.class.getName()));
                query.append(" WHERE 1 = 1");
                Map<String, Object> args = Maps.newLinkedHashMap();

                // domains
                if (growingPlanTopiaIds.size() == 1) {
                    query.append(DaoUtils.andAttributeEquals("gp", GrowingPlan.PROPERTY_TOPIA_ID, args, growingPlanTopiaIds.iterator().next()));
                } else {
                    query.append(DaoUtils.andAttributeIn("gp", GrowingPlan.PROPERTY_TOPIA_ID, args, growingPlanTopiaIds));
                }

                query.append(DaoUtils.andAttributeEquals("gp", GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, active));

                List<String> domainCodes = topiaJpaSupport.findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(domainCodes);

                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s", growingPlanTopiaIds, growingPlanToDomainCodeProperty, Domain.PROPERTY_TOPIA_ID, active);
        Set<String> result = cacheService.get("growingPlansToDomainCodes", key, loader);
        return result;

    }

    public Set<String> growingSystemsToDomainCodes(final Set<String> growingSystemsIds, final boolean active) {
        Preconditions.checkArgument(growingSystemsIds != null && !growingSystemsIds.isEmpty());

        final String growingSystemToDomainCodeProperty = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." +  Domain.PROPERTY_CODE;


        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                StringBuilder query = new StringBuilder(String.format("SELECT distinct gs.%s FROM %s gs", growingSystemToDomainCodeProperty, GrowingSystem.class.getName()));
                query.append(" WHERE 1 = 1");
                Map<String, Object> args = Maps.newLinkedHashMap();

                // domains
                if (growingSystemsIds.size() == 1) {
                    query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_TOPIA_ID, args, growingSystemsIds.iterator().next()));
                } else {
                    query.append(DaoUtils.andAttributeIn("gs", GrowingSystem.PROPERTY_TOPIA_ID, args, growingSystemsIds));
                }

                query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, active));

                List<String> domainCodes = topiaJpaSupport.findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(domainCodes);

                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s", growingSystemsIds, growingSystemToDomainCodeProperty, Domain.PROPERTY_TOPIA_ID, active);
        Set<String> result = cacheService.get("networksProjection", key, loader);
        return result;
    }

    public Set<String> campaignsToDomainCodes(final Set<Integer> campaigns, final boolean active) {
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());

        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                StringBuilder query = new StringBuilder(String.format("SELECT distinct d.%s FROM %s d", Domain.PROPERTY_CODE, Domain.class.getName()));
                query.append(" WHERE 1 = 1");
                Map<String, Object> args = Maps.newLinkedHashMap();

                // domains
                if (campaigns.size() == 1) {
                    query.append(DaoUtils.andAttributeEquals("d", Domain.PROPERTY_CAMPAIGN, args, campaigns.iterator().next()));
                } else {
                    query.append(DaoUtils.andAttributeIn("d", Domain.PROPERTY_CAMPAIGN, args, campaigns));
                }

                query.append(DaoUtils.andAttributeEquals("d", Domain.PROPERTY_ACTIVE, args, active));

                List<String> domainCodes = topiaJpaSupport.findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(domainCodes);

                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s", campaigns, Domain.PROPERTY_CODE, Domain.PROPERTY_TOPIA_ID, active);
        Set<String> result = cacheService.get("networksProjection", key, loader);
        return result;
    }
}
