package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PricesServiceImpl.java 4970 2015-06-09 11:56:01Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/PricesServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import fr.inra.agrosyst.api.entities.referential.RefFertiMinUNIFA;
import fr.inra.agrosyst.api.entities.referential.Referentials;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.PriceImpl;
import fr.inra.agrosyst.api.entities.PriceTopiaDao;
import fr.inra.agrosyst.api.entities.PriceType;
import fr.inra.agrosyst.api.entities.PriceUnit;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.services.common.PricesService;
import fr.inra.agrosyst.api.services.common.ProductPrices;
import fr.inra.agrosyst.api.services.practiced.DuplicateCropCyclesContext;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class PricesServiceImpl extends AbstractAgrosystService implements PricesService {

    protected PriceTopiaDao priceTopiaDao;
    protected ZoneTopiaDao zoneTopiaDao;

    public void setPriceTopiaDao(PriceTopiaDao priceTopiaDao) {
        this.priceTopiaDao = priceTopiaDao;
    }

    public void setZoneTopiaDao(ZoneTopiaDao zoneTopiaDao) {
        this.zoneTopiaDao = zoneTopiaDao;
    }

    @Override
    public ProductPrices computePricesIndication(Price filter,
                                                 String campaigns,
                                                 String excludeDomainIdRaw,
                                                 String excludeZoneId,
                                                 String excludePracticedSystemId) {
        Preconditions.checkArgument(filter != null, "Requires a filter");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(campaigns), "Requires a list of campaigns");

        Set<Integer> campaignsSet = Sets.newLinkedHashSet();
        Iterables.addAll(campaignsSet, CommonService.GET_CAMPAIGNS_SET.apply(campaigns));
        Preconditions.checkArgument(!campaignsSet.isEmpty(), "At least one campaign must be set");

        String excludeDomainId = excludeDomainIdRaw;
        if (!Strings.isNullOrEmpty(excludeZoneId)) {
            Zone zone = zoneTopiaDao.forTopiaIdEquals(excludeZoneId).findAny();
            Domain domain = zone.getPlot().getDomain();
            excludeDomainId = domain.getTopiaId();
        }

        return priceTopiaDao.computePriceIndication(filter, campaignsSet, excludeDomainId, excludePracticedSystemId);
    }

    @Override
    public void updatePrices(List<Price> prices, Domain domain, PracticedSystem practicedSystem) {
        if (prices == null) {
            prices = Collections.emptyList();
        }

        List<Price> existingPrices = getPrices0(Entities.GET_TOPIA_ID.apply(domain), Entities.GET_TOPIA_ID.apply(practicedSystem), null);
        Map<String, Price> existingPricesIndex = Maps.uniqueIndex(existingPrices, Prices.GET_PRICE_KEY);

        for (Price price : prices) {
            String priceKey = Prices.GET_PRICE_KEY.apply(price);
            Price priceEntity = existingPricesIndex.get(priceKey);
            if (priceEntity == null) {
                priceEntity = priceTopiaDao.newInstance();
                existingPrices.add(priceEntity);
                priceEntity.setType(price.getType());
                priceEntity.setObjectId(StringUtils.stripAccents(price.getObjectId()));
            }

            priceEntity.setCategory(price.getCategory());
            priceEntity.setDisplayName(price.getDisplayName());
            priceEntity.setPrice(price.getPrice());
            priceEntity.setPriceUnit(price.getPriceUnit());
            priceEntity.setSourceUnit(price.getSourceUnit());
            priceEntity.setSourceUnitLabel(price.getSourceUnitLabel());
        }

        for (Price priceEntity : existingPrices) {
            priceEntity.setDomain(domain);
            priceEntity.setPracticedSystem(practicedSystem);
            if (priceEntity.isPersisted()) {
                priceTopiaDao.update(priceEntity);
            } else {
                priceTopiaDao.create(priceEntity);
            }
        }
    }

    @Override
    public List<Price> getDomainPrices(String domainId, List<String> objectIds) {
        List<Price> result = getPrices0(domainId, null, objectIds);

        if (result.isEmpty()) {
            Iterables.addAll(result, newDefaultPrices());
        }

        return result;
    }

    @Override
    public List<Price> getPracticedSystemPrices(String practicedSystemId) {
        List<Price> result = getPrices0(null, practicedSystemId, null);

        if (result.isEmpty()) {
            Iterables.addAll(result, newDefaultPrices());
        }

        return result;
    }

    protected List<Price> getPrices0(String domainTopiaId, String practicedSystemId, List<String> objectIds) {
        List<Price> result;

        // If no domainId and no practicedSystemId specified, return an empty list
        if (Strings.isNullOrEmpty(domainTopiaId) && Strings.isNullOrEmpty(practicedSystemId)) {
            result = Lists.newLinkedList();
        } else {
            result = priceTopiaDao.getPrices0(domainTopiaId, practicedSystemId, objectIds);
        }

        return result;
    }

    protected List<Price> newDefaultPrices() {
        List<Price> result = Lists.newArrayListWithCapacity(1);

        // XXX AThimel 08/11/13 Move to conf ?
        // Maybe merge with fr.inra.agrosyst.services.practiced.PracticedSystemServiceImpl.newDefaultPrices ?
        PriceImpl price = new PriceImpl();
        price.setPrice(null);
        price.setCategory("NONE");
        price.setPriceUnit(PriceUnit.EURO_L);
        price.setDisplayName("Carburant");
        price.setType(PriceType.FUEL);
        result.add(price);

        return result;
    }

    @Override
    public void duplicatePracticedSystemPrices(DuplicateCropCyclesContext duplicateContext) {

        // duplication des prix seulement ceux qui concerne le pratiqué
        List<Price> prices = priceTopiaDao.forPracticedSystemEquals(duplicateContext.getPracticedSystem()).findAll();
        Binder<Price, Price> binder = BinderFactory.newBinder(Price.class);
        for (Price price : prices) {
            Price priceClone = priceTopiaDao.newInstance();
            binder.copyExcluding(price, priceClone,
                    Price.PROPERTY_TOPIA_ID,
                    Price.PROPERTY_TOPIA_CREATE_DATE,
                    Price.PROPERTY_TOPIA_VERSION,
                    Price.PROPERTY_PRACTICED_SYSTEM);

            priceClone.setPracticedSystem(duplicateContext.getPracticedSystemClone());

            // persist clone
            priceTopiaDao.create(priceClone);
        }
    }

    @Override
    public String getPriceFertiMinUnifaNaturalId (RefFertiMinUNIFA input) {
        String result = StringUtils.stripAccents(Referentials.GET_FERTI_MIN_UNIFA_NATURAL_ID_WITH_NULL.apply(input));
        return result;
    }
}
