package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EntityUsageService.java 4391 2014-10-03 22:06:47Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/EntityUsageService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.CroppingPlanEntryTopiaDao;
import fr.inra.agrosyst.api.entities.CroppingPlanSpecies;
import fr.inra.agrosyst.api.entities.CroppingPlanSpeciesTopiaDao;
import fr.inra.agrosyst.api.entities.DomainTopiaDao;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.entities.ToolsCoupling;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.ZoneTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNodeTopiaDao;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedSpeciesStade;
import fr.inra.agrosyst.api.entities.practiced.PracticedSpeciesStadeTopiaDao;
import fr.inra.agrosyst.api.services.common.UsageList;
import fr.inra.agrosyst.api.services.domain.CroppingPlanEntryDto;
import fr.inra.agrosyst.api.services.domain.CroppingPlans;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * Service has no interface because it should be used only internally in other services
 *
 * @author Arnaud Thimel (Code Lutin)
 */
public class EntityUsageService extends AbstractAgrosystService {

    private static final Log log = LogFactory.getLog(EntityUsageService.class);

    protected DomainTopiaDao domainTopiaDao;
    protected ZoneTopiaDao zoneTopiaDao;
    protected CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao;
    protected CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao;
    protected PracticedCropCycleNodeTopiaDao practicedCropCycleNodeTopiaDao;
    protected PracticedSpeciesStadeTopiaDao practicedSpeciesStadeTopiaDao;


    public void setPracticedSpeciesStadeTopiaDao(PracticedSpeciesStadeTopiaDao practicedSpeciesStadeTopiaDao) {
        this.practicedSpeciesStadeTopiaDao = practicedSpeciesStadeTopiaDao;
    }

    public void setPracticedCropCycleNodeTopiaDao(PracticedCropCycleNodeTopiaDao practicedCropCycleNodeTopiaDao) {
        this.practicedCropCycleNodeTopiaDao = practicedCropCycleNodeTopiaDao;
    }

    public void setCroppingPlanSpeciesTopiaDao(CroppingPlanSpeciesTopiaDao croppingPlanSpeciesTopiaDao) {
        this.croppingPlanSpeciesTopiaDao = croppingPlanSpeciesTopiaDao;
    }

    public void setZoneTopiaDao(ZoneTopiaDao zoneTopiaDao) {
        this.zoneTopiaDao = zoneTopiaDao;
    }

    public void setCroppingPlanEntryTopiaDao(CroppingPlanEntryTopiaDao croppingPlanEntryTopiaDao) {
        this.croppingPlanEntryTopiaDao = croppingPlanEntryTopiaDao;
    }

    public void setDomainTopiaDao(DomainTopiaDao domainTopiaDao) {
        this.domainTopiaDao = domainTopiaDao;
    }

    public UsageList<ToolsCoupling> getToolsCouplingUsageList(List<ToolsCoupling> toolsCouplings, Integer campaign) {
        Map<String, Boolean> map;
        if (toolsCouplings != null && !toolsCouplings.isEmpty()){
            Iterable<String> ids = Iterables.transform(toolsCouplings, Entities.GET_TOPIA_ID);
            String stCampaign = "%" + campaign + "%";
            map = getToolsCouplingUsageMap(ids, stCampaign);
        } else {
            map = Maps.newHashMap();
        }

        UsageList<ToolsCoupling> result = UsageList.of(toolsCouplings, map);
        return result;
    }

    public Map<String, Boolean> getToolsCouplingUsageMap(Iterable<String> toolsCouplingIds, String campaign) {
        Map<String, Boolean> result = Maps.newHashMap();

        Boolean forTCUsedForEffectiveInterventions = true;
        Boolean forTCUsedForPracticedCropCycleNodesAndConnections = true;
        Boolean forTCUsedForPracticedInterventionPhases = true;

//        long timeStart = System.currentTimeMillis();
        Map<String, Long> forTCUsedForEffectiveInterventionResult = forTCUsedForEffectiveInterventions ? domainTopiaDao.getTCUsedForEffectiveInterventions(toolsCouplingIds) : null;
//        if (log.isDebugEnabled()) {log.debug("forTCUsedForEffectiveInterventionResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forTCUsedForPracticedCropCycleNodesAndConnectionsResult = forTCUsedForPracticedCropCycleNodesAndConnections ? domainTopiaDao.getTCUsedForPracticedInterventionNodesAndConnections(toolsCouplingIds, campaign) : null;
//        if (log.isDebugEnabled()) {log.debug("forTCUsedForPracticedCropCycleNodesAndConnectionsResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forTCUsedForPracticedInterventionPhasesResult = forTCUsedForPracticedInterventionPhases ? domainTopiaDao.getTCUsedForPracticedInterventionPhases(toolsCouplingIds, campaign) : null;
//        if (log.isDebugEnabled()) {log.debug("forTCUsedForPracticedInterventionPhasesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

        for (String id : toolsCouplingIds) {
            Long affected = 0L;
            affected += forTCUsedForEffectiveInterventionResult != null && forTCUsedForEffectiveInterventionResult.get(id) != null ? forTCUsedForEffectiveInterventionResult.get(id) : 0;
            affected += forTCUsedForPracticedCropCycleNodesAndConnectionsResult != null && forTCUsedForPracticedCropCycleNodesAndConnectionsResult.get(id) != null ? forTCUsedForPracticedCropCycleNodesAndConnectionsResult.get(id) : 0;
            affected += forTCUsedForPracticedInterventionPhasesResult != null && forTCUsedForPracticedInterventionPhasesResult.get(id) != null ? forTCUsedForPracticedInterventionPhasesResult.get(id) : 0;
            result.put(id, affected>0);
        }
        return result;
    }

    public UsageList<Zone> getZonesUsageList(List<Zone> zones) {
        Map<String, Boolean> map;
        if (!zones.isEmpty()) {
            Iterable<String> ids = Iterables.transform(zones, Entities.GET_TOPIA_ID);
            map = getZonesUsageMap(ids);
        } else {
            map = Maps.newHashMap();
        }
        UsageList<Zone> result = UsageList.of(zones, map);
        return result;
    }

    public Map<String, Boolean> getZonesUsageMap(Iterable<String> zonesIds) {
        Map<String, Boolean> result = Maps.newHashMap();
        Boolean forZonesUsedForPerformances = true;
        Boolean forZonesUsedForMeasurementSessions = true;
        Boolean forZonesUsedForEffectivePerennialCropCycles = true;
        Boolean forZonesUsedForEffectiveSeasonalCropCycles = true;

//        long timeStart = System.currentTimeMillis();
        Map<String, Long> getZonesUsedForPerformancesResult = forZonesUsedForPerformances ? zoneTopiaDao.getZonesUsedForPerformances(zonesIds) : null;
//        if (log.isDebugEnabled()) {log.debug("getZonesUsedForPerformancesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> getZonesUsedForMeasurementSessionsResult = forZonesUsedForMeasurementSessions ? zoneTopiaDao.getZonesUsedForMesurementSessions(zonesIds) : null;
//        if (log.isDebugEnabled()) {log.debug("getZonesUsedForMeasurementSessionsResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> getZonesUsedForEffectivePerennialCropCyclesResult = forZonesUsedForEffectivePerennialCropCycles ? zoneTopiaDao.getZonesUsedForEffectivePerennialCropCycles(zonesIds) : null;
//        if (log.isDebugEnabled()) {log.debug("getZonesUsedForEffectivePerennialCropCyclesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> getZonesUsedForEffectiveSeasonalCropCyclesResult = forZonesUsedForEffectiveSeasonalCropCycles ? zoneTopiaDao.getZonesUsedForEffectiveSeasonalCropCycles(zonesIds) : null;
//        if (log.isDebugEnabled()) {log.debug("getZonesUsedForEffectiveSeasonalCropCyclesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

        for (String id : zonesIds) {
            Long affected = 0L;
            affected += getZonesUsedForPerformancesResult != null && getZonesUsedForPerformancesResult.get(id) != null ? getZonesUsedForPerformancesResult.get(id) : 0;
            affected += getZonesUsedForMeasurementSessionsResult != null && getZonesUsedForMeasurementSessionsResult.get(id) != null ? getZonesUsedForMeasurementSessionsResult.get(id) : 0;
            affected += getZonesUsedForEffectivePerennialCropCyclesResult != null && getZonesUsedForEffectivePerennialCropCyclesResult.get(id) != null ? getZonesUsedForEffectivePerennialCropCyclesResult.get(id) : 0;
            affected += getZonesUsedForEffectiveSeasonalCropCyclesResult != null && getZonesUsedForEffectiveSeasonalCropCyclesResult.get(id) != null ? getZonesUsedForEffectiveSeasonalCropCyclesResult.get(id) : 0;
            result.put(id, affected>0);
        }

        return result;
    }

    public UsageList<CroppingPlanEntryDto> getCroppingPlanEntryDtoAndUsage(List<CroppingPlanEntry> croppingPlanEntries, Integer campaign) {
        Map<String, Boolean> map;
        List<CroppingPlanEntryDto> croppingPlanEntryDtos;
        if (croppingPlanEntries != null && !croppingPlanEntries.isEmpty()) {
            Iterable<String> ids = Iterables.transform(croppingPlanEntries, Entities.GET_TOPIA_ID);
            String stCampaign = "%" + campaign + "%";
            map = getCroppingPlanEntryUsageMap(ids, stCampaign);
            croppingPlanEntryDtos = Lists.transform(croppingPlanEntries, CroppingPlans.CROPPING_PLAN_ENTRY_TO_DTO);
        } else {
            map = Maps.newHashMap();
            croppingPlanEntryDtos = Lists.newArrayList();
        }

        UsageList<CroppingPlanEntryDto> result = UsageList.of(croppingPlanEntryDtos, map);
        return result;
    }

    public Map<String, Boolean> getCroppingPlanEntryUsageMap(Iterable<String> croppingPlanEntriesIds, String campaign) {
        Map<String, Boolean> result = Maps.newHashMap();

        //TODO david 19/03/2014 il faudra pouvoir détecter un changement sur une culture et recalculer l'ensemble des sous requêtes concernées.
        // set it to true if target changed.
        Boolean forStrategies = true;
        Boolean forMeasurementSessions = true;
        Boolean forCPEUsedForEffectiveCropCycleNode = true;
        Boolean forEffectiveCropCycleConnections = true;
        Boolean forPracticedCropCycleConnections = true;
        Boolean forPracticedCropCycleNodes = true;
        Boolean forEffectivePerennialCropCycles = true;
        Boolean forPracticedPerennialCropCycles = true;

//        long timeStart = System.currentTimeMillis();
        Map<String, Long> forStrategiesResult = forStrategies? croppingPlanEntryTopiaDao.getCPEUsedForStrategies(croppingPlanEntriesIds): null;
//        if (log.isDebugEnabled()) {log.debug("forStrategiesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forMeasurementSessionsResult = forMeasurementSessions ? croppingPlanEntryTopiaDao.getCPEUsedForMesurementSessions(croppingPlanEntriesIds): null;
//        if (log.isDebugEnabled()) {log.debug("forMeasurementSessionsResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long>forCPEUsedForEffectiveCropCycleNodeResult = forCPEUsedForEffectiveCropCycleNode ? croppingPlanEntryTopiaDao.getCPEUsedForEffectiveCropCycleNode(croppingPlanEntriesIds): null;
//        if (log.isDebugEnabled()) {log.debug("forCPEUsedForEffectiveCropCycleNode:" + (System.currentTimeMillis() - timeStart) + " ms");}


//        timeStart = System.currentTimeMillis();
        Map<String, Long> forEffectiveCropCycleConnectionsResult = forEffectiveCropCycleConnections ? croppingPlanEntryTopiaDao.getCPEUsedForEffectiveCropCycleConnections(croppingPlanEntriesIds): null;
//        if (log.isDebugEnabled()) {log.debug("forEffectiveCropCycleConnectionsResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forPracticedCropCycleConnectionsResult = forPracticedCropCycleConnections ? croppingPlanEntryTopiaDao.getCPEUsedForPracticedCropCycleConnections(croppingPlanEntriesIds, campaign) : null;
//        if (log.isDebugEnabled()) {log.debug("forPracticedCropCycleConnectionsResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forPracticedCropCycleNodesResult = forPracticedCropCycleNodes ? croppingPlanEntryTopiaDao.getCPEUsedForPracticedCropCycleNodes(croppingPlanEntriesIds, campaign): null;
//        if (log.isDebugEnabled()) {log.debug("forPracticedCropCycleNodesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forEffectivePerennialCropCyclesResult = forEffectivePerennialCropCycles ? croppingPlanEntryTopiaDao.getCPEUsedForEffectivePerenialCropCycles(croppingPlanEntriesIds): null;
//        if (log.isDebugEnabled()) {log.debug("forEffectivePerennialCropCyclesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}

//        timeStart = System.currentTimeMillis();
        Map<String, Long> forPracticedPerenialCropCyclesResult = forPracticedPerennialCropCycles ? croppingPlanEntryTopiaDao.getCPEUsedPracticedPerenialCropCycles(croppingPlanEntriesIds, campaign): null;
//        if (log.isDebugEnabled()) {log.debug("forPracticedPerennialCropCyclesResult:" + (System.currentTimeMillis() - timeStart) + " ms");}


        for (String id : croppingPlanEntriesIds) {
            Long affected = 0L;
            affected += forStrategiesResult != null && forStrategiesResult.get(id) != null ? forStrategiesResult.get(id) : 0;
            affected += forCPEUsedForEffectiveCropCycleNodeResult != null && forCPEUsedForEffectiveCropCycleNodeResult.get(id) != null ? forCPEUsedForEffectiveCropCycleNodeResult.get(id) : 0;
            affected += forMeasurementSessionsResult != null && forMeasurementSessionsResult.get(id) != null ? forMeasurementSessionsResult.get(id) : 0;
            affected += forEffectiveCropCycleConnectionsResult != null && forEffectiveCropCycleConnectionsResult.get(id) != null ? forEffectiveCropCycleConnectionsResult.get(id) : 0;
            affected += forPracticedCropCycleConnectionsResult != null && forPracticedCropCycleConnectionsResult.get(id) != null ? forPracticedCropCycleConnectionsResult.get(id) : 0;
            affected += forPracticedCropCycleNodesResult != null && forPracticedCropCycleNodesResult.get(id) != null ? forPracticedCropCycleNodesResult.get(id) : 0;
            affected += forEffectivePerennialCropCyclesResult != null && forEffectivePerennialCropCyclesResult.get(id) != null ? forEffectivePerennialCropCyclesResult.get(id) : 0;
            affected += forPracticedPerenialCropCyclesResult != null && forPracticedPerenialCropCyclesResult.get(id) != null ? forPracticedPerenialCropCyclesResult.get(id) : 0;
            result.put(id, affected > 0) ;
        }
        return result;
    }

    public Map<String, Boolean> getCroppingPlanSpeciesUsageMap(List<CroppingPlanSpecies> croppingPlanSpecies, Integer campaign, String domainCode) {
        Map<String, Boolean> map;

        if (croppingPlanSpecies != null && !croppingPlanSpecies.isEmpty()) {
            map = Maps.newHashMapWithExpectedSize(croppingPlanSpecies.size());

            Iterable<String> croppingPlanSpeciesIds = Iterables.transform(croppingPlanSpecies, Entities.GET_TOPIA_ID);
            String stCampaign = "%"+ campaign +"%";

            //TODO david 19/03/2014 il faudra pouvoir détecter un changement sur une culture et recalculer l'ensemble des sous requêtes concernées.
            // set it to true if target changed.
            Boolean forPracticedPerennialCropCycleStades = true;
            Boolean forPracticedCropCycleNodesAndConnections = true;
            Boolean forPracticedCropCycleSpecies = true;
            Boolean forEffectiveSpeciesStades = true;
            Boolean forEffectiveCropCycleSpecies = true;
            Boolean forMeasures = true;

//            long timeStart = System.currentTimeMillis();
            Map<String, Long> forPracticedPerennialCropCycleStadesResults = forPracticedPerennialCropCycleStades ? croppingPlanSpeciesTopiaDao.getCPSpeciesUsedForPracticedPerennialCropCycleStades(croppingPlanSpeciesIds, stCampaign) : null;
//            if (log.isDebugEnabled()) {log.debug("forPracticedPerennialCropCycleStadesResults:" + (System.currentTimeMillis() - timeStart) + " ms");}

//            timeStart = System.currentTimeMillis();
            Map<String, Long> forPracticedCropCycleNodesAndConnectionsResults = forPracticedCropCycleNodesAndConnections ? getCPSpeciesUsedForPracticedCropCycleNodesAndConnections(croppingPlanSpeciesIds, domainCode, stCampaign) : null;
//            if (log.isDebugEnabled()) {log.debug("forPracticedCropCycleNodesAndConnectionsResults:" + (System.currentTimeMillis() - timeStart) + " ms");}

//            timeStart = System.currentTimeMillis();
            Map<String, Long> forPracticedCropCycleSpeciesResults = forPracticedCropCycleSpecies ? croppingPlanSpeciesTopiaDao.getCPSpeciesUsedForPracticedCropCycleSpecies(croppingPlanSpeciesIds, stCampaign) : null;
//            if (log.isDebugEnabled()) {log.debug("forPracticedCropCycleSpeciesResults:" + (System.currentTimeMillis() - timeStart)  + " ms");}

//            timeStart = System.currentTimeMillis();
            Map<String, Long> forEffectiveSpeciesStadesResults = forEffectiveSpeciesStades ? croppingPlanSpeciesTopiaDao.getCPSpeciesUsedForEffectiveSpeciesStades(croppingPlanSpeciesIds) : null;
//            if (log.isDebugEnabled()) {log.debug("forEffectiveSpeciesStadesResults:" + (System.currentTimeMillis() - timeStart) + " ms");}

//            timeStart = System.currentTimeMillis();
            Map<String, Long> forEffectiveCropCycleSpeciesResults = forEffectiveCropCycleSpecies ? croppingPlanSpeciesTopiaDao.getCPSpeciesUsedForEffectiveCropCycleSpecies(croppingPlanSpeciesIds) : null;
//            if (log.isDebugEnabled()) {log.debug("forEffectiveCropCycleSpeciesResults:" + (System.currentTimeMillis() - timeStart) + " ms");}

//            timeStart = System.currentTimeMillis();
            Map<String, Long> forMeasuresResults = forMeasures ? croppingPlanSpeciesTopiaDao.getCPSpeciesUsedForMeasures(croppingPlanSpeciesIds) : null;
//            if (log.isDebugEnabled()) {log.debug("forMeasuresResults:" + (System.currentTimeMillis() - timeStart) + " ms");}

            for (String id : croppingPlanSpeciesIds) {
                Long affected = 0L;
                affected += forPracticedPerennialCropCycleStadesResults != null && forPracticedPerennialCropCycleStadesResults.get(id) != null ? forPracticedPerennialCropCycleStadesResults.get(id) : 0;
                affected += forPracticedCropCycleNodesAndConnectionsResults != null && forPracticedCropCycleNodesAndConnectionsResults.get(id) != null ? forPracticedCropCycleNodesAndConnectionsResults.get(id) : 0;
                affected += forPracticedCropCycleSpeciesResults != null &&  forPracticedCropCycleSpeciesResults.get(id) != null ? forPracticedCropCycleSpeciesResults.get(id) : 0;
                affected += forEffectiveSpeciesStadesResults != null && forEffectiveSpeciesStadesResults.get(id) != null ? forEffectiveSpeciesStadesResults.get(id) : 0;
                affected += forEffectiveCropCycleSpeciesResults != null && forEffectiveCropCycleSpeciesResults.get(id) != null ? forEffectiveCropCycleSpeciesResults.get(id) : 0;
                affected += forMeasuresResults != null && forMeasuresResults.get(id) != null ? forMeasuresResults.get(id) : 0;
                map.put(id, affected > 0) ;
            }
        } else {
            map = Maps.newHashMap();
        }
        return map;
    }

    protected Map<String, Long> getCPSpeciesUsedForPracticedCropCycleNodesAndConnections(Iterable<String> croppingPlanSpeciesIds, String domainCode, String campaign) {

        Map<String, Long> result = Maps.newHashMap();
        // Find all nodes related to a domain with same domainCode as the given one.
        //   and practicedSystem matching the given campaign
        List<PracticedCropCycleNode> practicedCropCycleNodes = croppingPlanSpeciesTopiaDao.getPracticedCropCycleNodeForCampaignAndDomainCode(domainCode, campaign);


        // intervention where crops are used in stades
        if(practicedCropCycleNodes != null && practicedCropCycleNodes.size() > 0) {
            // find speciesStades where given species are used
            //    speciesStades: map of speciesStades and it's related Species TopiaId
            Map<PracticedSpeciesStade, String> speciesStades = croppingPlanSpeciesTopiaDao.getSpeciesPracticedSpeciesStades(croppingPlanSpeciesIds);
            if (speciesStades != null && speciesStades.size()>0) {

                // find intervention related to the given nodes
                List<PracticedIntervention> practicedInterventions = croppingPlanSpeciesTopiaDao.practicedInterventionsForSources(practicedCropCycleNodes);
                practicedInterventions.addAll(croppingPlanSpeciesTopiaDao.practicedInterventionsForTargets(practicedCropCycleNodes));
                Set<PracticedIntervention> practicedInterventionSet = Sets.newHashSet(practicedInterventions);

                // find practiced stades related to the interventions
                Set<PracticedSpeciesStade> speciesStadesForGivenCampaign = Sets.newHashSet();
                if (practicedInterventionSet != null) {
                    for (PracticedIntervention practicedIntervention : practicedInterventionSet) {
                        if (practicedIntervention.getSpeciesStades() != null) {
                            speciesStadesForGivenCampaign.addAll(ListUtils.removeAll(practicedIntervention.getSpeciesStades(), speciesStades.values()));
                        }
                    }
                }

                // set 1 for species is used into speciesStade otherwise 0
                for (PracticedSpeciesStade practicedSpeciesStade : speciesStadesForGivenCampaign) {
                    result.put(speciesStades.get(practicedSpeciesStade), 1L);
                }
                for (String speciesId : croppingPlanSpeciesIds) {
                    if (result.get(speciesId) == null) {
                        result.put(speciesId, 0L);
                    }
                }
            }
        }
        else {
            // No species are concerned
            for(String speciesId: croppingPlanSpeciesIds) {
                result.put(speciesId, 0L);
            }
        }
        return result;
    }

}
