package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: EmailService.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/EmailService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.List;

import javax.activation.DataSource;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.util.ByteArrayDataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.EmailAttachment;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.HtmlEmail;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class EmailService extends AbstractAgrosystService implements AgrosystService {

    private static final Log log = LogFactory.getLog(EmailService.class);

    protected HtmlEmail newEmptyEmail() {
        HtmlEmail newEmptyEmail = new HtmlEmail();
        newEmptyEmail.setHostName(getConfig().getSmtpHost());
        newEmptyEmail.setSmtpPort(getConfig().getSmtpPort());
        newEmptyEmail.setCharset(Charsets.UTF_8.name());
        try {
            newEmptyEmail.setFrom(getConfig().getEmailFrom());
        } catch (EmailException e) {
            throw new AgrosystTechnicalException("Unable to initialize email template", e);
        }
        return newEmptyEmail;

    }

    protected void sendEmail(String subject, String to, List<String> ccs, List<String> bccs, String plainBody, String htmlBody, byte[] screenshotData) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(subject));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(to));
        Preconditions.checkArgument(!Strings.isNullOrEmpty(plainBody) || !Strings.isNullOrEmpty(htmlBody));

        HtmlEmail email = newEmptyEmail();

        // subject
        email.setSubject(subject);

        try {
            // to
            InternetAddress internetAddressTo = new InternetAddress(to);
            email.setTo(Arrays.asList(internetAddressTo));

            // cc
            if (ccs != null) {
                List<InternetAddress> ccList = Lists.newArrayList();
                for (String cc : ccs) {
                    InternetAddress internetAddress = new InternetAddress(cc);
                    ccList.add(internetAddress);
                }
                email.setCc(ccList);
            }

            // bcc
            if (bccs != null) {
                List<InternetAddress> bccList = Lists.newArrayList();
                for (String cc : bccs) {
                    InternetAddress internetAddress = new InternetAddress(cc);
                    bccList.add(internetAddress);
                }
                email.setBcc(bccList);
            }

            // body
            if (plainBody != null) {
                email.setTextMsg(plainBody);
            }
            if (htmlBody != null) {
                email.setHtmlMsg(htmlBody);
            }

            // screenshot
            if (screenshotData != null) {
                DataSource ds = new ByteArrayDataSource(screenshotData, "image/jpeg");
                email.attach(ds, "screenshot.jpg", null, EmailAttachment.ATTACHMENT);
            }

            if (log.isInfoEnabled()) {
                String format = "Will send e-mail to '%s' with subject '%s'";
                log.info(String.format(format, email.getToAddresses(), email.getSubject()));
            }

            email.send();
        } catch (AddressException ae) {
            throw new AgrosystTechnicalException("Configured address is not usable: " + to, ae);
        } catch (EmailException ee) {
            throw new AgrosystTechnicalException("Unable to send email", ee);
        }
    }

    public void sendPasswordReminder(AgrosystUser user, String token) {
        String applicationBaseUrl = getConfig().getApplicationBaseUrl();
        String reminderUrl = String.format("%s/auth/retrieve-password-input.action?userId=%s&token=%s", applicationBaseUrl, user.getTopiaId(), token);

        StringBuilder body = new StringBuilder(String.format("Bonjour %s,\n", user.getFirstName()));
        body.append('\n');
        body.append("Une demande de regénération du mot de passe a été faite sur Agrosyst.\n\n");
        body.append("Pour regénérer votre mot de passe, il vous suffit de vous rendre à l'adresse suivante :\n");
        body.append(reminderUrl);
        body.append("\n\n");
        body.append("Cordialement,\n");
        body.append("L'équipe Agrosyst.\n");
        body.append('\n');
        body.append(applicationBaseUrl);

        sendEmail("Agrosyst - Mot de passe oublié", user.getEmail(), null, null, body.toString(), null, null);
    }

    /**
     * Send email with user feedback.
     */
    public void sendFeedback(String user, String env, String location, String locationTitle, String category, String feedback, String requested, String referer, byte[] screenshotData) {
        String feedbackEmail = getConfig().getFeedbackEmail();

        StringBuilder body = new StringBuilder("Le ");
        body.append(new SimpleDateFormat("dd MMMM yyyy à HH:mm").format(context.getCurrentDate()));
        body.append(",\n\n");

        body.append("Description\n");
        body.append("-----------\n\n");
        body.append(feedback);

        body.append("\n\n");
        body.append("Emplacement\n");
        body.append("-----------\n");
        body.append("Titre de la page : " + locationTitle + "\n");
        body.append("URL              : " + location + "\n");
        body.append("URL demandée     : " + requested + "\n");
        body.append("URL précédente   : " + referer + "\n");

        body.append("\n");
        body.append("Environnement\n");
        body.append("-------------\n");
        body.append(env);
        body.append("Version services : ");
        body.append(getConfig().getApplicationVersion());

        sendEmail("Feedback Agrosyst - " + category + " - " + user, feedbackEmail, null, null, body.toString(), null, screenshotData);
    }

    /**
     * Envoi un mail de notification de création de compte.
     * 
     * @param user updated user
     */
    public void sendCreatedAccountNotification(AgrosystUser user, String token) {
        String applicationBaseUrl = getConfig().getApplicationBaseUrl();
        String supportEmail = getConfig().getSupportEmail();
        String reminderUrl = String.format("%s/auth/retrieve-password-input.action?userId=%s&token=%s", applicationBaseUrl, user.getTopiaId(), token);

        StringBuilder body = new StringBuilder("Bonjour,\n");
        body.append('\n');
        body.append("Quelqu'un - sans doute vous ou un administrateur Agrosyst - a demandé d'allouer ou de renouveler votre mot de passe associé à votre compte sur le site " + applicationBaseUrl + "\n\n");
        body.append("Vous pouvez ignorer cette requête ou bien cliquer sur le lien suivant pour choisir votre mot de passe :\n");
        body.append(reminderUrl);
        body.append("\n\n");
        body.append("Si vous avez des questions par rapport à ce message vous pouvez contacter : " + supportEmail);
        body.append("\n\n");
        body.append("Cordialement,\n");
        body.append("L'équipe Agrosyst.\n");

        sendEmail("Agrosyst - Mise à jour de votre profil", user.getEmail(), null, null, body.toString(), null, null);
    }
}
