package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: CacheService.java 3960 2014-04-01 16:59:22Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/CacheService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.Serializable;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * @author Arnaud Thimel (Code Lutin)
 */
public class CacheService extends AbstractAgrosystService {

    public static final String CACHE_HAS_ROLE = "hasRole";

    // instance is static to be shared between several service instances
    protected static Cache<String, Serializable> cache;

    protected static Cache<String, Serializable> shortCache;

    protected static final Set<String> shortCacheDiscriminators = Sets.newHashSet(CACHE_HAS_ROLE);

    protected Cache<String, Serializable> getCacheInstance(String discriminator) {
        Cache<String, Serializable> result;
        if (shortCacheDiscriminators.contains(discriminator)) {
            if (shortCache == null && getConfig().isBusinessCachingEnabled()) {
                long duration = getConfig().getBusinessCachingShortDuration();
                shortCache = CacheBuilder.newBuilder().expireAfterWrite(duration, TimeUnit.SECONDS).build();
            }
            result = shortCache;
        } else {
            if (cache == null && getConfig().isBusinessCachingEnabled()) {
                long duration = getConfig().getBusinessCachingDuration();
                cache = CacheBuilder.newBuilder().expireAfterWrite(duration, TimeUnit.MINUTES).build();
            }
            result = cache;
        }
        return result;
    }

    public <V extends Serializable> V get(String discriminator, Callable<? extends V> loader) {
        V result = get(discriminator, null, loader);
        return result;
    }

    public <K, V extends Serializable> V get(String discriminator, K key, Callable<? extends V> loader) {
        try {
            V result;
            if (getConfig().isBusinessCachingEnabled()) {
                String cacheKey = String.format("%s#%s", discriminator, key);
                result = (V) getCacheInstance(discriminator).get(cacheKey, loader);
            } else {
                result = loader.call();
            }
            return result;
        } catch (Exception eee) {
            if (eee instanceof RuntimeException) {
                throw (RuntimeException)eee;
            } else {
                throw new AgrosystTechnicalException("Unable to load type from cache", eee);
            }
        }
    }

    public void clear() {
        if (cache != null) {
            cache.invalidateAll();
        }
        if (shortCache != null) {
            shortCache.invalidateAll();
        }
    }

    public void clear(String discriminator) {
        Cache<String, Serializable> instance = getCacheInstance(discriminator);
        if (instance != null) {
            instance.invalidateAll();
        }
    }

}
