package fr.inra.agrosyst.services.common;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AttachmentServiceImpl.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/common/AttachmentServiceImpl.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.List;

import org.hibernate.Hibernate;
import org.hibernate.Session;

import fr.inra.agrosyst.api.entities.AttachmentContent;
import fr.inra.agrosyst.api.entities.AttachmentContentTopiaDao;
import fr.inra.agrosyst.api.entities.AttachmentMetadata;
import fr.inra.agrosyst.api.entities.AttachmentMetadataTopiaDao;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.entities.security.AgrosystUserTopiaDao;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.common.AttachmentService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.services.AbstractAgrosystService;

/**
 * Default implementation of {@link AttachmentService}.
 *
 * @author <a href="mailto:sebastien.grimault@makina-corpus.com">S. Grimault</a>
 */
public class AttachmentServiceImpl extends AbstractAgrosystService implements AttachmentService {

    protected BusinessAuthorizationService authorizationService;

    protected AgrosystUserTopiaDao userDao;
    protected AttachmentMetadataTopiaDao attachmentMetadataDao;
    protected AttachmentContentTopiaDao attachmentContentDao;

    public void setAuthorizationService(BusinessAuthorizationService authorizationService) {
        this.authorizationService = authorizationService;
    }

    public void setUserDao(AgrosystUserTopiaDao userDao) {
        this.userDao = userDao;
    }

    public void setAttachmentMetadataDao(AttachmentMetadataTopiaDao attachmentMetadataDao) {
        this.attachmentMetadataDao = attachmentMetadataDao;
    }

    public void setAttachmentContentDao(AttachmentContentTopiaDao attachmentContentDao) {
        this.attachmentContentDao = attachmentContentDao;
    }

    @Override
    public AttachmentMetadata addAttachment(String objectReferenceId, InputStream inputStream, String fileName, String fileContentType) {

        authorizationService.checkAddAttachment(objectReferenceId);

        AgrosystUser author = userDao.forTopiaIdEquals(getSecurityContext().getUserId()).findUnique();

        AttachmentMetadata attachmentMetadata = attachmentMetadataDao.newInstance();
        attachmentMetadata.setObjectReferenceId(objectReferenceId);
        attachmentMetadata.setAuthor(author);
        attachmentMetadata.setName(fileName);
        attachmentMetadata.setContentType(fileContentType);

        try {
            // copy file into blob
            AttachmentContent attachmentContent = this.attachmentContentDao.newInstance();
            Session hibernateSession = getPersistenceContext().getHibernateSupport().getHibernateSession();
            Blob blob = Hibernate.getLobCreator(hibernateSession).createBlob(inputStream, inputStream.available());
            attachmentContent.setContent(blob);
            attachmentMetadata.setSize(blob.length());

            // persist blob and metadata
            attachmentMetadata = attachmentMetadataDao.create(attachmentMetadata);
            attachmentContent.setAttachmentMetadata(attachmentMetadata);
            attachmentContentDao.create(attachmentContent);
        } catch (IOException ex) {
            throw new AgrosystTechnicalException("Can't upload file in database", ex);
        } catch (SQLException ex) {
            throw new AgrosystTechnicalException("Can't upload file in database", ex);
        }

        getTransaction().commit();

        return attachmentMetadata;
    }

    @Override
    public void delete(String attachmentMetadataId) {
        authorizationService.checkDeleteAttachment(attachmentMetadataId);

        AttachmentMetadata attachmentMetadata = attachmentMetadataDao.forTopiaIdEquals(attachmentMetadataId).findUnique();
        AttachmentContent attachmentContent = attachmentContentDao.forAttachmentMetadataEquals(attachmentMetadata).findUnique();
        attachmentContentDao.delete(attachmentContent);
        attachmentMetadataDao.delete(attachmentMetadata);

        getTransaction().commit();
    }

    @Override
    public InputStream getAttachmentContent(String attachmentMetadataId) {
        authorizationService.checkReadAttachment(attachmentMetadataId);

        AttachmentMetadata attachmentMetadata = attachmentMetadataDao.forTopiaIdEquals(attachmentMetadataId).findUnique();
        AttachmentContent attachmentContents = this.attachmentContentDao.forAttachmentMetadataEquals(attachmentMetadata).findUnique();

        InputStream is;
        try {
            is = attachmentContents.getContent().getBinaryStream();
        } catch (SQLException ex) {
            throw new AgrosystTechnicalException("Can't get file content from database", ex);
        }
        return is;
    }

    @Override
    public List<AttachmentMetadata> getAttachmentMetadatas(String objectReferenceId) {
        List<AttachmentMetadata> result = attachmentMetadataDao.forObjectReferenceIdEquals(objectReferenceId).findAll();
        return result;
    }

    @Override
    public long getAttachmentMetadatasCount(String objectReferenceId) {
        long result = attachmentMetadataDao.forObjectReferenceIdEquals(objectReferenceId).count();
        return result;
    }

    @Override
    public AttachmentMetadata getAttachmentMetadata(String attachmentTopiaId) {
        AttachmentMetadata result = attachmentMetadataDao.forTopiaIdEquals(attachmentTopiaId).findUnique();
        return result;
    }
}
