package fr.inra.agrosyst.services;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DefaultServiceFactory.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/DefaultServiceFactory.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Supplier;

import fr.inra.agrosyst.api.services.history.MessageService;
import fr.inra.agrosyst.services.history.MessageServiceImpl;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaDaoSupplier;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.beans.BeanUtil;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.api.services.action.ActionService;
import fr.inra.agrosyst.api.services.common.AttachmentService;
import fr.inra.agrosyst.api.services.common.PricesService;
import fr.inra.agrosyst.api.services.context.NavigationContextService;
import fr.inra.agrosyst.api.services.domain.DomainService;
import fr.inra.agrosyst.api.services.edaplos.EdaplosService;
import fr.inra.agrosyst.api.services.effective.EffectiveCropCycleService;
import fr.inra.agrosyst.api.services.generic.GenericEntityService;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanService;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemService;
import fr.inra.agrosyst.api.services.input.InputService;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeService;
import fr.inra.agrosyst.api.services.measurement.MeasurementService;
import fr.inra.agrosyst.api.services.network.NetworkService;
import fr.inra.agrosyst.api.services.performance.PerformanceService;
import fr.inra.agrosyst.api.services.plot.PlotService;
import fr.inra.agrosyst.api.services.practiced.PracticedPlotService;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemService;
import fr.inra.agrosyst.api.services.referential.ExportService;
import fr.inra.agrosyst.api.services.referential.ImportService;
import fr.inra.agrosyst.api.services.referential.ReferentialService;
import fr.inra.agrosyst.api.services.security.AnonymizeService;
import fr.inra.agrosyst.api.services.security.AuthenticationService;
import fr.inra.agrosyst.api.services.security.AuthorizationService;
import fr.inra.agrosyst.api.services.security.BusinessAuthorizationService;
import fr.inra.agrosyst.api.services.security.TrackerService;
import fr.inra.agrosyst.api.services.users.UserService;
import fr.inra.agrosyst.services.action.ActionServiceImpl;
import fr.inra.agrosyst.services.common.AttachmentServiceImpl;
import fr.inra.agrosyst.services.common.CacheAware;
import fr.inra.agrosyst.services.common.CacheService;
import fr.inra.agrosyst.services.common.EmailService;
import fr.inra.agrosyst.services.common.EntityUsageService;
import fr.inra.agrosyst.services.common.PricesServiceImpl;
import fr.inra.agrosyst.services.context.NavigationContextServiceImpl;
import fr.inra.agrosyst.services.demo.DemoDatas;
import fr.inra.agrosyst.services.domain.DomainServiceImpl;
import fr.inra.agrosyst.services.edaplos.EdaplosServiceImpl;
import fr.inra.agrosyst.services.effective.EffectiveCropCycleServiceImpl;
import fr.inra.agrosyst.services.generic.GenericEntityServiceImpl;
import fr.inra.agrosyst.services.growingplan.GrowingPlanServiceImpl;
import fr.inra.agrosyst.services.growingsystem.GrowingSystemServiceImpl;
import fr.inra.agrosyst.services.input.InputServiceImpl;
import fr.inra.agrosyst.services.managementmode.ManagementModeServiceImpl;
import fr.inra.agrosyst.services.measurement.MeasurementServiceImpl;
import fr.inra.agrosyst.services.network.NetworkServiceImpl;
import fr.inra.agrosyst.services.performance.PerformanceServiceImpl;
import fr.inra.agrosyst.services.plot.PlotServiceImpl;
import fr.inra.agrosyst.services.practiced.PracticedPlotServiceImpl;
import fr.inra.agrosyst.services.practiced.PracticedSystemServiceImpl;
import fr.inra.agrosyst.services.referential.ExportServiceImpl;
import fr.inra.agrosyst.services.referential.ImportServiceImpl;
import fr.inra.agrosyst.services.referential.ReferentialServiceImpl;
import fr.inra.agrosyst.services.security.AnonymizeServiceImpl;
import fr.inra.agrosyst.services.security.AuthenticationServiceImpl;
import fr.inra.agrosyst.services.security.AuthorizationServiceImpl;
import fr.inra.agrosyst.services.security.BusinessAuthorizationServiceImpl;
import fr.inra.agrosyst.services.security.TrackerServiceImpl;
import fr.inra.agrosyst.services.users.UserServiceImpl;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class DefaultServiceFactory implements ServiceFactory {

    private static final Log log = LogFactory.getLog(DefaultServiceFactory.class);

    protected static final Map<Class<? extends AgrosystService>, Class<? extends AgrosystService>> INTERFACE_TO_IMPL = Maps.newHashMap();

    static {
        INTERFACE_TO_IMPL.put(DomainService.class, DomainServiceImpl.class);
        INTERFACE_TO_IMPL.put(GrowingPlanService.class, GrowingPlanServiceImpl.class);
        INTERFACE_TO_IMPL.put(GrowingSystemService.class, GrowingSystemServiceImpl.class);
        INTERFACE_TO_IMPL.put(NavigationContextService.class, NavigationContextServiceImpl.class);
        INTERFACE_TO_IMPL.put(ReferentialService.class, ReferentialServiceImpl.class);
        INTERFACE_TO_IMPL.put(ImportService.class, ImportServiceImpl.class);
        INTERFACE_TO_IMPL.put(ExportService.class, ExportServiceImpl.class);
        INTERFACE_TO_IMPL.put(EdaplosService.class, EdaplosServiceImpl.class);
        INTERFACE_TO_IMPL.put(UserService.class, UserServiceImpl.class);
        INTERFACE_TO_IMPL.put(PlotService.class, PlotServiceImpl.class);
        INTERFACE_TO_IMPL.put(PracticedSystemService.class, PracticedSystemServiceImpl.class);
        INTERFACE_TO_IMPL.put(PracticedPlotService.class, PracticedPlotServiceImpl.class);
        INTERFACE_TO_IMPL.put(GenericEntityService.class, GenericEntityServiceImpl.class);
        INTERFACE_TO_IMPL.put(NetworkService.class, NetworkServiceImpl.class);
        INTERFACE_TO_IMPL.put(AuthenticationService.class, AuthenticationServiceImpl.class);
        INTERFACE_TO_IMPL.put(AuthorizationService.class, AuthorizationServiceImpl.class);
        INTERFACE_TO_IMPL.put(BusinessAuthorizationService.class, BusinessAuthorizationServiceImpl.class);
        INTERFACE_TO_IMPL.put(AnonymizeService.class, AnonymizeServiceImpl.class);
        INTERFACE_TO_IMPL.put(ManagementModeService.class, ManagementModeServiceImpl.class);
        INTERFACE_TO_IMPL.put(EffectiveCropCycleService.class, EffectiveCropCycleServiceImpl.class);
        INTERFACE_TO_IMPL.put(AttachmentService.class, AttachmentServiceImpl.class);
        INTERFACE_TO_IMPL.put(MeasurementService.class, MeasurementServiceImpl.class);
        INTERFACE_TO_IMPL.put(PerformanceService.class, PerformanceServiceImpl.class);
        INTERFACE_TO_IMPL.put(PricesService.class, PricesServiceImpl.class);
        INTERFACE_TO_IMPL.put(ActionService.class, ActionServiceImpl.class);
        INTERFACE_TO_IMPL.put(InputService.class, InputServiceImpl.class);
        INTERFACE_TO_IMPL.put(TrackerService.class, TrackerServiceImpl.class);
        INTERFACE_TO_IMPL.put(MessageService.class, MessageServiceImpl.class);

        // Services without public interface
        INTERFACE_TO_IMPL.put(TrackerServiceImpl.class, TrackerServiceImpl.class);
        INTERFACE_TO_IMPL.put(EmailService.class, EmailService.class);
        INTERFACE_TO_IMPL.put(DemoDatas.class, DemoDatas.class);
        INTERFACE_TO_IMPL.put(EntityUsageService.class, EntityUsageService.class);
        INTERFACE_TO_IMPL.put(CacheService.class, CacheService.class);
    }

    protected static final String LEGACY_DAO_SUFFIX = "DAO";
    protected static final String DAO_SUFFIX = "TopiaDao";

    protected Map<Class<? extends AgrosystService>, AgrosystService> servicesCache = Maps.newConcurrentMap(); // TODO AThimel 04/10/13 Is it possible to improve generics ?
    protected Map<Class<? extends TopiaDao>, TopiaDao> daoCache = Maps.newConcurrentMap(); // TODO AThimel 04/10/13 Is it possible to improve generics ?

    protected ServiceContext serviceContext;

    public DefaultServiceFactory(ServiceContext serviceContext) {
        this.serviceContext = serviceContext;
    }

    public ServiceContext getServiceContext() {
        return serviceContext;
    }

    @Override
    public <E extends AgrosystService> E newService(Class<E> clazz) {
        Preconditions.checkNotNull(clazz);

        long start = System.currentTimeMillis();
        int nbServicesBefore = servicesCache.size();
        int nbDaoBefore = daoCache.size();

        E service = findOrCreateService(clazz);

        if (log.isTraceEnabled()) {
            long duration = System.currentTimeMillis() - start;
            int nbServicesCreated = servicesCache.size() - nbServicesBefore;
            int nbDaoCreated = daoCache.size() - nbDaoBefore;
            String format = "Service '%s' created in %dms. %d new services and %d new dao has been instantiated";
            String message = String.format(format, clazz.getSimpleName(), duration, nbServicesCreated, nbDaoCreated);
            log.trace(message);
        }

        return service;
    }

    protected <E extends AgrosystService> E findOrCreateService(Class<E> clazz) {
        // Load from cache
        E service = (E) servicesCache.get(clazz);

        if (service == null) {
            // instantiate service using empty constructor
            try {
                // TODO AThimel 14/06/13 Remplacer la map par quelque chose de viable
                Class<? extends AgrosystService> implClazz = INTERFACE_TO_IMPL.get(clazz);
                Preconditions.checkNotNull(implClazz, "Unable to find implementation class for service: " + clazz);

                service = (E) implClazz.getConstructor().newInstance();
            } catch (InstantiationException ie) {
                throw new AgrosystTechnicalException("Unable to instantiate service for class " + clazz.getName(), ie);
            } catch (InvocationTargetException ite) {
                throw new AgrosystTechnicalException("Unable to instantiate service for class " + clazz.getName(), ite);
            } catch (NoSuchMethodException nsme) {
                throw new AgrosystTechnicalException("Unable to instantiate service for class " + clazz.getName(), nsme);
            } catch (IllegalAccessException iae) {
                throw new AgrosystTechnicalException("Unable to instantiate service for class " + clazz.getName(), iae);
            }

            // Put instance in cache before init in case of some loop between the services
            servicesCache.put(clazz, service);

            // init instance
            injectProperties(service);

            // If the instance created in a service, set its serviceContext
            if (service instanceof AbstractAgrosystService) {
                ((AbstractAgrosystService) service).setContext(serviceContext);
            }
        }

        return service;
    }

    protected <E> void injectProperties(E instance) {
        // Check if some services has to be injected
        Set<PropertyDescriptor> descriptors =
                BeanUtil.getDescriptors(
                        instance.getClass(),
                        BeanUtil.IS_WRITE_DESCRIPTOR);

        for (PropertyDescriptor propertyDescriptor : descriptors) {

            Class<?> propertyType = propertyDescriptor.getPropertyType();
            Object toInject = null;

            if (AgrosystService.class.isAssignableFrom(propertyType)) {
                Class<? extends AgrosystService> serviceClass = (Class<? extends AgrosystService>) propertyType;
                toInject = findOrCreateService(serviceClass);
            } else if (AgrosystServiceConfig.class.isAssignableFrom(propertyType)) {
                toInject = serviceContext.getConfig();
            } else if (TopiaDao.class.isAssignableFrom(propertyType)) {
                Class<? extends TopiaDao> daoType = (Class<? extends TopiaDao>) propertyType;
                toInject = getDaoInstance(daoType);
            }

            if (toInject != null) {
                if (log.isTraceEnabled()) {
                    log.trace("injecting " + toInject + " in instance " + instance);
                }

                try {
                    propertyDescriptor.getWriteMethod().invoke(instance, toInject);
                } catch (IllegalAccessException iae) {
                    String message = String.format("Unable to inject '%s' in instance '%s'",
                            toInject.getClass().getName(), instance.getClass().getName());
                    throw new AgrosystTechnicalException(message, iae);
                } catch (InvocationTargetException ite) {
                    String message = String.format("Unable to inject '%s' in instance '%s'",
                            toInject.getClass().getName(), instance.getClass().getName());
                    throw new AgrosystTechnicalException(message, ite);
                }
            }
        }
    }

    protected <D extends TopiaDao> D getDaoInstance(Class<D> daoClass) {
        D toInject = (D) daoCache.get(daoClass);
        if (toInject == null) {
            String daoName = daoClass.getName();
            if (daoName.endsWith(LEGACY_DAO_SUFFIX) || daoName.endsWith(DAO_SUFFIX)) { // TODO AThimel 10/10/13 Improve this code
                TopiaDaoSupplier daoSupplier = serviceContext.getDaoSupplier();
                try {
                    String className = daoName.substring(0, daoName.length() - DAO_SUFFIX.length());
                    if (daoName.endsWith(LEGACY_DAO_SUFFIX)) {
                        if (log.isWarnEnabled()) {
                            log.warn("Legacy DAO detected: " + daoName);
                        }
                        className = daoName.substring(0, daoName.length() - LEGACY_DAO_SUFFIX.length());
                    }
                    Class<? extends TopiaEntity> aClass = (Class<TopiaEntity>) Class.forName(className);
                    toInject = (D) daoSupplier.getDao(aClass);
                    daoCache.put(daoClass, toInject);
                } catch (ClassNotFoundException e) {
                    if (log.isErrorEnabled()) {
                        log.error("An exception occurred", e);
                    }
                }
            } else {
                if (log.isWarnEnabled()) {
                    log.warn("Unable to guess entity name : " + daoClass);
                }
            }
        }

        // TODO DCossé 21/05/14 Check if necessary.
        if (toInject != null && toInject instanceof CacheAware) {
            ((CacheAware)toInject).setCacheServiceSupplier(new Supplier<CacheService>() {
                protected CacheService cacheService;
                @Override
                public CacheService get() {
                    if (cacheService == null) {
                        cacheService = newService(CacheService.class);
                    }
                    return cacheService;
                }
            });
        }

        return toInject;
    }

    @Override
    public <I> I newInstance(Class<I> clazz) {
        I instance;
        try {
            instance = clazz.getConstructor().newInstance();
        } catch (InstantiationException ie) {
            throw new AgrosystTechnicalException("Unable to instantiate object for class " + clazz.getName(), ie);
        } catch (InvocationTargetException ite) {
            throw new AgrosystTechnicalException("Unable to instantiate object for class " + clazz.getName(), ite);
        } catch (NoSuchMethodException nsme) {
            throw new AgrosystTechnicalException("Unable to instantiate object for class " + clazz.getName(), nsme);
        } catch (IllegalAccessException iae) {
            throw new AgrosystTechnicalException("Unable to instantiate object for class " + clazz.getName(), iae);
        }

        // init instance
        injectProperties(instance);

        return instance;
    }

}
