package fr.inra.agrosyst.services;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DefaultServiceContext.java 4085 2014-06-05 09:19:10Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/DefaultServiceContext.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Date;

import com.google.gson.Gson;

import fr.inra.agrosyst.api.entities.AgrosystTopiaDaoSupplier;
import fr.inra.agrosyst.commons.gson.AgrosystGsonSupplier;

import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.TopiaTransaction;

import com.google.common.base.Supplier;

import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.api.services.ServiceFactory;
import fr.inra.agrosyst.services.security.SecurityContext;

public class DefaultServiceContext implements ServiceContext {

    protected ServiceFactory serviceFactory;

    protected Supplier<AgrosystTopiaPersistenceContext> transactionSupplier;

    protected AgrosystTopiaPersistenceContext persistenceContext;

    protected AgrosystServiceConfig config;

    // private for security reasons
    private String authenticationToken;
    private SecurityContext securityContext;

    public DefaultServiceContext(AgrosystServiceConfig config, Supplier<AgrosystTopiaPersistenceContext> transactionSupplier) {
        this.config = config;
        this.transactionSupplier = transactionSupplier;
    }

    public DefaultServiceContext(AgrosystServiceConfig config, Supplier<AgrosystTopiaPersistenceContext> transactionSupplier, String authenticationToken) {
        this(config, transactionSupplier);
        this.authenticationToken = authenticationToken;
    }

    @Override
    public ServiceContext newServiceContext() {
        ServiceContext serviceContext = new DefaultServiceContext(config, transactionSupplier, authenticationToken);
        return serviceContext;
    }

    @Override
    public Date getCurrentDate() {
        return new Date();
    }

    @Override
    public TopiaTransaction getTransaction() {
        TopiaTransaction result = getTransaction(true);
        return result;
    }

    @Override
    public TopiaTransaction getTransaction(boolean create) {
        TopiaTransaction result = getTransaction0(create);
        return result;
    }

    @Override
    public AgrosystTopiaDaoSupplier getDaoSupplier() {
        AgrosystTopiaDaoSupplier result = getTransaction0(true);
        return result;
    }

    @Override
    public AgrosystTopiaPersistenceContext getPersistenceContext() {
        AgrosystTopiaPersistenceContext result = getTransaction0(true);
        return result;
    }

    @Override
    public AgrosystTopiaPersistenceContext getPersistenceContext(boolean create) {
        AgrosystTopiaPersistenceContext result = getTransaction0(create);
        return result;
    }

    @Override
    public void close() {
        // try to close non closed persistence context (false to not reopen it)
        AgrosystTopiaPersistenceContext context = getPersistenceContext(false);
        if (context != null) {
            context.close();
        }
    }

    protected AgrosystTopiaPersistenceContext getTransaction0(boolean create) {
        if (persistenceContext == null && create) {
            try {
                persistenceContext = transactionSupplier.get();
            } catch (TopiaException ex) {
                throw new AgrosystTechnicalException("Can't begin new transaction", ex);
            }
        }
        return persistenceContext;
    }

    public ServiceFactory getServiceFactory() {
        if (serviceFactory == null) {
            serviceFactory = new DefaultServiceFactory(this);
        }
        return serviceFactory;
    }

    @Override
    public <E extends AgrosystService> E newService(Class<E> clazz) {
        E result = getServiceFactory().newService(clazz);
        return result;
    }

    @Override
    public AgrosystServiceConfig getConfig() {
        return config;
    }

    @Override
    public SecurityContext getSecurityContext() {
        if (securityContext == null) {
            securityContext = new SecurityContext(getServiceFactory(), authenticationToken);
        }
        return securityContext;
    }

    @Override
    public SecurityContext getSecurityContextAsUser(String userId) {
        SecurityContext securityContext = new SecurityContext(getServiceFactory(), authenticationToken, userId);
        return securityContext;
    }

    @Override
    public <I> I newInstance(Class<I> clazz) {
        return getServiceFactory().newInstance(clazz);
    }

    public void setAuthenticationToken(String authenticationToken) {
        this.authenticationToken = authenticationToken;
        this.securityContext = null;
    }

    @Override
    public Gson getGson() {
        return new AgrosystGsonSupplier().get();
    }

}
