package fr.inra.agrosyst.services;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AgrosystServiceConfig.java 4287 2014-09-03 16:33:06Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/AgrosystServiceConfig.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Properties;

import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Environment;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.config.ArgumentsParserException;
import org.nuiton.config.ConfigOptionDef;

/**
 * Service module configuration.
 *
 * @author Eric Chatellier
 */
public class AgrosystServiceConfig {

    private static final Log log = LogFactory.getLog(AgrosystServiceConfig.class);

    /**
     * Configuration filename.
     */
    protected static final String AGROSYST_DEFAULT_CONF_FILENAME = "agrosyst-default.properties";

    /**
     * Delegate application config.
     */
    protected ApplicationConfig config;

    public AgrosystServiceConfig(String configFileName) {
        try {
            ApplicationConfig defaultConfig = new ApplicationConfig(AGROSYST_DEFAULT_CONF_FILENAME);
            defaultConfig.loadDefaultOptions(ServiceConfigOption.values());
            defaultConfig.parse();
            if (StringUtils.isNotBlank(configFileName)) {
                Properties flatOptions = defaultConfig.getFlatOptions(false);

                config = new ApplicationConfig(flatOptions, configFileName);
                config.parse();
            } else {
                if (log.isWarnEnabled()) {
                    log.warn("No specific configuration provided, using the default one");
                }
                config = defaultConfig;
            }
        } catch (ArgumentsParserException ex) {
            throw new AgrosystTechnicalException("Can't read configuration", ex);
        }
    }

    public Properties getFlatOptions() {
        return config.getFlatOptions();
    }

    public String getApplicationVersion() {
        return config.getOption(ServiceConfigOption.APPLICATION_VERSION.key);
    }

    public String getApplicationBaseUrl() {
        String result = config.getOption(ServiceConfigOption.APPLICATION_BASE_URL.key).trim();
        if (result.endsWith("/")) {
            result = result.substring(0, result.length() - 1);
        }
        return result;
    }

    public boolean isDemoModeEnabled() {
        return config.getOptionAsBoolean(ServiceConfigOption.DEMO_MODE.key);
    }

    public boolean isGenerateBuiltinReferentials() {
        return config.getOptionAsBoolean(ServiceConfigOption.GENERATE_BUILTIN_REFERENTIALS.key);
    }

    public boolean isGenerateBuiltinUsers() {
        return config.getOptionAsBoolean(ServiceConfigOption.GENERATE_BUILTIN_USERS.key);
    }

    public int getSpeciesGraftSupportCodeSection() {
        return config.getOptionAsInt(ServiceConfigOption.SPECIES_GRAFTSUPPORT_CODESECTION.key);
    }

    public String getSpeciesGraftSupportUtilisation() {
        return config.getOption(ServiceConfigOption.SPECIES_GRAFTSUPPORT_UTILISATION.key);
    }

    public String getSpeciesGraftCloneTypeVarietal() {
        return config.getOption(ServiceConfigOption.SPECIES_GRAFTCLONE_TYPEVARIETAL.key);
    }

    public boolean isTokenStorageEnabled() {
        return config.getOptionAsBoolean(ServiceConfigOption.TOKEN_STORAGE_ENABLED.key);
    }

    public boolean isBusinessCachingEnabled() {
        return config.getOptionAsBoolean(ServiceConfigOption.BUSINESS_CACHING_ENABLED.key);
    }

    public void setBusinessCachingEnabled(boolean value) {
        config.setOption(ServiceConfigOption.BUSINESS_CACHING_ENABLED.key, String.valueOf(value));
    }

    public long getBusinessCachingDuration() {
        return config.getOptionAsLong(ServiceConfigOption.BUSINESS_CACHING_DURATION.key);
    }

    public long getBusinessCachingShortDuration() {
        return config.getOptionAsLong(ServiceConfigOption.BUSINESS_CACHING_SHORT_DURATION.key);
    }

    public int getActaDosageSpcCroppingZonesGroupId() {
        return config.getOptionAsInt(ServiceConfigOption.ACTA_DOSAGE_SPC_CROPPING_ZONES_GROUP_ID.key);
    }

    public String getFileEncoding() {
        return config.getOption(ServiceConfigOption.FILE_ENCODING.key);
    }

    public String getSmtpHost() {
        return config.getOption(ServiceConfigOption.EMAIL_SMPT_HOST.key);
    }

    public int getSmtpPort() {
        return config.getOptionAsInt(ServiceConfigOption.EMAIL_SMPT_PORT.key);
    }

    public String getEmailFrom() {
        return config.getOption(ServiceConfigOption.EMAIL_FROM.key);
    }
    
    public String getFeedbackEmail() {
        return config.getOption(ServiceConfigOption.USER_FEEDBACK_EMAIL.key);
    }

    public String getSupportEmail() {
        return config.getOption(ServiceConfigOption.USER_SUPPORT_EMAIL.key);
    }
 
    public String getLogFileLocation() {
        return config.getOption(ServiceConfigOption.LOG_FILE_LOCATION.key);
    }

    static enum ServiceConfigOption implements ConfigOptionDef {

//        FILENAME(ApplicationConfig.CONFIG_FILE_NAME, AGROSYST_DEFAULT_CONF_FILENAME),

        APPLICATION_VERSION("agrosyst.services.application.version", null),
        APPLICATION_BASE_URL("agrosyst.services.application.base_url", "http://demo.codelutin.com/agrosyst-stable"),

        DATABASE_HBM2DLL_AUTO(Environment.HBM2DDL_AUTO, "none"), // Use "validate" on production env
        DATABASE_DIALECT(Environment.DIALECT, "org.hibernate.dialect.H2Dialect"),
        DATABASE_DRIVER(Environment.DRIVER, "org.h2.Driver"),
        DATABASE_URL(Environment.URL, "jdbc:h2:file:/tmp/agrosyst/h2data-web"),
        DATABASE_USER(Environment.USER, "sa"),
        DATABASE_PASS(Environment.PASS, ""),

        DATABASE_C3P0_MIN_SIZE(Environment.C3P0_MIN_SIZE, "5"),
        DATABASE_C3P0_MAX_SIZE(Environment.C3P0_MAX_SIZE, "20"),
        DATABASE_C3P0_TIMEOUT(Environment.C3P0_TIMEOUT, "300"),
        DATABASE_C3P0_MAX_STATEMENTS(Environment.C3P0_MAX_STATEMENTS, "50"),

        EMAIL_SMPT_HOST("agrosyst.services.email.smtp_host", "localhost"),
        EMAIL_SMPT_PORT("agrosyst.services.email.smtp_port", "25"),
        EMAIL_FROM("agrosyst.services.email.from", "no-reply+agrosyst@inra.fr"),
        USER_FEEDBACK_EMAIL("agrosyst.services.email.feedback", "agrosyst-commits@list.forge.codelutin.com"),
        USER_SUPPORT_EMAIL("agrosyst.services.email.support", "agrosyst-users@listes.inra.fr"),

        DEMO_MODE("agrosyst.services.demoMode", "false"),
        GENERATE_BUILTIN_REFERENTIALS("agrosyst.services.generateBuiltinReferentials", "true"),
        GENERATE_BUILTIN_USERS("agrosyst.services.generateBuiltinUsers", "true"),
        SPECIES_GRAFTSUPPORT_CODESECTION("agrosyst.services.species.graftsupport.codesection", "26"),
        SPECIES_GRAFTSUPPORT_UTILISATION("agrosyst.services.species.graftsupport.utilisation", "PG"),
        SPECIES_GRAFTCLONE_TYPEVARIETAL("agrosyst.services.species.graftslone.typevarietal", "CLO"),
        TOKEN_STORAGE_ENABLED("agrosyst.services.security.tokenStorageEnabled", "false"),
        FILE_ENCODING("agrosyst.services.file.encoding", "utf-8"),
        LOG_FILE_LOCATION("agrosyst.services.log.file.location", null),
        BUSINESS_CACHING_ENABLED("agrosyst.services.businessCaching.enabled", "true"),
        BUSINESS_CACHING_DURATION("agrosyst.services.businessCaching.duration", "5"), // minutes
        BUSINESS_CACHING_SHORT_DURATION("agrosyst.services.businessCaching.shortDuration", "30"), // seconds

        ACTA_DOSAGE_SPC_CROPPING_ZONES_GROUP_ID("agrosyst.services.referential.acta_dosage_spc_complet.id_groupe_culture_zones_cultivees", "428");

        private String key;
        private String defaultValue;

        private ServiceConfigOption(String key, String defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public Class<?> getType() {
            return null;
        }

        @Override
        public String getDescription() {
            return null;
        }

        @Override
        public boolean isTransient() {
            return false;
        }

        @Override
        public boolean isFinal() {
            return false;
        }

        @Override
        public void setDefaultValue(String defaultValue) {

        }

        @Override
        public void setTransient(boolean isTransient) {

        }

        @Override
        public void setFinal(boolean isFinal) {

        }
    }
}
