package fr.inra.agrosyst.services;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: AbstractAgrosystService.java 4318 2014-09-10 15:57:14Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/services/AbstractAgrosystService.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaTransaction;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.AgrosystTopiaPersistenceContext;
import fr.inra.agrosyst.api.entities.Entities;
import fr.inra.agrosyst.api.services.AgrosystService;
import fr.inra.agrosyst.services.security.SecurityContext;

/**
 * Tous les services doivent hériter de cette classe.
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public abstract class AbstractAgrosystService implements AgrosystService {

    private static final Log log = LogFactory.getLog(AbstractAgrosystService.class);

    /**
     * On va utiliser le contexte par délégation
     */
    protected ServiceContext context;

    public ServiceContext getContext() {
        return context;
    }

    public void setContext(ServiceContext context) {
        this.context = context;
    }

    public TopiaTransaction getTransaction() {
        return context.getTransaction();
    }

    public AgrosystTopiaPersistenceContext getPersistenceContext() {
        return context.getPersistenceContext();
    }

    protected AgrosystServiceConfig getConfig() {
        return context.getConfig();
    }

    public SecurityContext getSecurityContext() {
        return context.getSecurityContext();
    }

    protected SecurityContext getSecurityContextAsUser(String userId) {
        return context.getSecurityContextAsUser(userId);
    }

    public <I> I newInstance(Class<I> clazz) {
        return context.newInstance(clazz);
    }

    @Override
    protected void finalize() throws Throwable {
        if (log.isDebugEnabled()) {
            log.debug("FINALIZE " + getClass().getName());
        }
        super.finalize();
    }

    public <K extends TopiaEntity> void easyBind(TopiaDao<K> dao, List<K> existingRaw, List<K> newItemsRaw,
                                                 Function<K, Void> onNewItem, String... exclusions) {
        easyBind(dao, existingRaw, newItemsRaw, onNewItem, null, exclusions);
    }

    public <K extends TopiaEntity> void easyBind(TopiaDao<K> dao, List<K> existingRaw, List<K> newItemsRaw,
                                                 Function<K, Void> onNewItem, Function<K, Void> afterBind,
                                                 String... exclusions) {

        // This list represents the elements to remove at the end of the loop
        List<K> toDelete = easyBindNoDelete(dao, existingRaw, newItemsRaw, onNewItem, afterBind, exclusions);

        // Now delete all unused items
        dao.deleteAll(toDelete);
    }

    public <K extends TopiaEntity> List<K> easyBindNoDelete(TopiaDao<K> dao, List<K> existingRaw, List<K> newItemsRaw,
                                                            Function<K, Void> onNewItem, Function<K, Void> afterBind,
                                                            String... exclusions) {

        List<K> emptyList = Lists.newArrayList();
        List<K> existing = MoreObjects.firstNonNull(existingRaw, emptyList);
        List<K> newItems = MoreObjects.firstNonNull(newItemsRaw, emptyList);

        Set<String> exclusionsSet = Sets.newHashSet(TopiaEntity.PROPERTY_TOPIA_CREATE_DATE);
        if (exclusions != null) {
            Iterables.addAll(exclusionsSet, Sets.newHashSet(exclusions));
        }
        String[] exclusionsArray = Iterables.toArray(exclusionsSet, String.class);

        // This list represents the elements to remove at the end of the loop
        List<K> toDelete = Lists.newArrayList(existing);

        // Maintain an index of the existing entities
        Map<String, K> existingIndex = Maps.uniqueIndex(existing, Entities.GET_TOPIA_ID);

        // Loop over each element
        for (K item : newItems) {
            String topiaId = item.getTopiaId();
            // Look for the element in the existing element's cache
            K entityEntry = existingIndex.get(topiaId);
            if (entityEntry == null) {
                entityEntry = dao.newInstance();
            } else {
                // If found, that means the element is still used, do not schedule its deletion
                toDelete.remove(entityEntry);
            }

            Class<K> clazz = (Class<K>) item.getClass();
            Binder<K, K> binder = BinderFactory.newBinder(clazz);

            binder.copyExcluding(item, entityEntry, exclusionsArray);

            if (afterBind != null) {
                afterBind.apply(entityEntry);
            }

            if (entityEntry.isPersisted()) {
                dao.update(entityEntry);
            } else {
                if (onNewItem != null) {
                    onNewItem.apply(entityEntry);
                }
                dao.create(entityEntry);
            }
        }

        // No deletion made in this method, just return the list
        return toDelete;
    }
}
